<?php
// Lägg till säkerhetsheaders
require_once __DIR__ . '/../security.php';
setSecurityHeaders();

// Hämta versionsinformation
$versionFile = __DIR__ . '/version.php';
$version = file_exists($versionFile) ? include $versionFile : ['commit' => 'Okänd', 'branch' => 'Okänd'];
$versionString = is_array($version) ? $version['commit'] : $version;
?>
<?php require_once __DIR__ . '/admin_guard.php'; ?>
<?php require_once __DIR__ . '/../lib/data_functions.php'; ?>
<!DOCTYPE html>
<html lang="sv">
<head>
    <meta charset="utf-8" />
    <meta name="viewport" content="width=device-width,initial-scale=1" />
    <title>Admin - ByggArkiv</title>
    <link rel="stylesheet" href="common.css">
    <style>
        /* Sid-specifika stilar för admin.html */
        .admin-title { text-align: center; color: #333; margin-bottom: 15px; font-size: 28px; font-weight: 600; }
        .admin-card {
            background: white !important;
        }
        /* Säkerställ att alla admin-button har samma höjd */
        .admin-button,
        a.admin-button,
        button.admin-button {
            height: 44px !important;
            min-height: 44px !important;
            max-height: 44px !important;
            box-sizing: border-box !important;
            display: inline-flex !important;
            align-items: center !important;
            justify-content: center !important;
            padding: 0 20px !important;
            line-height: 1.2 !important;
            vertical-align: middle !important;
            margin: 0 !important;
            border: none !important;
        }
        .admin-search-icon {
            background: #1a73e8;
            color: white;
            border: none;
            border-radius: 8px;
            padding: 8px 12px;
            cursor: pointer;
            display: flex;
            align-items: center;
            justify-content: center;
            transition: all 0.2s;
            height: 36px;
            min-width: 36px;
        }
        .admin-search-icon:hover {
            background: #1557b0;
            transform: translateY(-1px);
            box-shadow: 0 4px 12px rgba(26, 115, 232, 0.3);
        }
        .admin-search-modal {
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(0,0,0,0.5);
            display: flex;
            align-items: center;
            justify-content: center;
            z-index: 10000;
        }
        .admin-search-modal-content {
            background: white;
            border-radius: 12px;
            padding: 24px;
            max-width: 700px;
            width: 90%;
            max-height: 80vh;
            display: flex;
            flex-direction: column;
            box-shadow: 0 8px 32px rgba(0,0,0,0.3);
        }
        .admin-search-modal-header {
            display: flex;
            align-items: center;
            justify-content: space-between;
            margin-bottom: 16px;
        }
        .admin-search-modal-title {
            font-size: 20px;
            font-weight: 600;
            color: #333;
            margin: 0;
        }
        .admin-search-modal-close {
            background: none;
            border: none;
            font-size: 24px;
            cursor: pointer;
            color: #666;
            padding: 0;
            width: 32px;
            height: 32px;
            display: flex;
            align-items: center;
            justify-content: center;
            border-radius: 4px;
            transition: background-color 0.2s;
        }
        .admin-search-modal-close:hover {
            background-color: rgba(0,0,0,0.05);
        }
        .admin-search-input-container {
            margin-bottom: 16px;
        }
        .admin-search-input {
            width: 100%;
            padding: 12px;
            border: 1px solid #ccc;
            border-radius: 8px;
            font-size: 16px;
            box-sizing: border-box;
        }
        .admin-search-input:focus {
            outline: none;
            border-color: #1a73e8;
        }
        .admin-search-results {
            flex: 1;
            overflow-y: auto;
            max-height: calc(80vh - 200px);
        }
        .admin-search-result-section {
            margin-bottom: 24px;
        }
        .admin-search-result-section-title {
            font-size: 16px;
            font-weight: 600;
            color: #1a73e8;
            margin-bottom: 12px;
            padding-bottom: 8px;
            border-bottom: 2px solid #e0e0e0;
        }
        .admin-search-result-item {
            padding: 12px;
            border-bottom: 1px solid #e0e0e0;
            cursor: pointer;
            transition: background-color 0.2s;
        }
        .admin-search-result-item:hover {
            background-color: #f5f5f5;
        }
        .admin-search-result-item:last-child {
            border-bottom: none;
        }
        .admin-search-result-name {
            font-weight: 600;
            color: #333;
            font-size: 14px;
            margin-bottom: 4px;
        }
        .admin-search-result-details {
            color: #666;
            font-size: 13px;
        }
        .admin-search-highlight {
            background-color: #fff59d;
            font-weight: 600;
            padding: 2px 0;
        }
        .admin-search-no-results {
            text-align: center;
            color: #666;
            padding: 40px 20px;
            font-size: 14px;
        }
        @media (max-width: 768px) {
            .admin-grid { grid-template-columns: 1fr; }
            .admin-card { padding: 20px; }
            .admin-search-modal-content {
                max-width: 100%;
                width: 100%;
                height: 100%;
                max-height: 100vh;
                border-radius: 0;
            }
            .admin-search-results {
                max-height: calc(100vh - 200px);
            }
            .admin-search-input {
                font-size: 16px; /* Förhindra zoom på iOS */
            }
        }
        
        /* Populära länkar */
        .popular-links-section {
            background: white;
            border-radius: 8px;
            padding: 20px;
            margin-bottom: 30px;
            box-shadow: 0 2px 4px rgba(0,0,0,0.1);
        }
        
        .popular-links-header {
            display: flex;
            align-items: center;
            justify-content: space-between;
            margin-bottom: 15px;
        }
        
        .popular-links-title {
            margin: 0;
            color: #333;
            font-size: 20px;
            font-weight: 600;
        }
        
        .popular-links-actions {
            display: flex;
            align-items: center;
            gap: 8px;
        }
        
        .reset-stats-btn,
        .pin-links-btn {
            background: none;
            border: 1px solid #ddd;
            border-radius: 4px;
            padding: 6px 10px;
            cursor: pointer;
            color: #666;
            display: flex;
            align-items: center;
            justify-content: center;
            transition: all 0.2s;
        }
        
        .reset-stats-btn:hover {
            background: #f5f5f5;
            border-color: #999;
            color: #333;
        }
        
        .pin-links-btn {
            min-width: 36px;
        }
        
        .pin-links-btn:hover {
            background: #f5f5f5;
            border-color: #999;
            color: #333;
        }
        
        .pin-links-btn.pinned {
            background: #e3f2fd;
            border-color: #2196f3;
            color: #2196f3;
        }
        
        .pin-links-btn.pinned:hover {
            background: #bbdefb;
            border-color: #1976d2;
            color: #1976d2;
        }
        
        .popular-links-container {
            display: flex;
            flex-wrap: wrap;
            gap: 10px;
        }
        
        .popular-link-btn {
            background: #1a73e8;
            color: white;
            border: none;
            border-radius: 6px;
            padding: 10px 16px;
            cursor: pointer;
            font-size: 14px;
            transition: all 0.2s;
            text-decoration: none;
            display: inline-block;
        }
        
        .popular-link-btn:hover {
            background: #1557b0;
            transform: translateY(-1px);
            box-shadow: 0 2px 8px rgba(26, 115, 232, 0.3);
        }
        
        .popular-links-placeholder {
            color: #999;
            font-style: italic;
            padding: 20px;
            text-align: center;
        }
        
        /* Trädstruktur */
        .admin-tree-section {
            background: white;
            border-radius: 8px;
            padding: 20px;
            box-shadow: 0 2px 4px rgba(0,0,0,0.1);
        }
        
        .admin-tree {
            margin-top: 20px;
        }
        
        .tree-category {
            margin-bottom: 8px;
        }
        
        .tree-category-header {
            display: flex;
            align-items: center;
            gap: 8px;
            padding: 10px 12px;
            background: #f8f9fa;
            border: 1px solid #e0e0e0;
            border-radius: 6px;
            cursor: pointer;
            user-select: none;
            transition: all 0.2s;
        }
        
        .tree-category-header:hover {
            background: #e9ecef;
            border-color: #ccc;
        }
        
        .tree-category-header.expanded .tree-expand-icon {
            transform: rotate(180deg);
        }
        
        .tree-icon {
            color: #1a73e8;
            flex-shrink: 0;
        }
        
        .tree-category-name {
            flex: 1;
            font-weight: 600;
            color: #333;
        }
        
        .tree-expand-icon {
            color: #666;
            transition: transform 0.2s;
            flex-shrink: 0;
        }
        
        .tree-category-content {
            display: none;
            padding-left: 24px;
            padding-top: 8px;
            padding-bottom: 8px;
        }
        
        .tree-category.expanded .tree-category-content {
            display: block;
        }
        
        .tree-link {
            display: block;
            padding: 8px 12px;
            color: #333;
            text-decoration: none;
            border-radius: 4px;
            margin-bottom: 4px;
            transition: all 0.2s;
        }
        
        .tree-link:hover {
            background: #f0f0f0;
            color: #1a73e8;
        }
        
        .tree-link-special {
            display: block;
            padding: 8px 12px;
            color: #333;
            margin-bottom: 4px;
        }
    </style>
</head>
<body>
    <div class="container">
        <header class="header">
            <a href="../index.html">
                <img src="../logo.png" alt="ByggArkiv Snickare Stefan" class="logo" onerror="this.style.display='none'">
            </a>
            <div style="display: flex; align-items: center; gap: 16px;">
                <button id="searchIcon" class="admin-search-icon" onclick="showAdminSearchModal()" title="Sök i användare, externa partners och projekt">
                    <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" width="20" height="20">
                        <path d="M15.5 14h-.79l-.28-.27C15.41 12.59 16 11.11 16 9.5 16 5.91 13.09 3 9.5 3S3 5.91 3 9.5 5.91 16 9.5 16c1.61 0 3.09-.59 4.23-1.57l.27.28v.79l5 4.99L20.49 19l-4.99-5zm-6 0C7.01 14 5 11.99 5 9.5S7.01 5 9.5 5 14 7.01 14 9.5 11.99 14 9.5 14z" fill="currentColor"/>
                    </svg>
                </button>
                <div class="version-info">
                    <?php if (!empty($_SESSION['user_id'] ?? '')): ?>
                        <?php 
                        $userId = $_SESSION['user_id'];
                        $userName = getUserFullName($userId);
                        ?>
                        <span class="version-label">User:</span> <?= htmlspecialchars($userName ?: $userId) ?> 
                        <?php if ($userName): ?>
                            (<?= htmlspecialchars($userId) ?>)
                        <?php endif; ?> |
                    <?php endif; ?>
                    <span class="version-label">Version:</span> <?= htmlspecialchars($versionString) ?>
                    <?php if (is_array($version) && isset($version['deploy_time'])): ?>
                        | <span class="version-label">Deploy:</span> <?= htmlspecialchars($version['deploy_time']) ?>
                    <?php endif; ?>
                    <?php if (is_array($version) && isset($version['branch']) && $version['branch'] !== 'main'): ?>
                        | <span class="version-label">Branch:</span> <?= htmlspecialchars($version['branch']) ?>
                    <?php endif; ?>
                </div>
            </div>
        </header>

        <!-- Populära länkar -->
        <div class="popular-links-section">
            <div class="popular-links-header">
                <h2 class="popular-links-title">Populära länkar</h2>
                <div class="popular-links-actions">
                    <button id="pinLinksBtn" class="pin-links-btn" onclick="togglePinLinks()" title="Lås/upplås favoritknappar">
                        <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" width="16" height="16" fill="currentColor">
                            <path d="M16,12V4H17V2H7V4H8V12L6,14V16H11.2V22H12.8V16H18V14L16,12Z"/>
                        </svg>
                    </button>
                    <button id="resetStatsBtn" class="reset-stats-btn" onclick="resetLinkStats()" title="Nollställ statistik">
                        <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" width="16" height="16" fill="currentColor">
                            <path d="M6 19c0 1.1.9 2 2 2h8c1.1 0 2-.9 2-2V7H6v12zM19 4h-3.5l-1-1h-5l-1 1H5v2h14V4z"/>
                        </svg>
                    </button>
                </div>
            </div>
            <div id="popularLinksContainer" class="popular-links-container">
                <div class="popular-links-placeholder">Inga länkar har klickats ännu. Klicka på länkar nedan för att se dem här.</div>
            </div>
        </div>

        <!-- Trädstruktur för alla länkar -->
        <div class="admin-tree-section">
            <h2 class="category-title">Alla länkar</h2>
            <div class="admin-tree" id="adminTree">
                <!-- Bildhantering -->
                <div class="tree-category" data-category="Bildhantering">
                    <div class="tree-category-header" onclick="toggleCategory(this)">
                        <svg class="tree-icon" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" width="16" height="16" fill="currentColor">
                            <path d="M10 4H4c-1.11 0-2 .89-2 2v12c0 1.11.89 2 2 2h16c1.11 0 2-.89 2-2V8c0-1.11-.89-2-2-2h-8l-2-2z"/>
                        </svg>
                        <span class="tree-category-name">Bildhantering</span>
                        <svg class="tree-expand-icon" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" width="16" height="16" fill="currentColor">
                            <path d="M16.59 8.59L12 13.17 7.41 8.59 6 10l6 6 6-6z"/>
                        </svg>
                    </div>
                    <div class="tree-category-content">
                        <a href="#" class="tree-link" data-feature="filtered-view" onclick="handleClick('filtered-view'); return false;">Filtrerad bildvisning</a>
                        <a href="#" class="tree-link" data-feature="all-images" onclick="handleClick('all-images'); return false;">Alla bilder</a>
                    </div>
                </div>

                <!-- Grunddata -->
                <div class="tree-category" data-category="Grunddata">
                    <div class="tree-category-header" onclick="toggleCategory(this)">
                        <svg class="tree-icon" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" width="16" height="16" fill="currentColor">
                            <path d="M10 4H4c-1.11 0-2 .89-2 2v12c0 1.11.89 2 2 2h16c1.11 0 2-.89 2-2V8c0-1.11-.89-2-2-2h-8l-2-2z"/>
                        </svg>
                        <span class="tree-category-name">Grunddata</span>
                        <svg class="tree-expand-icon" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" width="16" height="16" fill="currentColor">
                            <path d="M16.59 8.59L12 13.17 7.41 8.59 6 10l6 6 6-6z"/>
                        </svg>
                    </div>
                    <div class="tree-category-content">
                        <a href="#" class="tree-link" data-feature="manage-users" onclick="handleClick('manage-users'); return false;">Hantera användare</a>
                        <a href="#" class="tree-link" data-feature="manage-projects" onclick="handleClick('manage-projects'); return false;">Hantera projekt</a>
                        <a href="#" class="tree-link" data-feature="manage-external-parties" onclick="handleClick('manage-external-parties'); return false;">Hantera extern användare</a>
                        <a href="#" class="tree-link" data-feature="manage-external-companies" onclick="handleClick('manage-external-companies'); return false;">Hantera extern partner</a>
                    </div>
                </div>

                <!-- Projektchatt -->
                <div class="tree-category" data-category="Projektchatt">
                    <div class="tree-category-header" onclick="toggleCategory(this)">
                        <svg class="tree-icon" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" width="16" height="16" fill="currentColor">
                            <path d="M10 4H4c-1.11 0-2 .89-2 2v12c0 1.11.89 2 2 2h16c1.11 0 2-.89 2-2V8c0-1.11-.89-2-2-2h-8l-2-2z"/>
                        </svg>
                        <span class="tree-category-name">Projektchatt</span>
                        <svg class="tree-expand-icon" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" width="16" height="16" fill="currentColor">
                            <path d="M16.59 8.59L12 13.17 7.41 8.59 6 10l6 6 6-6z"/>
                        </svg>
                    </div>
                    <div class="tree-category-content">
                        <a href="chat/list_chats.php?user_id=<?= htmlspecialchars(urlencode($_SESSION['user_id'] ?? '')) ?>" class="tree-link" data-feature="chat-list" onclick="recordLinkClick('chat-list', this.href, 'Chattöversikt'); return true;">Chattöversikt</a>
                        <div class="tree-link-special">
                            <span>Öppna projektchatt</span>
                            <div style="display: flex; gap: 8px; align-items: center; margin-top: 8px;">
                                <select id="chatProjectSelect" style="flex: 1; padding: 8px; border: 1px solid #ddd; border-radius: 4px; font-size: 14px;">
                                    <option value="">Välj projekt...</option>
                                </select>
                                <button id="btn-open-chat" onclick="openSelectedChat()" class="admin-button" style="padding: 8px 12px; min-width: auto; height: 36px;" title="Öppna chatt">
                                    <svg width="16" height="16" viewBox="0 0 24 24" fill="currentColor">
                                        <path d="M20 2H4c-1.1 0-2 .9-2 2v18l4-4h14c1.1 0 2-.9 2-2V4c0-1.1-.9-2-2-2zm0 14H6l-2 2V4h16v12z"/>
                                    </svg>
                                </button>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Dokumenthantering och Paketering -->
                <div class="tree-category" data-category="Dokumenthantering och Paketering">
                    <div class="tree-category-header" onclick="toggleCategory(this)">
                        <svg class="tree-icon" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" width="16" height="16" fill="currentColor">
                            <path d="M10 4H4c-1.11 0-2 .89-2 2v12c0 1.11.89 2 2 2h16c1.11 0 2-.89 2-2V8c0-1.11-.89-2-2-2h-8l-2-2z"/>
                        </svg>
                        <span class="tree-category-name">Dokumenthantering och Paketering</span>
                        <svg class="tree-expand-icon" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" width="16" height="16" fill="currentColor">
                            <path d="M16.59 8.59L12 13.17 7.41 8.59 6 10l6 6 6-6z"/>
                        </svg>
                    </div>
                    <div class="tree-category-content">
                        <a href="packaging/upload_documents.html" class="tree-link" data-feature="upload-documents" onclick="recordLinkClick('upload-documents', this.href, 'Dokumentuppladdning'); return true;">Dokumentuppladdning</a>
                        <a href="packaging/underlag.php" class="tree-link" data-feature="package-underlag" onclick="recordLinkClick('package-underlag', this.href, 'Paketera underlag för kund'); return true;">Paketera underlag för kund</a>
                        <a href="packaging/paketerade_projekt.php" class="tree-link" data-feature="packaged-projects" onclick="recordLinkClick('packaged-projects', this.href, 'Paketerade projekt'); return true;">Paketerade projekt</a>
                    </div>
                </div>

                <!-- Systemverktyg -->
                <div class="tree-category" data-category="Systemverktyg">
                    <div class="tree-category-header" onclick="toggleCategory(this)">
                        <svg class="tree-icon" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" width="16" height="16" fill="currentColor">
                            <path d="M10 4H4c-1.11 0-2 .89-2 2v12c0 1.11.89 2 2 2h16c1.11 0 2-.89 2-2V8c0-1.11-.89-2-2-2h-8l-2-2z"/>
                        </svg>
                        <span class="tree-category-name">Systemverktyg</span>
                        <svg class="tree-expand-icon" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" width="16" height="16" fill="currentColor">
                            <path d="M16.59 8.59L12 13.17 7.41 8.59 6 10l6 6 6-6z"/>
                        </svg>
                    </div>
                    <div class="tree-category-content">
                        <a href="#" class="tree-link" data-feature="system-message" onclick="handleClick('system-message'); return false;">Systemmeddelande</a>
                        <a href="#" class="tree-link" data-feature="resources" onclick="handleClick('resources'); return false;">Resurser</a>
                        <a href="#" class="tree-link" data-feature="billing" onclick="handleClick('billing'); return false;">Debitering</a>
                        <a href="diagnostics/activity_timeline.php<?= !empty($_SESSION['user_id'] ?? '') ? '?user_id=' . urlencode($_SESSION['user_id']) : '' ?>" class="tree-link" data-feature="activity-timeline" onclick="recordLinkClick('activity-timeline', this.href, 'Aktivitetslogg'); return true;">Aktivitetslogg</a>
                        <a href="chat/project_activity.php" class="tree-link" data-feature="project-activity" onclick="recordLinkClick('project-activity', this.href, 'Projektaktivitet'); return true;">Projektaktivitet</a>
                    </div>
                </div>

                <!-- Diagnostik -->
                <div class="tree-category" data-category="Diagnostik">
                    <div class="tree-category-header" onclick="toggleCategory(this)">
                        <svg class="tree-icon" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" width="16" height="16" fill="currentColor">
                            <path d="M10 4H4c-1.11 0-2 .89-2 2v12c0 1.11.89 2 2 2h16c1.11 0 2-.89 2-2V8c0-1.11-.89-2-2-2h-8l-2-2z"/>
                        </svg>
                        <span class="tree-category-name">Diagnostik</span>
                        <svg class="tree-expand-icon" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" width="16" height="16" fill="currentColor">
                            <path d="M16.59 8.59L12 13.17 7.41 8.59 6 10l6 6 6-6z"/>
                        </svg>
                    </div>
                    <div class="tree-category-content">
                        <a href="#" class="tree-link" data-feature="repair" onclick="handleClick('repair'); return false;">Repair / Städning</a>
                        <a href="diagnostics/system_diag.php" class="tree-link" data-feature="system-diagnostics" target="_blank" onclick="recordLinkClick('system-diagnostics', this.href, 'Kör systemdiagnostik'); return true;">Kör systemdiagnostik</a>
                        <a href="#" class="tree-link" data-feature="php-info" onclick="handleClick('php-info'); return false;">PHP Information</a>
                        <a href="#" class="tree-link" data-feature="version-history" onclick="handleClick('version-history'); return false;">Versionshistorik</a>
                    </div>
                </div>

                <!-- Dokumentation -->
                <div class="tree-category" data-category="Dokumentation">
                    <div class="tree-category-header" onclick="toggleCategory(this)">
                        <svg class="tree-icon" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" width="16" height="16" fill="currentColor">
                            <path d="M10 4H4c-1.11 0-2 .89-2 2v12c0 1.11.89 2 2 2h16c1.11 0 2-.89 2-2V8c0-1.11-.89-2-2-2h-8l-2-2z"/>
                        </svg>
                        <span class="tree-category-name">Dokumentation</span>
                        <svg class="tree-expand-icon" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" width="16" height="16" fill="currentColor">
                            <path d="M16.59 8.59L12 13.17 7.41 8.59 6 10l6 6 6-6z"/>
                        </svg>
                    </div>
                    <div class="tree-category-content">
                        <a href="#" class="tree-link" data-feature="installation-guide" onclick="handleClick('installation-guide'); return false;">Installationsguide</a>
                        <a href="guides/specifikation.html<?= !empty($_SESSION['user_id'] ?? '') ? '?user_id=' . urlencode($_SESSION['user_id']) : '' ?>" class="tree-link" data-feature="specification" onclick="recordLinkClick('specification', this.href, 'Specifikation'); return true;">Specifikation</a>
                    </div>
                </div>
            </div>
        </div>

        <!-- Gammal struktur (döljs) -->
        <div style="display: none;">
        <h2 class="category-title">Bildhantering</h2>
        <div class="admin-grid">
            <div class="admin-card">
                <h3>Filtrerad bildvisning</h3>
                <p>Visa och filtrera uppladdade bilder efter projekt eller användare. Perfekt för att hitta specifika bilder eller se alla bilder från ett visst projekt eller användare.</p>
                <button id="btn-filtered-view" class="admin-button disabled" onclick="handleClick('filtered-view')">
                    Öppna filtrerad visning
                </button>
            </div>

            <div class="admin-card">
                <h3>Alla bilder</h3>
                <p>Visa alla uppladdade bilder organiserade efter projekt. Komplett översikt över alla bilder i systemet med metadata och anteckningar.</p>
                <button id="btn-all-images" class="admin-button" onclick="handleClick('all-images')">
                    Visa alla bilder
                </button>
            </div>
        </div>

        <h2 class="category-title">Grunddata</h2>
        <div class="admin-grid">
            <div class="admin-card">
                <h3>Hantera användare</h3>
                <p>Lägg till och hantera användare. Skapa personliga länkar för att spåra vem som laddar upp bilder.</p>
                <button id="btn-manage-users" class="admin-button" onclick="handleClick('manage-users')">
                    Hantera användare
                </button>
            </div>

            <div class="admin-card">
                <h3>Hantera extern användare</h3>
                <p>Lägg till, redigera och ta bort extern användare (t.ex. underleverantörer, kunder) som kan bjudas in till projekt.</p>
                <button id="btn-manage-external-parties" class="admin-button" onclick="handleClick('manage-external-parties')">
                    Hantera extern användare
                </button>
            </div>

            <div class="admin-card">
                <h3>Hantera Projekt</h3>
                <p>Skapa nya projekt, redigera befintliga och markera projekt som klara. Hantera projekt-ID, namn, beskrivningar och status för alla byggprojekt.</p>
                <button id="btn-manage-projects" class="admin-button" onclick="handleClick('manage-projects')">
                    Hantera projekt
                </button>
            </div>

            <div class="admin-card">
                <h3>Hantera extern partner</h3>
                <p>Lägg till, redigera och ta bort externa företag (partners) som kan kopplas till externa användare.</p>
                <button id="btn-manage-external-companies" class="admin-button" onclick="handleClick('manage-external-companies')">
                    Hantera extern partner
                </button>
            </div>

        </div>

        <h2 class="category-title">Admin av chatt</h2>
        <div class="admin-grid">
            <div class="admin-card">
                <h3>Chattöversikt</h3>
                <p>Visa alla projektchattar med statistik över antal meddelanden och bilder. Översiktlig vy över all chattaktivitet i systemet. Du kan också rensa chattar direkt från översikten.</p>
                <a href="chat/list_chats.php?user_id=<?= htmlspecialchars(urlencode($_SESSION['user_id'] ?? '')) ?>" class="admin-button">
                    Visa chattöversikt
                </a>
            </div>
            <div class="admin-card">
                <h3>Öppna projektchatt</h3>
                <p>Välj ett projekt och gå direkt till dess chatt. Snabb åtkomst till projektchattar för kommunikation.</p>
                <div style="display: flex; gap: 8px; align-items: center; margin-top: 12px;">
                    <select id="chatProjectSelect" style="flex: 1; padding: 10px; border: 1px solid #ddd; border-radius: 6px; font-size: 14px;">
                        <option value="">Välj projekt...</option>
                    </select>
                    <button id="btn-open-chat" onclick="openSelectedChat()" class="admin-button" style="padding: 10px 16px; min-width: auto; height: 44px;" title="Öppna chatt">
                        <svg width="20" height="20" viewBox="0 0 24 24" fill="currentColor" style="margin-right: 0;">
                            <path d="M20 2H4c-1.1 0-2 .9-2 2v18l4-4h14c1.1 0 2-.9 2-2V4c0-1.1-.9-2-2-2zm0 14H6l-2 2V4h16v12z"/>
                        </svg>
                    </button>
                </div>
            </div>
        </div>

        <h2 class="category-title">Dokumenthantering och Paketering</h2>
        <div class="admin-grid">
            <div class="admin-card">
                <h3>Dokumentuppladdning</h3>
                <p>Ladda upp dokument (PDF, TXT, CSV) för projekt. Separerat från bilduppladdning för snickare.</p>
                <a href="packaging/upload_documents.html" class="admin-button">
                    Ladda upp dokument
                </a>
            </div>
            
            <div class="admin-card">
                <h3>Paketera underlag för kund</h3>
                <p>Välj filer från ett projekt och skapa en zip-fil för nedladdning. Perfekt för att Paketera underlag för kunder med valda bilder och dokument.</p>
                <a href="packaging/underlag.php" class="admin-button">
                    Paketera underlag
                </a>
            </div>
            <div class="admin-card">
                <h3>Paketerade projekt</h3>
                <p>Visa och ladda ner tidigare skapade zip-filer. Hantera alla paketerade projekt och åtkomst till nedladdningsbara arkiv.</p>
                <a href="packaging/paketerade_projekt.php" class="admin-button">
                    Visa paketerade projekt
                </a>
            </div>
        </div>

        <h2 class="category-title">Systemverktyg</h2>
        <div class="admin-grid">
            <div class="admin-card">
                <h3>Systemmeddelande</h3>
                <p>Författa och hantera systemmeddelanden som visas för alla användare innan de kommer åt index.html eller chat.php.</p>
                <button id="btn-system-message" class="admin-button" onclick="handleClick('system-message')">
                    Hantera systemmeddelande
                </button>
            </div>

            <div class="admin-card">
                <h3>Resurser</h3>
                <p>Visa statistik över bilder och diskutrymme. Se antal bilder per projekt, totalt utrymme och attic-arkiv.</p>
                <button id="btn-resources" class="admin-button" onclick="handleClick('resources')">
                    Visa resurser
                </button>
            </div>

            <div class="admin-card">
                <h3>Repair / Städning</h3>
                <p>Kontrollera och ta bort backupfiler, tumnaglar och bilder som inte längre refereras.</p>
                <button id="btn-repair" class="admin-button disabled" onclick="handleClick('repair')">
                    Öppna repair-sidan
                </button>
            </div>

            <div class="admin-card">
                <h3>Debitering</h3>
                <p>Beräkna månadskostnad baserat på fast kostnad, antal användare och använd lagring. Visa detaljerad specifikation för valt år och månad.</p>
                <button id="btn-billing" class="admin-button" onclick="handleClick('billing')">
                    Visa debitering
                </button>
            </div>

            <div class="admin-card">
                <h3>Aktivitetslogg</h3>
                <p>Visa tidslinje över systemaktivitet: chat-meddelanden, bilduppladdningar, dokument, zip-paketeringar och systemmeddelanden. Filtrera efter datumintervall.</p>
                <a href="diagnostics/activity_timeline.php<?= !empty($_SESSION['user_id'] ?? '') ? '?user_id=' . urlencode($_SESSION['user_id']) : '' ?>" class="admin-button">
                    Öppna aktivitetslogg
                </a>
            </div>

        </div>

        <h2 class="category-title">Diagnostik</h2>
        <div class="admin-grid">

            <div class="admin-card">
                <h3>Kör systemdiagnostik</h3>
                <p>Direktkör systemdiagnostiken och visa rapport i nytt fönster/flik.</p>
                <a href="diagnostics/system_diag.php" class="admin-button" target="_blank">
                    Kör rapport
                </a>
            </div>

            <div class="admin-card">
                <h3>PHP Information</h3>
                <p>Visa detaljerad information om PHP-installationen och serverkonfiguration. Teknisk information för systemadministratörer.</p>
                <button id="btn-php-info" class="admin-button disabled" onclick="handleClick('php-info')">
                    Visa PHP-info
                </button>
            </div>

            <div class="admin-card">
                <h3>Versionshistorik</h3>
                <p>Visa Git-historik med commits, författare och datum. Hjälper att spåra ändringar och förstå systemets utveckling över tid.</p>
                <button id="btn-version-history" class="admin-button" onclick="handleClick('version-history')">
                    Visa versionshistorik
                </button>
            </div>
        </div>

        <h2 class="category-title">Dokumentation</h2>
        <div class="admin-grid">
            <div class="admin-card">
                <h3>Installationsguide</h3>
                <p>Steg-för-steg instruktioner för att installera ByggArkiv på Synology NAS. Innehåller alla nödvändiga konfigurationssteg.</p>
                <button id="btn-installation-guide" class="admin-button" onclick="handleClick('installation-guide')">
                    Läs installationsguide
                </button>
            </div>

            <div class="admin-card">
                <h3>Specifikation</h3>
                <p>Teknisk dokumentation och specifikation för ByggArkiv-systemet. Beskriver funktioner, arkitektur och användning.</p>
                <a href="guides/specifikation.html<?= !empty($_SESSION['user_id'] ?? '') ? '?user_id=' . urlencode($_SESSION['user_id']) : '' ?>" class="admin-button">
                    Läs specifikation
                </a>
            </div>
        </div>
        </div>
        <!-- Slut på gammal struktur -->

    </div>

    <script>
        // Konfiguration av vilka funktioner som ska vara aktiverade
        const config = {
            'filtered-view': true,
            'all-images': true,
            'manage-projects': true,
            'external-parties': true,
            'manage-external-parties': true,
            'manage-external-companies': true,
            'manage-users': true,
            'resources': true,
            'billing': true,
            'system-diagnostics': true,
            'repair': true,
            'php-info': false,
            'version-history': true,
            'installation-guide': false,
            'specification': true,
            'files': true,
            'system-message': true,
            'project-activity': true
        };

        // URL-mappning för alla länkar
        const urlMap = {
            'filtered-view': 'files/list1p.php',
            'all-images': 'files/listallp.php',
            'manage-projects': 'projects/projects.php',
            'external-parties': './extusers/external_parties.php',
            'manage-external-parties': './extusers/external_parties.php',
            'manage-external-companies': './extusers/external_companies.php',
            'manage-users': 'users/userids.php',
            'resources': 'guides/resurser.php',
            'billing': 'billing/billing.php',
            'system-diagnostics': 'diagnostics/diag.php',
            'repair': 'diagnostics/repair.php',
            'php-info': 'diagnostics/phpinfo.php',
            'version-history': 'guides/versionshistorik.php',
            'installation-guide': 'guides/installation.html',
            'specification': 'guides/specifikation.html',
            'files': 'files/files.html',
            'system-message': 'system_message.php',
            'chat-list': 'chat/list_chats.php',
            'upload-documents': 'packaging/upload_documents.html',
            'package-underlag': 'packaging/underlag.php',
            'packaged-projects': 'packaging/paketerade_projekt.php',
            'activity-timeline': 'diagnostics/activity_timeline.php',
            'project-activity': 'chat/project_activity.php'
        };
        
        // Titel-mappning
        const titleMap = {
            'filtered-view': 'Filtrerad bildvisning',
            'all-images': 'Alla bilder',
            'manage-projects': 'Hantera projekt',
            'manage-external-parties': 'Hantera extern användare',
            'manage-external-companies': 'Hantera extern partner',
            'manage-users': 'Hantera användare',
            'resources': 'Resurser',
            'billing': 'Debitering',
            'system-diagnostics': 'Kör systemdiagnostik',
            'repair': 'Repair / Städning',
            'php-info': 'PHP Information',
            'version-history': 'Versionshistorik',
            'installation-guide': 'Installationsguide',
            'specification': 'Specifikation',
            'files': 'Filer',
            'system-message': 'Systemmeddelande',
            'chat-list': 'Chattöversikt',
            'upload-documents': 'Dokumentuppladdning',
            'package-underlag': 'Paketera underlag för kund',
            'packaged-projects': 'Paketerade projekt',
            'activity-timeline': 'Aktivitetslogg',
            'project-activity': 'Projektaktivitet'
        };
        
        // Hämta user_id
        function getUserId() {
            const urlParams = new URLSearchParams(window.location.search);
            return urlParams.get('user_id') || '<?= htmlspecialchars($_SESSION['user_id'] ?? '') ?>';
        }
        
        // Registrera klick på en länk
        // Kontrollera om favoritknappar är låsta
        function areLinksPinned() {
            return localStorage.getItem('adminLinksPinned') === 'true';
        }
        
        // Spara pin-status
        function setLinksPinned(pinned) {
            localStorage.setItem('adminLinksPinned', pinned ? 'true' : 'false');
        }
        
        // Växla pin-status
        function togglePinLinks() {
            const currentlyPinned = areLinksPinned();
            const newPinned = !currentlyPinned;
            setLinksPinned(newPinned);
            updatePinButtonState(newPinned);
        }
        
        // Uppdatera pin-knappens visuella tillstånd
        function updatePinButtonState(pinned) {
            const pinBtn = document.getElementById('pinLinksBtn');
            if (pinBtn) {
                if (pinned) {
                    pinBtn.classList.add('pinned');
                    pinBtn.title = 'Favoritknappar är låsta - klicka för att upplåsa';
                } else {
                    pinBtn.classList.remove('pinned');
                    pinBtn.title = 'Lås favoritknappar - förhindra att de ändras';
                }
            }
        }
        
        function recordLinkClick(featureId, url, title) {
            // Om favoritknappar är låsta, registrera inte nya klick
            if (areLinksPinned()) {
                return;
            }
            
            const userId = getUserId();
            if (!userId || !featureId) return;
            
            // Använd fetch med keepalive för att säkerställa att anropet skickas
            // även om sidan navigerar bort
            const formData = new FormData();
            formData.append('action', 'record_click');
            formData.append('user_id', userId);
            formData.append('feature_id', featureId);
            formData.append('url', url);
            formData.append('title', title);
            
            // Använd fetch med keepalive för att säkerställa att anropet skickas
            // Ignorera alla fel tyst - klickregistrering är inte kritisk för funktionaliteten
            fetch('../api/admin_link_stats.php', {
                method: 'POST',
                body: formData,
                keepalive: true
            }).catch(err => {
                // Ignorera fel tyst - klickregistrering är inte kritisk
                return { ok: false };
            }).then(response => {
                // Kontrollera om svaret är OK, men ignorera fel
                if (!response || !response.ok) {
                    return;
                }
                if (!response.ok) {
                    console.debug('Klickregistrering returnerade fel status:', response.status);
                }
                return response.json().catch(() => ({}));
            }).catch(error => {
                // Ignorera fel tyst - klickregistrering är inte kritisk
                // Endast logga i debug-läge för att undvika spam i konsolen
                console.debug('Klickregistrering misslyckades (icke-kritiskt):', error);
            });
        }
        
        function handleClick(feature) {
            if (config[feature]) {
                let url = urlMap[feature];
                if (!url) {
                    alert('Länk hittades inte för: ' + feature);
                    return;
                }
                
                // Lägg till user_id om det finns i URL
                const userId = getUserId();
                if (userId && !url.includes('user_id=')) {
                    url += (url.includes('?') ? '&' : '?') + 'user_id=' + encodeURIComponent(userId);
                }
                
                // Registrera klick
                const title = titleMap[feature] || feature;
                recordLinkClick(feature, url, title);
                
                window.location.href = url;
            } else {
                alert('Denna funktion är inte riktigt färdig.');
            }
        }
        
        // Trädordning för länkar (i samma ordning som de visas i trädstrukturen)
        const treeOrder = [
            'filtered-view',
            'all-images',
            'manage-users',
            'manage-projects',
            'manage-external-parties',
            'manage-external-companies',
            'chat-list',
            'project-activity',
            'open-project-chat',
            'upload-documents',
            'package-underlag',
            'packaged-projects',
            'system-message',
            'resources',
            'billing',
            'activity-timeline',
            'repair',
            'system-diagnostics',
            'php-info',
            'version-history',
            'installation-guide',
            'specification'
        ];
        
        // Hämta index i trädordningen för en feature_id
        function getTreeOrderIndex(featureId) {
            const index = treeOrder.indexOf(featureId);
            return index >= 0 ? index : 9999; // Okända länkar hamnar sist
        }
        
        // Ladda populära länkar
        async function loadPopularLinks() {
            const userId = getUserId();
            if (!userId) return;
            
            try {
                const response = await fetch(`../api/admin_link_stats.php?action=get_top&user_id=${encodeURIComponent(userId)}&limit=100`).catch(err => {
                    console.warn('Kunde inte ladda populära länkar:', err);
                    return null;
                });
                if (!response || !response.ok) {
                    return;
                }
                const data = await response.json().catch(err => {
                    console.warn('Kunde inte parsa JSON från populära länkar:', err);
                    return { success: false };
                });
                
                const container = document.getElementById('popularLinksContainer');
                if (!container) return;
                
                if (data.success && data.links && data.links.length > 0) {
                    // Sortera länkar enligt trädordningen (behåll de 5 första)
                    const sortedLinks = data.links.sort((a, b) => {
                        const indexA = getTreeOrderIndex(a.feature_id);
                        const indexB = getTreeOrderIndex(b.feature_id);
                        return indexA - indexB;
                    }).slice(0, 5); // Ta de 5 första efter sortering
                    
                    container.innerHTML = '';
                    sortedLinks.forEach(link => {
                        const btn = document.createElement('a');
                        btn.href = link.url;
                        btn.className = 'popular-link-btn';
                        btn.textContent = link.title;
                        btn.onclick = function(e) {
                            e.preventDefault();
                            recordLinkClick(link.feature_id, link.url, link.title);
                            window.location.href = link.url;
                        };
                        container.appendChild(btn);
                    });
                } else {
                    container.innerHTML = '<div class="popular-links-placeholder">Inga länkar har klickats ännu. Klicka på länkar nedan för att se dem här.</div>';
                }
            } catch (error) {
                console.error('Fel vid laddning av populära länkar:', error);
            }
        }
        
        // Nollställ statistik
        async function resetLinkStats() {
            if (!confirm('Är du säker på att du vill nollställa statistiken för populära länkar?')) {
                return;
            }
            
            const userId = getUserId();
            if (!userId) return;
            
            try {
                const formData = new FormData();
                formData.append('action', 'reset');
                formData.append('user_id', userId);
                
                const response = await fetch('../api/admin_link_stats.php', {
                    method: 'POST',
                    body: formData
                });
                
                const data = await response.json();
                if (data.success) {
                    loadPopularLinks();
                } else {
                    alert('Kunde inte nollställa statistik: ' + (data.message || 'Okänt fel'));
                }
            } catch (error) {
                console.error('Fel vid nollställning av statistik:', error);
                alert('Ett fel uppstod vid nollställning av statistik');
            }
        }
        
        // Toggle kategori (expand/collapse)
        function toggleCategory(header) {
            const category = header.closest('.tree-category');
            category.classList.toggle('expanded');
        }
        
        // Initiera trädstruktur (expandera första kategorin som standard)
        function initTreeStructure() {
            const firstCategory = document.querySelector('.tree-category');
            if (firstCategory) {
                firstCategory.classList.add('expanded');
            }
        }
        
        // Spåra klick på direkt länkar (backup om onclick inte fungerar)
        function trackDirectLinks() {
            document.querySelectorAll('a.tree-link[data-feature]').forEach(link => {
                // Om länken redan har onclick, lägg till vår tracking också
                const existingOnclick = link.getAttribute('onclick');
                if (!existingOnclick || !existingOnclick.includes('recordLinkClick')) {
                    link.addEventListener('click', function(e) {
                        const featureId = this.getAttribute('data-feature');
                        const url = this.href;
                        const title = this.textContent.trim();
                        
                        if (featureId && url && !url.includes('javascript:') && !url.includes('#')) {
                            recordLinkClick(featureId, url, title);
                        }
                    });
                }
            });
        }

        function updateButtonStates() {
            Object.keys(config).forEach(feature => {
                const button = document.getElementById(`btn-${feature}`);
                if (!button) return;
                if (config[feature]) { button.classList.remove('disabled'); } else { button.classList.add('disabled'); }
            });
        }
        document.addEventListener('DOMContentLoaded', updateButtonStates);

        function goToChat() {
            // Gå till projektlistan för att välja projekt
            const urlParams = new URLSearchParams(window.location.search);
            const userId = urlParams.get('user_id');
            let url = 'projects/projects.php';
            if (userId) {
                url += '?user_id=' + encodeURIComponent(userId);
            }
            window.location.href = url;
        }

        // Ladda projekt för chatt-dropdown
        async function loadChatProjects() {
            try {
                // Hämta projekt med meddelandeantal från chattöversikten
                const response = await fetch('chat/list_chats_data.php').catch(err => {
                    console.warn('Kunde inte ladda chattprojekt:', err);
                    return null;
                });
                if (!response || !response.ok) {
                    return;
                }
                const data = await response.json().catch(err => {
                    console.warn('Kunde inte parsa JSON från chattprojekt:', err);
                    return { success: false };
                });
                const select = document.getElementById('chatProjectSelect');
                
                if (data.success && data.chatStats && data.chatStats.length > 0) {
                    // Filtrera: bara aktiva projekt med minst ett meddelande
                    const validProjects = data.chatStats.filter(stat => {
                        const isActive = (stat.status || 'active') === 'active';
                        const hasMessages = (stat.total_messages || 0) > 0;
                        return isActive && hasMessages;
                    });
                    
                    if (validProjects.length === 0) {
                        const option = document.createElement('option');
                        option.value = '';
                        option.textContent = 'Inga aktiva projekt med meddelanden';
                        option.disabled = true;
                        // Rensa befintliga alternativ (utom första)
                        while (select.options.length > 1) {
                            select.remove(1);
                        }
                        select.appendChild(option);
                        return;
                    }
                    
                    // Sortera projekt efter namn
                    const sortedProjects = [...validProjects].sort((a, b) => {
                        const nameA = (a.project_name || '').toLowerCase();
                        const nameB = (b.project_name || '').toLowerCase();
                        return nameA.localeCompare(nameB, 'sv');
                    });
                    
                    // Rensa befintliga alternativ (utom första)
                    while (select.options.length > 1) {
                        select.remove(1);
                    }
                    
                    // Lägg till projekt
                    sortedProjects.forEach(stat => {
                        const option = document.createElement('option');
                        option.value = stat.project_id;
                        const name = stat.project_name || stat.project_id || 'Okänt projekt';
                        option.textContent = name + (stat.project_id ? ' (' + stat.project_id + ')' : '');
                        select.appendChild(option);
                    });
                } else {
                    const option = document.createElement('option');
                    option.value = '';
                    option.textContent = 'Inga projekt hittades';
                    option.disabled = true;
                    select.appendChild(option);
                }
            } catch (error) {
                console.error('Fel vid laddning av projekt:', error);
                const select = document.getElementById('chatProjectSelect');
                const option = document.createElement('option');
                option.value = '';
                option.textContent = 'Fel vid laddning';
                option.disabled = true;
                if (select.options.length > 1) {
                    while (select.options.length > 1) {
                        select.remove(1);
                    }
                }
                select.appendChild(option);
            }
        }

        // Öppna vald chatt
        function openSelectedChat() {
            const select = document.getElementById('chatProjectSelect');
            const projectId = select.value;
            
            if (!projectId) {
                alert('Välj ett projekt först');
                return;
            }
            
            const urlParams = new URLSearchParams(window.location.search);
            const userId = urlParams.get('user_id') || '<?= htmlspecialchars($_SESSION['user_id'] ?? '') ?>';
            
            let url = '../chat/chat.php?project_id=' + encodeURIComponent(projectId);
            if (userId) {
                url += '&user_id=' + encodeURIComponent(userId);
            }
            
            // Spåra klick (använd ett generiskt feature-id för projektchatt)
            recordLinkClick('open-project-chat', url, 'Öppna projektchatt');
            
            window.location.href = url;
        }

        // Ladda projekt när sidan laddas
        document.addEventListener('DOMContentLoaded', function() {
            loadChatProjects();
            loadPopularLinks();
            initTreeStructure();
            trackDirectLinks();
            // Initiera pin-knappens tillstånd
            updatePinButtonState(areLinksPinned());
        });

        // Admin-sökfunktion
        let adminSearchEscHandler = null;

        function showAdminSearchModal() {
            let modal = document.getElementById('adminSearchModal');
            if (!modal) {
                modal = document.createElement('div');
                modal.id = 'adminSearchModal';
                modal.className = 'admin-search-modal';
                modal.innerHTML = `
                    <div class="admin-search-modal-content">
                        <div class="admin-search-modal-header">
                            <h3 class="admin-search-modal-title">Sök</h3>
                            <button class="admin-search-modal-close" onclick="closeAdminSearchModal()" title="Stäng">×</button>
                        </div>
                        <div class="admin-search-input-container">
                            <input type="text" id="adminSearchInput" class="admin-search-input" placeholder="Sök efter användare, externa partners, projekt, filer, zip-filer, chat-meddelanden, bildbeskrivningar eller externa företag..." autocomplete="off">
                        </div>
                        <div class="admin-search-results" id="adminSearchResults">
                            <div class="admin-search-no-results">Skriv för att söka...</div>
                        </div>
                    </div>
                `;
                document.body.appendChild(modal);

                const searchInput = document.getElementById('adminSearchInput');
                searchInput.addEventListener('input', handleAdminSearchInput);
                searchInput.addEventListener('keydown', handleAdminSearchKeydown);

                modal.addEventListener('click', (e) => {
                    if (e.target === modal) {
                        closeAdminSearchModal();
                    }
                });
            }

            if (adminSearchEscHandler) {
                document.removeEventListener('keydown', adminSearchEscHandler);
            }
            adminSearchEscHandler = (e) => {
                if (e.key === 'Escape') {
                    closeAdminSearchModal();
                }
            };
            document.addEventListener('keydown', adminSearchEscHandler);

            modal.style.display = 'flex';
            const searchInput = document.getElementById('adminSearchInput');
            if (searchInput) {
                searchInput.focus();
            }
        }

        function closeAdminSearchModal() {
            const modal = document.getElementById('adminSearchModal');
            if (modal) {
                modal.style.display = 'none';
                const searchInput = document.getElementById('adminSearchInput');
                if (searchInput) {
                    searchInput.value = '';
                }
                const results = document.getElementById('adminSearchResults');
                if (results) {
                    results.innerHTML = '<div class="admin-search-no-results">Skriv för att söka...</div>';
                }
            }
            if (adminSearchEscHandler) {
                document.removeEventListener('keydown', adminSearchEscHandler);
                adminSearchEscHandler = null;
            }
        }

        function handleAdminSearchInput(e) {
            const searchTerm = e.target.value.trim();
            if (searchTerm.length === 0) {
                const results = document.getElementById('adminSearchResults');
                if (results) {
                    results.innerHTML = '<div class="admin-search-no-results">Skriv för att söka...</div>';
                }
                return;
            }

            performAdminSearch(searchTerm);
        }

        function handleAdminSearchKeydown(e) {
            if (e.key === 'Enter') {
                const results = document.getElementById('adminSearchResults');
                const firstResult = results.querySelector('.admin-search-result-item');
                if (firstResult) {
                    firstResult.click();
                }
            }
        }

        async function performAdminSearch(searchTerm) {
            const searchLower = searchTerm.toLowerCase();
            const results = {
                users: [],
                externalStaff: [],
                projects: [],
                files: [],
                zips: [],
                chatMessages: [],
                images: [],
                companies: []
            };

            try {
                // Sök i användare
                const usersResponse = await fetch('api/users.php?action=list');
                if (usersResponse.ok) {
                    const usersData = await usersResponse.json();
                    if (usersData.success && usersData.users) {
                        usersData.users.forEach(user => {
                            const name = (user.full_name || '').toLowerCase();
                            const mobile = (user.mobile || '').toLowerCase();
                            const userId = (user.user_id || '').toLowerCase();
                            if (name.includes(searchLower) || mobile.includes(searchLower) || userId.includes(searchLower)) {
                                results.users.push(user);
                            }
                        });
                    }
                }
            } catch (e) {
                console.error('Fel vid sökning i användare:', e);
            }

            try {
                // Sök i externa partners
                const staffResponse = await fetch('../api/chat/external_staff.php?action=list&include_inactive=1');
                if (staffResponse.ok) {
                    const staffData = await staffResponse.json();
                    if (staffData.success && staffData.staff) {
                        staffData.staff.forEach(staff => {
                            const name = (staff.name || staff.full_name || '').toLowerCase();
                            const mobile = (staff.mobile || '').toLowerCase();
                            const email = (staff.email || '').toLowerCase();
                            const companyName = (staff.company_name || '').toLowerCase();
                            if (name.includes(searchLower) || mobile.includes(searchLower) || email.includes(searchLower) || companyName.includes(searchLower)) {
                                results.externalStaff.push(staff);
                            }
                        });
                    }
                }
            } catch (e) {
                console.error('Fel vid sökning i externa partners:', e);
            }

            try {
                // Sök i projekt
                const urlParams = new URLSearchParams(window.location.search);
                const userId = urlParams.get('user_id') || '';
                const projectsResponse = await fetch('api/projects.php?action=list' + (userId ? '&user_id=' + encodeURIComponent(userId) : ''));
                if (projectsResponse.ok) {
                    const projectsData = await projectsResponse.json();
                    if (projectsData.success && projectsData.projects) {
                        projectsData.projects.forEach(project => {
                            const name = (project.name || '').toLowerCase();
                            const id = (project.id || '').toLowerCase();
                            const description = (project.description || '').toLowerCase();
                            if (name.includes(searchLower) || id.includes(searchLower) || description.includes(searchLower)) {
                                results.projects.push(project);
                            }
                        });
                    }
                }
            } catch (e) {
                console.error('Fel vid sökning i projekt:', e);
            }

            try {
                // Sök i filer
                const filesResponse = await fetch('api/search.php?action=search_files&search_term=' + encodeURIComponent(searchTerm));
                if (filesResponse.ok) {
                    const filesData = await filesResponse.json();
                    if (filesData.success && filesData.results) {
                        results.files = filesData.results;
                    }
                }
            } catch (e) {
                console.error('Fel vid sökning i filer:', e);
            }

            try {
                // Sök i zip-filer
                const zipsResponse = await fetch('api/search.php?action=search_zips&search_term=' + encodeURIComponent(searchTerm));
                if (zipsResponse.ok) {
                    const zipsData = await zipsResponse.json();
                    if (zipsData.success && zipsData.results) {
                        results.zips = zipsData.results;
                    }
                }
            } catch (e) {
                console.error('Fel vid sökning i zip-filer:', e);
            }

            try {
                // Sök i chat-meddelanden
                const chatResponse = await fetch('api/search.php?action=search_chat&search_term=' + encodeURIComponent(searchTerm));
                if (chatResponse.ok) {
                    const chatData = await chatResponse.json();
                    if (chatData.success && chatData.results) {
                        results.chatMessages = chatData.results;
                    }
                }
            } catch (e) {
                console.error('Fel vid sökning i chat-meddelanden:', e);
            }

            try {
                // Sök i bildbeskrivningar
                const imagesResponse = await fetch('api/search.php?action=search_images&search_term=' + encodeURIComponent(searchTerm));
                if (imagesResponse.ok) {
                    const imagesData = await imagesResponse.json();
                    if (imagesData.success && imagesData.results) {
                        results.images = imagesData.results;
                    }
                }
            } catch (e) {
                console.error('Fel vid sökning i bildbeskrivningar:', e);
            }

            try {
                // Sök i externa företag
                const companiesResponse = await fetch('api/search.php?action=search_companies&search_term=' + encodeURIComponent(searchTerm));
                if (companiesResponse.ok) {
                    const companiesData = await companiesResponse.json();
                    if (companiesData.success && companiesData.results) {
                        results.companies = companiesData.results;
                    }
                }
            } catch (e) {
                console.error('Fel vid sökning i externa företag:', e);
            }

            displayAdminSearchResults(results, searchTerm);
        }

        function highlightSearchTerm(text, searchTerm) {
            if (!text || !searchTerm) return escapeHtmlAdmin(text || '');
            const regex = new RegExp(`(${escapeRegexAdmin(searchTerm)})`, 'gi');
            const escapedText = escapeHtmlAdmin(text);
            return escapedText.replace(regex, '<span class="admin-search-highlight">$1</span>');
        }

        function escapeRegexAdmin(str) {
            return str.replace(/[.*+?^${}()|[\]\\]/g, '\\$&');
        }

        function escapeHtmlAdmin(text) {
            if (!text) return '';
            const div = document.createElement('div');
            div.textContent = text;
            return div.innerHTML;
        }

        function displayAdminSearchResults(results, searchTerm) {
            const resultsContainer = document.getElementById('adminSearchResults');
            if (!resultsContainer) return;

            const totalResults = results.users.length + results.externalStaff.length + results.projects.length + 
                                results.files.length + results.zips.length + results.chatMessages.length + results.images.length +
                                results.companies.length;
            if (totalResults === 0) {
                resultsContainer.innerHTML = '<div class="admin-search-no-results">Inga resultat hittades</div>';
                return;
            }

            let html = '';
            const urlParams = new URLSearchParams(window.location.search);
            const userId = urlParams.get('user_id') || '<?= htmlspecialchars($_SESSION['user_id'] ?? '') ?>';
            const userIdParam = userId ? '&user_id=' + encodeURIComponent(userId) : '';

            // Användare
            if (results.users.length > 0) {
                html += '<div class="admin-search-result-section">';
                html += '<div class="admin-search-result-section-title">Användare (' + results.users.length + ')</div>';
                results.users.forEach(user => {
                    const name = user.full_name || 'Namn saknas';
                    const mobile = user.mobile || '';
                    const userType = user.user_type === 'admin' ? ' (Admin)' : '';
                    const inactive = user.inactive ? ' <span style="color: #dc3545;">(Inaktiv)</span>' : '';
                    const url = 'users/userids.php?user_id=' + encodeURIComponent(user.user_id) + userIdParam;
                    html += '<div class="admin-search-result-item" onclick="window.location.href=\'' + escapeHtmlAdmin(url) + '\'">';
                    html += '<div class="admin-search-result-name">' + highlightSearchTerm(name, searchTerm) + userType + inactive + '</div>';
                    html += '<div class="admin-search-result-details">' + highlightSearchTerm(mobile, searchTerm) + '</div>';
                    html += '</div>';
                });
                html += '</div>';
            }

            // Externa partners
            if (results.externalStaff.length > 0) {
                html += '<div class="admin-search-result-section">';
                html += '<div class="admin-search-result-section-title">Externa partners (' + results.externalStaff.length + ')</div>';
                results.externalStaff.forEach(staff => {
                    const name = staff.name || staff.full_name || 'Namn saknas';
                    const mobile = staff.mobile || '';
                    const companyName = staff.company_name || '';
                    const inactive = staff.active === false ? ' <span style="color: #dc3545;">(Inaktiv)</span>' : '';
                    const url = 'extusers/external_parties.php?user_id=' + encodeURIComponent(userId);
                    html += '<div class="admin-search-result-item" onclick="window.location.href=\'' + escapeHtmlAdmin(url) + '\'">';
                    html += '<div class="admin-search-result-name">' + highlightSearchTerm(name, searchTerm) + inactive + '</div>';
                    html += '<div class="admin-search-result-details">' + highlightSearchTerm(mobile, searchTerm) + (companyName ? ' • ' + highlightSearchTerm(companyName, searchTerm) : '') + '</div>';
                    html += '</div>';
                });
                html += '</div>';
            }

            // Projekt
            if (results.projects.length > 0) {
                html += '<div class="admin-search-result-section">';
                html += '<div class="admin-search-result-section-title">Projekt (' + results.projects.length + ')</div>';
                results.projects.forEach(project => {
                    const name = project.name || project.id || 'Namn saknas';
                    const id = project.id || '';
                    const status = project.status === 'archived' ? ' <span style="color: #666;">(Arkiverad)</span>' : '';
                    const url = 'projects/projects.php?user_id=' + encodeURIComponent(userId);
                    html += '<div class="admin-search-result-item" onclick="window.location.href=\'' + escapeHtmlAdmin(url) + '\'">';
                    html += '<div class="admin-search-result-name">' + highlightSearchTerm(name, searchTerm) + status + '</div>';
                    html += '<div class="admin-search-result-details">ID: ' + highlightSearchTerm(id, searchTerm) + '</div>';
                    html += '</div>';
                });
                html += '</div>';
            }

            // Filer
            if (results.files.length > 0) {
                html += '<div class="admin-search-result-section">';
                html += '<div class="admin-search-result-section-title">Filer (' + results.files.length + ')</div>';
                results.files.forEach(file => {
                    const fileName = file.original || file.name || 'Okänt filnamn';
                    const projectName = file.project_name || 'Okänt projekt';
                    const note = file.note || '';
                    const url = 'files/files.html?project_id=' + encodeURIComponent(file.project_id) + userIdParam;
                    html += '<div class="admin-search-result-item" onclick="window.location.href=\'' + escapeHtmlAdmin(url) + '\'">';
                    html += '<div class="admin-search-result-name">' + highlightSearchTerm(fileName, searchTerm) + '</div>';
                    html += '<div class="admin-search-result-details">Projekt: ' + escapeHtmlAdmin(projectName);
                    if (note) {
                        html += ' • ' + highlightSearchTerm(note.substring(0, 100), searchTerm) + (note.length > 100 ? '...' : '');
                    }
                    html += '</div>';
                    html += '</div>';
                });
                html += '</div>';
            }

            // Paketerade zip-filer
            if (results.zips.length > 0) {
                html += '<div class="admin-search-result-section">';
                html += '<div class="admin-search-result-section-title">Paketerade zip-filer (' + results.zips.length + ')</div>';
                results.zips.forEach(zip => {
                    const fileName = zip.filename || 'Okänt filnamn';
                    const projectName = zip.project_name || 'Okänt projekt';
                    const url = 'packaging/paketerade_projekt.php?user_id=' + encodeURIComponent(userId);
                    html += '<div class="admin-search-result-item" onclick="window.location.href=\'' + escapeHtmlAdmin(url) + '\'">';
                    html += '<div class="admin-search-result-name">' + highlightSearchTerm(fileName, searchTerm) + '</div>';
                    html += '<div class="admin-search-result-details">Projekt: ' + escapeHtmlAdmin(projectName);
                    if (zip.size) {
                        const sizeMB = (zip.size / 1024 / 1024).toFixed(2);
                        html += ' • ' + sizeMB + ' MB';
                    }
                    html += '</div>';
                    html += '</div>';
                });
                html += '</div>';
            }

            // Chat-meddelanden
            if (results.chatMessages.length > 0) {
                html += '<div class="admin-search-result-section">';
                html += '<div class="admin-search-result-section-title">Chat-meddelanden (' + results.chatMessages.length + ')</div>';
                results.chatMessages.forEach(msg => {
                    const projectName = msg.project_name || 'Okänt projekt';
                    const messageText = msg.message || '';
                    const senderName = msg.sender_name || msg.sender_id || 'Okänd avsändare';
                    const url = '../chat/chat.php?project_id=' + encodeURIComponent(msg.project_id) + userIdParam;
                    html += '<div class="admin-search-result-item" onclick="window.location.href=\'' + escapeHtmlAdmin(url) + '\'">';
                    html += '<div class="admin-search-result-name">' + highlightSearchTerm(messageText.substring(0, 80), searchTerm) + (messageText.length > 80 ? '...' : '') + '</div>';
                    html += '<div class="admin-search-result-details">Projekt: ' + escapeHtmlAdmin(projectName) + ' • Avsändare: ' + escapeHtmlAdmin(senderName) + '</div>';
                    html += '</div>';
                });
                html += '</div>';
            }

            // Bildbeskrivningar
            if (results.images.length > 0) {
                html += '<div class="admin-search-result-section">';
                html += '<div class="admin-search-result-section-title">Bildbeskrivningar (' + results.images.length + ')</div>';
                results.images.forEach(img => {
                    const projectName = img.project_name || 'Okänt projekt';
                    const description = img.description || '';
                    const imageFile = img.image_file || img.original || 'Okänt filnamn';
                    const imageNumber = img.image_number !== null && img.image_number !== undefined ? img.image_number : null;
                    const fileType = img.file_type || 'image';
                    const imageNumberText = imageNumber !== null 
                        ? (fileType === 'video' ? 'Video #' + imageNumber : 'Bild #' + imageNumber)
                        : '';
                    const url = 'files/files.html?project_id=' + encodeURIComponent(img.project_id) + userIdParam;
                    html += '<div class="admin-search-result-item" onclick="window.location.href=\'' + escapeHtmlAdmin(url) + '\'">';
                    html += '<div class="admin-search-result-name">' + escapeHtmlAdmin(imageFile);
                    if (imageNumberText) {
                        html += ' <span style="color: #666; font-weight: normal;">(' + escapeHtmlAdmin(imageNumberText) + ')</span>';
                    }
                    html += '</div>';
                    html += '<div class="admin-search-result-details">Projekt: ' + escapeHtmlAdmin(projectName);
                    if (imageNumberText) {
                        html += ' • ' + highlightSearchTerm(imageNumberText, searchTerm);
                    }
                    if (description) {
                        html += (imageNumberText ? ' • ' : ' • ') + highlightSearchTerm(description.substring(0, 100), searchTerm) + (description.length > 100 ? '...' : '');
                    }
                    html += '</div>';
                    html += '</div>';
                });
                html += '</div>';
            }

            // Externa företag
            if (results.companies.length > 0) {
                html += '<div class="admin-search-result-section">';
                html += '<div class="admin-search-result-section-title">Externa företag (' + results.companies.length + ')</div>';
                results.companies.forEach(company => {
                    const companyName = company.name || 'Okänt företag';
                    const companyId = company.id || '';
                    const url = 'extusers/external_parties.php?user_id=' + encodeURIComponent(userId);
                    html += '<div class="admin-search-result-item" onclick="window.location.href=\'' + escapeHtmlAdmin(url) + '\'">';
                    html += '<div class="admin-search-result-name">' + highlightSearchTerm(companyName, searchTerm) + '</div>';
                    html += '<div class="admin-search-result-details">ID: ' + highlightSearchTerm(companyId, searchTerm) + '</div>';
                    html += '</div>';
                });
                html += '</div>';
            }

            resultsContainer.innerHTML = html;
        }
    </script>
</body>
</html>


