<?php
// Starta output buffering tidigt för att förhindra att varningar skrivs ut
ob_start();

// Förhindra att PHP-varningar skrivs ut (de loggas fortfarande)
error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);

// Registrera error handler för att fånga alla fel (efter att require_once är klar)

// Funktioner (definieras utanför try-block för att vara tillgängliga)
function readProjects($filePath) {
    if (!file_exists($filePath)) {
        return [];
    }
    $content = file_get_contents($filePath);
    if ($content === false) {
        return [];
    }
    $projects = json_decode($content, true);
    return is_array($projects) ? $projects : [];
}

function readChatMessages($projectId, $chatsDir) {
    $chatFile = $chatsDir . '/' . $projectId . '/messages.json';
    if (!file_exists($chatFile)) {
        return [];
    }
    $content = file_get_contents($chatFile);
    if ($content === false) {
        return [];
    }
    $messages = json_decode($content, true);
    return is_array($messages) ? $messages : [];
}

function saveChatMessages($projectId, $messages, $chatsDir) {
    $chatDir = $chatsDir . '/' . $projectId;
    if (!is_dir($chatDir)) {
        mkdir($chatDir, 0755, true);
    }
    $chatFile = $chatDir . '/messages.json';
    $json = json_encode($messages, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
    $result = file_put_contents($chatFile, $json);
    return $result !== false;
}

function createThumbnailImage($sourcePath, $destPath, $maxWidth = 320) {
    $info = @getimagesize($sourcePath);
    if ($info === false) {
        return false;
    }
    $mime = $info['mime'] ?? '';
    switch ($mime) {
        case 'image/jpeg':
        case 'image/jpg':
        case 'image/pjpeg':
            $src = @imagecreatefromjpeg($sourcePath);
            break;
        case 'image/png':
            $src = @imagecreatefrompng($sourcePath);
            break;
        case 'image/webp':
            if (function_exists('imagecreatefromwebp')) {
                $src = @imagecreatefromwebp($sourcePath);
            } else {
                $src = false;
            }
            break;
        default:
            $src = false;
    }
    if (!$src) {
        return false;
    }
    $width = imagesx($src);
    $height = imagesy($src);
    if ($width <= 0 || $height <= 0) {
        imagedestroy($src);
        return false;
    }
    $targetWidth = $width > $maxWidth ? $maxWidth : $width;
    $targetHeight = (int)round(($targetWidth / $width) * $height);
    $dst = imagecreatetruecolor($targetWidth, $targetHeight);
    $white = imagecolorallocate($dst, 255, 255, 255);
    imagefill($dst, 0, 0, $white);
    imagecopyresampled($dst, $src, 0, 0, 0, 0, $targetWidth, $targetHeight, $width, $height);
    $ok = imagejpeg($dst, $destPath, 75);
    imagedestroy($dst);
    imagedestroy($src);
    return $ok;
}

try {
    require_once __DIR__ . '/../../admin_guard.php';
    require_once __DIR__ . '/../../../security.php';
    require_once __DIR__ . '/../../../config.php';
    require_once __DIR__ . '/../../../lib/participant_helper.php';
    require_once __DIR__ . '/../../../lib/data_functions.php';

    // Registrera error handler efter att alla filer är inkluderade
    register_shutdown_function(function() {
        $error = error_get_last();
        if ($error !== null && in_array($error['type'], [E_ERROR, E_PARSE, E_CORE_ERROR, E_COMPILE_ERROR])) {
            @ob_clean();
            @http_response_code(500);
            @header('Content-Type: application/json; charset=utf-8');
            $json = @json_encode([
                'success' => false,
                'message' => 'Ett fel uppstod: ' . $error['message'] . ' på rad ' . $error['line'] . ' i ' . basename($error['file'])
            ]);
            if ($json !== false) {
                echo $json;
            }
            @exit;
        }
    });

    // Säkerhetsheaders tidigt
    setSecurityHeaders();

    // Rensa output buffer (tar bort eventuella varningar)
    ob_clean();

    // Svara alltid JSON
    header('Content-Type: application/json; charset=utf-8');

    // Sökvägar
    $projectsFile = __DIR__ . '/../../../data/projects.json';
    $chatsDir = __DIR__ . '/../../../chats';
    $uploadsDir = __DIR__ . '/../../../uploads';

    // Endast POST tillåten
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        http_response_code(405);
        echo json_encode(['success' => false, 'message' => 'Endast POST tillåten']);
        exit;
    }
    $sourceProjectId = sanitizeInput($_POST['source_project_id'] ?? '', 'project_id');
    $targetProjectId = sanitizeInput($_POST['target_project_id'] ?? '', 'project_id');
    $userId = sanitizeInput($_POST['user_id'] ?? '', 'user_id');
    $imageIds = $_POST['image_ids'] ?? [];

    if (empty($sourceProjectId) || empty($targetProjectId) || empty($userId)) {
        throw new Exception('Källprojekt, målprojekt och user_id krävs');
    }

    if (!is_array($imageIds) || empty($imageIds)) {
        throw new Exception('Inga filer valda');
    }

    // Validera admin-rättigheter
    if (!isParticipantAdmin($userId)) {
        throw new Exception('Endast administratörer kan kopiera filer');
    }

    // Validera att målprojektet är aktivt (eller tillåt projekt som saknas i projects.json)
    $projects = readProjects($projectsFile);
    $targetProject = null;
    foreach ($projects as $project) {
        if (($project['id'] ?? '') === $targetProjectId) {
            $targetProject = $project;
            break;
        }
    }

    // Om projektet finns i projects.json, kontrollera att det är aktivt
    if ($targetProject) {
        $targetStatus = $targetProject['status'] ?? 'active';
        if ($targetStatus !== 'active') {
            throw new Exception('Kan endast kopiera till aktiva projekt');
        }
    }
    // Om projektet inte finns i projects.json, tillåt kopiering ändå
    // (projekt kan finnas i chatten men inte i projects.json)

    // Läsa meddelanden från källprojektet
    $sourceMessages = readChatMessages($sourceProjectId, $chatsDir);
    
    // Hitta mediameddelanden att kopiera (bilder, dokument, videor)
    $mediaToCopy = [];
    foreach ($sourceMessages as $message) {
        $messageType = $message['type'] ?? '';
        if (in_array($messageType, ['image', 'document', 'video'])) {
            $messageId = $message['id'] ?? '';
            if (in_array($messageId, $imageIds)) {
                $mediaToCopy[] = $message;
            }
        }
    }

    if (empty($mediaToCopy)) {
        throw new Exception('Inga filer hittades att kopiera');
    }

    // Skapa målprojektets uploads-mapp om den inte finns
    $targetUploadDir = $uploadsDir . '/' . $targetProjectId;
    if (!is_dir($targetUploadDir)) {
        mkdir($targetUploadDir, 0755, true);
    }

    $copiedCount = 0;
    $errors = [];
    $alreadyExists = []; // Lista över filer som redan finns

    foreach ($mediaToCopy as $sourceMessage) {
        $mediaFile = $sourceMessage['media_file'] ?? '';
        $mediaType = $sourceMessage['type'] ?? 'image';
        
        if (empty($mediaFile)) {
            $errors[] = 'Meddelande ' . ($sourceMessage['id'] ?? 'okänt') . ' saknar media_file';
            continue;
        }

        // Sökvägar för käll- och målfil
        // Källfiler finns nu i chats-mappen, målfil i uploads-mappen (projektalbumet)
        $sourceFilePath = $chatsDir . '/' . $sourceProjectId . '/' . $mediaFile;
        $targetFilePath = $targetUploadDir . '/' . $mediaFile;

        // Kontrollera att källfilen finns
        if (!file_exists($sourceFilePath)) {
            $errors[] = 'Fil hittades inte: ' . $mediaFile;
            continue;
        }

        // Kontrollera om filen redan finns i målprojektet
        // Kontrollera både filen och JSON-metadatafilen
        $pathInfo = pathinfo($mediaFile);
        $baseName = $pathInfo['filename'];
        $targetMetaPath = $targetUploadDir . '/' . $baseName . '.json';
        
        if (file_exists($targetFilePath) || file_exists($targetMetaPath)) {
            // Filen finns redan - informera användaren och hoppa över kopieringen
            $alreadyExists[] = $mediaFile;
            continue;
        }

        // Kopiera filen
        if (!copy($sourceFilePath, $targetFilePath)) {
            $errors[] = 'Kunde inte kopiera fil: ' . $mediaFile;
            continue;
        }

        // Skapa thumbnail endast för bilder
        $thumbCreated = false;
        $thumbPath = '';
        if ($mediaType === 'image') {
            $thumbPath = preg_replace('/\.(jpg|jpeg|png|webp|heic|heif)$/i', '', $targetFilePath) . '-thumb.jpg';
            $thumbCreated = createThumbnailImage($targetFilePath, $thumbPath, 320);
        }

        // Skapa JSON-metadatafil för kopierad fil (för att synas i list1p.php)
        // Använd samma basnamn som filen (utan filändelse)
        // Viktigt: JSON-filen måste ha exakt samma basnamn som filen för att getAllUploadData() ska hitta den
        // Använd targetFilePath för att få rätt basnamn (inklusive _copy1 etc om det finns)
        $baseName = pathinfo($targetFilePath, PATHINFO_FILENAME);
        $metaPath = $targetUploadDir . '/' . $baseName . '.json';
        
        // Försök hämta metadata från källfilen om den finns
        // Källfiler finns i chats-mappen, men metadata kan finnas där eller i uploads
        $sourceBaseName = pathinfo($sourceFilePath, PATHINFO_FILENAME);
        $sourceMetaPath = $chatsDir . '/' . $sourceProjectId . '/' . $sourceBaseName . '.json';
        // Om metadata inte finns i chats, försök i uploads (för bakåtkompatibilitet)
        if (!file_exists($sourceMetaPath)) {
            $sourceMetaPath = $uploadsDir . '/' . $sourceProjectId . '/' . $sourceBaseName . '.json';
        }
        $sourceMetadata = null;
        
        if (file_exists($sourceMetaPath)) {
            $sourceMetaContent = file_get_contents($sourceMetaPath);
            if ($sourceMetaContent !== false) {
                $sourceMetadata = json_decode($sourceMetaContent, true);
            }
        }
        
        // Skapa metadata för kopierad fil
        $fileSize = file_exists($targetFilePath) ? filesize($targetFilePath) : 0;
        $mime = 'application/octet-stream'; // Default
        
        // Försök bestämma MIME-typ
        if ($mediaType === 'image') {
            $fileInfo = @getimagesize($targetFilePath);
            $mime = $fileInfo ? ($fileInfo['mime'] ?? 'image/jpeg') : 'image/jpeg';
        } elseif ($mediaType === 'document') {
            // Försök bestämma MIME från filändelse
            $ext = strtolower($pathInfo['extension'] ?? '');
            $mimeMap = [
                'pdf' => 'application/pdf',
                'doc' => 'application/msword',
                'docx' => 'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
                'xls' => 'application/vnd.ms-excel',
                'xlsx' => 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
                'txt' => 'text/plain',
            ];
            $mime = $mimeMap[$ext] ?? 'application/octet-stream';
        } elseif ($mediaType === 'video') {
            $ext = strtolower($pathInfo['extension'] ?? '');
            $mimeMap = [
                'mp4' => 'video/mp4',
                'mov' => 'video/quicktime',
                'avi' => 'video/x-msvideo',
                'mkv' => 'video/x-matroska',
            ];
            $mime = $mimeMap[$ext] ?? 'video/mp4';
        }
        
        $meta = [
            'project_id' => $targetProjectId,
            'note' => $sourceMetadata['note'] ?? '',
            'user_id' => $sourceMessage['participant_id'] ?? ($sourceMetadata['user_id'] ?? ''),
            'original' => $sourceMetadata['original'] ?? $mediaFile,
            'mime' => $mime,
            'bytes' => $fileSize,
            'stored_file' => $mediaFile,
            'stored_at' => date('c'),
            'upload_id' => time(),
            'file_type' => $mediaType
        ];
        
        if ($thumbCreated && file_exists($thumbPath)) {
            $meta['thumb_file'] = basename($thumbPath);
        }
        
        // Verifiera att filen finns innan vi skapar JSON-filen
        if (!file_exists($targetFilePath)) {
            $errors[] = 'Filen finns inte efter kopiering: ' . $targetFilePath;
            continue;
        }
        
        // Verifiera att JSON-filnamnet matchar filens basnamn
        $expectedBaseName = pathinfo($targetFilePath, PATHINFO_FILENAME);
        $actualBaseName = pathinfo($metaPath, PATHINFO_FILENAME);
        if ($expectedBaseName !== $actualBaseName) {
            $errors[] = 'JSON-filnamn matchar inte fil: ' . $expectedBaseName . ' vs ' . $actualBaseName;
            // Korrigera JSON-filnamnet
            $metaPath = $targetUploadDir . '/' . $expectedBaseName . '.json';
        }
        
        // Spara metadata
        $metaResult = file_put_contents($metaPath, json_encode($meta, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
        if ($metaResult === false) {
            $errors[] = 'Kunde inte spara metadata för: ' . $mediaFile . ' (sökväg: ' . $metaPath . ')';
            continue; // Hoppa över denna fil om metadata inte kunde sparas
        }
        
        // Verifiera att JSON-filen faktiskt skapades
        if (!file_exists($metaPath)) {
            $errors[] = 'JSON-filen skapades inte: ' . $metaPath;
            continue;
        }
        
        // Tilldela bildnummer endast för bilder (använder funktionen från data_functions.php)
        // Kontrollera att funktionen finns innan anrop
        if ($mediaType === 'image' && function_exists('assignImageNumber')) {
            $imageNumber = assignImageNumber($targetFilePath, $metaPath);
            if ($imageNumber !== false) {
                $meta['image_number'] = $imageNumber;
                // Uppdatera metadata med bildnummer
                file_put_contents($metaPath, json_encode($meta, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
            }
        }

        // Filen kopieras bara till projektalbumet, inte till chatten
        $copiedCount++;
    }

    // Filerna kopieras bara till projektalbumet (via JSON-metadatafiler),
    // inte till chatten, så vi behöver inte spara meddelanden

    // Debug-information för att verifiera att JSON-filer skapades
    $debugInfo = [];
    if ($copiedCount > 0) {
        $jsonFiles = glob($targetUploadDir . '/*.json');
        $debugInfo['json_files_in_target'] = count($jsonFiles);
        $debugInfo['target_dir'] = $targetUploadDir;
    }
    
    // Rensa output buffer innan vi skriver JSON
    ob_clean();
    echo json_encode([
        'success' => true,
        'copied_count' => $copiedCount,
        'already_exists' => $alreadyExists,
        'already_exists_count' => count($alreadyExists),
        'errors' => $errors,
        'debug' => $debugInfo
    ]);
    exit;

} catch (Exception $e) {
    // Rensa output buffer innan vi skriver felmeddelandet
    ob_clean();
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'message' => $e->getMessage()
    ]);
    exit;
} catch (Error $e) {
    // Fånga även PHP-fel (fatal errors etc)
    ob_clean();
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => 'Ett fel uppstod: ' . $e->getMessage()
    ]);
    exit;
} catch (Throwable $e) {
    // Fånga alla andra fel (inklusive fel från require_once etc)
    ob_clean();
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => 'Ett oväntat fel uppstod: ' . $e->getMessage() . ' (' . get_class($e) . ')'
    ]);
    exit;
}
