<?php
require_once __DIR__ . '/../admin_guard.php';
require_once __DIR__ . '/../../security.php';
require_once __DIR__ . '/../../config.php';

setSecurityHeaders();
header('Content-Type: application/json; charset=utf-8');

$action = sanitizeInput($_GET['action'] ?? $_POST['action'] ?? '', 'string');
$searchTerm = sanitizeInput($_GET['search_term'] ?? $_POST['search_term'] ?? '', 'string');

if (empty($action)) {
    http_response_code(400);
    echo json_encode(['success' => false, 'error' => 'Action saknas']);
    exit;
}

if (empty($searchTerm)) {
    echo json_encode(['success' => true, 'results' => []]);
    exit;
}

$searchLower = mb_strtolower($searchTerm, 'UTF-8');
$maxResults = 50; // Begränsa antal resultat per kategori

// Sökvägar
$projectsFile = __DIR__ . '/../../data/projects.json';
$downloadsDir = __DIR__ . '/../../downloads';
$chatsDir = __DIR__ . '/../../chats';
$uploadRoot = __DIR__ . '/../../' . rtrim(UPLOAD_ROOT, '/\\');

// Funktion för att läsa projekt
function readProjects($filePath) {
    if (!file_exists($filePath)) {
        return [];
    }
    $content = file_get_contents($filePath);
    if ($content === false) {
        return [];
    }
    $projects = json_decode($content, true);
    return is_array($projects) ? $projects : [];
}

// Funktion för att läsa chatmeddelanden
function readChatMessages($projectId, $chatsDir) {
    $chatFile = $chatsDir . '/' . $projectId . '/messages.json';
    if (!file_exists($chatFile)) {
        return [];
    }
    $content = file_get_contents($chatFile);
    if ($content === false) {
        return [];
    }
    $messages = json_decode($content, true);
    return is_array($messages) ? $messages : [];
}

try {
    switch ($action) {
        case 'search_files':
            // Sök i alla filer över alla projekt
            $projects = readProjects($projectsFile);
            $results = [];
            
            foreach ($projects as $project) {
                $projectId = $project['id'] ?? '';
                $projectName = $project['name'] ?? 'Okänt projekt';
                $projectDir = $uploadRoot . DIRECTORY_SEPARATOR . basename($projectId);
                
                if (!is_dir($projectDir)) {
                    continue;
                }
                
                $entries = scandir($projectDir) ?: [];
                $imageExts = ['jpg', 'jpeg', 'png', 'webp', 'heic', 'heif'];
                
                foreach ($entries as $entry) {
                    if ($entry === '.' || $entry === '..') continue;
                    $fullPath = $projectDir . DIRECTORY_SEPARATOR . $entry;
                    if (!is_file($fullPath)) continue;
                    
                    $ext = strtolower(pathinfo($entry, PATHINFO_EXTENSION));
                    // Hoppa över bildfiler, thumbnails och metadata
                    if (in_array($ext, $imageExts, true)) continue;
                    if (str_ends_with($entry, '-thumb.jpg')) continue;
                    if ($ext === 'json') continue;
                    if ($ext === 'finfo') {
                        // Läsa .finfo för att söka i note
                        $meta = [];
                        $metaRaw = file_get_contents($fullPath);
                        if ($metaRaw !== false) {
                            $meta = json_decode($metaRaw, true) ?: [];
                        }
                        $note = mb_strtolower($meta['note'] ?? '', 'UTF-8');
                        if (strpos($note, $searchLower) !== false) {
                            // Hitta motsvarande fil
                            $baseName = preg_replace('/\.finfo$/', '', $entry);
                            $filePath = $projectDir . DIRECTORY_SEPARATOR . $baseName;
                            if (file_exists($filePath)) {
                                $results[] = [
                                    'project_id' => $projectId,
                                    'project_name' => $projectName,
                                    'name' => $baseName,
                                    'original' => $meta['original'] ?? $baseName,
                                    'note' => $meta['note'] ?? '',
                                    'match_type' => 'note'
                                ];
                                if (count($results) >= $maxResults) break 2;
                            }
                        }
                        continue;
                    }
                    
                    // Sök i filnamn
                    $nameLower = mb_strtolower($entry, 'UTF-8');
                    $matches = false;
                    
                    // Läsa metadata för originalnamn och note
                    $metaPath = preg_replace('/\.[^.]+$/', '', $fullPath) . '.finfo';
                    $meta = [];
                    if (is_file($metaPath)) {
                        $metaRaw = file_get_contents($metaPath);
                        $meta = json_decode($metaRaw, true) ?: [];
                    }
                    
                    $original = mb_strtolower($meta['original'] ?? $entry, 'UTF-8');
                    $note = mb_strtolower($meta['note'] ?? '', 'UTF-8');
                    
                    if (strpos($nameLower, $searchLower) !== false || 
                        strpos($original, $searchLower) !== false ||
                        strpos($note, $searchLower) !== false) {
                        $results[] = [
                            'project_id' => $projectId,
                            'project_name' => $projectName,
                            'name' => $entry,
                            'original' => $meta['original'] ?? $entry,
                            'note' => $meta['note'] ?? '',
                            'match_type' => strpos($nameLower, $searchLower) !== false ? 'filename' : 
                                          (strpos($original, $searchLower) !== false ? 'original' : 'note')
                        ];
                        if (count($results) >= $maxResults) break 2;
                    }
                }
            }
            
            echo json_encode(['success' => true, 'results' => $results], JSON_UNESCAPED_UNICODE);
            break;
            
        case 'search_zips':
            // Sök i paketerade zip-filer
            $results = [];
            
            if (is_dir($downloadsDir)) {
                $projects = readProjects($projectsFile);
                $projectNames = [];
                foreach ($projects as $project) {
                    $projectNames[$project['id'] ?? ''] = $project['name'] ?? 'Okänt projekt';
                }
                
                $files = glob($downloadsDir . '/*.zip');
                foreach ($files as $file) {
                    $filename = basename($file);
                    $filenameLower = mb_strtolower($filename, 'UTF-8');
                    
                    if (strpos($filenameLower, $searchLower) !== false) {
                        // Extrahera projekt-ID från filnamnet
                        $projectId = '';
                        if (preg_match('/projekt_(\d+)_/', $filename, $matches)) {
                            $projectId = $matches[1];
                        }
                        
                        $results[] = [
                            'filename' => $filename,
                            'project_id' => $projectId,
                            'project_name' => $projectNames[$projectId] ?? 'Okänt projekt',
                            'size' => file_exists($file) ? filesize($file) : 0,
                            'date' => file_exists($file) ? filemtime($file) : 0
                        ];
                        
                        if (count($results) >= $maxResults) break;
                    }
                }
            }
            
            echo json_encode(['success' => true, 'results' => $results], JSON_UNESCAPED_UNICODE);
            break;
            
        case 'search_chat':
            // Sök i chat-meddelanden
            $projects = readProjects($projectsFile);
            $results = [];
            
            foreach ($projects as $project) {
                $projectId = $project['id'] ?? '';
                $projectName = $project['name'] ?? 'Okänt projekt';
                $messages = readChatMessages($projectId, $chatsDir);
                
                foreach ($messages as $message) {
                    $messageText = $message['message'] ?? '';
                    $messageType = $message['type'] ?? 'text';
                    
                    // Sök endast i textmeddelanden (inte system-meddelanden eller bilder)
                    if ($messageType === 'text' && !empty($messageText)) {
                        $messageLower = mb_strtolower($messageText, 'UTF-8');
                        if (strpos($messageLower, $searchLower) !== false) {
                            // Hämta avsändarnamn om möjligt
                            $participantId = $message['participant_id'] ?? '';
                            $senderName = '';
                            if (!empty($participantId)) {
                                $userIdsFile = __DIR__ . '/../../data/userids.json';
                                if (file_exists($userIdsFile)) {
                                    $users = json_decode(file_get_contents($userIdsFile), true) ?: [];
                                    foreach ($users as $user) {
                                        if (($user['user_id'] ?? '') === $participantId) {
                                            $senderName = $user['full_name'] ?? '';
                                            break;
                                        }
                                    }
                                }
                            }
                            
                            $results[] = [
                                'project_id' => $projectId,
                                'project_name' => $projectName,
                                'message_id' => $message['id'] ?? '',
                                'message' => $messageText,
                                'sender_name' => $senderName,
                                'sender_id' => $participantId,
                                'created_at' => $message['created_at'] ?? ''
                            ];
                            
                            if (count($results) >= $maxResults) break 2;
                        }
                    }
                }
            }
            
            echo json_encode(['success' => true, 'results' => $results], JSON_UNESCAPED_UNICODE);
            break;
            
        case 'search_images':
            // Sök i bildbeskrivningar (metadata JSON-filer)
            $projects = readProjects($projectsFile);
            $results = [];
            
            foreach ($projects as $project) {
                $projectId = $project['id'] ?? '';
                $projectName = $project['name'] ?? 'Okänt projekt';
                $projectDir = $uploadRoot . DIRECTORY_SEPARATOR . basename($projectId);
                
                if (!is_dir($projectDir)) {
                    continue;
                }
                
                $entries = scandir($projectDir) ?: [];
                $imageExts = ['jpg', 'jpeg', 'png', 'webp', 'heic', 'heif'];
                
                foreach ($entries as $entry) {
                    if ($entry === '.' || $entry === '..') continue;
                    
                    $ext = strtolower(pathinfo($entry, PATHINFO_EXTENSION));
                    // Sök endast i JSON-metadatafiler för bilder
                    if ($ext !== 'json') continue;
                    if (str_ends_with($entry, '-thumb.jpg')) continue;
                    
                    $jsonPath = $projectDir . DIRECTORY_SEPARATOR . $entry;
                    if (!is_file($jsonPath)) continue;
                    
                    $metaRaw = file_get_contents($jsonPath);
                    if ($metaRaw === false) continue;
                    
                    $meta = json_decode($metaRaw, true);
                    if (!is_array($meta)) continue;
                    
                    // Hämta bildnummer och filtyp från metadata
                    $imageNumber = isset($meta['image_number']) && is_numeric($meta['image_number']) 
                        ? (int)$meta['image_number'] 
                        : null;
                    $fileType = $meta['file_type'] ?? 'image'; // 'image' eller 'video'
                    
                    // Sök i note-fältet och bildnummer
                    $note = mb_strtolower($meta['note'] ?? '', 'UTF-8');
                    $matches = false;
                    
                    // Matcha mot note
                    if (!empty($note) && strpos($note, $searchLower) !== false) {
                        $matches = true;
                    }
                    
                    // Matcha mot bildnummer
                    if (!$matches && $imageNumber !== null) {
                        // Exakt matchning om söktermen är numerisk
                        if (is_numeric($searchTerm) && (int)$searchTerm === $imageNumber) {
                            $matches = true;
                        } 
                        // Partiell matchning (t.ex. "12" matchar "123")
                        elseif (strpos((string)$imageNumber, $searchTerm) !== false) {
                            $matches = true;
                        }
                    }
                    
                    if ($matches) {
                        // Hitta motsvarande bildfil
                        $baseName = preg_replace('/\.json$/', '', $entry);
                        $imageFile = null;
                        foreach ($imageExts as $imgExt) {
                            $testPath = $projectDir . DIRECTORY_SEPARATOR . $baseName . '.' . $imgExt;
                            if (file_exists($testPath)) {
                                $imageFile = $baseName . '.' . $imgExt;
                                break;
                            }
                        }
                        
                        // Kontrollera också videofiler om det är en video
                        if (!$imageFile && $fileType === 'video') {
                            $videoExts = ['mp4', 'webm', 'mov', 'avi'];
                            foreach ($videoExts as $vidExt) {
                                $testPath = $projectDir . DIRECTORY_SEPARATOR . $baseName . '.' . $vidExt;
                                if (file_exists($testPath)) {
                                    $imageFile = $baseName . '.' . $vidExt;
                                    break;
                                }
                            }
                        }
                        
                        if ($imageFile) {
                            $results[] = [
                                'project_id' => $projectId,
                                'project_name' => $projectName,
                                'image_file' => $imageFile,
                                'description' => $meta['note'] ?? '',
                                'original' => $meta['original'] ?? $imageFile,
                                'stored_at' => $meta['stored_at'] ?? '',
                                'image_number' => $imageNumber,
                                'file_type' => $fileType
                            ];
                            
                            if (count($results) >= $maxResults) break 2;
                        }
                    }
                }
            }
            
            echo json_encode(['success' => true, 'results' => $results], JSON_UNESCAPED_UNICODE);
            break;
            
        case 'search_companies':
            // Sök i externa företag
            $results = [];
            $companiesFile = __DIR__ . '/../../data/external_companies.json';
            
            if (file_exists($companiesFile)) {
                $companies = json_decode(file_get_contents($companiesFile), true) ?: [];
                
                foreach ($companies as $company) {
                    $companyId = mb_strtolower($company['id'] ?? '', 'UTF-8');
                    $companyName = mb_strtolower($company['name'] ?? '', 'UTF-8');
                    
                    if (strpos($companyId, $searchLower) !== false || 
                        strpos($companyName, $searchLower) !== false) {
                        $results[] = [
                            'id' => $company['id'] ?? '',
                            'name' => $company['name'] ?? ''
                        ];
                        
                        if (count($results) >= $maxResults) break;
                    }
                }
            }
            
            echo json_encode(['success' => true, 'results' => $results], JSON_UNESCAPED_UNICODE);
            break;
            
        default:
            http_response_code(400);
            echo json_encode(['success' => false, 'error' => 'Ogiltig action: ' . htmlspecialchars($action)]);
            break;
    }
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['success' => false, 'error' => $e->getMessage()]);
}
?>

