<?php
// För get_image action, returnera bild direkt (inte JSON) - måste vara FÖRE admin_guard
if (isset($_GET['action']) && $_GET['action'] === 'get_image') {
  require_once __DIR__ . '/../../security.php';
  require_once __DIR__ . '/../../config.php';
  
  // Säkerhetsheaders tidigt
  setSecurityHeaders();
  
  // Sökvägar till JSON-filer
  $userIdsFile = __DIR__ . '/../../data/userids.json';
  
  // Funktion för att läsa user IDs från JSON-fil (definieras här för get_image, annars definieras den senare)
  if (!function_exists('readUserIds')) {
    function readUserIds($filePath) {
      if (!file_exists($filePath)) {
        return [];
      }
      
      $content = file_get_contents($filePath);
      if ($content === false) {
        return [];
      }
      
      $userIds = json_decode($content, true);
      return is_array($userIds) ? $userIds : [];
    }
  }
  
  // Hoppa över user_id-kontroll för get_image - den behövs inte här
  $userId = $_GET['user_id'] ?? '';
  if (empty($userId)) {
    http_response_code(404);
    exit;
  }
  
  // Säkerställ att user_id inte innehåller farliga tecken (tillåt UUID-format)
  if (preg_match('/[^a-zA-Z0-9\-_]/', $userId)) {
    http_response_code(400);
    exit;
  }
  
  $userIds = readUserIds($userIdsFile);
  $profileImage = null;
  
  foreach ($userIds as $user) {
    if (($user['user_id'] ?? '') === $userId && isset($user['profile_image'])) {
      $profileImage = $user['profile_image'];
      break;
    }
  }
  
  if (!$profileImage) {
    // Debug: logga om användaren inte hittades eller saknar profilbild
    error_log("get_image: Användare $userId hittades inte eller saknar profile_image. Antal användare i filen: " . count($userIds));
    // Logga alla user_id för debugging
    $userIdsList = array_map(function($u) { return $u['user_id'] ?? 'SAKNAR_ID'; }, $userIds);
    error_log("get_image: User IDs i filen: " . implode(', ', array_slice($userIdsList, 0, 10)));
    http_response_code(404);
    exit;
  }
  
  // Säkerställ att filnamnet är säkert
  $profileImage = basename($profileImage);
  if (!preg_match('/^user_[a-zA-Z0-9\-_]+\.jpg$/', $profileImage)) {
    error_log("get_image: Ogiltigt filnamn: $profileImage");
    http_response_code(400);
    exit;
  }
  
  // Använd samma sökväg som vid uppladdning
  $imagePath = __DIR__ . '/../../data/' . $profileImage;
  
  // Debug: logga sökväg för felsökning
  error_log("get_image: Söker efter bild: $imagePath, user_id: $userId, profile_image: $profileImage, __DIR__: " . __DIR__);
  
  if (!file_exists($imagePath) || !is_file($imagePath)) {
    // Debug: logga om filen inte finns
    error_log("get_image: Bildfilen finns inte: $imagePath");
    // Lista filer i data-mappen för debugging
    $dataDir = __DIR__ . '/../../data';
    if (is_dir($dataDir)) {
      $files = scandir($dataDir);
      error_log("get_image: Filer i data-mappen: " . implode(', ', array_filter($files, function($f) { return $f !== '.' && $f !== '..'; })));
    }
    http_response_code(404);
    exit;
  }
  
  // Kontrollera att filen faktiskt är läsbar
  if (!is_readable($imagePath)) {
    error_log("get_image: Bildfilen är inte läsbar: $imagePath");
    http_response_code(403);
    exit;
  }
  
  header('Content-Type: image/jpeg');
  header('Cache-Control: public, max-age=3600');
  header('Content-Length: ' . filesize($imagePath));
  readfile($imagePath);
  exit;
}

// För alla andra actions, kräv admin-rättigheter
require_once __DIR__ . '/../admin_guard.php';
require_once __DIR__ . '/../../security.php';
require_once __DIR__ . '/../../config.php';
require_once __DIR__ . '/../../lib/uuid_functions.php';
require_once __DIR__ . '/../../lib/participant_helper.php';

// Säkerhetsheaders tidigt
setSecurityHeaders();

// Sökvägar till JSON-filer
$userIdsFile = __DIR__ . '/../../data/userids.json';
$projectsFile = __DIR__ . '/../../data/projects.json';

// Funktion för att läsa user IDs från JSON-fil
function readUserIds($filePath) {
  if (!file_exists($filePath)) {
    return [];
  }
  
  $content = file_get_contents($filePath);
  if ($content === false) {
    return [];
  }
  
  $userIds = json_decode($content, true);
  return is_array($userIds) ? $userIds : [];
}

// Hämta user_id från GET, POST eller cookie (för andra actions)
if (session_status() === PHP_SESSION_NONE) {
    @session_start();
}

$rawUserId = '';
if (isset($_GET['user_id']) && $_GET['user_id'] !== '') {
    $rawUserId = $_GET['user_id'];
} elseif (isset($_POST['userId']) && $_POST['userId'] !== '') {
    $rawUserId = $_POST['userId'];
} elseif (isset($_COOKIE['user_id']) && $_COOKIE['user_id'] !== '') {
    $rawUserId = $_COOKIE['user_id'];
} else {
    $rawUserId = '';
}
$userId = sanitizeInput($rawUserId, 'user_id');

// Kontrollera att user_id finns (för actions som kräver det)
if (empty($userId)) {
    http_response_code(403);
    echo json_encode([
        'success' => false,
        'message' => 'User ID krävs'
    ]);
    exit;
}

// Svara alltid JSON för andra actions
header('Content-Type: application/json; charset=utf-8');

// Funktion för att spara user IDs till JSON-fil
function saveUserIds($filePath, $userIds) {
  $json = json_encode($userIds, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
  return file_put_contents($filePath, $json) !== false;
}

// Funktion för att läsa projekt
function readProjects($filePath) {
  if (!file_exists($filePath)) {
    return [];
  }
  $content = file_get_contents($filePath);
  if ($content === false) {
    return [];
  }
  $projects = json_decode($content, true);
  return is_array($projects) ? $projects : [];
}

// Funktion för att spara projekt
function saveProjects($filePath, $projects) {
  $json = json_encode($projects, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
  return file_put_contents($filePath, $json) !== false;
}

// Funktion för att ta bort en participant från alla projekt
function removeParticipantFromAllProjects($participantId, $projectsFile) {
  // Kontrollera att participant INTE är admin
  if (isParticipantAdmin($participantId)) {
    return []; // Admins tas aldrig bort
  }
  
  $projects = readProjects($projectsFile);
  $removedFromProjects = [];
  
  foreach ($projects as &$project) {
    $participants = $project['participants'] ?? [];
    $originalCount = count($participants);
    
    // Filtrera bort deltagare med matchande participant_id
    $participants = array_filter($participants, function($p) use ($participantId) {
      $pid = $p['participant_id'] ?? '';
      return $pid !== $participantId;
    });
    
    // Konvertera tillbaka till indexerad array
    $participants = array_values($participants);
    
    // Om någon deltagare togs bort, spara projektet
    if (count($participants) < $originalCount) {
      $project['participants'] = $participants;
      $removedFromProjects[] = $project['id'] ?? '';
    }
  }
  
  if (!empty($removedFromProjects)) {
    saveProjects($projectsFile, $projects);
  }
  
  return $removedFromProjects;
}

// Funktion för att skala ner bild till max 640x640 pixels med bibehållen aspect ratio
function resizeImageToMax($sourcePath, $destPath, $maxWidth = 640, $maxHeight = 640) {
  if (!function_exists('gd_info')) {
    return false;
  }
  
  $info = @getimagesize($sourcePath);
  if ($info === false) {
    return false;
  }
  
  $mime = $info['mime'] ?? '';
  $width = $info[0];
  $height = $info[1];
  
  // Ladda bild baserat på MIME-typ
  switch ($mime) {
    case 'image/jpeg':
    case 'image/jpg':
    case 'image/pjpeg':
      $src = @imagecreatefromjpeg($sourcePath);
      break;
    case 'image/png':
      $src = @imagecreatefrompng($sourcePath);
      break;
    case 'image/webp':
      if (function_exists('imagecreatefromwebp')) {
        $src = @imagecreatefromwebp($sourcePath);
      } else {
        $src = false;
      }
      break;
    default:
      $src = false;
  }
  
  if (!$src) {
    return false;
  }
  
  // Beräkna ny storlek med bibehållen aspect ratio
  if ($width <= $maxWidth && $height <= $maxHeight) {
    // Bilden är redan mindre än max, kopiera direkt
    $targetWidth = $width;
    $targetHeight = $height;
  } else {
    $ratio = min($maxWidth / $width, $maxHeight / $height);
    $targetWidth = (int)round($width * $ratio);
    $targetHeight = (int)round($height * $ratio);
  }
  
  // Skapa ny bild
  $dst = imagecreatetruecolor($targetWidth, $targetHeight);
  
  // För PNG med alfa - fyll vit bakgrund
  if ($mime === 'image/png') {
    $white = imagecolorallocate($dst, 255, 255, 255);
    imagefill($dst, 0, 0, $white);
  }
  
  // Skala om bilden
  imagecopyresampled($dst, $src, 0, 0, 0, 0, $targetWidth, $targetHeight, $width, $height);
  
  // Spara som JPEG med kvalitet 85
  $ok = imagejpeg($dst, $destPath, 85);
  
  imagedestroy($dst);
  imagedestroy($src);
  
  return $ok;
}

// Hämta action från URL
$action = $_GET['action'] ?? '';

try {
  switch ($action) {
    case 'list':
      $allUserIds = readUserIds($userIdsFile);
      // Filtrera bort externa användare (user_type === 'extern')
      $userIds = [];
      foreach ($allUserIds as $user) {
        $userType = $user['user_type'] ?? 'user';
        if ($userType !== 'extern') {
          $userIds[] = $user;
        }
      }
      echo json_encode([
        'success' => true,
        'users' => $userIds
      ]);
      break;
      
    case 'add':
      $userId = trim($_POST['userId'] ?? '');
      $fullName = trim($_POST['fullName'] ?? '');
      $mobile = trim($_POST['mobile'] ?? '');
      // Prioritera userType om det finns, annars använd admin-flagga för bakåtkompatibilitet
      $userType = trim($_POST['userType'] ?? '');
      if (empty($userType)) {
        $admin = isset($_POST['admin']) && $_POST['admin'] === '1';
        $userType = $admin ? 'admin' : 'user';
      }
      // Validera user_type
      if (!in_array($userType, ['user', 'admin', 'extern'])) {
        $userType = 'user';
      }
      $pushoverUserKey = trim($_POST['pushoverUserKey'] ?? '');
      $nickname = trim($_POST['nickname'] ?? '');
      
      if (empty($userId) || empty($fullName)) {
        throw new Exception('User ID och fullständigt namn krävs');
      }
      
      // Validera nickname unikhet om det är satt
      if (!empty($nickname)) {
        require_once __DIR__ . '/../../lib/participant_helper.php';
        if (!isNicknameUnique($nickname)) {
          throw new Exception('Detta nickname används redan av en annan användare');
        }
      }
      
      $userIds = readUserIds($userIdsFile);
      
      // Kontrollera om user ID redan finns
      foreach ($userIds as $user) {
        if ($user['user_id'] === $userId) {
          throw new Exception('Ett användar-ID med detta värde finns redan');
        }
      }
      
      $newUser = [
        'user_id' => $userId,
        'full_name' => $fullName,
        'mobile' => $mobile,
        'user_type' => $userType,
        'active' => true,
        'created_at' => date('Y-m-d H:i:s')
      ];
      
      if (!empty($pushoverUserKey)) {
        $newUser['pushover_user_key'] = $pushoverUserKey;
      }
      
      if (!empty($nickname)) {
        $newUser['nickname'] = $nickname;
      }
      
      $userIds[] = $newUser;
      
      if (saveUserIds($userIdsFile, $userIds)) {
        echo json_encode([
          'success' => true,
          'message' => 'Användare skapad',
          'user' => $newUser
        ]);
      } else {
        throw new Exception('Kunde inte spara användare');
      }
      break;
      
    case 'update':
      $userId = $_POST['userId'] ?? '';
      $fullName = trim($_POST['fullName'] ?? '');
      $mobile = trim($_POST['mobile'] ?? '');
      // Prioritera userType om det finns, annars använd admin-flagga för bakåtkompatibilitet
      $userType = trim($_POST['userType'] ?? '');
      if (empty($userType)) {
        $admin = isset($_POST['admin']) && $_POST['admin'] === '1';
        $userType = $admin ? 'admin' : 'user';
      }
      $inactive = isset($_POST['inactive']) && $_POST['inactive'] === '1';
      $pushoverUserKey = trim($_POST['pushoverUserKey'] ?? '');
      $notes = trim($_POST['notes'] ?? '');
      $publicNotes = trim($_POST['publicNotes'] ?? '');
      $nickname = trim($_POST['nickname'] ?? '');
      
      if (empty($userId) || empty($fullName)) {
        throw new Exception('User ID och fullständigt namn krävs');
      }
      
      // Validera nickname unikhet om det är satt
      if (!empty($nickname)) {
        require_once __DIR__ . '/../../lib/participant_helper.php';
        if (!isNicknameUnique($nickname, $userId)) {
          throw new Exception('Detta nickname används redan av en annan användare');
        }
      }
      
      $userIds = readUserIds($userIdsFile);
      $userFound = false;
      $wasActive = false;
      $isNowInactive = false;
      
      foreach ($userIds as &$user) {
        if ($user['user_id'] === $userId) {
          // Kontrollera om användaren var aktiv innan uppdatering
          $wasActive = ($user['active'] ?? true) !== false;
          
          $user['full_name'] = $fullName;
          $user['mobile'] = $mobile;
          
          // Uppdatera user_type baserat på userType-parameter eller admin-flagga
          // Behåll befintlig user_type om användaren är extern (skulle inte kunna ändras här)
          $currentUserType = $user['user_type'] ?? 'user';
          if ($currentUserType !== 'extern') {
            // Validera user_type
            if (in_array($userType, ['user', 'admin', 'extern'])) {
              $user['user_type'] = $userType;
            } else {
              $user['user_type'] = 'user';
            }
          }
          
          // Hantera active-status baserat på inactive-flagga
          if ($inactive) {
            $user['active'] = false;
            $user['inactive'] = true; // Behåll för bakåtkompatibilitet
            $isNowInactive = true;
          } else {
            $user['active'] = true;
            unset($user['inactive']);
            $isNowInactive = false;
          }
          
          // Pushover user key
          if (!empty($pushoverUserKey)) {
            $user['pushover_user_key'] = $pushoverUserKey;
          } else {
            unset($user['pushover_user_key']);
          }
          
          // Notes
          if (!empty($notes)) {
            $user['notes'] = $notes;
          } else {
            unset($user['notes']);
          }
          
          // Public notes
          if (!empty($publicNotes)) {
            $user['public_notes'] = $publicNotes;
          } else {
            unset($user['public_notes']);
          }
          
          // Nickname
          if (!empty($nickname)) {
            $user['nickname'] = $nickname;
          } else {
            unset($user['nickname']);
          }
          
          $user['updated_at'] = date('Y-m-d H:i:s');
          $userFound = true;
          break;
        }
      }
      
      if (!$userFound) {
        throw new Exception('Användare hittades inte');
      }
      
      if (saveUserIds($userIdsFile, $userIds)) {
        // Om användaren görs inaktiv och INTE är admin, ta bort från alla projekt
        $removedFromProjects = [];
        if ($wasActive && $isNowInactive && !isParticipantAdmin($userId)) {
          $removedFromProjects = removeParticipantFromAllProjects($userId, $projectsFile);
        }
        
        // Hitta den uppdaterade användaren för att returnera den
        $updatedUser = null;
        foreach ($userIds as $user) {
          if ($user['user_id'] === $userId) {
            $updatedUser = $user;
            break;
          }
        }
        
        $message = 'Användare uppdaterad';
        if (!empty($removedFromProjects)) {
          $message .= ' och borttagen från ' . count($removedFromProjects) . ' projekt';
        }
        
        echo json_encode([
          'success' => true,
          'message' => $message,
          'user' => $updatedUser
        ]);
      } else {
        throw new Exception('Kunde inte spara ändringar');
      }
      break;
      
    case 'delete':
      $userId = $_POST['userId'] ?? '';
      
      if (empty($userId)) {
        throw new Exception('User ID krävs');
      }
      
      // Kontrollera om användaren INTE är admin
      $isAdmin = isParticipantAdmin($userId);
      
      $userIds = readUserIds($userIdsFile);
      $userFound = false;
      $profileImage = null;
      
      foreach ($userIds as $index => $user) {
        if ($user['user_id'] === $userId) {
          // Spara profilbild innan radering
          $profileImage = $user['profile_image'] ?? null;
          unset($userIds[$index]);
          $userIds = array_values($userIds); // Re-index array
          $userFound = true;
          break;
        }
      }
      
      if (!$userFound) {
        throw new Exception('Användare hittades inte');
      }
      
      if (saveUserIds($userIdsFile, $userIds)) {
        // Ta bort från projekt om INTE admin
        $removedFromProjects = [];
        if (!$isAdmin) {
          $removedFromProjects = removeParticipantFromAllProjects($userId, $projectsFile);
        }
        
        // Ta bort profilbild om den finns
        if ($profileImage) {
          $imagePath = __DIR__ . '/../../data/' . $profileImage;
          if (file_exists($imagePath)) {
            @unlink($imagePath);
          }
        }
        
        $message = 'Användare borttagen';
        if (!empty($removedFromProjects)) {
          $message .= ' och borttagen från ' . count($removedFromProjects) . ' projekt';
        }
        
        echo json_encode([
          'success' => true,
          'message' => $message
        ]);
      } else {
        throw new Exception('Kunde inte spara ändringar');
      }
      break;
      
    case 'generate_uuid':
      $userId = $_POST['userId'] ?? '';
      
      if (empty($userId)) {
        throw new Exception('User ID krävs');
      }
      
      $userIds = readUserIds($userIdsFile);
      $userFound = false;
      
      foreach ($userIds as &$user) {
        if ($user['user_id'] === $userId) {
          $user['uuid'] = generateUUID();
          $user['updated_at'] = date('Y-m-d H:i:s');
          $userFound = true;
          break;
        }
      }
      
      if (!$userFound) {
        throw new Exception('Användare hittades inte');
      }
      
      if (saveUserIds($userIdsFile, $userIds)) {
        echo json_encode([
          'success' => true,
          'message' => 'UUID genererat',
          'uuid' => $user['uuid'],
          'user' => $user
        ]);
      } else {
        throw new Exception('Kunde inte spara UUID');
      }
      break;
      
    case 'get_user':
      $userId = $_GET['user_id'] ?? '';
      
      if (empty($userId)) {
        throw new Exception('User ID krävs');
      }
      
      $userIds = readUserIds($userIdsFile);
      $userFound = false;
      
      foreach ($userIds as $user) {
        // Kontrollera både user_id och uuid
        if ($user['user_id'] === $userId || (isset($user['uuid']) && $user['uuid'] === $userId)) {
          echo json_encode([
            'success' => true,
            'user' => $user
          ]);
          $userFound = true;
          break;
        }
      }
      
      if (!$userFound) {
        throw new Exception('Användare hittades inte');
      }
      break;
      
    case 'upload_image':
      $targetUserId = sanitizeInput($_POST['user_id'] ?? '', 'user_id');
      
      if (empty($targetUserId)) {
        throw new Exception('User ID krävs');
      }
      
      if (!isset($_FILES['image']) || $_FILES['image']['error'] !== UPLOAD_ERR_OK) {
        throw new Exception('Ingen bildfil uppladdad eller uppladdningsfel');
      }
      
      $file = $_FILES['image'];
      
      // Validera MIME-typ
      global $ALLOWED_IMAGE_MIME;
      require_once __DIR__ . '/../../config.php';
      
      $finfo = new finfo(FILEINFO_MIME_TYPE);
      $mime = $finfo->file($file['tmp_name']);
      
      if (!isset($ALLOWED_IMAGE_MIME[$mime])) {
        throw new Exception('Otillåten bildtyp: ' . $mime);
      }
      
      // Validera filstorlek (max 25MB)
      if ($file['size'] > MAX_FILE_BYTES) {
        throw new Exception('Bilden är för stor (max 25MB)');
      }
      
      // Kontrollera att användaren finns
      $userIds = readUserIds($userIdsFile);
      $userFound = false;
      $userIndex = -1;
      
      foreach ($userIds as $index => $user) {
        if ($user['user_id'] === $targetUserId) {
          $userFound = true;
          $userIndex = $index;
          break;
        }
      }
      
      if (!$userFound) {
        throw new Exception('Användare hittades inte');
      }
      
      // Kontrollera om användaren är inaktiv
      $isInactive = $userIds[$userIndex]['active'] ?? true;
      if ($isInactive === false || $isInactive === 0 || $isInactive === '0' || $isInactive === 'false') {
        throw new Exception('Du kan inte ändra profilbild på en inaktiv användare');
      }
      
      // Ta bort gammal bild om den finns
      if (isset($userIds[$userIndex]['profile_image'])) {
        $oldImagePath = __DIR__ . '/../../data/' . $userIds[$userIndex]['profile_image'];
        if (file_exists($oldImagePath)) {
          @unlink($oldImagePath);
        }
      }
      
      // Skapa filnamn
      $imageFilename = 'user_' . $targetUserId . '.jpg';
      $imagePath = __DIR__ . '/../../data/' . $imageFilename;
      
      // Säkerställ att data-mappen finns
      $dataDir = __DIR__ . '/../../data';
      if (!is_dir($dataDir)) {
        @mkdir($dataDir, 0755, true);
      }
      
      // Skala ner och spara bild
      if (!resizeImageToMax($file['tmp_name'], $imagePath, 640, 640)) {
        error_log("upload_image: Kunde inte skala ner bilden för user_id=$targetUserId, imagePath=$imagePath");
        throw new Exception('Kunde inte skala ner bilden');
      }
      
      // Verifiera att bilden faktiskt sparades
      if (!file_exists($imagePath) || !is_file($imagePath)) {
        error_log("upload_image: Bildfilen sparades inte korrekt: $imagePath");
        throw new Exception('Bildfilen kunde inte sparas');
      }
      
      // Uppdatera userids.json
      $userIds[$userIndex]['profile_image'] = $imageFilename;
      
      if (!saveUserIds($userIdsFile, $userIds)) {
        // Om sparandet misslyckas, ta bort bilden
        @unlink($imagePath);
        throw new Exception('Kunde inte spara användardata');
      }
      
      error_log("upload_image: Bild uppladdad för user_id=$targetUserId, filnamn=$imageFilename, sökväg=$imagePath, filstorlek=" . filesize($imagePath));
      
      echo json_encode([
        'success' => true,
        'message' => 'Bild uppladdad',
        'profile_image' => $imageFilename
      ]);
      break;
      
    case 'delete_image':
      $targetUserId = sanitizeInput($_POST['user_id'] ?? '', 'user_id');
      
      if (empty($targetUserId)) {
        throw new Exception('User ID krävs');
      }
      
      $userIds = readUserIds($userIdsFile);
      $userFound = false;
      $userIndex = -1;
      
      foreach ($userIds as $index => &$user) {
        if ($user['user_id'] === $targetUserId) {
          $userFound = true;
          $userIndex = $index;
          break;
        }
      }
      
      if (!$userFound) {
        throw new Exception('Användare hittades inte');
      }
      
      // Kontrollera om användaren är inaktiv
      $isInactive = $userIds[$userIndex]['active'] ?? true;
      if ($isInactive === false || $isInactive === 0 || $isInactive === '0' || $isInactive === 'false') {
        throw new Exception('Du kan inte ta bort profilbild på en inaktiv användare');
      }
      
      // Ta bort bildfil om den finns
      if (isset($userIds[$userIndex]['profile_image'])) {
        $imagePath = __DIR__ . '/../../data/' . $userIds[$userIndex]['profile_image'];
        if (file_exists($imagePath)) {
          @unlink($imagePath);
        }
        unset($userIds[$userIndex]['profile_image']);
      }
      
      if (!saveUserIds($userIdsFile, $userIds)) {
        throw new Exception('Kunde inte spara ändringar');
      }
      
      echo json_encode([
        'success' => true,
        'message' => 'Bild borttagen'
      ]);
      break;
      
    default:
      throw new Exception('Ogiltig action');
  }
  
} catch (Exception $e) {
  http_response_code(400);
  echo json_encode([
    'success' => false,
    'message' => $e->getMessage()
  ]);
}
?>
