<?php
require_once __DIR__ . '/../admin_guard.php';
require_once __DIR__ . '/../../config.php';

// Hjälpfunktion för att normalisera billing config-struktur (hantera både gammal och ny format)
function normalizeBillingConfig($allBillingConfigs) {
    // Om det redan finns en "default" nyckel, returnera som den är
    if (isset($allBillingConfigs['default'])) {
        return $allBillingConfigs;
    }
    
    // Kolla om det finns prisparametrar på root-nivå (gammal format)
    $priceKeys = ['base_monthly_cost', 'cost_per_user', 'cost_per_gb'];
    $hasRootPrices = false;
    foreach ($priceKeys as $key) {
        if (isset($allBillingConfigs[$key])) {
            $hasRootPrices = true;
            break;
        }
    }
    
    // Om det finns root-priser, migrera till ny format
    if ($hasRootPrices) {
        $defaultPrices = [];
        foreach ($priceKeys as $key) {
            if (isset($allBillingConfigs[$key])) {
                $defaultPrices[$key] = $allBillingConfigs[$key];
                unset($allBillingConfigs[$key]);
            }
        }
        $allBillingConfigs['default'] = $defaultPrices;
    }
    
    return $allBillingConfigs;
}

// Hjälpfunktion för att kontrollera om default-värden har något värde skilt från noll
function hasNonZeroDefault($defaultConfig) {
    if (!isset($defaultConfig)) {
        return false;
    }
    $priceKeys = ['base_monthly_cost', 'cost_per_user', 'cost_per_gb'];
    foreach ($priceKeys as $key) {
        if (isset($defaultConfig[$key]) && (float)$defaultConfig[$key] != 0) {
            return true;
        }
    }
    return false;
}

// Hjälpfunktion för att hitta föregående månad med prisparametrar
function findPreviousMonthConfig($allBillingConfigs, $year, $month) {
    $currentYear = $year;
    $currentMonth = $month;
    
    // Börja från föregående månad
    $currentMonth--;
    if ($currentMonth < 1) {
        $currentMonth = 12;
        $currentYear--;
    }
    
    // Sök bakåt max 24 månader
    $maxIterations = 24;
    $iterations = 0;
    
    while ($iterations < $maxIterations) {
        $yearKey = (string)$currentYear;
        $monthKey = (string)$currentMonth;
        
        if (isset($allBillingConfigs[$yearKey][$monthKey])) {
            return $allBillingConfigs[$yearKey][$monthKey];
        }
        
        // Gå till föregående månad
        $currentMonth--;
        if ($currentMonth < 1) {
            $currentMonth = 12;
            $currentYear--;
        }
        
        $iterations++;
    }
    
    return null;
}

// Hjälpfunktion för att hämta prisparametrar för en specifik månad med fallback-logik
function getBillingConfigForMonth($allBillingConfigs, $year, $month, $defaultConfig) {
    $yearKey = (string)$year;
    $monthKey = (string)$month;
    
    // 1. Först kolla om det finns specifika priser för månaden
    if (isset($allBillingConfigs[$yearKey][$monthKey])) {
        return [
            'config' => array_merge($defaultConfig, $allBillingConfigs[$yearKey][$monthKey]),
            'isLocked' => isset($allBillingConfigs[$yearKey][$monthKey]['locked']) && $allBillingConfigs[$yearKey][$monthKey]['locked'] === true
        ];
    }
    
    // 2. Om månaden inte finns, kolla om default finns och har värden skilt från noll
    if (isset($allBillingConfigs['default']) && hasNonZeroDefault($allBillingConfigs['default'])) {
        return [
            'config' => array_merge($defaultConfig, $allBillingConfigs['default']),
            'isLocked' => false
        ];
    }
    
    // 3. Annars använd föregående månad
    $previousConfig = findPreviousMonthConfig($allBillingConfigs, $year, $month);
    if ($previousConfig !== null) {
        return [
            'config' => array_merge($defaultConfig, $previousConfig),
            'isLocked' => isset($previousConfig['locked']) && $previousConfig['locked'] === true
        ];
    }
    
    // 4. Fallback till default även om alla är noll
    if (isset($allBillingConfigs['default'])) {
        return [
            'config' => array_merge($defaultConfig, $allBillingConfigs['default']),
            'isLocked' => false
        ];
    }
    
    // 5. Sista fallback: använd defaultConfig (alla noll)
    return [
        'config' => $defaultConfig,
        'isLocked' => false
    ];
}

// API endpoint för att hämta prisparametrar för en specifik månad
if (isset($_GET['action']) && $_GET['action'] === 'get_prices') {
    header('Content-Type: application/json; charset=utf-8');
    
    $year = isset($_GET['year']) ? (int)$_GET['year'] : (int)date('Y');
    $month = isset($_GET['month']) ? (int)$_GET['month'] : (int)date('m');
    
    // Validera år och månad
    if ($year < 2020 || $year > 2030) {
        $year = (int)date('Y');
    }
    if ($month < 1 || $month > 12) {
        $month = (int)date('m');
    }
    
    $billingConfigFile = __DIR__ . '/../../data/billing_config.json';
    $defaultConfig = [
        'base_monthly_cost' => 0,
        'cost_per_user' => 0,
        'cost_per_gb' => 0
    ];
    
    $prices = $defaultConfig;
    
    if (file_exists($billingConfigFile)) {
        $configContent = file_get_contents($billingConfigFile);
        if ($configContent !== false) {
            $allBillingConfigs = json_decode($configContent, true) ?: [];
            $allBillingConfigs = normalizeBillingConfig($allBillingConfigs);
            
            $result = getBillingConfigForMonth($allBillingConfigs, $year, $month, $defaultConfig);
            $prices = $result['config'];
        }
    }
    
    echo json_encode([
        'success' => true,
        'prices' => $prices
    ], JSON_UNESCAPED_UNICODE);
    exit;
}

// Funktion för att formatera bytes till läsbar storlek
function formatBytes($bytes, $precision = 2) {
    $units = array('B', 'KB', 'MB', 'GB', 'TB');
    
    for ($i = 0; $bytes > 1024 && $i < count($units) - 1; $i++) {
        $bytes /= 1024;
    }
    
    return round($bytes, $precision) . ' ' . $units[$i];
}

// Funktion för att räkna bilder, dokument, videor och storlek i en mapp för en specifik månad
// Returnerar även lista över aktiva användare (user_ids som laddat upp filer)
function getImageStatsForMonth($path, $year, $month) {
    $stats = [
        'image_files' => 0,
        'image_size' => 0,
        'thumbnail_files' => 0,
        'thumbnail_size' => 0,
        'document_files' => 0,
        'document_size' => 0,
        'video_files' => 0,
        'video_size' => 0,
        'total_size' => 0,
        'active_user_ids' => [], // Samla user_ids som laddat upp filer
        'user_details' => [] // Detaljerad information per användare
    ];
    
    if (!is_dir($path)) {
        return $stats;
    }
    
    // Skapa start- och slutdatum för månaden
    $startDate = sprintf('%04d-%02d-01 00:00:00', $year, $month);
    $endDate = date('Y-m-t 23:59:59', strtotime($startDate));
    $startTimestamp = strtotime($startDate);
    $endTimestamp = strtotime($endDate);
    
    $iterator = new RecursiveIteratorIterator(
        new RecursiveDirectoryIterator($path, RecursiveDirectoryIterator::SKIP_DOTS)
    );
    
    $imageExtensions = ['jpg', 'jpeg', 'png', 'webp', 'heic', 'heif'];
    $documentExtensions = ['pdf', 'txt'];
    $videoExtensions = ['mp4', 'avi', 'mov', 'wmv', 'flv', 'webm', 'mkv', 'm4v'];
    
    foreach ($iterator as $file) {
        if ($file->isFile()) {
            $extension = strtolower($file->getExtension());
            $filename = $file->getFilename();
            
            // Hoppa över metadata-filer
            if ($extension === 'json' || $extension === 'finfo') {
                continue;
            }
            
            // Hoppa över thumbnails (de räknas separat)
            if (strpos($filename, '-thumb.') !== false) {
                continue;
            }
            
            // Hitta metadata-fil (.json eller .finfo)
            $basePath = preg_replace('/\.[^.]+$/', '', $file->getPathname());
            $metaPath = $basePath . '.json';
            if (!file_exists($metaPath)) {
                $metaPath = $basePath . '.finfo';
            }
            
            // Kontrollera om filen är från rätt månad
            $fileInMonth = false;
            $userId = null;
            if (file_exists($metaPath)) {
                $metadata = json_decode(file_get_contents($metaPath), true);
                if ($metadata && isset($metadata['stored_at'])) {
                    $storedTimestamp = strtotime($metadata['stored_at']);
                    if ($storedTimestamp >= $startTimestamp && $storedTimestamp <= $endTimestamp) {
                        $fileInMonth = true;
                        // Hämta user_id från metadata om det finns
                        if (isset($metadata['user_id']) && $metadata['user_id'] !== '') {
                            $userId = $metadata['user_id'];
                        }
                    }
                }
            } else {
                // Om ingen metadata finns, använd filens mtime
                $fileMtime = $file->getMTime();
                if ($fileMtime >= $startTimestamp && $fileMtime <= $endTimestamp) {
                    $fileInMonth = true;
                }
            }
            
            if (!$fileInMonth) {
                continue;
            }
            
            // Lägg till user_id i lista över aktiva användare
            if ($userId !== null) {
                $stats['active_user_ids'][$userId] = true;
                
                // Initiera användardetaljer om de inte finns
                if (!isset($stats['user_details'][$userId])) {
                    $stats['user_details'][$userId] = [
                        'image_files' => 0,
                        'image_size' => 0,
                        'document_files' => 0,
                        'document_size' => 0,
                        'video_files' => 0,
                        'video_size' => 0,
                        'total_size' => 0
                    ];
                }
            }
            
            $fileSize = $file->getSize();
            $stats['total_size'] += $fileSize;
            
            if (in_array($extension, $imageExtensions)) {
                $stats['image_files']++;
                $stats['image_size'] += $fileSize;
                if ($userId !== null) {
                    $stats['user_details'][$userId]['image_files']++;
                    $stats['user_details'][$userId]['image_size'] += $fileSize;
                    $stats['user_details'][$userId]['total_size'] += $fileSize;
                }
            } elseif (in_array($extension, $documentExtensions)) {
                $stats['document_files']++;
                $stats['document_size'] += $fileSize;
                if ($userId !== null) {
                    $stats['user_details'][$userId]['document_files']++;
                    $stats['user_details'][$userId]['document_size'] += $fileSize;
                    $stats['user_details'][$userId]['total_size'] += $fileSize;
                }
            } elseif (in_array($extension, $videoExtensions)) {
                $stats['video_files']++;
                $stats['video_size'] += $fileSize;
                if ($userId !== null) {
                    $stats['user_details'][$userId]['video_files']++;
                    $stats['user_details'][$userId]['video_size'] += $fileSize;
                    $stats['user_details'][$userId]['total_size'] += $fileSize;
                }
            }
            
            // Räkna thumbnail om den finns
            $thumbPath = $basePath . '-thumb.jpg';
            if (file_exists($thumbPath)) {
                $thumbSize = filesize($thumbPath);
                $stats['thumbnail_files']++;
                $stats['thumbnail_size'] += $thumbSize;
                $stats['total_size'] += $thumbSize;
            }
        }
    }
    
    return $stats;
}

// Funktion för att räkna chat-resurser för en specifik månad
// Returnerar även lista över aktiva användare (user_ids som deltagit i chatt)
function getChatStatsForMonth($chatsPath, $year, $month) {
    $stats = [
        'chat_directories' => 0,
        'messages_files' => 0,
        'messages_size' => 0,
        'messages_count' => 0,
        'image_files' => 0,
        'image_size' => 0,
        'document_files' => 0,
        'document_size' => 0,
        'video_files' => 0,
        'video_size' => 0,
        'total_size' => 0,
        'active_user_ids' => [], // Samla user_ids som deltagit i chatt
        'user_details' => [] // Detaljerad information per användare
    ];
    
    if (!is_dir($chatsPath)) {
        return $stats;
    }
    
    // Skapa start- och slutdatum för månaden
    $startDate = sprintf('%04d-%02d-01 00:00:00', $year, $month);
    $endDate = date('Y-m-t 23:59:59', strtotime($startDate));
    $startTimestamp = strtotime($startDate);
    $endTimestamp = strtotime($endDate);
    
    $imageExtensions = ['jpg', 'jpeg', 'png', 'webp', 'heic', 'heif'];
    $documentExtensions = ['pdf', 'txt'];
    $videoExtensions = ['mp4', 'avi', 'mov', 'wmv', 'flv', 'webm', 'mkv', 'm4v'];
    
    $projectDirs = glob($chatsPath . '/*', GLOB_ONLYDIR);
    foreach ($projectDirs as $projectDir) {
        $messagesFile = $projectDir . '/messages.json';
        if (!file_exists($messagesFile)) {
            continue;
        }
        
        // Läs meddelanden och filtrera på månad
        $messages = json_decode(file_get_contents($messagesFile), true);
        if (!is_array($messages)) {
            continue;
        }
        
        $messagesInMonth = [];
        foreach ($messages as $message) {
            if (isset($message['created_at'])) {
                $messageTimestamp = strtotime($message['created_at']);
                if ($messageTimestamp >= $startTimestamp && $messageTimestamp <= $endTimestamp) {
                    $messagesInMonth[] = $message;
                    // Lägg till user_id från meddelandet om det finns
                    // Meddelanden kan ha participant_id eller user_id
                    $userId = $message['user_id'] ?? $message['participant_id'] ?? '';
                    
                    if ($userId !== '') {
                        $stats['active_user_ids'][$userId] = true;
                        
                        // Initiera användardetaljer om de inte finns
                        if (!isset($stats['user_details'][$userId])) {
                            $stats['user_details'][$userId] = [
                                'messages_count' => 0,
                                'image_messages' => 0,
                                'document_messages' => 0,
                                'video_messages' => 0,
                                'text_messages' => 0,
                                'chat_image_size' => 0
                            ];
                        }
                        
                        // Räkna meddelanden per typ
                        $stats['user_details'][$userId]['messages_count']++;
                        $messageType = $message['type'] ?? 'text';
                        if ($messageType === 'image') {
                            $stats['user_details'][$userId]['image_messages']++;
                            // Försök hitta bildfilen och räkna dess storlek
                            if (isset($message['image_url'])) {
                                $imagePath = $projectDir . '/' . basename($message['image_url']);
                                if (file_exists($imagePath)) {
                                    $stats['user_details'][$userId]['chat_image_size'] += filesize($imagePath);
                                }
                            }
                        } elseif ($messageType === 'document') {
                            $stats['user_details'][$userId]['document_messages']++;
                        } elseif ($messageType === 'video') {
                            $stats['user_details'][$userId]['video_messages']++;
                        } else {
                            $stats['user_details'][$userId]['text_messages']++;
                        }
                    }
                }
            }
        }
        
        if (count($messagesInMonth) > 0) {
            $stats['chat_directories']++;
            $stats['messages_count'] += count($messagesInMonth);
            // Beräkna storlek för meddelanden i månaden (ungefärlig)
            $stats['messages_size'] += strlen(json_encode($messagesInMonth));
        }
        
        // Räkna filer i projektmappen som är från månaden
        $files = glob($projectDir . '/*');
        foreach ($files as $file) {
            if (is_file($file)) {
                $filename = basename($file);
                // Hoppa över messages.json
                if ($filename === 'messages.json') {
                    continue;
                }
                
                // Kontrollera om filen är från rätt månad (använd mtime)
                $fileMtime = filemtime($file);
                if ($fileMtime >= $startTimestamp && $fileMtime <= $endTimestamp) {
                    $extension = strtolower(pathinfo($file, PATHINFO_EXTENSION));
                    $fileSize = filesize($file);
                    $stats['total_size'] += $fileSize;
                    
                    if (in_array($extension, $imageExtensions)) {
                        $stats['image_files']++;
                        $stats['image_size'] += $fileSize;
                    } elseif (in_array($extension, $documentExtensions)) {
                        $stats['document_files']++;
                        $stats['document_size'] += $fileSize;
                    } elseif (in_array($extension, $videoExtensions)) {
                        $stats['video_files']++;
                        $stats['video_size'] += $fileSize;
                    }
                }
            }
        }
    }
    
    // Lägg till messages.json-storlek i total storlek
    $stats['total_size'] += $stats['messages_size'];
    
    return $stats;
}

// Funktion för att räkna downloads-resurser för en specifik månad
function getDownloadsStatsForMonth($downloadsPath, $year, $month) {
    $stats = [
        'zip_files' => 0,
        'zip_size' => 0,
        'total_size' => 0
    ];
    
    if (!is_dir($downloadsPath)) {
        return $stats;
    }
    
    // Skapa start- och slutdatum för månaden
    $startDate = sprintf('%04d-%02d-01 00:00:00', $year, $month);
    $endDate = date('Y-m-t 23:59:59', strtotime($startDate));
    $startTimestamp = strtotime($startDate);
    $endTimestamp = strtotime($endDate);
    
    // Räkna ZIP-filer som skapades under månaden
    $zipFiles = glob($downloadsPath . '/*.zip');
    foreach ($zipFiles as $zipFile) {
        if (is_file($zipFile)) {
            // Kontrollera om filen är från rätt månad (använd mtime)
            $fileMtime = filemtime($zipFile);
            if ($fileMtime >= $startTimestamp && $fileMtime <= $endTimestamp) {
                $stats['zip_files']++;
                $fileSize = filesize($zipFile);
                $stats['zip_size'] += $fileSize;
                $stats['total_size'] += $fileSize;
            }
        }
    }
    
    return $stats;
}

// Hämta valt år och månad från GET-parametrar (behövs för att läsa rätt prisparametrar)
$selectedYear = isset($_GET['year']) ? (int)$_GET['year'] : (int)date('Y');
$selectedMonth = isset($_GET['month']) ? (int)$_GET['month'] : (int)date('m');

// Validera år och månad
if ($selectedYear < 2020 || $selectedYear > 2030) {
    $selectedYear = (int)date('Y');
}
if ($selectedMonth < 1 || $selectedMonth > 12) {
    $selectedMonth = (int)date('m');
}

// Läs prisparametrar för vald månad från billing_config.json
$billingConfigFile = __DIR__ . '/../../data/billing_config.json';
$defaultConfig = [
    'base_monthly_cost' => 0,
    'cost_per_user' => 0,
    'cost_per_gb' => 0
];

// Läs alla prisparametrar (struktur: { "default": {...}, "2025": { "1": {...}, "2": {...} } })
$allBillingConfigs = [];
if (file_exists($billingConfigFile)) {
    $configContent = file_get_contents($billingConfigFile);
    if ($configContent !== false) {
        $allBillingConfigs = json_decode($configContent, true) ?: [];
        // Normalisera strukturen (hantera både gammal och ny format)
        $allBillingConfigs = normalizeBillingConfig($allBillingConfigs);
    }
}

// Hämta prisparametrar för vald månad med fallback-logik
$result = getBillingConfigForMonth($allBillingConfigs, $selectedYear, $selectedMonth, $defaultConfig);
$billingConfig = $result['config'];
$isLocked = $result['isLocked'];

// Hantera POST för att spara prisparametrar
$saveMessage = '';
$saveError = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'save_prices') {
    require_once __DIR__ . '/../../security.php';
    
    // Hämta år och månad från POST (eller GET om de inte finns i POST)
    $saveYear = isset($_POST['year']) ? (int)$_POST['year'] : $selectedYear;
    $saveMonth = isset($_POST['month']) ? (int)$_POST['month'] : $selectedMonth;
    
    // Kontrollera om perioden är låst
    $yearKeyCheck = (string)$saveYear;
    $monthKeyCheck = (string)$saveMonth;
    $isPeriodLocked = false;
    if (isset($allBillingConfigs[$yearKeyCheck][$monthKeyCheck]['locked']) && $allBillingConfigs[$yearKeyCheck][$monthKeyCheck]['locked'] === true) {
        $isPeriodLocked = true;
    }
    
    // Validera år och månad
    if ($saveYear < 2020 || $saveYear > 2030) {
        $saveError = 'Ogiltigt år';
    } elseif ($saveMonth < 1 || $saveMonth > 12) {
        $saveError = 'Ogiltig månad';
    } elseif ($isPeriodLocked) {
        $saveError = 'Perioden är låst och kan inte ändras';
    } else {
        // Validera och hämta värden
        $baseCost = filter_var($_POST['base_monthly_cost'] ?? 0, FILTER_VALIDATE_FLOAT);
        $costPerUser = filter_var($_POST['cost_per_user'] ?? 0, FILTER_VALIDATE_FLOAT);
        $costPerGB = filter_var($_POST['cost_per_gb'] ?? 0, FILTER_VALIDATE_FLOAT);
        
        // Validera att värden är numeriska och >= 0
        if ($baseCost === false || $baseCost < 0) {
            $saveError = 'Fast månadskostnad måste vara ett positivt tal eller noll';
        } elseif ($costPerUser === false || $costPerUser < 0) {
            $saveError = 'Kostnad per användare måste vara ett positivt tal eller noll';
        } elseif ($costPerGB === false || $costPerGB < 0) {
            $saveError = 'Kostnad per GB måste vara ett positivt tal eller noll';
        } else {
            // Uppdatera konfiguration för vald månad
            $yearKey = (string)$saveYear;
            $monthKey = (string)$saveMonth;
            
            // Normalisera strukturen innan sparning (hantera både gammal och ny format)
            $allBillingConfigs = normalizeBillingConfig($allBillingConfigs);
            
            // Se till att "default" inte skrivs över (behåll den om den finns)
            if (!isset($allBillingConfigs[$yearKey])) {
                $allBillingConfigs[$yearKey] = [];
            }
            
            // Hämta locked-status från formuläret
            $locked = isset($_POST['locked']) && $_POST['locked'] === 'on';
            
            $allBillingConfigs[$yearKey][$monthKey] = [
                'base_monthly_cost' => round($baseCost, 2),
                'cost_per_user' => round($costPerUser, 2),
                'cost_per_gb' => round($costPerGB, 2),
                'locked' => $locked
            ];
            
            // Uppdatera även den aktuella billingConfig för att visa det sparade värdet direkt
            $billingConfig = $allBillingConfigs[$yearKey][$monthKey];
            
            // Spara till fil (strukturen är nu normaliserad med "default" nyckel)
            $json = json_encode($allBillingConfigs, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
            if (file_put_contents($billingConfigFile, $json) !== false) {
                $saveMessage = 'Prisparametrar sparade!';
            } else {
                $saveError = 'Kunde inte spara prisparametrar. Kontrollera filrättigheter.';
            }
        }
    }
}

// Hämta alla användare från userids.json för att kategorisera aktiva användare
$userIdsFile = __DIR__ . '/../../data/userids.json';
$allUsers = [];
if (file_exists($userIdsFile)) {
    $usersContent = file_get_contents($userIdsFile);
    if ($usersContent !== false) {
        $allUsers = json_decode($usersContent, true) ?: [];
    }
}

// Skapa en lookup-map för användare
$userLookup = [];
foreach ($allUsers as $user) {
    $userId = $user['user_id'] ?? '';
    if ($userId !== '') {
        $userLookup[$userId] = $user;
    }
}

// Beräkna lagring för vald månad (för information)
$uploadsPath = __DIR__ . '/../../uploads';
$chatsPath = __DIR__ . '/../../chats';
$downloadsPath = __DIR__ . '/../../downloads';

$uploadsStatsForMonth = getImageStatsForMonth($uploadsPath, $selectedYear, $selectedMonth);
$chatsStatsForMonth = getChatStatsForMonth($chatsPath, $selectedYear, $selectedMonth);
$downloadsStatsForMonth = getDownloadsStatsForMonth($downloadsPath, $selectedYear, $selectedMonth);

// Beräkna total diskutnyttjande (för fakturering)
function getTotalImageStats($path) {
    $stats = [
        'total_size' => 0
    ];
    
    if (!is_dir($path)) {
        return $stats;
    }
    
    $iterator = new RecursiveIteratorIterator(
        new RecursiveDirectoryIterator($path, RecursiveDirectoryIterator::SKIP_DOTS)
    );
    
    foreach ($iterator as $file) {
        if ($file->isFile()) {
            $extension = strtolower($file->getExtension());
            $filename = $file->getFilename();
            
            // Hoppa över metadata-filer
            if ($extension === 'json' || $extension === 'finfo') {
                continue;
            }
            
            // Hoppa över thumbnails (de räknas separat)
            if (strpos($filename, '-thumb.') !== false) {
                continue;
            }
            
            $stats['total_size'] += $file->getSize();
        }
    }
    
    return $stats;
}

function getTotalChatStats($chatsPath) {
    $stats = [
        'total_size' => 0
    ];
    
    if (!is_dir($chatsPath)) {
        return $stats;
    }
    
    $projectDirs = glob($chatsPath . '/*', GLOB_ONLYDIR);
    foreach ($projectDirs as $projectDir) {
        // Räkna messages.json
        $messagesFile = $projectDir . '/messages.json';
        if (file_exists($messagesFile)) {
            $stats['total_size'] += filesize($messagesFile);
        }
        
        // Räkna alla filer i projektmappen
        $files = glob($projectDir . '/*');
        foreach ($files as $file) {
            if (is_file($file)) {
                $filename = basename($file);
                // Hoppa över messages.json (redan räknat)
                if ($filename === 'messages.json') {
                    continue;
                }
                $stats['total_size'] += filesize($file);
            }
        }
    }
    
    return $stats;
}

function getTotalDownloadsStats($downloadsPath) {
    $stats = [
        'total_size' => 0
    ];
    
    if (!is_dir($downloadsPath)) {
        return $stats;
    }
    
    // Räkna alla ZIP-filer
    $zipFiles = glob($downloadsPath . '/*.zip');
    foreach ($zipFiles as $zipFile) {
        if (is_file($zipFile)) {
            $stats['total_size'] += filesize($zipFile);
        }
    }
    
    return $stats;
}

// Hämta total diskutnyttjande (för fakturering)
$uploadsStatsTotal = getTotalImageStats($uploadsPath);
$chatsStatsTotal = getTotalChatStats($chatsPath);
$downloadsStatsTotal = getTotalDownloadsStats($downloadsPath);

// Samla alla aktiva användare (från uploads och chats) - för vald månad
$activeUserIds = array_merge(
    array_keys($uploadsStatsForMonth['active_user_ids'] ?? []),
    array_keys($chatsStatsForMonth['active_user_ids'] ?? [])
);
$activeUserIds = array_unique($activeUserIds);

// Kategorisera aktiva användare
$internalUsers = [];
$externalUsers = [];

foreach ($activeUserIds as $userId) {
    if (isset($userLookup[$userId])) {
        $user = $userLookup[$userId];
        $userType = $user['user_type'] ?? 'user';
        if ($userType === 'extern') {
            $externalUsers[] = $user;
        } else {
            $internalUsers[] = $user;
        }
    }
}

$userCount = count($activeUserIds);
$internalUserCount = count($internalUsers);
$externalUserCount = count($externalUsers);

// Summera total diskutnyttjande i bytes (för fakturering)
$totalStorageBytes = $uploadsStatsTotal['total_size'] + $chatsStatsTotal['total_size'] + $downloadsStatsTotal['total_size'];

// Konvertera till GB (1 GB = 1024^3 bytes)
$totalStorageGB = $totalStorageBytes / (1024 * 1024 * 1024);

// Summera lagring uppladdad under månaden (för information)
$totalStorageBytesForMonth = $uploadsStatsForMonth['total_size'] + $chatsStatsForMonth['total_size'] + $downloadsStatsForMonth['total_size'];
$totalStorageGBForMonth = $totalStorageBytesForMonth / (1024 * 1024 * 1024);

// Beräkna kostnader (baserat på total diskutnyttjande)
$baseCost = (float)$billingConfig['base_monthly_cost'];
$userCost = $userCount * (float)$billingConfig['cost_per_user'];
$storageCost = $totalStorageGB * (float)$billingConfig['cost_per_gb'];
$subtotal = $baseCost + $userCost + $storageCost;
$vat = $subtotal * 0.25; // Moms 25%
$totalCost = $subtotal + $vat;

// Månadsnamn
$monthNames = [
    1 => 'Januari', 2 => 'Februari', 3 => 'Mars', 4 => 'April',
    5 => 'Maj', 6 => 'Juni', 7 => 'Juli', 8 => 'Augusti',
    9 => 'September', 10 => 'Oktober', 11 => 'November', 12 => 'December'
];
?>
<!DOCTYPE html>
<html lang="sv">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Debitering - ByggArkiv</title>
    <link rel="stylesheet" href="../common.css">
    <style>
        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Oxygen, Ubuntu, Cantarell, sans-serif;
            margin: 0;
            padding: 20px;
            background: #f5f5f5;
        }
        
        .container {
            max-width: 1200px;
            margin: 0 auto;
            background: white;
            border-radius: 8px;
            box-shadow: 0 2px 4px rgba(0,0,0,0.1);
            padding: 30px;
        }
        
        h1 {
            color: #1a73e8;
            margin: 0 0 30px 0;
            font-size: 28px;
            font-weight: 600;
        }
        
        .price-params {
            background: #f8f9fa;
            border: 1px solid #e0e0e0;
            border-radius: 8px;
            padding: 20px;
            margin-bottom: 30px;
        }
        
        .price-params h2 {
            color: #1a73e8;
            margin: 0 0 15px 0;
            font-size: 20px;
            font-weight: 600;
        }
        
        .price-params table {
            width: 100%;
            border-collapse: collapse;
            border: none;
        }
        
        .price-params tr {
            border: none !important;
        }
        
        .price-params td {
            padding: 8px 0;
            border: none !important;
            border-bottom: none !important;
        }
        
        .price-params td:first-child {
            font-weight: 600;
            width: 200px;
        }
        
        .price-params td:last-child {
            text-align: right;
            color: #1a73e8;
            font-weight: 600;
        }
        
        .form-section {
            margin-bottom: 30px;
            padding: 20px;
            background: #f8f9fa;
            border-radius: 8px;
        }
        
        .form-section label {
            display: inline-block;
            margin-right: 15px;
            font-weight: 600;
            color: #333;
        }
        
        .form-section select {
            padding: 8px 12px;
            border: 1px solid #ddd;
            border-radius: 4px;
            font-size: 14px;
            margin-right: 15px;
        }
        
        .form-section button {
            padding: 8px 20px;
            background: #1a73e8;
            color: white;
            border: none;
            border-radius: 4px;
            cursor: pointer;
            font-size: 14px;
            font-weight: 600;
        }
        
        .form-section button:hover {
            background: #1557b0;
        }
        
        .specification {
            margin-top: 30px;
        }
        
        .specification h2 {
            color: #1a73e8;
            margin: 0 0 20px 0;
            font-size: 22px;
            font-weight: 600;
        }
        
        .specification table {
            width: 100%;
            border-collapse: collapse;
            margin-bottom: 20px;
        }
        
        .specification th {
            background: #1a73e8;
            color: white;
            padding: 12px;
            text-align: left;
            font-weight: 600;
        }
        
        .specification td {
            padding: 12px;
            border-bottom: 1px solid #e0e0e0;
        }
        
        .specification tr:hover {
            background: #f8f9fa;
        }
        
        .specification .label {
            font-weight: 600;
            color: #333;
        }
        
        .specification .value {
            text-align: right;
            color: #1a73e8;
            font-weight: 600;
        }
        
        .specification .total-row {
            background: #e3f2fd;
            font-weight: 600;
            font-size: 16px;
        }
        
        .specification .total-row td {
            padding: 15px 12px;
            border-top: 2px solid #1a73e8;
            border-bottom: 2px solid #1a73e8;
        }
        
        .specification .total-row .label {
            color: #1a73e8;
        }
        
        .specification .total-row .value {
            color: #1a73e8;
            font-size: 18px;
        }
        
        .storage-details {
            margin-top: 20px;
            padding: 15px;
            background: #f8f9fa;
            border-radius: 4px;
            font-size: 14px;
        }
        
        .storage-details h3 {
            margin: 0 0 10px 0;
            color: #1a73e8;
            font-size: 16px;
        }
        
        .storage-details table {
            width: 100%;
            border-collapse: collapse;
            margin-top: 10px;
        }
        
        .storage-details td {
            padding: 6px 0;
            border-bottom: 1px solid #e0e0e0;
        }
        
        .storage-details td:first-child {
            font-weight: 600;
        }
        
        .storage-details td:last-child {
            text-align: right;
        }
        
        /* Ta bort spinner-pilarna från number inputs */
        input[type="number"]::-webkit-inner-spin-button,
        input[type="number"]::-webkit-outer-spin-button {
            -webkit-appearance: none;
            margin: 0;
        }
        
        input[type="number"] {
            -moz-appearance: textfield;
            appearance: textfield;
        }
        .header {
            text-align: center;
            margin-bottom: 30px;
            padding-bottom: 20px;
        }
        
        .header .logo {
            max-height: 120px;
            height: auto;
            display: block;
            margin: 0 auto;
        }
        
        .back-link {
            text-align: center;
            margin-top: 15px;
        }
        
        .back-link a {
            color: #1a73e8;
            text-decoration: none;
            font-weight: 500;
            font-size: 14px;
        }
        
        .back-link a:hover {
            text-decoration: underline;
        }
    </style>
</head>
<body>
    <div class="container">
        <header class="header">
            <a href="../../index.html">
                <img src="../../data/logo.png" alt="ByggArkiv Snickare Stefan" class="logo">
            </a>
            <div class="back-link">
                <?php
                    $script = $_SERVER['SCRIPT_NAME'] ?? '';
                    $admPos = strpos($script, '/adm/');
                    $adminUrl = ($admPos !== false ? substr($script, 0, $admPos + 4) : '/adm') . '/admin.php';
                    if (isset($_GET['user_id']) && $_GET['user_id'] !== '') {
                        $adminUrl .= '?user_id=' . urlencode($_GET['user_id']);
                    }
                ?>
                <a href="<?= htmlspecialchars($adminUrl) ?>">← Tillbaka till admin</a>
            </div>
        </header>
        
        <h1>Debitering</h1>
        
        <!-- Formulär för att välja år och månad -->
        <div class="form-section">
            <form method="GET" action="" id="dateForm">
                <input type="hidden" name="user_id" value="<?php echo htmlspecialchars($_GET['user_id'] ?? ''); ?>">
                <label for="year">År:</label>
                <select name="year" id="year">
                    <?php
                    $currentYear = (int)date('Y');
                    for ($year = 2020; $year <= 2030; $year++) {
                        $selected = $year === $selectedYear ? 'selected' : '';
                        echo "<option value=\"$year\" $selected>$year</option>";
                    }
                    ?>
                </select>
                
                <label for="month">Månad:</label>
                <select name="month" id="month">
                    <?php
                    for ($month = 1; $month <= 12; $month++) {
                        $selected = $month === $selectedMonth ? 'selected' : '';
                        $monthName = $monthNames[$month];
                        echo "<option value=\"$month\" $selected>$monthName</option>";
                    }
                    ?>
                </select>
                
                <button type="submit">Visa debitering</button>
                <button type="button" id="showDetailsBtn" style="margin-left: 10px; padding: 8px 20px; background: #28a745; color: white; border: none; border-radius: 4px; cursor: pointer; font-size: 14px; font-weight: 600;">
                    Visa detaljerat underlag
                </button>
            </form>
        </div>
        
        <!-- Prisparametrar -->
        <div class="price-params">
            <h2>Prisparametrar</h2>
            
            <?php if ($saveError): ?>
                <div style="background: #f8d7da; color: #721c24; padding: 12px; border-radius: 4px; margin-bottom: 15px; border: 1px solid #f5c6cb;">
                    <?php echo htmlspecialchars($saveError); ?>
                </div>
            <?php endif; ?>
            
            <form method="POST" action="" style="margin-top: 20px;" id="priceForm">
                <input type="hidden" name="action" value="save_prices">
                <input type="hidden" name="user_id" value="<?php echo htmlspecialchars($_GET['user_id'] ?? ''); ?>">
                <input type="hidden" name="year" id="priceFormYear" value="<?php echo htmlspecialchars($selectedYear); ?>">
                <input type="hidden" name="month" id="priceFormMonth" value="<?php echo htmlspecialchars($selectedMonth); ?>">
                
                <table>
                    <tr>
                        <td>Fast månadskostnad:</td>
                        <td>
                            <input type="number" 
                                   name="base_monthly_cost" 
                                   value="<?php echo htmlspecialchars($billingConfig['base_monthly_cost']); ?>" 
                                   step="0.01" 
                                   min="0" 
                                   style="width: 150px; padding: 6px; border: 1px solid #ddd; border-radius: 4px; text-align: right;"
                                   required>
                            <span style="margin-left: 8px;">SEK</span>
                        </td>
                    </tr>
                    <tr>
                        <td>Kostnad per användare:</td>
                        <td>
                            <input type="number" 
                                   name="cost_per_user" 
                                   value="<?php echo htmlspecialchars($billingConfig['cost_per_user']); ?>" 
                                   step="0.01" 
                                   min="0" 
                                   style="width: 150px; padding: 6px; border: 1px solid #ddd; border-radius: 4px; text-align: right;"
                                   <?php echo $isLocked ? 'disabled' : 'required'; ?>>
                            <span style="margin-left: 8px;">SEK</span>
                        </td>
                    </tr>
                    <tr>
                        <td>Kostnad per GB:</td>
                        <td>
                            <input type="number" 
                                   name="cost_per_gb" 
                                   value="<?php echo htmlspecialchars($billingConfig['cost_per_gb']); ?>" 
                                   step="0.01" 
                                   min="0" 
                                   style="width: 150px; padding: 6px; border: 1px solid #ddd; border-radius: 4px; text-align: right;"
                                   <?php echo $isLocked ? 'disabled' : 'required'; ?>>
                            <span style="margin-left: 8px;">SEK</span>
                        </td>
                    </tr>
                    <tr>
                        <td>Låst period:</td>
                        <td>
                            <input type="checkbox" 
                                   name="locked" 
                                   id="lockedCheckbox"
                                   <?php echo $isLocked ? 'checked' : ''; ?>
                                   style="width: 18px; height: 18px; cursor: pointer; margin-left: calc(150px + 12px - 18px);">
                            <span style="margin-left: 8px; visibility: hidden;">SEK</span>
                        </td>
                    </tr>
                    <tr>
                        <td></td>
                        <td style="text-align: right; padding-top: 15px;">
                            <button type="submit" 
                                    id="saveButton"
                                    style="padding: 10px 25px; <?php echo $isLocked ? 'background: #ccc; cursor: not-allowed;' : 'background: #1a73e8; cursor: pointer;'; ?> color: white; border: none; border-radius: 4px; font-weight: 600;"
                                    <?php echo $isLocked ? 'disabled' : ''; ?>>
                                Spara prisparametrar
                            </button>
                        </td>
                    </tr>
                </table>
            </form>
        </div>
        
        <script>
            // Vänta på att DOM är redo
            document.addEventListener('DOMContentLoaded', function() {
                // Uppdatera prisparametrar när år/månad ändras
                const yearSelect = document.getElementById('year');
                const monthSelect = document.getElementById('month');
                const dateForm = document.getElementById('dateForm');
                
                // När år eller månad ändras, skicka formuläret för att ladda om sidan
                // Detta gör att prisparametrar och beräkningar uppdateras korrekt
                if (yearSelect && dateForm) {
                    yearSelect.addEventListener('change', function() {
                        dateForm.submit();
                    });
                }
                
                if (monthSelect && dateForm) {
                    monthSelect.addEventListener('change', function() {
                        dateForm.submit();
                    });
                }
                
                // Hantera "Låst period" kryssruta
                const lockedCheckbox = document.getElementById('lockedCheckbox');
                const saveButton = document.getElementById('saveButton');
                const priceInputs = document.querySelectorAll('#priceForm input[type="number"]');
                const priceForm = document.getElementById('priceForm');
                
                if (lockedCheckbox) {
                    lockedCheckbox.addEventListener('change', function() {
                        const isLocked = this.checked;
                        
                        // Spara locked-statusen direkt genom att skicka formuläret
                        if (priceForm) {
                            // Skapa ett temporärt formulär för att spara locked-statusen
                            const formData = new FormData(priceForm);
                            formData.set('action', 'save_prices');
                            formData.set('locked', isLocked ? 'on' : '');
                            
                            fetch('', {
                                method: 'POST',
                                body: formData
                            }).then(function(response) {
                                // Ladda om sidan för att uppdatera prislistan
                                window.location.reload();
                            }).catch(function(error) {
                                console.error('Fel vid sparning:', error);
                                // Ladda om sidan ändå
                                window.location.reload();
                            });
                        }
                    });
                }
                
                // Hantera knapp för "Visa detaljerat underlag"
                const showDetailsBtn = document.getElementById('showDetailsBtn');
                if (showDetailsBtn) {
                    showDetailsBtn.addEventListener('click', function(e) {
                        e.preventDefault();
                        const year = document.getElementById('year').value;
                        const month = document.getElementById('month').value;
                        const userId = '<?php echo htmlspecialchars($_GET['user_id'] ?? ''); ?>';
                        let url = 'details.php?year=' + encodeURIComponent(year) + '&month=' + encodeURIComponent(month);
                        if (userId) {
                            url += '&user_id=' + encodeURIComponent(userId);
                        }
                        const newWindow = window.open(url, '_blank');
                        if (!newWindow) {
                            // Om popup blockerades, öppna i samma fönster
                            window.location.href = url;
                        }
                    });
                }
            });
            
            function updatePriceForm() {
                const year = document.getElementById('year').value;
                const month = document.getElementById('month').value;
                const priceFormYear = document.getElementById('priceFormYear');
                const priceFormMonth = document.getElementById('priceFormMonth');
                if (priceFormYear) priceFormYear.value = year;
                if (priceFormMonth) priceFormMonth.value = month;
            }
            
            function loadPricesForMonth() {
                const year = document.getElementById('year').value;
                const month = document.getElementById('month').value;
                const userId = '<?php echo htmlspecialchars($_GET['user_id'] ?? ''); ?>';
                
                // Hämta prisparametrar för vald månad via AJAX
                const url = '?action=get_prices&year=' + encodeURIComponent(year) + '&month=' + encodeURIComponent(month);
                const fullUrl = userId ? url + '&user_id=' + encodeURIComponent(userId) : url;
                
                fetch(fullUrl)
                    .then(response => response.json())
                    .then(data => {
                        if (data.success) {
                            const baseCostInput = document.querySelector('input[name="base_monthly_cost"]');
                            const costPerUserInput = document.querySelector('input[name="cost_per_user"]');
                            const costPerGBInput = document.querySelector('input[name="cost_per_gb"]');
                            
                            if (baseCostInput) baseCostInput.value = data.prices.base_monthly_cost || 0;
                            if (costPerUserInput) costPerUserInput.value = data.prices.cost_per_user || 0;
                            if (costPerGBInput) costPerGBInput.value = data.prices.cost_per_gb || 0;
                        }
                    })
                    .catch(error => {
                        // Tyst hantera fel (prisparametrar kan saknas för månaden)
                    });
            }
        </script>
        
        <!-- Specifikation -->
        <div class="specification">
            <h2>Specifikation för <?php echo $monthNames[$selectedMonth] . ' ' . $selectedYear; ?></h2>
            
            <table>
                <thead>
                    <tr>
                        <th>Post</th>
                        <th style="text-align: right;">Belopp</th>
                    </tr>
                </thead>
                <tbody>
                    <tr>
                        <td class="label">Fast månadskostnad</td>
                        <td class="value"><?php echo number_format($baseCost, 2, ',', ' '); ?> SEK</td>
                    </tr>
                    <tr>
                        <td class="label">Aktiva användare under månaden (<?php echo $userCount; ?> st × <?php echo number_format($billingConfig['cost_per_user'], 2, ',', ' '); ?> SEK)</td>
                        <td class="value"><?php echo number_format($userCost, 2, ',', ' '); ?> SEK</td>
                    </tr>
                    <tr>
                        <td class="label">Total diskutnyttjande (<?php echo number_format($totalStorageGB, 2, ',', ' '); ?> GB × <?php echo number_format($billingConfig['cost_per_gb'], 2, ',', ' '); ?> SEK)</td>
                        <td class="value"><?php echo number_format($storageCost, 2, ',', ' '); ?> SEK</td>
                    </tr>
                    <tr>
                        <td class="label">Summa exkl. moms</td>
                        <td class="value"><?php echo number_format($subtotal, 2, ',', ' '); ?> SEK</td>
                    </tr>
                    <tr>
                        <td class="label">Moms (25%)</td>
                        <td class="value"><?php echo number_format($vat, 2, ',', ' '); ?> SEK</td>
                    </tr>
                    <tr class="total-row">
                        <td class="label">Totalt inkl. moms</td>
                        <td class="value"><?php echo number_format($totalCost, 2, ',', ' '); ?> SEK</td>
                    </tr>
                </tbody>
            </table>
        </div>
    </div>
</body>
</html>

