<?php
require_once __DIR__ . '/../admin_guard.php';
require_once __DIR__ . '/../../config.php';
require_once __DIR__ . '/../../lib/participant_helper.php';

// Funktion för att formatera bytes till läsbar storlek
function formatBytes($bytes, $precision = 2) {
    $units = array('B', 'KB', 'MB', 'GB', 'TB');
    
    for ($i = 0; $bytes > 1024 && $i < count($units) - 1; $i++) {
        $bytes /= 1024;
    }
    
    return round($bytes, $precision) . ' ' . $units[$i];
}

// Funktion för att räkna bilder, dokument, videor och storlek i en mapp för en specifik månad
function getImageStatsForMonth($path, $year, $month) {
    $stats = [
        'image_files' => 0,
        'image_size' => 0,
        'thumbnail_files' => 0,
        'thumbnail_size' => 0,
        'document_files' => 0,
        'document_size' => 0,
        'video_files' => 0,
        'video_size' => 0,
        'total_size' => 0,
        'active_user_ids' => [],
        'user_details' => []
    ];
    
    if (!is_dir($path)) {
        return $stats;
    }
    
    $startDate = sprintf('%04d-%02d-01 00:00:00', $year, $month);
    $endDate = date('Y-m-t 23:59:59', strtotime($startDate));
    $startTimestamp = strtotime($startDate);
    $endTimestamp = strtotime($endDate);
    
    $iterator = new RecursiveIteratorIterator(
        new RecursiveDirectoryIterator($path, RecursiveDirectoryIterator::SKIP_DOTS)
    );
    
    $imageExtensions = ['jpg', 'jpeg', 'png', 'webp', 'heic', 'heif'];
    $documentExtensions = ['pdf', 'txt'];
    $videoExtensions = ['mp4', 'avi', 'mov', 'wmv', 'flv', 'webm', 'mkv', 'm4v'];
    
    foreach ($iterator as $file) {
        if ($file->isFile()) {
            $extension = strtolower($file->getExtension());
            $filename = $file->getFilename();
            
            if ($extension === 'json' || $extension === 'finfo') {
                continue;
            }
            
            if (strpos($filename, '-thumb.') !== false) {
                continue;
            }
            
            $basePath = preg_replace('/\.[^.]+$/', '', $file->getPathname());
            $metaPath = $basePath . '.json';
            if (!file_exists($metaPath)) {
                $metaPath = $basePath . '.finfo';
            }
            
            $fileInMonth = false;
            $userId = null;
            if (file_exists($metaPath)) {
                $metadata = json_decode(file_get_contents($metaPath), true);
                if ($metadata && isset($metadata['stored_at'])) {
                    $storedTimestamp = strtotime($metadata['stored_at']);
                    if ($storedTimestamp >= $startTimestamp && $storedTimestamp <= $endTimestamp) {
                        $fileInMonth = true;
                        if (isset($metadata['user_id']) && $metadata['user_id'] !== '') {
                            $userId = $metadata['user_id'];
                        }
                    }
                }
            } else {
                $fileMtime = $file->getMTime();
                if ($fileMtime >= $startTimestamp && $fileMtime <= $endTimestamp) {
                    $fileInMonth = true;
                }
            }
            
            if (!$fileInMonth) {
                continue;
            }
            
            if ($userId !== null) {
                $stats['active_user_ids'][$userId] = true;
                
                if (!isset($stats['user_details'][$userId])) {
                    $stats['user_details'][$userId] = [
                        'image_files' => 0,
                        'image_size' => 0,
                        'document_files' => 0,
                        'document_size' => 0,
                        'video_files' => 0,
                        'video_size' => 0,
                        'total_size' => 0
                    ];
                }
            }
            
            $fileSize = $file->getSize();
            $stats['total_size'] += $fileSize;
            
            if (in_array($extension, $imageExtensions)) {
                $stats['image_files']++;
                $stats['image_size'] += $fileSize;
                if ($userId !== null) {
                    $stats['user_details'][$userId]['image_files']++;
                    $stats['user_details'][$userId]['image_size'] += $fileSize;
                    $stats['user_details'][$userId]['total_size'] += $fileSize;
                }
            } elseif (in_array($extension, $documentExtensions)) {
                $stats['document_files']++;
                $stats['document_size'] += $fileSize;
                if ($userId !== null) {
                    $stats['user_details'][$userId]['document_files']++;
                    $stats['user_details'][$userId]['document_size'] += $fileSize;
                    $stats['user_details'][$userId]['total_size'] += $fileSize;
                }
            } elseif (in_array($extension, $videoExtensions)) {
                $stats['video_files']++;
                $stats['video_size'] += $fileSize;
                if ($userId !== null) {
                    $stats['user_details'][$userId]['video_files']++;
                    $stats['user_details'][$userId]['video_size'] += $fileSize;
                    $stats['user_details'][$userId]['total_size'] += $fileSize;
                }
            }
            
            $thumbPath = $basePath . '-thumb.jpg';
            if (file_exists($thumbPath)) {
                $thumbSize = filesize($thumbPath);
                $stats['thumbnail_files']++;
                $stats['thumbnail_size'] += $thumbSize;
                $stats['total_size'] += $thumbSize;
            }
        }
    }
    
    return $stats;
}

// Funktion för att räkna chat-resurser för en specifik månad
function getChatStatsForMonth($chatsPath, $year, $month) {
    $stats = [
        'chat_directories' => 0,
        'messages_files' => 0,
        'messages_size' => 0,
        'messages_count' => 0,
        'image_files' => 0,
        'image_size' => 0,
        'document_files' => 0,
        'document_size' => 0,
        'video_files' => 0,
        'video_size' => 0,
        'total_size' => 0,
        'active_user_ids' => [],
        'user_details' => []
    ];
    
    if (!is_dir($chatsPath)) {
        return $stats;
    }
    
    $startDate = sprintf('%04d-%02d-01 00:00:00', $year, $month);
    $endDate = date('Y-m-t 23:59:59', strtotime($startDate));
    $startTimestamp = strtotime($startDate);
    $endTimestamp = strtotime($endDate);
    
    $imageExtensions = ['jpg', 'jpeg', 'png', 'webp', 'heic', 'heif'];
    $documentExtensions = ['pdf', 'txt'];
    $videoExtensions = ['mp4', 'avi', 'mov', 'wmv', 'flv', 'webm', 'mkv', 'm4v'];
    
    $projectDirs = glob($chatsPath . '/*', GLOB_ONLYDIR);
    foreach ($projectDirs as $projectDir) {
        $messagesFile = $projectDir . '/messages.json';
        if (!file_exists($messagesFile)) {
            continue;
        }
        
        $messages = json_decode(file_get_contents($messagesFile), true);
        if (!is_array($messages)) {
            continue;
        }
        
        $messagesInMonth = [];
        foreach ($messages as $message) {
            if (!isset($message['created_at'])) {
                continue;
            }

            $messageTimestamp = strtotime($message['created_at']);
            if ($messageTimestamp < $startTimestamp || $messageTimestamp > $endTimestamp) {
                continue;
            }

            // Hoppa över systemgenererade meddelanden helt från statistik
            $messageType = $message['type'] ?? 'text';
            if ($messageType === 'system') {
                continue;
            }

            $messagesInMonth[] = $message;

            // Hämta user_id från meddelandet
            $userId = $message['user_id'] ?? '';
            $participantId = $message['participant_id'] ?? '';
            
            // Om det bara finns participant_id, använd den som user_id
            // (för interna användare är participant_id samma som user_id)
            if (empty($userId) && !empty($participantId)) {
                $userId = $participantId;
            }
            
            if ($userId === '') {
                continue;
            }

            $stats['active_user_ids'][$userId] = true;
            
            if (!isset($stats['user_details'][$userId])) {
                $stats['user_details'][$userId] = [
                    'messages_count' => 0,
                    'image_messages' => 0,
                    'document_messages' => 0,
                    'video_messages' => 0,
                    'text_messages' => 0,
                    'chat_image_size' => 0,
                    'chat_document_size' => 0,
                    'chat_video_size' => 0
                ];
            }
            
            $stats['user_details'][$userId]['messages_count']++;

            // För media-meddelanden, bygg en filsökväg baserat på media_file (primärt) eller image_url (fallback)
            $mediaPath = null;
            if (!empty($message['media_file'] ?? '')) {
                $mediaPath = $projectDir . '/' . basename($message['media_file']);
            } elseif (!empty($message['image_url'] ?? '')) {
                $mediaPath = $projectDir . '/' . basename($message['image_url']);
            }

            if ($messageType === 'image') {
                $stats['user_details'][$userId]['image_messages']++;
                if ($mediaPath && file_exists($mediaPath)) {
                    $stats['user_details'][$userId]['chat_image_size'] += filesize($mediaPath);
                }
            } elseif ($messageType === 'document') {
                $stats['user_details'][$userId]['document_messages']++;
                if ($mediaPath && file_exists($mediaPath)) {
                    $stats['user_details'][$userId]['chat_document_size'] += filesize($mediaPath);
                }
            } elseif ($messageType === 'video') {
                $stats['user_details'][$userId]['video_messages']++;
                if ($mediaPath && file_exists($mediaPath)) {
                    $stats['user_details'][$userId]['chat_video_size'] += filesize($mediaPath);
                }
            } else {
                $stats['user_details'][$userId]['text_messages']++;
            }
        }
        
        if (count($messagesInMonth) > 0) {
            $stats['chat_directories']++;
            $stats['messages_count'] += count($messagesInMonth);
            $stats['messages_size'] += strlen(json_encode($messagesInMonth));
        }
        
        $files = glob($projectDir . '/*');
        foreach ($files as $file) {
            if (is_file($file)) {
                $filename = basename($file);
                if ($filename === 'messages.json') {
                    continue;
                }
                
                $fileMtime = filemtime($file);
                if ($fileMtime >= $startTimestamp && $fileMtime <= $endTimestamp) {
                    $extension = strtolower(pathinfo($file, PATHINFO_EXTENSION));
                    $fileSize = filesize($file);
                    $stats['total_size'] += $fileSize;
                    
                    if (in_array($extension, $imageExtensions)) {
                        $stats['image_files']++;
                        $stats['image_size'] += $fileSize;
                    } elseif (in_array($extension, $documentExtensions)) {
                        $stats['document_files']++;
                        $stats['document_size'] += $fileSize;
                    } elseif (in_array($extension, $videoExtensions)) {
                        $stats['video_files']++;
                        $stats['video_size'] += $fileSize;
                    }
                }
            }
        }
    }
    
    $stats['total_size'] += $stats['messages_size'];
    
    return $stats;
}

// Funktion för att räkna downloads-resurser för en specifik månad
function getDownloadsStatsForMonth($downloadsPath, $year, $month) {
    $stats = [
        'zip_files' => 0,
        'zip_size' => 0,
        'total_size' => 0
    ];
    
    if (!is_dir($downloadsPath)) {
        return $stats;
    }
    
    // Skapa start- och slutdatum för månaden
    $startDate = sprintf('%04d-%02d-01 00:00:00', $year, $month);
    $endDate = date('Y-m-t 23:59:59', strtotime($startDate));
    $startTimestamp = strtotime($startDate);
    $endTimestamp = strtotime($endDate);
    
    // Räkna ZIP-filer som skapades under månaden
    $zipFiles = glob($downloadsPath . '/*.zip');
    foreach ($zipFiles as $zipFile) {
        if (is_file($zipFile)) {
            // Kontrollera om filen är från rätt månad (använd mtime)
            $fileMtime = filemtime($zipFile);
            if ($fileMtime >= $startTimestamp && $fileMtime <= $endTimestamp) {
                $stats['zip_files']++;
                $fileSize = filesize($zipFile);
                $stats['zip_size'] += $fileSize;
                $stats['total_size'] += $fileSize;
            }
        }
    }
    
    return $stats;
}

// Funktion för att räkna notifieringar (t.ex. Pushover) för en specifik månad
// Bygger på JSON-loggar i logs/notifications.log där varje rad innehåller:
// timestamp, project_id, message_id, notified_users (array), message_type, provider
function getNotificationStatsForMonth($logPath, $year, $month) {
    $stats = [
        'total_notifications' => 0, // antal notifieringar per användare
    ];
    
    if (!is_string($logPath) || $logPath === '' || !file_exists($logPath) || !is_readable($logPath)) {
        return $stats;
    }
    
    // Skapa start- och slutdatum för månaden
    $startDate = sprintf('%04d-%02d-01 00:00:00', $year, $month);
    $endDate = date('Y-m-t 23:59:59', strtotime($startDate));
    $startTimestamp = strtotime($startDate);
    $endTimestamp = strtotime($endDate);
    
    $handle = @fopen($logPath, 'r');
    if ($handle === false) {
        return $stats;
    }
    
    while (($line = fgets($handle)) !== false) {
        $line = trim($line);
        if ($line === '') {
            continue;
        }
        
        $entry = json_decode($line, true);
        if (!is_array($entry)) {
            continue;
        }
        
        $timestampStr = $entry['timestamp'] ?? null;
        if (!$timestampStr) {
            continue;
        }
        $ts = strtotime($timestampStr);
        if ($ts === false || $ts < $startTimestamp || $ts > $endTimestamp) {
            continue;
        }
        
        // Om provider är 'none' antar vi att inga riktiga notifieringar skickades
        $provider = $entry['provider'] ?? null;
        if ($provider === 'none') {
            continue;
        }
        
        $notifiedUsers = $entry['notified_users'] ?? [];
        if (is_array($notifiedUsers)) {
            $stats['total_notifications'] += count($notifiedUsers);
        } elseif (!empty($notifiedUsers)) {
            // Fallback om logged struktur inte är en array men ändå innehåller något
            $stats['total_notifications']++;
        }
    }
    
    fclose($handle);
    
    return $stats;
}

// Hämta valt år och månad
$selectedYear = isset($_GET['year']) ? (int)$_GET['year'] : (int)date('Y');
$selectedMonth = isset($_GET['month']) ? (int)$_GET['month'] : (int)date('m');

if ($selectedYear < 2020 || $selectedYear > 2030) {
    $selectedYear = (int)date('Y');
}
if ($selectedMonth < 1 || $selectedMonth > 12) {
    $selectedMonth = (int)date('m');
}

// Hjälpfunktion för att normalisera billing config-struktur (hantera både gammal och ny format)
function normalizeBillingConfig($allBillingConfigs) {
    // Om det redan finns en "default" nyckel, returnera som den är
    if (isset($allBillingConfigs['default'])) {
        return $allBillingConfigs;
    }
    
    // Kolla om det finns prisparametrar på root-nivå (gammal format)
    $priceKeys = ['base_monthly_cost', 'cost_per_user', 'cost_per_gb'];
    $hasRootPrices = false;
    foreach ($priceKeys as $key) {
        if (isset($allBillingConfigs[$key])) {
            $hasRootPrices = true;
            break;
        }
    }
    
    // Om det finns root-priser, migrera till ny format
    if ($hasRootPrices) {
        $defaultPrices = [];
        foreach ($priceKeys as $key) {
            if (isset($allBillingConfigs[$key])) {
                $defaultPrices[$key] = $allBillingConfigs[$key];
                unset($allBillingConfigs[$key]);
            }
        }
        $allBillingConfigs['default'] = $defaultPrices;
    }
    
    return $allBillingConfigs;
}

// Hjälpfunktion för att kontrollera om default-värden har något värde skilt från noll
function hasNonZeroDefault($defaultConfig) {
    if (!isset($defaultConfig)) {
        return false;
    }
    $priceKeys = ['base_monthly_cost', 'cost_per_user', 'cost_per_gb'];
    foreach ($priceKeys as $key) {
        if (isset($defaultConfig[$key]) && (float)$defaultConfig[$key] != 0) {
            return true;
        }
    }
    return false;
}

// Hjälpfunktion för att hitta föregående månad med prisparametrar
function findPreviousMonthConfig($allBillingConfigs, $year, $month) {
    $currentYear = $year;
    $currentMonth = $month;
    
    // Börja från föregående månad
    $currentMonth--;
    if ($currentMonth < 1) {
        $currentMonth = 12;
        $currentYear--;
    }
    
    // Sök bakåt max 24 månader
    $maxIterations = 24;
    $iterations = 0;
    
    while ($iterations < $maxIterations) {
        $yearKey = (string)$currentYear;
        $monthKey = (string)$currentMonth;
        
        if (isset($allBillingConfigs[$yearKey][$monthKey])) {
            return $allBillingConfigs[$yearKey][$monthKey];
        }
        
        // Gå till föregående månad
        $currentMonth--;
        if ($currentMonth < 1) {
            $currentMonth = 12;
            $currentYear--;
        }
        
        $iterations++;
    }
    
    return null;
}

// Hjälpfunktion för att hämta prisparametrar för en specifik månad med fallback-logik
function getBillingConfigForMonth($allBillingConfigs, $year, $month, $defaultConfig) {
    $yearKey = (string)$year;
    $monthKey = (string)$month;
    
    // 1. Först kolla om det finns specifika priser för månaden
    if (isset($allBillingConfigs[$yearKey][$monthKey])) {
        return [
            'config' => array_merge($defaultConfig, $allBillingConfigs[$yearKey][$monthKey]),
            'isLocked' => isset($allBillingConfigs[$yearKey][$monthKey]['locked']) && $allBillingConfigs[$yearKey][$monthKey]['locked'] === true
        ];
    }
    
    // 2. Om månaden inte finns, kolla om default finns och har värden skilt från noll
    if (isset($allBillingConfigs['default']) && hasNonZeroDefault($allBillingConfigs['default'])) {
        return [
            'config' => array_merge($defaultConfig, $allBillingConfigs['default']),
            'isLocked' => false
        ];
    }
    
    // 3. Annars använd föregående månad
    $previousConfig = findPreviousMonthConfig($allBillingConfigs, $year, $month);
    if ($previousConfig !== null) {
        return [
            'config' => array_merge($defaultConfig, $previousConfig),
            'isLocked' => isset($previousConfig['locked']) && $previousConfig['locked'] === true
        ];
    }
    
    // 4. Fallback till default även om alla är noll
    if (isset($allBillingConfigs['default'])) {
        return [
            'config' => array_merge($defaultConfig, $allBillingConfigs['default']),
            'isLocked' => false
        ];
    }
    
    // 5. Sista fallback: använd defaultConfig (alla noll)
    return [
        'config' => $defaultConfig,
        'isLocked' => false
    ];
}

// Läs prisparametrar för vald månad
$billingConfigFile = __DIR__ . '/../../data/billing_config.json';
$defaultConfig = [
    'base_monthly_cost' => 0,
    'cost_per_user' => 0,
    'cost_per_gb' => 0
];

$allBillingConfigs = [];
if (file_exists($billingConfigFile)) {
    $configContent = file_get_contents($billingConfigFile);
    if ($configContent !== false) {
        $allBillingConfigs = json_decode($configContent, true) ?: [];
        // Normalisera strukturen (hantera både gammal och ny format)
        $allBillingConfigs = normalizeBillingConfig($allBillingConfigs);
    }
}

// Hämta prisparametrar för vald månad med fallback-logik
$result = getBillingConfigForMonth($allBillingConfigs, $selectedYear, $selectedMonth, $defaultConfig);
$billingConfig = $result['config'];

// Hämta alla användare
$userIdsFile = __DIR__ . '/../../data/userids.json';
$allUsers = [];
if (file_exists($userIdsFile)) {
    $usersContent = file_get_contents($userIdsFile);
    if ($usersContent !== false) {
        $allUsers = json_decode($usersContent, true) ?: [];
    }
}

$userLookup = [];
foreach ($allUsers as $user) {
    $userId = $user['user_id'] ?? '';
    if ($userId !== '') {
        $userLookup[$userId] = $user;
    }
}

// Beräkna statistik för vald månad (för information)
$uploadsPath = __DIR__ . '/../../uploads';
$chatsPath = __DIR__ . '/../../chats';
$downloadsPath = __DIR__ . '/../../downloads';
$notificationsLogPath = __DIR__ . '/../../logs/notifications.log';

$uploadsStatsForMonth = getImageStatsForMonth($uploadsPath, $selectedYear, $selectedMonth);
$chatsStatsForMonth = getChatStatsForMonth($chatsPath, $selectedYear, $selectedMonth);
$downloadsStatsForMonth = getDownloadsStatsForMonth($downloadsPath, $selectedYear, $selectedMonth);
$notificationStatsForMonth = getNotificationStatsForMonth($notificationsLogPath, $selectedYear, $selectedMonth);

// Beräkna total diskutnyttjande (för fakturering)
function getTotalImageStats($path) {
    $stats = [
        'total_size' => 0
    ];
    
    if (!is_dir($path)) {
        return $stats;
    }
    
    $iterator = new RecursiveIteratorIterator(
        new RecursiveDirectoryIterator($path, RecursiveDirectoryIterator::SKIP_DOTS)
    );
    
    foreach ($iterator as $file) {
        if ($file->isFile()) {
            $extension = strtolower($file->getExtension());
            $filename = $file->getFilename();
            
            // Hoppa över metadata-filer
            if ($extension === 'json' || $extension === 'finfo') {
                continue;
            }
            
            // Hoppa över thumbnails (de räknas separat)
            if (strpos($filename, '-thumb.') !== false) {
                continue;
            }
            
            $stats['total_size'] += $file->getSize();
        }
    }
    
    return $stats;
}

function getTotalChatStats($chatsPath) {
    $stats = [
        'total_size' => 0
    ];
    
    if (!is_dir($chatsPath)) {
        return $stats;
    }
    
    $projectDirs = glob($chatsPath . '/*', GLOB_ONLYDIR);
    foreach ($projectDirs as $projectDir) {
        // Räkna messages.json
        $messagesFile = $projectDir . '/messages.json';
        if (file_exists($messagesFile)) {
            $stats['total_size'] += filesize($messagesFile);
        }
        
        // Räkna alla filer i projektmappen
        $files = glob($projectDir . '/*');
        foreach ($files as $file) {
            if (is_file($file)) {
                $filename = basename($file);
                // Hoppa över messages.json (redan räknat)
                if ($filename === 'messages.json') {
                    continue;
                }
                $stats['total_size'] += filesize($file);
            }
        }
    }
    
    return $stats;
}

function getTotalDownloadsStats($downloadsPath) {
    $stats = [
        'total_size' => 0
    ];
    
    if (!is_dir($downloadsPath)) {
        return $stats;
    }
    
    // Räkna alla ZIP-filer
    $zipFiles = glob($downloadsPath . '/*.zip');
    foreach ($zipFiles as $zipFile) {
        if (is_file($zipFile)) {
            $stats['total_size'] += filesize($zipFile);
        }
    }
    
    return $stats;
}

// Hämta total diskutnyttjande (för fakturering)
$uploadsStatsTotal = getTotalImageStats($uploadsPath);
$chatsStatsTotal = getTotalChatStats($chatsPath);
$downloadsStatsTotal = getTotalDownloadsStats($downloadsPath);

// Samla alla aktiva användare (från uploads och chats) - för vald månad
$activeUserIds = array_merge(
    array_keys($uploadsStatsForMonth['active_user_ids'] ?? []),
    array_keys($chatsStatsForMonth['active_user_ids'] ?? [])
);
$activeUserIds = array_unique($activeUserIds);

// Kategorisera aktiva användare
$internalUsers = [];
$externalUsers = [];

foreach ($activeUserIds as $userId) {
    if (isset($userLookup[$userId])) {
        $user = $userLookup[$userId];
        $userType = $user['user_type'] ?? 'user';
        if ($userType === 'extern') {
            $externalUsers[] = $user;
        } else {
            $internalUsers[] = $user;
        }
    }
}

$userCount = count($activeUserIds);
$internalUserCount = count($internalUsers);
$externalUserCount = count($externalUsers);

// Summera total diskutnyttjande i bytes (för fakturering)
$totalStorageBytes = $uploadsStatsTotal['total_size'] + $chatsStatsTotal['total_size'] + $downloadsStatsTotal['total_size'];

// Konvertera till GB (1 GB = 1024^3 bytes)
$totalStorageGB = $totalStorageBytes / (1024 * 1024 * 1024);

// Summera lagring uppladdad under månaden (för information)
$totalStorageBytesForMonth = $uploadsStatsForMonth['total_size'] + $chatsStatsForMonth['total_size'] + $downloadsStatsForMonth['total_size'];
$totalStorageGBForMonth = $totalStorageBytesForMonth / (1024 * 1024 * 1024);

// Beräkna kostnader (baserat på total diskutnyttjande)
$baseCost = (float)$billingConfig['base_monthly_cost'];
$userCost = $userCount * (float)$billingConfig['cost_per_user'];
$storageCost = $totalStorageGB * (float)$billingConfig['cost_per_gb'];
$subtotal = $baseCost + $userCost + $storageCost;
$vat = $subtotal * 0.25; // Moms 25%
$totalCost = $subtotal + $vat;

// Kombinera användardetaljer
$combinedUserDetails = [];

foreach ($uploadsStatsForMonth['user_details'] ?? [] as $userId => $details) {
    if (!isset($combinedUserDetails[$userId])) {
        $userName = getParticipantName($userId);
        if ($userName === $userId) {
            // Om getParticipantName returnerar samma som userId, försök från lookup
            $userName = $userLookup[$userId]['full_name'] ?? $userLookup[$userId]['name'] ?? 'Okänd användare';
        }
        $combinedUserDetails[$userId] = [
            'name' => $userName,
            'user_type' => $userLookup[$userId]['user_type'] ?? getParticipantType($userId) ?? 'user',
            // Chatt-statistik (antal)
            'messages_count' => 0,
            'image_messages' => 0,
            'document_messages' => 0,
            'video_messages' => 0,
            'text_messages' => 0,
            // Bildarkiv: antal
            'uploaded_images' => 0,
            'uploaded_documents' => 0,
            'uploaded_videos' => 0,
            // Lagring
            'total_size' => 0,           // behålls för bakåtkompatibilitet (uploads)
            'chat_image_size' => 0,      // behålls (för bilder i chatt)
            'chat_document_size' => 0,
            'chat_video_size' => 0,
            'chat_total_bytes' => 0,
            'archive_total_bytes' => 0,
            'grand_total_bytes' => 0,
        ];
    }
    $combinedUserDetails[$userId]['uploaded_images'] = $details['image_files'] ?? 0;
    $combinedUserDetails[$userId]['uploaded_documents'] = $details['document_files'] ?? 0;
    $combinedUserDetails[$userId]['uploaded_videos'] = $details['video_files'] ?? 0;

    $archiveBytes = $details['total_size'] ?? 0;
    $combinedUserDetails[$userId]['total_size'] += $archiveBytes;
    $combinedUserDetails[$userId]['archive_total_bytes'] += $archiveBytes;
    // grand_total_bytes fylls färdigt efter att chatt-statistik lagts på
}

foreach ($chatsStatsForMonth['user_details'] ?? [] as $userId => $details) {
    if (!isset($combinedUserDetails[$userId])) {
        $userName = getParticipantName($userId);
        if ($userName === $userId) {
            // Om getParticipantName returnerar samma som userId, försök från lookup
            $userName = $userLookup[$userId]['full_name'] ?? $userLookup[$userId]['name'] ?? 'Okänd användare';
        }
        $combinedUserDetails[$userId] = [
            'name' => $userName,
            'user_type' => $userLookup[$userId]['user_type'] ?? getParticipantType($userId) ?? 'user',
            // Chatt-statistik (antal)
            'messages_count' => 0,
            'image_messages' => 0,
            'document_messages' => 0,
            'video_messages' => 0,
            'text_messages' => 0,
            // Bildarkiv: antal
            'uploaded_images' => 0,
            'uploaded_documents' => 0,
            'uploaded_videos' => 0,
            // Lagring
            'total_size' => 0,
            'chat_image_size' => 0,
            'chat_document_size' => 0,
            'chat_video_size' => 0,
            'chat_total_bytes' => 0,
            'archive_total_bytes' => 0,
            'grand_total_bytes' => 0,
        ];
    }
    $combinedUserDetails[$userId]['messages_count'] = $details['messages_count'] ?? 0;
    $combinedUserDetails[$userId]['image_messages'] = $details['image_messages'] ?? 0;
    $combinedUserDetails[$userId]['document_messages'] = $details['document_messages'] ?? 0;
    $combinedUserDetails[$userId]['video_messages'] = $details['video_messages'] ?? 0;
    $combinedUserDetails[$userId]['text_messages'] = $details['text_messages'] ?? 0;

    $combinedUserDetails[$userId]['chat_image_size'] = $details['chat_image_size'] ?? 0;
    $combinedUserDetails[$userId]['chat_document_size'] = $details['chat_document_size'] ?? 0;
    $combinedUserDetails[$userId]['chat_video_size'] = $details['chat_video_size'] ?? 0;

    $chatBytes = ($details['chat_image_size'] ?? 0)
               + ($details['chat_document_size'] ?? 0)
               + ($details['chat_video_size'] ?? 0);
    $combinedUserDetails[$userId]['chat_total_bytes'] = $chatBytes;

    // Uppdatera totalsumma lagring per användare
    $archiveBytes = $combinedUserDetails[$userId]['archive_total_bytes'] ?? 0;
    $combinedUserDetails[$userId]['grand_total_bytes'] = $chatBytes + $archiveBytes;
}

uasort($combinedUserDetails, function($a, $b) {
    $typeOrder = ['admin' => 0, 'user' => 1, 'extern' => 2];
    $aType = $typeOrder[$a['user_type']] ?? 3;
    $bType = $typeOrder[$b['user_type']] ?? 3;
    if ($aType !== $bType) {
        return $aType <=> $bType;
    }
    return strcmp($a['name'], $b['name']);
});

$monthNames = [
    1 => 'Januari', 2 => 'Februari', 3 => 'Mars', 4 => 'April',
    5 => 'Maj', 6 => 'Juni', 7 => 'Juli', 8 => 'Augusti',
    9 => 'September', 10 => 'Oktober', 11 => 'November', 12 => 'December'
];
?>
<!DOCTYPE html>
<html lang="sv">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Detaljerat underlag - <?php echo $monthNames[$selectedMonth] . ' ' . $selectedYear; ?> - ByggArkiv</title>
    <style>
        @media print {
            body {
                margin: 0;
                padding: 20px;
            }
            .no-print {
                display: none !important;
            }
            @page {
                margin: 2cm;
            }
        }
        
        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Oxygen, Ubuntu, Cantarell, sans-serif;
            margin: 0;
            padding: 20px;
            background: white;
            color: #333;
        }
        
        .header {
            text-align: center;
            margin-bottom: 30px;
            border-bottom: 2px solid #1a73e8;
            padding-bottom: 20px;
        }
        
        .header h1 {
            color: #1a73e8;
            margin: 0 0 10px 0;
            font-size: 24px;
        }
        
        .header .period {
            color: #666;
            font-size: 16px;
        }
        
        .no-print {
            margin-bottom: 20px;
            text-align: center;
        }
        
        .no-print button {
            padding: 10px 20px;
            background: #1a73e8;
            color: white;
            border: none;
            border-radius: 4px;
            cursor: pointer;
            font-size: 14px;
            font-weight: 600;
        }
        
        .no-print button:hover {
            background: #1557b0;
        }
        
        table {
            width: 100%;
            border-collapse: collapse;
            margin-bottom: 30px;
            page-break-inside: avoid;
        }
        
        thead {
            background: #f8f9fa;
        }
        
        th {
            padding: 12px;
            text-align: left;
            font-weight: 600;
            border-bottom: 2px solid #1a73e8;
            color: #333;
        }
        
        th.text-right {
            text-align: right;
        }
        
        td {
            padding: 10px 12px;
            border-bottom: 1px solid #e0e0e0;
        }
        
        td.text-right {
            text-align: right;
        }
        
        tbody tr:hover {
            background: #f8f9fa;
        }
        
        .section {
            margin-bottom: 40px;
            page-break-inside: avoid;
        }
        
        .section h2 {
            color: #1a73e8;
            margin: 0 0 15px 0;
            font-size: 20px;
            border-bottom: 1px solid #e0e0e0;
            padding-bottom: 10px;
        }
        
        .summary {
            background: #f8f9fa;
            padding: 15px;
            border-radius: 4px;
            margin-bottom: 20px;
        }
        
        .summary table {
            margin-bottom: 0;
        }
        
        .summary td {
            border-bottom: none;
        }
        
        .user-type {
            font-size: 12px;
            color: #666;
            font-weight: normal;
        }
    </style>
</head>
<body>
    <div class="header">
        <h1>Detaljerat underlag för debitering</h1>
        <div class="period"><?php echo $monthNames[$selectedMonth] . ' ' . $selectedYear; ?></div>
    </div>
    
    <div class="no-print">
        <button onclick="window.print()">Skriv ut</button>
        <button onclick="window.close()" style="margin-left: 10px; background: #6c757d;">Stäng</button>
    </div>
    
    <div class="section">
        <h2>Kostnadsberäkning</h2>
        <div style="font-size: 0.85em; color: #666; margin-bottom: 15px;">
            Prisparametrar: Fast månadskostnad <?php echo number_format($billingConfig['base_monthly_cost'], 2, ',', ' '); ?> SEK | 
            Kostnad per användare <?php echo number_format($billingConfig['cost_per_user'], 2, ',', ' '); ?> SEK | 
            Kostnad per GB <?php echo number_format($billingConfig['cost_per_gb'], 2, ',', ' '); ?> SEK
        </div>
        <table>
            <tr>
                <td>Fast månadskostnad</td>
                <td class="text-right"><?php echo number_format($baseCost, 2, ',', ' '); ?> SEK</td>
            </tr>
            <tr>
                <td>Aktiva användare under månaden (<?php echo $userCount; ?> st × <?php echo number_format($billingConfig['cost_per_user'], 2, ',', ' '); ?> SEK)</td>
                <td class="text-right"><?php echo number_format($userCost, 2, ',', ' '); ?> SEK</td>
            </tr>
            <tr>
                <td>Total diskutnyttjande (<?php echo number_format($totalStorageGB, 2, ',', ' '); ?> GB × <?php echo number_format($billingConfig['cost_per_gb'], 2, ',', ' '); ?> SEK)</td>
                <td class="text-right"><?php echo number_format($storageCost, 2, ',', ' '); ?> SEK</td>
            </tr>
            <tr style="background: #f5f5f5;">
                <td style="font-size: 0.9em; color: #666;">Lagring uppladdad under månaden (<?php echo number_format($totalStorageGBForMonth, 2, ',', ' '); ?> GB - endast information)</td>
                <td class="text-right" style="font-size: 0.9em; color: #666;">-</td>
            </tr>
            <tr>
                <td>Summa exkl. moms</td>
                <td class="text-right"><?php echo number_format($subtotal, 2, ',', ' '); ?> SEK</td>
            </tr>
            <tr>
                <td>Moms (25%)</td>
                <td class="text-right"><?php echo number_format($vat, 2, ',', ' '); ?> SEK</td>
            </tr>
            <tr style="font-weight: 600; border-top: 2px solid #1a73e8;">
                <td>Totalt inkl. moms</td>
                <td class="text-right"><?php echo number_format($totalCost, 2, ',', ' '); ?> SEK</td>
            </tr>
        </table>
    </div>
    
    <div class="section">
        <h2>Sammanfattning</h2>
        <div class="summary">
            <table>
                <tr>
                    <td><strong>Totalt antal aktiva användare:</strong></td>
                    <td class="text-right"><?php echo count($combinedUserDetails); ?> st</td>
                </tr>
                <tr>
                    <td><strong>Totalt antal chattmeddelanden:</strong></td>
                    <td class="text-right"><?php echo $chatsStatsForMonth['messages_count']; ?> st</td>
                </tr>
                <tr>
                    <td><strong>Notifieringar:</strong></td>
                    <td class="text-right"><?php echo number_format($notificationStatsForMonth['total_notifications'] ?? 0, 0, ',', ' '); ?> st</td>
                </tr>
                <tr>
                    <td><strong>Total diskutnyttjande (uploads):</strong></td>
                    <td class="text-right"><?php echo formatBytes($uploadsStatsTotal['total_size']); ?></td>
                </tr>
                <tr>
                    <td><strong>Total diskutnyttjande (chats):</strong></td>
                    <td class="text-right"><?php echo formatBytes($chatsStatsTotal['total_size']); ?></td>
                </tr>
                <tr>
                    <td><strong>Total diskutnyttjande (paketeringar):</strong></td>
                    <td class="text-right"><?php echo formatBytes($downloadsStatsTotal['total_size']); ?></td>
                </tr>
                <tr>
                    <td><strong>Total diskutnyttjande:</strong></td>
                    <td class="text-right"><?php echo formatBytes($totalStorageBytes); ?> (<?php echo number_format($totalStorageGB, 2, ',', ' '); ?> GB)</td>
                </tr>
            </table>
        </div>
    </div>
    
    <div class="section">
        <h2>Detaljerad information per användare</h2>
        <table>
            <thead>
                <tr>
                    <th>Användare</th>
                    <th class="text-right">Chattmeddelanden</th>
                    <th class="text-right">Chattbilder</th>
                    <th class="text-right">Chattdokument</th>
                    <th class="text-right">Summa chatt</th>
                    <th class="text-right">Bilder i bildarkiv</th>
                    <th class="text-right">Videor i bildarkiv</th>
                    <th class="text-right">Dokument i bildarkiv</th>
                    <th class="text-right">Summa bildarkiv</th>
                    <th class="text-right">Totalsumma</th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($combinedUserDetails as $userId => $details):
                    $userTypeLabel = $details['user_type'] === 'extern' ? ' (Extern)' : ($details['user_type'] === 'admin' ? ' (Admin)' : '');
                ?>
                <tr>
                    <td>
                        <?php echo htmlspecialchars($details['name']); ?>
                        <span class="user-type"><?php echo htmlspecialchars($userTypeLabel); ?></span>
                    </td>
                    <td class="text-right"><?php echo $details['messages_count']; ?></td>
                    <td class="text-right"><?php echo $details['image_messages']; ?></td>
                    <td class="text-right"><?php echo $details['document_messages']; ?></td>
                    <td class="text-right"><?php echo formatBytes($details['chat_total_bytes'] ?? 0); ?></td>
                    <td class="text-right"><?php echo $details['uploaded_images']; ?></td>
                    <td class="text-right"><?php echo $details['uploaded_videos']; ?></td>
                    <td class="text-right"><?php echo $details['uploaded_documents']; ?></td>
                    <td class="text-right"><?php echo formatBytes($details['archive_total_bytes'] ?? 0); ?></td>
                    <td class="text-right"><?php echo formatBytes($details['grand_total_bytes'] ?? 0); ?></td>
                </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
    </div>
</body>
</html>

