<?php
require_once __DIR__ . '/../admin_guard.php';

// Sökvägar
$projectsFile = __DIR__ . '/../../data/projects.json';
$chatsDir = __DIR__ . '/../../chats';

// Funktion för att läsa projekt
function readProjects($filePath) {
    if (!file_exists($filePath)) {
        return [];
    }
    $content = file_get_contents($filePath);
    if ($content === false) {
        return [];
    }
    $projects = json_decode($content, true);
    return is_array($projects) ? $projects : [];
}

// Funktion för att läsa chatmeddelanden
function readChatMessages($projectId, $chatsDir) {
    $chatFile = $chatsDir . '/' . $projectId . '/messages.json';
    if (!file_exists($chatFile)) {
        return [];
    }
    $content = file_get_contents($chatFile);
    if ($content === false) {
        return [];
    }
    $messages = json_decode($content, true);
    return is_array($messages) ? $messages : [];
}

// Läsa projekt
$projects = readProjects($projectsFile);

// Styr om arkiverade projekt ska visas i listan
$showArchived = isset($_GET['show_archived']) && $_GET['show_archived'] === '1';

// Samla statistik för varje projekt
$chatStats = [];
foreach ($projects as $project) {
    $projectId = $project['id'] ?? '';
    if (empty($projectId)) {
        continue;
    }

    $status = $project['status'] ?? 'active';
    // Filtrera bort arkiverade projekt om inte visning är aktiverad
    if (!$showArchived && $status === 'archived') {
        continue;
    }
    
    $messages = readChatMessages($projectId, $chatsDir);
    $totalMessages = 0;
    $imageCount = 0;
    $hasMedia = false;
    
    foreach ($messages as $message) {
        $messageType = $message['type'] ?? 'text';
        // Räkna inte systemgenererade meddelanden i statistiken
        if ($messageType === 'system') {
            continue;
        }
        $totalMessages++;
        if ($messageType === 'image') {
            $imageCount++;
            $hasMedia = true;
        } elseif ($messageType === 'document' || $messageType === 'video') {
            // Dokument- och videomeddelanden räknas som media för knappens aktivering
            $hasMedia = true;
        }
    }
    
    // Kontrollera om projektet har deltagare
    $participants = $project['participants'] ?? [];
    $hasParticipants = count($participants) > 0;
    
    // Räkna antal deltagare (inklusive admins)
    $participants = $project['participants'] ?? [];
    $participantCount = count($participants);
    
    // Lägg till alla admins i räkningen
    require_once __DIR__ . '/../../lib/participant_helper.php';
    $adminIds = getAllAdminUserIds();
    $adminCount = count($adminIds);
    
    // Total deltagare = participants + admins (men räkna inte admins dubbelt om de redan är i participants)
    $uniqueParticipantIds = [];
    foreach ($participants as $p) {
        $pid = $p['participant_id'] ?? '';
        if (!empty($pid)) {
            $uniqueParticipantIds[] = $pid;
        }
    }
    foreach ($adminIds as $adminId) {
        if (!in_array($adminId, $uniqueParticipantIds)) {
            $uniqueParticipantIds[] = $adminId;
        }
    }
    $totalParticipantCount = count($uniqueParticipantIds);
    
    $chatStats[] = [
        'project_id' => $projectId,
        'project_name' => $project['name'] ?? 'Namnlöst projekt',
        'status' => $status,
        'total_messages' => $totalMessages,
        'image_count' => $imageCount,
        'has_media' => $hasMedia,
        'has_participants' => $hasParticipants,
        'participant_count' => $totalParticipantCount
    ];
}

// Sortera så att arkiverade projekt hamnar sist
usort($chatStats, function($a, $b) {
    // Först sortera efter status: aktiva först, arkiverade sist
    $aArchived = ($a['status'] ?? 'active') === 'archived';
    $bArchived = ($b['status'] ?? 'active') === 'archived';
    
    if ($aArchived && !$bArchived) {
        return 1; // a är arkiverad, b är aktiv - a ska hamna efter b
    }
    if (!$aArchived && $bArchived) {
        return -1; // a är aktiv, b är arkiverad - a ska hamna före b
    }
    
    // Om båda har samma status, sortera alfabetiskt på projektnamn
    return strcmp($a['project_name'], $b['project_name']);
});

// Generera admin URL
$script = $_SERVER['SCRIPT_NAME'] ?? '';
$admPos = strpos($script, '/adm/');
$adminUrl = ($admPos !== false ? substr($script, 0, $admPos + 4) : '/adm') . '/admin.php';
?>
<!DOCTYPE html>
<html lang="sv">
<head>
    <meta charset="utf-8" />
    <meta name="viewport" content="width=device-width,initial-scale=1" />
    <title>Chattöversikt - ByggArkiv Admin</title>
    <link rel="stylesheet" href="../common.css">
    <style>
        .container { max-width: 1000px; }
        .stats-summary {
            background: white;
            border-radius: 12px;
            padding: 20px;
            margin-bottom: 24px;
            box-shadow: 0 2px 8px rgba(0,0,0,0.1);
        }
        .stats-summary h3 {
            margin: 0 0 12px 0;
            color: #1a73e8;
            font-size: 18px;
        }
        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 16px;
        }
        .stat-item {
            text-align: center;
            padding: 12px;
            background: #f8f9fa;
            border-radius: 8px;
        }
        .stat-value {
            font-size: 24px;
            font-weight: 600;
            color: #1a73e8;
            margin-bottom: 4px;
        }
        .stat-label {
            font-size: 14px;
            color: #666;
        }
        .chat-table {
            width: 100%;
            border-collapse: collapse;
            background: white;
            border-radius: 12px;
            overflow: hidden;
            box-shadow: 0 2px 8px rgba(0,0,0,0.1);
        }
        .chat-table thead {
            background: #f8f9fa;
        }
        .chat-table th {
            padding: 16px;
            text-align: left;
            font-weight: 600;
            color: #333;
            border-bottom: 2px solid #e0e0e0;
        }
        .chat-table td {
            padding: 16px;
            border-bottom: 1px solid #e0e0e0;
            color: #333;
        }
        .chat-table tbody tr:hover {
            background: #f8f9fa;
        }
        .chat-table tbody tr:last-child td {
            border-bottom: none;
        }
        .status-badge {
            display: inline-block;
            padding: 4px 12px;
            border-radius: 12px;
            font-size: 12px;
            font-weight: 500;
        }
        .status-badge.active {
            background: #d4edda;
            color: #155724;
        }
        .status-badge.archived {
            background: #f8d7da;
            color: #721c24;
        }
        .project-link {
            color: #1a73e8;
            text-decoration: none;
            font-weight: 500;
        }
        .project-link:hover {
            text-decoration: underline;
        }
        .admin-button.disabled {
            background: #ccc;
            color: #666;
            cursor: not-allowed;
            pointer-events: none;
        }
        .admin-button:hover {
            opacity: 0.9;
        }
        .admin-button[style*="background: #dc3545"]:hover {
            background: #c82333 !important;
        }
        .admin-button[style*="background: #28a745"]:hover {
            background: #218838 !important;
        }
    </style>
</head>
<body>
    <div class="container">
        <header class="header">
            <a href="../../index.html">
                <img src="../../data/logo.png" alt="ByggArkiv Snickare Stefan" class="logo">
            </a>
            <div class="back-link">
                <a href="<?= htmlspecialchars($adminUrl) ?>">← Tillbaka till admin</a>
            </div>
        </header>

        <h1 class="page-title" style="text-align: left; color: #1a73e8;">Chattöversikt</h1>

        <?php
        // Beräkna totalsummor
        $totalProjects = count($chatStats);
        $totalMessages = array_sum(array_column($chatStats, 'total_messages'));
        $totalImages = array_sum(array_column($chatStats, 'image_count'));
        $projectsWithChat = count(array_filter($chatStats, function($stat) {
            return $stat['total_messages'] > 0;
        }));
        ?>

        <div class="stats-summary">
            <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 8px;">
                <h3 style="margin-bottom: 0;">Sammanfattning</h3>
                <a href="project_activity.php"
                   class="project-link"
                   style="font-size: 13px; font-weight: 500;">
                    Visa aktivitets-topplista →
                </a>
            </div>
            <div class="stats-grid">
                <div class="stat-item">
                    <div class="stat-value"><?= htmlspecialchars($totalProjects) ?></div>
                    <div class="stat-label">Totalt projekt</div>
                    <label style="margin-top: 8px; display: inline-flex; align-items: center; gap: 6px; font-size: 13px; color: #555;">
                        <input type="checkbox"
                               id="showArchived"
                               <?= $showArchived ? 'checked' : '' ?>
                               onchange="toggleShowArchived(this)">
                        Visa arkiverade projekt
                    </label>
                </div>
                <div class="stat-item">
                    <div class="stat-value"><?= htmlspecialchars($projectsWithChat) ?></div>
                    <div class="stat-label">Projekt med chatt</div>
                </div>
                <div class="stat-item">
                    <div class="stat-value"><?= htmlspecialchars($totalMessages) ?></div>
                    <div class="stat-label">Totalt meddelanden</div>
                </div>
                <div class="stat-item">
                    <div class="stat-value"><?= htmlspecialchars($totalImages) ?></div>
                    <div class="stat-label">Totalt bilder</div>
                </div>
            </div>
        </div>

        <div class="card">
            <table class="chat-table">
                <thead>
                    <tr>
                        <th>Projekt</th>
                        <th>Status</th>
                        <th style="text-align: right;">Meddelanden</th>
                        <th style="text-align: right;">Bilder</th>
                        <th style="text-align: center;">Deltagare</th>
                        <th style="text-align: center;">Åtgärder</th>
                        <th style="text-align: center;">Åtgärder</th>
                    </tr>
                </thead>
                <tbody>
                    <?php if (empty($chatStats)): ?>
                        <tr>
                            <td colspan="7" style="text-align: center; padding: 40px; color: #666;">
                                Inga projekt hittades
                            </td>
                        </tr>
                    <?php else: ?>
                        <?php foreach ($chatStats as $stat): 
                            // Använd relativ sökväg från adm/chat/ till chat/chat.php
                            // Från adm/chat/list_chats.php går vi upp två nivåer (../../) till roten, sedan ner till chat/chat.php
                            $chatUrl = '../../chat/chat.php?project_id=' . htmlspecialchars(urlencode($stat['project_id'])) . '&user_id=' . htmlspecialchars(urlencode($_SESSION['user_id'] ?? ''));
                            $projectUrl = '../projects/projects.php?user_id=' . htmlspecialchars(urlencode($_SESSION['user_id'] ?? ''));
                            // Chatt finns om det finns meddelanden/bilder ELLER deltagare
                            $hasChat = ($stat['total_messages'] > 0 || $stat['image_count'] > 0 || $stat['has_participants']);
                        ?>
                            <tr>
                                <td>
                                    <a href="<?= htmlspecialchars($projectUrl) ?>" 
                                       class="project-link">
                                        <?= htmlspecialchars($stat['project_name']) ?>
                                    </a>
                                </td>
                                <td>
                                    <span class="status-badge <?= htmlspecialchars($stat['status']) ?>">
                                        <?= htmlspecialchars($stat['status'] === 'active' ? 'Aktiv' : 'Arkiverad') ?>
                                    </span>
                                </td>
                                <td style="text-align: right;">
                                    <?= htmlspecialchars($stat['total_messages']) ?>
                                </td>
                                <td style="text-align: right;">
                                    <?= htmlspecialchars($stat['image_count']) ?>
                                </td>
                                <td style="text-align: center;">
                                    <?= htmlspecialchars($stat['participant_count'] ?? 0) ?>
                                </td>
                                <td style="text-align: center;">
                                    <?php 
                                    $isArchived = ($stat['status'] ?? 'active') === 'archived';
                                    if ($isArchived): ?>
                                        <span class="action-btn btn-edit" 
                                              style="display: inline-flex; align-items: center; justify-content: center; text-decoration: none; padding: 6px; min-width: 36px; height: 36px; border: none; box-sizing: border-box; opacity: 0.5; cursor: not-allowed;"
                                              title="Kan inte öppna chatt för arkiverade projekt">
                                            <svg width="20" height="20" viewBox="0 0 24 24" fill="currentColor">
                                                <path d="M20 2H4c-1.1 0-2 .9-2 2v18l4-4h14c1.1 0 2-.9 2-2V4c0-1.1-.9-2-2-2zm0 14H6l-2 2V4h16v12z"/>
                                            </svg>
                                        </span>
                                    <?php else: ?>
                                        <a href="<?= htmlspecialchars($chatUrl) ?>" 
                                           class="action-btn btn-edit" 
                                           style="display: inline-flex; align-items: center; justify-content: center; text-decoration: none; padding: 6px; min-width: 36px; height: 36px; border: none; box-sizing: border-box;"
                                           title="Gå till chatt">
                                            <svg width="20" height="20" viewBox="0 0 24 24" fill="currentColor">
                                                <path d="M20 2H4c-1.1 0-2 .9-2 2v18l4-4h14c1.1 0 2-.9 2-2V4c0-1.1-.9-2-2-2zm0 14H6l-2 2V4h16v12z"/>
                                            </svg>
                                        </a>
                                    <?php endif; ?>
                                </td>
                                <td style="text-align: center;">
                                    <div style="display: flex; flex-direction: row; gap: 6px; align-items: center; justify-content: center;">
                                        <?php 
                                        $hasMedia = !empty($stat['has_media']);
                                        $mediaBtnStyle = 'display: inline-flex; align-items: center; justify-content: center; text-decoration: none; padding: 6px; min-width: 36px; height: 36px; border: none; box-sizing: border-box;';
                                        if ($hasMedia) {
                                            $mediaBtnStyle .= ' background: #4dabf7; color: white;';
                                        } else {
                                            $mediaBtnStyle .= ' background: #cfd8dc; color: #ffffff; opacity: 0.6; cursor: not-allowed; pointer-events: none;';
                                        }
                                        ?>
                                        <a href="view_project_images.php?project_id=<?= htmlspecialchars(urlencode($stat['project_id'])) ?>&user_id=<?= htmlspecialchars(urlencode($_SESSION['user_id'] ?? '')) ?>" 
                                           class="action-btn btn-edit" 
                                           style="<?= $mediaBtnStyle ?>"
                                           title="Visa bilder och dokument i chatten">
                                            <img src="../icons/move.svg" width="20" height="20" alt="Visa bilder och dokument">
                                        </a>
                                        <a href="../api/chat/download_chat.php?project_id=<?= htmlspecialchars(urlencode($stat['project_id'])) ?>&user_id=<?= htmlspecialchars(urlencode($_SESSION['user_id'] ?? '')) ?>" 
                                           class="action-btn btn-edit" 
                                           style="display: inline-flex; align-items: center; justify-content: center; text-decoration: none; padding: 6px; min-width: 36px; height: 36px; background: #17a2b8; color: white; border: none; box-sizing: border-box;"
                                           title="Ladda ner alla bilder och meddelanden som ZIP">
                                            <svg width="20" height="20" viewBox="0 0 24 24" fill="currentColor">
                                                <path d="M19 9h-4V3H9v6H5l7 7 7-7zM5 18v2h14v-2H5z"/>
                                            </svg>
                                        </a>
                                        <?php 
                                        $clearBtnStyle = 'display: inline-flex; align-items: center; justify-content: center; text-decoration: none; padding: 6px; min-width: 36px; height: 36px; border: none; box-sizing: border-box;';
                                        if ($hasMedia) {
                                            $clearBtnStyle .= ' background: #ffc107; color: #212529;';
                                        } else {
                                            $clearBtnStyle .= ' background: #cfd8dc; color: #212529; opacity: 0.6; cursor: not-allowed; pointer-events: none;';
                                        }
                                        ?>
                                        <button onclick="<?= $hasMedia ? "clearChat('" . htmlspecialchars($stat['project_id']) . "', '" . htmlspecialchars(addslashes($stat['project_name'])) . "')" : "return false;" ?>" 
                                                class="action-btn btn-edit" 
                                                style="<?= $clearBtnStyle ?>"
                                                title="<?= $hasMedia ? 'Rensa alla meddelanden (behåller deltagare)' : 'Inga meddelanden att rensa' ?>">
                                            <svg width="20" height="20" viewBox="0 0 24 24">
                                                <!-- Förenklad dokument-ikon: vitt papper med nedvikt hörn -->
                                                <path fill="#ffffff"
                                                      d="M6 3h9l3 3v13a2 2 0 0 1-2 2H6a2 2 0 0 1-2-2V5a2 2 0 0 1 2-2zm8 1.5V8h3.5L14 4.5z"/>
                                                <!-- Rött kryss ovanpå dokumentet (ännu längre) -->
                                                <line x1="2" y1="2" x2="22" y2="22"
                                                      stroke="#d32f2f" stroke-width="2" stroke-linecap="round" />
                                                <line x1="22" y1="2" x2="2" y2="22"
                                                      stroke="#d32f2f" stroke-width="2" stroke-linecap="round" />
                                            </svg>
                                        </button>
                                        <button onclick="deleteChat('<?= htmlspecialchars($stat['project_id']) ?>', '<?= htmlspecialchars(addslashes($stat['project_name'])) ?>')" 
                                                class="action-btn btn-edit" 
                                                style="display: inline-flex; align-items: center; justify-content: center; text-decoration: none; padding: 6px; min-width: 36px; height: 36px; background: #dc3545; color: white; border: none; box-sizing: border-box;"
                                                title="Ta bort chatt, bilder och dokument">
                                            <svg width="20" height="20" viewBox="0 0 24 24" fill="currentColor">
                                                <path d="M6 19c0 1.1.9 2 2 2h8c1.1 0 2-.9 2-2V7H6v12zM19 4h-3.5l-1-1h-5l-1 1H5v2h14V4z"/>
                                            </svg>
                                        </button>
                                    </div>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </tbody>
            </table>
        </div>
    </div>
    
    <script>
        function toggleShowArchived(checkbox) {
            const url = new URL(window.location.href);
            if (checkbox.checked) {
                url.searchParams.set('show_archived', '1');
            } else {
                url.searchParams.delete('show_archived');
            }
            window.location.href = url.toString();
        }
        
        async function createChat(projectId, projectName, status) {
            // Kontrollera om projektet är arkiverat
            if (status === 'archived') {
                alert('Kan inte skapa chatt för arkiverade projekt');
                return;
            }
            
            try {
                const formData = new FormData();
                formData.append('action', 'create_chat');
                formData.append('project_id', projectId);
                formData.append('user_id', '<?= htmlspecialchars($_SESSION['user_id'] ?? '') ?>');
                
                const res = await fetch('../../api/chat/participants.php', {
                    method: 'POST',
                    body: formData
                });
                
                const data = await res.json();
                
                if (data.success) {
                    // Gå till chatten efter att den skapats
                    const chatUrl = '../../chat/chat.php?project_id=' + encodeURIComponent(projectId) + '&user_id=' + encodeURIComponent('<?= htmlspecialchars($_SESSION['user_id'] ?? '') ?>');
                    window.location.href = chatUrl;
                } else {
                    alert('Fel: ' + data.message);
                }
            } catch (e) {
                console.error('Fel:', e);
                alert('Ett fel uppstod vid skapande av chatt');
            }
        }
        
        async function clearChat(projectId, projectName) {
            if (!confirm(`Är du säker på att du vill rensa alla meddelanden i chatten för projektet "${projectName}"?\n\nDetta kommer att:\n- Ta bort alla meddelanden i chatten\n- Lägga till ett systemmeddelande "Chatten har rensats"\n- Alla deltagare behålls\n\nDetta går inte att ångra!`)) {
                return;
            }
            
            try {
                const formData = new FormData();
                formData.append('action', 'clear_messages');
                formData.append('project_id', projectId);
                formData.append('user_id', '<?= htmlspecialchars($_SESSION['user_id'] ?? '') ?>');
                
                const res = await fetch('../../api/chat/messages.php', {
                    method: 'POST',
                    body: formData
                });
                
                const data = await res.json();
                
                if (data.success) {
                    alert('Meddelanden rensade för projektet "' + projectName + '"');
                    // Ladda om sidan för att uppdatera statistiken
                    window.location.reload();
                } else {
                    alert('Fel: ' + data.message);
                }
            } catch (e) {
                console.error('Fel:', e);
                alert('Ett fel uppstod vid rensning av meddelanden');
            }
        }
        
        async function deleteChat(projectId, projectName) {
            if (!confirm(`Är du säker på att du vill ta bort chatten för projektet "${projectName}"?\n\nDetta kommer att:\n- Ta bort alla meddelanden i chatten\n- Ta bort alla deltagare från projektchatten (utom admins)\n\nDetta går inte att ångra!`)) {
                return;
            }
            
            try {
                const formData = new FormData();
                formData.append('action', 'delete_chat');
                formData.append('project_id', projectId);
                formData.append('user_id', '<?= htmlspecialchars($_SESSION['user_id'] ?? '') ?>');
                
                const res = await fetch('../../api/chat/participants.php', {
                    method: 'POST',
                    body: formData
                });
                
                const data = await res.json();
                
                if (data.success) {
                    alert('Chatt och deltagare borttagna för projektet "' + projectName + '"');
                    // Ladda om sidan för att uppdatera statistiken
                    window.location.reload();
                } else {
                    alert('Fel: ' + data.message);
                }
            } catch (e) {
                console.error('Fel:', e);
                alert('Ett fel uppstod vid borttagning av chatt');
            }
        }
    </script>
</body>
</html>

