<?php
require_once __DIR__ . '/../admin_guard.php';
require_once __DIR__ . '/../../config.php';
require_once __DIR__ . '/../../lib/dal/repositories/ProjectRepository.php';

// Grundsökvägar
$chatsDir = __DIR__ . '/../../chats';
// UPLOAD_ROOT kan vara absolut eller relativ från projektroten
$uploadRoot = is_dir(UPLOAD_ROOT)
    ? rtrim(UPLOAD_ROOT, '/\\')
    : __DIR__ . '/../../' . rtrim(UPLOAD_ROOT, '/\\');

/**
 * Beräkna tidsintervall baserat på period-nyckel.
 *
 * @param string $periodKey
 * @return array [$startTimestamp, $endTimestamp, $label]
 */
function getDateRangeForPeriod(string $periodKey): array
{
    $now = time();

    switch ($periodKey) {
        case '30d':
            $days = 30;
            $label = 'Senaste 30 dagarna';
            break;
        case '7d':
        default:
            $days = 7;
            $label = 'Senaste 7 dagarna';
            $periodKey = '7d';
            break;
    }

    $end = $now;
    $start = strtotime('-' . $days . ' days', $end);

    // Normalisera till hela sekunder
    $start = $start !== false ? $start : $now - ($days * 86400);

    return [$start, $end, $label];
}

/**
 * Hämta chattaktivitet (antal meddelanden och media) per projekt under ett intervall.
 *
 * - Räknar inte systemmeddelanden (type === 'system')
 * - Räknar media i chatten (image/document/video) separat
 *
 * @param string $chatsDir
 * @param int    $startTs
 * @param int    $endTs
 * @return array [project_id => ['messages' => int, 'media_chat' => int]]
 */
function getChatActivityForPeriod(string $chatsDir, int $startTs, int $endTs): array
{
    $result = [];

    if (!is_dir($chatsDir)) {
        return $result;
    }

    $projectDirs = glob($chatsDir . '/*', GLOB_ONLYDIR);
    if ($projectDirs === false) {
        return $result;
    }

    foreach ($projectDirs as $projectDir) {
        $projectId = basename($projectDir);
        $messagesFile = $projectDir . '/messages.json';
        if (!file_exists($messagesFile)) {
            continue;
        }

        $content = file_get_contents($messagesFile);
        if ($content === false) {
            continue;
        }

        $messages = json_decode($content, true);
        if (!is_array($messages)) {
            continue;
        }

        $msgCount = 0;
        $mediaChatCount = 0;

        foreach ($messages as $message) {
            $createdAt = $message['created_at'] ?? null;
            if (empty($createdAt)) {
                continue;
            }
            $ts = strtotime($createdAt);
            if ($ts === false || $ts < $startTs || $ts > $endTs) {
                continue;
            }

            $type = $message['type'] ?? 'text';
            // Hoppa över systemmeddelanden helt
            if ($type === 'system') {
                continue;
            }

            $msgCount++;

            if (in_array($type, ['image', 'document', 'video'], true)) {
                $mediaChatCount++;
            }
        }

        if ($msgCount > 0 || $mediaChatCount > 0) {
            $result[$projectId] = [
                'messages'    => $msgCount,
                'media_chat'  => $mediaChatCount,
            ];
        }
    }

    return $result;
}

/**
 * Hämta media-aktivitet (uploads) per projekt under ett intervall.
 *
 * Bygger på metadata-filerna i uploads/<project_id>/*.json och använder:
 * - stored_at (om finns) som primärt datum
 * - file_type eller filändelse för att avgöra om det är bild/dokument/video
 *
 * @param string $uploadRoot
 * @param int    $startTs
 * @param int    $endTs
 * @return array [project_id => ['media_uploads' => int]]
 */
function getUploadMediaActivityForPeriod(string $uploadRoot, int $startTs, int $endTs): array
{
    $result = [];

    if (!is_dir($uploadRoot)) {
        return $result;
    }

    $projectDirs = glob(rtrim($uploadRoot, '/\\') . '/*', GLOB_ONLYDIR);
    if ($projectDirs === false) {
        return $result;
    }

    // Stödda format för att avgöra filtyp när metadata saknar file_type
    $imageExtensions = ['jpg', 'jpeg', 'png', 'webp', 'heic', 'heif'];
    $videoExtensions = ['mp4', 'webm', 'mov', 'avi'];
    $documentExtensions = ['pdf', 'txt', 'csv'];

    foreach ($projectDirs as $projectDir) {
        $projectId = basename($projectDir);
        $jsonFiles = glob($projectDir . '/*.json');
        if ($jsonFiles === false) {
            continue;
        }

        $mediaUploads = 0;

        foreach ($jsonFiles as $jsonFile) {
            $metaContent = file_get_contents($jsonFile);
            if ($metaContent === false) {
                continue;
            }

            $meta = json_decode($metaContent, true);
            if (!is_array($meta)) {
                continue;
            }

            // Datum för uppladdning
            $storedAt = $meta['stored_at'] ?? null;
            if (empty($storedAt)) {
                // Fallback: använd filens mtime
                $fileMtime = filemtime($jsonFile);
                if ($fileMtime === false) {
                    continue;
                }
                $ts = $fileMtime;
            } else {
                $ts = strtotime($storedAt);
                if ($ts === false) {
                    continue;
                }
            }

            if ($ts < $startTs || $ts > $endTs) {
                continue;
            }

            // Bestäm filtyp
            $fileType = $meta['file_type'] ?? null;

            if ($fileType === null) {
                $storedFile = $meta['stored_file'] ?? basename($jsonFile);
                $ext = strtolower(pathinfo($storedFile, PATHINFO_EXTENSION));

                if (in_array($ext, $imageExtensions, true)) {
                    $fileType = 'image';
                } elseif (in_array($ext, $videoExtensions, true)) {
                    $fileType = 'video';
                } elseif (in_array($ext, $documentExtensions, true)) {
                    $fileType = 'document';
                } else {
                    // Okänd typ, räkna inte in i media-aktivitet
                    continue;
                }
            }

            if (!in_array($fileType, ['image', 'document', 'video'], true)) {
                continue;
            }

            $mediaUploads++;
        }

        if ($mediaUploads > 0) {
            $result[$projectId] = [
                'media_uploads' => $mediaUploads,
            ];
        }
    }

    return $result;
}

/**
 * Kombinera chatt- och media-aktivitet till aktivitetspoäng per projekt.
 *
 * @param array $projectsMap  [project_id => projectData]
 * @param array $chatStats    [project_id => ['messages', 'media_chat']]
 * @param array $uploadStats  [project_id => ['media_uploads']]
 * @param array $weights      ['chat' => int, 'media' => int]
 * @param bool  $includeArchived
 * @return array Lista av projekt med aktivitet, sorterad efter score
 */
function buildActivityRanking(array $projectsMap, array $chatStats, array $uploadStats, array $weights, bool $includeArchived): array
{
    $allProjectIds = array_unique(array_merge(array_keys($chatStats), array_keys($uploadStats)));

    $rows = [];
    $totalMessagesAll = 0;
    $totalMediaAll = 0;

    foreach ($allProjectIds as $projectId) {
        $project = $projectsMap[$projectId] ?? null;
        $status = $project['status'] ?? 'active';

        if (!$includeArchived && $status === 'archived') {
            continue;
        }

        $messages = $chatStats[$projectId]['messages'] ?? 0;
        $mediaChat = $chatStats[$projectId]['media_chat'] ?? 0;
        $mediaUploads = $uploadStats[$projectId]['media_uploads'] ?? 0;
        $mediaTotal = $mediaChat + $mediaUploads;

        // Hoppa över helt inaktiva projekt
        if ($messages === 0 && $mediaTotal === 0) {
            continue;
        }

        $chatPoints = $messages * ($weights['chat'] ?? 1);
        $mediaPoints = $mediaTotal * ($weights['media'] ?? 3);
        $score = $chatPoints + $mediaPoints;

        $totalMessagesAll += $messages;
        $totalMediaAll += $mediaTotal;

        $rows[] = [
            'project_id'     => $projectId,
            'project_name'   => $project['name'] ?? $projectId,
            'status'         => $status,
            'messages'       => $messages,
            'media_chat'     => $mediaChat,
            'media_uploads'  => $mediaUploads,
            'media_total'    => $mediaTotal,
            'score'          => $score,
        ];
    }

    // Sortera efter aktivitetspoäng (flest först), sedan på namn
    usort($rows, function ($a, $b) {
        if ($a['score'] === $b['score']) {
            return strcmp($a['project_name'], $b['project_name']);
        }
        return $b['score'] <=> $a['score'];
    });

    return [
        'rows'           => $rows,
        'total_messages' => $totalMessagesAll,
        'total_media'    => $totalMediaAll,
    ];
}

// ---- Huvudlogik för sidan ----

// Period (7d eller 30d)
$allowedPeriods = ['7d', '30d'];
$periodKey = $_GET['period'] ?? '7d';
if (!in_array($periodKey, $allowedPeriods, true)) {
    $periodKey = '7d';
}

// Visa arkiverade?
$showArchived = isset($_GET['show_archived']) && $_GET['show_archived'] === '1';

// Datumintervall
[$startTs, $endTs, $periodLabel] = getDateRangeForPeriod($periodKey);

// Viktad poängmodell
$weights = [
    'chat'  => 1, // 1 poäng per chattmeddelande
    'media' => 3, // 3 poäng per mediafil (bild/dokument/video)
];

// Hämta projekt via repository (respekterar DATA_STORE_TYPE)
$projectRepo = new ProjectRepository();
$allProjects = $projectRepo->getAll();
$projectsMap = [];
foreach ($allProjects as $project) {
    if (!empty($project['id'])) {
        $projectsMap[$project['id']] = $project;
    }
}

// Räkna chattaktivitet
$chatStats = getChatActivityForPeriod($chatsDir, $startTs, $endTs);

// Räkna mediaaktivitet i uploads
$uploadStats = getUploadMediaActivityForPeriod($uploadRoot, $startTs, $endTs);

// Bygg ranking
$ranking = buildActivityRanking($projectsMap, $chatStats, $uploadStats, $weights, $showArchived);
$rows = $ranking['rows'];
$totalMessagesAll = $ranking['total_messages'];
$totalMediaAll = $ranking['total_media'];

// Summeringsvärden
$totalProjects = count($rows);
$topProject = $rows[0]['project_name'] ?? null;

// Generera admin URL (tillbaka-länk)
$script = $_SERVER['SCRIPT_NAME'] ?? '';
$admPos = strpos($script, '/adm/');
$adminUrl = ($admPos !== false ? substr($script, 0, $admPos + 4) : '/adm') . '/admin.php';
?>
<!DOCTYPE html>
<html lang="sv">
<head>
    <meta charset="utf-8" />
    <meta name="viewport" content="width=device-width,initial-scale=1" />
    <title>Projektaktivitet - ByggArkiv Admin</title>
    <link rel="stylesheet" href="../common.css">
    <style>
        .container { max-width: 1100px; }
        .page-title {
            text-align: left;
            color: #1a73e8;
            margin-bottom: 20px;
        }
        .filters-card {
            background: #ffffff;
            border-radius: 12px;
            padding: 16px 20px;
            margin-bottom: 20px;
            box-shadow: 0 2px 8px rgba(0,0,0,0.1);
            display: flex;
            flex-wrap: wrap;
            gap: 16px;
            align-items: center;
            justify-content: space-between;
        }
        .filters-left {
            display: flex;
            flex-wrap: wrap;
            gap: 16px;
            align-items: center;
        }
        .filters-group label {
            font-size: 14px;
            color: #555;
            margin-right: 8px;
        }
        .filters-group select {
            padding: 6px 10px;
            border-radius: 6px;
            border: 1px solid #ccc;
            font-size: 14px;
        }
        .filters-group input[type="checkbox"] {
            margin-right: 6px;
        }
        .stats-summary {
            background: #ffffff;
            border-radius: 12px;
            padding: 16px 20px;
            margin-bottom: 20px;
            box-shadow: 0 2px 8px rgba(0,0,0,0.1);
        }
        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(180px, 1fr));
            gap: 12px;
        }
        .stat-item {
            text-align: center;
            padding: 10px;
            background: #f8f9fa;
            border-radius: 8px;
        }
        .stat-value {
            font-size: 20px;
            font-weight: 600;
            color: #1a73e8;
            margin-bottom: 4px;
        }
        .stat-label {
            font-size: 13px;
            color: #666;
        }
        .activity-table {
            width: 100%;
            border-collapse: collapse;
            background: #ffffff;
            border-radius: 12px;
            overflow: hidden;
            box-shadow: 0 2px 8px rgba(0,0,0,0.1);
        }
        .activity-table thead {
            background: #f8f9fa;
        }
        .activity-table th {
            padding: 12px 10px;
            text-align: left;
            font-weight: 600;
            color: #333;
            border-bottom: 2px solid #e0e0e0;
            font-size: 13px;
        }
        .activity-table th.text-right {
            text-align: right;
        }
        .activity-table td {
            padding: 10px;
            border-bottom: 1px solid #e0e0e0;
            font-size: 13px;
        }
        .activity-table td.text-right {
            text-align: right;
        }
        .activity-table tbody tr:hover {
            background: #f8f9fa;
        }
        .rank-badge {
            display: inline-block;
            min-width: 24px;
            padding: 4px 8px;
            border-radius: 999px;
            background: #e3f2fd;
            color: #1a73e8;
            font-weight: 600;
            text-align: center;
        }
        .rank-badge.top1 {
            background: #ffe082;
            color: #8d6e00;
        }
        .project-name {
            font-weight: 500;
        }
        .project-id {
            font-size: 11px;
            color: #777;
        }
        .status-badge {
            display: inline-block;
            padding: 2px 8px;
            border-radius: 999px;
            font-size: 11px;
            font-weight: 500;
        }
        .status-active {
            background: #d4edda;
            color: #155724;
        }
        .status-archived {
            background: #f8d7da;
            color: #721c24;
        }
        .empty-state {
            text-align: center;
            padding: 40px;
            color: #666;
            font-style: italic;
        }
        @media (max-width: 768px) {
            .filters-card {
                align-items: flex-start;
            }
        }
    </style>
</head>
<body>
    <div class="container">
        <header class="header">
            <a href="../../index.html">
                <img src="../../data/logo.png" alt="ByggArkiv Snickare Stefan" class="logo">
            </a>
            <div class="back-link">
                <a href="<?= htmlspecialchars($adminUrl) ?>">← Tillbaka till admin</a>
            </div>
        </header>

        <h1 class="page-title">Projektaktivitet</h1>

        <div class="filters-card">
            <form method="get" style="display: contents;">
                <div class="filters-left">
                    <div class="filters-group">
                        <label for="period">Tidsperiod:</label>
                        <select name="period" id="period" onchange="this.form.submit()">
                            <option value="7d" <?= $periodKey === '7d' ? 'selected' : '' ?>>Senaste 7 dagarna</option>
                            <option value="30d" <?= $periodKey === '30d' ? 'selected' : '' ?>>Senaste 30 dagarna</option>
                        </select>
                    </div>
                    <div class="filters-group">
                        <label style="display: inline-flex; align-items: center; gap: 6px; font-size: 13px; color: #555; margin: 0;">
                            <input type="checkbox" name="show_archived" value="1" <?= $showArchived ? 'checked' : '' ?> onchange="this.form.submit()">
                            Visa arkiverade projekt
                        </label>
                    </div>
                </div>
                <div>
                    <span style="font-size: 13px; color: #777;">
                        Period: <?= htmlspecialchars($periodLabel) ?>
                    </span>
                </div>
            </form>
        </div>

        <div class="stats-summary">
            <div class="stats-grid">
                <div class="stat-item">
                    <div class="stat-value"><?= htmlspecialchars($totalProjects) ?></div>
                    <div class="stat-label">Projekt med aktivitet</div>
                </div>
                <div class="stat-item">
                    <div class="stat-value"><?= htmlspecialchars(number_format($totalMessagesAll, 0, ',', ' ')) ?></div>
                    <div class="stat-label">Chattmeddelanden i perioden</div>
                </div>
                <div class="stat-item">
                    <div class="stat-value"><?= htmlspecialchars(number_format($totalMediaAll, 0, ',', ' ')) ?></div>
                    <div class="stat-label">Media (bilder/dokument/videor) i perioden</div>
                </div>
                <div class="stat-item">
                    <div class="stat-value"><?= $topProject ? htmlspecialchars($topProject) : '–' ?></div>
                    <div class="stat-label">Mest aktivt projekt</div>
                </div>
            </div>
        </div>

        <div class="card">
            <?php if (empty($rows)): ?>
                <div class="empty-state">
                    Inga aktiva projekt hittades för vald period.
                </div>
            <?php else: ?>
                <table class="activity-table">
                    <thead>
                        <tr>
                            <th>#</th>
                            <th>Projekt</th>
                            <th>Status</th>
                            <th class="text-right">Chattmeddelanden</th>
                            <th class="text-right">Media i chatt</th>
                            <th class="text-right">Media i bildarkiv</th>
                            <th class="text-right">Media totalt</th>
                            <th class="text-right">Aktivitetspoäng</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php $rank = 1; ?>
                        <?php foreach ($rows as $row): ?>
                            <tr>
                                <td>
                                    <span class="rank-badge <?= $rank === 1 ? 'top1' : '' ?>">
                                        <?= $rank ?>
                                    </span>
                                </td>
                                <td>
                                    <div class="project-name"><?= htmlspecialchars($row['project_name']) ?></div>
                                    <div class="project-id"><?= htmlspecialchars($row['project_id']) ?></div>
                                </td>
                                <td>
                                    <?php $isArchived = ($row['status'] ?? 'active') === 'archived'; ?>
                                    <span class="status-badge <?= $isArchived ? 'status-archived' : 'status-active' ?>">
                                        <?= $isArchived ? 'Arkiverad' : 'Aktiv' ?>
                                    </span>
                                </td>
                                <td class="text-right"><?= htmlspecialchars(number_format($row['messages'], 0, ',', ' ')) ?></td>
                                <td class="text-right"><?= htmlspecialchars(number_format($row['media_chat'], 0, ',', ' ')) ?></td>
                                <td class="text-right"><?= htmlspecialchars(number_format($row['media_uploads'], 0, ',', ' ')) ?></td>
                                <td class="text-right"><?= htmlspecialchars(number_format($row['media_total'], 0, ',', ' ')) ?></td>
                                <td class="text-right"><?= htmlspecialchars(number_format($row['score'], 0, ',', ' ')) ?></td>
                            </tr>
                            <?php $rank++; ?>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            <?php endif; ?>
        </div>
    </div>
</body>
</html>

