<?php
require_once __DIR__ . '/../admin_guard.php';
require_once __DIR__ . '/../../lib/participant_helper.php';

// Sökvägar
$projectsFile = __DIR__ . '/../../data/projects.json';
$chatsDir = __DIR__ . '/../../chats';
$uploadsDir = __DIR__ . '/../../uploads';

// Funktion för att läsa projekt
function readProjects($filePath) {
    if (!file_exists($filePath)) {
        return [];
    }
    $content = file_get_contents($filePath);
    if ($content === false) {
        return [];
    }
    $projects = json_decode($content, true);
    return is_array($projects) ? $projects : [];
}

// Funktion för att läsa chatmeddelanden
function readChatMessages($projectId, $chatsDir) {
    $chatFile = $chatsDir . '/' . $projectId . '/messages.json';
    if (!file_exists($chatFile)) {
        return [];
    }
    $content = file_get_contents($chatFile);
    if ($content === false) {
        return [];
    }
    $messages = json_decode($content, true);
    return is_array($messages) ? $messages : [];
}

// Hämta projekt-ID från URL
$projectId = sanitizeInput($_GET['project_id'] ?? '', 'project_id');
$userId = $_SESSION['user_id'] ?? '';
// Styr om arkiverade projekt ska visas i listan
$showArchived = isset($_GET['show_archived']) && $_GET['show_archived'] === '1';

if (empty($projectId)) {
    die('Projekt-ID saknas');
}

// Läsa projekt
$projects = readProjects($projectsFile);
$currentProject = null;
foreach ($projects as $project) {
    if (($project['id'] ?? '') === $projectId) {
        $currentProject = $project;
        break;
    }
}

if (!$currentProject) {
    die('Projekt hittades inte');
}

// Funktion för att formatera filstorlek
function formatFileSize($bytes) {
    if ($bytes == 0) return '0 B';
    $units = ['B', 'KB', 'MB', 'GB'];
    $i = floor(log($bytes, 1024));
    $size = round($bytes / pow(1024, $i), 1);
    return $size . ' ' . $units[$i];
}

// Läsa meddelanden och filtrera ut bilder, dokument och videor
$messages = readChatMessages($projectId, $chatsDir);
$mediaMessages = [];
$seenMessageIds = []; // För att undvika dubbletter
foreach ($messages as $message) {
    $messageType = $message['type'] ?? '';
    if (in_array($messageType, ['image', 'document', 'video'])) {
        $messageId = $message['id'] ?? '';
        // Hoppa över om vi redan sett detta meddelande-ID
        if (!empty($messageId) && isset($seenMessageIds[$messageId])) {
            continue;
        }
        $seenMessageIds[$messageId] = true;
        
        // Lägg till avsändarnamn
        $pid = $message['participant_id'] ?? '';
        if (!empty($pid)) {
            $message['sender_name'] = getParticipantName($pid);
        }
        
        // Hämta filstorlek från den faktiska filen
        $mediaFile = $message['media_file'] ?? '';
        $filePath = $chatsDir . '/' . $projectId . '/' . $mediaFile;
        $message['file_size'] = file_exists($filePath) ? filesize($filePath) : 0;
        
        // Hämta filnamn (display_name eller original_filename eller media_file)
        $message['display_filename'] = $message['display_name'] ?? $message['original_filename'] ?? $mediaFile;
        
        $mediaMessages[] = $message;
    }
}

// Hämta aktiva (och ev. arkiverade) projekt för dropdown
$activeProjects = [];
$archivedProjects = [];
$currentProjectIsActive = ($currentProject['status'] ?? 'active') === 'active';
foreach ($projects as $project) {
    $status = $project['status'] ?? 'active';
    if ($status === 'active') {
        $activeProjects[] = $project;
    } elseif ($status === 'archived') {
        $archivedProjects[] = $project;
    }
}
usort($activeProjects, function($a, $b) {
    return strcasecmp($a['name'] ?? '', $b['name'] ?? '');
});
usort($archivedProjects, function($a, $b) {
    return strcasecmp($a['name'] ?? '', $b['name'] ?? '');
});

// Generera admin URL
$script = $_SERVER['SCRIPT_NAME'] ?? '';
$admPos = strpos($script, '/adm/');
$adminUrl = ($admPos !== false ? substr($script, 0, $admPos + 4) : '/adm') . '/admin.php';
?>
<!DOCTYPE html>
<html lang="sv">
<head>
    <meta charset="utf-8" />
    <meta name="viewport" content="width=device-width,initial-scale=1" />
    <title>Kopiera från chatt till projekt - ByggArkiv Admin</title>
    <link rel="stylesheet" href="../common.css">
    <style>
        .container { max-width: 1200px; }
        .page-title {
            text-align: left;
            color: #1a73e8;
            margin-bottom: 20px;
        }
        .controls-section {
            background: white;
            border-radius: 12px;
            padding: 20px;
            margin-bottom: 24px;
            box-shadow: 0 2px 8px rgba(0,0,0,0.1);
        }
        .controls-row {
            display: flex;
            gap: 20px;
            align-items: flex-end;
            flex-wrap: wrap;
        }
        .control-group {
            display: flex;
            flex-direction: column;
            gap: 8px;
        }
        .control-group label {
            font-weight: 600;
            color: #333;
        }
        .control-group select {
            padding: 10px;
            border: 1px solid #ccc;
            border-radius: 8px;
            font-size: 16px;
            min-width: 250px;
        }
        .btn-copy {
            background: #28a745;
            color: white;
            border: none;
            padding: 12px 24px;
            border-radius: 8px;
            font-size: 16px;
            font-weight: 500;
            cursor: pointer;
            transition: background 0.2s;
        }
        .btn-copy:hover {
            background: #218838;
        }
        .btn-copy:disabled {
            background: #ccc;
            cursor: not-allowed;
        }
        .btn-view {
            background: #1a73e8;
            color: white;
            border: none;
            padding: 12px 24px;
            border-radius: 8px;
            font-size: 16px;
            font-weight: 500;
            cursor: pointer;
            transition: background 0.2s;
            text-decoration: none;
            display: inline-block;
        }
        .btn-view:hover {
            background: #1557b0;
        }
        .select-all-controls {
            display: flex;
            gap: 12px;
            align-items: center;
            margin-bottom: 16px;
        }
        .select-all-btn {
            background: #1a73e8;
            color: white;
            border: none;
            padding: 8px 16px;
            border-radius: 6px;
            font-size: 14px;
            cursor: pointer;
        }
        .select-all-btn:hover {
            background: #1557b0;
        }
        .images-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(250px, 1fr));
            gap: 20px;
        }
        .image-item {
            border: 1px solid #e0e0e0;
            border-radius: 8px;
            overflow: hidden;
            background: #fff;
            position: relative;
            transition: transform 0.2s, box-shadow 0.2s;
            display: flex;
            flex-direction: column;
        }
        .image-item:hover {
            transform: translateY(-2px);
            box-shadow: 0 4px 12px rgba(0,0,0,0.15);
        }
        .image-checkbox {
            position: absolute;
            top: 8px;
            left: 8px;
            z-index: 10;
            width: 24px;
            height: 24px;
            cursor: pointer;
        }
        .image-view-icon {
            position: absolute;
            top: 8px;
            right: 8px;
            z-index: 10;
            width: 32px;
            height: 32px;
            background: rgba(255, 255, 255, 0.9);
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            cursor: pointer;
            border: 1px solid #e0e0e0;
        }
        .image-view-icon svg {
            width: 18px;
            height: 18px;
            stroke: #1a73e8;
        }
        .image-number {
            position: absolute;
            top: 8px;
            left: 40px;
            z-index: 10;
            font-size: 14px;
            color: #000;
            font-weight: normal;
            background: rgba(255, 255, 255, 0.9);
            padding: 2px 8px;
            border-radius: 4px;
        }
        .image-container {
            width: 100%;
            height: 200px;
            overflow: hidden;
            background: #f0f0f0;
            display: flex;
            align-items: center;
            justify-content: center;
            position: relative;
        }
        .image-container img {
            width: 100%;
            height: 100%;
            object-fit: cover;
            cursor: pointer;
        }
        .image-info {
            padding: 12px;
            display: flex;
            flex-direction: column;
            gap: 8px;
        }
        .image-meta-row {
            display: flex;
            justify-content: space-between;
            align-items: center;
            font-size: 12px;
            color: #666;
        }
        .image-filename {
            font-size: 13px;
            margin-top: 4px;
        }
        .image-filename a {
            color: #1a73e8;
            text-decoration: underline;
        }
        .image-description {
            margin-top: 8px;
        }
        .image-description input {
            width: 100%;
            padding: 6px 8px;
            border: 1px solid #e0e0e0;
            border-radius: 4px;
            font-size: 13px;
            background: #e3f2fd;
            color: #000;
            box-sizing: border-box;
        }
        
        /* Lightbox Modal Styles */
        .lightbox {
            display: none;
            position: fixed;
            z-index: 1000;
            left: 0;
            top: 0;
            width: 100%;
            height: 100%;
            background-color: rgba(0, 0, 0, 0.9);
            backdrop-filter: blur(5px);
        }
        
        .lightbox-content {
            position: relative;
            margin: auto;
            padding: 20px;
            width: 90%;
            height: 90%;
            display: flex;
            align-items: center;
            justify-content: center;
        }
        
        .lightbox-image {
            max-width: 100%;
            max-height: 100%;
            object-fit: contain;
            border-radius: 8px;
            box-shadow: 0 4px 20px rgba(0, 0, 0, 0.5);
        }
        
        .lightbox-close {
            position: absolute;
            top: 20px;
            right: 35px;
            color: #f1f1f1;
            font-size: 40px;
            font-weight: bold;
            cursor: pointer;
            z-index: 1001;
            background: rgba(0, 0, 0, 0.5);
            border-radius: 50%;
            width: 50px;
            height: 50px;
            display: flex;
            align-items: center;
            justify-content: center;
            transition: background-color 0.2s;
        }
        
        .lightbox-close:hover {
            background: rgba(0, 0, 0, 0.8);
        }
        
        .lightbox-info {
            position: absolute;
            bottom: 20px;
            left: 20px;
            right: 20px;
            background: rgba(0, 0, 0, 0.8);
            color: white;
            padding: 15px;
            border-radius: 8px;
            font-size: 14px;
        }
        
        .lightbox-filename {
            font-weight: bold;
            margin-bottom: 5px;
        }
        
        .lightbox-meta {
            display: flex;
            justify-content: space-between;
            flex-wrap: wrap;
            gap: 10px;
        }
        
        @media (max-width: 768px) {
            .lightbox-content {
                width: 95%;
                height: 95%;
                padding: 10px;
            }
            .lightbox-close {
                top: 10px;
                right: 20px;
                font-size: 30px;
                width: 40px;
                height: 40px;
            }
            .lightbox-info {
                bottom: 10px;
                left: 10px;
                right: 10px;
                padding: 10px;
                font-size: 12px;
            }
        }
        .no-images {
            text-align: center;
            padding: 40px;
            color: #666;
            font-style: italic;
        }
    </style>
</head>
<body>
    <div class="container">
        <header class="header">
            <a href="../../index.html">
                <img src="../../data/logo.png" alt="ByggArkiv Snickare Stefan" class="logo">
            </a>
            <div class="back-link">
                <a href="list_chats.php?user_id=<?= htmlspecialchars(urlencode($userId)) ?>">← Tillbaka till chattöversikt</a>
            </div>
        </header>

        <h1 class="page-title">Kopiera från chatt till projekt</h1>

        <?php if (empty($mediaMessages)): ?>
            <div class="card">
                <div class="no-images">
                    Inga filer hittades i detta projekt.
                </div>
            </div>
        <?php else: ?>
            <div class="controls-section">
                <div class="select-all-controls">
                    <button class="select-all-btn" onclick="selectAll()">Välj alla</button>
                    <button class="select-all-btn" onclick="deselectAll()">Avmarkera alla</button>
                    <span id="selectedCount" style="color: #666; font-size: 14px;">0 filer valda</span>
                </div>
                <div class="controls-row">
                    <div class="control-group">
                        <label for="targetProject">Kopiera till projekt:</label>
                        <select id="targetProject">
                            <option value="">Välj projekt...</option>
                            <?php foreach ($activeProjects as $project): ?>
                                <option value="<?= htmlspecialchars($project['id']) ?>" 
                                        <?= ($project['id'] ?? '') === $projectId ? 'selected' : '' ?>>
                                    <?= htmlspecialchars($project['name']) ?> (<?= htmlspecialchars($project['id']) ?>)
                                </option>
                            <?php endforeach; ?>
                            <?php if ($showArchived): ?>
                                <?php foreach ($archivedProjects as $project): ?>
                                    <option value="<?= htmlspecialchars($project['id']) ?>" 
                                            <?= ($project['id'] ?? '') === $projectId ? 'selected' : '' ?>>
                                        <?= htmlspecialchars($project['name']) ?> (<?= htmlspecialchars($project['id']) ?>) [arkiverat]
                                    </option>
                                <?php endforeach; ?>
                            <?php endif; ?>
                        </select>
                        <label style="margin-top: 8px; display: flex; align-items: center; gap: 6px; font-size: 14px; color: #555;">
                            <input type="checkbox"
                                   id="showArchived"
                                   <?= $showArchived ? 'checked' : '' ?>
                                   onchange="toggleShowArchived(this)">
                            Visa arkiverade projekt
                        </label>
                    </div>
                    <div class="control-group">
                        <label style="visibility: hidden;">&nbsp;</label>
                        <div style="display: flex; gap: 12px;">
                            <button id="copyBtn" class="btn-copy" onclick="copyImages()" disabled>
                                Kopiera till projekt
                            </button>
                            <a href="../files/list1p.php?project_id=<?= htmlspecialchars(urlencode($projectId)) ?>&user_id=<?= htmlspecialchars(urlencode($userId)) ?>" 
                               class="btn-view">
                                Visa i bildvisning
                            </a>
                        </div>
                    </div>
                </div>
            </div>

            <div class="card">
                <div class="images-grid">
                    <?php foreach ($mediaMessages as $message): 
                        // Bygg en säker URL via download-endpointen istället för direktåtkomst till chats/-katalogen
                        $mediaFile = $message['media_file'] ?? '';
                        $mediaType = $message['type'] ?? 'image';
                        $mediaUrl = '../../chat/download.php?project_id=' . rawurlencode($projectId);
                        if ($mediaFile !== '') {
                            $mediaUrl .= '&media_file=' . rawurlencode($mediaFile);
                        }
                        $senderName = $message['sender_name'] ?? $message['participant_id'] ?? 'Okänd';
                        $createdAt = $message['created_at'] ?? '';
                        $date = !empty($createdAt) ? date('Y-m-d H:i', strtotime($createdAt)) : '';
                        $fileSize = $message['file_size'] ?? 0;
                        $displayFilename = $message['display_filename'] ?? $mediaFile;
                        $description = $message['message'] ?? '';
                        
                        // För bilder, kontrollera om det finns bildnummer i uploads-metadata
                        $imageNumber = null;
                        if ($mediaType === 'image') {
                            // Försök hitta bildnummer från uploads-metadata om filen finns där
                            $baseName = pathinfo($mediaFile, PATHINFO_FILENAME);
                            $uploadsMetaPath = $uploadsDir . '/' . $projectId . '/' . $baseName . '.json';
                            if (file_exists($uploadsMetaPath)) {
                                $metaContent = @file_get_contents($uploadsMetaPath);
                                if ($metaContent !== false) {
                                    $meta = @json_decode($metaContent, true);
                                    if (is_array($meta) && isset($meta['image_number'])) {
                                        $imageNumber = $meta['image_number'];
                                    }
                                }
                            }
                        }
                    ?>
                        <div class="image-item">
                            <input type="checkbox" 
                                   class="image-checkbox" 
                                   data-message-id="<?= htmlspecialchars($message['id'] ?? '') ?>"
                                   data-media-type="<?= htmlspecialchars($mediaType) ?>"
                                   onchange="updateSelectedCount()">
                            <?php if ($mediaType === 'image'): ?>
                                <div class="image-view-icon" 
                                     onclick="openLightbox('<?= htmlspecialchars($mediaUrl) ?>', '<?= htmlspecialchars(addslashes($displayFilename)) ?>', '<?= htmlspecialchars($date) ?>', '<?= htmlspecialchars(addslashes($senderName)) ?>', '<?= htmlspecialchars(addslashes($description)) ?>', <?= $imageNumber !== null ? "'Bild #" . htmlspecialchars($imageNumber) . "'" : "null" ?>)"
                                     title="Visa bild">
                                    <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                        <path d="M1 12s4-8 11-8 11 8 11 8-4 8-11 8-11-8-11-8z"></path>
                                        <circle cx="12" cy="12" r="3"></circle>
                                    </svg>
                                </div>
                                <?php if ($imageNumber !== null): ?>
                                    <div class="image-number">Bild #<?= htmlspecialchars($imageNumber) ?></div>
                                <?php endif; ?>
                            <?php endif; ?>
                            <div class="image-container">
                                <?php if ($mediaType === 'image'): ?>
                                    <img src="<?= htmlspecialchars($mediaUrl) ?>" 
                                         alt="Bild" 
                                         onclick="toggleImage(this)">
                                <?php elseif ($mediaType === 'document'): ?>
                                    <div style="display: flex; align-items: center; justify-content: center; height: 100%; background: #fff; flex-direction: column; gap: 8px;">
                                        <svg width="80" height="80" viewBox="0 0 24 24" fill="none" stroke="#666" stroke-width="1.5">
                                            <!-- Vitt papper med nedvikt hörn -->
                                            <path d="M6 3h9l3 3v13a2 2 0 0 1-2 2H6a2 2 0 0 1-2-2V5a2 2 0 0 1 2-2z" fill="#fff" stroke="#666"></path>
                                            <path d="M14 3v5h5" stroke="#666" fill="none"></path>
                                        </svg>
                                    </div>
                                <?php elseif ($mediaType === 'video'): ?>
                                    <div style="display: flex; align-items: center; justify-content: center; height: 100%; background: #000; flex-direction: column; gap: 8px;">
                                        <svg width="64" height="64" viewBox="0 0 24 24" fill="#fff">
                                            <path d="M8 5v14l11-7z"></path>
                                        </svg>
                                        <span style="font-size: 12px; color: #fff;">Video</span>
                                    </div>
                                <?php endif; ?>
                            </div>
                            <div class="image-info">
                                <div class="image-meta-row">
                                    <span><?= htmlspecialchars($date) ?> <?= formatFileSize($fileSize) ?></span>
                                    <span><?= htmlspecialchars($senderName) ?></span>
                                </div>
                                <div class="image-filename">
                                    <a href="<?= htmlspecialchars($mediaUrl) ?>" target="_blank"><?= htmlspecialchars($displayFilename) ?></a>
                                </div>
                                <?php if ($mediaType === 'document'): ?>
                                    <div class="image-description">
                                        <input type="text" value="<?= htmlspecialchars($description) ?>" readonly>
                                    </div>
                                <?php endif; ?>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
            </div>
        <?php endif; ?>
    </div>

    <!-- Lightbox Modal -->
    <div id="lightbox" class="lightbox">
        <div class="lightbox-content">
            <span class="lightbox-close" onclick="closeLightbox()">&times;</span>
            <img id="lightbox-image" class="lightbox-image" src="" alt="" style="display: none;">
            <video id="lightbox-video" class="lightbox-image" controls style="display: none;">
                Din webbläsare stöder inte videouppspelning.
            </video>
            <div class="lightbox-info">
                <div id="lightbox-image-number" class="lightbox-filename" style="font-weight: bold; color: #fff; margin-bottom: 5px;"></div>
                <div id="lightbox-filename" class="lightbox-filename"></div>
                <div class="lightbox-meta">
                    <span id="lightbox-date"></span>
                    <span id="lightbox-user"></span>
                </div>
                <div id="lightbox-note" style="margin-top: 8px; font-style: italic;"></div>
            </div>
        </div>
    </div>

    <script>
        function toggleImage(img) {
            const checkbox = img.closest('.image-item').querySelector('.image-checkbox');
            checkbox.checked = !checkbox.checked;
            updateSelectedCount();
        }

        function toggleShowArchived(checkbox) {
            const url = new URL(window.location.href);
            if (checkbox.checked) {
                url.searchParams.set('show_archived', '1');
            } else {
                url.searchParams.delete('show_archived');
            }
            window.location.href = url.toString();
        }

        function selectAll() {
            document.querySelectorAll('.image-checkbox').forEach(cb => cb.checked = true);
            updateSelectedCount();
        }

        function deselectAll() {
            document.querySelectorAll('.image-checkbox').forEach(cb => cb.checked = false);
            updateSelectedCount();
        }

        function updateSelectedCount() {
            const checked = document.querySelectorAll('.image-checkbox:checked').length;
            document.getElementById('selectedCount').textContent = checked + ' filer valda';
            
            const targetProject = document.getElementById('targetProject').value;
            document.getElementById('copyBtn').disabled = checked === 0 || !targetProject;
        }

        document.getElementById('targetProject').addEventListener('change', updateSelectedCount);
        
        // Uppdatera knappens status vid sidladdning om projekt är förvalt
        updateSelectedCount();

        // Lightbox functions
        function openLightbox(imageSrc, filename, date, user, note, imageNumber) {
            const lightbox = document.getElementById('lightbox');
            const lightboxImage = document.getElementById('lightbox-image');
            const lightboxVideo = document.getElementById('lightbox-video');
            const lightboxImageNumber = document.getElementById('lightbox-image-number');
            const lightboxFilename = document.getElementById('lightbox-filename');
            const lightboxDate = document.getElementById('lightbox-date');
            const lightboxUser = document.getElementById('lightbox-user');
            const lightboxNote = document.getElementById('lightbox-note');
            
            if (!lightbox || !lightboxImage || !lightboxVideo) {
                console.error('Lightbox elements not found');
                return;
            }
            
            // Kontrollera om det är en video baserat på filnamn
            const isVideo = /\.(mp4|webm|mov|avi)$/i.test(filename);
            
            if (isVideo) {
                lightboxImage.style.display = 'none';
                lightboxVideo.style.display = 'block';
                lightboxVideo.src = imageSrc;
                lightboxVideo.load(); // Ladda videon
            } else {
                // Visa bild, dölj video
                lightboxVideo.style.display = 'none';
                lightboxImage.style.display = 'block';
                lightboxImage.src = imageSrc;
            }
            
            // Visa bildnummer om det finns
            if (imageNumber && imageNumber.trim() !== '') {
                lightboxImageNumber.textContent = imageNumber;
                lightboxImageNumber.style.display = 'block';
            } else {
                lightboxImageNumber.style.display = 'none';
            }
            
            lightboxFilename.textContent = filename;
            lightboxDate.textContent = date;
            lightboxUser.textContent = user;
            lightboxNote.textContent = note || '';
            
            lightbox.style.display = 'block';
            document.body.style.overflow = 'hidden'; // Prevent background scrolling
        }
        
        function closeLightbox() {
            const lightbox = document.getElementById('lightbox');
            const lightboxVideo = document.getElementById('lightbox-video');
            if (lightbox) {
                lightbox.style.display = 'none';
                document.body.style.overflow = 'auto'; // Restore scrolling
                
                // Pausa video om den spelar
                if (lightboxVideo) {
                    lightboxVideo.pause();
                    lightboxVideo.currentTime = 0;
                }
            }
        }
        
        // Close lightbox when clicking outside the image
        document.addEventListener('click', function(event) {
            const lightbox = document.getElementById('lightbox');
            if (event.target === lightbox) {
                closeLightbox();
            }
        });
        
        // Close lightbox with Escape key
        document.addEventListener('keydown', function(event) {
            if (event.key === 'Escape') {
                closeLightbox();
            }
        });

        async function copyImages() {
            const targetProjectId = document.getElementById('targetProject').value;
            if (!targetProjectId) {
                alert('Välj ett målprojekt');
                return;
            }

            const checkedBoxes = document.querySelectorAll('.image-checkbox:checked');
            if (checkedBoxes.length === 0) {
                alert('Välj minst en fil att kopiera');
                return;
            }

            const imageIds = Array.from(checkedBoxes).map(cb => cb.dataset.messageId);

            if (!confirm(`Kopiera ${imageIds.length} fil(er) till valt projekt?`)) {
                return;
            }

            const copyBtn = document.getElementById('copyBtn');
            copyBtn.disabled = true;
            copyBtn.textContent = 'Kopierar...';

            try {
                const formData = new FormData();
                formData.append('source_project_id', '<?= htmlspecialchars($projectId) ?>');
                formData.append('target_project_id', targetProjectId);
                formData.append('user_id', '<?= htmlspecialchars($userId) ?>');
                imageIds.forEach(id => {
                    formData.append('image_ids[]', id);
                });

                const response = await fetch('../api/chat/copy_images.php', {
                    method: 'POST',
                    body: formData
                });

                // Läs response som text först, sedan parsa som JSON
                const responseText = await response.text();
                let data;
                try {
                    data = JSON.parse(responseText);
                } catch (e) {
                    console.error('JSON parse error:', e, 'Response:', responseText);
                    alert('Fel: Kunde inte läsa svar från servern. ' + responseText.substring(0, 100));
                    return;
                }

                if (!response.ok) {
                    console.error('API error:', data);
                    alert('Fel: ' + (data.message || 'Kunde inte kopiera bilder'));
                    return;
                }

                if (data.success) {
                    let message = '';
                    if (data.copied_count > 0) {
                        message = `Kopierade ${data.copied_count} fil(er) till projektet.`;
                    }
                    if (data.already_exists_count > 0) {
                        if (message) message += '\n\n';
                        message += `${data.already_exists_count} fil(er) fanns redan i projektet och kopierades inte:\n`;
                        message += data.already_exists.slice(0, 5).join(', ');
                        if (data.already_exists.length > 5) {
                            message += ` och ${data.already_exists.length - 5} till.`;
                        }
                    }
                    if (!message) {
                        message = 'Inga filer kopierades.';
                    }
                    alert(message);
                    // Uppdatera sidan för att visa nya filer (om några kopierades till samma projekt)
                    if (targetProjectId === '<?= htmlspecialchars($projectId) ?>' && data.copied_count > 0) {
                        window.location.reload();
                    }
                } else {
                    alert('Fel: ' + (data.message || 'Kunde inte kopiera filer'));
                }
            } catch (error) {
                console.error('Fel:', error);
                alert('Ett fel uppstod: ' + error.message);
            } finally {
                copyBtn.disabled = false;
                copyBtn.textContent = 'Kopiera till projekt';
                updateSelectedCount();
            }
        }
    </script>
</body>
</html>
