<?php
require_once __DIR__ . '/../admin_guard.php';

$script = $_SERVER['SCRIPT_NAME'] ?? '';
$admPos = strpos($script, '/adm/');
$adminUrl = ($admPos !== false ? substr($script, 0, $admPos + 4) : '/adm') . '/admin.php';
$userId = $_SESSION['user_id'] ?? $_GET['user_id'] ?? '';
if (!empty($userId)) {
    $adminUrl .= '?user_id=' . urlencode($userId);
}
?>
<!DOCTYPE html>
<html lang="sv">
<head>
    <meta charset="utf-8" />
    <meta name="viewport" content="width=device-width,initial-scale=1" />
    <title>Aktivitetslogg - ByggArkiv</title>
    <link rel="stylesheet" href="../common.css">
    <style>
        .timeline-container {
            max-width: 1200px;
            margin: 0 auto;
            padding: 20px;
        }
        
        .date-filter {
            background: white;
            border-radius: 8px;
            padding: 20px;
            margin-bottom: 30px;
            box-shadow: 0 2px 4px rgba(0,0,0,0.1);
        }
        
        .date-filter-header {
            display: flex;
            align-items: center;
            justify-content: space-between;
            margin-bottom: 15px;
            flex-wrap: wrap;
            gap: 15px;
        }
        
        .date-filter h3 {
            margin: 0;
            color: #333;
            font-size: 18px;
            min-width: 150px;
        }
        
        .date-inputs {
            display: flex;
            gap: 10px;
            align-items: center;
            flex-wrap: wrap;
        }
        
        .date-separator {
            color: #666;
            font-size: 16px;
            font-weight: 500;
        }
        
        .date-filter-header .date-inputs {
            flex: 1;
        }
        
        .date-filter button {
            margin-left: auto;
        }
        
        .filter-checkboxes {
            display: flex;
            gap: 20px;
            flex-wrap: wrap;
            align-items: center;
            padding-top: 15px;
            border-top: 1px solid #e0e0e0;
        }
        
        .filter-checkbox-group {
            display: flex;
            align-items: center;
            gap: 6px;
        }
        
        .filter-checkbox-group input[type="checkbox"] {
            width: 18px;
            height: 18px;
            cursor: pointer;
        }
        
        .filter-checkbox-group label {
            font-size: 14px;
            color: #333;
            cursor: pointer;
            user-select: none;
        }
        
        .date-input-group {
            display: flex;
            align-items: center;
        }
        
        .date-input-group input[type="date"] {
            padding: 8px 12px;
            border: 1px solid #ddd;
            border-radius: 6px;
            font-size: 14px;
        }
        
        .date-filter button {
            padding: 10px 20px;
            background: #1a73e8;
            color: white;
            border: none;
            border-radius: 6px;
            cursor: pointer;
            font-size: 14px;
            font-weight: 600;
            height: 40px;
        }
        
        .date-filter button:hover {
            background: #1557b0;
        }
        
        .timeline {
            background: white;
            border-radius: 8px;
            padding: 20px;
            box-shadow: 0 2px 4px rgba(0,0,0,0.1);
        }
        
        .timeline-header {
            margin-bottom: 20px;
            padding-bottom: 15px;
            border-bottom: 2px solid #e0e0e0;
        }
        
        .timeline-header h2 {
            margin: 0;
            color: #1a73e8;
            font-size: 24px;
        }
        
        .timeline-header .subtitle {
            margin-top: 5px;
            color: #666;
            font-size: 14px;
        }
        
        .timeline-empty {
            text-align: center;
            padding: 60px 20px;
            color: #666;
        }
        
        .timeline-empty svg {
            width: 64px;
            height: 64px;
            margin-bottom: 20px;
            opacity: 0.3;
        }
        
        .timeline-group {
            margin-bottom: 30px;
        }
        
        .timeline-group-date {
            font-size: 18px;
            font-weight: 600;
            color: #333;
            margin-bottom: 15px;
            padding-bottom: 8px;
            border-bottom: 1px solid #e0e0e0;
        }
        
        .timeline-item {
            display: flex;
            gap: 15px;
            padding: 15px;
            margin-bottom: 12px;
            border-left: 4px solid #1a73e8;
            background: #f8f9fa;
            border-radius: 4px;
        }
        
        .timeline-item.chat {
            border-left-color: #28a745;
        }
        
        .timeline-item.upload {
            border-left-color: #1a73e8;
        }
        
        .timeline-item.package {
            border-left-color: #ffc107;
        }
        
        .timeline-item.system_message {
            border-left-color: #6f42c1;
        }
        
        .timeline-item.project {
            border-left-color: #17a2b8;
        }
        
        .timeline-item.user {
            border-left-color: #fd7e14;
        }
        
        .timeline-item.participant {
            border-left-color: #20c997;
        }
        
        .timeline-item.file {
            border-left-color: #dc3545;
        }
        
        .timeline-item.external_user {
            border-left-color: #6c757d;
        }
        
        .timeline-item.version {
            border-left-color: #007bff;
        }
        
        .timeline-item.version.omdeploy {
            border-left-color: #6c757d;
            opacity: 0.85;
        }
        
        .timeline-item.chat_cleanup {
            border-left-color: #e83e8c;
        }
        
        .timeline-item.repair {
            border-left-color: #28a745;
        }
        
        .timeline-icon {
            width: 40px;
            height: 40px;
            min-width: 40px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            flex-shrink: 0;
        }
        
        .timeline-icon svg {
            width: 20px;
            height: 20px;
            fill: currentColor;
        }
        
        .timeline-item.chat .timeline-icon {
            background: #28a745;
            color: white;
        }
        
        .timeline-item.upload .timeline-icon {
            background: #1a73e8;
            color: white;
        }
        
        .timeline-item.package .timeline-icon {
            background: #ffc107;
            color: #333;
        }
        
        .timeline-item.system_message .timeline-icon {
            background: #6f42c1;
            color: white;
        }
        
        .timeline-item.project .timeline-icon {
            background: #17a2b8;
            color: white;
        }
        
        .timeline-item.user .timeline-icon {
            background: #fd7e14;
            color: white;
        }
        
        .timeline-item.participant .timeline-icon {
            background: #20c997;
            color: white;
        }
        
        .timeline-item.file .timeline-icon {
            background: #dc3545;
            color: white;
        }
        
        .timeline-item.external_user .timeline-icon {
            background: #6c757d;
            color: white;
        }
        
        .timeline-item.version .timeline-icon {
            background: #007bff;
            color: white;
        }
        
        .timeline-item.chat_cleanup .timeline-icon {
            background: #e83e8c;
            color: white;
        }
        
        .timeline-item.repair .timeline-icon {
            background: #28a745;
            color: white;
        }
        
        .timeline-item.version.omdeploy .timeline-icon {
            background: #6c757d;
            color: white;
        }
        
        .timeline-content {
            flex: 1;
            min-width: 0;
        }
        
        .timeline-title {
            font-size: 16px;
            font-weight: 600;
            color: #333;
            margin-bottom: 4px;
        }
        
        .timeline-description {
            font-size: 14px;
            color: #666;
            margin-bottom: 6px;
        }
        
        .timeline-meta {
            font-size: 12px;
            color: #999;
            display: flex;
            gap: 12px;
            flex-wrap: wrap;
        }
        
        .timeline-meta span {
            display: flex;
            align-items: center;
            gap: 4px;
        }
        
        .timeline-description-link {
            background: none;
            border: none;
            padding: 0;
            margin: 0;
            color: #1a73e8;
            cursor: pointer;
            font: inherit;
            text-align: left;
            text-decoration: none;
        }
        
        .timeline-description-link:hover {
            text-decoration: underline;
        }
        
        .timeline-time {
            font-size: 13px;
            color: #1a73e8;
            font-weight: 500;
            white-space: nowrap;
            min-width: 100px;
            text-align: right;
        }
        
        .loading {
            text-align: center;
            padding: 40px;
            color: #666;
        }
        
        .activity-popup-overlay {
            position: fixed;
            inset: 0;
            background: rgba(0, 0, 0, 0.4);
            display: flex;
            align-items: center;
            justify-content: center;
            z-index: 1000;
        }
        
        .activity-popup {
            background: #fff;
            border-radius: 8px;
            max-width: 700px;
            width: 90%;
            max-height: 80vh;
            display: flex;
            flex-direction: column;
            box-shadow: 0 4px 16px rgba(0,0,0,0.25);
        }
        
        .activity-popup-header {
            padding: 12px 16px;
            border-bottom: 1px solid #e0e0e0;
            display: flex;
            align-items: center;
            justify-content: space-between;
            font-weight: 600;
            font-size: 14px;
        }
        
        .activity-popup-body {
            padding: 16px;
            font-size: 14px;
            color: #333;
            overflow-y: auto;
            white-space: pre-wrap;
            word-wrap: break-word;
        }
        
        .activity-popup-close {
            background: none;
            border: none;
            font-size: 22px;
            line-height: 1;
            cursor: pointer;
            color: #666;
        }
        
        .error {
            background: #f8d7da;
            color: #721c24;
            padding: 15px;
            border-radius: 6px;
            margin-bottom: 20px;
            border: 1px solid #f5c6cb;
        }
        
        @media (max-width: 768px) {
            .date-filter-header {
                flex-direction: column;
                align-items: stretch;
            }
            
            .date-inputs {
                width: 100%;
                flex-direction: column;
                align-items: stretch;
            }
            
            .date-input-group {
                width: 100%;
            }
            
            .date-input-group input[type="date"] {
                width: 100%;
            }
            
            .date-filter button {
                width: 100%;
            }
            
            .timeline-item {
                flex-direction: column;
            }
            
            .timeline-time {
                text-align: left;
                min-width: auto;
            }
        }
    </style>
</head>
<body>
    <div class="container">
        <header class="header">
            <a href="../../index.html">
                <img src="../../data/logo.png" alt="ByggArkiv Snickare Stefan" class="logo">
            </a>
        </header>

        <div class="back-link">
            <a href="<?= htmlspecialchars($adminUrl) ?>">← Tillbaka till admin</a>
        </div>

        <div class="timeline-container">
            <h1 style="text-align: left; color: #1a73e8; margin-bottom: 30px;">Aktivitetslogg</h1>
            
            <div class="date-filter">
                <div class="date-filter-header">
                    <h3>Datumintervall:</h3>
                    <div class="date-inputs">
                        <div class="date-input-group">
                            <input type="date" id="fromDate" value="<?= date('Y-m-d') ?>">
                        </div>
                        <span class="date-separator">-</span>
                        <div class="date-input-group">
                            <input type="date" id="toDate" value="<?= date('Y-m-d') ?>">
                        </div>
                    </div>
                    <button onclick="loadActivities()">Visa aktivitet</button>
                </div>
                <div class="filter-checkboxes">
                    <div class="filter-checkbox-group">
                        <input type="checkbox" id="filterChat">
                        <label for="filterChat" title="Chattmeddelande, Bild uppladdad i chatt, Dokument uppladdad i chatt, Video uppladdad i chatt">Chattmeddelanden</label>
                    </div>
                    <div class="filter-checkbox-group">
                        <input type="checkbox" id="filterUpload" checked>
                        <label for="filterUpload" title="Bild uppladdad">Uppladdningar</label>
                    </div>
                    <div class="filter-checkbox-group">
                        <input type="checkbox" id="filterPackage" checked>
                        <label for="filterPackage" title="Zip-paket skapat">Zip-paketeringar</label>
                    </div>
                    <div class="filter-checkbox-group">
                        <input type="checkbox" id="filterSystemMessage" checked>
                        <label for="filterSystemMessage" title="Systemmeddelande skapat">Systemmeddelanden</label>
                    </div>
                    <div class="filter-checkbox-group">
                        <input type="checkbox" id="filterProject" checked>
                        <label for="filterProject" title="Projekt skapat, Projekt uppdaterat, Projekt arkiverat">Projekt-händelser</label>
                    </div>
                    <div class="filter-checkbox-group">
                        <input type="checkbox" id="filterUser" checked>
                        <label for="filterUser" title="Användare skapad, Användare uppdaterad, Användare inaktiverad">Användar-händelser</label>
                    </div>
                    <div class="filter-checkbox-group">
                        <input type="checkbox" id="filterParticipant" checked>
                        <label for="filterParticipant" title="Deltagare tillagd">Deltagar-händelser</label>
                    </div>
                    <div class="filter-checkbox-group">
                        <input type="checkbox" id="filterFile" checked>
                        <label for="filterFile" title="Fil återställd, Fil raderad">Fil-händelser</label>
                    </div>
                    <div class="filter-checkbox-group">
                        <input type="checkbox" id="filterFileMove" checked>
                        <label for="filterFileMove" title="Fil flyttad">Filflytt</label>
                    </div>
                    <div class="filter-checkbox-group">
                        <input type="checkbox" id="filterExternalUser" checked>
                        <label for="filterExternalUser" title="Extern användare skapad, Extern användare uppdaterad">Externa användare</label>
                    </div>
                    <div class="filter-checkbox-group">
                        <input type="checkbox" id="filterVersion">
                        <label for="filterVersion" title="Ny version av programvaran, Uppdaterat programvaran">Programuppdateringar</label>
                    </div>
                    <div class="filter-checkbox-group">
                        <input type="checkbox" id="filterChatCleanup" checked>
                        <label for="filterChatCleanup" title="Chatt rensad, Chattmeddelande raderat">Chatt rensning</label>
                    </div>
                    <div class="filter-checkbox-group">
                        <input type="checkbox" id="filterRepair" checked>
                        <label for="filterRepair" title="Bildnummer tilldelade, Backup-filer rensade, Thumbnails rensade, Projektbilder rensade, Chattbilder rensade, Metadata utan bild rensade, Metadata utan thumbnail rensade, Mappar utan projekt rensade, Filer utan metadata rensade">Reparation/Städning</label>
                    </div>
                </div>
            </div>
            
            <div id="errorContainer"></div>
            
            <div id="timelineContainer" class="timeline">
                <div class="loading">Laddar aktivitet...</div>
            </div>
        </div>
    </div>

    <script>
        const userId = '<?= htmlspecialchars($userId) ?>';
        
        function formatTimestamp(timestamp) {
            const date = new Date(timestamp);
            const now = new Date();
            const today = new Date(now.getFullYear(), now.getMonth(), now.getDate());
            const eventDate = new Date(date.getFullYear(), date.getMonth(), date.getDate());
            
            const timeStr = date.toLocaleTimeString('sv-SE', { hour: '2-digit', minute: '2-digit' });
            
            if (eventDate.getTime() === today.getTime()) {
                return 'Idag ' + timeStr;
            } else if (eventDate.getTime() === today.getTime() - 86400000) {
                return 'Igår ' + timeStr;
            } else {
                return date.toLocaleDateString('sv-SE', { year: 'numeric', month: 'short', day: 'numeric' }) + ' ' + timeStr;
            }
        }
        
        function formatDate(dateStr) {
            const date = new Date(dateStr);
            return date.toLocaleDateString('sv-SE', { year: 'numeric', month: 'long', day: 'numeric' });
        }
        
        function getTypeIcon(type) {
            switch(type) {
                case 'chat':
                    return '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24"><path d="M20 2H4c-1.1 0-2 .9-2 2v18l4-4h14c1.1 0 2-.9 2-2V4c0-1.1-.9-2-2-2z"/></svg>';
                case 'upload':
                    return '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24"><path d="M19.35 10.04C18.67 6.59 15.64 4 12 4 9.11 4 6.6 5.64 5.35 8.04 2.34 8.36 0 10.91 0 14c0 3.31 2.69 6 6 6h13c2.76 0 5-2.24 5-5 0-2.64-2.05-4.78-4.65-4.96zM14 13v4h-4v-4H7l5-5 5 5h-3z"/></svg>';
                case 'package':
                    return '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24"><path d="M20 6h-2.18c.11-.31.18-.65.18-1 0-1.66-1.34-3-3-3-1.05 0-1.96.54-2.5 1.35l-.5.67-.5-.68C10.96 2.54 10.05 2 9 2 7.34 2 6 3.34 6 5c0 .35.07.69.18 1H4c-1.11 0-1.99.89-1.99 2L2 19c0 1.11.89 2 2 2h16c1.11 0 2-.89 2-2V8c0-1.11-.89-2-2-2zm-5-2c.55 0 1 .45 1 1s-.45 1-1 1-1-.45-1-1 .45-1 1-1zM9 4c.55 0 1 .45 1 1s-.45 1-1 1-1-.45-1-1 .45-1 1-1zm11 15H4v-2h16v2zm0-5H4V8h5.08L7 10.83 8.62 12 11 8.76l1-1.36 1 1.36L15.38 12 17 10.83 14.92 8H20v6z"/></svg>';
                case 'system_message':
                    return '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24"><path d="M12 2C6.48 2 2 6.48 2 12s4.48 10 10 10 10-4.48 10-10S17.52 2 12 2zm1 15h-2v-2h2v2zm0-4h-2V7h2v6z"/></svg>';
                case 'project':
                    return '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24"><path d="M10 4H4c-1.11 0-2 .89-2 2v12c0 1.11.89 2 2 2h16c1.11 0 2-.89 2-2V8c0-1.11-.89-2-2-2h-8l-2-2z"/></svg>';
                case 'user':
                    return '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24"><path d="M16 7c0-2.21-1.79-4-4-4S8 4.79 8 7s1.79 4 4 4 4-1.79 4-4zm-4 6c-3.31 0-6 2.69-6 6v2h12v-2c0-3.31-2.69-6-6-6z"/></svg>';
                case 'participant':
                    return '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24"><path d="M19 13h-6v6h-2v-6H5v-2h6V5h2v6h6v2z"/></svg>';
                case 'file':
                    return '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24"><path d="M6 19c0 1.1.9 2 2 2h8c1.1 0 2-.9 2-2V7H6v12zM19 4h-3.5l-1-1h-5l-1 1H5v2h14V4z"/></svg>';
                case 'external_user':
                    return '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24"><path d="M12 7c-2.76 0-5 2.24-5 5s2.24 5 5 5 5-2.24 5-5-2.24-5-5-5zm0 8c-1.66 0-3-1.34-3-3s1.34-3 3-3 3 1.34 3 3-1.34 3-3 3zm0-13C6.48 2 2 6.48 2 12s4.48 10 10 10 10-4.48 10-10S17.52 2 12 2zm0 18c-4.41 0-8-3.59-8-8s3.59-8 8-8 8 3.59 8 8-3.59 8-8 8z"/></svg>';
                case 'version':
                    return '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24"><path d="M12 2C6.48 2 2 6.48 2 12s4.48 10 10 10 10-4.48 10-10S17.52 2 12 2zm-2 15l-5-5 1.41-1.41L10 14.17l7.59-7.59L19 8l-9 9z"/></svg>';
                case 'chat_cleanup':
                    return '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24"><path d="M19 6.41L17.59 5 12 10.59 6.41 5 5 6.41 10.59 12 5 17.59 6.41 19 12 13.41 17.59 19 19 17.59 13.41 12z"/></svg>';
                case 'repair':
                    return '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24"><path d="M22.7 19l-9.1-9.1c.9-2.3.4-5-1.5-6.9-2-2-5-2.4-7.4-1.3L9 6 6 9 1.6 4.7C.4 7.1.9 10.1 2.9 12.1c1.9 1.9 4.6 2.4 6.9 1.5l9.1 9.1c.4.4 1 .4 1.4 0l2.3-2.3c.5-.4.5-1.1.1-1.4z"/></svg>';
                case 'file_move':
                    return '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24"><path d="M14 2H6c-1.1 0-1.99.9-1.99 2L4 20c0 1.1.89 2 1.99 2H18c1.1 0 2-.9 2-2V8l-6-6zm2 16H8v-2h8v2zm0-4H8v-2h8v2zm-3-5V3.5L18.5 9H13z"/></svg>';
                default:
                    return '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24"><path d="M10 4H4c-1.11 0-2 .89-2 2v12c0 1.11.89 2 2 2h16c1.11 0 2-.89 2-2V8c0-1.11-.89-2-2-2h-8l-2-2z"/></svg>';
            }
        }
        
        function loadActivities() {
            const fromDate = document.getElementById('fromDate').value;
            const toDate = document.getElementById('toDate').value;
            
            if (!fromDate || !toDate) {
                showError('Välj både från- och till-datum');
                return;
            }
            
            const container = document.getElementById('timelineContainer');
            container.innerHTML = '<div class="loading">Laddar aktivitet...</div>';
            
            const errorContainer = document.getElementById('errorContainer');
            errorContainer.innerHTML = '';
            
            const url = '../../api/activity.php?action=list&from=' + encodeURIComponent(fromDate) + '&to=' + encodeURIComponent(toDate) + 
                       (userId ? '&user_id=' + encodeURIComponent(userId) : '');
            
            fetch(url)
                .then(response => response.json())
                .then(data => {
                    if (!data.success) {
                        showError(data.message || 'Kunde inte ladda aktivitet');
                        container.innerHTML = '<div class="timeline-empty">Ingen aktivitet hittades</div>';
                        return;
                    }
                    
                    // Filtrera aktiviteter baserat på valda kryssrutor
                    const filteredActivities = getFilteredActivities(data.activities);
                    
                    displayActivities(filteredActivities, fromDate, toDate).catch(error => {
                        console.error('Error displaying activities:', error);
                        showError('Fel vid visning av aktivitet: ' + error.message);
                    });
                })
                .catch(error => {
                    console.error('Error loading activities:', error);
                    showError('Fel vid laddning av aktivitet: ' + error.message);
                    container.innerHTML = '<div class="timeline-empty">Kunde inte ladda aktivitet</div>';
                });
        }
        
        function showError(message) {
            const errorContainer = document.getElementById('errorContainer');
            errorContainer.innerHTML = '<div class="error">' + escapeHtml(message) + '</div>';
        }
        
        function escapeHtml(text) {
            if (!text) return '';
            const div = document.createElement('div');
            div.textContent = text;
            return div.innerHTML;
        }
        
        function showDescriptionPopup(title, text) {
            const overlay = document.createElement('div');
            overlay.className = 'activity-popup-overlay';
            
            const safeTitle = title || 'Detaljer';
            const safeText = text || '';
            
            overlay.innerHTML = `
                <div class="activity-popup">
                    <div class="activity-popup-header">
                        <span>${escapeHtml(safeTitle)}</span>
                        <button type="button" class="activity-popup-close" aria-label="Stäng">&times;</button>
                    </div>
                    <div class="activity-popup-body">${escapeHtml(safeText)}</div>
                </div>
            `;
            
            function close() {
                if (overlay.parentNode) {
                    overlay.parentNode.removeChild(overlay);
                }
            }
            
            overlay.addEventListener('click', (event) => {
                if (event.target === overlay || event.target.classList.contains('activity-popup-close')) {
                    close();
                }
            });
            
            document.addEventListener('keydown', function onKeyDown(e) {
                if (e.key === 'Escape') {
                    document.removeEventListener('keydown', onKeyDown);
                    close();
                }
            });
            
            document.body.appendChild(overlay);
        }
        
        function basename(path) {
            return path.split('/').pop().split('\\').pop();
        }
        
        async function fetchProjectName(projectId) {
            try {
                const response = await fetch(`../../api/projects.php?action=get&project_id=${encodeURIComponent(projectId)}`);
                const data = await response.json();
                if (data.success && data.project && data.project.name) {
                    return data.project.name;
                }
            } catch (error) {
                console.error('Error fetching project name:', error);
            }
            return null;
        }
        
        async function displayActivities(activities, fromDate, toDate) {
            const container = document.getElementById('timelineContainer');
            
            if (!activities || activities.length === 0) {
                container.innerHTML = `
                    <div class="timeline-empty">
                        <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" fill="currentColor">
                            <path d="M12 2C6.48 2 2 6.48 2 12s4.48 10 10 10 10-4.48 10-10S17.52 2 12 2zm-2 15l-5-5 1.41-1.41L10 14.17l7.59-7.59L19 8l-9 9z"/>
                        </svg>
                        <p>Ingen aktivitet hittades för valt datumintervall</p>
                        <p style="font-size: 12px; margin-top: 10px; opacity: 0.7;">${formatDate(fromDate)} - ${formatDate(toDate)}</p>
                    </div>
                `;
                return;
            }
            
            // Hämta projektnamn för aktiviteter där project_name är samma som project_id
            const projectNamePromises = [];
            const projectNameMap = new Map();
            
            activities.forEach(activity => {
                if (activity.project_id && activity.project_name && activity.project_name === activity.project_id) {
                    if (!projectNameMap.has(activity.project_id)) {
                        projectNameMap.set(activity.project_id, null);
                        projectNamePromises.push(
                            fetchProjectName(activity.project_id).then(name => {
                                if (name) {
                                    projectNameMap.set(activity.project_id, name);
                                    // Uppdatera aktiviteten
                                    activity.project_name = name;
                                }
                            })
                        );
                    }
                }
            });
            
            // Vänta på att alla projektnamn är hämtade
            if (projectNamePromises.length > 0) {
                await Promise.all(projectNamePromises);
            }
            
            // Gruppera efter datum
            const grouped = {};
            activities.forEach(activity => {
                const date = new Date(activity.timestamp);
                const dateKey = date.toLocaleDateString('sv-SE', { year: 'numeric', month: 'long', day: 'numeric' });
                
                if (!grouped[dateKey]) {
                    grouped[dateKey] = [];
                }
                grouped[dateKey].push(activity);
            });
            
            let html = `
                <div class="timeline-header">
                    <h2>Aktivitetslogg</h2>
                    <div class="subtitle">${activities.length} händelse${activities.length !== 1 ? 'r' : ''} • ${formatDate(fromDate)} - ${formatDate(toDate)}</div>
                </div>
            `;
            
            // Funktion för att parsa svenska datum
            function parseSwedishDate(dateStr) {
                // Försök matcha formatet "5 januari 2025"
                const months = {
                    'januari': 0, 'februari': 1, 'mars': 2, 'april': 3,
                    'maj': 4, 'juni': 5, 'juli': 6, 'augusti': 7,
                    'september': 8, 'oktober': 9, 'november': 10, 'december': 11
                };
                const match = dateStr.match(/(\d+)\s+(\w+)\s+(\d+)/);
                if (match) {
                    const day = parseInt(match[1]);
                    const month = months[match[2].toLowerCase()];
                    const year = parseInt(match[3]);
                    if (month !== undefined) {
                        return new Date(year, month, day);
                    }
                }
                // Fallback till standard parsing
                return new Date(dateStr);
            }
            
            // Sortera datum (nyaste först)
            // Konvertera svenska datumsträngar till Date-objekt för sortering
            const sortedDates = Object.keys(grouped).sort((a, b) => {
                const dateA = parseSwedishDate(a);
                const dateB = parseSwedishDate(b);
                return dateB - dateA;
            });
            
            sortedDates.forEach(dateKey => {
                html += `<div class="timeline-group">`;
                html += `<div class="timeline-group-date">${dateKey}</div>`;
                
                grouped[dateKey].forEach(activity => {
                    let typeClass = activity.type || 'default';
                    // Lägg till omdeploy-klass om det är en omdeploy
                    if (activity.type === 'version' && activity.is_new_version === false) {
                        typeClass += ' omdeploy';
                    }
                    const icon = getTypeIcon(activity.type);
                    const timeStr = formatTimestamp(activity.timestamp);
                    
                    let metaHtml = '';
                    // Visa projektnamn med ikon och projekt-ID
                    if (activity.project_name) {
                        const projectId = activity.project_id || '';
                        // Om project_name är samma som project_id, visa bara projekt-ID i parentes (projektnamn saknas)
                        // Annars visa projektnamn med ID i parentes
                        const projectDisplay = (activity.project_name === projectId)
                            ? `(${escapeHtml(projectId)})`
                            : (projectId 
                                ? `${escapeHtml(activity.project_name)} (${escapeHtml(projectId)})`
                                : escapeHtml(activity.project_name));
                        metaHtml += `<span style="display: inline-flex; align-items: center; gap: 4px;">
                            <svg xmlns="http://www.w3.org/2000/svg" width="14" height="14" viewBox="0 0 24 24" fill="currentColor" style="opacity: 0.7;">
                                <path d="M10 4H4c-1.11 0-2 .89-2 2v12c0 1.11.89 2 2 2h16c1.11 0 2-.89 2-2V8c0-1.11-.89-2-2-2h-8l-2-2z"/>
                            </svg>
                            ${projectDisplay}
                        </span>`;
                    }
                    // Visa användarnamn med ikon, och user_id som tooltip
                    if (activity.user_name || activity.user_id) {
                        // Alltid visa user_name om det finns, annars user_id
                        const displayName = activity.user_name || activity.user_id;
                        const userId = activity.user_id || '';
                        // Lägg user_id i tooltip om det finns
                        const tooltip = userId ? ` title="User ID: ${escapeHtml(userId)}"` : '';
                        metaHtml += `<span${tooltip} style="display: inline-flex; align-items: center; gap: 4px;">
                            <svg xmlns="http://www.w3.org/2000/svg" width="14" height="14" viewBox="0 0 24 24" fill="currentColor" style="opacity: 0.7;">
                                <path d="M12 12c2.21 0 4-1.79 4-4s-1.79-4-4-4-4 1.79-4 4 1.79 4 4 4zm0 2c-2.67 0-8 1.34-8 4v2h16v-2c0-2.66-5.33-4-8-4z"/>
                            </svg>
                            ${escapeHtml(displayName)}
                        </span>`;
                    }
                    
                    // För bilder i chatt, visa bild-URL som länk
                    let descriptionHtml = escapeHtml(activity.description);
                    if (activity.type === 'chat' && activity.message_type === 'image' && activity.image_url) {
                        const imageUrl = activity.image_url.startsWith('http') ? activity.image_url : '../../' + activity.image_url;
                        descriptionHtml = `<a href="${escapeHtml(imageUrl)}" target="_blank" style="color: #1a73e8; text-decoration: none;">${escapeHtml(activity.description || basename(activity.image_url))}</a>`;
                    }
                    
                    // För versionshändelser (ny version), gör beskrivningen klickbar för att visa full committext
                    if (activity.type === 'version' && activity.is_new_version !== false && activity.description) {
                        const fullText = activity.full_description || activity.description;
                        const titleText = activity.title || '';
                        descriptionHtml = `
                            <button type="button"
                                    class="timeline-description-link"
                                    data-full-text="${escapeHtml(fullText)}"
                                    data-title="${escapeHtml(titleText)}">
                                ${escapeHtml(activity.description)}
                            </button>
                        `;
                    }
                    
                    // För omdeploy (uppdaterat programvaran), dölj beskrivning
                    if (activity.type === 'version' && activity.is_new_version === false) {
                        descriptionHtml = '';
                    }
                    
                    // För versioner (ny version), skapa en extra rad med branch och commit-id
                    let versionInfoHtml = '';
                    if (activity.type === 'version' && activity.is_new_version !== false) {
                        const branch = activity.branch || '';
                        const commit = activity.commit || '';
                        if (branch || commit) {
                            versionInfoHtml = '<div class="timeline-version-info" style="font-size: 12px; color: #666; margin-top: 4px;">';
                            if (branch) {
                                versionInfoHtml += `<span style="background: #e3f2fd; padding: 2px 6px; border-radius: 3px; margin-right: 8px;">${escapeHtml(branch)}</span>`;
                            }
                            if (commit) {
                                versionInfoHtml += `<span style="font-family: monospace; background: #f0f0f0; padding: 2px 6px; border-radius: 3px;">${escapeHtml(commit)}</span>`;
                            }
                            versionInfoHtml += '</div>';
                        }
                    }
                    
                    html += `
                        <div class="timeline-item ${typeClass}">
                            <div class="timeline-icon">${icon}</div>
                            <div class="timeline-content">
                                <div class="timeline-title">${escapeHtml(activity.title)}</div>
                                <div class="timeline-description">${descriptionHtml}</div>
                                ${versionInfoHtml}
                                ${metaHtml ? '<div class="timeline-meta">' + metaHtml + '</div>' : ''}
                            </div>
                            <div class="timeline-time">${timeStr}</div>
                        </div>
                    `;
                });
                
                html += `</div>`;
            });
            
            container.innerHTML = html;
            
            // Koppla klick-händelser för versionsbeskrivningar
            container.querySelectorAll('.timeline-description-link').forEach(button => {
                button.addEventListener('click', () => {
                    const fullText = button.getAttribute('data-full-text') || '';
                    const title = button.getAttribute('data-title') || '';
                    showDescriptionPopup(title, fullText);
                });
            });
        }
        
        function getFilteredActivities(activities) {
            const showChat = document.getElementById('filterChat').checked;
            const showUpload = document.getElementById('filterUpload').checked;
            const showPackage = document.getElementById('filterPackage').checked;
            const showSystemMessage = document.getElementById('filterSystemMessage').checked;
            const showProject = document.getElementById('filterProject').checked;
            const showUser = document.getElementById('filterUser').checked;
            const showParticipant = document.getElementById('filterParticipant').checked;
            const showFile = document.getElementById('filterFile').checked;
            const showFileMove = document.getElementById('filterFileMove').checked;
            const showExternalUser = document.getElementById('filterExternalUser').checked;
            const showVersion = document.getElementById('filterVersion').checked;
            const showChatCleanup = document.getElementById('filterChatCleanup').checked;
            const showRepair = document.getElementById('filterRepair').checked;
            
            return activities.filter(activity => {
                switch(activity.type) {
                    case 'chat':
                        return showChat;
                    case 'upload':
                        return showUpload;
                    case 'package':
                        return showPackage;
                    case 'system_message':
                        return showSystemMessage;
                    case 'project':
                        return showProject;
                    case 'user':
                        return showUser;
                    case 'participant':
                        return showParticipant;
                    case 'file':
                        return showFile;
                    case 'file_move':
                        return showFileMove;
                    case 'external_user':
                        return showExternalUser;
                    case 'version':
                        return showVersion;
                    case 'chat_cleanup':
                        return showChatCleanup;
                    case 'repair':
                        return showRepair;
                    default:
                        return true;
                }
            });
        }
        
        // Ladda filter-inställningar från localStorage
        function loadFilterSettings() {
            const saved = localStorage.getItem('activityTimelineFilters');
            if (saved) {
                try {
                    const filters = JSON.parse(saved);
                    // Chattmeddelanden och Versioner ska inte vara checked som default
                    document.getElementById('filterChat').checked = filters.chat === true;
                    document.getElementById('filterUpload').checked = filters.upload !== false;
                    document.getElementById('filterPackage').checked = filters.package !== false;
                    document.getElementById('filterSystemMessage').checked = filters.systemMessage !== false;
                    document.getElementById('filterProject').checked = filters.project !== false;
                    document.getElementById('filterUser').checked = filters.user !== false;
                    document.getElementById('filterParticipant').checked = filters.participant !== false;
                    document.getElementById('filterFile').checked = filters.file !== false;
                    document.getElementById('filterFileMove').checked = filters.fileMove !== false;
                    document.getElementById('filterExternalUser').checked = filters.externalUser !== false;
                    document.getElementById('filterVersion').checked = filters.version === true;
                    document.getElementById('filterChatCleanup').checked = filters.chatCleanup !== false;
                    document.getElementById('filterRepair').checked = filters.repair !== false;
                } catch (e) {
                    // Ignorera fel vid parsing
                }
            }
        }
        
        // Spara filter-inställningar till localStorage
        function saveFilterSettings() {
            const filters = {
                chat: document.getElementById('filterChat').checked,
                upload: document.getElementById('filterUpload').checked,
                package: document.getElementById('filterPackage').checked,
                systemMessage: document.getElementById('filterSystemMessage').checked,
                project: document.getElementById('filterProject').checked,
                user: document.getElementById('filterUser').checked,
                participant: document.getElementById('filterParticipant').checked,
                file: document.getElementById('filterFile').checked,
                externalUser: document.getElementById('filterExternalUser').checked,
                version: document.getElementById('filterVersion').checked,
                chatCleanup: document.getElementById('filterChatCleanup').checked,
                repair: document.getElementById('filterRepair').checked
            };
            localStorage.setItem('activityTimelineFilters', JSON.stringify(filters));
        }
        
        // Ladda aktivitet när sidan laddas
        document.addEventListener('DOMContentLoaded', function() {
            loadFilterSettings();
            
            // Lägg till event listeners för filter-kryssrutor
            ['filterChat', 'filterUpload', 'filterPackage', 'filterSystemMessage', 
             'filterProject', 'filterUser', 'filterParticipant', 'filterFile', 'filterFileMove', 'filterExternalUser', 'filterVersion', 
             'filterChatCleanup', 'filterRepair'].forEach(id => {
                document.getElementById(id).addEventListener('change', function() {
                    saveFilterSettings();
                    // Om aktiviteter redan är laddade, filtrera om dem
                    const container = document.getElementById('timelineContainer');
                    if (container && !container.querySelector('.loading')) {
                        // Hämta ursprungliga aktiviteter från data-attribute eller ladda om
                        loadActivities();
                    }
                });
            });
            
            loadActivities();
        });
    </script>
</body>
</html>

