<?php
/**
 * Script för att retroaktivt tilldela bildnummer till alla befintliga bilder och videor
 * 
 * Detta script skannar alla projektmappar, hittar alla bilder och videor,
 * sorterar dem efter stored_at datum (äldsta först) och tilldelar sekventiella nummer.
 */

require_once __DIR__ . '/../admin_guard.php';
require_once __DIR__ . '/../../config.php';
require_once __DIR__ . '/../../lib/data_functions.php';

header('Content-Type: text/html; charset=utf-8');
?>
<!DOCTYPE html>
<html lang="sv">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Tilldela bildnummer</title>
    <style>
        body {
            font-family: Arial, sans-serif;
            max-width: 800px;
            margin: 50px auto;
            padding: 20px;
            background: #f5f5f5;
        }
        .container {
            background: white;
            padding: 30px;
            border-radius: 8px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        }
        h1 {
            color: #333;
            margin-top: 0;
        }
        .info {
            background: #e3f2fd;
            padding: 15px;
            border-radius: 4px;
            margin-bottom: 20px;
        }
        .result {
            margin-top: 20px;
            padding: 15px;
            border-radius: 4px;
        }
        .success {
            background: #d4edda;
            color: #155724;
            border: 1px solid #c3e6cb;
        }
        .error {
            background: #f8d7da;
            color: #721c24;
            border: 1px solid #f5c6cb;
        }
        .warning {
            background: #fff3cd;
            color: #856404;
            border: 1px solid #ffeaa7;
        }
        button {
            background: #007bff;
            color: white;
            border: none;
            padding: 12px 24px;
            font-size: 16px;
            border-radius: 4px;
            cursor: pointer;
        }
        button:hover {
            background: #0056b3;
        }
        button:disabled {
            background: #ccc;
            cursor: not-allowed;
        }
        .stats {
            margin-top: 20px;
            padding: 15px;
            background: #f8f9fa;
            border-radius: 4px;
        }
        .stats-item {
            margin: 5px 0;
        }
        .checkbox-group {
            margin: 20px 0;
        }
        .checkbox-group label {
            display: flex;
            align-items: center;
            cursor: pointer;
            font-size: 14px;
        }
        .checkbox-group input[type="checkbox"] {
            margin-right: 8px;
            width: 18px;
            height: 18px;
            cursor: pointer;
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>Tilldela bildnummer retroaktivt</h1>
        
        <div class="info">
            <strong>Information:</strong><br>
            Detta script kommer att:
            <ul>
                <li>Skanna alla projektmappar efter bilder och videor</li>
                <li>Sortera dem efter uppladdningsdatum (äldsta först)</li>
                <li>Tilldela sekventiella nummer från 1 och uppåt</li>
                <li>Uppdatera JSON-metadatafilerna med bildnummer</li>
            </ul>
            <strong>Varning:</strong> Om bilder redan har nummer kommer de att behållas. Endast bilder utan nummer kommer att få nya nummer. Om du kryssar i "Initiera om alla bildnummer från 1" kommer alla bildnummer att omnumreras från 1 och alla JSON-metadatafiler kommer att uppdateras.
        </div>

        <?php
        if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['assign_numbers'])) {
            // Läsa checkbox-värdet för omnumrering
            $resetAll = isset($_POST['reset_all']) && $_POST['reset_all'] === '1';
            
            $uploadRoot = __DIR__ . '/../../' . rtrim(UPLOAD_ROOT, '/\\');
            $allFiles = [];
            
            // Stödda format
            $supportedImageFormats = ['jpg', 'jpeg', 'png', 'webp', 'heic', 'heif'];
            $supportedVideoFormats = ['mp4', 'webm', 'mov', 'avi'];
            $allSupportedFormats = array_merge($supportedImageFormats, $supportedVideoFormats);
            
            // Skanna alla projektmappar
            if (is_dir($uploadRoot)) {
                $projectDirs = glob($uploadRoot . '/*', GLOB_ONLYDIR);
                
                foreach ($projectDirs as $projectDir) {
                    $projectId = basename($projectDir);
                    
                    // Hämta alla JSON-filer (metadata)
                    $jsonFiles = glob($projectDir . '/*.json');
                    
                    foreach ($jsonFiles as $jsonFile) {
                        $metadata = json_decode(file_get_contents($jsonFile), true);
                        if (!$metadata) {
                            continue;
                        }
                        
                        // Hitta motsvarande bild/video-fil
                        $actualFile = null;
                        foreach ($allSupportedFormats as $format) {
                            $testFile = str_replace('.json', '.' . $format, $jsonFile);
                            if (file_exists($testFile)) {
                                $actualFile = $testFile;
                                break;
                            }
                        }
                        
                        if ($actualFile) {
                            // Kontrollera om bilden redan har ett nummer
                            $hasNumber = isset($metadata['image_number']) && is_numeric($metadata['image_number']);
                            
                            // Hämta stored_at datum för sortering
                            $storedAt = $metadata['stored_at'] ?? date('c', filemtime($actualFile));
                            
                            $allFiles[] = [
                                'json_path' => $jsonFile,
                                'file_path' => $actualFile,
                                'project_id' => $projectId,
                                'stored_at' => $storedAt,
                                'has_number' => $hasNumber,
                                'current_number' => $hasNumber ? (int)$metadata['image_number'] : null,
                                'metadata' => $metadata
                            ];
                        }
                    }
                }
            }
            
            // Sortera efter stored_at (äldsta först)
            usort($allFiles, function($a, $b) {
                $timeA = strtotime($a['stored_at']);
                $timeB = strtotime($b['stored_at']);
                if ($timeA === false) $timeA = 0;
                if ($timeB === false) $timeB = 0;
                return $timeA - $timeB;
            });
            
            // Separera filer med och utan nummer
            $filesWithNumbers = [];
            $filesWithoutNumbers = [];
            
            if ($resetAll) {
                // Om omnumrering är aktiverad, behandla alla som utan nummer
                $filesWithoutNumbers = $allFiles;
                $filesWithNumbers = [];
                $maxNumber = 0;
            } else {
                // Annars: behåll nuvarande logik
                foreach ($allFiles as $file) {
                    if ($file['has_number']) {
                        $filesWithNumbers[] = $file;
                    } else {
                        $filesWithoutNumbers[] = $file;
                    }
                }
                
                // Sortera filer med nummer efter nummer
                usort($filesWithNumbers, function($a, $b) {
                    return $a['current_number'] - $b['current_number'];
                });
                
                // Hitta högsta nummer
                $maxNumber = 0;
                foreach ($filesWithNumbers as $file) {
                    if ($file['current_number'] > $maxNumber) {
                        $maxNumber = $file['current_number'];
                    }
                }
            }
            
            // Initiera eller uppdatera image_numbers.json med rätt startvärde
            $numbersFile = __DIR__ . '/../../data/image_numbers.json';
            $nextNumber = $maxNumber + 1;
            $numbersData = ['next_number' => $nextNumber];
            file_put_contents($numbersFile, json_encode($numbersData, JSON_PRETTY_PRINT));
            
            // Tilldela nummer till filer utan nummer
            $assigned = 0;
            $skipped = 0;
            $errors = 0;
            
            foreach ($filesWithoutNumbers as $file) {
                // Om omnumrering är aktiverad, ta bort befintligt nummer från metadata först
                if ($resetAll && isset($file['metadata']['image_number'])) {
                    unset($file['metadata']['image_number']);
                    // Uppdatera metadata-filen
                    file_put_contents($file['json_path'], json_encode($file['metadata'], JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
                }
                
                $imageNumber = assignImageNumber($file['file_path'], $file['json_path']);
                if ($imageNumber !== false) {
                    $assigned++;
                } else {
                    $errors++;
                }
            }
            
            $skipped = count($filesWithNumbers);
            
            echo '<div class="result success">';
            echo '<h2>Klar!</h2>';
            echo '<div class="stats">';
            echo '<div class="stats-item"><strong>Totalt antal bilder/videor:</strong> ' . count($allFiles) . '</div>';
            if ($resetAll) {
                echo '<div class="stats-item"><strong>Alla bilder omnumrerades från 1:</strong> ' . $assigned . '</div>';
            } else {
                echo '<div class="stats-item"><strong>Redan hade nummer:</strong> ' . $skipped . '</div>';
                echo '<div class="stats-item"><strong>Nya nummer tilldelade:</strong> ' . $assigned . '</div>';
            }
            if ($errors > 0) {
                echo '<div class="stats-item"><strong>Fel:</strong> ' . $errors . '</div>';
            }
            echo '</div>';
            echo '</div>';
        } else {
            ?>
            <form method="POST">
                <div class="checkbox-group">
                    <label>
                        <input type="checkbox" name="reset_all" value="1">
                        Initiera om alla bildnummer från 1 (uppdaterar alla bilders JSON-metadatafiler)
                    </label>
                </div>
                <button type="submit" name="assign_numbers">Tilldela bildnummer</button>
            </form>
            <?php
        }
        ?>
    </div>
</body>
</html>

