<?php
require_once __DIR__ . '/../admin_guard.php';
require_once __DIR__ . '/../../config.php';
require_once __DIR__ . '/../../lib/participant_helper.php';
require_once __DIR__ . '/../../lib/mapping_helper.php';
require_once __DIR__ . '/../../lib/data_functions.php';

// Sökvägar
$rootDir = dirname(dirname(__DIR__));
$uploadRoot = $rootDir . '/' . rtrim(UPLOAD_ROOT, '/\\');
$chatsDir = $rootDir . '/chats';
$projectsFile = $rootDir . '/data/projects.json';
$userIdsFile = $rootDir . '/data/userids.json';

// Funktion för att logga reparation-händelser
function logRepairEvent($repairType, $userId, $changesCount, $description) {
    $logFile = __DIR__ . '/../../data/repair_log.json';
    $logDir = dirname($logFile);
    
    // Skapa data-mappen om den inte finns
    if (!is_dir($logDir)) {
        @mkdir($logDir, 0755, true);
    }
    
    // Läs befintliga händelser
    $events = [];
    if (file_exists($logFile)) {
        $content = @file_get_contents($logFile);
        if ($content !== false) {
            $decoded = json_decode($content, true);
            if (is_array($decoded)) {
                $events = $decoded;
            }
        }
    }
    
    // Lägg till ny händelse
    $event = [
        'timestamp' => date('c'),
        'user_id' => $userId,
        'repair_type' => $repairType,
        'changes_count' => $changesCount,
        'description' => $description
    ];
    
    $events[] = $event;
    
    // Begränsa till senaste 1000 händelser
    if (count($events) > 1000) {
        $events = array_slice($events, -1000);
    }
    
    // Spara tillbaka
    @file_put_contents($logFile, json_encode($events, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE), LOCK_EX);
}

// Funktion för att läsa chatmeddelanden
function readChatMessages($projectId, $chatsDir) {
    $chatFile = $chatsDir . '/' . $projectId . '/messages.json';
    if (!file_exists($chatFile)) {
        return [];
    }
    $content = file_get_contents($chatFile);
    if ($content === false) {
        return [];
    }
    $messages = json_decode($content, true);
    return is_array($messages) ? $messages : [];
}

// Funktion för att hitta alla refererade filer i JSON
function getAllReferencedFiles($uploadRoot, $chatsDir) {
    $referenced = [];
    
    // Hämta alla refererade filer från uploads (metadata-filer)
    if (is_dir($uploadRoot)) {
        $projectDirs = glob($uploadRoot . '/*', GLOB_ONLYDIR);
        foreach ($projectDirs as $projectDir) {
            $jsonFiles = glob($projectDir . '/*.json');
            foreach ($jsonFiles as $jsonFile) {
                $metadata = json_decode(file_get_contents($jsonFile), true);
                if ($metadata) {
                    // Lägg till stored_file
                    if (isset($metadata['stored_file'])) {
                        $referenced[basename($metadata['stored_file'])] = true;
                    }
                    // Lägg till thumb_file
                    if (isset($metadata['thumb_file'])) {
                        $referenced[basename($metadata['thumb_file'])] = true;
                    }
                }
            }
        }
    }
    
    // Hämta alla refererade filer från chats (messages.json)
    if (is_dir($chatsDir)) {
        $chatDirs = glob($chatsDir . '/*', GLOB_ONLYDIR);
        foreach ($chatDirs as $chatDir) {
            $messagesFile = $chatDir . '/messages.json';
            if (file_exists($messagesFile)) {
                $messages = json_decode(file_get_contents($messagesFile), true);
                if (is_array($messages)) {
                    foreach ($messages as $message) {
                        if (isset($message['type']) && $message['type'] === 'image' && isset($message['media_file'])) {
                            $referenced[basename($message['media_file'])] = true;
                        }
                    }
                }
            }
        }
    }
    
    return $referenced;
}

// Läsa projektlista
function readProjectsList($projectsFile) {
    if (!file_exists($projectsFile)) return [];
    $content = file_get_contents($projectsFile);
    $projects = json_decode($content, true);
    return is_array($projects) ? $projects : [];
}

function saveProjectsList($projectsFile, $projects) {
    $json = json_encode($projects, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
    return file_put_contents($projectsFile, $json) !== false;
}

// Läsa användare (user_id => full_name)
function readUsersMap($userIdsFile) {
    if (!file_exists($userIdsFile)) return [];
    $content = file_get_contents($userIdsFile);
    $users = json_decode($content, true);
    if (!is_array($users)) return [];
    $map = [];
    foreach ($users as $user) {
        $uid = $user['user_id'] ?? '';
        if ($uid !== '') {
            $map[$uid] = $user['full_name'] ?? $uid;
        }
    }
    return $map;
}

// Funktion för att hitta backup-filer som inte refereras
function findBackupFiles($uploadRoot, $chatsDir) {
    $backupExtensions = ['.bak', '.old', '.backup', '.tmp', '.temp'];
    $referenced = getAllReferencedFiles($uploadRoot, $chatsDir);
    $backupFiles = [];
    
    // Sök i uploads
    if (is_dir($uploadRoot)) {
        $projectDirs = glob($uploadRoot . '/*', GLOB_ONLYDIR);
        foreach ($projectDirs as $projectDir) {
            $files = scandir($projectDir);
            foreach ($files as $file) {
                if ($file === '.' || $file === '..') continue;
                $filePath = $projectDir . '/' . $file;
                if (!is_file($filePath)) continue;
                
                $ext = strtolower(pathinfo($file, PATHINFO_EXTENSION));
                $isBackup = false;
                foreach ($backupExtensions as $backupExt) {
                    if (str_ends_with(strtolower($file), $backupExt)) {
                        $isBackup = true;
                        break;
                    }
                }
                
                if ($isBackup && !isset($referenced[$file])) {
                    $backupFiles[] = [
                        'path' => $filePath,
                        'name' => $file,
                        'size' => filesize($filePath),
                        'project' => basename($projectDir)
                    ];
                }
            }
        }
    }
    
    // Sök i chats
    if (is_dir($chatsDir)) {
        $chatDirs = glob($chatsDir . '/*', GLOB_ONLYDIR);
        foreach ($chatDirs as $chatDir) {
            $files = scandir($chatDir);
            foreach ($files as $file) {
                if ($file === '.' || $file === '..' || $file === 'messages.json') continue;
                $filePath = $chatDir . '/' . $file;
                if (!is_file($filePath)) continue;
                
                $ext = strtolower(pathinfo($file, PATHINFO_EXTENSION));
                $isBackup = false;
                foreach ($backupExtensions as $backupExt) {
                    if (str_ends_with(strtolower($file), $backupExt)) {
                        $isBackup = true;
                        break;
                    }
                }
                
                if ($isBackup && !isset($referenced[$file])) {
                    $backupFiles[] = [
                        'path' => $filePath,
                        'name' => $file,
                        'size' => filesize($filePath),
                        'project' => basename($chatDir)
                    ];
                }
            }
        }
    }
    
    return $backupFiles;
}

// Funktion för att hitta tumnaglar utan originalbild
function findOrphanedThumbnails($uploadRoot, $chatsDir) {
    $orphaned = [];
    
    // Sök i uploads
    if (is_dir($uploadRoot)) {
        $projectDirs = glob($uploadRoot . '/*', GLOB_ONLYDIR);
        foreach ($projectDirs as $projectDir) {
            $files = scandir($projectDir);
            foreach ($files as $file) {
                if ($file === '.' || $file === '..') continue;
                if (!str_ends_with(strtolower($file), '-thumb.jpg')) continue;
                
                $filePath = $projectDir . '/' . $file;
                if (!is_file($filePath)) continue;
                
                // Hitta originalbild (ta bort -thumb.jpg)
                $originalName = preg_replace('/-thumb\.jpg$/i', '', $file);
                $supportedFormats = ['jpg', 'jpeg', 'png', 'webp', 'heic', 'heif'];
                $originalFound = false;
                
                foreach ($supportedFormats as $format) {
                    $originalPath = $projectDir . '/' . $originalName . '.' . $format;
                    if (file_exists($originalPath)) {
                        $originalFound = true;
                        break;
                    }
                }
                
                if (!$originalFound) {
                    $orphaned[] = [
                        'path' => $filePath,
                        'name' => $file,
                        'size' => filesize($filePath),
                        'project' => basename($projectDir)
                    ];
                }
            }
        }
    }
    
    // Sök i chats
    if (is_dir($chatsDir)) {
        $chatDirs = glob($chatsDir . '/*', GLOB_ONLYDIR);
        foreach ($chatDirs as $chatDir) {
            $files = scandir($chatDir);
            foreach ($files as $file) {
                if ($file === '.' || $file === '..' || $file === 'messages.json') continue;
                if (!str_ends_with(strtolower($file), '-thumb.jpg')) continue;
                
                $filePath = $chatDir . '/' . $file;
                if (!is_file($filePath)) continue;
                
                // Hitta originalbild
                $originalName = preg_replace('/-thumb\.jpg$/i', '', $file);
                $supportedFormats = ['jpg', 'jpeg', 'png', 'webp', 'heic', 'heif'];
                $originalFound = false;
                
                foreach ($supportedFormats as $format) {
                    $originalPath = $chatDir . '/' . $originalName . '.' . $format;
                    if (file_exists($originalPath)) {
                        $originalFound = true;
                        break;
                    }
                }
                
                if (!$originalFound) {
                    $orphaned[] = [
                        'path' => $filePath,
                        'name' => $file,
                        'size' => filesize($filePath),
                        'project' => basename($chatDir)
                    ];
                }
            }
        }
    }
    
    return $orphaned;
}

// Funktion för att hitta projektalbum-bilder som inte refereras
function findOrphanedProjectImages($uploadRoot, $chatsDir) {
    $orphaned = [];
    $referenced = getAllReferencedFiles($uploadRoot, $chatsDir);
    $supportedFormats = ['jpg', 'jpeg', 'png', 'webp', 'heic', 'heif', 'mp4', 'webm', 'mov', 'avi'];
    
    if (is_dir($uploadRoot)) {
        $projectDirs = glob($uploadRoot . '/*', GLOB_ONLYDIR);
        foreach ($projectDirs as $projectDir) {
            $projectId = basename($projectDir);
            $files = scandir($projectDir);
            
            foreach ($files as $file) {
                if ($file === '.' || $file === '..') continue;
                if (str_ends_with(strtolower($file), '-thumb.jpg')) continue; // Hoppa över tumnaglar
                if (str_ends_with(strtolower($file), '.json')) continue; // Hoppa över metadata
                
                $filePath = $projectDir . '/' . $file;
                if (!is_file($filePath)) continue;
                
                $ext = strtolower(pathinfo($file, PATHINFO_EXTENSION));
                if (!in_array($ext, $supportedFormats)) continue;
                
                // Kontrollera om filen refereras
                if (!isset($referenced[$file])) {
                    // Kontrollera om det finns en JSON-metadatafil som refererar till denna bild
                    $baseName = pathinfo($file, PATHINFO_FILENAME);
                    $jsonPath = $projectDir . '/' . $baseName . '.json';
                    $isReferenced = false;
                    
                    if (file_exists($jsonPath)) {
                        $metadata = json_decode(file_get_contents($jsonPath), true);
                        if ($metadata && isset($metadata['stored_file']) && basename($metadata['stored_file']) === $file) {
                            $isReferenced = true;
                        }
                    }
                    
                    // Kontrollera även om bilden refereras i chatten
                    $messages = readChatMessages($projectId, $chatsDir);
                    foreach ($messages as $message) {
                        if (isset($message['type']) && $message['type'] === 'image' && isset($message['media_file']) && basename($message['media_file']) === $file) {
                            $isReferenced = true;
                            break;
                        }
                    }
                    
                    if (!$isReferenced) {
                        $orphaned[] = [
                            'path' => $filePath,
                            'name' => $file,
                            'size' => filesize($filePath),
                            'project' => $projectId
                        ];
                    }
                }
            }
        }
    }
    
    return $orphaned;
}

// Funktion för att hitta projektchat-bilder som inte refereras
function findOrphanedChatImages($chatsDir) {
    $orphaned = [];
    $supportedFormats = ['jpg', 'jpeg', 'png', 'webp', 'heic', 'heif', 'mp4', 'webm', 'mov', 'avi'];
    
    if (is_dir($chatsDir)) {
        $chatDirs = glob($chatsDir . '/*', GLOB_ONLYDIR);
        foreach ($chatDirs as $chatDir) {
            $projectId = basename($chatDir);
            $messagesFile = $chatDir . '/messages.json';
            
            // Hämta alla refererade bilder från messages.json
            $referencedFiles = [];
            if (file_exists($messagesFile)) {
                $messages = json_decode(file_get_contents($messagesFile), true);
                if (is_array($messages)) {
                    foreach ($messages as $message) {
                        if (isset($message['type']) && $message['type'] === 'image' && isset($message['media_file'])) {
                            $referencedFiles[basename($message['media_file'])] = true;
                        }
                    }
                }
            }
            
            // Sök efter bilder i chat-mappen
            $files = scandir($chatDir);
            foreach ($files as $file) {
                if ($file === '.' || $file === '..' || $file === 'messages.json') continue;
                if (str_ends_with(strtolower($file), '-thumb.jpg')) continue; // Hoppa över tumnaglar
                
                $filePath = $chatDir . '/' . $file;
                if (!is_file($filePath)) continue;
                
                $ext = strtolower(pathinfo($file, PATHINFO_EXTENSION));
                if (!in_array($ext, $supportedFormats)) continue;
                
                // Kontrollera om filen refereras i messages.json
                if (!isset($referencedFiles[$file])) {
                    $orphaned[] = [
                        'path' => $filePath,
                        'name' => $file,
                        'size' => filesize($filePath),
                        'project' => $projectId
                    ];
                }
            }
        }
    }
    
    return $orphaned;
}

// Metadata som saknar bildfil (stored_file saknas)
function findMetadataMissingImages($uploadRoot) {
    $missing = [];
    $supportedFormats = ['jpg', 'jpeg', 'png', 'webp', 'heic', 'heif', 'mp4', 'webm', 'mov', 'avi'];
    if (!is_dir($uploadRoot)) return $missing;
    $projectDirs = glob($uploadRoot . '/*', GLOB_ONLYDIR);
    foreach ($projectDirs as $projectDir) {
        $projectId = basename($projectDir);
        $jsonFiles = glob($projectDir . '/*.json');
        foreach ($jsonFiles as $jsonFile) {
            $metadata = json_decode(file_get_contents($jsonFile), true);
            if (!$metadata) continue;
            if (!empty($metadata['stored_file'])) {
                $stored = basename($metadata['stored_file']);
                $storedPath = $projectDir . '/' . $stored;
                $ext = strtolower(pathinfo($storedPath, PATHINFO_EXTENSION));
                if (!file_exists($storedPath) && in_array($ext, $supportedFormats)) {
                    $missing[] = [
                        'path' => $jsonFile,
                        'name' => basename($jsonFile),
                        'project' => $projectId,
                        'stored_file' => $stored,
                        'size' => @filesize($jsonFile) ?: 0
                    ];
                }
            }
        }
    }
    return $missing;
}

// Thumb-filer som saknas enligt metadata
function findMetadataMissingThumbs($uploadRoot) {
    $missing = [];
    if (!is_dir($uploadRoot)) return $missing;
    $projectDirs = glob($uploadRoot . '/*', GLOB_ONLYDIR);
    foreach ($projectDirs as $projectDir) {
        $projectId = basename($projectDir);
        $jsonFiles = glob($projectDir . '/*.json');
        foreach ($jsonFiles as $jsonFile) {
            $metadata = json_decode(file_get_contents($jsonFile), true);
            if (!$metadata) continue;
            if (!empty($metadata['thumb_file'])) {
                $thumb = basename($metadata['thumb_file']);
                $thumbPath = $projectDir . '/' . $thumb;
                if (!file_exists($thumbPath)) {
                    $missing[] = [
                        'path' => $jsonFile,
                        'name' => basename($jsonFile),
                        'project' => $projectId,
                        'thumb_file' => $thumb,
                        'size' => @filesize($jsonFile) ?: 0
                    ];
                }
            }
        }
    }
    return $missing;
}

// Projektmappar som saknas i projects.json
function findDirsWithoutProject($projectsFile, $uploadRoot, $chatsDir) {
    $missing = [];
    $projects = readProjectsList($projectsFile);
    $projectIds = [];
    foreach ($projects as $p) {
        if (!empty($p['id'])) $projectIds[$p['id']] = true;
    }
    $scanDirs = [];
    if (is_dir($uploadRoot)) $scanDirs[] = $uploadRoot;
    if (is_dir($chatsDir)) $scanDirs[] = $chatsDir;
    foreach ($scanDirs as $dir) {
        $dirs = glob($dir . '/*', GLOB_ONLYDIR);
        foreach ($dirs as $d) {
            $pid = basename($d);
            if (!isset($projectIds[$pid])) {
                $missing[] = [
                    'path' => $d,
                    'project' => $pid,
                    'location' => $dir === $uploadRoot ? 'uploads' : 'chats'
                ];
            }
        }
    }
    return $missing;
}

// Deltagare i projects.json som saknas i userids.json
function findParticipantsMissingUsers($projectsFile, $userIdsFile) {
    $missing = [];
    $projects = readProjectsList($projectsFile);
    $users = readUsersMap($userIdsFile);
    foreach ($projects as $p) {
        $pid = $p['id'] ?? '';
        if (empty($pid)) continue;
        $participants = $p['participants'] ?? [];
        foreach ($participants as $part) {
            $uid = $part['participant_id'] ?? '';
            if ($uid !== '' && !isset($users[$uid])) {
                $missing[] = [
                    'project' => $pid,
                    'participant_id' => $uid
                ];
            }
        }
    }
    return $missing;
}

// Meddelanden med participant_id som saknas i userids.json
function findMessagesMissingUsers($chatsDir, $userIdsFile) {
    $missing = [];
    $users = readUsersMap($userIdsFile);
    if (!is_dir($chatsDir)) return $missing;
    $chatDirs = glob($chatsDir . '/*', GLOB_ONLYDIR);
    foreach ($chatDirs as $chatDir) {
        $projectId = basename($chatDir);
        $messagesFile = $chatDir . '/messages.json';
        if (!file_exists($messagesFile)) continue;
        $messages = json_decode(file_get_contents($messagesFile), true);
        if (!is_array($messages)) continue;
        foreach ($messages as $msg) {
            $pid = $msg['participant_id'] ?? '';
            if ($pid !== '' && !isset($users[$pid])) {
                $missing[] = [
                    'project' => $projectId,
                    'participant_id' => $pid,
                    'message_id' => $msg['id'] ?? ''
                ];
            }
        }
    }
    return $missing;
}

// Chat-meddelanden som refererar till bilder som saknas
function findChatMessagesMissingImages($chatsDir) {
    $missing = [];
    $supportedFormats = ['jpg', 'jpeg', 'png', 'webp', 'heic', 'heif', 'mp4', 'webm', 'mov', 'avi'];
    if (!is_dir($chatsDir)) return $missing;
    $chatDirs = glob($chatsDir . '/*', GLOB_ONLYDIR);
    foreach ($chatDirs as $chatDir) {
        $projectId = basename($chatDir);
        $messagesFile = $chatDir . '/messages.json';
        if (!file_exists($messagesFile)) continue;
        $messages = json_decode(file_get_contents($messagesFile), true);
        if (!is_array($messages)) continue;
        foreach ($messages as $msg) {
            if (($msg['type'] ?? '') === 'image' && !empty($msg['media_file'])) {
                $fileName = basename($msg['media_file']);
                $filePath = $chatDir . '/' . $fileName;
                $ext = strtolower(pathinfo($filePath, PATHINFO_EXTENSION));
                if (in_array($ext, $supportedFormats) && !file_exists($filePath)) {
                    $missing[] = [
                        'project' => $projectId,
                        'media_file' => $fileName,
                        'message_id' => $msg['id'] ?? ''
                    ];
                }
            }
        }
    }
    return $missing;
}

// Filer utan metadata (bild/video utan motsvarande .json)
function findFilesMissingMetadata($uploadRoot) {
    $missing = [];
    $supportedFormats = ['jpg', 'jpeg', 'png', 'webp', 'heic', 'heif', 'mp4', 'webm', 'mov', 'avi'];
    if (!is_dir($uploadRoot)) return $missing;
    $projectDirs = glob($uploadRoot . '/*', GLOB_ONLYDIR);
    foreach ($projectDirs as $projectDir) {
        $projectId = basename($projectDir);
        $files = scandir($projectDir);
        foreach ($files as $file) {
            if ($file === '.' || $file === '..') continue;
            if (str_ends_with(strtolower($file), '-thumb.jpg')) continue;
            if (str_ends_with(strtolower($file), '.json')) continue;
            $filePath = $projectDir . '/' . $file;
            if (!is_file($filePath)) continue;
            $ext = strtolower(pathinfo($file, PATHINFO_EXTENSION));
            if (!in_array($ext, $supportedFormats)) continue;
            $base = pathinfo($file, PATHINFO_FILENAME);
            $jsonPath = $projectDir . '/' . $base . '.json';
            if (!file_exists($jsonPath)) {
                $missing[] = [
                    'path' => $filePath,
                    'name' => $file,
                    'project' => $projectId,
                    'size' => filesize($filePath)
                ];
            }
        }
    }
    return $missing;
}

// Beräkna storlek av katalog rekursivt
function getDirSize($dir) {
    $size = 0;
    if (!is_dir($dir)) return 0;
    $items = scandir($dir);
    foreach ($items as $item) {
        if ($item === '.' || $item === '..') continue;
        $path = $dir . '/' . $item;
        if (is_file($path)) {
            $size += filesize($path);
        } elseif (is_dir($path)) {
            $size += getDirSize($path);
        }
    }
    return $size;
}

// Ta bort katalog rekursivt
function rrmdir($dir) {
    if (!is_dir($dir)) return;
    $items = scandir($dir);
    foreach ($items as $item) {
        if ($item === '.' || $item === '..') continue;
        $path = $dir . '/' . $item;
        if (is_dir($path)) {
            rrmdir($path);
        } else {
            @unlink($path);
        }
    }
    @rmdir($dir);
}

// Funktion för att hitta profilbilder som saknas
function findMissingProfileImages($userIdsFile) {
    $missing = [];
    if (!file_exists($userIdsFile)) return $missing;
    
    $content = file_get_contents($userIdsFile);
    if ($content === false) return $missing;
    
    $users = json_decode($content, true);
    if (!is_array($users)) return $missing;
    
    $dataDir = dirname($userIdsFile);
    
    foreach ($users as $index => $user) {
        $userId = $user['user_id'] ?? '';
        $profileImage = $user['profile_image'] ?? null;
        
        if (!empty($profileImage)) {
            // Säkerställ att filnamnet är säkert
            $profileImage = basename($profileImage);
            // Kontrollera att filnamnet matchar förväntat format
            if (preg_match('/^user_[a-zA-Z0-9\-_]+\.jpg$/', $profileImage)) {
                $imagePath = $dataDir . '/' . $profileImage;
                
                if (!file_exists($imagePath) || !is_file($imagePath)) {
                    $missing[] = [
                        'user_id' => $userId,
                        'full_name' => $user['full_name'] ?? 'Okänt namn',
                        'profile_image' => $profileImage,
                        'index' => $index
                    ];
                }
            }
        }
    }
    
    return $missing;
}

// Kontrollera chat_mapping.json och project_mapping.json för ogiltiga referenser
function findMappingIssues($projectsFile, $userIdsFile, $chatsDir) {
    $issues = [
        'chat_mapping' => [
            'missing_projects' => [],
            'missing_users' => [],
            'missing_chats' => []
        ],
        'project_mapping' => [
            'missing_projects' => [],
            'missing_users' => []
        ]
    ];
    
    // Läs projekt och användare
    $projects = readProjectsList($projectsFile);
    $projectIds = [];
    foreach ($projects as $p) {
        if (!empty($p['id'])) {
            $projectIds[$p['id']] = true;
        }
    }
    
    $users = readUsersMap($userIdsFile);
    
    // Kontrollera chat_mapping.json
    $chatMappingFile = dirname($projectsFile) . '/chat_mapping.json';
    if (file_exists($chatMappingFile)) {
        $chatMapping = readChatMapping(null, $chatMappingFile);
        
        foreach ($chatMapping as $projectId => $chatters) {
            // Kontrollera om projektet finns
            if (!isset($projectIds[$projectId])) {
                $issues['chat_mapping']['missing_projects'][] = [
                    'project_id' => $projectId,
                    'type' => 'chat_mapping'
                ];
            }
            
            // Kontrollera om chat-mappen finns
            if (isset($projectIds[$projectId]) && is_dir($chatsDir)) {
                $chatDir = $chatsDir . '/' . $projectId;
                if (!is_dir($chatDir)) {
                    $issues['chat_mapping']['missing_chats'][] = [
                        'project_id' => $projectId,
                        'type' => 'chat_mapping'
                    ];
                }
            }
            
            // Kontrollera om användarna finns
            if (is_array($chatters)) {
                foreach ($chatters as $chatter) {
                    $participantId = $chatter['participant_id'] ?? '';
                    if ($participantId !== '' && !isset($users[$participantId])) {
                        $issues['chat_mapping']['missing_users'][] = [
                            'project_id' => $projectId,
                            'participant_id' => $participantId,
                            'type' => 'chat_mapping'
                        ];
                    }
                }
            }
        }
    }
    
    // Kontrollera project_mapping.json
    $projectMappingFile = dirname($projectsFile) . '/project_mapping.json';
    if (file_exists($projectMappingFile)) {
        $projectMapping = readProjectMapping(null, $projectMappingFile);
        
        foreach ($projectMapping as $projectId => $participants) {
            // Kontrollera om projektet finns
            if (!isset($projectIds[$projectId])) {
                $issues['project_mapping']['missing_projects'][] = [
                    'project_id' => $projectId,
                    'type' => 'project_mapping'
                ];
            }
            
            // Kontrollera om användarna finns
            if (is_array($participants)) {
                foreach ($participants as $participant) {
                    $participantId = $participant['participant_id'] ?? '';
                    if ($participantId !== '' && !isset($users[$participantId])) {
                        $issues['project_mapping']['missing_users'][] = [
                            'project_id' => $projectId,
                            'participant_id' => $participantId,
                            'type' => 'project_mapping'
                        ];
                    }
                }
            }
        }
    }
    
    return $issues;
}

// Funktion för att formatera filstorlek
function formatFileSize($bytes) {
    if ($bytes < 1024) return $bytes . ' B';
    if ($bytes < 1024 * 1024) return round($bytes / 1024, 2) . ' KB';
    if ($bytes < 1024 * 1024 * 1024) return round($bytes / (1024 * 1024), 2) . ' MB';
    return round($bytes / (1024 * 1024 * 1024), 2) . ' GB';
}

// Funktion för att tilldela bildnummer retroaktivt
function assignImageNumbers($uploadRoot) {
    $allFiles = [];
    $supportedImageFormats = ['jpg', 'jpeg', 'png', 'webp', 'heic', 'heif'];
    $supportedVideoFormats = ['mp4', 'webm', 'mov', 'avi'];
    $allSupportedFormats = array_merge($supportedImageFormats, $supportedVideoFormats);
    
    // Skanna alla projektmappar
    if (is_dir($uploadRoot)) {
        $projectDirs = glob($uploadRoot . '/*', GLOB_ONLYDIR);
        
        foreach ($projectDirs as $projectDir) {
            $projectId = basename($projectDir);
            
            // Hämta alla JSON-filer (metadata)
            $jsonFiles = glob($projectDir . '/*.json');
            
            foreach ($jsonFiles as $jsonFile) {
                $metadata = json_decode(file_get_contents($jsonFile), true);
                if (!$metadata) {
                    continue;
                }
                
                // Hitta motsvarande bild/video-fil
                $actualFile = null;
                foreach ($allSupportedFormats as $format) {
                    $testFile = str_replace('.json', '.' . $format, $jsonFile);
                    if (file_exists($testFile)) {
                        $actualFile = $testFile;
                        break;
                    }
                }
                
                if ($actualFile) {
                    // Kontrollera om bilden redan har ett nummer
                    $hasNumber = isset($metadata['image_number']) && is_numeric($metadata['image_number']);
                    
                    // Hämta stored_at datum för sortering
                    $storedAt = $metadata['stored_at'] ?? date('c', filemtime($actualFile));
                    
                    $allFiles[] = [
                        'json_path' => $jsonFile,
                        'file_path' => $actualFile,
                        'project_id' => $projectId,
                        'stored_at' => $storedAt,
                        'has_number' => $hasNumber,
                        'current_number' => $hasNumber ? (int)$metadata['image_number'] : null,
                        'metadata' => $metadata
                    ];
                }
            }
        }
    }
    
    // Sortera efter stored_at (äldsta först)
    usort($allFiles, function($a, $b) {
        $timeA = strtotime($a['stored_at']);
        $timeB = strtotime($b['stored_at']);
        if ($timeA === false) $timeA = 0;
        if ($timeB === false) $timeB = 0;
        return $timeA - $timeB;
    });
    
    // Separera filer med och utan nummer
    $filesWithNumbers = [];
    $filesWithoutNumbers = [];
    
    foreach ($allFiles as $file) {
        if ($file['has_number']) {
            $filesWithNumbers[] = $file;
        } else {
            $filesWithoutNumbers[] = $file;
        }
    }
    
    // Sortera filer med nummer efter nummer
    usort($filesWithNumbers, function($a, $b) {
        return $a['current_number'] - $b['current_number'];
    });
    
    // Hitta högsta nummer
    $maxNumber = 0;
    foreach ($filesWithNumbers as $file) {
        if ($file['current_number'] > $maxNumber) {
            $maxNumber = $file['current_number'];
        }
    }
    
    // Initiera eller uppdatera image_numbers.json med rätt startvärde
    $numbersFile = dirname($uploadRoot) . '/data/image_numbers.json';
    $nextNumber = $maxNumber + 1;
    $numbersData = ['next_number' => $nextNumber];
    file_put_contents($numbersFile, json_encode($numbersData, JSON_PRETTY_PRINT));
    
    // Tilldela nummer till filer utan nummer
    $assigned = 0;
    $skipped = 0;
    $errors = 0;
    
    foreach ($filesWithoutNumbers as $file) {
        $imageNumber = assignImageNumber($file['file_path'], $file['json_path']);
        if ($imageNumber !== false) {
            $assigned++;
        } else {
            $errors++;
        }
    }
    
    $skipped = count($filesWithNumbers);
    
    return [
        'total_files' => count($allFiles),
        'assigned' => $assigned,
        'skipped' => $skipped,
        'errors' => $errors
    ];
}

// Hantera POST-request
$results = [];
$totalSize = 0;
$dryRun = true;
// Kom ihåg vilka rutor som var ikryssade
$checkBackup = true;
$checkThumbnails = true;
$checkProjectImages = true;
$checkChatImages = true;
$checkMetaMissingImage = true;
$checkMetaMissingThumb = true;
$checkDirsMissingProject = true;
$checkParticipantsMissingUser = true;
$checkMessagesMissingUser = true;
$checkChatMissingFiles = true;
$checkFilesMissingMetadata = true;
$checkMissingProfileImages = true;
$checkMappingIssues = true;
$checkAssignImageNumbers = false;

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $dryRun = isset($_POST['dry_run']) && $_POST['dry_run'] === '1';
    $checkBackup = isset($_POST['check_backup']) && $_POST['check_backup'] === '1';
    $checkThumbnails = isset($_POST['check_thumbnails']) && $_POST['check_thumbnails'] === '1';
    $checkProjectImages = isset($_POST['check_project_images']) && $_POST['check_project_images'] === '1';
    $checkChatImages = isset($_POST['check_chat_images']) && $_POST['check_chat_images'] === '1';
    $checkMetaMissingImage = isset($_POST['check_meta_missing_image']) && $_POST['check_meta_missing_image'] === '1';
    $checkMetaMissingThumb = isset($_POST['check_meta_missing_thumb']) && $_POST['check_meta_missing_thumb'] === '1';
    $checkDirsMissingProject = isset($_POST['check_dirs_missing_project']) && $_POST['check_dirs_missing_project'] === '1';
    $checkParticipantsMissingUser = isset($_POST['check_participants_missing_user']) && $_POST['check_participants_missing_user'] === '1';
    $checkMessagesMissingUser = isset($_POST['check_messages_missing_user']) && $_POST['check_messages_missing_user'] === '1';
    $checkChatMissingFiles = isset($_POST['check_chat_missing_files']) && $_POST['check_chat_missing_files'] === '1';
    $checkFilesMissingMetadata = isset($_POST['check_files_missing_metadata']) && $_POST['check_files_missing_metadata'] === '1';
    $checkMissingProfileImages = isset($_POST['check_missing_profile_images']) && $_POST['check_missing_profile_images'] === '1';
    $checkMappingIssues = isset($_POST['check_mapping_issues']) && $_POST['check_mapping_issues'] === '1';
    $checkAssignImageNumbers = isset($_POST['check_assign_image_numbers']) && $_POST['check_assign_image_numbers'] === '1';
    
    // Hämta user_id för loggning
    $userId = $_SESSION['user_id'] ?? $_GET['user_id'] ?? '';
    
    if ($checkBackup) {
        $backupFiles = findBackupFiles($uploadRoot, $chatsDir);
        $results['backup'] = $backupFiles;
        foreach ($backupFiles as $file) {
            $totalSize += $file['size'];
        }
        
        if (!$dryRun && count($backupFiles) > 0) {
            foreach ($backupFiles as $file) {
                @unlink($file['path']);
            }
            logRepairEvent('cleanup_backup_files', $userId, count($backupFiles), 'Raderade ' . count($backupFiles) . ' backup-filer');
        }
    }
    
    if ($checkThumbnails) {
        $thumbnails = findOrphanedThumbnails($uploadRoot, $chatsDir);
        $results['thumbnails'] = $thumbnails;
        foreach ($thumbnails as $file) {
            $totalSize += $file['size'];
        }
        
        if (!$dryRun && count($thumbnails) > 0) {
            foreach ($thumbnails as $file) {
                @unlink($file['path']);
            }
            logRepairEvent('cleanup_thumbnails', $userId, count($thumbnails), 'Raderade ' . count($thumbnails) . ' föräldralösa thumbnails');
        }
    }
    
    if ($checkProjectImages) {
        $projectImages = findOrphanedProjectImages($uploadRoot, $chatsDir);
        $results['project_images'] = $projectImages;
        foreach ($projectImages as $file) {
            $totalSize += $file['size'];
        }
        
        if (!$dryRun && count($projectImages) > 0) {
            foreach ($projectImages as $file) {
                @unlink($file['path']);
            }
            logRepairEvent('cleanup_project_images', $userId, count($projectImages), 'Raderade ' . count($projectImages) . ' föräldralösa projektbilder');
        }
    }
    
    if ($checkChatImages) {
        $chatImages = findOrphanedChatImages($chatsDir);
        $results['chat_images'] = $chatImages;
        foreach ($chatImages as $file) {
            $totalSize += $file['size'];
        }
        
        if (!$dryRun && count($chatImages) > 0) {
            foreach ($chatImages as $file) {
                @unlink($file['path']);
            }
            logRepairEvent('cleanup_chat_images', $userId, count($chatImages), 'Raderade ' . count($chatImages) . ' föräldralösa chattbilder');
        }
    }
    
    // Nya konsistenskontroller (rapport endast, ingen borttagning)
    if ($checkMetaMissingImage) {
        $metaNoImg = findMetadataMissingImages($uploadRoot);
        $results['meta_missing_image'] = $metaNoImg;
        if (!$dryRun && count($metaNoImg) > 0) {
            foreach ($metaNoImg as $m) {
                @unlink($m['path']);
            }
            logRepairEvent('cleanup_metadata_missing_image', $userId, count($metaNoImg), 'Raderade ' . count($metaNoImg) . ' metadata-filer utan bild');
        }
    }
    if ($checkMetaMissingThumb) {
        $metaNoThumb = findMetadataMissingThumbs($uploadRoot);
        $results['meta_missing_thumb'] = $metaNoThumb;
        if (!$dryRun && count($metaNoThumb) > 0) {
            foreach ($metaNoThumb as $m) {
                @unlink($m['path']);
            }
            logRepairEvent('cleanup_metadata_missing_thumb', $userId, count($metaNoThumb), 'Raderade ' . count($metaNoThumb) . ' metadata-filer utan thumbnail');
        }
    }
    if ($checkDirsMissingProject) {
        $dirsMissing = findDirsWithoutProject($projectsFile, $uploadRoot, $chatsDir);
        // Lägg till storlek för rapport
        foreach ($dirsMissing as &$d) {
            $d['size'] = getDirSize($d['path']);
        }
        unset($d);
        $results['dirs_missing_project'] = $dirsMissing;
        if (!$dryRun && count($dirsMissing) > 0) {
            foreach ($dirsMissing as $d) {
                rrmdir($d['path']);
            }
            logRepairEvent('cleanup_dirs_missing_project', $userId, count($dirsMissing), 'Raderade ' . count($dirsMissing) . ' mappar utan projekt');
        }
    }
    if ($checkParticipantsMissingUser) {
        $missingParts = findParticipantsMissingUsers($projectsFile, $userIdsFile);
        $results['participants_missing_user'] = $missingParts;
        if (!$dryRun && !empty($missingParts)) {
            // Ta bort saknade deltagare från projects.json
            $projects = readProjectsList($projectsFile);
            $userMap = readUsersMap($userIdsFile);
            $updated = false;
            foreach ($projects as &$proj) {
                if (empty($proj['id'])) continue;
                $parts = $proj['participants'] ?? [];
                $filtered = [];
                foreach ($parts as $p) {
                    $pid = $p['participant_id'] ?? '';
                    if ($pid === '' || isset($userMap[$pid])) {
                        $filtered[] = $p;
                    } else {
                        $updated = true;
                    }
                }
                $proj['participants'] = $filtered;
            }
            unset($proj);
            if ($updated) {
                saveProjectsList($projectsFile, $projects);
            }
        }
    }
    if ($checkMessagesMissingUser) {
        $results['messages_missing_user'] = findMessagesMissingUsers($chatsDir, $userIdsFile);
    }
    if ($checkChatMissingFiles) {
        $results['chat_missing_files'] = findChatMessagesMissingImages($chatsDir);
    }
    if ($checkFilesMissingMetadata) {
        $filesNoMeta = findFilesMissingMetadata($uploadRoot);
        $results['files_missing_metadata'] = $filesNoMeta;
        if (!$dryRun && count($filesNoMeta) > 0) {
            foreach ($filesNoMeta as $f) {
                @unlink($f['path']);
            }
            logRepairEvent('cleanup_files_missing_metadata', $userId, count($filesNoMeta), 'Raderade ' . count($filesNoMeta) . ' filer utan metadata');
        }
    }
    
    if ($checkMissingProfileImages) {
        $missingProfileImages = findMissingProfileImages($userIdsFile);
        $results['missing_profile_images'] = $missingProfileImages;
        if (!$dryRun && !empty($missingProfileImages)) {
            // Ta bort profile_image från userids.json
            $users = json_decode(file_get_contents($userIdsFile), true);
            if (is_array($users)) {
                $updated = false;
                foreach ($missingProfileImages as $missing) {
                    $index = $missing['index'];
                    if (isset($users[$index]) && isset($users[$index]['profile_image'])) {
                        unset($users[$index]['profile_image']);
                        $updated = true;
                    }
                }
                if ($updated) {
                    $json = json_encode($users, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
                    file_put_contents($userIdsFile, $json);
                }
            }
        }
    }
    
    if ($checkMappingIssues) {
        $mappingIssues = findMappingIssues($projectsFile, $userIdsFile, $chatsDir);
        $results['mapping_issues'] = $mappingIssues;
        
        if (!$dryRun) {
            // Reparera chat_mapping.json
            $chatMappingFile = dirname($projectsFile) . '/chat_mapping.json';
            if (file_exists($chatMappingFile)) {
                $chatMapping = readChatMapping(null, $chatMappingFile);
                $updated = false;
                
                // Ta bort projekt som inte finns
                foreach ($mappingIssues['chat_mapping']['missing_projects'] as $issue) {
                    if (isset($chatMapping[$issue['project_id']])) {
                        unset($chatMapping[$issue['project_id']]);
                        $updated = true;
                    }
                }
                
                // Ta bort projekt där chat-mappen saknas
                foreach ($mappingIssues['chat_mapping']['missing_chats'] as $issue) {
                    if (isset($chatMapping[$issue['project_id']])) {
                        unset($chatMapping[$issue['project_id']]);
                        $updated = true;
                    }
                }
                
                // Ta bort användare som inte finns
                foreach ($mappingIssues['chat_mapping']['missing_users'] as $issue) {
                    if (isset($chatMapping[$issue['project_id']])) {
                        $chatters = $chatMapping[$issue['project_id']];
                        $chatters = array_filter($chatters, function($chatter) use ($issue) {
                            return ($chatter['participant_id'] ?? '') !== $issue['participant_id'];
                        });
                        $chatMapping[$issue['project_id']] = array_values($chatters);
                        $updated = true;
                    }
                }
                
                if ($updated) {
                    $json = json_encode($chatMapping, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
                    file_put_contents($chatMappingFile, $json);
                    logRepairEvent('fix_mapping_issues', $userId, 0, 'Reparerade ogiltiga referenser i chat_mapping.json');
                }
            }
            
            // Reparera project_mapping.json
            $projectMappingFile = dirname($projectsFile) . '/project_mapping.json';
            if (file_exists($projectMappingFile)) {
                $projectMapping = readProjectMapping(null, $projectMappingFile);
                $updated = false;
                
                // Ta bort projekt som inte finns
                foreach ($mappingIssues['project_mapping']['missing_projects'] as $issue) {
                    if (isset($projectMapping[$issue['project_id']])) {
                        unset($projectMapping[$issue['project_id']]);
                        $updated = true;
                    }
                }
                
                // Ta bort användare som inte finns
                foreach ($mappingIssues['project_mapping']['missing_users'] as $issue) {
                    if (isset($projectMapping[$issue['project_id']])) {
                        $participants = $projectMapping[$issue['project_id']];
                        $participants = array_filter($participants, function($participant) use ($issue) {
                            return ($participant['participant_id'] ?? '') !== $issue['participant_id'];
                        });
                        $projectMapping[$issue['project_id']] = array_values($participants);
                        $updated = true;
                    }
                }
                
                if ($updated) {
                    $json = json_encode($projectMapping, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
                    file_put_contents($projectMappingFile, $json);
                }
            }
        }
    }
    
    if ($checkAssignImageNumbers) {
        // Kör funktionen (den uppdaterar bara JSON-metadata, tar inte bort filer)
        // Men respektera dry-run-läget för att visa vad som skulle hända
        if (!$dryRun) {
            $imageNumbersResult = assignImageNumbers($uploadRoot);
            // Logga om bildnummer faktiskt tilldelades
            if (($imageNumbersResult['assigned'] ?? 0) > 0) {
                logRepairEvent('assign_image_numbers', $userId, $imageNumbersResult['assigned'], 'Tilldelade bildnummer till ' . $imageNumbersResult['assigned'] . ' bilder');
            }
        } else {
            // I dry-run-läge, simulera genom att räkna filer utan att göra ändringar
            $allFiles = [];
            $supportedImageFormats = ['jpg', 'jpeg', 'png', 'webp', 'heic', 'heif'];
            $supportedVideoFormats = ['mp4', 'webm', 'mov', 'avi'];
            $allSupportedFormats = array_merge($supportedImageFormats, $supportedVideoFormats);
            
            if (is_dir($uploadRoot)) {
                $projectDirs = glob($uploadRoot . '/*', GLOB_ONLYDIR);
                foreach ($projectDirs as $projectDir) {
                    $jsonFiles = glob($projectDir . '/*.json');
                    foreach ($jsonFiles as $jsonFile) {
                        $metadata = json_decode(file_get_contents($jsonFile), true);
                        if (!$metadata) continue;
                        
                        $actualFile = null;
                        foreach ($allSupportedFormats as $format) {
                            $testFile = str_replace('.json', '.' . $format, $jsonFile);
                            if (file_exists($testFile)) {
                                $actualFile = $testFile;
                                break;
                            }
                        }
                        
                        if ($actualFile) {
                            $hasNumber = isset($metadata['image_number']) && is_numeric($metadata['image_number']);
                            $allFiles[] = ['has_number' => $hasNumber];
                        }
                    }
                }
            }
            
            $filesWithNumbers = array_filter($allFiles, function($f) { return $f['has_number']; });
            $filesWithoutNumbers = array_filter($allFiles, function($f) { return !$f['has_number']; });
            
            $imageNumbersResult = [
                'total_files' => count($allFiles),
                'assigned' => count($filesWithoutNumbers),
                'skipped' => count($filesWithNumbers),
                'errors' => 0
            ];
        }
        $imageNumbersResult['dry_run'] = $dryRun;
        $results['assign_image_numbers'] = $imageNumbersResult;
    }
    
    // Uppdatera checkbox-status: behåll ikryssad om problem kvarstår, avmarkera om allt är OK
    // Endast om dry-run kördes - om faktisk borttagning kördes, behåll statusen
    if ($dryRun) {
        $checkBackup = !empty($results['backup'] ?? []);
        $checkThumbnails = !empty($results['thumbnails'] ?? []);
        $checkProjectImages = !empty($results['project_images'] ?? []);
        $checkChatImages = !empty($results['chat_images'] ?? []);
        $checkMetaMissingImage = !empty($results['meta_missing_image'] ?? []);
        $checkMetaMissingThumb = !empty($results['meta_missing_thumb'] ?? []);
        $checkDirsMissingProject = !empty($results['dirs_missing_project'] ?? []);
        $checkParticipantsMissingUser = !empty($results['participants_missing_user'] ?? []);
        $checkMessagesMissingUser = !empty($results['messages_missing_user'] ?? []);
        $checkChatMissingFiles = !empty($results['chat_missing_files'] ?? []);
        $checkFilesMissingMetadata = !empty($results['files_missing_metadata'] ?? []);
        $checkMissingProfileImages = !empty($results['missing_profile_images'] ?? []);
        // För assign_image_numbers, avmarkera om allt gick bra (inga fel och inga ändringar behövdes)
        if (isset($results['assign_image_numbers'])) {
            $imgResult = $results['assign_image_numbers'];
            $hasErrors = ($imgResult['errors'] ?? 0) > 0;
            $hasChanges = ($imgResult['assigned'] ?? 0) > 0;
            // Avmarkera om allt är OK (inga fel och inga ändringar behövdes)
            if (!$hasErrors && !$hasChanges) {
                $checkAssignImageNumbers = false;
            } else {
                // Behåll ikryssad om det fanns fel eller ändringar
                $checkAssignImageNumbers = true;
            }
        }
        // För mapping_issues, kontrollera om det finns några problem
        if (isset($results['mapping_issues'])) {
            $mappingIssues = $results['mapping_issues'];
            $totalChatIssues = count($mappingIssues['chat_mapping']['missing_projects'] ?? []) + 
                              count($mappingIssues['chat_mapping']['missing_users'] ?? []) + 
                              count($mappingIssues['chat_mapping']['missing_chats'] ?? []);
            $totalProjectIssues = count($mappingIssues['project_mapping']['missing_projects'] ?? []) + 
                                 count($mappingIssues['project_mapping']['missing_users'] ?? []);
            $checkMappingIssues = ($totalChatIssues + $totalProjectIssues) > 0;
        } else {
            // Om kontrollen inte kördes, behåll statusen
            // (checkMappingIssues redan satt från POST eller default)
        }
    }

    // Efter körning, sätt dry-run till true igen för säkerhet och förvalt läge
    $dryRun = true;
}
?>
<!DOCTYPE html>
<html lang="sv">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Repair - Städning av oanvända filer</title>
    <link rel="stylesheet" href="../common.css">
    <?php
    // Bestäm admin URL
    $script = $_SERVER['SCRIPT_NAME'] ?? '';
    $admPos = strpos($script, '/adm/');
    $adminUrl = ($admPos !== false ? substr($script, 0, $admPos + 4) : '/adm') . '/admin.php';
    $userId = $_SESSION['user_id'] ?? $_GET['user_id'] ?? '';
    if (!empty($userId)) {
        $adminUrl .= '?user_id=' . htmlspecialchars($userId);
    }
    // Bestäm bas-sökväg för logo
    $basePath = ($admPos !== false ? substr($script, 0, $admPos) : '') . '/';
    ?>
    <style>
        .repair-container {
            max-width: 1200px;
            margin: 20px auto;
            padding: 20px;
        }
        .repair-form {
            background: white;
            border-radius: 8px;
            padding: 20px;
            margin-bottom: 20px;
            box-shadow: 0 2px 4px rgba(0,0,0,0.1);
        }
        .form-group {
            margin-bottom: 15px;
        }
        .form-group label {
            display: block;
            margin-bottom: 5px;
            font-weight: bold;
        }
        .checkbox-group {
            display: flex;
            align-items: center;
            gap: 8px;
        }
        .checkbox-group input[type="checkbox"] {
            width: 18px;
            height: 18px;
        }
        .results {
            background: white;
            border-radius: 8px;
            padding: 20px;
            box-shadow: 0 2px 4px rgba(0,0,0,0.1);
        }
        .result-section {
            margin-bottom: 30px;
        }
        .result-section h3 {
            margin-top: 0;
            color: #333;
            border-bottom: 2px solid #007bff;
            padding-bottom: 10px;
        }
        .file-list {
            list-style: none;
            padding: 0;
            margin: 0;
        }
        .file-item {
            padding: 8px 12px;
            border-bottom: 1px solid #eee;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        .file-item:last-child {
            border-bottom: none;
        }
        .file-info {
            flex: 1;
        }
        .file-name {
            font-weight: 500;
            color: #333;
        }
        .file-path {
            font-size: 12px;
            color: #666;
            margin-top: 2px;
        }
        .file-size {
            color: #666;
            font-size: 14px;
        }
        .summary {
            background: #f8f9fa;
            padding: 15px;
            border-radius: 6px;
            margin-bottom: 20px;
        }
        .summary-item {
            display: flex;
            justify-content: space-between;
            padding: 5px 0;
        }
        .btn-primary {
            background: #007bff;
            color: white;
            border: none;
            padding: 10px 20px;
            border-radius: 6px;
            cursor: pointer;
            font-size: 16px;
        }
        .btn-primary:hover {
            background: #0056b3;
        }
        .btn-danger {
            background: #dc3545;
            color: white;
            border: none;
            padding: 10px 20px;
            border-radius: 6px;
            cursor: pointer;
            font-size: 16px;
        }
        .btn-danger:hover {
            background: #c82333;
        }
        .dry-run-badge {
            display: inline-block;
            background: #ffc107;
            color: #212529;
            padding: 4px 8px;
            border-radius: 4px;
            font-size: 12px;
            font-weight: bold;
            margin-left: 10px;
        }
        .no-files {
            color: #28a745;
            font-style: italic;
            padding: 10px;
        }
    </style>
</head>
<body>
    <header class="header">
        <a href="<?= htmlspecialchars($basePath) ?>index.html<?= !empty($userId) ? '?user_id=' . htmlspecialchars($userId) : '' ?>">
            <img src="<?= htmlspecialchars($basePath) ?>data/logo.png" alt="ByggArkiv Snickare Stefan" class="logo">
        </a>
        <div class="back-link">
            <a href="<?= htmlspecialchars($adminUrl) ?>">← Tillbaka till admin</a>
        </div>
    </header>
    
    <div class="repair-container">
        <h1>Repair - Städning av oanvända filer</h1>
        
        <div class="repair-form">
            <form method="POST">
                <div class="form-group">
                    <label>Åtgärder (kontroll + ev. borttagning)</label>
                    <div class="checkbox-group">
                        <input type="checkbox" id="check_backup" name="check_backup" value="1" <?= $checkBackup ? 'checked' : '' ?>>
                        <label for="check_backup">Backup-filer (.bak, .old, .backup, .tmp, .temp) som inte refereras</label>
                    </div>
                    <div class="checkbox-group">
                        <input type="checkbox" id="check_thumbnails" name="check_thumbnails" value="1" <?= $checkThumbnails ? 'checked' : '' ?>>
                        <label for="check_thumbnails">Tumnaglar (-thumb.jpg) utan motsvarande originalbild</label>
                    </div>
                    <div class="checkbox-group">
                        <input type="checkbox" id="check_project_images" name="check_project_images" value="1" <?= $checkProjectImages ? 'checked' : '' ?>>
                        <label for="check_project_images">Projektalbum-bilder (uploads/) som inte refereras</label>
                    </div>
                    <div class="checkbox-group">
                        <input type="checkbox" id="check_chat_images" name="check_chat_images" value="1" <?= $checkChatImages ? 'checked' : '' ?>>
                        <label for="check_chat_images">Projektchat-bilder (chats/) som inte refereras</label>
                    </div>
                    <div class="checkbox-group">
                        <input type="checkbox" id="check_dirs_missing_project" name="check_dirs_missing_project" value="1" <?= $checkDirsMissingProject ? 'checked' : '' ?>>
                        <label for="check_dirs_missing_project">Mappar i uploads/chats saknar projekt i projects.json</label>
                    </div>
                    <div class="checkbox-group">
                        <input type="checkbox" id="check_participants_missing_user" name="check_participants_missing_user" value="1" <?= $checkParticipantsMissingUser ? 'checked' : '' ?>>
                        <label for="check_participants_missing_user">Projekt-deltagare saknas i userids.json</label>
                    </div>
                    <div class="checkbox-group">
                        <input type="checkbox" id="check_files_missing_metadata" name="check_files_missing_metadata" value="1" <?= $checkFilesMissingMetadata ? 'checked' : '' ?>>
                        <label for="check_files_missing_metadata">Filer i uploads saknar metadata (.json)</label>
                    </div>
                    <div class="checkbox-group">
                        <input type="checkbox" id="check_missing_profile_images" name="check_missing_profile_images" value="1" <?= $checkMissingProfileImages ? 'checked' : '' ?>>
                        <label for="check_missing_profile_images">Profilbilder som refereras i userids.json men saknas på servern</label>
                    </div>
                    <div class="checkbox-group">
                        <input type="checkbox" id="check_mapping_issues" name="check_mapping_issues" value="1" <?= $checkMappingIssues ? 'checked' : '' ?>>
                        <label for="check_mapping_issues">Ogiltiga referenser i chat_mapping.json och project_mapping.json</label>
                    </div>
                    <div class="checkbox-group">
                        <input type="checkbox" id="check_assign_image_numbers" name="check_assign_image_numbers" value="1" <?= $checkAssignImageNumbers ? 'checked' : '' ?>>
                        <label for="check_assign_image_numbers">Kontrollera bildnummer</label>
                    </div>
                </div>

                <div class="form-group">
                    <label>Endast kontroll (rapport)</label>
                    <div class="checkbox-group">
                        <input type="checkbox" id="check_messages_missing_user" name="check_messages_missing_user" value="1" <?= $checkMessagesMissingUser ? 'checked' : '' ?>>
                        <label for="check_messages_missing_user">Meddelanden från saknade användare</label>
                    </div>
                    <div class="checkbox-group">
                        <input type="checkbox" id="check_chat_missing_files" name="check_chat_missing_files" value="1" <?= $checkChatMissingFiles ? 'checked' : '' ?>>
                        <label for="check_chat_missing_files">Chat-meddelanden pekar på filer som saknas</label>
                    </div>
                </div>
                
                <div class="form-group">
                    <div class="checkbox-group">
                        <input type="checkbox" id="dry_run" name="dry_run" value="1" <?= $dryRun ? 'checked' : '' ?>>
                        <label for="dry_run">Dry-run (visa endast vad som skulle tas bort, ta inte bort filerna)</label>
                    </div>
                </div>
                
                <button type="submit" class="btn-primary">Kör städning</button>
            </form>
        </div>
        
        <?php if (!empty($results)): ?>
        <div class="results">
            
            <?php if (isset($results['backup'])): ?>
            <div class="result-section">
                <h3>Backup-filer (<?= count($results['backup']) ?>)</h3>
                <?php if (empty($results['backup'])): ?>
                    <div class="no-files">Inga backup-filer hittades.</div>
                <?php else: ?>
                    <ul class="file-list">
                        <?php foreach ($results['backup'] as $file): ?>
                        <li class="file-item">
                            <div class="file-info">
                                <div class="file-name"><?= htmlspecialchars($file['name']) ?></div>
                                <div class="file-path"><?= htmlspecialchars($file['path']) ?> (Projekt: <?= htmlspecialchars($file['project']) ?>)</div>
                            </div>
                            <div class="file-size"><?= formatFileSize($file['size']) ?></div>
                        </li>
                        <?php endforeach; ?>
                    </ul>
                <?php endif; ?>
            </div>
            <?php endif; ?>
            
            <?php if (isset($results['thumbnails'])): ?>
            <div class="result-section">
                <h3>Tumnaglar utan originalbild (<?= count($results['thumbnails']) ?>)</h3>
                <?php if (empty($results['thumbnails'])): ?>
                    <div class="no-files">Inga tumnaglar utan originalbild hittades.</div>
                <?php else: ?>
                    <ul class="file-list">
                        <?php foreach ($results['thumbnails'] as $file): ?>
                        <li class="file-item">
                            <div class="file-info">
                                <div class="file-name"><?= htmlspecialchars($file['name']) ?></div>
                                <div class="file-path"><?= htmlspecialchars($file['path']) ?> (Projekt: <?= htmlspecialchars($file['project']) ?>)</div>
                            </div>
                            <div class="file-size"><?= formatFileSize($file['size']) ?></div>
                        </li>
                        <?php endforeach; ?>
                    </ul>
                <?php endif; ?>
            </div>
            <?php endif; ?>
            
            <?php if (isset($results['project_images'])): ?>
            <div class="result-section">
                <h3>Projektalbum-bilder som inte refereras (<?= count($results['project_images']) ?>)</h3>
                <?php if (empty($results['project_images'])): ?>
                    <div class="no-files">Inga oanvända projektalbum-bilder hittades.</div>
                <?php else: ?>
                    <ul class="file-list">
                        <?php foreach ($results['project_images'] as $file): ?>
                        <li class="file-item">
                            <div class="file-info">
                                <div class="file-name"><?= htmlspecialchars($file['name']) ?></div>
                                <div class="file-path"><?= htmlspecialchars($file['path']) ?> (Projekt: <?= htmlspecialchars($file['project']) ?>)</div>
                            </div>
                            <div class="file-size"><?= formatFileSize($file['size']) ?></div>
                        </li>
                        <?php endforeach; ?>
                    </ul>
                <?php endif; ?>
            </div>
            <?php endif; ?>
            
            <?php if (isset($results['chat_images'])): ?>
            <div class="result-section">
                <h3>Projektchat-bilder som inte refereras (<?= count($results['chat_images']) ?>)</h3>
                <?php if (empty($results['chat_images'])): ?>
                    <div class="no-files">Inga oanvända projektchat-bilder hittades.</div>
                <?php else: ?>
                    <ul class="file-list">
                        <?php foreach ($results['chat_images'] as $file): ?>
                        <li class="file-item">
                            <div class="file-info">
                                <div class="file-name"><?= htmlspecialchars($file['name']) ?></div>
                                <div class="file-path"><?= htmlspecialchars($file['path']) ?> (Projekt: <?= htmlspecialchars($file['project']) ?>)</div>
                            </div>
                            <div class="file-size"><?= formatFileSize($file['size']) ?></div>
                        </li>
                        <?php endforeach; ?>
                    </ul>
                <?php endif; ?>
            </div>
            <?php endif; ?>
            
            <?php if (isset($results['meta_missing_image'])): ?>
            <div class="result-section">
                <h3>Metadata utan bild (<?= count($results['meta_missing_image']) ?>)</h3>
                <?php if (empty($results['meta_missing_image'])): ?>
                    <div class="no-files">Inga metadatafiler pekar på saknade bilder.</div>
                <?php else: ?>
                    <ul class="file-list">
                        <?php foreach ($results['meta_missing_image'] as $item): ?>
                        <li class="file-item">
                            <div class="file-info">
                                <div class="file-name"><?= htmlspecialchars($item['name']) ?></div>
                                <div class="file-path"><?= htmlspecialchars($item['path']) ?> (Projekt: <?= htmlspecialchars($item['project']) ?>) – saknar fil: <?= htmlspecialchars($item['stored_file'] ?? '') ?></div>
                            </div>
                        </li>
                        <?php endforeach; ?>
                    </ul>
                <?php endif; ?>
            </div>
            <?php endif; ?>
            
            <?php if (isset($results['meta_missing_thumb'])): ?>
            <div class="result-section">
                <h3>Metadata pekar på saknad tumnagel (<?= count($results['meta_missing_thumb']) ?>)</h3>
                <?php if (empty($results['meta_missing_thumb'])): ?>
                    <div class="no-files">Inga metadatafiler pekar på saknade tumnaglar.</div>
                <?php else: ?>
                    <ul class="file-list">
                        <?php foreach ($results['meta_missing_thumb'] as $item): ?>
                        <li class="file-item">
                            <div class="file-info">
                                <div class="file-name"><?= htmlspecialchars($item['name']) ?></div>
                                <div class="file-path"><?= htmlspecialchars($item['path']) ?> (Projekt: <?= htmlspecialchars($item['project']) ?>) – saknar thumb: <?= htmlspecialchars($item['thumb_file'] ?? '') ?></div>
                            </div>
                        </li>
                        <?php endforeach; ?>
                    </ul>
                <?php endif; ?>
            </div>
            <?php endif; ?>
            
            <?php if (isset($results['dirs_missing_project'])): ?>
            <div class="result-section">
                <h3>Mappar utan projekt i projects.json (<?= count($results['dirs_missing_project']) ?>)</h3>
                <?php if (empty($results['dirs_missing_project'])): ?>
                    <div class="no-files">Inga mappar saknar projektdefinition.</div>
                <?php else: ?>
                    <ul class="file-list">
                        <?php foreach ($results['dirs_missing_project'] as $item): ?>
                        <li class="file-item">
                            <div class="file-info">
                                <div class="file-name"><?= htmlspecialchars($item['project']) ?></div>
                                <div class="file-path"><?= htmlspecialchars($item['path']) ?> (<?= htmlspecialchars($item['location']) ?>)</div>
                            </div>
                            <?php if (isset($item['size'])): ?>
                            <div class="file-size"><?= formatFileSize($item['size']) ?></div>
                            <?php endif; ?>
                        </li>
                        <?php endforeach; ?>
                    </ul>
                <?php endif; ?>
            </div>
            <?php endif; ?>
            
            <?php if (isset($results['participants_missing_user'])): ?>
            <div class="result-section">
                <h3>Deltagare i projekt saknas i userids.json (<?= count($results['participants_missing_user']) ?>)</h3>
                <?php if (empty($results['participants_missing_user'])): ?>
                    <div class="no-files">Alla deltagare finns i userids.json.</div>
                <?php else: ?>
                    <ul class="file-list">
                        <?php foreach ($results['participants_missing_user'] as $item): ?>
                        <li class="file-item">
                            <div class="file-info">
                                <div class="file-name">Projekt: <?= htmlspecialchars($item['project']) ?></div>
                                <div class="file-path">Deltagare saknas: <?= htmlspecialchars($item['participant_id']) ?></div>
                            </div>
                        </li>
                        <?php endforeach; ?>
                    </ul>
                <?php endif; ?>
            </div>
            <?php endif; ?>
            
            <?php if (isset($results['messages_missing_user'])): ?>
            <div class="result-section">
                <h3>Meddelanden från saknade användare (<?= count($results['messages_missing_user']) ?>)</h3>
                <?php if (empty($results['messages_missing_user'])): ?>
                    <div class="no-files">Inga meddelanden från saknade användare hittades.</div>
                <?php else: ?>
                    <ul class="file-list">
                        <?php foreach ($results['messages_missing_user'] as $item): ?>
                        <li class="file-item">
                            <div class="file-info">
                                <div class="file-name">Projekt: <?= htmlspecialchars($item['project']) ?></div>
                                <div class="file-path">participant_id: <?= htmlspecialchars($item['participant_id']) ?> (message_id: <?= htmlspecialchars($item['message_id']) ?>)</div>
                            </div>
                        </li>
                        <?php endforeach; ?>
                    </ul>
                <?php endif; ?>
            </div>
            <?php endif; ?>
            
            <?php if (isset($results['chat_missing_files'])): ?>
            <div class="result-section">
                <h3>Chat-meddelanden pekar på saknade filer (<?= count($results['chat_missing_files']) ?>)</h3>
                <?php if (empty($results['chat_missing_files'])): ?>
                    <div class="no-files">Inga saknade chat-filer hittades.</div>
                <?php else: ?>
                    <ul class="file-list">
                        <?php foreach ($results['chat_missing_files'] as $item): ?>
                        <li class="file-item">
                            <div class="file-info">
                                <div class="file-name">Projekt: <?= htmlspecialchars($item['project']) ?></div>
                                <div class="file-path">media_file: <?= htmlspecialchars($item['media_file']) ?> (message_id: <?= htmlspecialchars($item['message_id']) ?>)</div>
                            </div>
                        </li>
                        <?php endforeach; ?>
                    </ul>
                <?php endif; ?>
            </div>
            <?php endif; ?>
            
            <?php if (isset($results['files_missing_metadata'])): ?>
            <div class="result-section">
                <h3>Filer utan metadata (<?= count($results['files_missing_metadata']) ?>)</h3>
                <?php if (empty($results['files_missing_metadata'])): ?>
                    <div class="no-files">Inga filer saknar metadata.</div>
                <?php else: ?>
                    <ul class="file-list">
                        <?php foreach ($results['files_missing_metadata'] as $file): ?>
                        <li class="file-item">
                            <div class="file-info">
                                <div class="file-name"><?= htmlspecialchars($file['name']) ?></div>
                                <div class="file-path"><?= htmlspecialchars($file['path']) ?> (Projekt: <?= htmlspecialchars($file['project']) ?>)</div>
                            </div>
                            <div class="file-size"><?= formatFileSize($file['size']) ?></div>
                        </li>
                        <?php endforeach; ?>
                    </ul>
                <?php endif; ?>
            </div>
            <?php endif; ?>
            
            <?php if (isset($results['missing_profile_images'])): ?>
            <div class="result-section">
                <h3>Profilbilder som saknas (<?= count($results['missing_profile_images']) ?>)</h3>
                <?php if (empty($results['missing_profile_images'])): ?>
                    <div class="no-files">Alla profilbilder som refereras i userids.json finns på servern.</div>
                <?php else: ?>
                    <ul class="file-list">
                        <?php foreach ($results['missing_profile_images'] as $item): ?>
                        <li class="file-item">
                            <div class="file-info">
                                <div class="file-name"><?= htmlspecialchars($item['full_name']) ?> (<?= htmlspecialchars($item['user_id']) ?>)</div>
                                <div class="file-path">Saknar profilbild: <?= htmlspecialchars($item['profile_image']) ?></div>
                            </div>
                        </li>
                        <?php endforeach; ?>
                    </ul>
                <?php endif; ?>
            </div>
            <?php endif; ?>
            
            <?php if (isset($results['mapping_issues'])): ?>
            <div class="result-section">
                <?php
                $mappingIssues = $results['mapping_issues'];
                $totalChatIssues = count($mappingIssues['chat_mapping']['missing_projects']) + 
                                  count($mappingIssues['chat_mapping']['missing_users']) + 
                                  count($mappingIssues['chat_mapping']['missing_chats']);
                $totalProjectIssues = count($mappingIssues['project_mapping']['missing_projects']) + 
                                     count($mappingIssues['project_mapping']['missing_users']);
                $totalIssues = $totalChatIssues + $totalProjectIssues;
                ?>
                <h3>Ogiltiga referenser i mapping-filer (<?= $totalIssues ?>)</h3>
                <?php if ($totalIssues === 0): ?>
                    <div class="no-files">Inga ogiltiga referenser hittades i chat_mapping.json eller project_mapping.json.</div>
                <?php else: ?>
                    <?php if ($totalChatIssues > 0): ?>
                    <h4>chat_mapping.json (<?= $totalChatIssues ?> problem)</h4>
                    <ul class="file-list">
                        <?php if (!empty($mappingIssues['chat_mapping']['missing_projects'])): ?>
                            <?php foreach ($mappingIssues['chat_mapping']['missing_projects'] as $issue): ?>
                            <li class="file-item">
                                <div class="file-info">
                                    <div class="file-name">Projekt saknas: <?= htmlspecialchars($issue['project_id']) ?></div>
                                    <div class="file-path">Projekt-ID finns i chat_mapping.json men saknas i projects.json</div>
                                </div>
                            </li>
                            <?php endforeach; ?>
                        <?php endif; ?>
                        <?php if (!empty($mappingIssues['chat_mapping']['missing_users'])): ?>
                            <?php foreach ($mappingIssues['chat_mapping']['missing_users'] as $issue): ?>
                            <li class="file-item">
                                <div class="file-info">
                                    <div class="file-name">Användare saknas: <?= htmlspecialchars($issue['participant_id']) ?></div>
                                    <div class="file-path">Projekt: <?= htmlspecialchars($issue['project_id']) ?> - participant_id finns i chat_mapping.json men saknas i userids.json</div>
                                </div>
                            </li>
                            <?php endforeach; ?>
                        <?php endif; ?>
                        <?php if (!empty($mappingIssues['chat_mapping']['missing_chats'])): ?>
                            <?php foreach ($mappingIssues['chat_mapping']['missing_chats'] as $issue): ?>
                            <li class="file-item">
                                <div class="file-info">
                                    <div class="file-name">Chat-mapp saknas: <?= htmlspecialchars($issue['project_id']) ?></div>
                                    <div class="file-path">Projekt finns i projects.json men chat-mappen saknas</div>
                                </div>
                            </li>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </ul>
                    <?php endif; ?>
                    
                    <?php if ($totalProjectIssues > 0): ?>
                    <h4>project_mapping.json (<?= $totalProjectIssues ?> problem)</h4>
                    <ul class="file-list">
                        <?php if (!empty($mappingIssues['project_mapping']['missing_projects'])): ?>
                            <?php foreach ($mappingIssues['project_mapping']['missing_projects'] as $issue): ?>
                            <li class="file-item">
                                <div class="file-info">
                                    <div class="file-name">Projekt saknas: <?= htmlspecialchars($issue['project_id']) ?></div>
                                    <div class="file-path">Projekt-ID finns i project_mapping.json men saknas i projects.json</div>
                                </div>
                            </li>
                            <?php endforeach; ?>
                        <?php endif; ?>
                        <?php if (!empty($mappingIssues['project_mapping']['missing_users'])): ?>
                            <?php foreach ($mappingIssues['project_mapping']['missing_users'] as $issue): ?>
                            <li class="file-item">
                                <div class="file-info">
                                    <div class="file-name">Användare saknas: <?= htmlspecialchars($issue['participant_id']) ?></div>
                                    <div class="file-path">Projekt: <?= htmlspecialchars($issue['project_id']) ?> - participant_id finns i project_mapping.json men saknas i userids.json</div>
                                </div>
                            </li>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </ul>
                    <?php endif; ?>
                <?php endif; ?>
            </div>
            <?php endif; ?>
            
            <?php if (isset($results['assign_image_numbers'])): ?>
            <div class="result-section">
                <?php
                $imgResult = $results['assign_image_numbers'];
                $hasErrors = ($imgResult['errors'] ?? 0) > 0;
                $hasChanges = ($imgResult['assigned'] ?? 0) > 0;
                $totalFiles = $imgResult['total_files'] ?? 0;
                ?>
                <h3>Kontrollera bildnummer (<?= $totalFiles ?> filer)</h3>
                <?php if ($totalFiles === 0): ?>
                    <div class="no-files">Inga bilder eller videor hittades att tilldela nummer till.</div>
                <?php elseif (!$hasErrors && !$hasChanges): ?>
                    <div class="no-files">Alla bilder har redan nummer och inga ändringar behövdes.</div>
                <?php else: ?>
                    <div class="summary">
                        <?php if (($imgResult['skipped'] ?? 0) > 0): ?>
                        <div class="summary-item">
                            <span><strong>Redan hade nummer:</strong></span>
                            <span><?= $imgResult['skipped'] ?></span>
                        </div>
                        <?php endif; ?>
                        <?php if ($hasChanges): ?>
                        <div class="summary-item">
                            <span><strong>Nya nummer tilldelade:</strong></span>
                            <span><?= $imgResult['assigned'] ?></span>
                        </div>
                        <?php endif; ?>
                        <?php if ($hasErrors): ?>
                            <div class="summary-item">
                                <span><strong>Fel:</strong></span>
                                <span style="color: #dc3545;"><?= $imgResult['errors'] ?></span>
                            </div>
                        <?php endif; ?>
                        <?php if (!$dryRun && $hasChanges): ?>
                        <div class="summary-item">
                            <span><strong>Status:</strong></span>
                            <span style="color: #28a745;">Bildnummer tilldelade</span>
                        </div>
                        <?php endif; ?>
                    </div>
                <?php endif; ?>
            </div>
            <?php endif; ?>
            
        </div>
        <?php endif; ?>
    </div>
</body>
</html>
