<?php
require_once __DIR__ . '/../admin_guard.php';
// Visa fel på skärmen + logga till fil
ini_set('display_errors', 1);
error_reporting(E_ALL);

$logFile = __DIR__ . '/diag-log.txt';
function dlog($msg){ global $logFile; @file_put_contents($logFile, '['.date('c')."] $msg\n", FILE_APPEND); }

header('Content-Type: text/plain; charset=utf-8');

// Använd rotkatalogen istället för adm-katalogen
// __DIR__ är adm/diagnostics, dirname(__DIR__) är adm, dirname(dirname(__DIR__)) är rotmappen
$rootDir = dirname(dirname(__DIR__));
// Normalisera sökvägen för att undvika problem med relativa sökvägar
$rootDir = realpath($rootDir) ?: $rootDir;

echo "========================================\n";
echo "  BYGGARKIV SYSTEMDIAGNOSTIK\n";
echo "  " . date('Y-m-d H:i:s') . "\n";
echo "========================================\n\n";

// ===== PHP INFORMATION =====
echo "== PHP INFORMATION ==\n";
echo "php_version: " . PHP_VERSION . "\n";
echo "sapi: " . php_sapi_name() . "\n";
echo "doc_root: " . ($_SERVER['DOCUMENT_ROOT'] ?? '(n/a)') . "\n";
echo "__DIR__: " . __DIR__ . "\n";
echo "root_dir: $rootDir\n";
echo "cwd: " . getcwd() . "\n";
echo "open_basedir: " . (ini_get('open_basedir') ?: '(none)') . "\n";
echo "upload_max_filesize: " . ini_get('upload_max_filesize') . "\n";
echo "post_max_size: " . ini_get('post_max_size') . "\n";
echo "memory_limit: " . ini_get('memory_limit') . "\n";
echo "max_execution_time: " . ini_get('max_execution_time') . "\n";
echo "date.timezone: " . ini_get('date.timezone') . "\n";

// ===== PHP MODULES =====
echo "\n== PHP MODULES ==\n";
$requiredModules = [
    'gd' => 'GD (för thumbnail-generering)',
    'fileinfo' => 'Fileinfo (för MIME-typ validering)',
    'json' => 'JSON (för datahantering)',
    'zip' => 'ZipArchive (för projektpaketering)',
    'mbstring' => 'mbstring (för UTF-8 hantering)'
];
foreach ($requiredModules as $module => $description) {
    $loaded = extension_loaded($module);
    echo "$module: " . ($loaded ? 'OK' : 'MISSING') . " ($description)\n";
}

// ===== KONFIGURATION =====
echo "\n== KONFIGURATION ==\n";
$configFile = $rootDir . '/config.php';
echo "config.php: " . (file_exists($configFile) ? 'OK' : 'MISSING') . "\n";
if (file_exists($configFile)) {
    require_once $configFile;
    echo "UPLOAD_ROOT: " . (defined('UPLOAD_ROOT') ? UPLOAD_ROOT : 'NOT DEFINED') . "\n";
    echo "MAX_FILE_BYTES: " . (defined('MAX_FILE_BYTES') ? (MAX_FILE_BYTES / 1024 / 1024) . ' MB' : 'NOT DEFINED') . "\n";
    echo "MAX_VIDEO_BYTES: " . (defined('MAX_VIDEO_BYTES') ? (MAX_VIDEO_BYTES / 1024 / 1024) . ' MB' : 'NOT DEFINED') . "\n";
}

// ===== DATA FILES =====
echo "\n== DATA FILES ==\n";
$dataFiles = [
    'projects.json' => 'Projektlista',
    'userids.json' => 'Användardata',
    'image_numbers.json' => 'Bildnummer-räknare'
];

foreach ($dataFiles as $file => $description) {
    $path = $rootDir . '/data/' . $file;
    echo "\n-- $file ($description) --\n";
    echo "path: $path\n";
    echo "exists: " . (file_exists($path) ? 'yes' : 'no') . "\n";
    echo "readable: " . (is_readable($path) ? 'yes' : 'no') . "\n";
    echo "writable: " . (is_writable($path) ? 'yes' : 'no') . "\n";
    
    if (file_exists($path) && is_readable($path)) {
        $raw = @file_get_contents($path);
        echo "bytes: " . ($raw === false ? 'read-failed' : strlen($raw)) . "\n";
        if ($raw !== false) {
            $encOk = mb_check_encoding($raw, 'UTF-8') ? 'utf-8' : 'not-utf8';
            echo "encoding: $encOk\n";
            $data = json_decode($raw, true);
            if (is_array($data)) {
                echo "json_valid: yes (" . count($data) . " items)\n";
                if ($file === 'projects.json') {
                    $active = 0;
                    $archived = 0;
                    foreach ($data as $item) {
                        $status = $item['status'] ?? 'active';
                        if ($status === 'archived') $archived++;
                        else $active++;
                    }
                    echo "projects_active: $active\n";
                    echo "projects_archived: $archived\n";
                } elseif ($file === 'userids.json') {
                    $active = 0;
                    $inactive = 0;
                    foreach ($data as $item) {
                        if (($item['inactive'] ?? false) === true) $inactive++;
                        else $active++;
                    }
                    echo "users_active: $active\n";
                    echo "users_inactive: $inactive\n";
                } elseif ($file === 'image_numbers.json') {
                    echo "next_number: " . ($data['next_number'] ?? 'not set') . "\n";
                }
            } else {
                echo "json_valid: FAIL - " . json_last_error_msg() . "\n";
            }
        }
    }
}

// ===== DIRECTORIES =====
echo "\n== DIRECTORIES ==\n";
$directories = [
    'uploads' => 'Uppladdade bilder och videor',
    'logs' => 'Loggfiler',
    'downloads' => 'Paketerade projekt',
    'data' => 'Konfigurationsdata'
];

foreach ($directories as $dir => $description) {
    $path = $rootDir . '/' . $dir;
    echo "\n-- $dir ($description) --\n";
    echo "path: $path\n";
    echo "exists: " . (is_dir($path) ? 'yes' : 'no') . "\n";
    if (is_dir($path)) {
        echo "readable: " . (is_readable($path) ? 'yes' : 'no') . "\n";
        echo "writable: " . (is_writable($path) ? 'yes' : 'no') . "\n";
        
        // Testa skrivning
        if ($dir === 'uploads' || $dir === 'downloads' || $dir === 'logs') {
            $probe = $path . '/__write_test__.txt';
            $w = @file_put_contents($probe, "test " . date('c'));
            echo "write_test: " . ($w === false ? 'FAIL' : 'OK') . "\n";
            if ($w !== false) { @unlink($probe); }
        }
        
        // Räkna filer i uploads
        if ($dir === 'uploads' && is_dir($path)) {
            $projectDirs = glob($path . '/*', GLOB_ONLYDIR);
            $totalFiles = 0;
            $totalSize = 0;
            foreach ($projectDirs as $projectDir) {
                $files = glob($projectDir . '/*.{jpg,jpeg,png,webp,heic,heif,mp4,webm,mov,avi}', GLOB_BRACE);
                $totalFiles += count($files);
                foreach ($files as $file) {
                    $totalSize += filesize($file);
                }
            }
            echo "projects: " . count($projectDirs) . "\n";
            echo "total_files: $totalFiles\n";
            echo "total_size: " . round($totalSize / 1024 / 1024, 2) . " MB\n";
        }
        
        // Räkna zip-filer i downloads
        if ($dir === 'downloads' && is_dir($path)) {
            $zipFiles = glob($path . '/*.zip');
            $totalSize = 0;
            foreach ($zipFiles as $zip) {
                $totalSize += filesize($zip);
            }
            echo "zip_files: " . count($zipFiles) . "\n";
            echo "total_size: " . round($totalSize / 1024 / 1024, 2) . " MB\n";
        }
    }
}

// ===== SECURITY FILES =====
echo "\n== SECURITY FILES ==\n";
$securityFiles = [
    'security.php' => 'Säkerhetsfunktioner',
    'admin_guard.php' => 'Admin-skydd',
    'rights.php' => 'Rättighetshantering'
];

foreach ($securityFiles as $file => $description) {
    if ($file === 'admin_guard.php') {
        $path = $rootDir . '/adm/' . $file;
    } else {
        $path = $rootDir . '/' . $file;
    }
    echo "$file: " . (file_exists($path) ? 'OK' : 'MISSING') . " ($description)\n";
}

// ===== LOG FILES =====
echo "\n== LOG FILES ==\n";
$logFiles = [
    'logs/upload_limits.json' => 'Rate limiting data',
    'logs/security.log' => 'Säkerhetsloggar'
];

foreach ($logFiles as $file => $description) {
    $path = $rootDir . '/' . $file;
    echo "$file: " . (file_exists($path) ? 'OK (' . filesize($path) . ' bytes)' : 'NOT CREATED YET') . " ($description)\n";
}

// ===== CHATT =====
echo "\n== CHATT ==\n";
$chatsDir = $rootDir . '/chats';
echo "chats_dir: $chatsDir\n";
echo "exists: " . (is_dir($chatsDir) ? 'yes' : 'no') . "\n";

if (is_dir($chatsDir)) {
    echo "readable: " . (is_readable($chatsDir) ? 'yes' : 'no') . "\n";
    echo "writable: " . (is_writable($chatsDir) ? 'yes' : 'no') . "\n";
    
    // Räkna projektchattar
    $projectChatDirs = glob($chatsDir . '/*', GLOB_ONLYDIR);
    $totalChats = count($projectChatDirs);
    echo "project_chats: $totalChats\n";
    
    $totalMessages = 0;
    $totalChatImages = 0;
    $totalChatSize = 0;
    $invalidMessages = 0;
    $projectsWithChat = [];
    
    foreach ($projectChatDirs as $chatDir) {
        $projectId = basename($chatDir);
        $messagesFile = $chatDir . '/messages.json';
        
        if (file_exists($messagesFile)) {
            $projectsWithChat[] = $projectId;
            $content = @file_get_contents($messagesFile);
            if ($content !== false) {
                $messages = json_decode($content, true);
                if (is_array($messages)) {
                    $messageCount = count($messages);
                    $totalMessages += $messageCount;
                    
                    // Kontrollera meddelandestruktur
                    foreach ($messages as $msg) {
                        $required = ['id', 'created_at'];
                        $hasRequired = true;
                        foreach ($required as $key) {
                            if (!isset($msg[$key])) {
                                $hasRequired = false;
                                break;
                            }
                        }
                        if (!$hasRequired) {
                            $invalidMessages++;
                        }
                        
                        // Räkna bilder i chatt
                        if (isset($msg['type']) && $msg['type'] === 'image' && isset($msg['media_file'])) {
                            $imagePath = $chatDir . '/' . $msg['media_file'];
                            if (file_exists($imagePath)) {
                                $totalChatImages++;
                                $totalChatSize += filesize($imagePath);
                            }
                        }
                    }
                } else {
                    echo "WARNING: Invalid JSON in $messagesFile\n";
                }
            }
        }
    }
    
    echo "total_messages: $totalMessages\n";
    echo "total_chat_images: $totalChatImages\n";
    echo "total_chat_size: " . round($totalChatSize / 1024 / 1024, 2) . " MB\n";
    echo "invalid_messages: $invalidMessages\n";
    echo "projects_with_chat: " . count($projectsWithChat) . "\n";
    
    // Kontrollera mapping-filer om de finns
    $projectMappingFile = $rootDir . '/data/project_mapping.json';
    $chatMappingFile = $rootDir . '/data/chat_mapping.json';
    
    if (file_exists($projectMappingFile)) {
        echo "project_mapping.json: OK (" . filesize($projectMappingFile) . " bytes)\n";
        $mapping = json_decode(file_get_contents($projectMappingFile), true);
        if (is_array($mapping)) {
            echo "project_mappings: " . count($mapping) . "\n";
        }
    } else {
        echo "project_mapping.json: NOT FOUND (using repository)\n";
    }
    
    if (file_exists($chatMappingFile)) {
        echo "chat_mapping.json: OK (" . filesize($chatMappingFile) . " bytes)\n";
        $mapping = json_decode(file_get_contents($chatMappingFile), true);
        if (is_array($mapping)) {
            echo "chat_mappings: " . count($mapping) . "\n";
        }
    } else {
        echo "chat_mapping.json: NOT FOUND (using repository)\n";
    }
} else {
    echo "WARNING: chats directory does not exist\n";
}

// ===== PAKETERING =====
echo "\n== PAKETERING ==\n";
$downloadsDir = $rootDir . '/downloads';
echo "downloads_dir: $downloadsDir\n";
echo "exists: " . (is_dir($downloadsDir) ? 'yes' : 'no') . "\n";

if (is_dir($downloadsDir)) {
    echo "readable: " . (is_readable($downloadsDir) ? 'yes' : 'no') . "\n";
    echo "writable: " . (is_writable($downloadsDir) ? 'yes' : 'no') . "\n";
    
    // Hitta alla zip-filer
    $zipFiles = glob($downloadsDir . '/*.zip');
    $totalZipFiles = count($zipFiles);
    echo "zip_files: $totalZipFiles\n";
    
    $totalZipSize = 0;
    $validZipFiles = 0;
    $invalidZipFiles = 0;
    $zipFilesWithMetadata = 0;
    $zipFilesWithoutMetadata = 0;
    $packagesByProject = [];
    
    foreach ($zipFiles as $zipFile) {
        $fileSize = filesize($zipFile);
        $totalZipSize += $fileSize;
        
        // Kontrollera om zip-filen är giltig
        $zip = new ZipArchive();
        $result = $zip->open($zipFile, ZipArchive::CHECKCONS);
        if ($result === TRUE) {
            $validZipFiles++;
            $zip->close();
        } else {
            $invalidZipFiles++;
        }
        
        // Kontrollera om metadata-fil finns
        $metadataFile = preg_replace('/\.zip$/', '.json', $zipFile);
        if (file_exists($metadataFile)) {
            $zipFilesWithMetadata++;
            
            // Läs metadata
            $metadata = json_decode(file_get_contents($metadataFile), true);
            if (is_array($metadata)) {
                $projectId = $metadata['project_id'] ?? 'unknown';
                if (!isset($packagesByProject[$projectId])) {
                    $packagesByProject[$projectId] = 0;
                }
                $packagesByProject[$projectId]++;
            }
        } else {
            $zipFilesWithoutMetadata++;
        }
    }
    
    echo "total_zip_size: " . round($totalZipSize / 1024 / 1024, 2) . " MB\n";
    echo "valid_zip_files: $validZipFiles\n";
    echo "invalid_zip_files: $invalidZipFiles\n";
    echo "zip_files_with_metadata: $zipFilesWithMetadata\n";
    echo "zip_files_without_metadata: $zipFilesWithoutMetadata\n";
    echo "unique_projects_packaged: " . count($packagesByProject) . "\n";
    
    // Visa största zip-filer
    if ($totalZipFiles > 0) {
        usort($zipFiles, function($a, $b) {
            return filesize($b) - filesize($a);
        });
        echo "largest_zip_files:\n";
        $topCount = min(5, $totalZipFiles);
        for ($i = 0; $i < $topCount; $i++) {
            $size = filesize($zipFiles[$i]);
            $name = basename($zipFiles[$i]);
            echo "  - $name: " . round($size / 1024 / 1024, 2) . " MB\n";
        }
    }
} else {
    echo "WARNING: downloads directory does not exist\n";
}

// ===== ROOT DIRECTORY STRUCTURE =====
echo "\n== ROOT DIRECTORY STRUCTURE ==\n";
$items = @scandir($rootDir);
if ($items !== false) {
    $importantItems = array_filter($items, function($item) {
        return $item !== '.' && $item !== '..' && substr($item, 0, 1) !== '.';
    });
    foreach ($importantItems as $it) {
        $fullPath = $rootDir . '/' . $it;
        $type = is_dir($fullPath) ? '[DIR]' : '[FILE]';
        echo "$type $it\n";
    }
} else {
    echo "scandir failed\n";
}

echo "\n========================================\n";
echo "  DIAGNOSTIK KLAR\n";
echo "========================================\n";

dlog("diag run done");

