<?php
require_once __DIR__ . '/../../config.php';
require_once __DIR__ . '/../../security.php';

setSecurityHeaders();
require_once __DIR__ . '/../admin_guard.php';

// admin_guard.php sätter redan $userId, men vi behöver den för JavaScript
// Om den inte är satt, försök hämta från GET eller session
if (!isset($userId) || empty($userId)) {
    $userId = $_GET['user_id'] ?? $_SESSION['user_id'] ?? '';
}
?>
<!DOCTYPE html>
<html lang="sv">
<head>
    <meta charset="utf-8" />
    <meta name="viewport" content="width=device-width,initial-scale=1" />
    <title>Extern användare - ByggArkiv</title>
    <link rel="stylesheet" href="../common.css">
    <style>
        /* Sid-specifika stilar för external_parties.php */
        .modal {
            display: none;
            position: fixed;
            z-index: 1000;
            left: 0;
            top: 0;
            width: 100%;
            height: 100%;
            background-color: rgba(0,0,0,0.5);
            align-items: center;
            justify-content: center;
        }
        .modal-content {
            background: white;
            border-radius: 12px;
            padding: 30px;
            max-width: 500px;
            width: 90%;
            max-height: 90vh;
            overflow-y: auto;
        }
        #modalTitle {
            color: #1a73e8;
        }
        #newCompanyModal h2 {
            color: #1a73e8;
        }
        .btn {
            margin-right: 10px;
            margin-bottom: 10px;
        }
        th, td {
            padding: 12px;
        }
        .content-wrapper {
            background: white;
            border-radius: 12px;
            padding: 20px;
            box-shadow: 0 2px 8px rgba(0,0,0,0.1);
            margin-top: 20px;
        }
        .staff-profile-image {
            width: 48px;
            height: 48px;
            border-radius: 50%;
            object-fit: cover;
            cursor: pointer;
            border: 2px solid #e0e0e0;
            transition: all 0.2s;
            margin-right: 12px;
        }
        .staff-profile-image:hover {
            border-color: #1a73e8;
            transform: scale(1.05);
        }
        .staff-profile-placeholder {
            width: 48px;
            height: 48px;
            border-radius: 50%;
            background: #f0f0f0;
            display: flex;
            align-items: center;
            justify-content: center;
            cursor: pointer;
            border: 2px solid #e0e0e0;
            margin-right: 12px;
            transition: all 0.2s;
        }
        .staff-profile-placeholder:hover {
            border-color: #1a73e8;
            background: #f5f5f5;
        }
        .staff-profile-placeholder svg {
            width: 24px;
            height: 24px;
            color: #999;
        }
        .image-upload-overlay {
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(0, 0, 0, 0.5);
            display: flex;
            align-items: center;
            justify-content: center;
            z-index: 2000;
        }
        .image-upload-box {
            background: white;
            border-radius: 12px;
            padding: 24px;
            box-shadow: 0 8px 32px rgba(0,0,0,0.3);
            max-width: 500px;
            width: 90%;
        }
        .image-upload-preview {
            width: 200px;
            height: 200px;
            border-radius: 50%;
            object-fit: cover;
            margin: 0 auto 20px;
            display: block;
        }
        .image-upload-preview-placeholder {
            width: 200px;
            height: 200px;
            border-radius: 50%;
            background: #f0f0f0;
            display: flex;
            align-items: center;
            justify-content: center;
            margin: 0 auto 20px;
        }
        .image-upload-preview-placeholder svg {
            width: 64px;
            height: 64px;
            color: #999;
        }
        @keyframes spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }
        .popup-overlay {
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(0,0,0,0.5);
            display: flex;
            align-items: center;
            justify-content: center;
            z-index: 10000;
        }
        .popup-title {
            margin: 0 0 16px 0;
            color: #333;
            font-size: 20px;
            font-weight: 600;
        }
        .popup-description {
            margin: 0 0 16px 0;
            color: #666;
            font-size: 14px;
        }
        .popup-code {
            background: #f5f5f5;
            padding: 12px;
            border-radius: 8px;
            margin-bottom: 20px;
            word-break: break-all;
            font-family: monospace;
            font-size: 14px;
            color: #333;
            border: 1px solid #ddd;
        }
        .popup-buttons {
            display: flex;
            gap: 12px;
            justify-content: center;
        }
        .popup-btn {
            padding: 10px 20px;
            border-radius: 8px;
            border: none;
            font-size: 16px;
            cursor: pointer;
            font-weight: 500;
            transition: background-color 0.2s;
        }
        .popup-btn-primary {
            background: #1a73e8;
            color: white;
        }
        .popup-btn-primary:hover {
            background: #1557b0;
        }
        .popup-btn-secondary {
            background: #6c757d;
            color: white;
        }
        .popup-btn-secondary:hover {
            background: #5a6268;
        }
        /* Kortbaserad lista */
        .staff-list {
            display: flex;
            flex-direction: column;
            gap: 12px;
        }
        
        .staff-card {
            display: flex;
            align-items: center;
            padding: 16px;
            background: #f8f9fa;
            border: 1px solid #e9ecef;
            border-radius: 8px;
            transition: all 0.2s;
            gap: 16px;
        }
        
        .staff-card:hover {
            background: #e9ecef;
            border-color: #1a73e8;
        }
        
        .staff-card.inactive {
            background-color: #ffebee;
            border-color: #ffcdd2;
        }
        
        .staff-card-image {
            flex-shrink: 0;
        }
        
        .staff-card-info {
            flex: 1;
            min-width: 0;
        }
        
        .staff-card-name {
            font-size: 16px;
            font-weight: 600;
            color: #333;
            margin-bottom: 4px;
            cursor: pointer;
            transition: color 0.2s;
        }
        
        .staff-card-name:hover {
            color: #1a73e8;
        }
        
        .staff-card-company {
            font-size: 14px;
            color: #666;
            margin-bottom: 4px;
        }
        .staff-card-company a {
            color: #1a73e8;
            text-decoration: none;
            cursor: pointer;
        }
        .staff-card-company a:hover {
            text-decoration: underline;
        }
        
        .staff-card-contact {
            font-size: 13px;
            color: #888;
        }
        
        .staff-card-meta {
            flex-shrink: 0;
            text-align: right;
            min-width: 150px;
        }
        
        .staff-card-type {
            font-size: 13px;
            color: #666;
            margin-bottom: 4px;
        }
        
        .staff-card-ids {
            font-family: monospace;
            font-size: 11px;
            color: #999;
            word-break: break-all;
        }
        
        .staff-card-actions {
            display: flex;
            gap: 8px;
            align-items: center;
            flex-shrink: 0;
        }
        
        @media (max-width: 768px) {
            .staff-card {
                flex-wrap: wrap;
            }
            .staff-card-meta {
                text-align: left;
                min-width: auto;
                width: 100%;
                margin-top: 8px;
            }
        }
    </style>
</head>
<body>
    <div class="container">
        <header class="header">
            <a href="../../index.html<?= !empty($userId) ? '?user_id=' . htmlspecialchars($userId) : '' ?>">
                <img src="../../data/logo.png" alt="ByggArkiv Snickare Stefan" class="logo">
            </a>
            <div class="back-link" style="margin-top: 15px;">
                <a href="../admin.php<?= !empty($userId) ? '?user_id=' . htmlspecialchars($userId) : '' ?>">← Tillbaka till admin</a>
            </div>
        </header>
        
        <div class="content-wrapper">
            <h1 style="color: #1a73e8; text-align: left; margin-top: 0; margin-bottom: 20px;">Extern användare</h1>
            <p style="color: #666; font-size: 14px; margin-bottom: 20px; line-height: 1.6;">
                Här kan du hantera externa användare som kan delas med i projektchattar. Lägg till nya användare, redigera deras uppgifter, välj vilka projektchattar de ska vara med i, och hantera deras profilbilder.
            </p>
            
            <button class="btn btn-primary" onclick="showAddModal()">Lägg till</button>
            
            <div id="status" style="display: none; padding: 12px; margin: 15px 0; border-radius: 8px; text-align: center; font-weight: 500;"></div>
            
            <div id="externalStaffList" style="margin-top: 20px;">
                <p>Laddar...</p>
            </div>
        </div>
    </div>
    
    <!-- Modal för att lägga till/redigera -->
    <div id="editModal" class="modal">
        <div class="modal-content">
            <h2 id="modalTitle">Lägg till extern användare</h2>
            <form id="externalStaffForm">
                <input type="hidden" id="editId" name="id">
                
                <div class="form-group">
                    <label for="name">Namn *</label>
                    <input type="text" id="name" name="name" required>
                </div>
                
                <div class="form-group">
                    <label for="nickname">Nickname (valfritt, unikt):</label>
                    <input type="text" id="nickname" name="nickname" placeholder="Kortnamn som visas i chatten">
                    <small style="color: #666; font-size: 12px; display: block; margin-top: 4px;">
                        Om satt, visas detta istället för namn i chatten. Måste vara unikt.
                    </small>
                </div>
                
                <div class="form-group">
                    <label for="mobile">Mobilnummer *</label>
                    <input type="text" id="mobile" name="mobile" required>
                </div>
                
                <div class="form-group">
                    <label for="email">E-post</label>
                    <input type="email" id="email" name="email">
                </div>
                
                <div class="form-group" style="display: flex; gap: 20px;">
                    <div style="flex: 1;">
                        <label for="company_id">Företag</label>
                        <select id="company_id" name="company_id" style="width: 100%;">
                            <option value="">Välj företag...</option>
                        </select>
                    </div>
                    <div style="flex: 1;">
                        <label for="type_id">Yrkeskategori</label>
                        <select id="type_id" name="type_id" style="width: 100%;">
                            <option value="">Välj yrkeskategori...</option>
                        </select>
                    </div>
                </div>
                
                <div class="form-group">
                    <label for="pushover_user_key">Pushover User Key (valfritt)</label>
                    <input type="text" id="pushover_user_key" name="pushover_user_key" placeholder="u1234567890abcdefghijklmnop" maxlength="50" style="font-family: monospace;">
                    <p style="font-size: 12px; color: #666; margin-top: 4px; margin-bottom: 0;">Hämta från Pushover-appen: Inställningar → Your User Key</p>
                </div>
                
                <div class="form-group">
                    <label for="notes">Anteckningar (valfritt)</label>
                    <textarea id="notes" name="notes" rows="3" placeholder="Interna anteckningar om denna partner..." style="width: 100%; padding: 10px; border: 1px solid #ccc; border-radius: 8px; font-size: 14px; font-family: inherit; resize: vertical;"></textarea>
                </div>
                
                <div class="form-group">
                    <label for="public_notes">Publika anteckningar (valfritt)</label>
                    <textarea id="public_notes" name="public_notes" rows="3" placeholder="Anteckningar som visas i visitkortet..." style="width: 100%; padding: 10px; border: 1px solid #ccc; border-radius: 8px; font-size: 14px; font-family: inherit; resize: vertical;"></textarea>
                    <p style="font-size: 12px; color: #666; margin-top: 4px; margin-bottom: 0;">Dessa anteckningar visas när visitkortet öppnas, t.ex. i chatten</p>
                </div>
                
                <div class="form-group">
                    <div class="checkbox-group">
                        <input type="checkbox" id="active" name="active" value="1" checked>
                        <label for="active" style="margin-bottom: 0; font-weight: 400; cursor: pointer;">Aktiv</label>
                    </div>
                    <p style="font-size: 12px; color: #666; margin-top: 4px; margin-bottom: 0;">Avmarkera för att inaktivera den externa partnern</p>
                </div>
                
                <div class="form-group">
                    <div class="checkbox-group" style="display: flex; align-items: center; gap: 8px;">
                        <input type="checkbox" id="can_chat" name="can_chat" value="1" checked style="margin: 0;">
                        <label for="can_chat" style="margin-bottom: 0; font-weight: 600; cursor: pointer;">Kan delta i projektchatt</label>
                    </div>
                </div>
                
                <div class="form-group">
                    <div class="checkbox-group" style="display: flex; align-items: center; gap: 8px;">
                        <input type="checkbox" id="can_upload" name="can_upload" value="1" style="margin: 0;">
                        <label for="can_upload" style="margin-bottom: 0; font-weight: 600; cursor: pointer;">Kommer åt byggarkiv</label>
                    </div>
                </div>
                
                <div style="margin-top: 20px; display: flex; justify-content: space-between; align-items: center;">
                    <div>
                        <button type="button" class="btn btn-danger" id="deleteButton" onclick="deleteStaffFromModal()" style="display: none;">Ta bort</button>
                    </div>
                    <div>
                        <button type="submit" class="btn btn-primary">Spara</button>
                        <button type="button" class="btn" onclick="closeModal()">Avbryt</button>
                    </div>
                </div>
            </form>
        </div>
    </div>
    
    <!-- Modal för att skapa nytt företag -->
    <div id="newCompanyModal" class="modal">
        <div class="modal-content">
            <h2>Skapa nytt företag</h2>
            <form id="newCompanyForm">
                <div class="form-group">
                    <label for="company_name">Företagsnamn *</label>
                    <input type="text" id="company_name" name="name" required>
                </div>
                
                <div class="form-group">
                    <label for="company_visit_address">Besöksadress</label>
                    <input type="text" id="company_visit_address" name="visit_address">
                </div>
                
                <div class="form-group">
                    <label for="company_postal_address">Postadress</label>
                    <input type="text" id="company_postal_address" name="postal_address">
                </div>
                
                <div class="form-group">
                    <label for="company_phone">Telefon</label>
                    <input type="text" id="company_phone" name="phone">
                </div>
                
                <div class="form-group">
                    <label for="company_contact_person">Kontaktperson</label>
                    <input type="text" id="company_contact_person" name="contact_person">
                </div>
                
                <div style="margin-top: 20px; display: flex; justify-content: flex-end; gap: 10px;">
                    <button type="submit" class="btn btn-primary">Spara</button>
                    <button type="button" class="btn" onclick="closeNewCompanyModal()">Avbryt</button>
                </div>
            </form>
        </div>
    </div>
    
    <script>
        const userId = '<?= htmlspecialchars($userId ?? '') ?>';
        let companies = [];
        let types = [];
        let externalStaff = [];
        
        // Funktion för att visa statusmeddelanden
        function showStatus(message, type) {
            const statusElement = document.getElementById('status');
            if (!statusElement) return;
            
            statusElement.textContent = message;
            statusElement.className = 'status ' + type;
            
            // Sätt färg baserat på typ
            if (type === 'success') {
                statusElement.style.backgroundColor = '#d4edda';
                statusElement.style.color = '#155724';
                statusElement.style.border = '1px solid #c3e6cb';
            } else if (type === 'error') {
                statusElement.style.backgroundColor = '#f8d7da';
                statusElement.style.color = '#721c24';
                statusElement.style.border = '1px solid #f5c6cb';
            } else if (type === 'info') {
                statusElement.style.backgroundColor = '#d1ecf1';
                statusElement.style.color = '#0c5460';
                statusElement.style.border = '1px solid #bee5eb';
            }
            
            statusElement.style.display = 'block';
            
            setTimeout(function() {
                statusElement.style.display = 'none';
            }, 5000);
        }
        
        async function loadCompanies() {
            try {
                let url = `../../api/chat/external_staff.php?action=list_companies`;
                if (userId) {
                    url += `&user_id=${encodeURIComponent(userId)}`;
                }
                const res = await fetch(url);
                if (!res.ok) {
                    throw new Error(`HTTP ${res.status}: ${res.statusText}`);
                }
                const data = await res.json();
                if (data.success && data.companies && Array.isArray(data.companies)) {
                    companies = data.companies;
                    const select = document.getElementById('company_id');
                    if (select) {
                        while (select.options.length > 1) {
                            select.remove(1);
                        }
                        companies.forEach(company => {
                            const option = document.createElement('option');
                            option.value = company.id;
                            option.textContent = company.name;
                            select.appendChild(option);
                        });
                        
                        // Lägg till "> Nytt företag..." som sista alternativ om det är en ny användare
                        const editId = document.getElementById('editId');
                        if (editId && !editId.value) {
                            const newCompanyOption = document.createElement('option');
                            newCompanyOption.value = '__new_company__';
                            newCompanyOption.textContent = '▶ Nytt företag...';
                            newCompanyOption.style.fontStyle = 'italic';
                            select.appendChild(newCompanyOption);
                        }
                        
                    } else {
                    }
                } else {
                }
            } catch (e) {
            }
        }
        
        async function loadTypes() {
            try {
                let url = `../../api/chat/external_staff.php?action=list_types`;
                if (userId) {
                    url += `&user_id=${encodeURIComponent(userId)}`;
                }
                const res = await fetch(url);
                if (!res.ok) {
                    throw new Error(`HTTP ${res.status}: ${res.statusText}`);
                }
                const data = await res.json();
                if (data.success && data.types && Array.isArray(data.types)) {
                    types = data.types;
                    // Sortera yrkeskategorier efter namn
                    types.sort((a, b) => {
                        const nameA = (a.name || a.type_name || '').toLowerCase();
                        const nameB = (b.name || b.type_name || '').toLowerCase();
                        return nameA.localeCompare(nameB, 'sv');
                    });
                    const select = document.getElementById('type_id');
                    if (select) {
                        // Rensa alla options utom första (placeholder)
                        while (select.options.length > 1) {
                            select.remove(1);
                        }
                        // Lägg till yrkeskategorier
                        types.forEach(type => {
                            const option = document.createElement('option');
                            option.value = type.id || type.type_id || '';
                            option.textContent = type.name || type.type_name || '';
                            select.appendChild(option);
                        });
                    } else {
                        console.error('type_id select element not found');
                    }
                } else {
                    console.error('Failed to load types:', data);
                }
            } catch (e) {
                console.error('Error loading types:', e);
            }
        }
        
        async function loadExternalStaff() {
            try {
                const res = await fetch(`../../api/chat/external_staff.php?action=list&user_id=${encodeURIComponent(userId)}&include_inactive=1`);
                const data = await res.json();
                if (data.success) {
                    externalStaff = data.staff || [];
                    renderList();
                }
            } catch (e) {
                document.getElementById('externalStaffList').innerHTML = '<p style="color: red;">Fel vid laddning av extern användare</p>';
            }
        }
        
        function renderList() {
            const container = document.getElementById('externalStaffList');
            if (externalStaff.length === 0) {
                container.innerHTML = '<p>Inga externa användare registrerade än.</p>';
                return;
            }
            
            // Sortera så att inaktiva visas sist
            const sortedStaff = [...externalStaff].sort((a, b) => {
                const aActive = a.active !== false;
                const bActive = b.active !== false;
                if (aActive && !bActive) return -1;
                if (!aActive && bActive) return 1;
                return (a.name || '').localeCompare(b.name || '');
            });
            
            let html = '<div class="staff-list">';
            
            sortedStaff.forEach(staff => {
                const isActive = staff.active !== false;
                const cardClass = isActive ? '' : 'inactive';
                
                const staffId = staff.user_id || staff.id || '';
                const safeStaffId = escapeHtml(staffId);
                
                const profileImageUrl = staff.profile_image ? `../../data/${staff.profile_image}?t=${Date.now()}` : null;
                let profileImageHtml = '';
                
                if (profileImageUrl) {
                    profileImageHtml = `<img src="${profileImageUrl}" alt="Profilbild" class="staff-profile-image" onclick="uploadStaffImage('${safeStaffId}')" title="Klicka för att ändra bild" onerror="this.onerror=null; this.style.display='none'; const ph=this.nextElementSibling; if(ph) ph.style.display='flex';">` +
                        `<div class="staff-profile-placeholder" onclick="uploadStaffImage('${safeStaffId}')" title="Klicka för att ladda upp bild" style="display:none;">` +
                        `<svg width="24" height="24" viewBox="0 0 24 24" fill="currentColor"><path d="M12 12c2.21 0 4-1.79 4-4s-1.79-4-4-4-4 1.79-4 4 1.79 4 4 4zm0 2c-2.67 0-8 1.34-8 4v2h16v-2c0-2.66-5.33-4-8-4z"/></svg>` +
                        `</div>`;
                } else {
                    profileImageHtml = `<div class="staff-profile-placeholder" onclick="uploadStaffImage('${safeStaffId}')" title="Klicka för att ladda upp bild">` +
                        `<svg width="24" height="24" viewBox="0 0 24 24" fill="currentColor"><path d="M12 12c2.21 0 4-1.79 4-4s-1.79-4-4-4-4 1.79-4 4 1.79 4 4 4zm0 2c-2.67 0-8 1.34-8 4v2h16v-2c0-2.66-5.33-4-8-4z"/></svg>` +
                        `</div>`;
                }
                
                const userId = staff.user_id || staff.id || '';
                const pushoverKey = staff.pushover_user_key || '';
                
                // Kontrollera can_chat och can_upload
                const canChat = staff.can_chat !== false; // true om can_chat är true eller undefined
                const canUpload = staff.can_upload !== false; // true om can_upload är true eller undefined
                
                // Ikoner för can_chat och can_upload
                const chatIcon = canChat 
                    ? '<svg width="18" height="18" viewBox="0 0 24 24" fill="#4caf50" style="vertical-align: middle; margin-left: 8px;" title="Kan delta i projektchatt"><path d="M20 2H4c-1.1 0-2 .9-2 2v18l4-4h14c1.1 0 2-.9 2-2V4c0-1.1-.9-2-2-2zm0 14H6l-2 2V4h16v12z"/></svg>'
                    : '<svg width="18" height="18" viewBox="0 0 24 24" fill="#999" style="vertical-align: middle; margin-left: 8px; opacity: 0.4;" title="Kan inte delta i projektchatt"><path d="M20 2H4c-1.1 0-2 .9-2 2v18l4-4h14c1.1 0 2-.9 2-2V4c0-1.1-.9-2-2-2zm0 14H6l-2 2V4h16v12z"/></svg>';
                
                const uploadIcon = canUpload
                    ? '<svg width="18" height="18" viewBox="0 0 24 24" fill="#2196f3" style="vertical-align: middle; margin-left: 8px;" title="Kommer åt byggarkiv"><path d="M9 2L7.17 4H4c-1.1 0-2 .9-2 2v12c0 1.1.9 2 2 2h16c1.1 0 2-.9 2-2V6c0-1.1-.9-2-2-2h-3.17L15 2H9zm3 15c-2.76 0-5-2.24-5-5s2.24-5 5-5 5 2.24 5 5-2.24 5-5 5z"/></svg>'
                    : '<svg width="18" height="18" viewBox="0 0 24 24" fill="#999" style="vertical-align: middle; margin-left: 8px; opacity: 0.4;" title="Kommer inte åt byggarkiv"><path d="M9 2L7.17 4H4c-1.1 0-2 .9-2 2v12c0 1.1.9 2 2 2h16c1.1 0 2-.9 2-2V6c0-1.1-.9-2-2-2h-3.17L15 2H9zm3 15c-2.76 0-5-2.24-5-5s2.24-5 5-5 5 2.24 5 5-2.24 5-5 5z"/></svg>';
                
                html += `<div class="staff-card ${cardClass}">
                    <div class="staff-card-image">${profileImageHtml}</div>
                    <div class="staff-card-info">
                        <div class="staff-card-name" onmouseenter="showUserCard('${safeStaffId}', event)" onmouseleave="hideUserCard()">
                            ${escapeHtml(staff.name)}${!isActive ? ' <span style="color: #d32f2f; font-weight: normal; font-size: 12px;">(inaktiv)</span>' : ''}
                            ${chatIcon}
                            ${uploadIcon}
                        </div>
                        <div class="staff-card-company">${staff.company_id && staff.company_name ? `<a href="external_companies.php?company_id=${encodeURIComponent(staff.company_id)}" title="Hantera extern partner">${escapeHtml(staff.company_name)}</a>` : escapeHtml(staff.company_name || '')}</div>
                        <div class="staff-card-contact">${escapeHtml(staff.mobile || '')}${staff.email ? ' • ' + escapeHtml(staff.email) : ''}</div>
                    </div>
                    <div class="staff-card-meta">
                        <div class="staff-card-type">${escapeHtml(staff.type_name || '')}</div>
                        <div class="staff-card-ids">
                            <div>u=${escapeHtml(userId || 'Saknas')}</div>
                            <div>p=${escapeHtml(pushoverKey || 'Saknas')}</div>
                        </div>
                    </div>
                    <div class="staff-card-actions">
                        <button class="action-btn btn-link" onclick="openStaffLink('${escapeHtml(staff.id)}')" title="Skapa app-länk för extern användare" id="link-btn-${escapeHtml(staff.id)}">
                            <svg width="20" height="20" viewBox="0 0 24 24" fill="currentColor">
                                <path d="M3.9 12c0-1.71 1.39-3.1 3.1-3.1h4V7H7c-2.76 0-5 2.24-5 5s2.24 5 5 5h4v-1.9H7c-1.71 0-3.1-1.39-3.1-3.1zM8 13h8v-2H8v2zm9-6h-4v1.9h4c1.71 0 3.1 1.39 3.1 3.1s-1.39 3.1-3.1 3.1h-4V17h4c2.76 0 5-2.24 5-5s-2.24-5-5-5z"/>
                            </svg>
                        </button>
                        <button class="action-btn btn-edit" onclick="editStaff('${escapeHtml(staff.id)}')" title="Redigera extern användare">
                            <svg width="20" height="20" viewBox="0 0 24 24" fill="currentColor">
                                <path d="M3 17.25V21h3.75L17.81 9.94l-3.75-3.75L3 17.25zM20.71 7.04c.39-.39.39-1.02 0-1.41l-2.34-2.34c-.39-.39-1.02-.39-1.41 0l-1.83 1.83 3.75 3.75 1.83-1.83z"></path>
                            </svg>
                        </button>
                    </div>
                </div>`;
            });
            
            html += '</div>';
            container.innerHTML = html;
        }
        
        function escapeHtml(text) {
            const div = document.createElement('div');
            div.textContent = text;
            return div.innerHTML;
        }
        
        let userCardTimeout = null;
        let currentUserCard = null;
        
        // Visa visitkort när man hovrar över namnet
        async function showUserCard(userId, event) {
            // Rensa tidigare timeout
            if (userCardTimeout) {
                clearTimeout(userCardTimeout);
                userCardTimeout = null;
            }
            
            // Ta bort tidigare visitkort om det finns
            if (currentUserCard) {
                currentUserCard.remove();
                currentUserCard = null;
            }
            
            // Vänta lite innan visitkortet visas (för att undvika att det blinkar)
            userCardTimeout = setTimeout(async () => {
                try {
                    // Hämta användardata
                    const response = await fetch(`../../chat/participant_info_api.php?participant_id=${encodeURIComponent(userId)}`);
                    const data = await response.json();
                    
                    if (!data.success || !data.data) {
                        return; // Tyst misslyckande
                    }
                    
                    const userData = data.data;
                    
                    // Hämta position för namnet
                    const nameElement = event.target;
                    const rect = nameElement.getBoundingClientRect();
                    
                    // Beräkna position för visitkortet
                    let top = rect.bottom + 10;
                    let left = rect.left;
                    
                    // Kontrollera om visitkortet hamnar utanför skärmen
                    const cardWidth = 350;
                    const cardHeight = 400; // Uppskattad höjd
                    
                    // Justera horisontellt om det hamnar utanför högerkanten
                    if (left + cardWidth > window.innerWidth) {
                        left = window.innerWidth - cardWidth - 10;
                    }
                    
                    // Justera vertikalt om det hamnar utanför nedre kanten
                    if (top + cardHeight > window.innerHeight) {
                        top = rect.top - cardHeight - 10;
                    }
                    
                    // Säkerställ att visitkortet inte hamnar utanför vänsterkanten
                    if (left < 10) {
                        left = 10;
                    }
                    
                    // Säkerställ att visitkortet inte hamnar utanför övre kanten
                    if (top < 10) {
                        top = rect.bottom + 10;
                    }
                    
                    // Skapa visitkort
                    const card = document.createElement('div');
                    card.className = 'user-card-hover';
                    card.style.cssText = `
                        position: fixed;
                        top: ${top}px;
                        left: ${left}px;
                        background: white;
                        border-radius: 12px;
                        padding: 20px;
                        max-width: 350px;
                        width: auto;
                        box-shadow: 0 4px 20px rgba(0,0,0,0.3);
                        text-align: center;
                        z-index: 10000;
                        pointer-events: auto;
                    `;
                    
                    // Profilbild URL
                    let profileImageUrl = null;
                    if (userData.profile_image) {
                        if (userData.type === 'user' || userData.type === 'admin') {
                            profileImageUrl = `../../adm/api/users.php?action=get_image&user_id=${encodeURIComponent(userId)}&t=${Date.now()}`;
                        } else if (userData.type === 'extern') {
                            profileImageUrl = `../../api/chat/external_staff.php?action=get_image&id=${encodeURIComponent(userId)}&t=${Date.now()}`;
                        }
                    }
                    
                    let html = '';
                    
                    // Profilbild
                    html += '<div style="margin-bottom: 16px;">';
                    if (profileImageUrl) {
                        html += `<img src="${escapeHtml(profileImageUrl)}" alt="Profilbild" style="width: 120px; height: 120px; border-radius: 50%; object-fit: cover; border: 2px solid #e0e0e0; display: block; margin: 0 auto;" onerror="this.style.display='none'; const placeholder = this.nextElementSibling; if(placeholder) placeholder.style.display='flex';">`;
                    }
                    html += `<div style="width: 120px; height: 120px; border-radius: 50%; background: #f0f0f0; display: ${profileImageUrl ? 'none' : 'flex'}; align-items: center; justify-content: center; margin: 0 auto; border: 2px solid #e0e0e0;">`;
                    html += '<svg width="60" height="60" viewBox="0 0 24 24" fill="#999"><path d="M12 12c2.21 0 4-1.79 4-4s-1.79-4-4-4-4 1.79-4 4 1.79 4 4 4zm0 2c-2.67 0-8 1.34-8 4v2h16v-2c0-2.66-5.33-4-8-4z"/></svg>';
                    html += '</div>';
                    html += '</div>';
                    
                    // Namn
                    html += `<h3 style="margin: 0 0 10px 0; color: #333; font-size: 20px;">${escapeHtml(userData.name)}</h3>`;
                    
                    // Yrkesroll
                    if (userData.role) {
                        html += '<div style="margin-bottom: 12px;">';
                        html += `<span style="color: #666; font-size: 14px; font-style: italic;">${escapeHtml(userData.role)}</span>`;
                        html += '</div>';
                    }
                    
                    // Telefonnummer
                    if (userData.mobile) {
                        const cleanPhone = userData.mobile.replace(/[\s\-\(\)]/g, '');
                        html += '<div style="margin-bottom: 12px; display: flex; align-items: center; justify-content: center; gap: 6px;">';
                        html += '<svg width="18" height="18" viewBox="0 0 24 24" fill="#666"><path d="M6.62 10.79c1.44 2.83 3.76 5.14 6.59 6.59l2.2-2.2c.27-.27.67-.36 1.02-.24 1.12.37 2.33.57 3.57.57.55 0 1 .45 1 1V20c0 .55-.45 1-1 1-9.39 0-17-7.61-17-17 0-.55.45-1 1-1h3.5c.55 0 1 .45 1 1 0 1.25.2 2.45.57 3.57.11.35.03.74-.25 1.02l-2.2 2.2z"/></svg>';
                        html += `<a href="tel:${escapeHtml(cleanPhone)}" style="color: #1a73e8; font-size: 14px; text-decoration: none; font-weight: 500;">${escapeHtml(userData.mobile)}</a>`;
                        html += '</div>';
                    }
                    
                    // Företag (endast för staff)
                    if (userData.type === 'extern' && userData.company_name) {
                        html += '<div style="margin-bottom: 12px; display: flex; align-items: center; justify-content: center; gap: 6px;">';
                        html += '<svg width="18" height="18" viewBox="0 0 24 24" fill="#666"><path d="M12 7V3H2v18h20V7H12zM6 19H4v-2h2v2zm0-4H4v-2h2v2zm0-4H4V9h2v2zm0-4H4V5h2v2zm4 12H8v-2h2v2zm0-4H8v-2h2v2zm0-4H8V9h2v2zm0-4H8V5h2v2zm10 12h-8v-2h2v-2h-2v-2h2v-2h-2V9h8v10zm-2-8h-2v2h2v-2zm0 4h-2v2h2v-2z"/></svg>';
                        html += `<span style="color: #666; font-size: 14px;">${escapeHtml(userData.company_name)}</span>`;
                        html += '</div>';
                    }
                    
                    // Publika anteckningar
                    if (userData.public_notes) {
                        html += '<div style="margin-top: 12px; padding: 12px; background: #f5f5f5; border-radius: 8px; text-align: left;">';
                        html += `<div style="color: #333; font-size: 13px; line-height: 1.5; white-space: pre-wrap;">${escapeHtml(userData.public_notes)}</div>`;
                        html += '</div>';
                    }
                    
                    card.innerHTML = html;
                    document.body.appendChild(card);
                    currentUserCard = card;
                    
                    // Håll visitkortet synligt när man hovrar över det
                    card.addEventListener('mouseenter', () => {
                        if (userCardTimeout) {
                            clearTimeout(userCardTimeout);
                            userCardTimeout = null;
                        }
                    });
                    
                    card.addEventListener('mouseleave', () => {
                        hideUserCard();
                    });
                    
                } catch (error) {
                    // Tyst misslyckande
                }
            }, 300); // 300ms delay innan visitkortet visas
        }
        
        // Dölj visitkort
        function hideUserCard() {
            if (userCardTimeout) {
                clearTimeout(userCardTimeout);
                userCardTimeout = null;
            }
            
            if (currentUserCard) {
                currentUserCard.remove();
                currentUserCard = null;
            }
        }
        
        // Event listener för company_id dropdown - hantera "> Nytt företag..."
        document.addEventListener('change', async function(e) {
            if (e.target && e.target.id === 'company_id' && e.target.value === '__new_company__') {
                // Återställ dropdownen till tomt val först
                e.target.value = '';
                showNewCompanyModal();
            }
        });
        
        // Funktioner för nytt företag-modal
        function showNewCompanyModal() {
            document.getElementById('newCompanyForm').reset();
            document.getElementById('newCompanyModal').style.display = 'flex';
        }
        
        function closeNewCompanyModal() {
            document.getElementById('newCompanyModal').style.display = 'none';
            // Återställ INTE dropdownen här - värdet ska behållas när företaget skapats
        }
        
        // Hantera skapande av nytt företag
        document.getElementById('newCompanyForm').addEventListener('submit', async (e) => {
            e.preventDefault();
            
            const formData = new FormData(e.target);
            formData.append('action', 'add_company');
            formData.append('user_id', userId);
            
            try {
                const res = await fetch('../../api/chat/external_staff.php', {
                    method: 'POST',
                    body: formData
                });
                
                const data = await res.json();
                if (data.success && data.company) {
                    // Lägg till det nya företaget i companies-arrayen
                    companies.push(data.company);
                    
                    // Uppdatera dropdownen
                    const select = document.getElementById('company_id');
                    if (select) {
                        // Ta bort "> Nytt företag..." alternativet
                        const newCompanyOption = select.querySelector('option[value="__new_company__"]');
                        if (newCompanyOption) {
                            newCompanyOption.remove();
                        }
                        
                        // Lägg till det nya företaget
                        const option = document.createElement('option');
                        option.value = data.company.id;
                        option.textContent = data.company.name;
                        select.appendChild(option);
                        
                        // Välj det nya företaget FÖRE vi lägger tillbaka "> Nytt företag..."
                        select.value = data.company.id;
                        
                        // Lägg tillbaka "> Nytt företag..." som sista alternativ
                        const editId = document.getElementById('editId');
                        if (editId && !editId.value) {
                            const newCompanyOption = document.createElement('option');
                            newCompanyOption.value = '__new_company__';
                            newCompanyOption.textContent = '▶ Nytt företag...';
                            newCompanyOption.style.fontStyle = 'italic';
                            select.appendChild(newCompanyOption);
                        }
                        
                        // Säkerställ att värdet är valt (sätt igen efter att "> Nytt företag..." lagts till)
                        // Använd setTimeout för att säkerställa att DOM är uppdaterad
                        setTimeout(() => {
                            select.value = data.company.id;
                            
                            // Säkerställ att selected-attribute är satt
                            const selectedOption = select.querySelector(`option[value="${data.company.id}"]`);
                            if (selectedOption) {
                                selectedOption.selected = true;
                                const index = Array.from(select.options).indexOf(selectedOption);
                                if (index >= 0) {
                                    select.selectedIndex = index;
                                }
                            }
                            
                        }, 10);
                    }
                    
                    // Stäng modalen
                    closeNewCompanyModal();
                } else {
                    alert('Fel: ' + (data.message || 'Kunde inte skapa företag'));
                }
            } catch (error) {
                alert('Ett fel uppstod vid skapande av företag');
            }
        });
        
        async function showAddModal() {
            // Ladda data först innan modalen öppnas
            if (companies.length === 0) {
                await loadCompanies();
            }
            // Ladda alltid types för att säkerställa att dropdownen är fylld
            await loadTypes();
            
            // Vänta lite extra för att säkerställa att DOM är uppdaterad
            await new Promise(resolve => setTimeout(resolve, 50));
            
            document.getElementById('modalTitle').textContent = 'Lägg till extern användare';
            document.getElementById('editId').value = '';
            
            // Spara dropdownens options innan reset
            const typeSelect = document.getElementById('type_id');
            const savedOptions = [];
            if (typeSelect) {
                for (let i = 1; i < typeSelect.options.length; i++) {
                    savedOptions.push({
                        value: typeSelect.options[i].value,
                        text: typeSelect.options[i].textContent
                    });
                }
            }
            
            document.getElementById('externalStaffForm').reset();
            
            // Återställ dropdownens options efter reset
            if (typeSelect && savedOptions.length > 0) {
                while (typeSelect.options.length > 1) {
                    typeSelect.remove(1);
                }
                savedOptions.forEach(opt => {
                    const option = document.createElement('option');
                    option.value = opt.value;
                    option.textContent = opt.text;
                    typeSelect.appendChild(option);
                });
            } else if (typeSelect && typeSelect.options.length <= 1) {
                // Om dropdownen är tom, ladda om types
                console.warn('Type dropdown is empty after reset, reloading types...');
                await loadTypes();
            }
            
            // Sätt aktiv-checkboxen som checked som standard
            document.getElementById('active').checked = true;
            // can_chat ska vara checked som standard, can_upload ska vara unchecked
            document.getElementById('can_chat').checked = true;
            document.getElementById('can_upload').checked = false;
            
            // Dölj delete-knappen när man lägger till ny
            const deleteButton = document.getElementById('deleteButton');
            deleteButton.style.display = 'none';
            deleteButton.removeAttribute('data-staff-id');
            
            document.getElementById('editModal').style.display = 'flex';
        }
        
        function closeModal() {
            document.getElementById('editModal').style.display = 'none';
        }
        
        async function editStaff(id) {
            const staff = externalStaff.find(s => s.id === id);
            if (!staff) return;
            
            if (companies.length === 0) {
                await loadCompanies();
            }
            // Ladda alltid types för att säkerställa att dropdownen är fylld
            await loadTypes();
            
            document.getElementById('modalTitle').textContent = 'Redigera extern användare';
            document.getElementById('editId').value = staff.id;
            document.getElementById('name').value = staff.name;
            document.getElementById('nickname').value = staff.nickname || '';
            document.getElementById('mobile').value = staff.mobile;
            document.getElementById('email').value = staff.email || '';
            // Ta bort "> Nytt företag..." alternativet om det finns (vid redigering)
            const companySelect = document.getElementById('company_id');
            if (companySelect) {
                const newCompanyOption = companySelect.querySelector('option[value="__new_company__"]');
                if (newCompanyOption) {
                    newCompanyOption.remove();
                }
            }
            document.getElementById('company_id').value = staff.company_id || '';
            document.getElementById('type_id').value = staff.type_id || '';
            document.getElementById('pushover_user_key').value = staff.pushover_user_key || '';
            document.getElementById('notes').value = staff.notes || '';
            document.getElementById('public_notes').value = staff.public_notes || '';
            // Sätt aktiv-status (default true om inte satt)
            const isActive = staff.active !== false; // true om active är true eller undefined
            document.getElementById('active').checked = isActive;
            
            // Sätt can_chat och can_upload (default true om inte satt)
            const canChat = staff.can_chat !== false; // true om can_chat är true eller undefined
            const canUpload = staff.can_upload !== false; // true om can_upload är true eller undefined
            document.getElementById('can_chat').checked = canChat;
            document.getElementById('can_upload').checked = canUpload;
            
            // Visa delete-knappen och spara ID för borttagning
            const deleteButton = document.getElementById('deleteButton');
            deleteButton.style.display = 'block';
            deleteButton.setAttribute('data-staff-id', staff.id);
            
            document.getElementById('editModal').style.display = 'flex';
        }
        
        async function deleteStaffFromModal() {
            const deleteButton = document.getElementById('deleteButton');
            const staffId = deleteButton.getAttribute('data-staff-id');
            
            if (!staffId) {
                return;
            }
            
            if (!confirm('Är du säker på att du vill ta bort denna externa partner? Detta kommer också ta bort personen från alla projekt där de är deltagare.')) {
                return;
            }
            
            try {
                const formData = new FormData();
                formData.append('action', 'delete');
                formData.append('id', staffId);
                formData.append('user_id', userId);
                
                const res = await fetch('../../api/chat/external_staff.php', {
                    method: 'POST',
                    body: formData
                });
                
                const data = await res.json();
                if (data.success) {
                    closeModal();
                    await loadExternalStaff();
                    alert(data.message || 'Extern användare borttagen');
                } else {
                    alert('Fel: ' + data.message);
                }
            } catch (e) {
                alert('Ett fel uppstod');
            }
        }
        
        // Funktion för att visa bilduppladdningspopup
        function uploadStaffImage(staffId) {
            // Hitta extern användare för att få nuvarande bild
            const staff = externalStaff.find(s => (s.user_id || s.id) === staffId);
            const hasProfileImage = staff && staff.profile_image;
            const currentImageUrl = hasProfileImage ? `../../data/${staff.profile_image}?t=${Date.now()}` : null;
            
            const popup = document.createElement('div');
            popup.className = 'image-upload-overlay';
            
            // Skapa preview container
            const previewContainer = document.createElement('div');
            previewContainer.id = 'imagePreviewContainer';
            
            if (currentImageUrl) {
                const img = document.createElement('img');
                img.src = currentImageUrl;
                img.alt = 'Nuvarande bild';
                img.className = 'image-upload-preview';
                img.id = 'imagePreview';
                img.onerror = function() {
                    const placeholder = document.createElement('div');
                    placeholder.className = 'image-upload-preview-placeholder';
                    placeholder.id = 'imagePreviewPlaceholder';
                    placeholder.innerHTML = '<svg width="64" height="64" viewBox="0 0 24 24" fill="currentColor"><path d="M12 12c2.21 0 4-1.79 4-4s-1.79-4-4-4-4 1.79-4 4 1.79 4 4 4zm0 2c-2.67 0-8 1.34-8 4v2h16v-2c0-2.66-5.33-4-8-4z"/></svg>';
                    previewContainer.innerHTML = '';
                    previewContainer.appendChild(placeholder);
                };
                previewContainer.appendChild(img);
            } else {
                const placeholder = document.createElement('div');
                placeholder.className = 'image-upload-preview-placeholder';
                placeholder.id = 'imagePreviewPlaceholder';
                placeholder.innerHTML = '<svg width="64" height="64" viewBox="0 0 24 24" fill="currentColor"><path d="M12 12c2.21 0 4-1.79 4-4s-1.79-4-4-4-4 1.79-4 4 1.79 4 4 4zm0 2c-2.67 0-8 1.34-8 4v2h16v-2c0-2.66-5.33-4-8-4z"/></svg>';
                previewContainer.appendChild(placeholder);
            }
            
            popup.innerHTML = `
                <div class="image-upload-box">
                    <h3 class="confirm-title">Ladda upp profilbild</h3>
                </div>
            `;
            
            const box = popup.querySelector('.image-upload-box');
            box.appendChild(previewContainer);
            
            const fileInput = document.createElement('input');
            fileInput.type = 'file';
            fileInput.id = 'imageFileInput';
            fileInput.accept = 'image/*';
            fileInput.style.display = 'none';
            fileInput.onchange = function(event) {
                handleImageFileSelect(event, staffId);
            };
            
            const buttonContainer = document.createElement('div');
            buttonContainer.style.cssText = 'display: flex; gap: 12px; justify-content: flex-start; align-items: center; margin-top: 20px;';
            
            const selectBtn = document.createElement('button');
            selectBtn.type = 'button';
            selectBtn.className = 'confirm-btn confirm-btn-blue';
            selectBtn.textContent = 'Välj bild';
            selectBtn.onclick = function() {
                fileInput.click();
            };
            
            const cancelBtn = document.createElement('button');
            cancelBtn.type = 'button';
            cancelBtn.className = 'confirm-btn confirm-btn-cancel';
            cancelBtn.textContent = 'Avbryt';
            cancelBtn.onclick = function() {
                popup.remove();
            };
            
            buttonContainer.appendChild(selectBtn);
            
            // Visa "Ta bort bild"-knappen om användaren har en profilbild
            if (currentImageUrl) {
                const deleteBtn = document.createElement('button');
                deleteBtn.type = 'button';
                deleteBtn.className = 'confirm-btn confirm-btn-cancel';
                deleteBtn.textContent = 'Ta bort bild';
                deleteBtn.onclick = function() {
                    deleteStaffImage(staffId);
                };
                buttonContainer.appendChild(deleteBtn);
            }
            
            buttonContainer.appendChild(cancelBtn);
            box.appendChild(fileInput);
            box.appendChild(buttonContainer);
            
            document.body.appendChild(popup);
            
            // Stäng popup när man klickar utanför
            popup.addEventListener('click', (e) => {
                if (e.target === popup) {
                    popup.remove();
                }
            });
        }
        
        // Hantera filval för bild
        function handleImageFileSelect(event, staffId) {
            const file = event.target.files[0];
            if (!file) return;
            
            // Visa förhandsvisning
            const reader = new FileReader();
            reader.onload = function(e) {
                const previewContainer = document.getElementById('imagePreviewContainer');
                previewContainer.innerHTML = `<img src="${e.target.result}" alt="Förhandsvisning" class="image-upload-preview" id="imagePreview">`;
                
                // Lägg till knapp för att spara
                const uploadBtn = document.createElement('button');
                uploadBtn.type = 'button';
                uploadBtn.className = 'confirm-btn confirm-btn-primary';
                uploadBtn.textContent = 'Spara';
                uploadBtn.onclick = () => uploadImageFile(staffId, file);
                
                const buttonContainer = document.querySelector('.image-upload-box > div:last-child');
                if (!buttonContainer.querySelector('.confirm-btn-primary[onclick*="uploadImageFile"]')) {
                    const selectBtn = buttonContainer.querySelector('.confirm-btn-blue');
                    if (selectBtn && selectBtn.nextSibling) {
                        buttonContainer.insertBefore(uploadBtn, selectBtn.nextSibling);
                    } else {
                        buttonContainer.appendChild(uploadBtn);
                    }
                }
            };
            reader.readAsDataURL(file);
        }
        
        // Ladda upp bildfil
        async function uploadImageFile(staffId, file) {
            const formData = new FormData();
            formData.append('action', 'upload_image');
            formData.append('id', staffId);
            formData.append('user_id', userId);
            formData.append('image', file);
            
            // Hitta popup och knappar
            const popup = document.querySelector('.image-upload-overlay');
            const buttonContainer = popup ? popup.querySelector('.image-upload-box > div:last-child') : null;
            const allButtons = buttonContainer ? buttonContainer.querySelectorAll('button') : [];
            const saveBtn = Array.from(allButtons).find(btn => btn.textContent === 'Spara');
            
            // Disable alla knappar och visa spinner
            allButtons.forEach(btn => {
                btn.disabled = true;
                btn.style.opacity = '0.6';
                btn.style.cursor = 'not-allowed';
            });
            
            // Lägg till spinner om den inte redan finns
            let spinner = popup ? popup.querySelector('.upload-spinner') : null;
            if (!spinner && popup) {
                spinner = document.createElement('div');
                spinner.className = 'upload-spinner';
                spinner.style.cssText = 'text-align: center; margin-top: 15px; color: #1a73e8;';
                spinner.innerHTML = '<div style="display: inline-block; width: 20px; height: 20px; border: 3px solid #f3f3f3; border-top: 3px solid #1a73e8; border-radius: 50%; animation: spin 1s linear infinite;"></div><span style="margin-left: 10px;">Bearbetar bild...</span>';
                const box = popup.querySelector('.image-upload-box');
                if (box) {
                    box.appendChild(spinner);
                }
            }
            
            try {
                const response = await fetch('../../api/chat/external_staff.php', {
                    method: 'POST',
                    body: formData
                });
                
                // Läs svaret som text först för att hantera både JSON och fel
                const responseText = await response.text();
                let data;
                
                try {
                    data = JSON.parse(responseText);
                } catch (parseError) {
                    // Om svaret inte är JSON, visa felmeddelandet
                    if (typeof showStatus === 'function') {
                        showStatus('Fel vid uppladdning: ' + responseText.slice(0, 200), 'error');
                    } else {
                        alert('Fel: ' + (responseText || 'Okänt fel'));
                    }
                    if (spinner) spinner.remove();
                    allButtons.forEach(btn => {
                        btn.disabled = false;
                        btn.style.opacity = '1';
                        btn.style.cursor = 'pointer';
                    });
                    return;
                }
                
                if (data.success) {
                    // Uppdatera extern användares profilbild i listan
                    const staff = externalStaff.find(s => (s.user_id || s.id) === staffId);
                    if (staff) {
                        staff.profile_image = data.profile_image;
                    }
                    // Stäng popup
                    if (popup && popup.parentNode) {
                        popup.remove();
                    }
                    // Ladda om listan från servern för att säkerställa att allt är synkroniserat
                    setTimeout(() => {
                        loadExternalStaff();
                    }, 500);
                    
                    // Ta bort spinner och återställ knappar (om popupen fortfarande finns)
                    if (spinner && popup && popup.parentNode) {
                        spinner.remove();
                    }
                    if (popup && popup.parentNode) {
                        allButtons.forEach(btn => {
                            btn.disabled = false;
                            btn.style.opacity = '1';
                            btn.style.cursor = 'pointer';
                        });
                    }
                } else {
                    if (typeof showStatus === 'function') {
                        showStatus('Kunde inte ladda upp bild: ' + data.message, 'error');
                    } else {
                        alert('Fel: ' + (data.message || 'Okänt fel'));
                    }
                    // Ta bort spinner och återställ knappar vid fel
                    if (spinner) {
                        spinner.remove();
                    }
                    allButtons.forEach(btn => {
                        btn.disabled = false;
                        btn.style.opacity = '1';
                        btn.style.cursor = 'pointer';
                    });
                }
            } catch (error) {
                if (typeof showStatus === 'function') {
                    showStatus('Fel vid uppladdning: ' + error.message, 'error');
                } else {
                    alert('Ett fel uppstod vid uppladdning: ' + (error.message || error));
                }
                // Ta bort spinner och återställ knappar vid fel
                if (spinner) {
                    spinner.remove();
                }
                allButtons.forEach(btn => {
                    btn.disabled = false;
                    btn.style.opacity = '1';
                    btn.style.cursor = 'pointer';
                });
            }
        }
        
        // Ta bort bild
        async function deleteStaffImage(staffId) {
            if (!confirm('Är du säker på att du vill ta bort profilbilden?')) {
                return;
            }
            
            const formData = new FormData();
            formData.append('action', 'delete_image');
            formData.append('id', staffId);
            formData.append('user_id', userId);
            
            try {
                const response = await fetch('../../api/chat/external_staff.php', {
                    method: 'POST',
                    body: formData
                });
                
                // Läs svaret som text först för att hantera både JSON och fel
                const responseText = await response.text();
                let data;
                
                try {
                    data = JSON.parse(responseText);
                } catch (parseError) {
                    // Om svaret inte är JSON, visa felmeddelandet
                    if (typeof showStatus === 'function') {
                        showStatus('Fel: ' + (responseText || 'Okänt fel'), 'error');
                    } else {
                        alert('Fel: ' + (responseText || 'Okänt fel'));
                    }
                    return;
                }
                
                if (data.success) {
                    // Uppdatera extern användares profilbild i listan
                    const staff = externalStaff.find(s => (s.user_id || s.id) === staffId);
                    if (staff) {
                        delete staff.profile_image;
                    }
                    // Ladda om listan från servern
                    await loadExternalStaff();
                    // Stäng popup
                    document.querySelector('.image-upload-overlay')?.remove();
                } else {
                    if (typeof showStatus === 'function') {
                        showStatus('Kunde inte ta bort bild: ' + data.message, 'error');
                    } else {
                        alert('Fel: ' + (data.message || 'Okänt fel'));
                    }
                }
            } catch (error) {
                if (typeof showStatus === 'function') {
                    showStatus('Fel vid borttagning: ' + error.message, 'error');
                } else {
                    alert('Ett fel uppstod: ' + (error.message || error));
                }
            }
        }
        
        document.getElementById('externalStaffForm').addEventListener('submit', async (e) => {
            e.preventDefault();
            
            const formData = new FormData(e.target);
            const editId = formData.get('id');
            formData.append('action', editId ? 'update' : 'add');
            formData.append('user_id', userId);
            
            try {
                const res = await fetch('../../api/chat/external_staff.php', {
                    method: 'POST',
                    body: formData
                });
                
                const data = await res.json();
                if (data.success) {
                    closeModal();
                    // Vänta lite innan vi laddar om listan för att säkerställa att alla ändringar har sparats
                    await new Promise(resolve => setTimeout(resolve, 200));
                    await loadExternalStaff();
                } else {
                    alert('Fel: ' + data.message);
                }
            } catch (error) {
                alert('Ett fel uppstod');
            }
        });
        
        // Stäng modal när man klickar utanför
        document.getElementById('editModal').addEventListener('click', (e) => {
            if (e.target.id === 'editModal') {
                closeModal();
            }
        });
        
        // Ladda data när sidan laddas
        async function initializePage() {
            try {
                await Promise.all([
                    loadCompanies(),
                    loadTypes(),
                    loadExternalStaff()
                ]);
            } catch (e) {
            }
        }
        
        // Kör när DOM är redo
        if (document.readyState === 'loading') {
            document.addEventListener('DOMContentLoaded', initializePage);
        } else {
            // DOM är redan laddad
            initializePage();
        }
        
        async function openStaffLink(userId) {
            console.log('[openStaffLink] Anropad med user_id:', userId);
            
            if (!userId) {
                console.warn('[openStaffLink] Användar-ID saknas');
                showErrorPopup('Användar-ID saknas. Kontakta administratören om problemet kvarstår.');
                return;
            }
            
            // Hämta information om användarens access
            try {
                const url = `../../api/chat/external_staff.php?action=get_user_access&user_id=${encodeURIComponent(userId)}`;
                console.log('[openStaffLink] Hämtar användaraccess från:', url);
                
                const response = await fetch(url);
                
                if (!response.ok) {
                    console.error('[openStaffLink] HTTP-fel:', response.status, response.statusText);
                    showErrorPopup('Kunde inte kontrollera användarrättigheter. Serverfel: ' + response.status + '. Försök igen senare.');
                    return;
                }
                
                const data = await response.json();
                console.log('[openStaffLink] API-svar:', data);
                
                if (!data.success) {
                    console.error('[openStaffLink] API returnerade success=false:', data.message);
                    showErrorPopup('Kunde inte hämta användarinformation: ' + (data.message || 'Okänt fel'));
                    return;
                }
                
                console.log('[openStaffLink] Användardata:', {
                    can_upload: data.can_upload,
                    can_chat: data.can_chat,
                    projects_count: data.projects?.length || 0,
                    chat_projects_count: data.chat_projects?.length || 0,
                    link_type: data.link_type,
                    target_project: data.target_project
                });
                
                // Kontrollera om användaren uppfyller kraven
                if (!data.projects || data.projects.length === 0) {
                    console.warn('[openStaffLink] Användaren är inte deltagare i något projekt');
                    showErrorPopup('Användaren måste vara deltagare i minst ett projekt innan en länk kan skapas. Lägg till användaren som deltagare i ett projekt först.');
                    return;
                }
                
                const canUpload = data.can_upload || false;
                const canChat = data.can_chat || false;
                
                console.log('[openStaffLink] Rättigheter:', { canUpload, canChat });
                
                // Om användaren uppfyller kraven, generera och visa länk
                if (data.link_type === 'landing') {
                    // Landing-länk kan skapas om användaren är deltagare i projekt och har can_upload eller can_chat
                    console.log('[openStaffLink] Skapar landing-länk');
                    showStaffLinkPopup(userId, 'landing', null, data);
                } else if (data.link_type === 'upload' && data.target_project) {
                    // Upload-länk kan skapas om användaren är deltagare i projekt och har can_upload
                    console.log('[openStaffLink] Skapar upload-länk för projekt:', data.target_project);
                    showStaffLinkPopup(userId, 'upload', data.target_project);
                } else if (data.link_type === 'chat' && data.target_project) {
                    // Chat-länk kräver både projekt- och chatt-deltagande
                    console.log('[openStaffLink] Skapar chat-länk för projekt:', data.target_project);
                    showStaffLinkPopup(userId, 'chat', data.target_project);
                } else {
                    // Inga länkar kunde skapas - ge specifikt felmeddelande
                    console.warn('[openStaffLink] Ingen länk kunde skapas', {
                        canUpload,
                        canChat,
                        projects_count: data.projects?.length || 0,
                        chat_projects_count: data.chat_projects?.length || 0,
                        link_type: data.link_type
                    });
                    
                    let reason = '';
                    
                    if (!canUpload && !canChat) {
                        reason = 'Användaren saknar rättigheter. Användaren måste ha rättigheter att antingen ladda upp bilder (Kommer åt byggarkiv) eller delta i projektchatt. Aktivera minst en av dessa rättigheter i användarens inställningar.';
                    } else if (canUpload && !canChat) {
                        // Användaren har upload-rättigheter men ingen länk skapades
                        // Detta borde inte hända om användaren är deltagare i projekt
                        reason = 'Användaren har rätt att ladda upp bilder men ingen länk kunde skapas. Kontrollera att användaren är deltagare i minst ett projekt.';
                    } else if (!canUpload && canChat) {
                        // Användaren har chat-rättigheter men saknar chat-access
                        if (!data.chat_projects || data.chat_projects.length === 0) {
                            reason = 'Användaren har rätt att delta i chatt men är inte deltagare i någon projektchatt. Lägg till användaren i projektchatten för ett projekt där hen är deltagare.';
                        } else {
                            reason = 'Användaren har rätt att delta i chatt men ingen länk kunde skapas. Kontakta administratören om problemet kvarstår.';
                        }
                    } else {
                        // Både can_upload och can_chat är true men ingen länk skapades
                        reason = 'Användaren har rättigheter men ingen länk kunde skapas. Kontrollera att användaren är deltagare i minst ett projekt.';
                    }
                    
                    showErrorPopup(reason);
                }
            } catch (error) {
                console.error('[openStaffLink] Fel vid hämtning av användaraccess:', error);
                showErrorPopup('Ett fel uppstod vid kontroll av användarrättigheter. Kontrollera webbläsarens konsol för mer information eller försök igen senare.');
            }
        }
        
        function showErrorPopup(message) {
            const popup = document.createElement('div');
            popup.className = 'popup-overlay';
            popup.style.cssText = `
                position: fixed;
                top: 0;
                left: 0;
                width: 100%;
                height: 100%;
                background: rgba(0,0,0,0.5);
                display: flex;
                align-items: center;
                justify-content: center;
                z-index: 10000;
            `;
            
            const popupContent = document.createElement('div');
            popupContent.style.cssText = `
                background: white;
                padding: 30px;
                border-radius: 12px;
                box-shadow: 0 8px 32px rgba(0,0,0,0.3);
                max-width: 600px;
                width: 90%;
                text-align: center;
            `;
            
            const projectsUrl = `../projects/projects.php${userId ? '?user_id=' + encodeURIComponent(userId) : ''}`;
            
            popupContent.innerHTML = `
                <h3 class="popup-title" style="color: #d32f2f; margin-bottom: 20px;">Länk kan inte skapas</h3>
                <div class="popup-description" style="text-align: left; line-height: 1.8; color: #333; background: #f8f9fa; padding: 20px; border-radius: 8px;">
                    <p style="margin: 0;">${escapeHtml(message)}</p>
                </div>
                <div style="margin-top: 20px; text-align: center;">
                    <a href="${projectsUrl}" style="color: #1a73e8; text-decoration: none; font-weight: 500; font-size: 14px;">→ Hantera projekt</a>
                </div>
                <div class="popup-buttons" style="margin-top: 24px;">
                    <button onclick="this.closest('.popup-overlay').remove()" class="popup-btn popup-btn-primary">
                        Förstått
                    </button>
                </div>
            `;
            
            popup.appendChild(popupContent);
            document.body.appendChild(popup);
            
            // Stäng popup när man klickar utanför
            popup.addEventListener('click', (e) => {
                if (e.target === popup) {
                    popup.remove();
                }
            });
        }
        
        function showStaffLinkPopup(userId, linkType, targetProject, accessData = null) {
            const baseUrl = window.location.origin + window.location.pathname.replace('/adm/extusers/external_parties.php', '');
            let staffLink = '';
            let linkDescription = '';
            
            if (linkType === 'landing') {
                staffLink = baseUrl + '/external_landing.php?user_id=' + encodeURIComponent(userId);
                const projectsCount = accessData?.projects?.length || 0;
                const canUpload = accessData?.can_upload || false;
                const canChat = accessData?.can_chat || false;
                
                let features = [];
                if (canUpload) features.push('ladda upp bilder');
                if (canChat) features.push('delta i projektchattar');
                
                linkDescription = `Denna länk ger användaren tillgång till en landningssida med ${projectsCount} projekt. `;
                linkDescription += `Användaren kan ${features.join(' och ')} för de projekt hen är med i.`;
            } else if (linkType === 'upload') {
                staffLink = baseUrl + '/index.html?user_id=' + encodeURIComponent(userId);
                linkDescription = 'Denna länk ger användaren tillgång till bilduppladdning för projekt: ' + escapeHtml(targetProject.name);
            } else if (linkType === 'chat') {
                staffLink = baseUrl + '/chat/chat.php?project_id=' + encodeURIComponent(targetProject.id) + '&user_id=' + encodeURIComponent(userId);
                linkDescription = 'Denna länk ger användaren tillgång till projektchatten för: ' + escapeHtml(targetProject.name);
            }
            
            // Skapa popup
            const popup = document.createElement('div');
            popup.style.cssText = `
                position: fixed;
                top: 0;
                left: 0;
                width: 100%;
                height: 100%;
                background: rgba(0,0,0,0.5);
                display: flex;
                align-items: center;
                justify-content: center;
                z-index: 10000;
            `;
            
            const popupContent = document.createElement('div');
            popupContent.style.cssText = `
                background: white;
                padding: 30px;
                border-radius: 12px;
                box-shadow: 0 8px 32px rgba(0,0,0,0.3);
                max-width: 500px;
                width: 90%;
                text-align: center;
            `;
            
            popupContent.innerHTML = `
                <h3 class="popup-title">Länk för extern användare</h3>
                <p class="popup-description" style="text-align: left; line-height: 1.6; margin-bottom: 16px;">${linkDescription}</p>
                <div class="popup-code">
                    ${escapeHtml(staffLink)}
                </div>
                <div class="popup-buttons">
                    <button onclick="copyToClipboard('${staffLink.replace(/'/g, "\\'")}'); this.textContent='Kopierad!'; setTimeout(() => this.textContent='Kopiera länk', 2000)" class="popup-btn popup-btn-primary">
                        Kopiera länk
                    </button>
                    <button onclick="this.closest('.popup-overlay').remove()" class="popup-btn popup-btn-secondary">
                        Stäng
                    </button>
                </div>
            `;
            
            popup.className = 'popup-overlay';
            popup.appendChild(popupContent);
            document.body.appendChild(popup);
            
            // Stäng popup när man klickar utanför
            popup.addEventListener('click', (e) => {
                if (e.target === popup) {
                    popup.remove();
                }
            });
        }
        
        function copyToClipboard(text) {
            navigator.clipboard.writeText(text).then(() => {
                // Visa statusmeddelande om showStatus finns
                if (typeof showStatus === 'function') {
                    showStatus('Länk kopierad till urklipp!', 'success');
                } else {
                    alert('Länk kopierad till urklipp!');
                }
            }).catch(() => {
                // Fallback för äldre webbläsare
                const textArea = document.createElement('textarea');
                textArea.value = text;
                document.body.appendChild(textArea);
                textArea.select();
                document.execCommand('copy');
                document.body.removeChild(textArea);
                if (typeof showStatus === 'function') {
                    showStatus('Länk kopierad till urklipp!', 'success');
                } else {
                    alert('Länk kopierad till urklipp!');
                }
            });
        }
    </script>
</body>
</html>

