<?php
require_once __DIR__ . '/../admin_guard.php';
require_once __DIR__ . '/../../lib/data_functions.php';

// Hämta all data
$data = getAllUploadData();
$allUploads = $data['allUploads'];
$userIds = $data['userIds'];
$projectNames = $data['projectNames'];

// Hämta alla projekt (både aktiva och arkiverade) direkt från projects.json
$projectsData = [];
if (file_exists(__DIR__ . '/../../data/projects.json')) {
    $allProjects = json_decode(file_get_contents(__DIR__ . '/../../data/projects.json'), true) ?: [];
    $projectsData = $allProjects; // Inkludera alla projekt, inte bara aktiva
}

// Hämta användardata för att visa namn
$userNames = [];
$userData = [];
if (file_exists(__DIR__ . '/../../data/userids.json')) {
    $userContent = file_get_contents(__DIR__ . '/../../data/userids.json');
    if ($userContent !== false) {
        $userData = json_decode($userContent, true) ?: [];
        foreach ($userData as $user) {
            $userNames[$user['user_id']] = $user['full_name'];
        }
    }
}

// Funktion getUserFullName() finns redan i ../lib/data_functions.php



// Hämta parametrar från URL
$selectedProject = $_GET['project_id'] ?? '';
$selectedUserId = $_GET['filter_user_id'] ?? '';

// Bestäm vilken vy som ska visas
$activeView = '';
if ($selectedProject && $selectedUserId) {
    $activeView = 'combined';
} elseif ($selectedProject) {
    $activeView = 'project';
} elseif ($selectedUserId) {
    $activeView = 'user';
}
?>
<!DOCTYPE html>
<html lang="sv">
<head>
    <meta charset="utf-8" />
    <meta name="viewport" content="width=device-width,initial-scale=1" />
    <title>Lista uppladdningar - ByggArkiv</title>
    <link rel="stylesheet" href="../common.css">
    <style>
        body { 
            font-family: system-ui, -apple-system, Segoe UI, Roboto, sans-serif; 
            margin: 16px; 
            background-color: #f5f5f5;
            line-height: 1.6;
        }
        .header {
            text-align: center;
            margin-bottom: 30px;
        }
        .header img {
            max-width: 100%;
            height: auto;
            max-height: 120px;
            display: block;
            margin: 0 auto;
        }
        .container {
            max-width: 1000px;
            margin: 0 auto;
        }
        .filters {
            background: white;
            border-radius: 12px;
            padding: 20px;
            margin-bottom: 24px;
            box-shadow: 0 2px 8px rgba(0,0,0,0.1);
        }
        .filter-row {
            display: flex;
            gap: 20px;
            margin-bottom: 16px;
        }
        .filter-group {
            flex: 1;
            min-width: 0;
        }
        @media (max-width: 768px) {
            .filter-row {
                flex-direction: column;
                gap: 16px;
            }
        }
        
        @media (min-width: 1200px) {
            .filter-row {
                justify-content: center;
            }
            .filter-group {
                flex: 0 1 400px;
                max-width: 400px;
            }
        }
        .filter-group {
            display: flex;
            flex-direction: column;
            gap: 4px;
            flex-direction: column;
        }
        .filter-group label {
            font-weight: 600;
            margin-top: 8px;
            padding: 8px;
            background: #e3f2fd;
            border-radius: 4px;
            font-style: italic;
            margin-bottom: 8px;
            color: #1976d2;
        }
        .filter-group select {
            padding: 10px;
            border: 1px solid #ccc;
            border-radius: 8px;
            font-size: 16px;
            background: white;
            width: 100% !important;
            min-width: 0 !important;
            max-width: 100% !important;
            box-sizing: border-box !important;
            flex-shrink: 1;
        }
        .filter-group select:focus {
            outline: none;
            border-color: #1a73e8;
            box-shadow: 0 0 0 2px rgba(26, 115, 232, 0.2);
        }
        .archive-toggle {
            display: flex;
            align-items: center;
            gap: 8px;
            background: #f5f5f5;
            padding: 4px 10px;
            border-radius: 6px;
            margin-top: 8px;
            height: 38px;
            box-sizing: border-box;
        }
        .archive-toggle input[type="checkbox"] {
            cursor: pointer;
        }
        .archive-toggle label {
            cursor: pointer;
            font-size: 13px;
            color: #666;
            line-height: 1.2;
            background: transparent;
            padding: 0;
            margin: 0;
            font-weight: normal;
            font-style: normal;
        }
        
        /* Ensure equal width for all select elements */
        #projectFilter, #userIdFilter {
            width: 100% !important;
            min-width: 0 !important;
            max-width: 100% !important;
            box-sizing: border-box !important;
        }
        .results-section {
            background: white;
            border-radius: 12px;
            box-shadow: 0 2px 8px rgba(0,0,0,0.1);
            overflow: hidden;
        }
        .results-header {
            background: #1a73e8;
            color: white;
            padding: 16px 20px;
            font-size: 18px;
            font-weight: 600;
        }
        /* Add vertical spacing under header similar to listallp.php */
        .results-header + .results-content,
        .results-header + div {
            padding-top: 20px;
        }
        .file-image {
            width: 100%;
            height: 200px;
            object-fit: cover;
            background: #f0f0f0;
            transition: opacity 0.2s;
            pointer-events: auto;
            z-index: 1;
            cursor: default;
        }
        .file-info {
            padding: 16px;
        }
        .file-project {
            font-weight: 600;
            color: #1a73e8;
            margin-top: 8px;
            padding: 0;
            background: none;
            border-radius: 0;
            font-style: italic;
            margin-bottom: 8px;
        }
        .file-project.no-background {
            background: none;
            padding: 0;
            margin-bottom: 4px;
        }
        .file-project-name {
            font-size: 12px;
            color: #666;
            font-weight: normal;
            margin-top: 2px;
        }
        .file-note {
            margin-top: 6px;
            padding: 6px 8px 2px 8px;
            background: #e3f2fd;
            border-radius: 4px;
            font-style: normal;
            font-weight: normal;
            color: #000;
            margin-bottom: 2px;
            word-wrap: break-word;
        }
        
        .edit-btn {
            position: absolute;
            top: 8px;
            right: 119px;
            background: white;
            color: #ff9800;
            border: 2px solid #ff9800;
            border-radius: 50%;
            width: 30px;
            height: 30px;
            font-size: 16px;
            cursor: pointer;
            display: flex;
            align-items: center;
            justify-content: center;
            opacity: 0.9;
            transition: opacity 0.2s;
            z-index: 10;
            pointer-events: auto;
        }
        .edit-btn:hover {
            opacity: 1;
            background: #ff9800;
            color: white;
        }
        .file-meta {
            font-size: 12px;
            color: #666;
            display: flex;
            flex-direction: column;
            gap: 4px;
            justify-content: flex-start;
            margin-bottom: -4px;
        }
        .file-meta-row {
            display: flex;
            justify-content: space-between;
            align-items: center;
            gap: 8px;
        }
        .file-date {
            flex: 1;
        }
        .file-user-id {
            color: #666;
            font-weight: normal;
        }

        .file-filename {
            font-size: 12px;
            color: #1a73e8;
            text-decoration: underline;
        }
        .file-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(300px, 1fr));
            gap: 20px;
        }
        
        /* Button styles for list1p.php */
        .view-btn {
            position: absolute;
            top: 8px;
            right: 82px;
            background: white;
            color: #1a73e8;
            border: 2px solid #1a73e8;
            border-radius: 50%;
            width: 30px;
            height: 30px;
            font-size: 16px;
            cursor: pointer;
            display: flex;
            align-items: center;
            justify-content: center;
            opacity: 0.9;
            transition: opacity 0.2s;
            z-index: 10;
            pointer-events: auto;
        }
        .view-btn:hover {
            opacity: 1;
            background: #1a73e8;
            color: white;
        }

        .move-btn {
            position: absolute;
            top: 8px;
            right: 45px;
            background: white;
            color: #28a745;
            border: 2px solid #28a745;
            border-radius: 50%;
            width: 30px;
            height: 30px;
            font-size: 16px;
            cursor: pointer;
            display: flex;
            align-items: center;
            justify-content: center;
            opacity: 0.9;
            transition: opacity 0.2s;
            z-index: 10;
            pointer-events: auto;
        }
        .move-btn:hover {
            opacity: 1;
            background: #28a745;
            color: white;
        }

        .delete-btn {
            position: absolute;
            top: 8px;
            right: 8px;
            background: white;
            color: #dc3545;
            border: 2px solid #dc3545;
            border-radius: 50%;
            width: 30px;
            height: 30px;
            font-size: 16px;
            cursor: pointer;
            display: flex;
            align-items: center;
            justify-content: center;
            opacity: 0.9;
            transition: opacity 0.2s;
            z-index: 10;
            pointer-events: auto;
        }
        .delete-btn:hover {
            opacity: 1;
            background: #dc3545;
            color: white;
        }

        .edit-btn {
            position: absolute;
            top: 8px;
            right: 119px;
            background: white;
            color: #ff9800;
            border: 2px solid #ff9800;
            border-radius: 50%;
            width: 30px;
            height: 30px;
            font-size: 16px;
            cursor: pointer;
            display: flex;
            align-items: center;
            justify-content: center;
            opacity: 0.9;
            transition: opacity 0.2s;
            z-index: 10;
            pointer-events: auto;
        }
        .edit-btn:hover {
            opacity: 1;
            background: #ff9800;
            color: white;
        }
        
        .file-item {
            border: 1px solid #e0e0e0;
            border-radius: 8px;
            overflow: hidden;
            background: #fafafa;
            transition: transform 0.2s, box-shadow 0.2s;
            position: relative;
        }
        .file-item:hover {
            transform: translateY(-2px);
            box-shadow: 0 4px 12px rgba(0,0,0,0.15);
        }
        .file-overlay {
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: transparent;
            opacity: 1;
            transition: none;
            z-index: 5;
        }
        /* Buttons are always visible now */
        .file-image {
            width: 100%;
            height: 200px;
            object-fit: cover;
            background: #f0f0f0;
            cursor: pointer;
            transition: opacity 0.2s;
        }
        .file-image:hover {
            opacity: 0.9;
        }
        .file-info {
            padding: 16px;
        }
        .no-results {
            text-align: center;
            padding: 40px;
            font-size: 14px;
            color: #1a73e8;
            word-break: break-all;
            text-decoration: none;
        }
        .no-results:hover {
            text-decoration: underline;
        }
        .no-results {
            color: #666;
            font-style: italic;
        }

        .delete-btn {
            position: absolute;
            top: 8px;
            right: 8px;
            background: white;
            color: #dc3545;
            border: 2px solid #dc3545;
            border-radius: 50%;
            width: 30px;
            height: 30px;
            font-size: 16px;
            cursor: pointer;
            display: flex;
            align-items: center;
            justify-content: center;
            opacity: 0.9;
            transition: opacity 0.2s;
            z-index: 10;
            pointer-events: auto;
        }
        .delete-btn:hover {
            opacity: 1;
            background: #dc3545;
            color: white;
        }
        
        .move-btn {
            position: absolute;
            top: 8px;
            right: 45px;
            background: white;
            color: #28a745;
            border: 2px solid #28a745;
            border-radius: 50%;
            width: 30px;
            height: 30px;
            font-size: 16px;
            cursor: pointer;
            display: flex;
            align-items: center;
            justify-content: center;
            opacity: 0.9;
            transition: opacity 0.2s;
            z-index: 10;
            pointer-events: auto;
        }
        .move-btn:hover {
            opacity: 1;
            background: #28a745;
            color: white;
        }
        
        .view-btn {
            position: absolute;
            top: 8px;
            right: 82px;
            background: white;
            color: #1a73e8;
            border: 2px solid #1a73e8;
            border-radius: 50%;
            width: 30px;
            height: 30px;
            font-size: 16px;
            cursor: pointer;
            display: flex;
            align-items: center;
            justify-content: center;
            opacity: 0.9;
            transition: opacity 0.2s;
            z-index: 10;
            pointer-events: auto;
        }
        .view-btn:hover {
            opacity: 1;
            background: #1a73e8;
            color: white;
        }
        
        /* Lightbox Modal Styles */
        .lightbox {
            display: none;
            position: fixed;
            z-index: 1000;
            left: 0;
            top: 0;
            width: 100%;
            height: 100%;
            background-color: rgba(0, 0, 0, 0.9);
            backdrop-filter: blur(5px);
        }
        
        .lightbox-content {
            position: relative;
            margin: auto;
            padding: 20px;
            width: 90%;
            height: 90%;
            display: flex;
            align-items: center;
            justify-content: center;
        }
        
        .lightbox-image {
            max-width: 100%;
            max-height: 100%;
            object-fit: contain;
            border-radius: 8px;
            box-shadow: 0 4px 20px rgba(0, 0, 0, 0.5);
        }
        
        .lightbox-close {
            position: absolute;
            top: 20px;
            right: 35px;
            color: #f1f1f1;
            font-size: 40px;
            font-weight: bold;
            cursor: pointer;
            z-index: 1001;
            background: rgba(0, 0, 0, 0.5);
            border-radius: 50%;
            width: 50px;
            height: 50px;
            display: flex;
            align-items: center;
            justify-content: center;
            transition: background-color 0.2s;
        }
        
        .lightbox-close:hover {
            background: rgba(0, 0, 0, 0.8);
        }
        
        .lightbox-info {
            position: absolute;
            bottom: 20px;
            left: 20px;
            right: 20px;
            background: rgba(0, 0, 0, 0.8);
            color: white;
            padding: 15px;
            border-radius: 8px;
            font-size: 14px;
        }
        
        .lightbox-filename {
            font-weight: bold;
            margin-bottom: 5px;
        }
        
        .lightbox-meta {
            display: flex;
            justify-content: space-between;
            flex-wrap: wrap;
            gap: 10px;
        }
        
        @media (max-width: 768px) {
            .filter-row {
                grid-template-columns: 1fr;
                gap: 16px;
            }
            .file-grid {
                grid-template-columns: 1fr;
            }
            .lightbox-content {
                width: 95%;
                height: 95%;
                padding: 10px;
            }
            .lightbox-close {
                top: 10px;
                right: 20px;
                font-size: 30px;
                width: 40px;
                height: 40px;
            }
            .lightbox-info {
                bottom: 10px;
                left: 10px;
                right: 10px;
                padding: 10px;
                font-size: 12px;
            }
        .lightbox-meta {
            flex-direction: column;
            gap: 5px;
        }
    }
    
    /* Modal styles for description popup */
    .modal {
        position: fixed;
        z-index: 1000;
        left: 0;
        top: 0;
        width: 100%;
        height: 100%;
        background-color: rgba(0, 0, 0, 0.5);
        display: flex;
        align-items: center;
        justify-content: center;
    }
    
    .modal-content {
        background-color: white;
        border-radius: 12px;
        width: 90%;
        max-width: 500px;
        max-height: 90vh;
        overflow-y: auto;
        box-shadow: 0 4px 20px rgba(0, 0, 0, 0.3);
    }
    
    .modal-header {
        padding: 20px 20px 0 20px;
        display: flex;
        justify-content: space-between;
        align-items: center;
        border-bottom: 1px solid #eee;
        padding-bottom: 15px;
    }
    
    .modal-header h3 {
        margin: 0;
        color: #1a73e8;
    }
    
    .close {
        color: #aaa;
        font-size: 28px;
        font-weight: bold;
        cursor: pointer;
        line-height: 1;
    }
    
    .close:hover {
        color: #000;
    }
    
    .modal-body {
        padding: 20px;
    }
    
    .modal-footer {
        padding: 0 20px 20px 20px;
        display: flex;
        gap: 12px;
        justify-content: flex-end;
    }
    
    .form-group {
        margin-bottom: 20px;
    }
    
    .form-group label {
        display: block;
        font-weight: 600;
        margin-bottom: 8px;
        color: #333;
    }
    
    .form-group textarea {
        width: 100%;
        padding: 10px;
        border: 1px solid #ccc;
        border-radius: 8px;
        font-size: 16px;
        box-sizing: border-box;
        resize: vertical;
        min-height: 80px;
    }
    
    .btn-primary {
        background: #1a73e8;
        color: white;
        border: none;
        padding: 12px 20px;
        border-radius: 8px;
        font-size: 16px;
        cursor: pointer;
        font-weight: 500;
    }
    
    .btn-primary:hover {
        background: #1557b0;
    }
    
    .btn-secondary {
        background: #6c757d;
        color: white;
        border: none;
        padding: 12px 20px;
        border-radius: 8px;
        font-size: 16px;
        cursor: pointer;
        font-weight: 500;
    }
    
    .btn-secondary:hover {
        background: #5a6268;
    }
    </style>
</head>
<body>
    <div class="container">
        <header class="header">
            <a href="../../index.html">
                <img src="../../data/logo.png" alt="ByggArkiv Snickare Stefan" class="logo">
            </a>
            <div style="text-align: center; margin-top: 16px;">
                <a href="../admin.php" style="color: #1a73e8; text-decoration: none; font-weight: 500;">← Tillbaka till admin</a>
            </div>
        </header>

        <div class="filters">
            <div class="filter-row">
                <div class="filter-group">
                    <label for="projectFilter">Projekt</label>
                    <select id="projectFilter">
                        <option value="">Välj projekt...</option>
                        <?php 
                        // Separera aktiva och arkiverade projekt
                        $activeProjects = [];
                        $archivedProjects = [];
                        foreach ($projectsData as $project) {
                            $status = $project['status'] ?? 'active';
                            if ($status === 'archived') {
                                $archivedProjects[] = $project;
                            } else {
                                $activeProjects[] = $project;
                            }
                        }
                        
                        // Sortera projekt alfabetiskt på namn
                        usort($activeProjects, function($a, $b) {
                            return strcasecmp($a['name'], $b['name']);
                        });
                        usort($archivedProjects, function($a, $b) {
                            return strcasecmp($a['name'], $b['name']);
                        });
                        
                        // Kontrollera om valt projekt är arkiverat
                        $selectedProjectIsArchived = false;
                        if ($selectedProject) {
                            foreach ($archivedProjects as $proj) {
                                if ($proj['id'] == $selectedProject) {
                                    $selectedProjectIsArchived = true;
                                    break;
                                }
                            }
                        }
                        
                        // Visa endast aktiva projekt initialt
                        foreach ($activeProjects as $project): ?>
                            <?php $selected = ($project['id'] == $selectedProject) ? 'selected' : ''; ?>
                            <option value="<?= htmlspecialchars($project['id']) ?>" <?= $selected ?> data-status="active">
                                <?= htmlspecialchars($project['name']) ?> (<?= htmlspecialchars($project['id']) ?>)
                            </option>
                        <?php endforeach; ?>
                        
                        <?php foreach ($archivedProjects as $project): ?>
                            <?php $selected = ($project['id'] == $selectedProject) ? 'selected' : ''; ?>
                            <option value="<?= htmlspecialchars($project['id']) ?>" <?= $selected ?> data-status="archived" style="display: <?= $selectedProjectIsArchived ? '' : 'none' ?>;">
                                <?= htmlspecialchars($project['name']) ?> (<?= htmlspecialchars($project['id']) ?>) (arkiverad)
                            </option>
                        <?php endforeach; ?>
                    </select>
                    <div class="archive-toggle">
                        <input type="checkbox" id="includeArchivedProjects">
                        <label for="includeArchivedProjects">Visa även arkiverade</label>
                    </div>
                </div>
                <div class="filter-group">
                    <label for="userIdFilter">Användare</label>
                    <select id="userIdFilter">
                        <option value="">Välj användare...</option>
                        <?php 
                        // Hämta användardata för att visa fullständiga namn
                        $userData = [];
                        if (file_exists(__DIR__ . '/../../data/userids.json')) {
                            $userContent = file_get_contents(__DIR__ . '/../../data/userids.json');
                            if ($userContent !== false) {
                                $userData = json_decode($userContent, true) ?: [];
                            }
                        }
                        
                        // Filtrera bort externa partners - endast visa user och admin
                        $filteredUserData = array_filter($userData, function($user) {
                            $userType = $user['user_type'] ?? 'user';
                            return $userType === 'user' || $userType === 'admin';
                        });
                        
                        // Separera aktiva och inaktiva användare
                        $activeUsers = [];
                        $inactiveUsers = [];
                        foreach ($filteredUserData as $user) {
                            $isActive = !isset($user['active']) || ($user['active'] !== false && $user['active'] !== 0 && $user['active'] !== '0' && $user['active'] !== 'false');
                            if ($isActive) {
                                $activeUsers[] = $user;
                            } else {
                                $inactiveUsers[] = $user;
                            }
                        }
                        
                        // Sortera användare alfabetiskt på förnamn
                        usort($activeUsers, function($a, $b) {
                            $aFirstName = explode(' ', $a['full_name'] ?? '')[0];
                            $bFirstName = explode(' ', $b['full_name'] ?? '')[0];
                            return strcasecmp($aFirstName, $bFirstName);
                        });
                        usort($inactiveUsers, function($a, $b) {
                            $aFirstName = explode(' ', $a['full_name'] ?? '')[0];
                            $bFirstName = explode(' ', $b['full_name'] ?? '')[0];
                            return strcasecmp($aFirstName, $bFirstName);
                        });
                        
                        // Kontrollera om vald användare är inaktiv
                        $selectedUserIsInactive = false;
                        if ($selectedUserId) {
                            foreach ($inactiveUsers as $usr) {
                                if ($usr['user_id'] == $selectedUserId) {
                                    $selectedUserIsInactive = true;
                                    break;
                                }
                            }
                        }
                        
                        // Visa endast aktiva användare initialt
                        foreach ($activeUsers as $user): 
                            $value = $user['user_id'];
                            $selected = ($value == $selectedUserId) ? 'selected' : ''; 
                            $displayName = $user['full_name'] ?? '';
                        ?>
                            <option value="<?= htmlspecialchars($value) ?>" <?= $selected ?> data-status="active">
                                <?= htmlspecialchars($displayName) ?>
                            </option>
                        <?php endforeach; ?>
                        
                        <?php foreach ($inactiveUsers as $user): 
                            $value = $user['user_id'];
                            $selected = ($value == $selectedUserId) ? 'selected' : ''; 
                            $displayName = $user['full_name'] ?? ' (inaktiv)';
                        ?>
                            <option value="<?= htmlspecialchars($value) ?>" <?= $selected ?> data-status="inactive" style="display: <?= $selectedUserIsInactive ? '' : 'none' ?>;">
                                <?= htmlspecialchars($user['full_name'] ?? '') ?> (inaktiv)
                            </option>
                        <?php endforeach; ?>
                    </select>
                    <div class="archive-toggle">
                        <input type="checkbox" id="includeInactiveUsers">
                        <label for="includeInactiveUsers">Visa även inaktiverade</label>
                    </div>
                </div>
            </div>
        </div>

        <div id="resultsContainer" <?php 
            // Dölj projektet från början om det är arkiverat (kryssrutan är inte ikryssad som standard)
            if ($activeView === 'project' && $selectedProject) {
                $selectedProjectIsArchived = false;
                foreach ($projectsData as $proj) {
                    if ($proj['id'] == $selectedProject) {
                        $status = $proj['status'] ?? 'active';
                        if ($status === 'archived') {
                            $selectedProjectIsArchived = true;
                        }
                        break;
                    }
                }
                if ($selectedProjectIsArchived) {
                    echo 'style="display: none;"';
                }
            }
        ?>>
            <?php if ($activeView === 'project'): ?>
                <?php 
                // Kontrollera om valt projekt är arkiverat
                $selectedProjectIsArchived = false;
                if ($selectedProject) {
                    foreach ($projectsData as $proj) {
                        if ($proj['id'] == $selectedProject) {
                            $status = $proj['status'] ?? 'active';
                            if ($status === 'archived') {
                                $selectedProjectIsArchived = true;
                            }
                            break;
                        }
                    }
                }
                // Kryssrutan ska INTE vara ikryssad som standard
                // Projektet ska döljas om det är arkiverat och kryssrutan inte är ikryssad
                $showArchived = false; // Alltid false som standard
                ?>
                <?php include '../projects/project_view.php'; ?>
            <?php elseif ($activeView === 'user'): ?>
                <?php include '../user_view.php'; ?>
            <?php elseif ($activeView === 'combined'): ?>
                <?php include 'user_project_view.php'; ?>
            <?php else: ?>
                <!-- Visa instruktioner när inga filter är valda -->
                <div class="results-section">
                    <div class="results-header">
                        <span id="resultsTitle">Välj filter för att visa filer</span>
                        <span id="resultsCount"></span>
                    </div>
                    <div class="results-content">
                        <div class="no-results" style="text-align: center; font-size: 14px; color: #666; padding: 40px;">
                            Välj ett projekt eller användare ovan för att visa filer
                        </div>
                    </div>
                </div>
            <?php endif; ?>
        </div>
    </div>

    <!-- Lightbox Modal -->
    <div id="lightbox" class="lightbox">
        <div class="lightbox-content">
            <span class="lightbox-close" onclick="closeLightbox()">&times;</span>
            <img id="lightbox-image" class="lightbox-image" src="" alt="" style="display: none;">
            <video id="lightbox-video" class="lightbox-image" controls style="display: none;">
                Din webbläsare stöder inte videouppspelning.
            </video>
            <div class="lightbox-info">
                <div id="lightbox-image-number" class="lightbox-filename" style="font-weight: bold; color: #fff; margin-bottom: 5px;"></div>
                <div id="lightbox-filename" class="lightbox-filename"></div>
                <div class="lightbox-meta">
                    <span id="lightbox-date"></span>
                    <span id="lightbox-user"></span>
                </div>
                <div id="lightbox-note" style="margin-top: 8px; font-style: italic;"></div>
            </div>
        </div>
    </div>

    <!-- Description Modal -->
    <div id="descriptionModal" class="modal" style="display: none;">
        <div class="modal-content">
            <div class="modal-header">
                <h3>Ändra beskrivning</h3>
                <span class="close" onclick="closeDescriptionModal()">&times;</span>
            </div>
            <div class="modal-body">
                <div class="form-group">
                    <textarea id="descriptionText" rows="4" placeholder="Skriv en beskrivning för denna fil..."></textarea>
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn-secondary" onclick="closeDescriptionModal()">Avbryt</button>
                <button type="button" class="btn-primary" onclick="saveDescription()">Spara</button>
            </div>
        </div>
    </div>

    <script>
        // Lightbox functions
        function openLightbox(imageSrc, filename, date, user, note, imageNumber) {
            // Convert thumbnail URL to full-size image URL
            // Handle both direct file paths and view_file.php URLs
            if (imageSrc && imageSrc.includes('-thumb.')) {
                if (imageSrc.includes('view_file.php')) {
                    // For view_file.php URLs, update the file parameter
                    const url = new URL(imageSrc, window.location.origin);
                    const currentFile = url.searchParams.get('file');
                    if (currentFile && currentFile.includes('-thumb.')) {
                        const fullSizeFile = currentFile.replace('-thumb.', '.');
                        url.searchParams.set('file', fullSizeFile);
                        imageSrc = url.pathname + '?' + url.searchParams.toString();
                    }
                } else {
                    // For direct file paths, simple replace
                    imageSrc = imageSrc.replace('-thumb.', '.');
                }
            }
            
            const lightbox = document.getElementById('lightbox');
            const lightboxImage = document.getElementById('lightbox-image');
            const lightboxVideo = document.getElementById('lightbox-video');
            const lightboxImageNumber = document.getElementById('lightbox-image-number');
            const lightboxFilename = document.getElementById('lightbox-filename');
            const lightboxDate = document.getElementById('lightbox-date');
            const lightboxUser = document.getElementById('lightbox-user');
            const lightboxNote = document.getElementById('lightbox-note');
            
            if (!lightbox || !lightboxImage || !lightboxVideo) {
                console.error('Lightbox elements not found');
                return;
            }
            
            // Kontrollera om det är en video baserat på filnamn
            const isVideo = /\.(mp4|webm|mov|avi)$/i.test(filename);
            
            if (isVideo) {
                // Visa video, dölj bild
                lightboxImage.style.display = 'none';
                lightboxVideo.style.display = 'block';
                lightboxVideo.src = imageSrc;
                lightboxVideo.load(); // Ladda videon
            } else {
                // Visa bild, dölj video
                lightboxVideo.style.display = 'none';
                lightboxImage.style.display = 'block';
                lightboxImage.src = imageSrc;
            }
            
            // Visa bildnummer om det finns
            if (imageNumber && imageNumber.trim() !== '') {
                lightboxImageNumber.textContent = imageNumber;
                lightboxImageNumber.style.display = 'block';
            } else {
                lightboxImageNumber.style.display = 'none';
            }
            
            lightboxFilename.textContent = filename;
            lightboxDate.textContent = date;
            lightboxUser.textContent = user;
            lightboxNote.textContent = note || '';
            
            lightbox.style.display = 'block';
            document.body.style.overflow = 'hidden'; // Prevent background scrolling
        }
        
        function closeLightbox() {
            const lightbox = document.getElementById('lightbox');
            const lightboxVideo = document.getElementById('lightbox-video');
            if (lightbox) {
                lightbox.style.display = 'none';
                document.body.style.overflow = 'auto'; // Restore scrolling
                
                // Pausa video om den spelar
                if (lightboxVideo) {
                    lightboxVideo.pause();
                    lightboxVideo.currentTime = 0;
                }
            }
        }
        
        // Close lightbox when clicking outside the image
        document.addEventListener('click', function(event) {
            const lightbox = document.getElementById('lightbox');
            if (event.target === lightbox) {
                closeLightbox();
            }
        });
        
        // Close lightbox with Escape key
        document.addEventListener('keydown', function(event) {
            if (event.key === 'Escape') {
                closeLightbox();
            }
        });

        // Delegated handler for view buttons - reads from data attributes to get current values
        document.addEventListener('click', function(e) {
            const btn = e.target.closest('.view-btn');
            if (!btn) return;
            const item = btn.closest('.file-item');
            if (!item) return;
            let imageUrl = item.getAttribute('data-full-image-url') || item.getAttribute('data-image-url');
            const filename = item.getAttribute('data-filename') || '';
            const date = item.getAttribute('data-date') || '';
            const user = item.getAttribute('data-user') || '';
            const note = item.getAttribute('data-note') || ''; // Read from data-note attribute
            const imageNumber = item.getAttribute('data-image-number') || '';
            if (!imageUrl) {
                const imgEl = item.querySelector('img.file-image');
                if (imgEl) imageUrl = imgEl.getAttribute('src');
            }
            openLightbox(imageUrl, filename, date, user, note, imageNumber);
        });

        // Wait for DOM to be ready
        document.addEventListener('DOMContentLoaded', function() {
            const projectFilter = document.getElementById('projectFilter');
            const userIdFilter = document.getElementById('userIdFilter');
            const includeArchivedCheckbox = document.getElementById('includeArchivedProjects');
            const includeInactiveCheckbox = document.getElementById('includeInactiveUsers');
            const resultsContainer = document.getElementById('resultsContainer');
            
            // Funktion för att uppdatera projekt-dropdown baserat på kryssruta
            async function updateProjectFilter() {
                if (!projectFilter || !includeArchivedCheckbox) return;
                
                const includeArchived = includeArchivedCheckbox.checked;
                const selectedValue = projectFilter.value; // Spara valt värde
                
                try {
                    // Hämta alla projekt från API:et
                    const response = await fetch('../api/projects.php?action=list');
                    if (!response.ok) {
                        throw new Error('Kunde inte hämta projekt');
                    }
                    const data = await response.json();
                    if (!data.success || !data.projects) {
                        throw new Error('Ogiltigt svar från servern');
                    }
                    
                    const allProjects = data.projects;
                    
                    // Separera aktiva och arkiverade projekt
                    const activeProjects = allProjects.filter(p => (p.status || 'active') === 'active');
                    const archivedProjects = allProjects.filter(p => (p.status || 'active') === 'archived');
                    
                    // Sortera projekt alfabetiskt på namn
                    activeProjects.sort((a, b) => (a.name || '').localeCompare(b.name || '', 'sv'));
                    archivedProjects.sort((a, b) => (a.name || '').localeCompare(b.name || '', 'sv'));
                    
                    // Rensa dropdown-menyn
                    projectFilter.innerHTML = '';
                    
                    // Lägg till första option
                    const firstOption = document.createElement('option');
                    firstOption.value = '';
                    firstOption.textContent = 'Välj projekt...';
                    projectFilter.appendChild(firstOption);
                    
                    // Lägg till alla aktiva projekt först
                    activeProjects.forEach(project => {
                        const option = document.createElement('option');
                        option.value = project.id;
                        option.textContent = `${project.name} (${project.id})`;
                        option.setAttribute('data-status', 'active');
                        if (project.id === selectedValue) {
                            option.selected = true;
                        }
                        projectFilter.appendChild(option);
                    });
                    
                    // Lägg till arkiverade projekt efter aktiva om kryssrutan är ikryssad
                    if (includeArchived) {
                        archivedProjects.forEach(project => {
                            const option = document.createElement('option');
                            option.value = project.id;
                            option.textContent = `${project.name} (${project.id}) (arkiverad)`;
                            option.setAttribute('data-status', 'archived');
                            if (project.id === selectedValue) {
                                option.selected = true;
                            }
                            projectFilter.appendChild(option);
                        });
                    }
                    
                    // Återställ valt värde om det fortfarande finns
                    if (selectedValue) {
                        const optionExists = Array.from(projectFilter.options).some(opt => opt.value === selectedValue);
                        if (optionExists) {
                            projectFilter.value = selectedValue;
                        } else {
                            // Om valt projekt är arkiverat och kryssrutan inte är ikryssad, rensa valet
                            projectFilter.value = '';
                        }
                    }
                    
                    // Uppdatera första option med räkning
                    const visibleCount = activeProjects.length + (includeArchived ? archivedProjects.length : 0);
                    firstOption.textContent = `Välj projekt (${visibleCount} tillgängliga)`;
                    
                    // Kontrollera om valt projekt är arkiverat och dölj/visa projektet
                    const selectedOption = projectFilter.options[projectFilter.selectedIndex];
                    if (selectedOption && selectedOption.getAttribute('data-status') === 'archived') {
                        const resultsContainer = document.getElementById('resultsContainer');
                        if (resultsContainer) {
                            if (!includeArchived) {
                                // Dölj projektet om kryssrutan inte är ikryssad
                                resultsContainer.style.display = 'none';
                            } else {
                                // Visa projektet om kryssrutan är ikryssad
                                resultsContainer.style.display = '';
                            }
                        }
                    } else if (selectedOption && selectedOption.getAttribute('data-status') === 'active') {
                        // Visa projektet om det är aktivt
                        const resultsContainer = document.getElementById('resultsContainer');
                        if (resultsContainer) {
                            resultsContainer.style.display = '';
                        }
                    }
                } catch (error) {
                    console.error('Fel vid uppdatering av projektfilter:', error);
                    alert('Kunde inte uppdatera projektlista: ' + error.message);
                }
            }
            
            // Funktion för att uppdatera användar-dropdown baserat på kryssruta
            async function updateUserFilter() {
                if (!userIdFilter || !includeInactiveCheckbox) return;
                
                const includeInactive = includeInactiveCheckbox.checked;
                const selectedValue = userIdFilter.value; // Spara valt värde
                
                try {
                    // Hämta alla användare från API:et
                    const response = await fetch('../api/users.php?action=list');
                    if (!response.ok) {
                        throw new Error('Kunde inte hämta användare');
                    }
                    const data = await response.json();
                    if (!data.success || !data.users) {
                        throw new Error('Ogiltigt svar från servern');
                    }
                    
                    const allUsers = data.users;
                    
                    // Filtrera bort externa användare - endast visa user och admin
                    const filteredUsers = allUsers.filter(user => {
                        const userType = user.user_type || 'user';
                        return userType === 'user' || userType === 'admin';
                    });
                    
                    // Separera aktiva och inaktiva användare
                    const activeUsers = filteredUsers.filter(user => {
                        const active = user.active;
                        return active !== false && active !== 0 && active !== '0' && active !== 'false';
                    });
                    const inactiveUsers = filteredUsers.filter(user => {
                        const active = user.active;
                        return active === false || active === 0 || active === '0' || active === 'false';
                    });
                    
                    // Sortera användare alfabetiskt på förnamn
                    activeUsers.sort((a, b) => {
                        const aFirstName = (a.full_name || '').split(' ')[0];
                        const bFirstName = (b.full_name || '').split(' ')[0];
                        return aFirstName.localeCompare(bFirstName, 'sv');
                    });
                    inactiveUsers.sort((a, b) => {
                        const aFirstName = (a.full_name || '').split(' ')[0];
                        const bFirstName = (b.full_name || '').split(' ')[0];
                        return aFirstName.localeCompare(bFirstName, 'sv');
                    });
                    
                    // Rensa dropdown-menyn
                    userIdFilter.innerHTML = '';
                    
                    // Lägg till första option
                    const firstOption = document.createElement('option');
                    firstOption.value = '';
                    firstOption.textContent = 'Välj användare...';
                    userIdFilter.appendChild(firstOption);
                    
                    // Lägg till alla aktiva användare först
                    activeUsers.forEach(user => {
                        const option = document.createElement('option');
                        option.value = user.user_id;
                        option.textContent = user.full_name || user.user_id;
                        option.setAttribute('data-status', 'active');
                        if (user.user_id === selectedValue) {
                            option.selected = true;
                        }
                        userIdFilter.appendChild(option);
                    });
                    
                    // Lägg till inaktiva användare efter aktiva om kryssrutan är ikryssad
                    if (includeInactive) {
                        inactiveUsers.forEach(user => {
                            const option = document.createElement('option');
                            option.value = user.user_id;
                            option.textContent = `${user.full_name || user.user_id} (inaktiv)`;
                            option.setAttribute('data-status', 'inactive');
                            if (user.user_id === selectedValue) {
                                option.selected = true;
                            }
                            userIdFilter.appendChild(option);
                        });
                    }
                    
                    // Återställ valt värde om det fortfarande finns
                    if (selectedValue) {
                        const optionExists = Array.from(userIdFilter.options).some(opt => opt.value === selectedValue);
                        if (optionExists) {
                            userIdFilter.value = selectedValue;
                        } else {
                            // Om vald användare är inaktiv och kryssrutan inte är ikryssad, rensa valet
                            userIdFilter.value = '';
                        }
                    }
                    
                    // Uppdatera första option med räkning
                    const visibleCount = activeUsers.length + (includeInactive ? inactiveUsers.length : 0);
                    firstOption.textContent = `Välj användare (${visibleCount} tillgängliga)`;
                } catch (error) {
                    console.error('Fel vid uppdatering av användarfilter:', error);
                    alert('Kunde inte uppdatera användarlista: ' + error.message);
                }
            }
            
            // Event listeners för kryssrutorna
            if (includeArchivedCheckbox) {
                includeArchivedCheckbox.addEventListener('change', updateProjectFilter);
            }
            
            if (includeInactiveCheckbox) {
                includeInactiveCheckbox.addEventListener('change', updateUserFilter);
            }
            
            // Initial uppdatering
            updateProjectFilter();
            updateUserFilter();
            
            // Se till att projektet döljs från början om det är arkiverat och kryssrutan inte är ikryssad
            if (includeArchivedCheckbox && !includeArchivedCheckbox.checked) {
                const selectedOption = projectFilter ? projectFilter.options[projectFilter.selectedIndex] : null;
                if (selectedOption && selectedOption.getAttribute('data-status') === 'archived') {
                    if (resultsContainer) {
                        resultsContainer.style.display = 'none';
                    }
                }
            }

            function navigateToFilter() {
                const selectedProject = projectFilter.value;
                const selectedUserId = userIdFilter.value;
                
                // Build URL with all active filters
                const params = new URLSearchParams();
                
                if (selectedProject) {
                    params.append('project_id', selectedProject);
                }
                
                if (selectedUserId !== undefined && selectedUserId !== '') {
                    params.append('filter_user_id', selectedUserId);
                }
                
                // Navigate to the appropriate URL
                const queryString = params.toString();
                if (queryString) {
                    window.location.href = `?${queryString}`;
                } else {
                    window.location.href = window.location.pathname;
                }
            }

            // Event listeners
            if (projectFilter) {
                projectFilter.addEventListener('change', function() {
                    navigateToFilter();
                });
            }
            
            if (userIdFilter) {
                userIdFilter.addEventListener('change', function() {
                    navigateToFilter();
                });
            }
        });

        async function deleteFile(button) {
            console.log('deleteFile called');
            if (!confirm('Är du säker på att du vill ta bort denna fil?\n\nFilen kommer att tas bort permanent och kan inte återställas.')) {
                return;
            }

            const fileItem = button.closest('.file-item');
            if (!fileItem) {
                console.error('File item not found');
                return;
            }
            
            const imagePath = fileItem.dataset.imagePath;
            const jsonPath = fileItem.dataset.jsonPath;

            if (!imagePath || !jsonPath) {
                console.error('Missing file paths:', { imagePath, jsonPath });
                alert('Fel: Saknade filvägar');
                return;
            }

            const formData = new FormData();
            formData.append('imagePath', imagePath);
            formData.append('jsonPath', jsonPath);

            try {
                console.log('Sending delete request...');
                const response = await fetch('delete_file.php', {
                    method: 'POST',
                    body: formData
                });

                if (!response.ok) {
                    throw new Error(`HTTP error! status: ${response.status}`);
                }

                const responseText = await response.text();
                console.log('Delete response:', responseText);
                
                let result;
                try {
                    result = JSON.parse(responseText);
                } catch (parseError) {
                    console.error('Non-JSON response:', responseText);
                    throw new Error('Servern returnerade ett ogiltigt svar. Kontrollera att delete_file.php fungerar korrekt.');
                }

                if (result.ok) {
                    // Remove the file item from the display
                    fileItem.remove();
                    alert('Fil borttagen');
                } else {
                    alert('Fel: ' + result.error);
                }
            } catch (error) {
                console.error('Delete error:', error);
                alert('Ett fel uppstod: ' + error.message);
            }
        }

        let currentEditButton = null;
        let currentFileItem = null;

        function addDescription(button) {
            currentEditButton = button;
            currentFileItem = button.closest('.file-item');
            if (!currentFileItem) {
                console.error('File item not found');
                return;
            }
            const imagePath = currentFileItem.dataset.imagePath;
            const jsonPath = currentFileItem.dataset.jsonPath;
            if (!imagePath || !jsonPath) {
                console.error('Missing file paths:', { imagePath, jsonPath });
                alert('Fel: Saknade filvägar');
                return;
            }
            
            // Get existing note if any
            const existingNote = currentFileItem.dataset.note || '';
            
            // Fill in existing note and show modal
            document.getElementById('descriptionText').value = existingNote;
            document.getElementById('descriptionModal').style.display = 'flex';
            document.getElementById('descriptionText').focus();
        }

        function closeDescriptionModal() {
            document.getElementById('descriptionModal').style.display = 'none';
            currentEditButton = null;
            currentFileItem = null;
        }

        function saveDescription() {
            if (!currentFileItem || !currentEditButton) {
                return;
            }
            
            const description = document.getElementById('descriptionText').value.trim();
            if (!description) {
                alert('Ange en beskrivning');
                return;
            }
            
            const imagePath = currentFileItem.dataset.imagePath;
            const jsonPath = currentFileItem.dataset.jsonPath;
            
            const formData = new FormData();
            formData.append('imagePath', imagePath);
            formData.append('jsonPath', jsonPath);
            formData.append('description', description);
            
            // Hämta user_id från URL eller cookie
            const urlParams = new URLSearchParams(window.location.search);
            const userId = urlParams.get('user_id') || (function() {
                const cookies = document.cookie.split(';');
                for (let cookie of cookies) {
                    const [name, value] = cookie.trim().split('=');
                    if (name === 'user_id') {
                        return decodeURIComponent(value);
                    }
                }
                return '';
            })();
            if (userId) {
                formData.append('user_id', userId);
            }
            
            fetch('../../api/files/add_description.php', { method: 'POST', body: formData })
                .then(response => response.json())
                .then(result => {
                    if (result.ok) {
                        // Update existing note or create new one
                        let noteDiv = currentFileItem.querySelector('.file-note');
                        if (noteDiv) {
                            // Update existing note
                            noteDiv.textContent = description;
                        } else {
                            // Create new note
                            noteDiv = document.createElement('div');
                            noteDiv.className = 'file-note';
                            noteDiv.style.cssText = 'margin-top: 6px; padding: 6px 8px 2px 8px; background: #e3f2fd; border-radius: 4px; font-style: normal; font-weight: normal; color: #000; margin-bottom: 2px; word-wrap: break-word;';
                            noteDiv.textContent = description;
                            const fileInfo = currentFileItem.querySelector('.file-info');
                            if (fileInfo) {
                                const filenameLink = fileInfo.querySelector('.file-filename');
                                if (filenameLink && filenameLink.parentNode === fileInfo) {
                                    fileInfo.insertBefore(noteDiv, filenameLink.nextSibling);
                                } else {
                                    fileInfo.appendChild(noteDiv);
                                }
                            } else {
                                currentEditButton.parentNode.appendChild(noteDiv);
                            }
                        }
                        
                        // Update the data attribute
                        currentFileItem.dataset.note = description;
                        
                        // Update button title
                        currentEditButton.title = 'Redigera beskrivning';
                        
                        closeDescriptionModal();
                    } else {
                        alert('Fel: ' + result.error);
                    }
                })
                .catch(error => {
                    console.error('Add description error:', error);
                    alert('Ett fel uppstod: ' + error.message);
                });
        }

        // Wait for DOM to be ready
        document.addEventListener('DOMContentLoaded', function() {
            // Close modal with Escape key
            document.addEventListener('keydown', function(e) {
                if (e.key === 'Escape' && document.getElementById('descriptionModal').style.display === 'flex') {
                    closeDescriptionModal();
                }
            });
        });

        async function moveFile(button) {
            console.log('moveFile called');
            const fileItem = button.closest('.file-item');
            if (!fileItem) {
                console.error('File item not found');
                return;
            }
            
            const imagePath = fileItem.dataset.imagePath;
            const jsonPath = fileItem.dataset.jsonPath;
            const currentProjectId = fileItem.dataset.projectId;
            
            if (!imagePath || !jsonPath || !currentProjectId) {
                console.error('Missing required data:', { imagePath, jsonPath, currentProjectId });
                alert('Fel: Saknade filuppgifter');
                return;
            }
            
            // Hämta alla projekt via API (inklusive inaktiva för flytt)
            let allProjects = [];
            let activeProjects = [];
            let archivedProjects = [];
            
            try {
                // Använd API:et istället för direkt JSON-läsning (samma som på rad 1122)
                const response = await fetch('../api/projects.php?action=list');
                if (!response.ok) {
                    throw new Error(`HTTP error! status: ${response.status}`);
                }
                const data = await response.json();
                if (!data.success || !data.projects) {
                    throw new Error('Ogiltigt svar från servern');
                }
                const projectsData = data.projects;
                // Exkludera bara nuvarande projekt, inkludera både aktiva och arkiverade
                const filteredProjects = projectsData.filter(project => project.id !== currentProjectId);
                
                // Separera aktiva och arkiverade projekt
                activeProjects = filteredProjects.filter(project => (project.status || 'active') === 'active');
                archivedProjects = filteredProjects.filter(project => (project.status || 'active') === 'archived');
                
                // Sortera båda grupperna alfabetiskt
                activeProjects.sort((a, b) => a.name.localeCompare(b.name, 'sv'));
                archivedProjects.sort((a, b) => a.name.localeCompare(b.name, 'sv'));
                
                // Kombinera: aktiva först, sedan arkiverade
                allProjects = [...activeProjects, ...archivedProjects];
            } catch (error) {
                console.error('Kunde inte hämta projekt:', error);
                // Fallback: använd projektfilter
                const projectFilter = document.getElementById('projectFilter');
                if (projectFilter) {
                    allProjects = Array.from(projectFilter.options)
                        .filter(option => option.value !== '' && option.value !== currentProjectId)
                        .map(opt => ({ id: opt.value, name: opt.textContent }));
                }
            }
            
            if (allProjects.length === 0) {
                alert('Inga andra projekt tillgängliga att flytta till');
                return;
            }
            
            // Skapa en dropdown för att välja målprojekt
            const projectSelect = document.createElement('select');
            projectSelect.style.cssText = 'padding: 8px; margin: 10px 0; width: 100%; border: 1px solid #ccc; border-radius: 4px;';
            
            const defaultOption = document.createElement('option');
            defaultOption.value = '';
            defaultOption.textContent = 'Välj målprojekt...';
            projectSelect.appendChild(defaultOption);
            
            // Lägg till aktiva projekt
            activeProjects.forEach(project => {
                const option = document.createElement('option');
                option.value = project.id;
                option.textContent = project.name || project.id || 'Okänt projekt';
                projectSelect.appendChild(option);
            });
            
            // Lägg till avgränsare om det finns både aktiva och arkiverade projekt
            if (activeProjects.length > 0 && archivedProjects.length > 0) {
                const separator = document.createElement('option');
                separator.disabled = true;
                separator.textContent = '───────────────';
                separator.style.cssText = 'color: #999; font-style: italic;';
                projectSelect.appendChild(separator);
            }
            
            // Lägg till arkiverade projekt
            archivedProjects.forEach(project => {
                const option = document.createElement('option');
                option.value = project.id;
                option.textContent = project.name || project.id || 'Okänt projekt';
                option.style.cssText = 'color: #666; font-style: italic;';
                projectSelect.appendChild(option);
            });
            
            // Skapa en dialog
            const dialog = document.createElement('div');
            dialog.style.cssText = `
                position: fixed;
                top: 0;
                left: 0;
                width: 100%;
                height: 100%;
                background: rgba(0,0,0,0.5);
                display: flex;
                align-items: center;
                justify-content: center;
                z-index: 1000;
            `;
            
            const dialogContent = document.createElement('div');
            dialogContent.style.cssText = `
                background: white;
                padding: 20px;
                border-radius: 8px;
                max-width: 400px;
                width: 90%;
                text-align: center;
            `;
            
            dialogContent.innerHTML = `
                <h3>Flytta fil till annat projekt</h3>
                <p>Välj målprojekt för denna fil:</p>
            `;
            
            dialogContent.appendChild(projectSelect);
            
            const buttonContainer = document.createElement('div');
            buttonContainer.style.cssText = 'margin-top: 20px; display: flex; gap: 10px; justify-content: flex-end;';
            
            const cancelBtn = document.createElement('button');
            cancelBtn.textContent = 'Avbryt';
            cancelBtn.className = 'btn-cancel';
            cancelBtn.onclick = () => document.body.removeChild(dialog);
            
            const moveBtn = document.createElement('button');
            moveBtn.textContent = 'Flytta';
            moveBtn.style.cssText = 'padding: 8px 16px; background: #28a745; color: white; border: none; border-radius: 4px; cursor: pointer;';
            moveBtn.onclick = async () => {
                const selectedProject = projectSelect.value;
                if (!selectedProject) {
                    alert('Välj ett målprojekt');
                    return;
                }
                
                const projectName = projectSelect.options[projectSelect.selectedIndex].textContent;
                
                // Hämta user_id från URL-parametrar
                const urlParams = new URLSearchParams(window.location.search);
                const userId = urlParams.get('user_id') || '';
                
                const formData = new FormData();
                formData.append('imagePath', imagePath);
                formData.append('jsonPath', jsonPath);
                formData.append('newProjectId', selectedProject);
                if (userId) {
                    formData.append('user_id', userId);
                }
                
                try {
                    console.log('Sending move request...');
                    const response = await fetch('move_file.php', {
                        method: 'POST',
                        body: formData
                    });
                    
                    if (!response.ok) {
                        throw new Error(`HTTP error! status: ${response.status}`);
                    }
                    
                    // Läs svaret som text först för att hantera eventuella parsing-fel
                    const responseText = await response.text();
                    console.log('Move response text:', responseText);
                    
                    let result;
                    try {
                        result = JSON.parse(responseText);
                    } catch (parseError) {
                        console.error('JSON parse error:', parseError, 'Response text:', responseText);
                        throw new Error('Ogiltigt svar från servern: ' + responseText.substring(0, 100));
                    }
                    
                    console.log('Move response:', result);
                    
                    if (result.ok) {
                        // Stäng dialogen
                        document.body.removeChild(dialog);
                        // Uppdatera sidan för att visa den nya situationen
                        window.location.reload();
                    } else {
                        alert('Fel: ' + result.error);
                    }
                } catch (error) {
                    console.error('Move error:', error);
                    alert('Ett fel uppstod: ' + error.message);
                }
            };
            
            buttonContainer.appendChild(moveBtn);
            buttonContainer.appendChild(cancelBtn);
            dialogContent.appendChild(buttonContainer);
            dialog.appendChild(dialogContent);
            document.body.appendChild(dialog);
        }
        
        // Scroll to top funktionalitet
        function scrollToTop() {
            window.scrollTo({
                top: 0,
                behavior: 'smooth'
            });
        }
        
        // Visa/dölj scroll-to-top knapp baserat på scroll-position
        window.addEventListener('scroll', function() {
            const scrollBtn = document.getElementById('scrollToTopBtn');
            if (window.pageYOffset > 300) {
                scrollBtn.style.display = 'block';
            } else {
                scrollBtn.style.display = 'none';
            }
        });
    </script>
    
    <!-- Scroll to top knapp -->
    <button id="scrollToTopBtn" onclick="scrollToTop()" style="
        position: fixed;
        bottom: 30px;
        right: 30px;
        width: 50px;
        height: 50px;
        background: #1a73e8;
        color: white;
        border: none;
        border-radius: 50%;
        cursor: pointer;
        font-size: 20px;
        box-shadow: 0 4px 12px rgba(26, 115, 232, 0.3);
        transition: all 0.3s ease;
        display: none;
        z-index: 1000;
    " onmouseover="this.style.background='#1557b0'; this.style.transform='scale(1.1)'" onmouseout="this.style.background='#1a73e8'; this.style.transform='scale(1)'" title="Scrolla till toppen">
        ↑
    </button>
</body>
</html>
