<?php
require_once __DIR__ . '/../admin_guard.php';
require_once __DIR__ . '/../../config.php';
require_once __DIR__ . '/../../lib/data_functions.php';

// Kontrollera om arkiverade projekt ska visas
$showArchived = isset($_GET['show_archived']) && $_GET['show_archived'] === '1';

// Hämta projektdata för namn
$projectsData = [];
$projectNames = [];
if (file_exists(__DIR__ . '/../../data/projects.json')) {
    $allProjects = json_decode(file_get_contents(__DIR__ . '/../../data/projects.json'), true) ?: [];
    // Lägg till alla projekt (aktiva och arkiverade)
    foreach ($allProjects as $project) {
        $projectNames[$project['id']] = $project['name'];
    }
}

// Hämta endast projekt från projects.json
$uploadRoot = __DIR__ . '/../../' . rtrim(UPLOAD_ROOT, '/\\');
$projects = [];


// Lägg till projekt baserat på showArchived parameter
foreach ($allProjects as $project) {
    $project_id = $project['id'];
    $projectStatus = $project['status'] ?? 'active';
    
    // Visa endast aktiva projekt om showArchived inte är satt
    if (!$showArchived && $projectStatus !== 'active') {
        continue;
    }
    
    
    $projectDir = $uploadRoot . '/' . $project_id;
    $projectFiles = [];
    
    // Kontrollera om projektmappen finns
    if (is_dir($projectDir)) {
        // Hämta alla JSON-filer (metadata)
        $jsonFiles = glob($projectDir . '/*.json');
        
        foreach ($jsonFiles as $jsonFile) {
            $metadata = json_decode(file_get_contents($jsonFile), true);
            if ($metadata) {
                $imageFile = null;
                $imageExt = null;
                
                // Först: Försök använda stored_file från JSON-metadata om det finns
                if (!empty($metadata['stored_file'])) {
                    $storedFile = $projectDir . '/' . basename($metadata['stored_file']);
                    if (file_exists($storedFile) && is_file($storedFile)) {
                        $imageFile = $storedFile;
                        $imageExt = strtolower(pathinfo($metadata['stored_file'], PATHINFO_EXTENSION));
                    }
                }
                
                // Fallback: Om stored_file inte finns eller inte matchar, försök hitta baserat på JSON-filnamn
                if (!$imageFile) {
                    $baseName = str_replace('.json', '', $jsonFile);
                    
                    // Kontrollera alla stödda bildformat och videoformat
                    $supportedImageFormats = ['jpg', 'jpeg', 'png', 'webp', 'heic', 'heif'];
                    $supportedVideoFormats = ['mp4', 'webm', 'mov', 'avi'];
                    $allSupportedFormats = array_merge($supportedImageFormats, $supportedVideoFormats);
                    
                    foreach ($allSupportedFormats as $format) {
                        $testFile = $baseName . '.' . $format;
                        if (file_exists($testFile)) {
                            $imageFile = $testFile;
                            $imageExt = $format;
                            break;
                        }
                    }
                }
                
                if ($imageFile) {
                    $metadata['image_path'] = $imageFile;
                    $metadata['image_url'] = '../../view_file.php?project_id=' . urlencode($project_id) . '&file=' . urlencode(basename($imageFile));
                    
                    // Bestäm filtyp (använd från JSON om det finns, annars gissa från filändelse)
                    if (!isset($metadata['file_type'])) {
                        $supportedVideoFormats = ['mp4', 'webm', 'mov', 'avi'];
                        $isVideo = in_array($imageExt, $supportedVideoFormats);
                        $metadata['file_type'] = $isVideo ? 'video' : 'image';
                    }
                    
                    // Thumb detection (använd thumb_file från JSON om det finns, annars gissa)
                    if (($metadata['file_type'] ?? 'image') !== 'video') {
                        if (!empty($metadata['thumb_file'])) {
                            $thumbPath = $projectDir . '/' . basename($metadata['thumb_file']);
                            if (file_exists($thumbPath)) {
                                $metadata['thumb_path'] = $thumbPath;
                                $metadata['thumb_url'] = '../../view_file.php?project_id=' . urlencode($project_id) . '&file=' . urlencode(basename($thumbPath));
                            }
                        } else {
                            // Fallback: Gissa thumbnail-namn
                            $baseNoExt = preg_replace('/\.(jpg|jpeg|png|webp|heic|heif)$/i', '', $imageFile);
                            $thumbPath = $baseNoExt . '-thumb.jpg';
                            if (file_exists($thumbPath)) {
                                $metadata['thumb_path'] = $thumbPath;
                                $metadata['thumb_url'] = '../../view_file.php?project_id=' . urlencode($project_id) . '&file=' . urlencode(basename($thumbPath));
                            }
                        }
                    }
                    $projectFiles[] = $metadata;
                }
            }
        }
        
        // Sortera efter datum (nyaste först)
        usort($projectFiles, function($a, $b) {
            return strtotime($b['stored_at']) - strtotime($a['stored_at']);
        });
    }
    
    // Lägg alltid till projektet i $projects (även om det är tomt eller saknar mapp)
    $projects[$project_id] = $projectFiles;
}


// Separera aktiva och arkiverade projekt för sortering
$activeProjects = [];
$archivedProjects = [];

foreach ($projects as $projectId => $files) {
    // Hitta projektstatus
    $projectStatus = 'active';
    foreach ($allProjects as $project) {
        if ($project['id'] === $projectId) {
            $projectStatus = $project['status'] ?? 'active';
            break;
        }
    }
    
    $projectName = isset($projectNames[$projectId]) ? $projectNames[$projectId] : $projectId;
    // Ta bort projekt-ID prefix om det finns (t.ex. "1234 – Markvägen" blir "Markvägen")
    if (str_starts_with($projectName, $projectId . ' – ')) {
        $projectName = substr($projectName, strlen($projectId . ' – '));
    }
    
    $projectData = [
        'id' => $projectId,
        'name' => $projectName,
        'files' => $files,
        'status' => $projectStatus
    ];
    
    if ($projectStatus === 'archived') {
        $archivedProjects[] = $projectData;
    } else {
        $activeProjects[] = $projectData;
    }
}

// Filtrera bort tomma projekt (alltid)
$activeProjects = array_values(array_filter($activeProjects, function($projectData) {
    return !empty($projectData['files']);
}));
$archivedProjects = array_values(array_filter($archivedProjects, function($projectData) {
    return !empty($projectData['files']);
}));

// Sortera båda grupperna alfabetiskt
usort($activeProjects, function($a, $b) {
    return strcasecmp($a['name'], $b['name']);
});
usort($archivedProjects, function($a, $b) {
    return strcasecmp($a['name'], $b['name']);
});

// Kombinera: aktiva först, sedan arkiverade
$sortedProjects = [];
foreach ($activeProjects as $projectData) {
    $sortedProjects[$projectData['id']] = $projectData['files'];
}
foreach ($archivedProjects as $projectData) {
    $sortedProjects[$projectData['id']] = $projectData['files'];
}

$projects = $sortedProjects;

// Hämta arkiverade projekt-ID:n för att kunna identifiera dem i HTML (efter filtrering och sortering)
$archivedProjectIds = [];
foreach ($archivedProjects as $projectData) {
    $archivedProjectIds[] = $projectData['id'];
}
?>
<!DOCTYPE html>
<html lang="sv">
<head>
    <meta charset="utf-8" />
    <meta name="viewport" content="width=device-width,initial-scale=1" />
    <title>Lista uppladdningar - ByggArkiv</title>
    <link rel="stylesheet" href="../common.css">
    <style>
        body { 
            font-family: system-ui, -apple-system, Segoe UI, Roboto, sans-serif; 
            margin: 16px; 
            background-color: #f5f5f5;
            line-height: 1.6;
        }
        .header {
            text-align: center;
            margin-bottom: 30px;
        }
        .header img {
            max-width: 100%;
            height: auto;
            max-height: 120px;
            display: block;
            margin: 0 auto;
        }
        .container {
            max-width: 1200px;
            margin: 0 auto;
        }
        .project-section {
            background: white;
            border-radius: 12px;
            margin-bottom: 24px;
            box-shadow: 0 2px 8px rgba(0,0,0,0.1);
            overflow: hidden;
        }
        /* Archived project styling */
        .project-section.archived {
            background: #ffe6e6;
            border: 1px solid #dc3545;
        }
        /* Kompakt styling för tomma projekt */
        .project-section.empty {
            margin-bottom: 4px;
        }
        .project-section.empty .project-header {
            padding: 4px 12px;
            font-size: 12px;
            display: flex;
            align-items: center;
            justify-content: space-between;
        }
        .project-section.empty .project-header .project-name {
            font-size: 12px;
        }
        .project-section.empty .project-header .project-info {
            font-size: 11px;
            margin-left: 12px;
        }
        .project-section.empty .project-content {
            padding: 4px 12px 4px 12px;
            min-height: auto;
        }
        .project-section.empty .empty-state {
            padding: 4px 0;
            font-size: 11px;
            margin: 0;
        }
        .project-header {
            background: #1a73e8;
            color: white;
            padding: 16px 20px;
            font-size: 18px;
            font-weight: 600;
        }
        .project-section.archived .project-header,
        .project-header.archived {
            background: #d9534f !important;
        }
        .project-content {
            padding: 20px;
        }
        .file-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(300px, 1fr));
            gap: 20px;
        }
        .file-item {
            border: 1px solid #e0e0e0;
            border-radius: 8px;
            overflow: hidden;
            background: #fafafa;
        }
        .edit-btn {
            position: absolute;
            top: 8px;
            right: 119px;
            background: white;
            color: #ff9800;
            border: 2px solid #ff9800;
            border-radius: 50%;
            width: 30px;
            height: 30px;
            font-size: 16px;
            cursor: pointer;
            display: flex;
            align-items: center;
            justify-content: center;
            opacity: 0.9;
            transition: opacity 0.2s;
            z-index: 10;
            pointer-events: auto;
        }
        .edit-btn:hover {
            opacity: 1;
            background: #ff9800;
            color: white;
        }
        .file-image {
            width: 100%;
            height: 200px;
            object-fit: cover;
            background: #f0f0f0;
            cursor: default;
            transition: opacity 0.2s;
        }
        .file-image:hover {
            opacity: 0.9;
        }
        .file-info {
            padding: 16px;
        }
        .file-name {
            font-weight: 600;
            margin-bottom: 8px;
            color: #333;
        }
        .file-filename {
            font-size: 12px;
            color: #1a73e8;
            text-decoration: underline;
        }
        .file-meta {
            font-size: 14px;
            color: #666;
            margin-bottom: -4px;
        }
        .file-note {
            margin-top: 8px;
            padding: 8px;
            background: #e3f2fd;
            border-radius: 4px;
            font-style: normal;
            color: #1976d2;
        }
        .empty-project {
            text-align: center;
            padding: 40px;
            color: #666;
            font-style: italic;
        }
        .back-link {
            text-align: center;
            margin: 20px 0;
        }
        .back-link a {
            color: #1a73e8;
            text-decoration: none;
            font-weight: 600;
        }
        .back-link a:hover {
            text-decoration: underline;
        }
        .stats {
            background: white;
            border-radius: 12px;
            padding: 20px;
            margin-bottom: 24px;
            box-shadow: 0 2px 8px rgba(0,0,0,0.1);
        }
        .stats-center {
            display: flex;
            justify-content: center;
            align-items: center;
            gap: 40px;
            flex-wrap: wrap;
        }
        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 16px;
        }
        .stat-item {
            text-align: center;
        }
        .stat-number {
            font-size: 24px;
            font-weight: bold;
            color: #1a73e8;
        }
        .stat-label {
            font-size: 14px;
            color: #666;
            margin-top: 4px;
        }
        .archive-toggle {
            display: flex;
            align-items: center;
            gap: 8px;
            background: #f5f5f5;
            padding: 8px 12px;
            border-radius: 6px;
        }
        .archive-toggle input[type="checkbox"] {
            cursor: pointer;
        }
        .archive-toggle label {
            cursor: pointer;
            font-size: 14px;
            color: #666;
        }

        .file-item {
            position: relative;
        }
        
        .delete-btn {
            position: absolute;
            top: 8px;
            right: 8px;
            background: white;
            color: #dc3545;
            border: 2px solid #dc3545;
            border-radius: 50%;
            width: 30px;
            height: 30px;
            font-size: 16px;
            cursor: pointer;
            display: flex;
            align-items: center;
            justify-content: center;
            opacity: 0.9;
            transition: opacity 0.2s;
            z-index: 10;
            pointer-events: auto;
        }
        .delete-btn:hover {
            opacity: 1;
            background: #dc3545;
            color: white;
        }
        
        .move-btn {
            position: absolute;
            top: 8px;
            right: 45px;
            background: white;
            color: #28a745;
            border: 2px solid #28a745;
            border-radius: 50%;
            width: 30px;
            height: 30px;
            font-size: 16px;
            cursor: pointer;
            display: flex;
            align-items: center;
            justify-content: center;
            opacity: 0.9;
            transition: opacity 0.2s;
            z-index: 10;
            pointer-events: auto;
        }
        .move-btn:hover {
            opacity: 1;
            background: #28a745;
            color: white;
        }
        
        /* Button styles for listallp.php */
        .view-btn {
            position: absolute;
            top: 8px;
            right: 82px;
            background: white;
            color: #1a73e8;
            border: 2px solid #1a73e8;
            border-radius: 50%;
            width: 30px;
            height: 30px;
            font-size: 16px;
            cursor: pointer;
            display: flex;
            align-items: center;
            justify-content: center;
            opacity: 0.9;
            transition: opacity 0.2s;
            z-index: 10;
            pointer-events: auto;
        }
        .view-btn:hover {
            opacity: 1;
            background: #1a73e8;
            color: white;
        }
        /* Lightbox Modal Styles */
        .lightbox {
            display: none;
            position: fixed;
            z-index: 1000;
            left: 0;
            top: 0;
            width: 100%;
            height: 100%;
            background-color: rgba(0, 0, 0, 0.9);
            backdrop-filter: blur(5px);
        }
        
        .lightbox-content {
            position: relative;
            margin: auto;
            padding: 20px;
            width: 90%;
            height: 90%;
            display: flex;
            align-items: center;
            justify-content: center;
        }
        
        .lightbox-image {
            max-width: 100%;
            max-height: 100%;
            object-fit: contain;
            border-radius: 8px;
            box-shadow: 0 4px 20px rgba(0, 0, 0, 0.5);
        }
        
        .lightbox-close {
            position: absolute;
            top: 20px;
            right: 35px;
            color: #f1f1f1;
            font-size: 40px;
            font-weight: bold;
            cursor: pointer;
            z-index: 1001;
            background: rgba(0, 0, 0, 0.5);
            border-radius: 50%;
            width: 50px;
            height: 50px;
            display: flex;
            align-items: center;
            justify-content: center;
            transition: background-color 0.2s;
        }
        
        .lightbox-close:hover {
            background: rgba(0, 0, 0, 0.8);
        }
        
        .lightbox-info {
            position: absolute;
            bottom: 20px;
            left: 20px;
            right: 20px;
            background: rgba(0, 0, 0, 0.8);
            color: white;
            padding: 15px;
            border-radius: 8px;
            font-size: 14px;
        }
        
        .lightbox-filename {
            font-weight: bold;
            margin-bottom: 5px;
        }
        
        .lightbox-meta {
            display: flex;
            justify-content: space-between;
            flex-wrap: wrap;
            gap: 10px;
        }
        
        @media (max-width: 768px) {
            .file-grid {
                grid-template-columns: 1fr;
            }
            .stats-grid {
                grid-template-columns: 1fr;
            }
            .lightbox-content {
                width: 95%;
                height: 95%;
                padding: 10px;
            }
            .lightbox-close {
                top: 10px;
                right: 20px;
                font-size: 30px;
                width: 40px;
                height: 40px;
            }
            .lightbox-info {
                bottom: 10px;
                left: 10px;
                right: 10px;
                padding: 10px;
                font-size: 12px;
            }
            .lightbox-meta {
                flex-direction: column;
                gap: 5px;
            }
        }
        
        
        /* Styling för raderade bilder */
        .deleted-file {
            opacity: 0.7;
            position: relative;
        }
        .restore-btn {
            position: absolute;
            top: 50%;
            left: 50%;
            transform: translate(-50%, -50%);
            background: rgba(220, 53, 69, 0.9);
            color: white;
            padding: 6px 12px;
            border-radius: 6px;
            font-size: 11px;
            font-weight: bold;
            z-index: 5;
            cursor: pointer;
            border: none;
            transition: all 0.2s ease;
            box-shadow: 0 2px 4px rgba(0,0,0,0.2);
        }
        .restore-btn:hover {
            background: rgba(220, 53, 69, 1);
            transform: translate(-50%, -50%) scale(1.05);
            box-shadow: 0 4px 8px rgba(0,0,0,0.3);
        }
        .deleted-file .file-image {
            filter: grayscale(50%);
        }
        
        /* Modal styles for description popup */
        .modal {
            position: fixed;
            z-index: 1000;
            left: 0;
            top: 0;
            width: 100%;
            height: 100%;
            background-color: rgba(0, 0, 0, 0.5);
            display: flex;
            align-items: center;
            justify-content: center;
        }
        
        .modal-content {
            background-color: white;
            border-radius: 12px;
            width: 90%;
            max-width: 500px;
            max-height: 90vh;
            overflow-y: auto;
            box-shadow: 0 4px 20px rgba(0, 0, 0, 0.3);
        }
        
        .modal-header {
            padding: 20px 20px 0 20px;
            display: flex;
            justify-content: space-between;
            align-items: center;
            border-bottom: 1px solid #eee;
            padding-bottom: 15px;
        }
        
        .modal-header h3 {
            margin: 0;
            color: #1a73e8;
        }
        
        .close {
            color: #aaa;
            font-size: 28px;
            font-weight: bold;
            cursor: pointer;
            line-height: 1;
        }
        
        .close:hover {
            color: #000;
        }
        
        .modal-body {
            padding: 20px;
        }
        
        .modal-footer {
            padding: 0 20px 20px 20px;
            display: flex;
            gap: 12px;
            justify-content: flex-end;
        }
        
        .form-group {
            margin-bottom: 20px;
        }
        
        .form-group label {
            display: block;
            font-weight: 600;
            margin-bottom: 8px;
            color: #333;
        }
        
        .form-group textarea {
            width: 100%;
            padding: 10px;
            border: 1px solid #ccc;
            border-radius: 8px;
            font-size: 16px;
            box-sizing: border-box;
            resize: vertical;
            min-height: 80px;
        }
        
        .btn-primary {
            background: #1a73e8;
            color: white;
            border: none;
            padding: 12px 20px;
            border-radius: 8px;
            font-size: 16px;
            cursor: pointer;
            font-weight: 500;
        }
        
        .btn-primary:hover {
            background: #1557b0;
        }
        
        .btn-secondary {
            background: #6c757d;
            color: white;
            border: none;
            padding: 12px 20px;
            border-radius: 8px;
            font-size: 16px;
            cursor: pointer;
            font-weight: 500;
        }
        
        .btn-secondary:hover {
            background: #5a6268;
        }
    </style>
</head>
<body>
    <div class="header">
        <a href="../../index.html">
            <img src="../../data/logo.png" alt="ByggArkiv Snickare Stefan">
        </a>
        <div class="back-link">
            <a href="../admin.php">← Tillbaka till admin</a>
        </div>
    </div>
    
    <div class="container">
        <div class="stats">
            <div class="stats-center">
                <div class="stat-item">
                    <div class="stat-number"><?= count($projects) ?></div>
                    <div class="stat-label">Projekt</div>
                </div>
                <div class="stat-item">
                    <?php 
                    $totalActiveImages = 0;
                    $totalArchivedImages = 0;
                    $totalActiveVideos = 0;
                    $totalArchivedVideos = 0;
                    
                    // Räkna bilder och videor direkt från $allProjects
                    foreach ($allProjects as $project) {
                        $projectId = $project['id'];
                        $projectStatus = $project['status'] ?? 'active';
                        
                        // Hoppa över om projektet inte finns i $projects
                        if (!isset($projects[$projectId])) {
                            continue;
                        }
                        
                        $files = $projects[$projectId];
                        
                        // Räkna bilder och videor separat
                        foreach ($files as $file) {
                            $fileType = $file['file_type'] ?? 'image';
                            if ($projectStatus === 'archived') {
                                if ($fileType === 'video') {
                                    $totalArchivedVideos++;
                                } else {
                                    $totalArchivedImages++;
                                }
                            } else {
                                if ($fileType === 'video') {
                                    $totalActiveVideos++;
                                } else {
                                    $totalActiveImages++;
                                }
                            }
                        }
                    }
                    ?>
                    <div class="stat-number" id="total-images-count"><?= $showArchived ? $totalActiveImages + $totalArchivedImages : $totalActiveImages ?></div>
                    <div class="stat-label" id="total-images-label"><?= $showArchived ? "Bilder totalt ({$totalActiveImages} aktiva, {$totalArchivedImages} arkiverade)" : 'Bilder totalt' ?></div>
                </div>
                <div class="stat-item">
                    <div class="stat-number" id="total-videos-count"><?= $showArchived ? $totalActiveVideos + $totalArchivedVideos : $totalActiveVideos ?></div>
                    <div class="stat-label" id="total-videos-label"><?= $showArchived ? "Videor totalt ({$totalActiveVideos} aktiva, {$totalArchivedVideos} arkiverade)" : 'Videor totalt' ?></div>
                </div>
                <div class="archive-toggle">
                    <input type="checkbox" id="showArchived" onchange="toggleArchivedView()" <?= $showArchived ? 'checked' : '' ?>>
                    <label for="showArchived">Visa även arkiverade</label>
                </div>
            </div>
        </div>
        

        
        <?php if (empty($projects)): ?>
            <div class="project-section">
                <div class="empty-project">
                    <h2>Inga uppladdningar hittades</h2>
                    <p>Det finns inga uppladdade bilder ännu.</p>
                </div>
            </div>
        <?php else: ?>
            <?php foreach ($projects as $project_id => $files): ?>
                <?php 
                // Använd bara vanliga filer (arkiverade projekt hanteras separat)
                $allFiles = $files;
                $activeFileCount = count($files);
                $deletedFileCount = 0;
                $totalFileCount = $activeFileCount;
                ?>
                <?php 
                // Kontrollera om projektet är arkiverat - använd array
                $isArchived = in_array($project_id, $archivedProjectIds);
                
                
                ?>
                <div class="project-section<?= $isArchived ? ' archived' : '' ?><?= empty($allFiles) ? ' empty' : '' ?>" id="project-<?= htmlspecialchars($project_id) ?>">
                    <div class="project-header<?= $isArchived ? ' archived' : '' ?>">
                        <?php if (isset($projectNames[$project_id])): ?>
                            <div class="project-name">
                                <?php
                                $displayName = htmlspecialchars($projectNames[$project_id]);
                                if (str_starts_with($displayName, $project_id . ' – ')) {
                                    $displayName = substr($displayName, strlen($project_id . ' – '));
                                }
                                echo $displayName;
                                if ($isArchived) {
                                    echo ' <span style="font-weight: normal; opacity: 0.9;">(Arkiverat)</span>';
                                }
                                ?>
                            </div>
                            <div class="project-info">
                                Projekt: <?= htmlspecialchars($project_id) ?>
                                <span>(<?= $activeFileCount ?> bilder<?= $deletedFileCount > 0 ? ', ' . $deletedFileCount . ' raderade' : '' ?>)</span>
                            </div>
                        <?php else: ?>
                            <div class="project-name">
                                Projekt: <?= htmlspecialchars($project_id) ?><?= $isArchived ? ' <span style="font-weight: normal; opacity: 0.9;">(Arkiverat)</span>' : '' ?>
                            </div>
                            <div class="project-info">
                                <span>(<?= $activeFileCount ?> bilder<?= $deletedFileCount > 0 ? ', ' . $deletedFileCount . ' raderade' : '' ?>)</span>
                            </div>
                        <?php endif; ?>
                    </div>
                    <div class="project-content">
                        <?php if (empty($allFiles)): ?>
                            <div class="empty-state" style="text-align: center; font-size: 14px; color: #666; padding: 0;">
                                Inga filer uppladdade än
                            </div>
                        <?php else: ?>
                            <div class="file-grid">
                                <?php foreach ($allFiles as $file): ?>
                                    <?php $isDeleted = isset($file['is_deleted']) && $file['is_deleted']; ?>
                                    <div class="file-item <?= $isDeleted ? 'deleted-file' : '' ?>" 
                                         data-image-path="<?= htmlspecialchars($file['image_path']) ?>"
                                         data-json-path="<?= htmlspecialchars(str_replace(['.jpg', '.jpeg', '.png', '.webp', '.heic', '.heif', '.mp4', '.webm', '.mov', '.avi'], '.json', $file['image_path'])) ?>"
                                         data-project-id="<?= htmlspecialchars($isDeleted ? ($file['original_project_id'] ?? $project_id) : $project_id) ?>"
                                         data-is-deleted="<?= $isDeleted ? 'true' : 'false' ?>"
                                         <?php 
                                         $displayDate = date('Y-m-d H:i', strtotime($file['stored_at']));
                                         $filename = basename($file['image_path']);
                                         $size = number_format($file['bytes'] / 1024 / 1024, 1) . ' MB';
                                         $note = !empty($file['note']) ? $file['note'] : '';
                                         $isVideo = ($file['file_type'] ?? 'image') === 'video';
                                         $imageNumber = isset($file['image_number']) && is_numeric($file['image_number']) ? (int)$file['image_number'] : null;
                                         $imageNumberText = $imageNumber !== null ? ($isVideo ? 'Video #' . $imageNumber : 'Bild #' . $imageNumber) : '';
                                         ?>
                                         data-image-url="<?= htmlspecialchars($file['image_url']) ?>"
                                         data-full-image-url="<?= htmlspecialchars($file['image_url']) ?>"
                                         data-filename="<?= htmlspecialchars($filename) ?>"
                                         data-date="<?= htmlspecialchars($displayDate) ?>"
                                         data-size="<?= htmlspecialchars($size) ?>"
                                         data-note="<?= htmlspecialchars($note) ?>"
                                         data-image-number="<?= htmlspecialchars($imageNumberText) ?>"
                                         >
                                        <button class="view-btn" title="Visa fil i full storlek"><img src="../icons/view.svg" width="16" height="16" alt="Visa"></button>
                                        <?php if ($isDeleted): ?>
                                            <button class="restore-btn" onclick="restoreFile(this)" title="Återställ denna bild till sitt ursprungliga projekt">ÅTERSTÄLL RADERAD</button>
                                        <?php else: ?>
                                            <button class="move-btn" onclick="moveFile(this)" title="Flytta till annat projekt"><img src="../icons/move.svg" width="16" height="16" alt="Flytta"></button>
                                            <button class="delete-btn" onclick="deleteFile(this)" title="Ta bort fil"><img src="../icons/delete.svg" width="16" height="16" alt="Ta bort"></button>
                                            <button class="edit-btn" onclick="addDescription(this)" title="<?= empty($file['note']) ? 'Ändra beskrivning' : 'Redigera beskrivning' ?>"><img src="../icons/edit.svg" width="16" height="16" alt="Redigera"></button>
                                        <?php endif; ?>
                                        <?php 
                                        if ($isVideo): ?>
                                            <video class="file-image" 
                                                   controls 
                                                   preload="metadata">
                                                <source src="<?= htmlspecialchars($file['image_url']) ?>" type="<?= htmlspecialchars($file['mime'] ?? 'video/mp4') ?>">
                                                Din webbläsare stöder inte videouppspelning.
                                            </video>
                                        <?php else: ?>
                                            <img src="<?= htmlspecialchars($file['thumb_url'] ?? $file['image_url']) ?>" 
                                                 alt="" 
                                                 class="file-image"
                                                 loading="lazy"
                                                 onerror="this.style.display='none'">
                                        <?php endif; ?>
                                        <div class="file-info">
                                            <?php if ($imageNumberText): ?>
                                                <div class="file-image-number" style="font-weight: bold; color: #1976d2; margin-bottom: 4px;">
                                                    <?= htmlspecialchars($imageNumberText) ?>
                                                </div>
                                            <?php endif; ?>
                                            <?php 
                                            $uid = $file['user_id'] ?? '';
                                            $uname = getUserFullName($uid);
                                            $displayUser = $uname !== '' ? htmlspecialchars($uname) : 'Ingen användare';
                                            $displayDate = date('Y-m-d H:i', strtotime($file['stored_at']));
                                            $origSizeText = number_format(($file['bytes'] ?? 0) / 1024 / 1024, 1) . 'M';
                                            $thumbSizeText = '';
                                            if (!empty($file['thumb_path']) && file_exists($file['thumb_path'])) {
                                                $thumbBytes = @filesize($file['thumb_path']);
                                                if ($thumbBytes !== false) {
                                                    $thumbSizeText = ' (' . number_format($thumbBytes / 1024, 1) . 'k)';
                                                }
                                            }
                                            $filename = basename($file['image_path']);
                                            ?>
                                            <div class="file-meta">
                                                <div class="file-meta-row">
                                                    <span class="file-date"><?= $displayDate ?>    <?= $origSizeText ?><?= $thumbSizeText ?></span>
                                                    <span class="file-user-id"><?= $displayUser ?></span>
                                                </div>
                                            </div>
                                            <a href="<?= htmlspecialchars($file['image_url']) ?>" class="file-filename" download><?= htmlspecialchars($filename) ?></a>

                                            <?php if (!empty($file['note'])): ?>
                                                <div class="file-note">
                                                    <?= htmlspecialchars($file['note']) ?>
                                                </div>
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                <?php endforeach; ?>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
            <?php endforeach; ?>
        <?php endif; ?>
    </div>

    <!-- Lightbox Modal -->
    <div id="lightbox" class="lightbox">
        <div class="lightbox-content">
            <span class="lightbox-close" onclick="closeListAllLightbox()">&times;</span>
            <img id="lightbox-image" class="lightbox-image" src="" alt="" style="display: none;">
            <video id="lightbox-video" class="lightbox-image" controls style="display: none;">
                Din webbläsare stöder inte videouppspelning.
            </video>
            <div class="lightbox-info">
                <div id="lightbox-image-number" class="lightbox-filename" style="font-weight: bold; color: #fff; margin-bottom: 5px;"></div>
                <div id="lightbox-filename" class="lightbox-filename"></div>
                <div class="lightbox-meta">
                    <span id="lightbox-date"></span>
                    <span id="lightbox-size"></span>
                </div>
                <div id="lightbox-note" class="lightbox-note"></div>
            </div>
        </div>
    </div>

    <!-- Description Modal -->
    <div id="descriptionModal" class="modal" style="display: none;">
        <div class="modal-content">
            <div class="modal-header">
                <h3>Ändra beskrivning</h3>
                <span class="close" onclick="closeDescriptionModal()">&times;</span>
            </div>
            <div class="modal-body">
                <div class="form-group">
                    <textarea id="descriptionText" rows="4" placeholder="Skriv en beskrivning för denna fil..."></textarea>
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn-secondary" onclick="closeDescriptionModal()">Avbryt</button>
                <button type="button" class="btn-primary" onclick="saveDescription()">Spara</button>
            </div>
        </div>
    </div>

    <script>
        // Lightbox functions
        function openListAllLightbox(imageSrc, filename, date, size, note, imageNumber) {
            // Convert thumbnail URL to full-size image URL if needed
            // Handle both direct file paths and view_file.php URLs
            if (imageSrc && imageSrc.includes('-thumb.')) {
                if (imageSrc.includes('view_file.php')) {
                    // For view_file.php URLs, update the file parameter
                    const url = new URL(imageSrc, window.location.origin);
                    const currentFile = url.searchParams.get('file');
                    if (currentFile && currentFile.includes('-thumb.')) {
                        const fullSizeFile = currentFile.replace('-thumb.', '.');
                        url.searchParams.set('file', fullSizeFile);
                        imageSrc = url.pathname + '?' + url.searchParams.toString();
                    }
                } else {
                    // For direct file paths, simple replace
                    imageSrc = imageSrc.replace('-thumb.', '.');
                }
            }
            const lightbox = document.getElementById('lightbox');
            const lightboxImage = document.getElementById('lightbox-image');
            const lightboxVideo = document.getElementById('lightbox-video');
            const lightboxImageNumber = document.getElementById('lightbox-image-number');
            const lightboxFilename = document.getElementById('lightbox-filename');
            const lightboxDate = document.getElementById('lightbox-date');
            const lightboxSize = document.getElementById('lightbox-size');
            const lightboxNote = document.getElementById('lightbox-note');
            
            // Kontrollera om det är en video baserat på filnamn
            const isVideo = /\.(mp4|webm|mov|avi)$/i.test(filename);
            
            if (isVideo) {
                // Visa video, dölj bild
                lightboxImage.style.display = 'none';
                lightboxVideo.style.display = 'block';
                lightboxVideo.src = imageSrc;
                lightboxVideo.load(); // Ladda videon
            } else {
                // Visa bild, dölj video
                lightboxVideo.style.display = 'none';
                lightboxImage.style.display = 'block';
                lightboxImage.src = imageSrc;
            }
            
            // Visa bildnummer om det finns
            if (imageNumber && imageNumber.trim() !== '') {
                lightboxImageNumber.textContent = imageNumber;
                lightboxImageNumber.style.display = 'block';
            } else {
                lightboxImageNumber.style.display = 'none';
            }
            
            lightboxFilename.textContent = filename;
            lightboxDate.textContent = date;
            lightboxSize.textContent = size;
            lightboxNote.textContent = note || '';
            
            lightbox.style.display = 'block';
            document.body.style.overflow = 'hidden'; // Prevent background scrolling
        }
        
        function closeListAllLightbox() {
            const lightbox = document.getElementById('lightbox');
            const lightboxVideo = document.getElementById('lightbox-video');
            if (lightbox) {
                lightbox.style.display = 'none';
                document.body.style.overflow = 'auto'; // Restore scrolling
                
                // Pausa video om den spelar
                if (lightboxVideo) {
                    lightboxVideo.pause();
                    lightboxVideo.currentTime = 0;
                }
            }
        }
        
        // Close lightbox when clicking outside the image
        document.addEventListener('click', function(event) {
            const lightbox = document.getElementById('lightbox');
            if (event.target === lightbox) {
                closeListAllLightbox();
            }
        });
        
        // Close lightbox with Escape key
        document.addEventListener('keydown', function(event) {
            if (event.key === 'Escape') {
                closeListAllLightbox();
            }
        });

        // Helper function to open lightbox from a file item
        function openLightboxFromItem(item) {
            if (!item) return;
            let imageUrl = item.getAttribute('data-full-image-url') || item.getAttribute('data-image-url');
            const filename = item.getAttribute('data-filename') || '';
            const date = item.getAttribute('data-date') || '';
            const size = item.getAttribute('data-size') || '';
            const note = item.getAttribute('data-note') || ''; // Always read from data-note attribute
            const imageNumber = item.getAttribute('data-image-number') || '';
            if (!imageUrl) {
                const imgEl = item.querySelector('img.file-image');
                if (imgEl) imageUrl = imgEl.getAttribute('src');
            }
            openListAllLightbox(imageUrl, filename, date, size, note, imageNumber);
        }

        // Delegated handler for view buttons
        document.addEventListener('click', function(e) {
            const btn = e.target.closest('.view-btn');
            if (btn) {
                const item = btn.closest('.file-item');
                openLightboxFromItem(item);
                return;
            }
            
            // Also handle clicks directly on images/videos
            const img = e.target.closest('img.file-image, video.file-image');
            if (img) {
                const item = img.closest('.file-item');
                // Don't open if clicking on buttons
                if (!e.target.closest('.view-btn, .move-btn, .delete-btn, .edit-btn, .restore-btn')) {
                    openLightboxFromItem(item);
                }
            }
        });


        // Funktion för att toggla visning av arkiverade projekt via sidladdning
        function toggleArchivedView() {
            const checkbox = document.getElementById('showArchived');
            const currentUrl = new URL(window.location);
            
            if (checkbox.checked) {
                currentUrl.searchParams.set('show_archived', '1');
            } else {
                currentUrl.searchParams.delete('show_archived');
            }
            
            // Ladda om sidan med den nya parametern
            window.location.href = currentUrl.toString();
        }

        

        async function restoreFile(button) {
            const fileItem = button.closest('.file-item');
            const imagePath = fileItem.dataset.imagePath;
            const jsonPath = fileItem.dataset.jsonPath;
            const originalProjectId = fileItem.dataset.projectId;
            
            // Hitta projektnamnet för bättre användarupplevelse
            const projectSection = document.getElementById(`project-${originalProjectId}`);
            let projectDisplayName = originalProjectId;
            
            if (projectSection) {
                const projectHeader = projectSection.querySelector('.project-header');
                if (projectHeader) {
                    // Hämta projektnamnet från första div:en i headern
                    const nameDiv = projectHeader.querySelector('div:first-child');
                    if (nameDiv) {
                        projectDisplayName = nameDiv.textContent.trim();
                    }
                }
            }
            
            // Bekräfta återställning med projektnamn
            if (!confirm(`Är du säker på att du vill återställa denna bild till projekt "${projectDisplayName}"?`)) {
                return;
            }
            
            const formData = new FormData();
            formData.append('imagePath', imagePath);
            formData.append('jsonPath', jsonPath);
            formData.append('targetProjectId', originalProjectId);
            
            try {
                const response = await fetch('restore_file.php', {
                    method: 'POST',
                    body: formData
                });
                
                const result = await response.json();
                
                if (result.ok) {
                    // Använd projektnamnet från server-responsen om det finns, annars använd det vi hämtade från DOM
                    const finalProjectName = result.projectName || projectDisplayName;
                    alert(`Bild återställd till projekt "${finalProjectName}"`);
                    
                    // Ladda om sidan efter att användaren stängt popupen
                    window.location.reload();
                } else {
                    alert('Fel vid återställning: ' + result.error);
                }
            } catch (error) {
                alert('Ett fel uppstod: ' + error.message);
            }
        }

        async function deleteFile(button) {
            if (!confirm('Är du säker på att du vill ta bort denna bild?\n\nBilden kommer att tas bort permanent och kan inte återställas.')) {
                return;
            }

            const fileItem = button.closest('.file-item');
            const imagePath = fileItem.dataset.imagePath;
            const jsonPath = fileItem.dataset.jsonPath;

            const formData = new FormData();
            formData.append('imagePath', imagePath);
            formData.append('jsonPath', jsonPath);

            try {
                const response = await fetch('delete_file.php', {
                    method: 'POST',
                    body: formData
                });

                const result = await response.json();

                if (result.ok) {
                    // Remove the file item from the display
                    fileItem.remove();
                    
                    // Update the count in the project header
                    const projectSection = fileItem.closest('.project-section');
                    if (projectSection) {
                        const projectHeader = projectSection.querySelector('.project-header');
                        if (projectHeader) {
                            const countSpan = projectHeader.querySelector('span');
                            if (countSpan) {
                                const match = countSpan.textContent.match(/\((\d+) bilder\)/);
                                if (match) {
                                    const currentCount = parseInt(match[1]);
                                    countSpan.textContent = `(${currentCount - 1} bilder)`;
                                }
                            }
                        }
                        
                        // If no images left in project, show empty message
                        const fileGrid = projectSection.querySelector('.file-grid');
                        if (fileGrid && fileGrid.children.length === 0) {
                            projectSection.innerHTML = '<div class="empty-project"><h2>Inga bilder kvar</h2><p>Alla bilder i detta projekt har tagits bort.</p></div>';
                        }
                    }
                    
                    // Ladda om sidan för att uppdatera räknare
                    window.location.reload();
                } else {
                    alert('Fel: ' + result.error);
                }
            } catch (error) {
                alert('Ett fel uppstod: ' + error.message);
            }
        }

        let currentEditButton = null;
        let currentFileItem = null;

        function addDescription(button) {
            currentEditButton = button;
            currentFileItem = button.closest('.file-item');
            if (!currentFileItem) {
                console.error('File item not found');
                return;
            }
            const imagePath = currentFileItem.dataset.imagePath;
            const jsonPath = currentFileItem.dataset.jsonPath;
            if (!imagePath || !jsonPath) {
                console.error('Missing file paths:', { imagePath, jsonPath });
                alert('Fel: Saknade filvägar');
                return;
            }
            
            // Get existing note if any
            const existingNote = currentFileItem.dataset.note || '';
            
            // Fill in existing note and show modal
            document.getElementById('descriptionText').value = existingNote;
            document.getElementById('descriptionModal').style.display = 'flex';
            document.getElementById('descriptionText').focus();
        }

        function closeDescriptionModal() {
            document.getElementById('descriptionModal').style.display = 'none';
            currentEditButton = null;
            currentFileItem = null;
        }

        function saveDescription() {
            if (!currentFileItem || !currentEditButton) {
                return;
            }
            
            const description = document.getElementById('descriptionText').value.trim();
            if (!description) {
                alert('Ange en beskrivning');
                return;
            }
            
            const imagePath = currentFileItem.dataset.imagePath;
            const jsonPath = currentFileItem.dataset.jsonPath;
            
            const formData = new FormData();
            formData.append('imagePath', imagePath);
            formData.append('jsonPath', jsonPath);
            formData.append('description', description);
            
            // Hämta user_id från URL eller cookie
            const urlParams = new URLSearchParams(window.location.search);
            const userId = urlParams.get('user_id') || (function() {
                const cookies = document.cookie.split(';');
                for (let cookie of cookies) {
                    const [name, value] = cookie.trim().split('=');
                    if (name === 'user_id') {
                        return decodeURIComponent(value);
                    }
                }
                return '';
            })();
            if (userId) {
                formData.append('user_id', userId);
            }
            
            fetch('../../api/files/add_description.php', { method: 'POST', body: formData })
                .then(response => response.json())
                .then(result => {
                    if (result.ok) {
                        // Update existing note or create new one
                        let noteDiv = currentFileItem.querySelector('.file-note');
                        if (noteDiv) {
                            // Update existing note
                            noteDiv.textContent = description;
                        } else {
                            // Create new note
                            noteDiv = document.createElement('div');
                            noteDiv.className = 'file-note';
                            noteDiv.style.cssText = 'margin-top: 6px; padding: 6px 8px 2px 8px; background: #e3f2fd; border-radius: 4px; font-style: normal; font-weight: normal; color: #1976d2; margin-bottom: 2px; word-wrap: break-word;';
                            noteDiv.textContent = description;
                            const fileInfo = currentFileItem.querySelector('.file-info');
                            if (fileInfo) {
                                const filenameLink = fileInfo.querySelector('.file-filename');
                                if (filenameLink && filenameLink.parentNode === fileInfo) {
                                    fileInfo.insertBefore(noteDiv, filenameLink.nextSibling);
                                } else {
                                    fileInfo.appendChild(noteDiv);
                                }
                            } else {
                                currentEditButton.parentNode.appendChild(noteDiv);
                            }
                        }
                        
                        // Update the data attribute
                        currentFileItem.dataset.note = description;
                        
                        // Update button title
                        currentEditButton.title = 'Redigera beskrivning';
                        
                        closeDescriptionModal();
                    } else {
                        alert('Fel: ' + result.error);
                    }
                })
                .catch(error => {
                    console.error('Add description error:', error);
                    alert('Ett fel uppstod: ' + error.message);
                });
        }

        // Wait for DOM to be ready
        document.addEventListener('DOMContentLoaded', function() {
            // Close modal with Escape key
            document.addEventListener('keydown', function(e) {
                if (e.key === 'Escape' && document.getElementById('descriptionModal').style.display === 'flex') {
                    closeDescriptionModal();
                }
            });
        });
        async function moveFile(button) {
            const fileItem = button.closest('.file-item');
            const imagePath = fileItem.dataset.imagePath;
            const jsonPath = fileItem.dataset.jsonPath;
            const currentProjectId = fileItem.dataset.projectId;
            
            // Hämta alla tillgängliga projekt från projektlistan
            let projectSections = document.querySelectorAll('.project-section');
            
            // Om inga hittades, försök med en annan selektor
            if (projectSections.length === 0) {
                projectSections = document.querySelectorAll('[id^="project-"]');
            }
            
            // Om fortfarande inga hittades, försök hitta alla div-element som innehåller projekt
            if (projectSections.length === 0) {
                const allDivs = document.querySelectorAll('div');
                projectSections = Array.from(allDivs).filter(div => 
                    div.textContent.includes('Projekt:') && div.querySelector('.file-grid')
                );
            }
            
            // Om fortfarande inga hittades, försök hitta alla element som innehåller projekt
            if (projectSections.length === 0) {
                const allElements = document.querySelectorAll('*');
                projectSections = Array.from(allElements).filter(element => 
                    element.textContent && 
                    element.textContent.includes('Projekt:') && 
                    element.querySelector('.file-grid')
                );
            }
            
            // Hämta alla projekt från API (inklusive arkiverade för flytt)
            let allProjects = [];
            let activeProjects = [];
            let archivedProjects = [];
            
            try {
                // Använd API-endpoint istället för direkt JSON-fil (pga .htaccess-skydd)
                const response = await fetch(`../../api/projects.php?action=list&t=${Date.now()}`);
                if (!response.ok) {
                    throw new Error(`HTTP error! status: ${response.status}`);
                }
                const result = await response.json();
                const projectsData = result.success ? result.projects : [];
                // Exkludera bara nuvarande projekt, inkludera både aktiva och arkiverade
                const filteredProjects = projectsData.filter(project => project.id !== currentProjectId);
                
                // Separera aktiva och arkiverade projekt
                activeProjects = filteredProjects.filter(project => (project.status || 'active') === 'active');
                archivedProjects = filteredProjects.filter(project => (project.status || 'active') === 'archived');
                
                // Sortera båda grupperna alfabetiskt
                activeProjects.sort((a, b) => a.name.localeCompare(b.name, 'sv'));
                archivedProjects.sort((a, b) => a.name.localeCompare(b.name, 'sv'));
                
                // Kombinera: aktiva först, sedan arkiverade
                allProjects = [...activeProjects, ...archivedProjects];
            } catch (error) {
                console.error('Kunde inte hämta projekt:', error);
                alert('Kunde inte hämta projektlista. Försök igen.');
                return;
            }
            
            if (allProjects.length === 0) {
                alert('Inga andra projekt tillgängliga att flytta till');
                return;
            }
            
            // Skapa en dropdown för att välja målprojekt
            const projectSelect = document.createElement('select');
            projectSelect.style.cssText = 'padding: 8px; margin: 10px 0; width: 100%; border: 1px solid #ccc; border-radius: 4px;';
            
            const defaultOption = document.createElement('option');
            defaultOption.value = '';
            defaultOption.textContent = 'Välj målprojekt...';
            projectSelect.appendChild(defaultOption);
            
            // Lägg till aktiva projekt
            activeProjects.forEach(project => {
                const option = document.createElement('option');
                option.value = project.id;
                option.textContent = project.name || project.id || 'Okänt projekt';
                projectSelect.appendChild(option);
            });
            
            // Lägg till avgränsare om det finns både aktiva och arkiverade projekt
            if (activeProjects.length > 0 && archivedProjects.length > 0) {
                const separator = document.createElement('option');
                separator.disabled = true;
                separator.textContent = '───────────────';
                separator.style.cssText = 'color: #999; font-style: italic;';
                projectSelect.appendChild(separator);
            }
            
            // Lägg till arkiverade projekt
            if (archivedProjects.length > 0) {
                archivedProjects.forEach(project => {
                    const option = document.createElement('option');
                    option.value = project.id;
                    option.textContent = project.name || project.id || 'Okänt projekt';
                    option.style.cssText = 'color: #666; font-style: italic;';
                    projectSelect.appendChild(option);
                });
            } else {
                // Visa "Inga arkiverade projekt" om det inte finns några
                const noArchivedOption = document.createElement('option');
                noArchivedOption.disabled = true;
                noArchivedOption.textContent = 'Inga arkiverade projekt';
                noArchivedOption.style.cssText = 'color: #999; font-style: italic;';
                projectSelect.appendChild(noArchivedOption);
            }
            
            // Skapa en dialog
            const dialog = document.createElement('div');
            dialog.style.cssText = `
                position: fixed;
                top: 0;
                left: 0;
                width: 100%;
                height: 100%;
                background: rgba(0,0,0,0.5);
                display: flex;
                align-items: center;
                justify-content: center;
                z-index: 1000;
            `;
            
            const dialogContent = document.createElement('div');
            dialogContent.style.cssText = `
                background: white;
                padding: 20px;
                border-radius: 8px;
                max-width: 400px;
                width: 90%;
                text-align: center;
            `;
            
            dialogContent.innerHTML = `
                <h3>Flytta fil till annat projekt</h3>
                <p>Välj målprojekt för denna fil:</p>
            `;
            
            dialogContent.appendChild(projectSelect);
            
            const buttonContainer = document.createElement('div');
            buttonContainer.style.cssText = 'margin-top: 20px; display: flex; gap: 10px; justify-content: flex-end;';
            
            const cancelBtn = document.createElement('button');
            cancelBtn.textContent = 'Avbryt';
            cancelBtn.className = 'btn-cancel';
            cancelBtn.onclick = () => document.body.removeChild(dialog);
            
            const moveBtn = document.createElement('button');
            moveBtn.textContent = 'Flytta';
            moveBtn.style.cssText = 'padding: 8px 16px; background: #28a745; color: white; border: none; border-radius: 4px; cursor: pointer;';
            moveBtn.onclick = async () => {
                const selectedProject = projectSelect.value;
                if (!selectedProject) {
                    alert('Välj ett målprojekt');
                    return;
                }
                
                const projectName = projectSelect.options[projectSelect.selectedIndex].textContent;
                
                // Fortsätt direkt med flytten utan bekräftelse
                proceedWithMove();
                
                async function proceedWithMove() {
                    // Hämta user_id från URL-parametrar
                    const urlParams = new URLSearchParams(window.location.search);
                    const userId = urlParams.get('user_id') || '';
                    
                    const formData = new FormData();
                    formData.append('imagePath', imagePath);
                    formData.append('jsonPath', jsonPath);
                    formData.append('newProjectId', selectedProject);
                    if (userId) {
                        formData.append('user_id', userId);
                    }
                    
                    try {
                        const response = await fetch('move_file.php', {
                            method: 'POST',
                            body: formData
                        });
                        
                        if (!response.ok) {
                            throw new Error(`HTTP error! status: ${response.status}`);
                        }
                        
                        // Läs svaret som text först för att hantera eventuella parsing-fel
                        const responseText = await response.text();
                        console.log('Move response text:', responseText);
                        
                        let result;
                        try {
                            result = JSON.parse(responseText);
                        } catch (parseError) {
                            console.error('JSON parse error:', parseError, 'Response text:', responseText);
                            throw new Error('Ogiltigt svar från servern: ' + responseText.substring(0, 100));
                        }
                        
                        if (result.ok) {
                            // Stäng dialogen
                            document.body.removeChild(dialog);
                            // Uppdatera sidan för att visa den nya situationen
                            window.location.reload();
                        } else {
                            alert('Fel: ' + result.error);
                        }
                    } catch (error) {
                        alert('Ett fel uppstod: ' + error.message);
                    }
                }
            };
            
            buttonContainer.appendChild(moveBtn);
            buttonContainer.appendChild(cancelBtn);
            dialogContent.appendChild(buttonContainer);
            dialog.appendChild(dialogContent);
            document.body.appendChild(dialog);
        }
        
        // Scroll to top funktionalitet
        function scrollToTop() {
            window.scrollTo({
                top: 0,
                behavior: 'smooth'
            });
        }
        
        // Visa/dölj scroll-to-top knapp baserat på scroll-position
        window.addEventListener('scroll', function() {
            const scrollBtn = document.getElementById('scrollToTopBtn');
            if (window.pageYOffset > 300) {
                scrollBtn.style.display = 'block';
            } else {
                scrollBtn.style.display = 'none';
            }
        });
    </script>
    
    <!-- Scroll to top knapp -->
    <button id="scrollToTopBtn" onclick="scrollToTop()" class="scroll-to-top" style="display: none;" title="Scrolla till toppen">
        ↑
    </button>
</body>
</html>
