<?php
// Aktivera output buffering för att förhindra output före JSON
ob_start();

require_once __DIR__ . '/../../lib/data_functions.php';
require_once __DIR__ . '/../../config.php';
require_once __DIR__ . '/../../security.php';

// Starta session om den inte redan är startad
if (session_status() === PHP_SESSION_NONE) {
    @session_start();
}

// Rensa output buffer och sätt header
ob_clean();
header('Content-Type: application/json');

// Funktion för att logga filflytt-händelser
function logFileMoveEvent($userId, $fromProjectId, $toProjectId, $filename, $fromProjectName, $toProjectName) {
    $logFile = __DIR__ . '/../../data/file_move_events.json';
    $logDir = dirname($logFile);
    
    // Skapa data-mappen om den inte finns
    if (!is_dir($logDir)) {
        @mkdir($logDir, 0755, true);
    }
    
    // Läs befintliga händelser
    $events = [];
    if (file_exists($logFile)) {
        $content = @file_get_contents($logFile);
        if ($content !== false) {
            $decoded = json_decode($content, true);
            if (is_array($decoded)) {
                $events = $decoded;
            }
        }
    }
    
    // Lägg till ny händelse
    $event = [
        'timestamp' => date('c'),
        'user_id' => $userId,
        'from_project_id' => $fromProjectId,
        'to_project_id' => $toProjectId,
        'filename' => $filename,
        'from_project_name' => $fromProjectName,
        'to_project_name' => $toProjectName
    ];
    
    $events[] = $event;
    
    // Begränsa till senaste 1000 händelser
    if (count($events) > 1000) {
        $events = array_slice($events, -1000);
    }
    
    // Spara tillbaka
    @file_put_contents($logFile, json_encode($events, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE), LOCK_EX);
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    ob_clean();
    http_response_code(405);
    echo json_encode(['ok' => false, 'error' => 'Method not allowed']);
    exit;
}

// Hämta parametrar
$imagePath = $_POST['imagePath'] ?? '';
$jsonPath = $_POST['jsonPath'] ?? '';
$newProjectId = $_POST['newProjectId'] ?? '';

// Hämta user_id från POST, GET, cookie eller session (som admin_guard.php gör)
$userId = '';
if (isset($_POST['user_id']) && $_POST['user_id'] !== '') {
    $userId = sanitizeInput($_POST['user_id'], 'user_id');
} elseif (isset($_GET['user_id']) && $_GET['user_id'] !== '') {
    $userId = sanitizeInput($_GET['user_id'], 'user_id');
} elseif (isset($_COOKIE['user_id']) && $_COOKIE['user_id'] !== '') {
    $userId = sanitizeInput($_COOKIE['user_id'], 'user_id');
} elseif (isset($_SESSION['user_id']) && $_SESSION['user_id'] !== '') {
    $userId = sanitizeInput($_SESSION['user_id'], 'user_id');
}

if (empty($imagePath) || empty($jsonPath) || empty($newProjectId)) {
    ob_clean();
    echo json_encode(['ok' => false, 'error' => 'Alla parametrar krävs']);
    exit;
}

// Validera sökvägar
if (!file_exists($imagePath) || !file_exists($jsonPath)) {
    ob_clean();
    echo json_encode(['ok' => false, 'error' => 'En eller flera filer hittades inte']);
    exit;
}

// Hämta projektdata för att validera målprojektet (inklusive inaktiva projekt)
$projectsData = [];
$projectNames = [];
if (file_exists(__DIR__ . '/../../data/projects.json')) {
    $projectsData = json_decode(file_get_contents(__DIR__ . '/../../data/projects.json'), true) ?: [];
    foreach ($projectsData as $project) {
        $projectNames[$project['id']] = $project['name'];
    }
}

if (!isset($projectNames[$newProjectId])) {
    ob_clean();
    echo json_encode(['ok' => false, 'error' => 'Ogiltigt målprojekt']);
    exit;
}

// Hämta nuvarande projekt-ID från JSON-filen
$currentMetadata = json_decode(file_get_contents($jsonPath), true);
if (!$currentMetadata) {
    ob_clean();
    echo json_encode(['ok' => false, 'error' => 'Kunde inte läsa metadata']);
    exit;
}

$currentProjectId = $currentMetadata['project_id'] ?? '';
if ($currentProjectId === $newProjectId) {
    ob_clean();
    echo json_encode(['ok' => false, 'error' => 'Bilden är redan i det valda projektet']);
    exit;
}

// Skapa målprojektmapp om den inte finns
$uploadRoot = __DIR__ . '/../../' . rtrim(UPLOAD_ROOT, '/\\');
$targetProjectDir = $uploadRoot . '/' . $newProjectId;
if (!is_dir($targetProjectDir)) {
    if (!mkdir($targetProjectDir, 0755, true)) {
        ob_clean();
        echo json_encode(['ok' => false, 'error' => 'Kunde inte skapa målprojektmapp']);
        exit;
    }
}

// Hämta filnamn
$imageFilename = basename($imagePath);
$jsonFilename = basename($jsonPath);
// Thumb filnamn
$thumbPath = preg_replace('/\.(jpg|jpeg|png|webp|heic|heif)$/i', '', $imagePath) . '-thumb.jpg';
$thumbFilename = basename($thumbPath);

// Skapa nya sökvägar
$targetImagePath = $targetProjectDir . '/' . $imageFilename;
$targetJsonPath = $targetProjectDir . '/' . $jsonFilename;
$targetThumbPath = $targetProjectDir . '/' . $thumbFilename;

// Kontrollera om filen redan finns i målprojektet
if (file_exists($targetImagePath) || file_exists($targetJsonPath) || (file_exists($thumbPath) && file_exists($targetThumbPath))) {
    ob_clean();
    echo json_encode(['ok' => false, 'error' => 'En fil med samma namn finns redan i målprojektet']);
    exit;
}

try {
    // Flytta bildfilen
    if (!rename($imagePath, $targetImagePath)) {
        throw new Exception('Kunde inte flytta bildfilen');
    }
    
    // Flytta tumnagel om den finns
    if (file_exists($thumbPath)) {
        if (!rename($thumbPath, $targetThumbPath)) {
            // Om tumnagel inte kunde flyttas, flytta tillbaka bildfilen
            rename($targetImagePath, $imagePath);
            throw new Exception('Kunde inte flytta tumnagel');
        }
    }

    // Uppdatera metadata
    $currentMetadata['project_id'] = $newProjectId;
    $currentMetadata['project_name'] = $projectNames[$newProjectId];
    if (isset($currentMetadata['thumb_file'])) {
        // Lämna filnamnet oförändrat; endast mapp byts
    } else {
        // Om det finns en tumnagel på disk men metadata saknar fältet, fyll i det
        if (file_exists($targetThumbPath)) {
            $currentMetadata['thumb_file'] = $thumbFilename;
        }
    }
    
    // Skriv uppdaterad metadata
    if (file_put_contents($targetJsonPath, json_encode($currentMetadata, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE)) === false) {
        // Om JSON-skrivningen misslyckas, flytta tillbaka bildfilen
        rename($targetImagePath, $imagePath);
        throw new Exception('Kunde inte uppdatera metadata');
    }
    
    // Ta bort gamla JSON-filen
    unlink($jsonPath);
    
    // Logga filflytt-händelse (logga alltid, även om user_id saknas)
    // Använd stored_file (det lagrade filnamnet) om det finns, annars original, annars basename
    $filename = $currentMetadata['stored_file'] ?? ($currentMetadata['original'] ?? basename($imagePath));
    $fromProjectName = $projectNames[$currentProjectId] ?? $currentProjectId;
    $toProjectName = $projectNames[$newProjectId] ?? $newProjectId;
    // Använd tom sträng om user_id saknas
    $logUserId = $userId ?: '';
    logFileMoveEvent($logUserId, $currentProjectId, $newProjectId, $filename, $fromProjectName, $toProjectName);
    
    ob_clean();
    echo json_encode([
        'ok' => true, 
        'message' => 'Bild flyttad till projekt: ' . $projectNames[$newProjectId],
        'newProjectId' => $newProjectId,
        'newProjectName' => $projectNames[$newProjectId]
    ]);
    
} catch (Exception $e) {
    ob_clean();
    http_response_code(500);
    echo json_encode(['ok' => false, 'error' => $e->getMessage()]);
    exit;
}
