<?php
require_once __DIR__ . '/../admin_guard.php';
require_once '../../config.php';

// API endpoint för statistik
if (isset($_GET['action']) && $_GET['action'] === 'stats') {
    header('Content-Type: application/json; charset=utf-8');
    
    // Hämta statistik för uploads-mappen
    $uploadsPath = __DIR__ . '/../../uploads';
    $uploadsStats = getImageStats($uploadsPath);
    
    echo json_encode([
        'success' => true,
        'uploadsStats' => $uploadsStats
    ], JSON_UNESCAPED_UNICODE);
    exit;
}

// Funktion för att formatera bytes till läsbar storlek
function formatBytes($bytes, $precision = 2) {
    $units = array('B', 'KB', 'MB', 'GB', 'TB');
    
    for ($i = 0; $bytes > 1024 && $i < count($units) - 1; $i++) {
        $bytes /= 1024;
    }
    
    return round($bytes, $precision) . ' ' . $units[$i];
}

// Funktion för att räkna bilder, dokument, videor och storlek i en mapp
function getImageStats($path) {
    $stats = [
        'image_files' => 0,
        'image_size' => 0,
        'thumbnail_files' => 0,
        'thumbnail_size' => 0,
        'document_files' => 0,
        'document_size' => 0,
        'video_files' => 0,
        'video_size' => 0
    ];
    
    if (!is_dir($path)) {
        return $stats;
    }
    
    $iterator = new RecursiveIteratorIterator(
        new RecursiveDirectoryIterator($path, RecursiveDirectoryIterator::SKIP_DOTS)
    );
    
    $imageExtensions = ['jpg', 'jpeg', 'png', 'webp', 'heic', 'heif'];
    $documentExtensions = ['pdf', 'txt'];
    $videoExtensions = ['mp4', 'avi', 'mov', 'wmv', 'flv', 'webm', 'mkv', 'm4v'];
    
    foreach ($iterator as $file) {
        if ($file->isFile()) {
            $extension = strtolower($file->getExtension());
            $filename = $file->getFilename();
            
            // Hoppa över metadata-filer
            if ($extension === 'json' || $extension === 'finfo') {
                continue;
            }
            
            if (in_array($extension, $imageExtensions)) {
                // Kontrollera om det är en thumbnail
                if (strpos($filename, '-thumb.') !== false) {
                    $stats['thumbnail_files']++;
                    $stats['thumbnail_size'] += $file->getSize();
                } else {
                    $stats['image_files']++;
                    $stats['image_size'] += $file->getSize();
                }
            } elseif (in_array($extension, $documentExtensions)) {
                $stats['document_files']++;
                $stats['document_size'] += $file->getSize();
            } elseif (in_array($extension, $videoExtensions)) {
                $stats['video_files']++;
                $stats['video_size'] += $file->getSize();
            }
        }
    }
    
    return $stats;
}

// Hämta statistik för uploads-mappen
$uploadsPath = __DIR__ . '/../../uploads';
$uploadsStats = getImageStats($uploadsPath);

// Funktion för att räkna chat-resurser
function getChatStats($chatsPath) {
    $stats = [
        'chat_directories' => 0,
        'messages_files' => 0,
        'messages_size' => 0,
        'image_files' => 0,
        'image_size' => 0,
        'document_files' => 0,
        'document_size' => 0,
        'video_files' => 0,
        'video_size' => 0,
        'total_size' => 0
    ];
    
    if (!is_dir($chatsPath)) {
        return $stats;
    }
    
    $imageExtensions = ['jpg', 'jpeg', 'png', 'webp', 'heic', 'heif'];
    $documentExtensions = ['pdf', 'txt'];
    $videoExtensions = ['mp4', 'avi', 'mov', 'wmv', 'flv', 'webm', 'mkv', 'm4v'];
    
    $projectDirs = glob($chatsPath . '/*', GLOB_ONLYDIR);
    foreach ($projectDirs as $projectDir) {
        $stats['chat_directories']++;
        
        // Räkna messages.json
        $messagesFile = $projectDir . '/messages.json';
        if (file_exists($messagesFile)) {
            $stats['messages_files']++;
            $stats['messages_size'] += filesize($messagesFile);
        }
        
        // Räkna filer i projektmappen
        $files = glob($projectDir . '/*');
        foreach ($files as $file) {
            if (is_file($file)) {
                $filename = basename($file);
                // Hoppa över messages.json (redan räknat)
                if ($filename === 'messages.json') {
                    continue;
                }
                
                $extension = strtolower(pathinfo($file, PATHINFO_EXTENSION));
                $fileSize = filesize($file);
                $stats['total_size'] += $fileSize;
                
                if (in_array($extension, $imageExtensions)) {
                    $stats['image_files']++;
                    $stats['image_size'] += $fileSize;
                } elseif (in_array($extension, $documentExtensions)) {
                    $stats['document_files']++;
                    $stats['document_size'] += $fileSize;
                } elseif (in_array($extension, $videoExtensions)) {
                    $stats['video_files']++;
                    $stats['video_size'] += $fileSize;
                }
            }
        }
    }
    
    // Lägg till messages.json-storlek i total storlek
    $stats['total_size'] += $stats['messages_size'];
    
    return $stats;
}

// Hämta statistik för chats-mappen
$chatsPath = __DIR__ . '/../../chats';
$chatsStats = getChatStats($chatsPath);

// Funktion för att räkna downloads-resurser
function getDownloadsStats($downloadsPath) {
    $stats = [
        'zip_files' => 0,
        'zip_size' => 0,
        'total_size' => 0
    ];
    
    if (!is_dir($downloadsPath)) {
        return $stats;
    }
    
    // Räkna ZIP-filer
    $zipFiles = glob($downloadsPath . '/*.zip');
    foreach ($zipFiles as $zipFile) {
        if (is_file($zipFile)) {
            $stats['zip_files']++;
            $fileSize = filesize($zipFile);
            $stats['zip_size'] += $fileSize;
            $stats['total_size'] += $fileSize;
        }
    }
    
    return $stats;
}

// Hämta statistik för downloads-mappen
$downloadsPath = __DIR__ . '/../../downloads';
$downloadsStats = getDownloadsStats($downloadsPath);

// Funktion för att kontrollera bilder och videor utan thumbnails
function checkMissingThumbnails($path) {
    $missingThumbnails = [];
    $imageExtensions = ['jpg', 'jpeg', 'png', 'webp', 'heic', 'heif'];
    $videoExtensions = ['mp4', 'avi', 'mov', 'wmv', 'flv', 'webm', 'mkv', 'm4v'];
    $allMediaExtensions = array_merge($imageExtensions, $videoExtensions);
    
    if (!is_dir($path)) {
        return $missingThumbnails;
    }
    
    $iterator = new RecursiveIteratorIterator(
        new RecursiveDirectoryIterator($path, RecursiveDirectoryIterator::SKIP_DOTS)
    );
    
    foreach ($iterator as $file) {
        if ($file->isFile()) {
            $extension = strtolower($file->getExtension());
            $filename = $file->getFilename();
            $filepath = $file->getPathname();
            
            // Hoppa över metadata-filer och thumbnails
            if ($extension === 'json' || $extension === 'finfo' || strpos($filename, '-thumb.') !== false) {
                continue;
            }
            
            if (in_array($extension, $allMediaExtensions)) {
                // Kontrollera om thumbnail finns
                $baseName = pathinfo($filepath, PATHINFO_FILENAME);
                $thumbnailPath = dirname($filepath) . '/' . $baseName . '-thumb.jpg';
                
                if (!file_exists($thumbnailPath)) {
                    $fileType = in_array($extension, $videoExtensions) ? 'video' : 'image';
                    $missingThumbnails[] = [
                        'image' => $filepath,
                        'relative_path' => str_replace($path . '/', '', $filepath),
                        'size' => $file->getSize(),
                        'type' => $fileType
                    ];
                }
            }
        }
    }
    
    return $missingThumbnails;
}

$missingThumbnails = checkMissingThumbnails($uploadsPath);

// Kontrollera vilka video thumbnail-metoder som är tillgängliga
function getAvailableVideoMethods() {
    $methods = [];
    
    // ImageMagick (prioriteras)
    if (extension_loaded('imagick')) {
        $methods[] = 'ImageMagick (PHP extension)';
    }
    
    // FFmpeg
    $ffmpegPaths = ['/usr/bin/ffmpeg', '/usr/local/bin/ffmpeg', '/opt/homebrew/bin/ffmpeg'];
    foreach ($ffmpegPaths as $path) {
        if (file_exists($path) && is_executable($path)) {
            $methods[] = 'FFmpeg (kommandorad)';
            break;
        }
    }
    
    // VLC
    $vlcPaths = ['/usr/bin/vlc', '/usr/local/bin/vlc', '/Applications/VLC.app/Contents/MacOS/VLC'];
    foreach ($vlcPaths as $path) {
        if (file_exists($path) && is_executable($path)) {
            $methods[] = 'VLC Media Player';
            break;
        }
    }
    
    // PHP FFmpeg extension
    if (extension_loaded('ffmpeg')) {
        $methods[] = 'PHP FFmpeg extension';
    }
    
    return $methods;
}

$availableVideoMethods = getAvailableVideoMethods();

// API endpoint för att generera saknade thumbnails
if (isset($_GET['action']) && $_GET['action'] === 'generate_thumbnails') {
    header('Content-Type: application/json; charset=utf-8');
    
    $results = generateMissingThumbnails($uploadsPath);
    
    echo json_encode([
        'success' => true,
        'results' => $results,
        'debug' => [
            'available_methods' => getAvailableVideoMethods(),
            'imagick_loaded' => extension_loaded('imagick'),
            'ffmpeg_available' => checkFFmpegAvailable(),
            'vlc_available' => checkVLCAvailable()
        ]
    ], JSON_UNESCAPED_UNICODE);
    exit;
}


// API endpoint för att skapa placeholder thumbnail
if (isset($_GET['action']) && $_GET['action'] === 'create_placeholder') {
    header('Content-Type: application/json; charset=utf-8');
    
    $videoPath = $_GET['video'] ?? '';
    if (empty($videoPath)) {
        echo json_encode(['success' => false, 'error' => 'No video path provided']);
        exit;
    }
    
    // Rensa sökvägen från mellanslag och extra tecken
    $videoPath = trim($videoPath);
    $videoPath = str_replace(' ', '', $videoPath);
    
    $fullPath = $uploadsPath . '/' . $videoPath;
    if (!file_exists($fullPath)) {
        echo json_encode(['success' => false, 'error' => 'Video file not found: ' . $fullPath]);
        exit;
    }
    
    $results = createVideoPlaceholderThumbnail($fullPath);
    echo json_encode($results, JSON_UNESCAPED_UNICODE);
    exit;
}



// Funktion för att testa video thumbnail-generering
function testVideoThumbnailGeneration($videoPath) {
    $results = [
        'video_path' => $videoPath,
        'file_exists' => file_exists($videoPath),
        'file_size' => file_exists($videoPath) ? filesize($videoPath) : 0,
        'file_readable' => file_exists($videoPath) ? is_readable($videoPath) : false,
        'file_permissions' => file_exists($videoPath) ? substr(sprintf('%o', fileperms($videoPath)), -4) : 'N/A',
        'file_owner' => file_exists($videoPath) ? posix_getpwuid(fileowner($videoPath))['name'] : 'N/A',
        'file_group' => file_exists($videoPath) ? posix_getgrgid(filegroup($videoPath))['name'] : 'N/A',
        'php_user' => get_current_user(),
        'methods' => []
    ];
    
    // Test ImageMagick
    if (extension_loaded('imagick')) {
        $tempPath = sys_get_temp_dir() . '/test_imagick_' . time() . '.jpg';
        $start = microtime(true);
        $success = createVideoThumbnailImageMagick($videoPath, $tempPath, 320);
        $time = microtime(true) - $start;
        
        $results['methods']['ImageMagick'] = [
            'success' => $success,
            'time' => round($time, 3),
            'output_exists' => file_exists($tempPath),
            'output_size' => file_exists($tempPath) ? filesize($tempPath) : 0
        ];
        
        if (file_exists($tempPath)) {
            unlink($tempPath);
        }
    }
    
    // Test FFmpeg
    if (checkFFmpegAvailable()) {
        $tempPath = sys_get_temp_dir() . '/test_ffmpeg_' . time() . '.jpg';
        $start = microtime(true);
        $success = createVideoThumbnailFFmpeg($videoPath, $tempPath, 320);
        $time = microtime(true) - $start;
        
        $results['methods']['FFmpeg'] = [
            'success' => $success,
            'time' => round($time, 3),
            'output_exists' => file_exists($tempPath),
            'output_size' => file_exists($tempPath) ? filesize($tempPath) : 0
        ];
        
        if (file_exists($tempPath)) {
            unlink($tempPath);
        }
    }
    
    return $results;
}

// Funktion för att testa FFmpeg direkt
function testFFmpegDirect($videoPath) {
    $results = [
        'video_path' => $videoPath,
        'file_exists' => file_exists($videoPath),
        'file_size' => file_exists($videoPath) ? filesize($videoPath) : 0,
        'file_readable' => file_exists($videoPath) ? is_readable($videoPath) : false,
        'ffmpeg_available' => checkFFmpegAvailable(),
        'ffmpeg_path' => '',
        'test_command' => '',
        'command_output' => [],
        'return_code' => 0,
        'success' => false,
        'errors' => []
    ];
    
    if (!checkFFmpegAvailable()) {
        $results['errors'][] = 'FFmpeg is not available';
        return $results;
    }
    
    // Hitta FFmpeg-sökväg
    $ffmpegPaths = ['/usr/bin/ffmpeg', '/usr/local/bin/ffmpeg', '/opt/homebrew/bin/ffmpeg'];
    foreach ($ffmpegPaths as $path) {
        if (file_exists($path) && is_executable($path)) {
            $results['ffmpeg_path'] = $path;
            break;
        }
    }
    
    if (empty($results['ffmpeg_path'])) {
        // Testa PATH
        $output = [];
        $returnCode = 0;
        exec('which ffmpeg 2>/dev/null', $output, $returnCode);
        if ($returnCode === 0 && !empty($output)) {
            $results['ffmpeg_path'] = 'ffmpeg';
        }
    }
    
    if (empty($results['ffmpeg_path'])) {
        $results['errors'][] = 'Could not find FFmpeg executable';
        return $results;
    }
    
    // Testa FFmpeg med videon - försök olika format och kommandon
    $extension = strtolower(pathinfo($videoPath, PATHINFO_EXTENSION));
    
    // Försök olika format i prioritetsordning (börja med enklare format)
    $formats = ['bmp', 'png', 'jpg', 'tiff'];
    $success = false;
    $lastOutput = [];
    $lastReturnCode = 0;
    $workingFormat = '';
    
    foreach ($formats as $format) {
        $tempImage = sys_get_temp_dir() . '/test_ffmpeg_' . time() . '.' . $format;
        
        // Försök olika kommandon för varje format
        $commands = [];
        
        // Kommando 1: Enkel frame-extraktion
        $commands[] = sprintf(
            '%s -i %s -ss 1 -vframes 1 -y %s 2>&1',
            escapeshellarg($results['ffmpeg_path']),
            escapeshellarg($videoPath),
            escapeshellarg($tempImage)
        );
        
        // Kommando 2: Med scale-filter
        $commands[] = sprintf(
            '%s -i %s -ss 1 -vframes 1 -vf "scale=320:-1" -y %s 2>&1',
            escapeshellarg($results['ffmpeg_path']),
            escapeshellarg($videoPath),
            escapeshellarg($tempImage)
        );
        
        // Kommando 3: För MOV-filer med tidsbegränsning
        if ($extension === 'mov') {
            $commands[] = sprintf(
                '%s -i %s -ss 1 -t 1 -vframes 1 -y %s 2>&1',
                escapeshellarg($results['ffmpeg_path']),
                escapeshellarg($videoPath),
                escapeshellarg($tempImage)
            );
        }
        
        foreach ($commands as $index => $command) {
            $output = [];
            $returnCode = 0;
            exec($command, $output, $returnCode);
            
            $lastOutput = $output;
            $lastReturnCode = $returnCode;
            
            error_log("FFmpeg format " . $format . " attempt " . ($index + 1) . ": " . $command);
            error_log("Return code: " . $returnCode);
            error_log("Output: " . implode("\n", $output));
            
            if ($returnCode === 0 && file_exists($tempImage)) {
                $success = true;
                $workingFormat = $format;
                $results['working_format'] = $format;
                $results['working_command'] = $command;
                break 2; // Bryt båda looparna
            }
        }
        
        // Rensa temp-fil om den skapades
        if (file_exists($tempImage)) {
            unlink($tempImage);
        }
    }
    
    $results['command_output'] = $lastOutput;
    $results['return_code'] = $lastReturnCode;
    
    // Logga detaljerad information
    error_log("Video file exists: " . (file_exists($videoPath) ? 'Yes' : 'No'));
    error_log("Video file readable: " . (is_readable($videoPath) ? 'Yes' : 'No'));
    error_log("Video file size: " . (file_exists($videoPath) ? filesize($videoPath) : 'N/A') . " bytes");
    
    if ($success) {
        $results['success'] = true;
        $results['output_size'] = filesize($tempImage);
        unlink($tempImage);
    } else {
        $results['errors'][] = 'FFmpeg failed with return code: ' . $lastReturnCode;
        $results['errors'][] = 'Output: ' . implode("\n", $lastOutput);
    }
    
    return $results;
}

// Funktion för att testa ImageMagick direkt
function testImageMagickDirect($videoPath) {
    $results = [
        'video_path' => $videoPath,
        'file_exists' => file_exists($videoPath),
        'file_size' => file_exists($videoPath) ? filesize($videoPath) : 0,
        'file_readable' => file_exists($videoPath) ? is_readable($videoPath) : false,
        'imagick_loaded' => extension_loaded('imagick'),
        'success' => false,
        'errors' => [],
        'test_attempts' => []
    ];
    
    if (!extension_loaded('imagick')) {
        $results['errors'][] = 'ImageMagick extension not loaded';
        return $results;
    }
    
    // Testa olika tidsstämplar
    $timeStamps = [0, 1, 2, 5, 10];
    
    foreach ($timeStamps as $seconds) {
        $tempPath = sys_get_temp_dir() . '/test_imagick_' . time() . '_' . $seconds . '.jpg';
        
        try {
            // @phpstan-ignore-next-line
            $imagick = new Imagick();
            
            $videoPathWithTime = $videoPath . '[' . $seconds . ']';
            if ($seconds === 0) {
                $videoPathWithTime = $videoPath;
            }
            
            $start = microtime(true);
            $imagick->readImage($videoPathWithTime);
            $time = microtime(true) - $start;
            
            $imagick->thumbnailImage(320, 0);
            $imagick->setImageFormat('jpeg');
            $imagick->setImageCompressionQuality(75);
            
            if ($imagick->writeImage($tempPath)) {
                $results['success'] = true;
                $results['working_timestamp'] = $seconds;
                $results['output_size'] = filesize($tempPath);
                $results['processing_time'] = round($time, 3);
                
                $results['test_attempts'][] = [
                    'timestamp' => $seconds,
                    'success' => true,
                    'time' => round($time, 3),
                    'output_size' => filesize($tempPath)
                ];
                
                $imagick->destroy();
                unlink($tempPath);
                break;
            }
            
            $imagick->destroy();
            
        } catch (Exception $e) {
            $results['test_attempts'][] = [
                'timestamp' => $seconds,
                'success' => false,
                'error' => $e->getMessage()
            ];
            
            error_log("ImageMagick test failed at {$seconds}s: " . $e->getMessage());
        }
    }
    
    if (!$results['success']) {
        $results['errors'][] = 'ImageMagick failed to process video at any timestamp';
    }
    
    return $results;
}

// Funktion för att skapa placeholder thumbnail för video
function createVideoPlaceholderThumbnail($videoPath) {
    $results = [
        'video_path' => $videoPath,
        'success' => false,
        'thumbnail_path' => '',
        'errors' => []
    ];
    
    if (!file_exists($videoPath)) {
        $results['errors'][] = 'Video file does not exist';
        return $results;
    }
    
    // Skapa thumbnail-sökväg
    $pathInfo = pathinfo($videoPath);
    $thumbnailPath = $pathInfo['dirname'] . '/' . $pathInfo['filename'] . '-thumb.jpg';
    
    try {
        // Skapa en placeholder thumbnail med GD
        $width = 320;
        $height = 240;
        
        // Skapa bild
        $image = imagecreatetruecolor($width, $height);
        
        // Bakgrundsfärg (mörkblå)
        $bgColor = imagecolorallocate($image, 25, 25, 112);
        imagefill($image, 0, 0, $bgColor);
        
        // Textfärg (vit)
        $textColor = imagecolorallocate($image, 255, 255, 255);
        
        // Lägg till video-ikon (enkel rektangel)
        $iconColor = imagecolorallocate($image, 255, 255, 255);
        $iconX = $width / 2 - 30;
        $iconY = $height / 2 - 20;
        imagefilledrectangle($image, $iconX, $iconY, $iconX + 60, $iconY + 40, $iconColor);
        
        // Lägg till text
        $fontSize = 3;
        $text = 'VIDEO';
        $textWidth = imagefontwidth($fontSize) * strlen($text);
        $textX = ($width - $textWidth) / 2;
        $textY = $height / 2 + 30;
        imagestring($image, $fontSize, $textX, $textY, $text, $textColor);
        
        // Lägg till filnamn (trunkerat)
        $filename = $pathInfo['filename'];
        if (strlen($filename) > 20) {
            $filename = substr($filename, 0, 17) . '...';
        }
        $filenameWidth = imagefontwidth(2) * strlen($filename);
        $filenameX = ($width - $filenameWidth) / 2;
        $filenameY = $height / 2 + 50;
        imagestring($image, 2, $filenameX, $filenameY, $filename, $textColor);
        
        // Spara bild
        if (imagejpeg($image, $thumbnailPath, 75)) {
            $results['success'] = true;
            $results['thumbnail_path'] = $thumbnailPath;
            $results['file_size'] = filesize($thumbnailPath);
            error_log("Created placeholder thumbnail: " . $thumbnailPath);
            
            // Uppdatera JSON-filen
            $jsonPath = $pathInfo['dirname'] . '/' . $pathInfo['filename'] . '.json';
            if (file_exists($jsonPath)) {
                $metadata = json_decode(file_get_contents($jsonPath), true);
                if ($metadata) {
                    $metadata['thumb_file'] = basename($thumbnailPath);
                    $metadata['thumbnail_type'] = 'placeholder';
                    $metadata['thumbnail_created'] = date('Y-m-d H:i:s');
                    file_put_contents($jsonPath, json_encode($metadata, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
                    error_log("Updated JSON file: " . $jsonPath);
                }
            } else {
                // Skapa ny JSON-fil om den inte finns
                $metadata = [
                    'thumb_file' => basename($thumbnailPath),
                    'thumbnail_type' => 'placeholder',
                    'thumbnail_created' => date('Y-m-d H:i:s'),
                    'original_file' => basename($videoPath),
                    'file_type' => 'video'
                ];
                file_put_contents($jsonPath, json_encode($metadata, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
                error_log("Created new JSON file: " . $jsonPath);
            }
        } else {
            $results['errors'][] = 'Failed to save thumbnail image';
        }
        
        imagedestroy($image);
        
    } catch (Exception $e) {
        $results['errors'][] = 'Exception: ' . $e->getMessage();
    }
    
    return $results;
}

// Hjälpfunktioner för att kontrollera tillgänglighet
function checkFFmpegAvailable() {
    $ffmpegPaths = ['/usr/bin/ffmpeg', '/usr/local/bin/ffmpeg', '/opt/homebrew/bin/ffmpeg'];
    foreach ($ffmpegPaths as $path) {
        if (file_exists($path) && is_executable($path)) {
            return true;
        }
    }
    
    // Testa PATH
    $output = [];
    $returnCode = 0;
    exec('which ffmpeg 2>/dev/null', $output, $returnCode);
    return $returnCode === 0 && !empty($output);
}

function checkVLCAvailable() {
    $vlcPaths = ['/usr/bin/vlc', '/usr/local/bin/vlc', '/Applications/VLC.app/Contents/MacOS/VLC'];
    foreach ($vlcPaths as $path) {
        if (file_exists($path) && is_executable($path)) {
            return true;
        }
    }
    
    // Testa PATH
    $output = [];
    $returnCode = 0;
    exec('which vlc 2>/dev/null', $output, $returnCode);
    return $returnCode === 0 && !empty($output);
}

// Funktion för att generera saknade thumbnails
function generateMissingThumbnails($path) {
    $results = [
        'success' => 0,
        'failed' => 0,
        'errors' => []
    ];
    
    $missingThumbnails = checkMissingThumbnails($path);
    
    foreach ($missingThumbnails as $missing) {
        $imagePath = $missing['image'];
        $baseName = pathinfo($imagePath, PATHINFO_FILENAME);
        $thumbnailPath = dirname($imagePath) . '/' . $baseName . '-thumb.jpg';
        $fileType = $missing['type'] ?? 'image';
        
        // Generera thumbnail baserat på filtyp
        $thumbCreated = false;
        if ($fileType === 'video') {
            // För videor, skapa placeholder direkt
            error_log("Creating placeholder thumbnail for video: " . $imagePath);
            $placeholderResult = createVideoPlaceholderThumbnail($imagePath);
            if ($placeholderResult['success']) {
                $thumbCreated = true;
                error_log("Created placeholder thumbnail: " . $placeholderResult['thumbnail_path']);
            }
        } else {
            $thumbCreated = createThumbnailImage($imagePath, $thumbnailPath, 320);
        }
        
        if ($thumbCreated) {
            $results['success']++;
            error_log("Successfully created thumbnail for: " . $missing['relative_path']);
            
            // Uppdatera JSON-filen
            $jsonPath = dirname($imagePath) . '/' . $baseName . '.json';
            if (file_exists($jsonPath)) {
                $metadata = json_decode(file_get_contents($jsonPath), true);
                if ($metadata) {
                    $metadata['thumb_file'] = basename($thumbnailPath);
                    file_put_contents($jsonPath, json_encode($metadata, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
                }
            }
        } else {
            $results['failed']++;
            $errorMsg = 'Misslyckades att generera thumbnail för: ' . $missing['relative_path'] . ' (' . $fileType . ')';
            $results['errors'][] = $errorMsg;
            error_log("Failed to create thumbnail for: " . $missing['relative_path'] . " (Type: " . $fileType . ")");
            
            // Lägg till specifik debug-information för video-filer
            if ($fileType === 'video') {
                $extension = strtolower(pathinfo($imagePath, PATHINFO_EXTENSION));
                error_log("Video file extension: " . $extension);
                error_log("Video file size: " . filesize($imagePath) . " bytes");
                error_log("Video file readable: " . (is_readable($imagePath) ? 'Yes' : 'No'));
                error_log("Thumbnail path: " . $thumbnailPath);
                error_log("Thumbnail directory writable: " . (is_writable(dirname($thumbnailPath)) ? 'Yes' : 'No'));
            }
        }
    }
    
    return $results;
}

// Funktion för att skapa thumbnail (kopierad från upload.php)
function createThumbnailImage($sourcePath, $destPath, $maxWidth = 320) {
    // GD fallback
    $info = @getimagesize($sourcePath);
    if ($info === false) {
        return false;
    }
    $mime = $info['mime'] ?? '';
    switch ($mime) {
        case 'image/jpeg':
        case 'image/jpg':
        case 'image/pjpeg':
            $src = @imagecreatefromjpeg($sourcePath);
            break;
        case 'image/png':
            $src = @imagecreatefrompng($sourcePath);
            break;
        case 'image/webp':
            if (function_exists('imagecreatefromwebp')) {
                $src = @imagecreatefromwebp($sourcePath);
            } else {
                $src = false;
            }
            break;
        default:
            // GD stödjer ej HEIC/HEIF – hoppa över
            $src = false;
    }
    if (!$src) {
        return false;
    }
    $width = imagesx($src);
    $height = imagesy($src);
    if ($width <= 0 || $height <= 0) {
        imagedestroy($src);
        return false;
    }
    $targetWidth = $width > $maxWidth ? $maxWidth : $width;
    $targetHeight = (int)round(($targetWidth / $width) * $height);
    $dst = imagecreatetruecolor($targetWidth, $targetHeight);
    // För PNG med alfa – fyll vit bakgrund
    $white = imagecolorallocate($dst, 255, 255, 255);
    imagefill($dst, 0, 0, $white);
    imagecopyresampled($dst, $src, 0, 0, 0, 0, $targetWidth, $targetHeight, $width, $height);
    $ok = imagejpeg($dst, $destPath, 75);
    imagedestroy($dst);
    imagedestroy($src);
    return $ok;
}

// Funktion för att skapa video thumbnail med olika metoder
function createVideoThumbnail($sourcePath, $destPath, $maxWidth = 320) {
    // Försök olika metoder i prioritetsordning (ImageMagick först)
    $methods = [
        'imagemagick' => 'createVideoThumbnailImageMagick',
        'ffmpeg' => 'createVideoThumbnailFFmpeg',
        'vlc' => 'createVideoThumbnailVLC',
        'php_ffmpeg' => 'createVideoThumbnailPHPFFmpeg'
    ];
    
    foreach ($methods as $methodName => $methodFunction) {
        if (function_exists($methodFunction)) {
            $result = $methodFunction($sourcePath, $destPath, $maxWidth);
            if ($result) {
                return true;
            }
        }
    }
    
    return false;
}

// FFmpeg metod (original)
function createVideoThumbnailFFmpeg($sourcePath, $destPath, $maxWidth = 320) {
    $ffmpegPath = '';
    $possiblePaths = [
        '/usr/bin/ffmpeg',
        '/usr/local/bin/ffmpeg',
        '/opt/homebrew/bin/ffmpeg',
        'ffmpeg'
    ];
    
    foreach ($possiblePaths as $path) {
        if ($path === 'ffmpeg') {
            $output = [];
            $returnCode = 0;
            exec('which ffmpeg 2>/dev/null', $output, $returnCode);
            if ($returnCode === 0 && !empty($output)) {
                $ffmpegPath = 'ffmpeg';
                break;
            }
        } else {
            if (file_exists($path) && is_executable($path)) {
                $ffmpegPath = $path;
                break;
            }
        }
    }
    
    if (empty($ffmpegPath)) {
        return false;
    }
    
    // Försök olika format eftersom JPG kanske inte fungerar
    $formats = ['png', 'jpg', 'bmp'];
    $tempImage = '';
    $workingFormat = '';
    
    foreach ($formats as $format) {
        $tempImage = tempnam(sys_get_temp_dir(), 'video_thumb_') . '.' . $format;
        $testCommand = sprintf(
            '%s -i %s -ss 1 -vframes 1 -y %s 2>/dev/null',
            escapeshellarg($ffmpegPath),
            escapeshellarg($sourcePath),
            escapeshellarg($tempImage)
        );
        
        $output = [];
        $returnCode = 0;
        exec($testCommand, $output, $returnCode);
        
        if ($returnCode === 0 && file_exists($tempImage)) {
            $workingFormat = $format;
            break;
        }
        
        if (file_exists($tempImage)) {
            unlink($tempImage);
        }
    }
    
    if (empty($workingFormat)) {
        return false;
    }
    
    try {
        // Använd det fungerande formatet
        $tempImage = tempnam(sys_get_temp_dir(), 'video_thumb_') . '.' . $workingFormat;
        
        // För MOV-filer, använd lite annorlunda parametrar
        $extension = strtolower(pathinfo($sourcePath, PATHINFO_EXTENSION));
        if ($extension === 'mov') {
            // För MOV-filer, använd -t 1 för att säkerställa att vi får en frame
            $command = sprintf(
                '%s -i %s -ss 1 -t 1 -vframes 1 -vf "scale=%d:-1" -y %s 2>/dev/null',
                escapeshellarg($ffmpegPath),
                escapeshellarg($sourcePath),
                $maxWidth,
                escapeshellarg($tempImage)
            );
        } else {
            $command = sprintf(
                '%s -i %s -ss 1 -vframes 1 -vf "scale=%d:-1" -y %s 2>/dev/null',
                escapeshellarg($ffmpegPath),
                escapeshellarg($sourcePath),
                $maxWidth,
                escapeshellarg($tempImage)
            );
        }
        
        $output = [];
        $returnCode = 0;
        exec($command, $output, $returnCode);
        
        error_log("FFmpeg command: " . $command);
        error_log("FFmpeg return code: " . $returnCode);
        error_log("FFmpeg output: " . implode("\n", $output));
        error_log("Source file exists: " . (file_exists($sourcePath) ? 'Yes' : 'No'));
        error_log("Source file readable: " . (is_readable($sourcePath) ? 'Yes' : 'No'));
        error_log("Source file size: " . (file_exists($sourcePath) ? filesize($sourcePath) : 'N/A') . " bytes");
        
        if ($returnCode === 0 && file_exists($tempImage)) {
            // Om vi använder ett annat format än JPG, konvertera det
            if ($workingFormat !== 'jpg') {
                $finalPath = $destPath;
            } else {
                $finalPath = $destPath;
            }
            
            if (rename($tempImage, $finalPath)) {
                error_log("FFmpeg successfully created thumbnail: " . $finalPath . " (format: " . $workingFormat . ")");
                return true;
            } else {
                error_log("FFmpeg failed to move thumbnail from " . $tempImage . " to " . $finalPath);
            }
        } else {
            error_log("FFmpeg failed to create thumbnail. Return code: " . $returnCode . ", Output: " . implode("\n", $output));
        }
        
        if (file_exists($tempImage)) {
            unlink($tempImage);
        }
        
        return false;
        
    } catch (Exception $e) {
        if (file_exists($tempImage)) {
            unlink($tempImage);
        }
        return false;
    }
}

// ImageMagick metod (kräver ImageMagick PHP extension)
function createVideoThumbnailImageMagick($sourcePath, $destPath, $maxWidth = 320) {
    if (!extension_loaded('imagick')) {
        error_log("ImageMagick extension not loaded");
        return false;
    }
    
    if (!file_exists($sourcePath)) {
        error_log("Source video file does not exist: " . $sourcePath);
        return false;
    }
    
    // Kontrollera filstorlek (om filen är för liten, kan det vara korrupt)
    $fileSize = filesize($sourcePath);
    if ($fileSize < 1024) { // Mindre än 1KB
        error_log("Video file too small, possibly corrupt: " . $sourcePath . " (size: " . $fileSize . " bytes)");
        return false;
    }
    
    try {
        // Imagick är en extern klass från ImageMagick PHP extension
        // @phpstan-ignore-next-line
        $imagick = new Imagick();
        
        // Försök olika tidsstämplar för att hitta en bra frame
        $timeStamps = [0, 1, 2, 5, 10]; // Sekunder in i videon (inkluderar 0 för första frame)
        $success = false;
        $lastError = '';
        
        foreach ($timeStamps as $seconds) {
            try {
                // För MOV-filer, försök olika syntax
                $videoPath = $sourcePath;
                if (strtolower(pathinfo($sourcePath, PATHINFO_EXTENSION)) === 'mov') {
                    // För MOV-filer, försök utan tidsstämpel först
                    if ($seconds === 0) {
                        $imagick->readImage($videoPath);
                    } else {
                        $imagick->readImage($videoPath . '[' . $seconds . ']');
                    }
                } else {
                    $imagick->readImage($videoPath . '[' . $seconds . ']');
                }
                $success = true;
                break;
            } catch (Exception $e) {
                $lastError = $e->getMessage();
                error_log("ImageMagick failed at {$seconds}s for {$sourcePath}: " . $lastError);
                // Rensa Imagick-objektet för nästa försök
                try {
                    $imagick->clear();
                } catch (Exception $clearError) {
                    // Ignorera clear-fel
                }
                continue;
            }
        }
        
        if (!$success) {
            error_log("ImageMagick failed to read video at any timestamp: " . $lastError);
            $imagick->destroy();
            return false;
        }
        
        // Resize och konvertera
        $imagick->thumbnailImage($maxWidth, 0);
        $imagick->setImageFormat('jpeg');
        $imagick->setImageCompressionQuality(75);
        
        if ($imagick->writeImage($destPath)) {
            $imagick->destroy();
            error_log("ImageMagick successfully created thumbnail: " . $destPath);
            return true;
        }
        
        error_log("ImageMagick failed to write thumbnail to: " . $destPath);
        $imagick->destroy();
        return false;
        
    } catch (Exception $e) {
        error_log("ImageMagick exception: " . $e->getMessage());
        return false;
    }
}

// VLC metod
function createVideoThumbnailVLC($sourcePath, $destPath, $maxWidth = 320) {
    $vlcPath = '';
    $possiblePaths = [
        '/usr/bin/vlc',
        '/usr/local/bin/vlc',
        '/Applications/VLC.app/Contents/MacOS/VLC',
        'vlc'
    ];
    
    foreach ($possiblePaths as $path) {
        if ($path === 'vlc') {
            $output = [];
            $returnCode = 0;
            exec('which vlc 2>/dev/null', $output, $returnCode);
            if ($returnCode === 0 && !empty($output)) {
                $vlcPath = 'vlc';
                break;
            }
        } else {
            if (file_exists($path) && is_executable($path)) {
                $vlcPath = $path;
                break;
            }
        }
    }
    
    if (empty($vlcPath)) {
        return false;
    }
    
    $tempImage = tempnam(sys_get_temp_dir(), 'video_thumb_') . '.jpg';
    
    try {
        // VLC kan extrahera thumbnails
        $command = sprintf(
            '%s --intf dummy --extraintf snapshot --snapshot-path %s --snapshot-format jpg --snapshot-time 1 %s vlc://quit 2>/dev/null',
            escapeshellarg($vlcPath),
            escapeshellarg(dirname($tempImage)),
            escapeshellarg($sourcePath)
        );
        
        $output = [];
        $returnCode = 0;
        exec($command, $output, $returnCode);
        
        // VLC skapar filer med specifika namn, vi behöver hitta den genererade filen
        $snapshotFiles = glob(dirname($tempImage) . '/*.jpg');
        if (!empty($snapshotFiles)) {
            $latestFile = max($snapshotFiles);
            if (rename($latestFile, $destPath)) {
                return true;
            }
        }
        
        return false;
        
    } catch (Exception $e) {
        return false;
    }
}

// PHP FFmpeg extension metod (kräver PHP-FFMpeg Composer package)
function createVideoThumbnailPHPFFmpeg($sourcePath, $destPath, $maxWidth = 320) {
    if (!extension_loaded('ffmpeg')) {
        return false;
    }
    
    try {
        // FFmpeg och TimeCode är externa klasser från PHP-FFMpeg package
        // @phpstan-ignore-next-line
        $ffmpeg = new FFmpeg();
        // @phpstan-ignore-next-line
        $video = $ffmpeg->open($sourcePath);
        // @phpstan-ignore-next-line
        $frame = $video->frame(TimeCode::fromSeconds(1));
        $frame->save($destPath);
        return true;
    } catch (Exception $e) {
        return false;
    }
}

// Hämta användarnamn baserat på user_id
function getUserNameById($userId) {
    if (empty($userId)) {
        return 'Okänd användare';
    }
    
    // Läs userids.json för att hitta användarnamn
    $userIdsFile = __DIR__ . '/../data/userids.json';
    if (file_exists($userIdsFile)) {
        $userIds = json_decode(file_get_contents($userIdsFile), true) ?: [];
        foreach ($userIds as $user) {
            if (isset($user['user_id']) && $user['user_id'] === $userId) {
                return $user['full_name'] ?? $userId;
            }
        }
    }
    
    return $userId; // Fallback till user_id om namn inte hittas
}

?>

<!DOCTYPE html>
<html lang="sv">
<head>
    <meta charset="utf-8" />
    <meta name="viewport" content="width=device-width,initial-scale=1" />
    <title>Resurser - ByggArkiv</title>
    <style>
        body { 
            font-family: system-ui, -apple-system, Segoe UI, Roboto, sans-serif; 
            margin: 16px; 
            background-color: #f5f5f5;
            line-height: 1.6;
        }
        .header {
            text-align: center;
            margin-bottom: 40px;
        }
        .header img, .logo {
            max-width: 100%;
            height: auto;
            max-height: 120px;
            display: block;
            margin: 0 auto;
        }
        .container {
            max-width: 1000px;
            margin: 0 auto;
        }
        .page-title {
            text-align: left;
            color: #1a73e8;
            margin-bottom: 30px;
            font-size: 28px;
            font-weight: 600;
        }
        .stats-table {
            background: white;
            border-radius: 12px;
            box-shadow: 0 2px 8px rgba(0,0,0,0.1);
            overflow: hidden;
            margin-bottom: 30px;
        }
        .stats-table table {
            width: 100%;
            border-collapse: collapse;
            table-layout: fixed;
        }
        .stats-table th {
            background: #f8f9fa;
            color: #1a73e8;
            font-weight: 600;
            padding: 16px;
            text-align: left;
            border-bottom: 1px solid #e0e0e0;
        }
        .stats-table th:first-child {
            width: auto;
        }
        .stats-table th:nth-child(2),
        .stats-table th:nth-child(3) {
            text-align: right !important;
            width: 120px;
        }
        .stats-table td {
            padding: 12px 16px;
            border-bottom: 1px solid #f0f0f0;
        }
        .stats-table tr:last-child td {
            border-bottom: none;
        }
        .stat-label {
            color: #666;
            font-size: 14px;
        }
        .stat-value {
            color: #333;
            font-weight: 500;
            font-size: 14px;
            text-align: right;
        }
        .back-link {
            text-align: center;
            margin-top: 30px;
        }
        .back-link a {
            color: #1a73e8;
            text-decoration: none;
            font-weight: 500;
        }
        .back-link a:hover {
            text-decoration: underline;
        }
        @media (max-width: 768px) {
            .stats-table {
                overflow-x: auto;
            }
            .stats-table table {
                min-width: 400px;
            }
        }
    </style>
</head>
<body>
    <div class="container">
        <header class="header">
            <a href="../../index.html">
                <img src="../../data/logo.png" alt="ByggArkiv Snickare Stefan" class="logo">
            </a>
        </header>

        <div class="back-link" style="margin-top: 20px; margin-bottom: 20px;">
            <?php
                $script = $_SERVER['SCRIPT_NAME'] ?? '';
                $admPos = strpos($script, '/adm/');
                $adminUrl = ($admPos !== false ? substr($script, 0, $admPos + 4) : '/adm') . '/admin.php';
            ?>
            <a href="<?= htmlspecialchars($adminUrl) ?>">← Tillbaka till admin</a>
        </div>

        <h1 class="page-title">Resurser</h1>

        <div class="stats-table">
            <h3 style="color: #1a73e8; margin: 0 0 16px 0; font-size: 18px; font-weight: 600; padding: 16px; border-bottom: 1px solid #e0e0e0;">Projektalbum (uploads/)</h3>
            <table>
                <thead>
                    <tr>
                        <th>Filtyp</th>
                        <th>Antal</th>
                        <th>Storlek</th>
                    </tr>
                </thead>
                <tbody>
                    <tr>
                        <td class="stat-label">Bilder</td>
                        <td class="stat-value"><?php echo number_format($uploadsStats['image_files']); ?></td>
                        <td class="stat-value"><?php echo formatBytes($uploadsStats['image_size']); ?></td>
                    </tr>
                    <tr>
                        <td class="stat-label">Videor</td>
                        <td class="stat-value"><?php echo number_format($uploadsStats['video_files']); ?></td>
                        <td class="stat-value"><?php echo formatBytes($uploadsStats['video_size']); ?></td>
                    </tr>
                    <tr>
                        <td class="stat-label">Dokument</td>
                        <td class="stat-value"><?php echo number_format($uploadsStats['document_files']); ?></td>
                        <td class="stat-value"><?php echo formatBytes($uploadsStats['document_size']); ?></td>
                    </tr>
                    <tr>
                        <td class="stat-label">Thumbnails</td>
                        <td class="stat-value"><?php echo number_format($uploadsStats['thumbnail_files']); ?></td>
                        <td class="stat-value"><?php echo formatBytes($uploadsStats['thumbnail_size']); ?></td>
                    </tr>
                    <tr style="background: #f8f9fa; font-weight: 600;">
                        <td class="stat-label">Totalt projektalbum</td>
                        <td class="stat-value"><?php echo number_format($uploadsStats['image_files'] + $uploadsStats['video_files'] + $uploadsStats['document_files'] + $uploadsStats['thumbnail_files']); ?></td>
                        <td class="stat-value"><?php echo formatBytes($uploadsStats['image_size'] + $uploadsStats['video_size'] + $uploadsStats['document_size'] + $uploadsStats['thumbnail_size']); ?></td>
                    </tr>
                </tbody>
            </table>
        </div>

        <div class="stats-table" style="margin-top: 30px;">
            <h3 style="color: #1a73e8; margin: 0 0 16px 0; font-size: 18px; font-weight: 600; padding: 16px; border-bottom: 1px solid #e0e0e0;">Chatt (chats/)</h3>
            <table>
                <thead>
                    <tr>
                        <th>Resurstyp</th>
                        <th>Antal</th>
                        <th>Storlek</th>
                    </tr>
                </thead>
                <tbody>
                    <tr>
                        <td class="stat-label">Chat-mappar</td>
                        <td class="stat-value"><?php echo number_format($chatsStats['chat_directories']); ?></td>
                        <td class="stat-value">-</td>
                    </tr>
                    <tr>
                        <td class="stat-label">Meddelanden</td>
                        <td class="stat-value"><?php echo number_format($chatsStats['messages_files']); ?></td>
                        <td class="stat-value"><?php echo formatBytes($chatsStats['messages_size']); ?></td>
                    </tr>
                    <tr>
                        <td class="stat-label">Chat-bilder</td>
                        <td class="stat-value"><?php echo number_format($chatsStats['image_files']); ?></td>
                        <td class="stat-value"><?php echo formatBytes($chatsStats['image_size']); ?></td>
                    </tr>
                    <tr>
                        <td class="stat-label">Chat-dokument</td>
                        <td class="stat-value"><?php echo number_format($chatsStats['document_files']); ?></td>
                        <td class="stat-value"><?php echo formatBytes($chatsStats['document_size']); ?></td>
                    </tr>
                    <tr>
                        <td class="stat-label">Chat-videor</td>
                        <td class="stat-value"><?php echo number_format($chatsStats['video_files']); ?></td>
                        <td class="stat-value"><?php echo formatBytes($chatsStats['video_size']); ?></td>
                    </tr>
                    <tr style="background: #f8f9fa; font-weight: 600;">
                        <td class="stat-label">Totalt chatt</td>
                        <td class="stat-value"><?php echo number_format($chatsStats['messages_files'] + $chatsStats['image_files'] + $chatsStats['document_files'] + $chatsStats['video_files']); ?></td>
                        <td class="stat-value"><?php echo formatBytes($chatsStats['total_size']); ?></td>
                    </tr>
                </tbody>
            </table>
        </div>

        <div class="stats-table" style="margin-top: 30px;">
            <h3 style="color: #1a73e8; margin: 0 0 16px 0; font-size: 18px; font-weight: 600; padding: 16px; border-bottom: 1px solid #e0e0e0;">Nedladdningar (downloads/)</h3>
            <table>
                <thead>
                    <tr>
                        <th>Resurstyp</th>
                        <th>Antal</th>
                        <th>Storlek</th>
                    </tr>
                </thead>
                <tbody>
                    <tr>
                        <td class="stat-label">ZIP-filer</td>
                        <td class="stat-value"><?php echo number_format($downloadsStats['zip_files']); ?></td>
                        <td class="stat-value"><?php echo formatBytes($downloadsStats['zip_size']); ?></td>
                    </tr>
                    <tr style="background: #f8f9fa; font-weight: 600;">
                        <td class="stat-label">Totalt nedladdningar</td>
                        <td class="stat-value"><?php echo number_format($downloadsStats['zip_files']); ?></td>
                        <td class="stat-value"><?php echo formatBytes($downloadsStats['total_size']); ?></td>
                    </tr>
                </tbody>
            </table>
        </div>

        <div class="stats-table" style="margin-top: 30px;">
            <h3 style="color: #1a73e8; margin: 0 0 16px 0; font-size: 18px; font-weight: 600; padding: 16px; border-bottom: 1px solid #e0e0e0;">Totalt system</h3>
            <table>
                <thead>
                    <tr>
                        <th>Resurstyp</th>
                        <th>Antal</th>
                        <th>Storlek</th>
                    </tr>
                </thead>
                <tbody>
                    <?php
                    $totalFiles = ($uploadsStats['image_files'] + $uploadsStats['video_files'] + $uploadsStats['document_files'] + $uploadsStats['thumbnail_files']) + 
                                  ($chatsStats['messages_files'] + $chatsStats['image_files'] + $chatsStats['document_files'] + $chatsStats['video_files']) +
                                  $downloadsStats['zip_files'];
                    $totalSize = ($uploadsStats['image_size'] + $uploadsStats['video_size'] + $uploadsStats['document_size'] + $uploadsStats['thumbnail_size']) + 
                                $chatsStats['total_size'] +
                                $downloadsStats['total_size'];
                    ?>
                    <tr style="background: #e3f2fd; font-weight: 600;">
                        <td class="stat-label">Totalt antal filer</td>
                        <td class="stat-value"><?php echo number_format($totalFiles); ?></td>
                        <td class="stat-value"><?php echo formatBytes($totalSize); ?></td>
                    </tr>
                </tbody>
            </table>
        </div>

        <?php if (!empty($missingThumbnails)): ?>
        <div class="stats-table" style="margin-top: 30px;">
            <h3 style="color: #1a73e8; margin: 0 0 16px 0; font-size: 18px; font-weight: 600; padding: 16px;">Kontroll av thumbnails</h3>
            <div style="background: #fff3cd; border: 1px solid #ffeaa7; border-radius: 6px; padding: 16px; margin-bottom: 16px;">
                <?php 
                $imageCount = 0;
                $videoCount = 0;
                foreach ($missingThumbnails as $missing) {
                    if (($missing['type'] ?? 'image') === 'video') {
                        $videoCount++;
                    } else {
                        $imageCount++;
                    }
                }
                ?>
                <strong>Hittade <?= count($missingThumbnails) ?> filer utan thumbnails:</strong>
                <p style="margin: 8px 0 8px 0; font-size: 14px; color: #666;">
                    <?php if ($imageCount > 0): ?>
                        <?= $imageCount ?> bilder
                    <?php endif; ?>
                    <?php if ($imageCount > 0 && $videoCount > 0): ?>, <?php endif; ?>
                    <?php if ($videoCount > 0): ?>
                        <?= $videoCount ?> videor
                    <?php endif; ?>
                </p>
                <p style="margin: 8px 0 8px 0; font-size: 14px; color: #666;">
                    Detta kan bero på att filerna nyligen laddades upp och thumbnails inte har genererats än, 
                    eller att thumbnail-genereringen misslyckades.
                </p>
                <button id="generateThumbnailsBtn" onclick="generateMissingThumbnails()" 
                        style="background: #1a73e8; color: white; border: none; padding: 8px 16px; border-radius: 4px; cursor: pointer; font-size: 14px; font-weight: 500; margin-right: 8px;">
                    Generera saknade thumbnails
                </button>
                <div id="generationStatus" style="margin-top: 12px; display: none;"></div>
            </div>
            
            <div style="max-height: 300px; overflow-y: auto; border: 1px solid #e0e0e0; border-radius: 6px;">
                <table style="width: 100%; border-collapse: collapse;">
                    <thead style="background: #f8f9fa; position: sticky; top: 0;">
                        <tr>
                            <th style="padding: 12px; text-align: left; border-bottom: 1px solid #e0e0e0; font-size: 14px;">Fil</th>
                            <th style="padding: 12px; text-align: center; border-bottom: 1px solid #e0e0e0; font-size: 14px;">Typ</th>
                            <th style="padding: 12px; text-align: right; border-bottom: 1px solid #e0e0e0; font-size: 14px;">Storlek</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($missingThumbnails as $missing): ?>
                        <tr style="border-bottom: 1px solid #f0f0f0;">
                            <td style="padding: 12px; font-size: 14px; color: #333;">
                                <code style="background: #f5f5f5; padding: 2px 6px; border-radius: 3px; font-size: 12px;">
                                    <?= htmlspecialchars($missing['relative_path']) ?>
                                </code>
                            </td>
                            <td style="padding: 12px; text-align: center; font-size: 14px; color: #666;">
                                <span style="background: <?= ($missing['type'] ?? 'image') === 'video' ? '#e3f2fd' : '#f3e5f5'; ?>; color: <?= ($missing['type'] ?? 'image') === 'video' ? '#1976d2' : '#7b1fa2'; ?>; padding: 4px 8px; border-radius: 12px; font-size: 12px; font-weight: 500;">
                                    <?= ($missing['type'] ?? 'image') === 'video' ? '🎥 Video' : '🖼️ Bild' ?>
                                </span>
                            </td>
                            <td style="padding: 12px; text-align: right; font-size: 14px; color: #666;">
                                <?= formatBytes($missing['size']) ?>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </div>
        <?php endif; ?>

    </div>

    <script>
        function showProgressPopup() {
            // Create popup overlay
            const overlay = document.createElement('div');
            overlay.id = 'progressOverlay';
            overlay.style.cssText = `
                position: fixed;
                top: 0;
                left: 0;
                width: 100%;
                height: 100%;
                background: rgba(0, 0, 0, 0.5);
                display: flex;
                justify-content: center;
                align-items: center;
                z-index: 1000;
            `;
            
            // Create popup content
            const popup = document.createElement('div');
            popup.style.cssText = `
                background: white;
                padding: 30px;
                border-radius: 8px;
                box-shadow: 0 4px 20px rgba(0, 0, 0, 0.3);
                text-align: center;
                min-width: 300px;
            `;
            
            popup.innerHTML = `
                <div style="color: #1a73e8; font-size: 18px; font-weight: 600; margin-bottom: 20px;">
                    Genererar thumbnails
                </div>
                <div style="margin-bottom: 20px;">
                    <div style="width: 100%; height: 8px; background: #e0e0e0; border-radius: 4px; overflow: hidden;">
                        <div id="progressBar" style="width: 0%; height: 100%; background: #1a73e8; border-radius: 4px; transition: width 0.3s ease;"></div>
                    </div>
                </div>
                <div id="progressText" style="color: #666; font-size: 14px;">
                    Förbereder...
                </div>
            `;
            
            overlay.appendChild(popup);
            document.body.appendChild(overlay);
            
            // Animate progress bar
            let progress = 0;
            const progressBar = document.getElementById('progressBar');
            const progressText = document.getElementById('progressText');
            
            const interval = setInterval(() => {
                // Fast start, slow finish
                if (progress < 30) {
                    progress += Math.random() * 8 + 4; // Fast start (Förbereder)
                } else if (progress < 60) {
                    progress += Math.random() * 6 + 3; // Fast generation
                } else if (progress < 85) {
                    progress += Math.random() * 2 + 1; // Slower processing
                } else {
                    progress += Math.random() * 0.3 + 0.1; // Very slow finish
                }
                
                // Cap at 90% until completion
                if (progress > 90) progress = 90;
                
                progressBar.style.width = progress + '%';
                
                if (progress < 30) {
                    progressText.textContent = 'Förbereder...';
                } else if (progress < 60) {
                    progressText.textContent = 'Genererar thumbnails...';
                } else if (progress < 85) {
                    progressText.textContent = 'Bearbetar bilder...';
                } else {
                    progressText.textContent = 'Slutför...';
                }
            }, 200);
            
            // Store interval ID for cleanup
            overlay.dataset.intervalId = interval;
        }
        
        function hideProgressPopup() {
            const overlay = document.getElementById('progressOverlay');
            if (overlay) {
                clearInterval(overlay.dataset.intervalId);
                
                // Complete the progress bar before hiding
                const progressBar = document.getElementById('progressBar');
                const progressText = document.getElementById('progressText');
                if (progressBar) {
                    progressBar.style.width = '100%';
                }
                if (progressText) {
                    progressText.textContent = 'Klar!';
                }
                
                // Wait a moment before hiding
                setTimeout(() => {
                    document.body.removeChild(overlay);
                }, 500);
            }
        }
        
        async function generateMissingThumbnails() {
            const button = document.getElementById('generateThumbnailsBtn');
            const statusDiv = document.getElementById('generationStatus');
            
            // Disable button and show loading
            button.disabled = true;
            button.textContent = 'Genererar thumbnails...';
            // Show popup with progress bar
            showProgressPopup();
            
            try {
                const response = await fetch('?action=generate_thumbnails');
                const result = await response.json();
                
                // Hide progress popup
                hideProgressPopup();
                
                if (result.success) {
                    const results = result.results;
                    const debug = result.debug || {};
                    let statusHtml = '<div style="margin-top: 12px;">';
                    
                    if (results.success > 0) {
                        statusHtml += `<div style="color: #155724; background: #d4edda; border: 1px solid #c3e6cb; border-radius: 4px; padding: 12px; margin-bottom: 8px;">
                            <strong>✅ Lyckades!</strong> Genererade ${results.success} thumbnails.
                        </div>`;
                    }
                    
                    if (results.failed > 0) {
                        statusHtml += `<div style="color: #721c24; background: #f8d7da; border: 1px solid #f5c6cb; border-radius: 4px; padding: 12px; margin-bottom: 8px;">
                            <strong>❌ Misslyckades:</strong> ${results.failed} thumbnails kunde inte genereras.
                        </div>`;
                        
                        if (results.errors.length > 0) {
                            statusHtml += '<div style="margin-top: 8px;"><strong>Fel:</strong><ul style="margin: 4px 0 0 16px;">';
                            results.errors.forEach(error => {
                                statusHtml += `<li style="font-size: 12px; color: #666;">${error}</li>`;
                            });
                            statusHtml += '</ul></div>';
                        }
                    }
                    
                    // Debug information
                    if (debug.available_methods && debug.available_methods.length === 0) {
                        statusHtml += `<div style="color: #d32f2f; background: #ffebee; border: 1px solid #f5c6cb; border-radius: 4px; padding: 12px; margin-top: 8px;">
                            <strong>⚠️ Inga video thumbnail-metoder tillgängliga!</strong><br>
                            <strong>Debug info:</strong><br>
                            • ImageMagick: ${debug.imagick_loaded ? '✅ Laddad' : '❌ Inte laddad'}<br>
                            • FFmpeg: ${debug.ffmpeg_available ? '✅ Tillgänglig' : '❌ Inte tillgänglig'}<br>
                            • VLC: ${debug.vlc_available ? '✅ Tillgänglig' : '❌ Inte tillgänglig'}<br>
                            <em>Installera ImageMagick PHP extension för bästa resultat.</em>
                        </div>`;
                    }
                    
                    statusHtml += '</div>';
                    statusDiv.innerHTML = statusHtml;
                    statusDiv.style.display = 'block';
                    
                    // Reload page after 2 seconds to show updated statistics
                    setTimeout(() => {
                        window.location.reload();
                    }, 2000);
                    
                } else {
                    statusDiv.innerHTML = '<div style="color: #721c24; background: #f8d7da; border: 1px solid #f5c6cb; border-radius: 4px; padding: 12px;">❌ Ett fel uppstod vid generering av thumbnails.</div>';
                    statusDiv.style.display = 'block';
                }
                
            } catch (error) {
                hideProgressPopup();
                statusDiv.innerHTML = `<div style="color: #721c24; background: #f8d7da; border: 1px solid #f5c6cb; border-radius: 4px; padding: 12px;">❌ Ett fel uppstod: ${error.message}</div>`;
                statusDiv.style.display = 'block';
            }
            
            // Re-enable button
            button.disabled = false;
            button.textContent = 'Generera saknade thumbnails';
        }
        
    </script>
</body>
</html>
