<?php
require_once __DIR__ . '/../../lib/data_functions.php';

// Starta session om den inte redan är startad
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

header('Content-Type: application/json');

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['success' => false, 'error' => 'Method not allowed']);
    exit;
}

$action = $_POST['action'] ?? '';
$projectId = $_POST['project_id'] ?? '';
$files = $_POST['files'] ?? [];
$userId = $_POST['user_id'] ?? $_GET['user_id'] ?? $_SESSION['user_id'] ?? '';

if ($action !== 'create_package' || empty($projectId) || empty($files)) {
    echo json_encode(['success' => false, 'error' => 'Missing required parameters']);
    exit;
}

try {
    // Create downloads directory first
    $downloadsDir = __DIR__ . '/../../downloads';
    if (!is_dir($downloadsDir)) {
        if (!mkdir($downloadsDir, 0755, true)) {
            throw new Exception('Could not create downloads directory');
        }
    }
    
    // Generate unique filename
    $timestamp = date('Y-m-d_H-i-s');
    $zipFilename = "projekt_{$projectId}_{$timestamp}.zip";
    $zipPath = $downloadsDir . '/' . $zipFilename;
    
    // Create ZipArchive instance
    $zip = new ZipArchive();
    $result = $zip->open($zipPath, ZipArchive::CREATE);
    if ($result !== TRUE) {
        throw new Exception('Cannot create zip file. Error code: ' . $result);
    }
    
    $addedFiles = 0;
    $errors = [];
    $debugInfo = [
        'project_id' => $projectId,
        'files_count' => count($files),
        'downloads_dir' => $downloadsDir,
        'zip_path' => $zipPath
    ];
    
    // Beräkna projektroten (där index.html ligger)
    $projectRoot = realpath(__DIR__ . '/../../');
    
    foreach ($files as $filePath) {
        // Handle different path formats
        if (strpos($filePath, '/volume1/web/') === 0 && $projectRoot !== false) {
            // Absolut NAS-sökväg: /volume1/web/<site>[/...]
            // Ta bort /volume1/web/<site>/ och ersätt med lokal projektrot
            $relativeFromSiteRoot = preg_replace('#^/volume1/web/[^/]+/#', '', $filePath);
            $fullPath = $projectRoot . '/' . $relativeFromSiteRoot;
        } elseif (strpos($filePath, __DIR__ . '/../') === 0) {
            // Already relative to project root
            $fullPath = $filePath;
        } elseif (strpos($filePath, 'uploads/') === 0) {
            // Relative path starting with uploads/
            $fullPath = __DIR__ . '/../../' . $filePath;
        } else {
            // Try as absolute path first, then relative
            if (file_exists($filePath)) {
                $fullPath = $filePath;
            } else {
                $fullPath = __DIR__ . '/../../' . $filePath;
            }
        }
        
        // Add debug info for each file
        $debugInfo['file_paths'][] = [
            'original' => $filePath,
            'resolved' => $fullPath,
            'exists' => file_exists($fullPath),
            'size' => file_exists($fullPath) ? filesize($fullPath) : 0
        ];
        
        if (file_exists($fullPath)) {
            // Get original filename from metadata if available
            $metadataPath = str_replace(['.jpg', '.png', '.heic', '.heif', '.pdf', '.txt', '.csv'], ['.json', '.json', '.json', '.json', '.finfo', '.finfo', '.finfo'], $fullPath);
            $filename = basename($filePath);
            
            // Try to get original filename from metadata
            if (file_exists($metadataPath)) {
                $metadata = json_decode(file_get_contents($metadataPath), true);
                if ($metadata && isset($metadata['original'])) {
                    $filename = $metadata['original'];
                }
            }
            
            // Add file to zip with original filename
            if ($zip->addFile($fullPath, $filename)) {
                $addedFiles++;
            } else {
                $errors[] = "Could not add file: $filename";
            }
        } else {
            $errors[] = "File not found: $filePath (tried: $fullPath)";
        }
    }
    
    // Close the zip file
    $closeResult = $zip->close();
    if ($closeResult !== TRUE) {
        throw new Exception('Failed to close zip file. Error code: ' . $closeResult);
    }
    
    if ($addedFiles === 0) {
        // Clean up empty zip
        if (file_exists($zipPath)) {
            unlink($zipPath);
        }
        $errorMessage = 'No files could be added to the package';
        if (!empty($errors)) {
            $errorMessage .= '. Errors: ' . implode(', ', $errors);
        }
        echo json_encode([
            'success' => false, 
            'error' => $errorMessage, 
            'debug_info' => $debugInfo,
            'file_paths_debug' => $debugInfo['file_paths'] ?? []
        ]);
        exit;
    }
    
    // Set proper permissions
    chmod($zipPath, 0644);
    
    // Spara metadata för zip-filen (inklusive user_id)
    if (!empty($userId)) {
        $metadataPath = preg_replace('/\.zip$/', '.json', $zipPath);
        $metadata = [
            'filename' => $zipFilename,
            'project_id' => $projectId,
            'user_id' => $userId,
            'created_at' => date('c'),
            'files_count' => $addedFiles
        ];
        file_put_contents($metadataPath, json_encode($metadata, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
    }
    
    // Verify file exists and is readable
    if (!file_exists($zipPath)) {
        throw new Exception('Zip file was not created successfully');
    }
    
    if (!is_readable($zipPath)) {
        throw new Exception('Zip file is not readable');
    }
    
    // Check file size - should be more than just headers
    $fileSize = filesize($zipPath);
    if ($fileSize < 100) {
        throw new Exception("Zip file is too small ($fileSize bytes). Likely corrupted or empty.");
    }
    
    // Test if ZIP is valid by trying to open it
    $testZip = new ZipArchive();
    $testResult = $testZip->open($zipPath, ZipArchive::CHECKCONS);
    if ($testResult !== TRUE) {
        throw new Exception('Created ZIP file is corrupted. Error code: ' . $testResult);
    }
    $testZip->close();
    
    // Skapa bas-URL till adm-mappen dynamiskt baserat på nuvarande script-sökväg
    $script = $_SERVER['SCRIPT_NAME'] ?? '';
    $admPos = strpos($script, '/adm/');
    if ($admPos !== false) {
        // Behåll allt inklusive /adm
        $admBase = substr($script, 0, $admPos + 4);
    } else {
        $admBase = '/adm';
    }
    $admBase = rtrim($admBase, '/');
    
    // Create download URL - use PHP download handler (utan hårdkodad deployment-mapp)
    $downloadUrl = $admBase . '/download.php?file=' . urlencode($zipFilename);
    
    echo json_encode([
        'success' => true,
        'message' => "Paket skapat med $addedFiles filer",
        'filename' => $zipFilename,
        'download_url' => $downloadUrl,
        'files_added' => $addedFiles,
        'errors' => $errors,
        'debug_info' => $debugInfo
    ]);
    
} catch (Exception $e) {
    // Clean up temp file if it exists
    if (isset($zipPath) && file_exists($zipPath)) {
        unlink($zipPath);
    }
    
    
    echo json_encode([
        'success' => false,
        'error' => $e->getMessage()
    ]);
}
?>
