<?php
require_once __DIR__ . '/../admin_guard.php';
require_once __DIR__ . '/../../lib/data_functions.php';

// Hämta all data
$data = getAllUploadData();
$projectsData = $data['projectsData'];
$projectNames = $data['projectNames'];

// Hämta alla zip-filer från downloads-mappen
$downloadsDir = __DIR__ . '/../../downloads';
$packagedFiles = [];

if (is_dir($downloadsDir)) {
    $files = glob($downloadsDir . '/*.zip');
    foreach ($files as $file) {
        $filename = basename($file);
        $filePath = $file;
        $fileSize = file_exists($file) ? filesize($file) : 0;
        $fileDate = filemtime($file);
        
        // Extrahera projekt-ID från filnamnet (format: projekt_1005_2025-01-01_12-30-45.zip)
        $projectId = '';
        if (preg_match('/projekt_(\d+)_/', $filename, $matches)) {
            $projectId = $matches[1];
        }
        
        $packagedFiles[] = [
            'filename' => $filename,
            'filepath' => $filePath,
            'size' => $fileSize,
            'date' => $fileDate,
            'project_id' => $projectId,
            'project_name' => $projectNames[$projectId] ?? 'Okänt projekt'
        ];
    }
    
    // Sortera efter datum (nyaste först)
    usort($packagedFiles, function($a, $b) {
        return $b['date'] - $a['date'];
    });
}
?>
<!DOCTYPE html>
<html lang="sv">
<head>
    <meta charset="utf-8" />
    <meta name="viewport" content="width=device-width,initial-scale=1" />
    <title>Paketerade projekt - ByggArkiv</title>
    <link rel="stylesheet" href="../common.css">
    <?php
    // Beräkna dynamiska URL:er baserat på nuvarande script-sökväg
    $script = $_SERVER['SCRIPT_NAME'] ?? '';
    $admPos = strpos($script, '/adm/');
    $admBase = ($admPos !== false ? substr($script, 0, $admPos + 4) : '/adm');
    $admBase = rtrim($admBase, '/');
    $adminUrl = $admBase . '/admin.php';
    $downloadBaseUrl = $admBase . '/download.php';
    ?>
    <style>
        /* Sid-specifika stilar för paketerade_projekt.php */
        .filter-group {
            display: flex;
            flex-direction: column;
            gap: 4px;
            flex-direction: column;
        }
        .filter-group label {
            font-weight: 600;
            margin-top: 8px;
            padding: 8px;
            background: #e3f2fd;
            border-radius: 4px;
            font-style: italic;
            margin-bottom: 8px;
            color: #1976d2;
        }
        .filter-group select {
            padding: 10px;
            border: 1px solid #ccc;
            border-radius: 8px;
            font-size: 16px;
            background: white;
            width: 100% !important;
            min-width: 0 !important;
            max-width: 100% !important;
            box-sizing: border-box !important;
            flex-shrink: 1;
        }
        .filter-group select:focus {
            outline: none;
            border-color: #1a73e8;
            box-shadow: 0 0 0 2px rgba(26, 115, 232, 0.2);
        }
        .results-section {
            background: white;
            border-radius: 12px;
            box-shadow: 0 2px 8px rgba(0,0,0,0.1);
            overflow: hidden;
        }
        .results-header {
            background: linear-gradient(135deg, #1a73e8 0%, #1557b0 100%);
            color: white;
            padding: 20px;
            text-align: center;
        }
        .results-content {
            padding: 20px;
        }
        .file-item {
            display: flex;
            align-items: center;
            padding: 16px;
            border: 1px solid #e0e0e0;
            border-radius: 8px;
            margin-bottom: 12px;
            background: #fafafa;
            transition: all 0.2s ease;
        }
        .file-item:hover {
            background: #f0f8ff;
            border-color: #1a73e8;
            transform: translateY(-1px);
            box-shadow: 0 4px 12px rgba(26, 115, 232, 0.15);
        }
        .file-info {
            flex: 1;
            min-width: 0;
        }
        .file-name {
            font-weight: 600;
            font-size: 16px;
            color: #333;
            margin-bottom: 4px;
            word-break: break-all;
        }
        .file-details {
            display: flex;
            flex-wrap: wrap;
            gap: 16px;
            font-size: 14px;
            color: #666;
        }
        .file-detail {
            display: flex;
            align-items: center;
            gap: 4px;
        }
        .file-detail svg {
            width: 16px;
            height: 16px;
            opacity: 0.7;
        }
        .btn-primary {
            background: #1a73e8;
            color: white;
            border: none;
            padding: 8px 16px;
            border-radius: 6px;
            font-size: 14px;
            font-weight: 500;
            cursor: pointer;
            transition: background-color 0.2s;
            text-decoration: none;
            display: inline-flex;
            align-items: center;
            gap: 6px;
            white-space: nowrap;
        }
        .btn-primary:hover {
            background: #1557b0;
        }
        .btn-primary svg {
            width: 16px;
            height: 16px;
        }
        .btn-danger {
            background: #dc3545;
            color: white;
            border: none;
            padding: 8px 16px;
            border-radius: 6px;
            font-size: 14px;
            font-weight: 500;
            cursor: pointer;
            transition: background-color 0.2s;
            display: inline-flex;
            align-items: center;
            gap: 6px;
            white-space: nowrap;
        }
        .btn-danger:hover {
            background: #c82333;
        }
        .btn-danger svg {
            width: 16px;
            height: 16px;
        }
        .delete-btn svg {
            width: 16px;
            height: 16px;
        }
        .no-results {
            text-align: center;
            padding: 40px;
            color: #666;
            font-style: italic;
        }
        .stats {
            display: flex;
            gap: 20px;
            margin-bottom: 20px;
            flex-wrap: wrap;
        }
        .stat-item {
            background: #f8f9fa;
            padding: 12px 16px;
            border-radius: 8px;
            border-left: 4px solid #1a73e8;
        }
        .stat-label {
            font-size: 12px;
            color: #666;
            text-transform: uppercase;
            font-weight: 500;
        }
        .stat-value {
            font-size: 18px;
            font-weight: 600;
            color: #333;
        }
        @media (max-width: 768px) {
            .file-item {
                flex-direction: column;
                align-items: flex-start;
                gap: 12px;
            }
            .file-details {
                flex-direction: column;
                gap: 8px;
            }
            .download-btn {
                width: 100%;
                justify-content: center;
            }
        }
    </style>
</head>
<body>
    <div class="container">
        <header class="header">
            <a href="../../index.html">
                <img src="../../data/logo.png" alt="ByggArkiv Snickare Stefan" class="logo">
            </a>
            <div class="back-link">
                <a href="<?= htmlspecialchars($adminUrl) ?>">← Tillbaka till admin</a>
            </div>
        </header>

        <div class="filters">
            <h1>Paketerade projekt</h1>
            <?php if (isset($_GET['error'])): ?>
                <div class="error-message">
                    ❌ Ett fel uppstod: 
                    <?php
                    switch ($_GET['error']) {
                        case 'method_not_allowed':
                            echo 'Ogiltig metod.';
                            break;
                        case 'no_filename':
                            echo 'Inget filnamn angivet.';
                            break;
                        case 'invalid_filename':
                            echo 'Ogiltigt filnamn.';
                            break;
                        case 'delete_failed':
                            echo 'Kunde inte ta bort filen.';
                            break;
                        case 'file_not_found':
                            echo 'Filen hittades inte.';
                            break;
                        case 'exception':
                            echo 'Ett oväntat fel uppstod.';
                            break;
                        default:
                            echo 'Okänt fel.';
                    }
                    ?>
                </div>
            <?php endif; ?>
            <div class="filter-row">
                <div class="filter-group">
                    <label for="projectFilter">Filtrera efter projekt</label>
                    <select id="projectFilter">
                        <option value="">Alla projekt (<?= count($packagedFiles) ?> paket)</option>
                        <?php 
                        // Sortera projekt alfabetiskt efter namn
                        $sortedProjects = $projectsData;
                        usort($sortedProjects, function($a, $b) {
                            return strcasecmp($a['name'], $b['name']);
                        });
                        foreach ($sortedProjects as $project): ?>
                            <option value="<?= htmlspecialchars($project['id']) ?>">
                                <?= htmlspecialchars($project['name']) ?> (<?= htmlspecialchars($project['id']) ?>)
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
            </div>
        </div>

        <div class="results-section">
            <div class="results-header">
                <div class="results-header-title">
                    Paketerade projekt
                </div>
                <div class="results-header-subtitle">
                    Nedladdningsbara zip-filer med valda bilder och dokument
                </div>
            </div>
            <div class="results-content">
                <?php if (empty($packagedFiles)): ?>
                    <div class="no-results">
                        Inga paketerade projekt hittades. Skapa ditt första paket via "Paketera underlag för kund".
                    </div>
                <?php else: ?>
                    <?php foreach ($packagedFiles as $file): ?>
                        <div class="file-item" data-project="<?= htmlspecialchars($file['project_id']) ?>">
                            <div class="file-info">
                                <div class="file-name"><?= htmlspecialchars($file['filename']) ?></div>
                                <div class="file-details">
                                    <div class="file-detail">
                                        <svg width="16" height="16" viewBox="0 0 24 24" fill="currentColor">
                                            <path d="M12 2C6.48 2 2 6.48 2 12s4.48 10 10 10 10-4.48 10-10S17.52 2 12 2zm-2 15l-5-5 1.41-1.41L10 14.17l7.59-7.59L19 8l-9 9z"/>
                                        </svg>
                                        <?= htmlspecialchars($file['project_name']) ?>
                                    </div>
                                    <div class="file-detail">
                                        <svg width="16" height="16" viewBox="0 0 24 24" fill="currentColor">
                                            <path d="M12 2l3.09 6.26L22 9.27l-5 4.87 1.18 6.88L12 17.77l-6.18 3.25L7 14.14 2 9.27l6.91-1.01L12 2z"/>
                                        </svg>
                                        <?php
                                        if ($file['size'] >= 1024 * 1024 * 1024) {
                                            echo number_format($file['size'] / (1024 * 1024 * 1024), 1) . ' GB';
                                        } elseif ($file['size'] >= 1024 * 1024) {
                                            echo number_format($file['size'] / (1024 * 1024), 1) . ' MB';
                                        } else {
                                            echo number_format($file['size'] / 1024, 1) . ' KB';
                                        }
                                        ?>
                                    </div>
                                    <div class="file-detail">
                                        <svg width="16" height="16" viewBox="0 0 24 24" fill="currentColor">
                                            <path d="M19 3h-1V1h-2v2H8V1H6v2H5c-1.11 0-1.99.9-1.99 2L3 19c0 1.1.89 2 2 2h14c1.1 0 2-.9 2-2V5c0-1.1-.9-2-2-2zm0 16H5V8h14v11zM7 10h5v5H7z"/>
                                        </svg>
                                        <?= date('Y-m-d H:i', $file['date']) ?>
                                    </div>
                                </div>
                            </div>
                            <div class="file-actions">
                                <a href="<?= htmlspecialchars($downloadBaseUrl) ?>?file=<?= htmlspecialchars($file['filename']) ?>" class="btn-primary download-btn" download>
                                    Ladda ner
                                </a>
                                <button class="btn-danger" onclick="deletePackage('<?= htmlspecialchars($file['filename']) ?>')">
                                    Ta bort
                                </button>
                            </div>
                        </div>
                    <?php endforeach; ?>
                <?php endif; ?>
            </div>
        </div>
    </div>

    <script>
        console.log('paketerade_projekt.php JavaScript starting...');
        
        document.addEventListener('DOMContentLoaded', function() {
            console.log('DOM loaded, initializing...');
            const projectFilter = document.getElementById('projectFilter');
            const fileItems = document.querySelectorAll('.file-item');
            
            console.log('Project filter found:', projectFilter);
            console.log('File items found:', fileItems.length);
            
            function filterFiles() {
                const selectedProject = projectFilter.value;
                console.log('Filtering files for project:', selectedProject);
                
                fileItems.forEach(item => {
                    const itemProject = item.getAttribute('data-project');
                    
                    if (!selectedProject || itemProject === selectedProject) {
                        item.style.display = 'flex';
                    } else {
                        item.style.display = 'none';
                    }
                });
                
                console.log('Filtering complete');
            }
            
            projectFilter.addEventListener('change', filterFiles);
            
            // Add click logging for download links
            const downloadLinks = document.querySelectorAll('.download-btn');
            console.log('Download links found:', downloadLinks.length);
            
            downloadLinks.forEach((link, index) => {
                link.addEventListener('click', function(e) {
                    console.log('Download link clicked:', index, this.href);
                    console.log('Download target:', this.target);
                    console.log('Download attribute:', this.getAttribute('download'));
                });
            });
        });
        
        function deletePackage(filename) {
            console.log('Delete package called for:', filename);
            if (confirm('Är du säker på att du vill ta bort detta paket? Detta går inte att ångra.')) {
                console.log('User confirmed deletion');
                // Create form and submit to delete endpoint
                const form = document.createElement('form');
                form.method = 'POST';
                form.action = 'delete_package.php';
                
                const input = document.createElement('input');
                input.type = 'hidden';
                input.name = 'filename';
                input.value = filename;
                
                form.appendChild(input);
                document.body.appendChild(form);
                console.log('Submitting delete form...');
                form.submit();
            }
        }
    </script>
</body>
</html>
