<?php
require_once __DIR__ . '/../admin_guard.php';
require_once __DIR__ . '/../../config.php';
require_once __DIR__ . '/../../lib/data_functions.php';
require_once __DIR__ . '/../../lib/dal/repositories/ProjectRepository.php';

// Hämta parametrar från URL
$selectedProject = $_GET['project_id'] ?? '';
$showArchived = isset($_GET['show_archived']) && $_GET['show_archived'] === '1';

// Hämta all data
$data = getAllUploadData();
$projectsData = $data['projectsData'];
$allUploads = $data['allUploads'];
$userIds = $data['userIds'];
$projectNames = $data['projectNames'];

// Om arkiverade projekt ska visas, lägg till dem också
if ($showArchived) {
    $projectRepo = new ProjectRepository();
    $allProjects = $projectRepo->getAll();
    $archivedProjects = array_filter($allProjects, function($project) {
        return ($project['status'] ?? 'active') === 'archived';
    });
    // Lägg till arkiverade projekt i listan (sortera så att aktiva kommer först)
    $projectsData = array_merge($projectsData, $archivedProjects);
    foreach ($archivedProjects as $project) {
        $projectNames[$project['id']] = $project['name'];
    }
}

// Hämta användardata för att visa namn
$userNames = [];
$userData = [];
if (file_exists(__DIR__ . '/../../data/userids.json')) {
    $userContent = file_get_contents(__DIR__ . '/../../data/userids.json');
    if ($userContent !== false) {
        $userData = json_decode($userContent, true) ?: [];
        foreach ($userData as $user) {
            $userNames[$user['user_id']] = $user['full_name'];
        }
    }
}

// Funktion för att hämta alla filer (bilder och dokument) för ett projekt
function getAllProjectFiles($projectId) {
    // Använd absolut sökväg från projektets rot
    $uploadRoot = __DIR__ . '/../../uploads';
    $projectDir = $uploadRoot . '/' . $projectId;
    $files = [];
    
    if (!is_dir($projectDir)) {
        return $files;
    }
    
    // Hämta alla metadata-filer och sortera så att .finfo kommer först
    $finfoFiles = glob($projectDir . '/*.finfo');
    $jsonFiles = glob($projectDir . '/*.json');
    $allMetadataFiles = array_merge($finfoFiles, $jsonFiles);
    
    foreach ($allMetadataFiles as $metadataFile) {
        $metadata = json_decode(file_get_contents($metadataFile), true);
        if ($metadata) {
            $isImage = false;
            $isDocument = false;
            $actualFile = null;
            
            // Bestäm filtyp baserat på metadata-filens extension
            $metadataExtension = pathinfo($metadataFile, PATHINFO_EXTENSION);
            
            if ($metadataExtension === 'finfo') {
                // .finfo filer = dokument
                $docExtensions = ['.pdf', '.txt', '.csv'];
                foreach ($docExtensions as $ext) {
                    $docFile = str_replace('.finfo', $ext, $metadataFile);
                    if (file_exists($docFile)) {
                        $actualFile = $docFile;
                        $isDocument = true;
                        break;
                    }
                }
            } else {
                // .json filer = bilder och videor
                $imageExtensions = ['.jpg', '.png', '.heic', '.heif'];
                $videoExtensions = ['.mp4', '.webm', '.mov', '.avi'];
                $allMediaExtensions = array_merge($imageExtensions, $videoExtensions);
                
                foreach ($allMediaExtensions as $ext) {
                    $mediaFile = str_replace('.json', $ext, $metadataFile);
                    if (file_exists($mediaFile)) {
                        $actualFile = $mediaFile;
                        $isImage = in_array($ext, $imageExtensions);
                        break;
                    }
                }
            }
            
            if ($actualFile) {
                $metadata['file_path'] = $actualFile;
                
                // Bestäm filtyp korrekt
                $fileExtension = strtolower(pathinfo($actualFile, PATHINFO_EXTENSION));
                $isVideo = in_array($fileExtension, ['mp4', 'webm', 'mov', 'avi']);
                
                if ($isVideo) {
                    $metadata['file_type'] = 'video';
                } elseif ($isImage) {
                    $metadata['file_type'] = 'image';
                } else {
                    $metadata['file_type'] = 'document';
                }
                
                $metadata['project_id'] = $projectId;
                // Include actual file size from filesystem
                $metadata['size'] = file_exists($actualFile) ? filesize($actualFile) : 0;
                
                // Lägg till URL för visning/nedladdning via säker endpoint
                if ($isVideo) {
                    // För videor, använd view_file.php
                    $metadata['file_url'] = '../../view_file.php?project_id=' . urlencode($projectId) . '&file=' . urlencode(basename($actualFile));
                } elseif ($isImage) {
                    $metadata['file_url'] = '../../view_file.php?project_id=' . urlencode($projectId) . '&file=' . urlencode(basename($actualFile));
                    // Lägg till thumb_url om tumnagel finns (endast för bilder)
                    $baseNoExt = preg_replace('/\.(jpg|jpeg|png|webp|heic|heif)$/i', '', $actualFile);
                    $thumbPath = $baseNoExt . '-thumb.jpg';
                    if (file_exists($thumbPath)) {
                        $metadata['thumb_path'] = $thumbPath;
                        $metadata['thumb_url'] = '../../view_file.php?project_id=' . urlencode($projectId) . '&file=' . urlencode(basename($thumbPath));
                    }
                } else {
                    // För dokument, använd view_file.php
                    $metadata['file_url'] = '../../view_file.php?project_id=' . urlencode($projectId) . '&file=' . urlencode(basename($actualFile));
                }
                
                $files[] = $metadata;
            }
        }
    }
    
    // Sortera efter datum (nyaste först)
    usort($files, function($a, $b) {
        return strtotime($b['stored_at']) - strtotime($a['stored_at']);
    });
    
    return $files;
}

// Bestäm vilken vy som ska visas
$activeView = $selectedProject ? 'project' : '';
?>
<!DOCTYPE html>
<html lang="sv">
<head>
    <meta charset="utf-8" />
    <meta name="viewport" content="width=device-width,initial-scale=1" />
    <title>Paketera underlag till kund - ByggArkiv</title>
    <link rel="stylesheet" href="../common.css">
    <?php
    $script = $_SERVER['SCRIPT_NAME'] ?? '';
    $admPos = strpos($script, '/adm/');
    $adminUrl = ($admPos !== false ? substr($script, 0, $admPos + 4) : '/adm') . '/admin.php';
    ?>
    <style>
        /* Sid-specifika stilar för underlag.php */
        .filter-group {
            display: flex;
            flex-direction: column;
            gap: 4px;
            flex-direction: column;
        }
        .filter-group label {
            margin-top: 8px;
            padding: 8px;
            border-radius: 4px;
            margin-bottom: 8px;
        }
        /* Blått kort bakom huvudetiketten "Projekt" */
        .filter-group .primary-label {
            background: #e3f2fd;
            color: #1976d2;
        }
        .filter-group select {
            padding: 10px;
            border: 1px solid #ccc;
            border-radius: 8px;
            font-size: 16px;
            background: white;
            width: 100% !important;
            min-width: 0 !important;
            max-width: 100% !important;
            box-sizing: border-box !important;
            flex-shrink: 1;
        }
        .filter-group select:focus {
            outline: none;
            border-color: #1a73e8;
            box-shadow: 0 0 0 2px rgba(26, 115, 232, 0.2);
        }
        
        /* Ensure equal width for all select elements */
        #projectFilter, #userIdFilter {
            width: 100% !important;
            min-width: 0 !important;
            max-width: 100% !important;
            box-sizing: border-box !important;
        }
        .results-section {
            background: white;
            border-radius: 12px;
            box-shadow: 0 2px 8px rgba(0,0,0,0.1);
            overflow: hidden;
        }
        .results-header {
            background: #1a73e8;
            color: white;
            padding: 16px 20px;
            font-size: 18px;
            font-weight: 600;
        }
        .results-content {
            padding: 20px;
        }
        .file-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(300px, 1fr));
            gap: 20px;
        }
        .file-item {
            border: 1px solid #e0e0e0;
            border-radius: 8px;
            overflow: hidden;
            background: #fafafa;
            transition: transform 0.2s, box-shadow 0.2s;
        }
        .file-item:hover {
            transform: translateY(-2px);
            box-shadow: 0 4px 12px rgba(0,0,0,0.15);
        }
        .file-image {
            width: 100%;
            height: 200px;
            object-fit: cover;
            background: #f0f0f0;
            transition: opacity 0.2s;
            pointer-events: auto;
            z-index: 1;
            cursor: default;
        }
        .file-info {
            padding: 16px;
        }
        .file-project {
            font-weight: 600;
            color: #1a73e8;
            margin-top: 8px;
            padding: 0;
            background: none;
            border-radius: 0;
            font-style: italic;
            margin-bottom: 8px;
        }
        .file-project.no-background {
            background: none;
            padding: 0;
            margin-bottom: 4px;
        }
        .file-project-name {
            font-size: 12px;
            color: #666;
            font-weight: normal;
            margin-top: 2px;
        }
        .file-note {
            margin-top: 6px;
            padding: 6px 8px 2px 8px;
            background: #e3f2fd;
            border-radius: 4px;
            font-style: normal;
            font-weight: normal;
            color: #000;
            margin-bottom: 2px;
            word-wrap: break-word;
        }
        
        .edit-btn {
            position: absolute;
            top: 8px;
            right: 119px;
            background: white;
            color: #ff9800;
            border: 2px solid #ff9800;
            border-radius: 50%;
            width: 30px;
            height: 30px;
            font-size: 16px;
            cursor: pointer;
            display: flex;
            align-items: center;
            justify-content: center;
            opacity: 0.9;
            transition: opacity 0.2s;
            z-index: 10;
            pointer-events: auto;
        }
        .edit-btn:hover {
            opacity: 1;
            background: #ff9800;
            color: white;
        }
        .file-meta {
            font-size: 12px;
            color: #666;
            display: flex;
            flex-direction: column;
            gap: 4px;
            justify-content: flex-start;
            margin-bottom: -4px;
        }
        .file-meta-row {
            display: flex;
            justify-content: space-between;
            align-items: center;
            gap: 8px;
        }
        .file-date {
            flex: 1;
        }
        .file-user-id {
            color: #666;
            font-weight: normal;
        }

        .file-filename {
            font-size: 12px;
            color: #1a73e8;
            text-decoration: underline;
        }
        .no-results {
            text-align: center;
            padding: 40px;
            font-size: 11px;
            color: #1a73e8;
            word-break: break-all;
            text-decoration: none;
        }
        .no-results:hover {
            text-decoration: underline;
        }
        .no-results {
            color: #666;
            font-style: italic;
        }

        .delete-btn {
            position: absolute;
            top: 8px;
            right: 8px;
            background: white;
            color: #dc3545;
            border: 2px solid #dc3545;
            border-radius: 50%;
            width: 30px;
            height: 30px;
            font-size: 16px;
            cursor: pointer;
            display: flex;
            align-items: center;
            justify-content: center;
            opacity: 0.9;
            transition: opacity 0.2s;
            z-index: 10;
            pointer-events: auto;
        }
        .delete-btn:hover {
            opacity: 1;
            background: #dc3545;
            color: white;
        }
        
        .move-btn {
            position: absolute;
            top: 8px;
            right: 45px;
            background: white;
            color: #28a745;
            border: 2px solid #28a745;
            border-radius: 50%;
            width: 30px;
            height: 30px;
            font-size: 16px;
            cursor: pointer;
            display: flex;
            align-items: center;
            justify-content: center;
            opacity: 0.9;
            transition: opacity 0.2s;
            z-index: 10;
            pointer-events: auto;
        }
        .move-btn:hover {
            opacity: 1;
            background: #28a745;
            color: white;
        }
        
        .view-btn {
            position: absolute;
            top: 8px;
            right: 8px;
            background: white;
            color: #1a73e8;
            border: 2px solid #1a73e8;
            border-radius: 50%;
            width: 30px;
            height: 30px;
            font-size: 16px;
            cursor: pointer;
            display: flex;
            align-items: center;
            justify-content: center;
            opacity: 0.9;
            transition: opacity 0.2s;
            z-index: 10;
            pointer-events: auto;
        }
        .view-btn:hover {
            opacity: 1;
            background: #1a73e8;
            color: white;
        }
        /* Ensure SVG icon inside view button is visible and properly sized */
        .view-btn svg {
            width: 16px;
            height: 16px;
            display: block;
            fill: currentColor;
        }
        
        .file-item {
            position: relative;
        }
        
        /* Checkbox styles for file selection */
        .file-checkbox {
            position: absolute;
            top: 8px;
            left: 8px;
            width: 20px;
            height: 20px;
            background: white;
            border: 2px solid #1a73e8;
            border-radius: 4px;
            cursor: pointer;
            z-index: 10;
            display: flex;
            align-items: center;
            justify-content: center;
            transition: all 0.2s;
        }
        
        .file-checkbox:hover {
            background: #e3f2fd;
        }
        
        .file-checkbox.checked {
            background: #1a73e8;
            color: white;
        }
        
        .file-checkbox.checked::after {
            content: '✓';
            font-weight: bold;
            font-size: 14px;
        }
        
        /* Package button styles */
        .package-section {
            background: white;
            border-radius: 12px;
            padding: 20px;
            margin-bottom: 24px;
            box-shadow: 0 2px 8px rgba(0,0,0,0.1);
            display: block;
        }
        
        .package-controls {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 16px;
        }
        
        .package-info {
            font-size: 14px;
            color: #666;
        }
        
        .package-buttons {
            display: flex;
            gap: 12px;
        }
        
        .btn {
            padding: 10px 20px;
            border: none;
            border-radius: 8px;
            cursor: pointer;
            font-size: 14px;
            font-weight: 600;
            transition: all 0.2s;
        }
        
        .btn-primary {
            background: #28a745;
            color: white;
        }
        
        .btn-primary:hover {
            background: #218838;
        }
        
        .btn-secondary {
            background: #f8f9fa;
            color: #666;
            border: 1px solid #ddd;
        }
        
        .btn-secondary:hover {
            background: #e9ecef;
        }
        
        .btn:disabled {
            opacity: 0.5;
            cursor: not-allowed;
        }
        
        /* Document icon styles - same as images but with icon */
        .document-icon {
            width: 100%;
            height: 200px;
            object-fit: cover;
            background: #f0f0f0;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 48px;
            color: #666;
        }
        
        /* Lightbox Modal Styles */
        .lightbox {
            display: none;
            position: fixed;
            z-index: 1000;
            left: 0;
            top: 0;
            width: 100%;
            height: 100%;
            background-color: rgba(0, 0, 0, 0.9);
            backdrop-filter: blur(5px);
        }
        
        .lightbox-content {
            position: relative;
            margin: auto;
            padding: 20px;
            width: 90%;
            height: 90%;
            display: flex;
            align-items: center;
            justify-content: center;
        }
        
        .lightbox-image {
            max-width: 100%;
            max-height: 100%;
            object-fit: contain;
            border-radius: 8px;
            box-shadow: 0 4px 20px rgba(0, 0, 0, 0.5);
        }
        
        .lightbox-close {
            position: absolute;
            top: 20px;
            right: 35px;
            color: #f1f1f1;
            font-size: 40px;
            font-weight: bold;
            cursor: pointer;
            z-index: 1001;
            background: rgba(0, 0, 0, 0.5);
            border-radius: 50%;
            width: 50px;
            height: 50px;
            display: flex;
            align-items: center;
            justify-content: center;
            transition: background-color 0.2s;
        }
        
        .lightbox-close:hover {
            background: rgba(0, 0, 0, 0.8);
        }
        
        .lightbox-info {
            position: absolute;
            bottom: 20px;
            left: 20px;
            right: 20px;
            background: rgba(0, 0, 0, 0.8);
            color: white;
            padding: 15px;
            border-radius: 8px;
            font-size: 14px;
        }
        
        .lightbox-filename {
            font-weight: bold;
            margin-bottom: 5px;
        }
        
        .lightbox-meta {
            display: flex;
            justify-content: space-between;
            flex-wrap: wrap;
            gap: 10px;
        }
        
        @media (max-width: 768px) {
            .filter-row {
                grid-template-columns: 1fr;
                gap: 16px;
            }
            .file-grid {
                grid-template-columns: 1fr;
            }
            .lightbox-content {
                width: 95%;
                height: 95%;
                padding: 10px;
            }
            .lightbox-close {
                top: 10px;
                right: 20px;
                font-size: 30px;
                width: 40px;
                height: 40px;
            }
            .lightbox-info {
                bottom: 10px;
                left: 10px;
                right: 10px;
                padding: 10px;
                font-size: 12px;
            }
            .lightbox-meta {
                flex-direction: column;
                gap: 5px;
            }
        }
    </style>
</head>
<body>
    <div class="container">
        <header class="header">
            <a href="../../index.html">
                <img src="../../data/logo.png" alt="ByggArkiv Snickare Stefan" class="logo">
            </a>
            <div class="back-link">
                <a href="<?= htmlspecialchars($adminUrl) ?>">← Tillbaka till admin</a>
            </div>
        </header>

        <div class="filters">
            <h1>Paketering till kund</h1>
            <div class="filter-row">
                <div class="filter-group max-width-600">
                    <label for="projectFilter" class="primary-label">Projekt</label>
                    <select id="projectFilter">
                        <option value="">Välj projekt (<?= count($projectsData) ?> tillgängliga)</option>
                        <?php 
                        // Sortera projekt alfabetiskt på namn
                        $sortedProjects = $projectsData;
                        usort($sortedProjects, function($a, $b) {
                            // Sortera så att aktiva kommer först, sedan arkiverade
                            $aStatus = $a['status'] ?? 'active';
                            $bStatus = $b['status'] ?? 'active';
                            $aArchived = $aStatus === 'archived';
                            $bArchived = $bStatus === 'archived';
                            
                            if ($aArchived && !$bArchived) {
                                return 1; // a är arkiverad, b är aktiv - a ska hamna efter b
                            }
                            if (!$aArchived && $bArchived) {
                                return -1; // a är aktiv, b är arkiverad - a ska hamna före b
                            }
                            
                            // Om båda har samma status, sortera alfabetiskt
                            return strcasecmp($a['name'], $b['name']);
                        });
                        // Om valt projekt inte finns i listan (t.ex. arkiverat), inkludera det som förvalt
                        $existingIds = array_map(function($p){ return $p['id']; }, $sortedProjects);
                        if ($selectedProject && !in_array($selectedProject, $existingIds, true)) {
                            $projectRepo = new ProjectRepository();
                            $allProjects = $projectRepo->getAll();
                            foreach ($allProjects as $p) {
                                if (($p['id'] ?? '') === $selectedProject) {
                                    $name = $p['name'] ?? $selectedProject;
                                    $status = $p['status'] ?? 'active';
                                    $statusLabel = $status === 'archived' ? ' [arkiverat]' : '';
                                    ?>
                                    <option value="<?= htmlspecialchars($selectedProject) ?>" selected>
                                        <?= htmlspecialchars($name) ?> (<?= htmlspecialchars($selectedProject) ?>)<?= htmlspecialchars($statusLabel) ?>
                                    </option>
                                    <?php
                                    break;
                                }
                            }
                        }

                        foreach ($sortedProjects as $project): 
                            $status = $project['status'] ?? 'active';
                            $statusLabel = $status === 'archived' ? ' [arkiverat]' : '';
                            $selected = ($project['id'] == $selectedProject) ? 'selected' : '';
                        ?>
                            <option value="<?= htmlspecialchars($project['id']) ?>" <?= $selected ?>>
                                <?= htmlspecialchars($project['name']) ?> (<?= htmlspecialchars($project['id']) ?>)<?= htmlspecialchars($statusLabel) ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                    <label style="margin-top: 8px; display: flex; align-items: center; gap: 6px; font-size: 14px; color: #555; background-color: #f5f5f5; padding: 6px 10px; border-radius: 4px; font-weight: normal;">
                        <input type="checkbox"
                               id="showArchived"
                               <?= $showArchived ? 'checked' : '' ?>
                               onchange="toggleShowArchived(this)">
                        Visa även arkiverade
                    </label>
                </div>
            </div>
        </div>

        <!-- Package section -->
        <div id="packageSection" class="package-section">
            <div class="package-controls">
                <div class="package-info">
                    <span id="selectedCount">0</span> filer valda
                    <span id="selectedSizes" class="selected-sizes"></span>
                </div>
                <div class="package-buttons">
                    <button class="btn btn-secondary" onclick="selectAll()">Välj alla</button>
                    <button class="btn btn-secondary" onclick="selectNone()">Välj inga</button>
                    <button class="btn btn-primary" id="packageBtn" onclick="createPackage()" disabled>Paketera</button>
                </div>
            </div>
        </div>

        <div id="resultsContainer">
            <?php if ($activeView === 'project'): ?>
                <?php
                // Hämta alla filer för det valda projektet
                $projectFiles = getAllProjectFiles($selectedProject);
                $displayProjectName = $projectNames[$selectedProject] ?? $selectedProject;
                ?>
                <div class="results-section">
                    <div class="results-header">
                        <div class="results-header-title">
                            <?= htmlspecialchars($displayProjectName) ?>
                        </div>
                        <div class="results-header-subtitle">
                            Projekt: <?= htmlspecialchars($selectedProject) ?>
                            <span>(<?= count($projectFiles) ?> filer)</span>
                        </div>
                    </div>
                    <div class="results-content">
                        <?php if (empty($projectFiles)): ?>
                            <div class="no-results">
                                Inga filer hittades för det valda projektet.
                            </div>
                        <?php else: ?>
                            <div class="file-grid">
                                <?php foreach ($projectFiles as $file): ?>
                                    <?php 
                                    $uid = $file['user_id'] ?? '';
                                    $uname = getUserFullName($uid);
                                    $displayUser = $uname !== '' ? $uname : 'Ingen användare';
                                    $displayDate = date('Y-m-d H:i', strtotime($file['stored_at']));
                                    $filename = basename($file['file_path']);
                                    $note = $file['note'] ?? '';
                                    $originalName = $file['original'] ?? $filename;
                                    $isVideo = ($file['file_type'] ?? 'image') === 'video';
                                    $imageNumber = isset($file['image_number']) && is_numeric($file['image_number']) ? (int)$file['image_number'] : null;
                                    $imageNumberText = $imageNumber !== null ? ($isVideo ? 'Video #' . $imageNumber : 'Bild #' . $imageNumber) : '';
                                    ?>
                                    <div class="file-item" 
                                         data-project="<?= htmlspecialchars($file['project_id']) ?>"
                                         data-file-path="<?= htmlspecialchars($file['file_path']) ?>"
                                         data-file-type="<?= htmlspecialchars($file['file_type']) ?>"
                                         data-file-size="<?= htmlspecialchars((string)($file['size'] ?? '0')) ?>"
                                         data-filename="<?= htmlspecialchars($originalName) ?>"
                                         data-date="<?= htmlspecialchars($displayDate) ?>"
                                         data-user="<?= htmlspecialchars($displayUser) ?>"
                                         data-note="<?= htmlspecialchars($note) ?>"
                                         >
                                        <div class="file-checkbox" onclick="toggleFileSelection(this)" data-file-path="<?= htmlspecialchars($file['file_path']) ?>" data-file-type="<?= htmlspecialchars($file['file_type']) ?>"></div>
                                        
                                        <?php 
                                        // Kontrollera filtyp baserat på filändelse om file_type inte är satt
                                        $extension = strtolower(pathinfo($file['file_path'], PATHINFO_EXTENSION));
                                        $isImageFile = in_array($extension, ['jpg', 'jpeg', 'png', 'heic', 'heif']);
                                        $isVideoFile = in_array($extension, ['mp4', 'webm', 'mov', 'avi']);
                                        $isDocumentFile = in_array($extension, ['pdf', 'txt', 'csv']);
                                        
                                        if ($file['file_type'] === 'document'): ?>
                                            <div class="document-icon">
                                                📄
                                            </div>
                                        <?php elseif ($file['file_type'] === 'video' || $isVideoFile): ?>
                                            <button class="view-btn" title="Visa video i full storlek" onclick="openLightbox('<?= htmlspecialchars($file['file_url']) ?>', '<?= htmlspecialchars($originalName) ?>', '<?= htmlspecialchars($displayDate) ?>', '<?= htmlspecialchars($displayUser) ?>', '<?= htmlspecialchars($note) ?>', '<?= htmlspecialchars($imageNumberText) ?>')"><img src="../icons/view.svg" width="16" height="16" alt="Visa"></button>
                                            <video class="file-image" 
                                                   controls 
                                                   preload="metadata"
                                                   onclick="openLightbox('<?= htmlspecialchars($file['file_url']) ?>', '<?= htmlspecialchars($originalName) ?>', '<?= htmlspecialchars($displayDate) ?>', '<?= htmlspecialchars($displayUser) ?>', '<?= htmlspecialchars($note) ?>', '<?= htmlspecialchars($imageNumberText) ?>')">
                                                <source src="<?= htmlspecialchars($file['file_url']) ?>" type="<?= htmlspecialchars($file['mime'] ?? 'video/mp4') ?>">
                                                Din webbläsare stöder inte videouppspelning.
                                            </video>
                                        <?php else: ?>
                                            <button class="view-btn" title="Visa bild i full storlek" onclick="openLightbox('<?= htmlspecialchars($file['thumb_url'] ?? $file['file_url']) ?>', '<?= htmlspecialchars($originalName) ?>', '<?= htmlspecialchars($displayDate) ?>', '<?= htmlspecialchars($displayUser) ?>', '<?= htmlspecialchars($note) ?>', '<?= htmlspecialchars($imageNumberText) ?>')"><img src="../icons/view.svg" width="16" height="16" alt="Visa"></button>
                                            <img src="<?= htmlspecialchars($file['thumb_url'] ?? $file['file_url']) ?>" 
                                                 alt="Bild" 
                                                 class="file-image"
                                                 loading="lazy"
                                                 onerror="this.style.display='none'"
                                                 onclick="openLightbox('<?= htmlspecialchars($file['thumb_url'] ?? $file['file_url']) ?>', '<?= htmlspecialchars($originalName) ?>', '<?= htmlspecialchars($displayDate) ?>', '<?= htmlspecialchars($displayUser) ?>', '<?= htmlspecialchars($note) ?>', '<?= htmlspecialchars($imageNumberText) ?>')">
                                        <?php endif; ?>
                                        
                                        <div class="file-info">
                                            <?php if ($imageNumberText): ?>
                                                <div class="file-image-number" style="color: #000; margin-bottom: 4px;">
                                                    <?= htmlspecialchars($imageNumberText) ?>
                                                </div>
                                            <?php endif; ?>
                                            <div class="file-meta">
                                                <div class="file-meta-row">
                                                    <span class="file-date"><?= $displayDate ?>    <?= number_format(($file['bytes'] ?? 0) / 1024 / 1024, 1) ?>M</span>
                                                    <span class="file-user-id">
                                                        <?= htmlspecialchars($displayUser) ?>
                                                    </span>
                                                </div>
                                            </div>
                                            <a href="<?= htmlspecialchars($file['file_url']) ?>" class="file-filename" download><?= htmlspecialchars($originalName) ?></a>
                                            <?php if (!empty($note)): ?>
                                                <div class="file-note">
                                                    <?= htmlspecialchars($note) ?>
                                                </div>
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                <?php endforeach; ?>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
            <?php else: ?>
                <div class="results-section">
                    <div class="results-header">
                        <span id="resultsTitle">Välj ett projekt för att visa filer</span>
                        <span id="resultsCount"></span>
                    </div>
                    <div class="results-content">
                        <div class="no-results">
                            Välj ett projekt ovan för att visa alla filer (bilder, videor och dokument).
                        </div>
                    </div>
                </div>
            <?php endif; ?>
        </div>
    </div>

    <!-- Lightbox Modal -->
    <div id="lightbox" class="lightbox">
        <div class="lightbox-content">
            <span class="lightbox-close" onclick="closeLightbox()">&times;</span>
            <img id="lightbox-image" class="lightbox-image" src="" alt="" style="display: none;">
            <video id="lightbox-video" class="lightbox-image" controls style="display: none;">
                Din webbläsare stöder inte videouppspelning.
            </video>
            <div class="lightbox-info">
                <div id="lightbox-image-number" class="lightbox-filename" style="font-weight: bold; color: #fff; margin-bottom: 5px;"></div>
                <div id="lightbox-filename" class="lightbox-filename"></div>
                <div class="lightbox-meta">
                    <span id="lightbox-date"></span>
                    <span id="lightbox-user"></span>
                </div>
                <div id="lightbox-note" class="lightbox-note"></div>
            </div>
        </div>
    </div>

    <script>
        
        // Lightbox functions
        function openLightbox(imageSrc, filename, date, user, note, imageNumber) {
            // Convert thumbnail URL to full-size image URL
            if (imageSrc && imageSrc.includes('-thumb.')) {
                imageSrc = imageSrc.replace('-thumb.', '.');
            }
            
            const lightbox = document.getElementById('lightbox');
            const lightboxImage = document.getElementById('lightbox-image');
            const lightboxVideo = document.getElementById('lightbox-video');
            const lightboxImageNumber = document.getElementById('lightbox-image-number');
            const lightboxFilename = document.getElementById('lightbox-filename');
            const lightboxDate = document.getElementById('lightbox-date');
            const lightboxUser = document.getElementById('lightbox-user');
            const lightboxNote = document.getElementById('lightbox-note');
            
            if (!lightbox || !lightboxImage || !lightboxVideo) {
                console.error('Lightbox elements not found');
                return;
            }
            
            // Kontrollera om det är en video baserat på filnamn
            const isVideo = /\.(mp4|webm|mov|avi)$/i.test(filename);
            
            if (isVideo) {
                // Visa video, dölj bild
                lightboxImage.style.display = 'none';
                lightboxVideo.style.display = 'block';
                lightboxVideo.src = imageSrc;
                lightboxVideo.load(); // Ladda videon
            } else {
                // Visa bild, dölj video
                lightboxVideo.style.display = 'none';
                lightboxImage.style.display = 'block';
                lightboxImage.src = imageSrc;
            }
            
            // Visa bildnummer om det finns
            if (imageNumber && imageNumber.trim() !== '') {
                lightboxImageNumber.textContent = imageNumber;
                lightboxImageNumber.style.display = 'block';
            } else {
                lightboxImageNumber.style.display = 'none';
            }
            
            lightboxFilename.textContent = filename;
            lightboxDate.textContent = date;
            lightboxUser.textContent = user;
            lightboxNote.textContent = note || '';
            
            lightbox.style.display = 'block';
            document.body.style.overflow = 'hidden'; // Prevent background scrolling
        }
        
        function closeLightbox() {
            const lightbox = document.getElementById('lightbox');
            const lightboxVideo = document.getElementById('lightbox-video');
            if (lightbox) {
                lightbox.style.display = 'none';
                document.body.style.overflow = 'auto'; // Restore scrolling
                
                // Pausa video om den spelar
                if (lightboxVideo) {
                    lightboxVideo.pause();
                    lightboxVideo.currentTime = 0;
                }
            }
        }
        
        // Close lightbox when clicking outside the image
        document.addEventListener('click', function(event) {
            const lightbox = document.getElementById('lightbox');
            if (event.target === lightbox) {
                closeLightbox();
            }
        });
        
        // Close lightbox with Escape key
        document.addEventListener('keydown', function(event) {
            if (event.key === 'Escape') {
                closeLightbox();
            }
        });

        // File selection functionality
        let selectedFiles = new Set();
        
        async function toggleFileSelection(checkbox) {
            const filePath = checkbox.dataset.filePath;
            const fileType = checkbox.dataset.fileType;
            
            if (checkbox.classList.contains('checked')) {
                checkbox.classList.remove('checked');
                selectedFiles.delete(filePath);
            } else {
                checkbox.classList.add('checked');
                selectedFiles.add(filePath);
            }
            
            // Update UI feedback
            const selectedCountEl = document.getElementById('selectedCount');
            if (selectedCountEl) selectedCountEl.textContent = `${selectedFiles.size}`;
            
            await updatePackageSection();
        }
        
        async function selectAll() {
            const checkboxes = document.querySelectorAll('.file-checkbox');
            checkboxes.forEach(checkbox => {
                if (!checkbox.classList.contains('checked')) {
                    checkbox.classList.add('checked');
                    selectedFiles.add(checkbox.dataset.filePath);
                }
            });
            await updatePackageSection();
        }
        
        async function selectNone() {
            const checkboxes = document.querySelectorAll('.file-checkbox');
            checkboxes.forEach(checkbox => {
                checkbox.classList.remove('checked');
            });
            selectedFiles.clear();
            await updatePackageSection();
        }
        
        async function updatePackageSection() {
            const packageSection = document.getElementById('packageSection');
            const selectedCount = document.getElementById('selectedCount');
            const packageBtn = document.getElementById('packageBtn');
            
            // Package section is always visible now
            packageSection.style.display = 'block';
            
            if (selectedFiles.size > 0) {
                // Räkna dokument och bilder
                let documentCount = 0;
                let imageCount = 0;
                let documentSize = 0;
                let imageSize = 0;
                
                // Summera filstorlekar för alla valda filer (från data-attribut)
                for (const filePath of selectedFiles) {
                    const file = allFiles.find(f => f.file_path === filePath);
                    if (file) {
                        const actualSize = parseInt(file.size || 0);
                        if (file.file_type === 'document') {
                            documentCount++;
                            documentSize += actualSize;
                        } else {
                            imageCount++;
                            imageSize += actualSize;
                        }
                    }
                }
                
                // Formatera storlekar
                const formatSize = (bytes) => {
                    if (bytes === 0) return '0 B';
                    const k = 1024;
                    const sizes = ['B', 'KB', 'MB', 'GB'];
                    const i = Math.floor(Math.log(bytes) / Math.log(k));
                    return parseFloat((bytes / Math.pow(k, i)).toFixed(1)) + ' ' + sizes[i];
                };
                
                // Uppdatera räknare med detaljerad information
                // Uppdatera separat: antal och storlekar
                selectedCount.textContent = `${selectedFiles.size}`;
                const sizesEl = document.getElementById('selectedSizes');
                let sizesText = '';
                if (imageCount > 0) sizesText += `${imageCount} bilder (${formatSize(imageSize)})`;
                if (documentCount > 0) sizesText += `${sizesText ? ' • ' : ''}${documentCount} dokument (${formatSize(documentSize)})`;
                sizesEl.textContent = sizesText;
                packageBtn.disabled = false;
                packageBtn.textContent = `Paketera (${selectedFiles.size} filer)`;
            } else {
                // Inga filer valda
                selectedCount.textContent = '0';
                const sizesEl = document.getElementById('selectedSizes');
                sizesEl.textContent = '';
                packageBtn.disabled = true;
                packageBtn.textContent = 'Paketera (0 filer)';
            }
        }
        
        async function createPackage() {
            if (selectedFiles.size === 0) {
                alert('Välj minst en fil att paketera');
                return;
            }
            
            const packageBtn = document.getElementById('packageBtn');
            const originalText = packageBtn.textContent;
            packageBtn.textContent = 'Skapar paket...';
            packageBtn.disabled = true;
            
            // Show popup while creating package
            const overlay = document.createElement('div');
            overlay.className = 'confirm-overlay';
            overlay.id = 'packagePopup';
            
            const popup = document.createElement('div');
            popup.className = 'confirm-box';
            popup.innerHTML = `
                <h3 class="confirm-title">📦 Skapar zip-paket...</h3>
                <p class="confirm-message">Detta kan ta en stund beroende på antal filer, nedladdning startar när zip-paketet är klart</p>
            `;
            
            overlay.appendChild(popup);
            document.body.appendChild(overlay);
            
            try {
                const formData = new FormData();
                formData.append('action', 'create_package');
                formData.append('project_id', document.getElementById('projectFilter').value);
                
                // Add user_id from URL, PHP session variable, or empty string
                const urlParams = new URLSearchParams(window.location.search);
                const phpUserId = '<?= htmlspecialchars($_SESSION['user_id'] ?? $_GET['user_id'] ?? '', ENT_QUOTES, 'UTF-8') ?>';
                const userId = urlParams.get('user_id') || phpUserId || '';
                if (userId) {
                    formData.append('user_id', userId);
                }
                
                // Add selected files
                selectedFiles.forEach(filePath => {
                    formData.append('files[]', filePath);
                });
                
                const response = await fetch('create_package.php', {
                    method: 'POST',
                    body: formData
                });
                
                if (!response.ok) {
                    throw new Error(`HTTP error! status: ${response.status}`);
                }
                
                const responseText = await response.text();
                console.log('Raw response:', responseText);
                
                let result;
                try {
                    result = JSON.parse(responseText);
                } catch (parseError) {
                    console.error('JSON parse error:', parseError);
                    console.error('Response text:', responseText);
                    throw new Error('Invalid JSON response: ' + parseError.message);
                }
                
                if (result.success) {
                    // Remove popup
                    const popup = document.getElementById('packagePopup');
                    if (popup) {
                        popup.remove();
                    }
                    
                    // Use download URL directly - force download
                    const link = document.createElement('a');
                    link.href = result.download_url;
                    link.download = result.filename;
                    link.target = '_blank';
                    document.body.appendChild(link);
                    link.click();
                    document.body.removeChild(link);
                } else {
                    let errorMsg = 'Fel: ' + result.error;
                    if (result.debug_info) {
                        errorMsg += '\n\nDebug info:\n';
                        errorMsg += 'Zip path: ' + result.debug_info.zip_path + '\n';
                        errorMsg += 'Downloads dir: ' + result.debug_info.downloads_dir + '\n';
                        errorMsg += 'Project ID: ' + result.debug_info.project_id + '\n';
                        errorMsg += 'Files count: ' + result.debug_info.files_count;
                    }
                    if (result.file_paths_debug) {
                        errorMsg += '\n\nFile paths debug:\n';
                        result.file_paths_debug.forEach((file, index) => {
                            errorMsg += `${index + 1}. Original: ${file.original}\n`;
                            errorMsg += `   Resolved: ${file.resolved}\n`;
                            errorMsg += `   Exists: ${file.exists}\n`;
                            errorMsg += `   Size: ${file.size} bytes\n\n`;
                        });
                    }
                    alert(errorMsg);
                }
            } catch (error) {
                // Remove popup on error
                const popup = document.getElementById('packagePopup');
                if (popup) {
                    popup.remove();
                }
                
                console.error('Package creation error:', error);
                alert('Ett fel uppstod: ' + error.message);
            } finally {
                packageBtn.textContent = originalText;
                packageBtn.disabled = false;
            }
        }

        // Wait for DOM to be ready
        document.addEventListener('DOMContentLoaded', function() {
            const projectFilter = document.getElementById('projectFilter');
            const resultsContainer = document.getElementById('resultsContainer');
            
            // Samla alla filer från sidan för att kunna räkna storlekar
            function collectAllFiles() {
                allFiles = [];
                const fileItems = document.querySelectorAll('.file-item');
                fileItems.forEach(item => {
                    const filePath = item.getAttribute('data-file-path');
                    const fileType = item.getAttribute('data-file-type');
                    const fileSize = item.getAttribute('data-file-size');
                    if (filePath) {
                        allFiles.push({
                            file_path: filePath,
                            file_type: fileType || 'image',
                            size: fileSize || '0'
                        });
                    }
                });
            }
            
            // Ingen AJAX behövs; storlek kommer från servern via data-attribut
            
            // Samla filer när sidan laddas
            collectAllFiles();

            function navigateToFilter() {
                const selectedProject = projectFilter.value;
                const showArchived = document.getElementById('showArchived').checked;
                
                // Build URL with project filter and archived flag
                const params = new URLSearchParams();
                
                if (selectedProject) {
                    params.append('project_id', selectedProject);
                }
                
                if (showArchived) {
                    params.append('show_archived', '1');
                }
                
                // Navigate to the appropriate URL
                const queryString = params.toString();
                if (queryString) {
                    window.location.href = `?${queryString}`;
                } else {
                    window.location.href = window.location.pathname;
                }
            }

            // Event listeners
            if (projectFilter) {
                projectFilter.addEventListener('change', function() {
                    navigateToFilter();
                });
            }
        });

        // Global funktion som anropas från checkboxens onchange-attribut
        function toggleShowArchived(checkbox) {
            const projectFilter = document.getElementById('projectFilter');
            const selectedProject = projectFilter ? projectFilter.value : '';
            const showArchived = checkbox && checkbox.checked;

            const params = new URLSearchParams();
            if (selectedProject) {
                params.append('project_id', selectedProject);
            }
            if (showArchived) {
                params.append('show_archived', '1');
            }

            const queryString = params.toString();
            if (queryString) {
                window.location.href = `?${queryString}`;
            } else {
                window.location.href = window.location.pathname;
            }
        }

        
        // Scroll to top funktionalitet
        function scrollToTop() {
            window.scrollTo({
                top: 0,
                behavior: 'smooth'
            });
        }
        
        // Visa/dölj scroll-to-top knapp baserat på scroll-position
        window.addEventListener('scroll', function() {
            const scrollBtn = document.getElementById('scrollToTopBtn');
            if (window.pageYOffset > 300) {
                scrollBtn.style.display = 'block';
            } else {
                scrollBtn.style.display = 'none';
            }
        });
    </script>
    
    <!-- Scroll to top knapp -->
    <button id="scrollToTopBtn" onclick="scrollToTop()" class="scroll-to-top" style="display: none;" title="Scrolla till toppen">
        ↑
    </button>
</body>
</html>
