<?php 
require_once __DIR__ . '/../../security.php';

// Säkerhetsheaders tidigt
setSecurityHeaders();

// Hämta project_id från GET eller cookie
if (session_status() === PHP_SESSION_NONE) {
    @session_start();
}

$rawProjectId = '';
if (isset($_GET['project_id']) && $_GET['project_id'] !== '') {
    $rawProjectId = $_GET['project_id'];
} elseif (isset($_COOKIE['project_id']) && $_COOKIE['project_id'] !== '') {
    $rawProjectId = $_COOKIE['project_id'];
} else {
    $rawProjectId = '';
}
$projectId = sanitizeInput($rawProjectId, 'project_id');

// Kontrollera att project_id finns
if (empty($projectId)) {
    http_response_code(403);
    ?>
<!DOCTYPE html>
<html lang="sv">
<head>
    <meta charset="utf-8" />
    <meta name="viewport" content="width=device-width,initial-scale=1" />
    <title>Åtkomst nekad - ByggArkiv</title>
    <link rel="stylesheet" href="../common.css">
</head>
<body>
    <div class="container">
        <div class="card">
            <h1>Åtkomst nekad</h1>
            <p>Du måste ange ett projekt-ID för att komma åt denna sida.</p>
            <a href="../../index.html" class="btn btn-primary">Till startsidan</a>
        </div>
    </div>
</body>
</html>
    <?php
    exit;
}

// Generera admin URL precis som andra admin-sidor
$script = $_SERVER['SCRIPT_NAME'] ?? '';
$admPos = strpos($script, '/adm/');
$adminUrl = ($admPos !== false ? substr($script, 0, $admPos + 4) : '/adm') . '/admin.php';
?>
<!DOCTYPE html>
<html lang="sv">
<head>
  <meta charset="utf-8" />
  <meta name="viewport" content="width=device-width,initial-scale=1" />
  <title>Redigera Projekt - ByggArkiv Admin</title>
  <link rel="stylesheet" href="../common.css">
  <style>
    /* Sid-specifika stilar för edit_project.php */
    .container { max-width: 600px; }
    .form-group input,
    .form-group textarea {
      width: 500px;
    }
    .form-group textarea {
      height: 100px;
    }
    .status {
      display: none;
      padding: 12px;
      border-radius: 4px;
      margin-top: 20px;
      font-weight: 500;
    }
    .status.success {
      background: #d4edda;
      color: #155724;
      border: 1px solid #c3e6cb;
    }
    .status.error {
      background: #f8d7da;
      color: #721c24;
      border: 1px solid #f5c6cb;
    }
    .status.info {
      background: #d1ecf1;
      color: #0c5460;
      border: 1px solid #bee5eb;
    }
    .form-help {
      font-size: 12px;
      color: #666;
      margin-top: 5px;
      font-weight: 400;
    }
  </style>
</head>
<body>
  <div class="container">
    <header class="header">
      <a href="../../index.html">
        <img src="../../data/logo.png" alt="ByggArkiv Snickare Stefan" class="logo">
      </a>
      <div class="back-link">
        <a href="projects.php">← Tillbaka till projekthantering</a>
      </div>
    </header>
    
    <div class="card">
      <h1>Redigera Projekt</h1>
      <p class="subtitle">Uppdatera projektinformation</p>
      
      <form id="editForm">
        <div class="form-group">
          <label for="projectId">Projekt-ID:</label>
          <input type="text" id="projectId" name="projectId" readonly>
        </div>
        
        <div class="form-group">
          <label for="projectName">Adress eller företagsnamn:</label>
          <input type="text" id="projectName" name="projectName" placeholder="t.ex. Ny villa" maxlength="100" required>
        </div>
        
        <div class="form-group">
          <label for="startDate">Startdatum:</label>
          <input type="date" id="startDate" name="startDate">
        </div>
        
        <div class="form-group">
          <label for="description">Beskrivning:</label>
          <textarea id="description" name="description" placeholder="Beskrivning av projektet (valfritt)"></textarea>
        </div>
        
        <div class="form-group">
          <div class="checkbox-group">
            <input type="checkbox" id="archived" name="archived">
            <label for="archived">Arkiverat (projektet är avslutat)</label>
          </div>
        </div>
        
        <div class="button-group">
          <button type="submit" class="btn-primary">Spara ändringar</button>
          <button type="button" class="btn-danger" onclick="deleteProject()">Ta bort projekt</button>
          <button type="button" class="btn-secondary" onclick="goBack()" style="margin-left: auto;">Avbryt</button>
        </div>
      </form>
      
      <div id="status" class="status"></div>
    </div>
  </div>

  <script>
    let currentProject = null;
    
    // Ladda projektdata
    async function loadProjectData() {
      const projectId = new URLSearchParams(window.location.search).get('project_id');
      if (!projectId) {
        showStatus('Inget projekt-ID angivet', 'error');
        return;
      }
      
      try {
        const response = await fetch(`../api/projects.php?action=get&project_id=${projectId}`);
        const data = await response.json();
        
        if (data.success) {
          currentProject = data.project;
          displayProjectData();
        } else {
          showStatus('Kunde inte ladda projektdata: ' + data.error, 'error');
        }
      } catch (error) {
        showStatus('Fel vid laddning: ' + error.message, 'error');
      }
    }
    
    // Visa projektdata i formuläret
    function displayProjectData() {
      if (!currentProject) return;
      
      document.getElementById('projectId').value = currentProject.id;
      document.getElementById('projectName').value = currentProject.name;
      document.getElementById('startDate').value = currentProject.start_date || '';
      document.getElementById('description').value = currentProject.description || '';
      document.getElementById('archived').checked = currentProject.status === 'archived';
    }
    
    // Hantera formulärsubmit
    document.getElementById('editForm').addEventListener('submit', async function(e) {
      e.preventDefault();
      
      if (!currentProject) {
        showStatus('Inget projekt att uppdatera', 'error');
        return;
      }
      
      // Disabla knappar under sparande
      const saveBtn = document.querySelector('button[type="submit"]');
      const deleteBtn = document.querySelector('button[onclick="deleteProject()"]');
      const cancelBtn = document.querySelector('button[onclick="goBack()"]');
      
      saveBtn.disabled = true;
      saveBtn.textContent = 'Sparar...';
      deleteBtn.disabled = true;
      cancelBtn.disabled = true;
      
      try {
        const formData = new FormData();
        formData.append('projectId', currentProject.id);
        formData.append('name', document.getElementById('projectName').value.trim());
        formData.append('description', document.getElementById('description').value.trim());
        formData.append('startDate', document.getElementById('startDate').value);
        formData.append('archived', document.getElementById('archived').checked ? '1' : '0');
        formData.append('status', currentProject.status || 'active');
        
        const response = await fetch('../api/projects.php?action=update', {
          method: 'POST',
          body: formData
        });
        
        const data = await response.json();
        
        if (data.success) {
          showStatus('Projekt uppdaterat', 'success');
          setTimeout(() => {
            goBack();
          }, 2000);
        } else {
          showStatus('Kunde inte uppdatera projekt: ' + data.message, 'error');
          // Återställ knappar vid fel
          saveBtn.disabled = false;
          saveBtn.textContent = 'Spara ändringar';
          deleteBtn.disabled = false;
          cancelBtn.disabled = false;
        }
      } catch (error) {
        showStatus('Fel vid uppdatering: ' + error.message, 'error');
        // Återställ knappar vid fel
        saveBtn.disabled = false;
        saveBtn.textContent = 'Spara ändringar';
        deleteBtn.disabled = false;
        cancelBtn.disabled = false;
      }
    });
    
    // Ta bort projekt
    async function deleteProject() {
      if (!currentProject) {
        showStatus('Inget projekt att ta bort', 'error');
        return;
      }
      
      if (!confirm(`Är du säker på att du vill ta bort projektet "${currentProject.name}" (ID: ${currentProject.id}) permanent?\n\nDetta kan inte ångras!`)) {
        return;
      }
      
      try {
        const formData = new FormData();
        formData.append('projectId', currentProject.id);
        
        const response = await fetch('../api/projects.php?action=delete', {
          method: 'POST',
          body: formData
        });
        
        const data = await response.json();
        
        if (data.success) {
          showStatus('Projekt har tagits bort', 'success');
          setTimeout(() => {
            goBack();
          }, 2000);
        } else {
          showStatus('Kunde inte ta bort projekt: ' + data.message, 'error');
        }
      } catch (error) {
        showStatus('Fel vid borttagning: ' + error.message, 'error');
      }
    }
    
    // Gå tillbaka
    function goBack() {
      window.location.href = 'projects.php';
    }
    
    // Visa statusmeddelande
    function showStatus(message, type) {
      const statusElement = document.getElementById('status');
      statusElement.textContent = message;
      statusElement.className = 'status ' + type;
      statusElement.style.display = 'block';
      
      // Dölj efter 5 sekunder
      setTimeout(() => {
        statusElement.style.display = 'none';
      }, 5000);
    }
    
    // Ladda projektdata när sidan laddas
    document.addEventListener('DOMContentLoaded', loadProjectData);
  </script>
</body>
</html>
