<?php 
require_once __DIR__ . '/../admin_guard.php';
require_once __DIR__ . '/../../lib/participant_helper.php';
?>
<!DOCTYPE html>
<html lang="sv">
<head>
  <meta charset="utf-8" />
<?php
    $script = $_SERVER['SCRIPT_NAME'] ?? '';
    $admPos = strpos($script, '/adm/');
    $adminUrl = ($admPos !== false ? substr($script, 0, $admPos + 4) : '/adm') . '/admin.php';
  ?>
  <meta name="viewport" content="width=device-width,initial-scale=1" />
  <title>Hantera Projekt - ByggArkiv Admin</title>
  <link rel="stylesheet" href="../common.css">
  <style>
    /* Sid-specifika stilar för projects.html */
    .container { max-width: 900px; }
    .project-list {
      margin: 20px 0;
    }
    .project-item.completed {
      background: #f0f8f0;
      border-color: #28a745;
    }
    .project-item.archived {
      background: #f8d7da;
      border-color: #dc3545;
      opacity: 0.9;
    }
    .project-info {
      flex: 1;
    }
    .project-name {
      font-size: 20px;
      font-weight: 600;
      color: #333;
      margin-bottom: 4px;
    }
    .project-id {
      font-family: monospace;
      font-size: 14px;
      font-weight: 400;
      color: #666;
    }
    .project-details {
      font-size: 14px;
      color: #666;
      margin-top: 4px;
    }
    .project-actions {
      display: flex;
      gap: 8px;
      align-items: center;
    }

    .confirm-btn-danger {
      background: #dc3545;
      color: white;
    }
    .confirm-btn-danger:hover {
      background: #c82333;
    }
    .add-form, .edit-form {
      display: grid;
      gap: 16px;
      margin-top: 20px;
    }
    .form-row {
      display: grid;
      grid-template-columns: 1fr 1fr;
      gap: 16px;
    }
    .form-row.full-width {
      grid-template-columns: 1fr;
    }
    /* Vänsterjustera fält-rubriker i popup (lägg till/redigera) */
    .confirm-box .form-group label {
      text-align: left;
    }

    @media (max-width: 768px) {
      .form-row {
        grid-template-columns: 1fr;
      }
      .project-actions {
        flex-direction: column;
        gap: 4px;
      }
      .action-btn {
        width: 100%;
        text-align: center;
      }
    }
  </style>
</head>
<body>
  <div class="container">
    <header class="header">
      <a href="../../index.html">
        <img src="../../data/logo.png" alt="ByggArkiv Snickare Stefan" class="logo">
      </a>
      <div class="back-link">
        <a href="<?= htmlspecialchars($adminUrl) ?>">← Tillbaka till admin</a>
      </div>
    </header>
    
    <div class="card">
      <h1>Hantera Projekt</h1>
      <p class="subtitle">Skapa, redigera och markera projekt som klara</p>
      
      <div style="margin: 20px 0;">
        <button id="addProjectBtn" onclick="showAddProjectPopup()" class="admin-button">Lägg till</button>
      </div>
      
      <div class="project-list" id="projectList">
        <!-- Projekt kommer att laddas här -->
      </div>

      <!-- Popup för att lägga till nytt projekt -->
      <div id="addProjectPopup" class="confirm-overlay" style="display: none;">
        <div class="confirm-box" style="max-width: 600px;">
          <h3 class="confirm-title">Lägg till</h3>
          <form id="addProjectForm">
            <div class="form-group">
              <label for="newProjectId">Projekt-ID:</label>
              <input type="text" id="newProjectId" placeholder="t.ex. 1005" maxlength="10" required>
            </div>
            <div class="form-group">
              <label for="newProjectName">Adress eller företagsnamn: *</label>
              <input type="text" id="newProjectName" placeholder="t.ex. Ny villa" maxlength="100" required>
            </div>
            <div class="form-group">
              <label for="newStartDate">Startdatum:</label>
              <input type="date" id="newStartDate">
            </div>
            <div class="form-group">
              <label for="newDescription">Beskrivning: *</nlabel>
              <textarea id="newDescription" placeholder="Beskrivning av projektet (valfritt)"></textarea>
            </div>
            <div class="confirm-actions">
              <button type="submit" class="confirm-btn confirm-btn-primary">Lägg till</button>
              <button type="button" onclick="hideAddProjectPopup()" class="confirm-btn confirm-btn-cancel">Avbryt</button>
            </div>
          </form>
        </div>
      </div>

      <!-- Popup för att redigera projekt -->
      <div id="editProjectPopup" class="confirm-overlay" style="display: none;">
        <div class="confirm-box" style="max-width: 600px;">
          <h3 class="confirm-title">Redigera projekt</h3>
          <form id="editProjectForm">
            <div class="form-group">
              <label for="editProjectId">Projekt-ID:</label>
              <input type="text" id="editProjectId" readonly>
            </div>
            <div class="form-group">
              <label for="editProjectName">Adress eller företagsnamn: *</label>
              <input type="text" id="editProjectName" placeholder="t.ex. Ny villa" maxlength="100" required>
            </div>
            <div class="form-group">
              <label for="editStartDate">Startdatum:</label>
              <input type="date" id="editStartDate">
            </div>
            <div class="form-group">
              <label for="editDescription">Beskrivning: *</label>
              <textarea id="editDescription" placeholder="Beskrivning av projektet (valfritt)"></textarea>
            </div>
            <div class="form-group">
              <div class="checkbox-group" style="display: flex; align-items: center; gap: 8px;">
                <input type="checkbox" id="editArchived" name="archived">
                <label for="editArchived" style="margin: 0;">Arkiverat (projektet är avslutat)</label>
              </div>
            </div>
            <div class="confirm-actions">
              <button type="submit" class="confirm-btn confirm-btn-primary">Spara</button>
              <button type="button" onclick="deleteProjectFromPopup()" class="confirm-btn confirm-btn-danger">Ta bort</button>
              <button type="button" onclick="hideEditProjectPopup()" class="confirm-btn confirm-btn-cancel">Avbryt</button>
            </div>
          </form>
        </div>
      </div>
      
      <!-- Popup för att hantera deltagare -->
      <div id="participantsPopup" class="confirm-overlay" style="display: none;">
        <div class="confirm-box" style="max-width: 800px; max-height: 90vh; display: flex; flex-direction: column;">
          <div style="flex-shrink: 0;">
            <h3 class="confirm-title" id="participantsTitle" style="font-size: 24px;">Hantera deltagare</h3>
            <p id="participantsProjectName" style="color: #000; margin-bottom: 20px; font-weight: 500;"></p>
          </div>
          
          <div id="participantsContent" style="flex: 1; overflow-y: auto; min-height: 0;">
            <p>Laddar...</p>
          </div>
          
          <div class="confirm-actions" style="margin-top: 20px; flex-shrink: 0;">
            <button type="button" onclick="cancelParticipants()" class="confirm-btn confirm-btn-cancel">Avbryt</button>
            <button type="button" onclick="saveParticipants()" class="confirm-btn confirm-btn-primary">Spara</button>
          </div>
        </div>
      </div>
      
      <div id="status" class="status"></div>
    </div>
  </div>

  <script>
    // Lista över alla admin user_ids
    const adminUserIds = <?= json_encode(getAllAdminUserIds()) ?>;
    
    // Hämta user_id från PHP session (satt av admin_guard.php)
    const currentUserId = <?= json_encode($_SESSION['user_id'] ?? '') ?>;
    
    // Funktion för att kontrollera om en participant är admin
    function isParticipantAdmin(participantId) {
        return adminUserIds.includes(participantId);
    }
    
    let projects = [];
    
    // Ladda projekt från filen
    async function loadProjects() {
      try {
        const response = await fetch('../api/projects.php?action=list');
        const data = await response.json();
        
        if (data.success) {
          projects = data.projects;
          displayProjects();
        } else {
          showStatus('Kunde inte ladda projekt: ' + data.error, 'error');
        }
      } catch (error) {
        showStatus('Fel vid laddning: ' + error.message, 'error');
      }
    }
    
    // Visa popup för att lägga till nytt projekt
    function showAddProjectPopup() {
      document.getElementById('addProjectPopup').style.display = 'flex';
      // Rensa formuläret
      document.getElementById('addProjectForm').reset();
    }
    
    // Dölj popup för att lägga till nytt projekt
    function hideAddProjectPopup() {
      document.getElementById('addProjectPopup').style.display = 'none';
    }
    
    // Visa popup för att redigera projekt
    async function showEditProjectPopup(projectId) {
      try {
        const response = await fetch(`../api/projects.php?action=get&project_id=${projectId}`);
        const data = await response.json();
        
        if (data.success) {
          const project = data.project;
          document.getElementById('editProjectId').value = project.id;
          document.getElementById('editProjectName').value = project.name;
          document.getElementById('editStartDate').value = project.startDate || '';
          document.getElementById('editDescription').value = project.description || '';
          document.getElementById('editArchived').checked = project.status === 'archived';
          
          document.getElementById('editProjectPopup').style.display = 'flex';
        } else {
          showStatus('Kunde inte ladda projektdata: ' + data.error, 'error');
        }
      } catch (error) {
        showStatus('Fel vid laddning: ' + error.message, 'error');
      }
    }
    
    // Dölj popup för att redigera projekt
    function hideEditProjectPopup() {
      document.getElementById('editProjectPopup').style.display = 'none';
    }
    
    // Hantera formulärskickning för popup
    document.addEventListener('DOMContentLoaded', function() {
      const addProjectForm = document.getElementById('addProjectForm');
      if (addProjectForm) {
        addProjectForm.addEventListener('submit', function(e) {
          e.preventDefault();
          addProject();
        });
      }
      
      const editProjectForm = document.getElementById('editProjectForm');
      if (editProjectForm) {
        editProjectForm.addEventListener('submit', function(e) {
          e.preventDefault();
          saveProject();
        });
      }
      
      // Stäng popup när man klickar utanför
      const addPopup = document.getElementById('addProjectPopup');
      if (addPopup) {
        addPopup.addEventListener('click', function(e) {
          if (e.target === addPopup) {
            hideAddProjectPopup();
          }
        });
      }
      
      const editPopup = document.getElementById('editProjectPopup');
      if (editPopup) {
        editPopup.addEventListener('click', function(e) {
          if (e.target === editPopup) {
            hideEditProjectPopup();
          }
        });
      }
    });
    
    // Visa projekt i listan
    function displayProjects() {
      const listElement = document.getElementById('projectList');
      
      if (projects.length === 0) {
        listElement.innerHTML = '<p class="no-results">Inga projekt hittades</p>';
        return;
      }
      
      // Sortera projekt: aktiva först, sedan arkiverade, sedan alfabetiskt efter namn
      const sortedProjects = [...projects].sort((a, b) => {
        const statusOrder = { 'active': 0, 'completed': 1, 'archived': 2 };
        const statusA = statusOrder[a.status] ?? 0;
        const statusB = statusOrder[b.status] ?? 0;
        if (statusA !== statusB) {
          return statusA - statusB;
        }
        return a.name.localeCompare(b.name, 'sv');
      });
      
      listElement.innerHTML = sortedProjects.map(project => {
        const status = project.status || 'active';
        const isArchived = status === 'archived';
        const isCompleted = status === 'completed';
        const archivedDate = project.archived_at ? new Date(project.archived_at).toLocaleDateString('sv-SE') : '';
        const completedDate = project.completed_at ? new Date(project.completed_at).toLocaleDateString('sv-SE') : '';
        const startDate = project.start_date ? new Date(project.start_date).toLocaleDateString('sv-SE') : '';
        const createdDate = project.created_at ? new Date(project.created_at).toLocaleDateString('sv-SE') : '';
        
        let statusText = '';
        if (isArchived) {
          statusText = 'Arkiverad: ' + archivedDate;
        } else if (isCompleted) {
          statusText = 'Slutförd: ' + completedDate;
        } else {
          statusText = startDate ? 'Start: ' + startDate : 'Skapat: ' + createdDate;
        }
        
        return `
          <div class="project-item ${isArchived ? 'archived' : isCompleted ? 'completed' : ''}" id="project-${project.id}">
            <div class="project-info">
              <div class="project-name">${project.name}</div>
              <div class="project-id">${project.id}</div>
              <div class="project-details">
                ${project.description ? project.description + ' • ' : ''}
                ${statusText}
              </div>
            </div>
            <div class="project-actions">
              <button class="action-btn btn-edit" onclick="editProject('${project.id}')" title="Redigera projekt">
                <svg width="20" height="20" viewBox="0 0 24 24" fill="currentColor">
                  <path d="M3 17.25V21h3.75L17.81 9.94l-3.75-3.75L3 17.25zM20.71 7.04c.39-.39.39-1.02 0-1.41l-2.34-2.34c-.39-.39-1.02-.39-1.41 0l-1.83 1.83 3.75 3.75 1.83-1.83z"/>
                </svg>
              </button>
              <button class="action-btn btn-participants" onclick="manageParticipants('${project.id}')" title="Hantera deltagare">
                <svg width="20" height="20" viewBox="0 0 24 24" fill="currentColor">
                  <path d="M16 11c1.66 0 2.99-1.34 2.99-3S17.66 5 16 5c-1.66 0-3 1.34-3 3s1.34 3 3 3zm-8 0c1.66 0 2.99-1.34 2.99-3S9.66 5 8 5C6.34 5 5 6.34 5 8s1.34 3 3 3zm0 2c-2.33 0-7 1.17-7 3.5V19h14v-2.5c0-2.33-4.67-3.5-7-3.5zm8 0c-.29 0-.62.02-.97.05 1.16.84 1.97 1.97 1.97 3.45V19h6v-2.5c0-2.33-4.67-3.5-7-3.5z"/>
                </svg>
              </button>
              <button class="action-btn btn-edit" onclick="openChat('${project.id}')" title="Öppna chatt">
                <svg width="20" height="20" viewBox="0 0 24 24" fill="currentColor">
                  <path d="M20 2H4c-1.1 0-2 .9-2 2v18l4-4h14c1.1 0 2-.9 2-2V4c0-1.1-.9-2-2-2zm0 14H6l-2 2V4h16v12z"/>
                </svg>
              </button>
            </div>
          </div>
        `;
      }).join('');
    }
    

    
    // Lägg till nytt projekt
    async function addProject() {
      const inputProjectId = document.getElementById('newProjectId');
      const inputProjectName = document.getElementById('newProjectName');
      const inputStartDate = document.getElementById('newStartDate');
      const inputDescription = document.getElementById('newDescription');
      
      const newProjectId = inputProjectId.value.trim();
      const newProjectName = inputProjectName.value.trim();
      const newStartDate = inputStartDate.value;
      const newDescription = inputDescription.value.trim();
      
      if (!newProjectId) {
        showStatus('Ange ett projekt-ID', 'error');
        return;
      }
      
      if (!newProjectName) {
        showStatus('Ange adress eller företagsnamn', 'error');
        return;
      }
      
      if (projects.some(project => project.id === newProjectId)) {
        showStatus('Projekt-ID finns redan', 'error');
        return;
      }
      
      try {
        const formData = new FormData();
        formData.append('id', newProjectId);
        formData.append('name', newProjectName);
        formData.append('startDate', newStartDate);
        formData.append('description', newDescription);
        
        const response = await fetch('../api/projects.php?action=add', {
          method: 'POST',
          body: formData
        });
        
        // Kontrollera om svaret är OK
        if (!response.ok) {
          throw new Error(`HTTP ${response.status}: ${response.statusText}`);
        }
        
        // Kontrollera om svaret är tomt
        const responseText = await response.text();
        if (!responseText.trim()) {
          throw new Error('Tomt svar från servern');
        }
        
        let data;
        try {
          data = JSON.parse(responseText);
        } catch (parseError) {
          console.error('Kunde inte parsa JSON:', responseText);
          throw new Error('Ogiltigt svar från servern: ' + responseText);
        }
        
        if (data.success) {
          projects.push(data.project);
          displayProjects();
          hideAddProjectPopup(); // Stäng popup-fönstret
        } else {
          showStatus('Kunde inte lägga till: ' + (data.error || 'Okänt fel'), 'error');
        }
      } catch (error) {
        console.error('Fel i addProject:', error);
        showStatus('Fel vid tillägg: ' + error.message, 'error');
      }
    }
    
    // Spara projektändringar
    async function saveProject() {
      const projectId = document.getElementById('editProjectId').value;
      const projectName = document.getElementById('editProjectName').value.trim();
      const startDate = document.getElementById('editStartDate').value;
      const description = document.getElementById('editDescription').value.trim();
      const archived = document.getElementById('editArchived').checked;
      
      if (!projectName) {
        showStatus('Ange adress eller företagsnamn', 'error');
        return;
      }
      
      try {
        const formData = new FormData();
        formData.append('projectId', projectId);
        formData.append('name', projectName);
        formData.append('startDate', startDate);
        formData.append('description', description);
        formData.append('archived', archived ? '1' : '0');
        
        const response = await fetch('../api/projects.php?action=update', {
          method: 'POST',
          body: formData
        });
        
        if (!response.ok) {
          throw new Error(`HTTP ${response.status}: ${response.statusText}`);
        }
        
        const responseText = await response.text();
        if (!responseText.trim()) {
          throw new Error('Tomt svar från servern');
        }
        
        let data;
        try {
          data = JSON.parse(responseText);
        } catch (parseError) {
          console.error('Kunde inte parsa JSON:', responseText);
          throw new Error('Ogiltigt svar från servern: ' + responseText);
        }
        
        if (data.success) {
          // Ladda om hela projektlistan
          await loadProjects();
          hideEditProjectPopup();
        } else {
          showStatus('Kunde inte uppdatera: ' + (data.error || 'Okänt fel'), 'error');
        }
      } catch (error) {
        console.error('Fel i saveProject:', error);
        showStatus('Fel vid uppdatering: ' + error.message, 'error');
      }
    }
    
    // Ta bort projekt från popup
    async function deleteProjectFromPopup() {
      const projectId = document.getElementById('editProjectId').value;
      const projectName = document.getElementById('editProjectName').value;
      
      if (!confirm(`Är du säker på att du vill ta bort projektet "${projectName}"? Detta går inte att ångra.`)) {
        return;
      }
      
      try {
        const formData = new FormData();
        formData.append('projectId', projectId);
        
        const response = await fetch('../api/projects.php?action=delete', {
          method: 'POST',
          body: formData
        });
        
        if (!response.ok) {
          throw new Error(`HTTP ${response.status}: ${response.statusText}`);
        }
        
        const responseText = await response.text();
        if (!responseText.trim()) {
          throw new Error('Tomt svar från servern');
        }
        
        let data;
        try {
          data = JSON.parse(responseText);
        } catch (parseError) {
          console.error('Kunde inte parsa JSON:', responseText);
          throw new Error('Ogiltigt svar från servern: ' + responseText);
        }
        
        if (data.success) {
          // Ladda om hela projektlistan
          await loadProjects();
          hideEditProjectPopup();
          showStatus('Projekt "' + projectName + '" har tagits bort', 'success');
        } else {
          showStatus('Kunde inte ta bort: ' + (data.error || 'Okänt fel'), 'error');
        }
      } catch (error) {
        console.error('Fel i deleteProjectFromPopup:', error);
        showStatus('Fel vid borttagning: ' + error.message, 'error');
      }
    }
    
    // Redigera projekt
    function openChat(projectId) {
      // Använd currentUserId från PHP session, eller fallback till URL-parameter
      const urlParams = new URLSearchParams(window.location.search);
      const userId = currentUserId || urlParams.get('user_id') || '';
      let href = '../../chat/chat.php?project_id=' + encodeURIComponent(projectId);
      if (userId) {
        href += '&user_id=' + encodeURIComponent(userId);
      }
      window.location.href = href;
    }
    
    function editProject(projectId) {
      showEditProjectPopup(projectId);
    }
    
    // Växla mellan aktiv och arkiverad
    async function toggleStatus(projectId) {
      try {
        const formData = new FormData();
        formData.append('projectId', projectId);
        
        const response = await fetch('../api/projects.php?action=toggle_status', {
          method: 'POST',
          body: formData
        });
        
        // Kontrollera om svaret är OK
        if (!response.ok) {
          throw new Error(`HTTP ${response.status}: ${response.statusText}`);
        }
        
        // Kontrollera om svaret är tomt
        const responseText = await response.text();
        if (!responseText.trim()) {
          throw new Error('Tomt svar från servern');
        }
        
        let data;
        try {
          data = JSON.parse(responseText);
        } catch (parseError) {
          console.error('Kunde inte parsa JSON:', responseText);
          throw new Error('Ogiltigt svar från servern: ' + responseText);
        }
        
        if (data.success) {
          const index = projects.findIndex(p => p.id === projectId);
          if (index !== -1 && data.project) {
            projects[index] = data.project;
            displayProjects();
            showStatus(data.message, 'success');
          } else {
            // Om vi inte fick projektet tillbaka, ladda om hela listan
            loadProjects();
            showStatus(data.message, 'success');
          }
        } else {
          showStatus('Kunde inte uppdatera: ' + (data.error || 'Okänt fel'), 'error');
        }
      } catch (error) {
        console.error('Fel i toggleStatus:', error);
        showStatus('Fel vid uppdatering: ' + error.message, 'error');
      }
    }
    
    
    // Visa statusmeddelande
    function showStatus(message, type) {
      const statusElement = document.getElementById('status');
      statusElement.textContent = message;
      statusElement.className = 'status ' + type;
      statusElement.style.display = 'block';
      
      // Dölj efter 5 sekunder
      setTimeout(() => {
        statusElement.style.display = 'none';
      }, 5000);
    }
    
    // ========== DELTAGARHANTERING ==========
    
    let currentParticipantsProjectId = null;
    let availableUsers = [];
    let availableStaff = [];
    let currentParticipants = [];
    let originalParticipants = []; // Spara ursprunglig state
    let pendingChanges = []; // Spåra ändringar som inte sparats än
    let types = [];
    
    // Ladda yrkeskategorier
    async function loadTypes() {
      // Använd currentUserId från PHP session, eller fallback till URL-parameter
      const urlParams = new URLSearchParams(window.location.search);
      const userId = currentUserId || urlParams.get('user_id') || '';
      
      const response = await fetch(`../../api/chat/external_staff.php?action=list_types&user_id=${encodeURIComponent(userId)}`);
      const data = await response.json();
      console.log('Yrkeskategorier API-svar:', data);
      
      if (data.success) {
        types = data.types || [];
        // Sortera alfabetiskt
        types.sort((a, b) => (a.name || '').localeCompare(b.name || '', 'sv'));
      } else {
        types = [];
      }
    }
    
    // Visa popup för deltagarhantering
    async function manageParticipants(projectId) {
      currentParticipantsProjectId = projectId;
      const project = projects.find(p => p.id === projectId);
      const projectName = project ? project.name : projectId;
      document.getElementById('participantsTitle').textContent = 'Hantera deltagare';
      document.getElementById('participantsProjectName').textContent = 'Projekt: ' + projectName;
      document.getElementById('participantsPopup').style.display = 'flex';
      document.getElementById('participantsContent').innerHTML = '<p>Laddar...</p>';
      
      // Rensa pending changes
      pendingChanges = [];
      
      try {
        // Ladda deltagare, användare, hantverkare och yrkeskategorier parallellt
        await Promise.all([
          loadParticipantsForProject(projectId),
          loadAvailableUsers(),
          loadAvailableStaff(),
          loadTypes()
        ]);
        
        // Spara ursprunglig state
        originalParticipants = JSON.parse(JSON.stringify(currentParticipants));
        
        displayParticipants();
      } catch (error) {
        console.error('Fel vid laddning av deltagare:', error);
        document.getElementById('participantsContent').innerHTML = '<p style="color: red;">Fel vid laddning: ' + error.message + '</p>';
      }
    }
    
    // Dölj popup
    function hideParticipantsPopup() {
      document.getElementById('participantsPopup').style.display = 'none';
      currentParticipantsProjectId = null;
      pendingChanges = [];
      originalParticipants = [];
    }
    
    // Avbryt och återställ till ursprunglig state
    function cancelParticipants() {
      // Återställ till ursprunglig state
      currentParticipants = JSON.parse(JSON.stringify(originalParticipants));
      displayParticipants();
      hideParticipantsPopup();
    }
    
    // Spara alla ändringar
    async function saveParticipants() {
      if (pendingChanges.length === 0) {
        hideParticipantsPopup();
        return;
      }
      
      const urlParams = new URLSearchParams(window.location.search);
      const userId = currentUserId || urlParams.get('user_id') || '';
      
      let successCount = 0;
      let errorCount = 0;
      
      // Spara alla ändringar sekventiellt
      for (const change of pendingChanges) {
        try {
          const formData = new FormData();
          formData.append('action', change.action);
          formData.append('project_id', currentParticipantsProjectId);
          formData.append('type', change.apiType);
          formData.append('participant_id', change.id);
          formData.append('user_id', userId);
          
          const response = await fetch('../../api/chat/participants.php', {
            method: 'POST',
            body: formData
          });
          
          const text = await response.text();
          let data;
          try {
            data = JSON.parse(text);
          } catch (parseError) {
            console.error('Kunde inte parsa JSON-svar:', text);
            errorCount++;
            continue;
          }
          
          if (data.success) {
            successCount++;
          } else {
            errorCount++;
            console.error('Fel vid ' + change.action + ':', data.message);
          }
        } catch (error) {
          console.error('Fel vid ' + change.action + ':', error);
          errorCount++;
        }
      }
      
      // Ladda om deltagare för att få senaste state
      if (successCount > 0) {
        await loadParticipantsForProject(currentParticipantsProjectId);
        originalParticipants = JSON.parse(JSON.stringify(currentParticipants));
        displayParticipants();
      }
      
      if (errorCount > 0) {
        showStatus('Vissa ändringar kunde inte sparas', 'error');
      } else if (successCount > 0) {
        showStatus('Alla ändringar sparade', 'success');
      }
      
      pendingChanges = [];
      hideParticipantsPopup();
    }
    
    // Ladda deltagare för projektet
    async function loadParticipantsForProject(projectId) {
      // Använd currentUserId från PHP session, eller fallback till URL-parameter
      const urlParams = new URLSearchParams(window.location.search);
      const userId = currentUserId || urlParams.get('user_id') || '';
      
      const url = `../../api/chat/participants.php?action=list&project_id=${encodeURIComponent(projectId)}&user_id=${encodeURIComponent(userId)}`;
      console.log('Laddar deltagare från:', url);
      
      const response = await fetch(url);
      const data = await response.json();
      console.log('Deltagare API-svar:', data);
      
      if (data.success) {
        currentParticipants = data.participants || [];
        console.log('Antal deltagare:', currentParticipants.length, currentParticipants);
      } else {
        throw new Error(data.message || 'Kunde inte ladda deltagare');
      }
    }
    
    // Ladda alla användare
    async function loadAvailableUsers() {
      // Använd currentUserId från PHP session, eller fallback till URL-parameter
      const urlParams = new URLSearchParams(window.location.search);
      const userId = currentUserId || urlParams.get('user_id') || '';
      
      const response = await fetch(`../../api/user_info.php?action=list&user_id=${encodeURIComponent(userId)}`);
      const data = await response.json();
      console.log('Användare API-svar:', data);
      
      if (data.success) {
        // Filtrera bort externa användare och inaktiverade - de ska inte visas
        availableUsers = (data.users || []).filter(user => {
          const userType = user.user_type || 'user';
          const isActive = user.active !== false; // Default till true om inte satt
          return userType !== 'extern' && isActive;
        });
        console.log('Antal användare (efter filtrering):', availableUsers.length, availableUsers);
      } else {
        console.log('Kunde inte ladda användare:', data.message);
        availableUsers = [];
      }
    }
    
    // Ladda alla hantverkare
    async function loadAvailableStaff() {
      // Använd currentUserId från PHP session, eller fallback till URL-parameter
      const urlParams = new URLSearchParams(window.location.search);
      const userId = currentUserId || urlParams.get('user_id') || '';
      
      const response = await fetch(`../../api/chat/external_staff.php?action=list&user_id=${encodeURIComponent(userId)}`);
      const data = await response.json();
      console.log('Hantverkare API-svar:', data);
      
      if (data.success) {
        availableStaff = data.staff || [];
        console.log('Antal hantverkare:', availableStaff.length, availableStaff);
      } else {
        availableStaff = [];
      }
    }
    
    // Visa deltagarlistan med kryssrutor
    function displayParticipants() {
      const container = document.getElementById('participantsContent');
      console.log('displayParticipants anropad, currentParticipants:', currentParticipants);
      
      let html = '<div style="display: flex; flex-direction: column; gap: 20px;">';
      
      // Skapa en karta över vilka som är deltagare för snabb lookup
      const participantMap = {};
      currentParticipants.forEach(p => {
        const pid = p.participant_id;
        if (pid) {
          participantMap[pid] = {
            isParticipant: true,
            type: p.type || 'user'
          };
        }
      });
      
      // 1. Egna hantverkare först
      if (availableUsers.length > 0) {
        html += '<div>';
        html += '<h4 style="margin: 0 0 15px 0; font-size: 20px; font-weight: 600; color: #1a73e8; text-align: left;">Egna hantverkare</h4>';
        html += '<div style="display: flex; flex-direction: column; gap: 4px; margin-left: 0;">';
        
        availableUsers.forEach(user => {
          const pid = user.user_id;
          const isChecked = participantMap[pid] ? true : false;
          const isAdminUser = isParticipantAdmin(pid);
          const safeUserId = safeId(pid);
          const safeUserIdAttr = escapeHtml(pid).replace(/'/g, "\\'");
          const disabledAttr = isAdminUser && isChecked ? 'disabled' : '';
          const cursorStyle = isAdminUser && isChecked ? 'not-allowed' : 'pointer';
          const adminLabel = isAdminUser && isChecked ? '<span style="color: #999; font-size: 12px; margin-left: 8px;">(Admin)</span>' : '';
          
          html += `<div style="display: flex; align-items: center; gap: 10px; padding: 4px; border-radius: 4px; background: ${isChecked ? '#e8f5e9' : '#fff'};">
            <input type="checkbox" 
                   id="participant_user_${safeUserId}" 
                   ${isChecked ? 'checked' : ''} 
                   ${disabledAttr}
                   onchange="toggleParticipant('user', '${safeUserIdAttr}')"
                   style="width: 18px; height: 18px; cursor: ${cursorStyle}; flex-shrink: 0;">
            <label for="participant_user_${safeUserId}" style="flex: 1; cursor: ${cursorStyle}; margin: 0; text-align: left; color: #666; font-weight: normal; font-size: 14px;">
              <span>${escapeHtml(user.full_name || pid)}</span>
              ${adminLabel}
              ${user.mobile ? '<span style="margin-left: 8px;">' + escapeHtml(user.mobile) + '</span>' : ''}
            </label>
          </div>`;
        });
        
        html += '</div></div>';
      }
      
      // 2. Hantverkare grupperade efter yrkeskategori
      if (availableStaff.length > 0) {
        html += '<div>';
        html += '<h4 style="margin: 0 0 15px 0; font-size: 20px; font-weight: 600; color: #1a73e8; text-align: left;">Externa hantverkare</h4>';
        
        // Gruppera hantverkare efter type_id
        const staffByType = {};
        const staffWithoutType = [];
        
        availableStaff.forEach(staff => {
          const typeId = staff.type_id || '';
          if (typeId) {
            if (!staffByType[typeId]) {
              staffByType[typeId] = [];
            }
            staffByType[typeId].push(staff);
          } else {
            staffWithoutType.push(staff);
          }
        });
        
        // Sortera yrkeskategorier alfabetiskt
        const sortedTypeIds = Object.keys(staffByType).sort((a, b) => {
          const typeA = types.find(t => t.id === a);
          const typeB = types.find(t => t.id === b);
          const nameA = typeA ? typeA.name : '';
          const nameB = typeB ? typeB.name : '';
          return nameA.localeCompare(nameB, 'sv');
        });
        
        // Visa hantverkare grupperade efter yrkeskategori
        sortedTypeIds.forEach(typeId => {
          const type = types.find(t => t.id === typeId);
          const typeName = type ? type.name : 'Okänd kategori';
          const staffList = staffByType[typeId];
          
          html += `<div style="margin-bottom: 15px;">`;
          html += `<div style="font-weight: 600; color: #1a73e8; margin-bottom: 10px; font-size: 16px; text-align: left;">${escapeHtml(typeName)}</div>`;
          html += `<div style="display: flex; flex-direction: column; gap: 4px; margin-left: 20px;">`;
          
          staffList.forEach(staff => {
            const pid = staff.id;
            const isChecked = participantMap[pid] ? true : false;
            const safeStaffId = safeId(pid);
            const safeStaffIdAttr = escapeHtml(pid).replace(/'/g, "\\'");
            const companyName = staff.company_name || '';
            const mobile = staff.mobile || '';
            
            html += `<div style="display: flex; align-items: center; gap: 10px; padding: 4px 6px; border-radius: 4px; background: ${isChecked ? '#ffe6e6' : '#fff'};">
              <input type="checkbox" 
                     id="participant_staff_${safeStaffId}" 
                     ${isChecked ? 'checked' : ''} 
                     onchange="toggleParticipant('staff', '${safeStaffIdAttr}')"
                     style="width: 18px; height: 18px; cursor: pointer; flex-shrink: 0;">
              <label for="participant_staff_${safeStaffId}" style="flex: 1; cursor: pointer; margin: 0; text-align: left; color: #666; font-weight: normal; font-size: 14px;">
                <span>${escapeHtml(staff.name || pid)}</span>
                ${mobile ? '<span style="margin-left: 8px;">' + escapeHtml(mobile) + '</span>' : ''}
                ${companyName ? '<span style="margin-left: 8px;">(' + escapeHtml(companyName) + ')</span>' : ''}
              </label>
            </div>`;
          });
          
          html += '</div></div>';
        });
        
        // Visa hantverkare utan yrkeskategori
        if (staffWithoutType.length > 0) {
          html += `<div style="margin-bottom: 15px;">`;
          html += `<div style="font-weight: 600; color: #1a73e8; margin-bottom: 10px; font-size: 16px; text-align: left;">Utan yrkeskategori</div>`;
          html += `<div style="display: flex; flex-direction: column; gap: 4px; margin-left: 20px;">`;
          
          staffWithoutType.forEach(staff => {
            const pid = staff.id;
            const isChecked = participantMap[pid] ? true : false;
            const safeStaffId = safeId(pid);
            const safeStaffIdAttr = escapeHtml(pid).replace(/'/g, "\\'");
            const companyName = staff.company_name || '';
            const mobile = staff.mobile || '';
            
            html += `<div style="display: flex; align-items: center; gap: 10px; padding: 4px 6px; border-radius: 4px; background: ${isChecked ? '#ffe6e6' : '#fff'};">
              <input type="checkbox" 
                     id="participant_staff_${safeStaffId}" 
                     ${isChecked ? 'checked' : ''} 
                     onchange="toggleParticipant('staff', '${safeStaffIdAttr}')"
                     style="width: 18px; height: 18px; cursor: pointer; flex-shrink: 0;">
              <label for="participant_staff_${safeStaffId}" style="flex: 1; cursor: pointer; margin: 0; text-align: left; color: #666; font-weight: normal; font-size: 14px;">
                <span>${escapeHtml(staff.name || pid)}</span>
                ${mobile ? '<span style="margin-left: 8px;">' + escapeHtml(mobile) + '</span>' : ''}
                ${companyName ? '<span style="margin-left: 8px;">(' + escapeHtml(companyName) + ')</span>' : ''}
              </label>
            </div>`;
          });
          
          html += '</div></div>';
        }
        
        html += '</div>';
      }
      
      if (availableUsers.length === 0 && availableStaff.length === 0) {
        html += '<p style="color: #666;">Inga användare eller hantverkare tillgängliga.</p>';
      }
      
      html += '</div>';
      container.innerHTML = html;
    }
    
    // Växla deltagare (lägg till/ta bort) - spara inte direkt, bara uppdatera UI
    function toggleParticipant(type, id) {
      // Kontrollera om deltagaren är admin och försöker tas bort
      if (type === 'user' && isParticipantAdmin(id)) {
        const checkboxId = `participant_user_${safeId(id)}`;
        const checkbox = document.getElementById(checkboxId);
        if (checkbox && !checkbox.checked) {
          // Försöker avmarkera admin - förhindra detta
          checkbox.checked = true;
          showStatus('Admins kan inte tas bort från projekt', 'error');
          return;
        }
      }
      
      let checkboxId;
      if (type === 'user') {
        checkboxId = `participant_user_${safeId(id)}`;
      } else if (type === 'extern' || type === 'staff') {
        checkboxId = `participant_staff_${safeId(id)}`;
      } else {
        console.error('Okänd typ:', type);
        return;
      }
      
      // Normalisera type för API:et (API:et förväntar sig 'extern', inte 'staff')
      const apiType = (type === 'staff') ? 'extern' : type;
      
      const checkbox = document.getElementById(checkboxId);
      if (!checkbox) {
        console.error('Checkbox hittades inte:', checkboxId);
        return;
      }
      const isChecked = checkbox.checked;
      
      // Hitta om det redan finns en pending change för denna deltagare
      const existingChangeIndex = pendingChanges.findIndex(c => c.id === id && c.apiType === apiType);
      
      if (isChecked) {
        // Lägg till i pending changes
        if (existingChangeIndex >= 0) {
          // Om det redan finns en 'remove', ta bort den
          if (pendingChanges[existingChangeIndex].action === 'remove') {
            pendingChanges.splice(existingChangeIndex, 1);
          }
          // Om det redan finns en 'add', gör ingenting
        } else {
          // Kontrollera om deltagaren redan fanns i originalParticipants
          const wasInOriginal = originalParticipants.some(p => p.participant_id === id);
          if (!wasInOriginal) {
            // Lägg till ny pending change
            pendingChanges.push({
              action: 'add',
              id: id,
              apiType: apiType,
              type: type
            });
          }
        }
      } else {
        // Ta bort från pending changes eller lägg till remove
        if (existingChangeIndex >= 0) {
          if (pendingChanges[existingChangeIndex].action === 'add') {
            // Om det var en 'add', ta bort den helt
            pendingChanges.splice(existingChangeIndex, 1);
          } else {
            // Om det redan var en 'remove', gör ingenting
          }
        } else {
          // Kontrollera om deltagaren fanns i originalParticipants
          const wasInOriginal = originalParticipants.some(p => p.participant_id === id);
          if (wasInOriginal) {
            // Lägg till remove change
            pendingChanges.push({
              action: 'remove',
              id: id,
              apiType: apiType,
              type: type
            });
          }
        }
      }
      
      // Uppdatera UI (bakgrundsfärg baserat på checkbox state)
      updateParticipantBackground(checkboxId, isChecked, type);
    }
    
    // Uppdatera bakgrundsfärg för deltagare
    function updateParticipantBackground(checkboxId, isChecked, type) {
      const checkbox = document.getElementById(checkboxId);
      if (!checkbox) return;
      
      const container = checkbox.closest('div[style*="background"]');
      if (!container) return;
      
      if (type === 'user') {
        container.style.background = isChecked ? '#e8f5e9' : '#fff';
      } else {
        container.style.background = isChecked ? '#ffe6e6' : '#fff';
      }
    }
    
    // Hjälpfunktion för att escape HTML
    function escapeHtml(text) {
      if (!text) return '';
      const div = document.createElement('div');
      div.textContent = text;
      return div.innerHTML;
    }
    
    // Hjälpfunktion för att skapa säkra ID:n för HTML-attribut
    function safeId(id) {
      if (!id) return '';
      // Ersätt specialtecken som kan orsaka problem i HTML-ID:n
      return String(id).replace(/[^a-zA-Z0-9_-]/g, '_');
    }
    
    // ========== SLUT DELTAGARHANTERING ==========
    
    // Ladda projekt när sidan laddas
    document.addEventListener('DOMContentLoaded', loadProjects);
    
    // Enter-tangent för att lägga till
    document.getElementById('newProjectId').addEventListener('keypress', function(e) {
      if (e.key === 'Enter') {
        addProject();
      }
    });
    
    document.getElementById('newProjectName').addEventListener('keypress', function(e) {
      if (e.key === 'Enter') {
        addProject();
      }
    });

    // Enkel custom confirm med primärknappen "TAG BORT"
    function confirmAction(message, primaryLabel = 'OK', extraLinkHref = null, extraLinkText = null) {
      return new Promise(resolve => {
        const overlay = document.createElement('div');
        overlay.style.cssText = 'position:fixed;inset:0;background:rgba(0,0,0,0.4);z-index:2000;display:flex;align-items:center;justify-content:center;';

        const box = document.createElement('div');
        box.style.cssText = 'background:white;border-radius:12px;max-width:420px;width:90%;padding:20px;box-shadow:0 10px 30px rgba(0,0,0,0.3);';
        
        const linkHtml = extraLinkHref && extraLinkText 
          ? `<a href="${extraLinkHref}" class="confirm-link" target="_blank" rel="noopener">${extraLinkText} →</a>`
          : '';
        
        box.innerHTML = `
          <h3 class="confirm-title">Bekräfta borttagning</h3>
          <p class="confirm-message">${message}</p>
          <div class="confirm-actions">
            ${linkHtml}
            <button id="cancelBtn" class="confirm-btn confirm-btn-cancel">Avbryt</button>
            <button id="okBtn" class="confirm-btn confirm-btn-primary">${primaryLabel}</button>
          </div>
        `;

        overlay.appendChild(box);
        document.body.appendChild(overlay);

        const cleanup = (result) => {
          document.body.removeChild(overlay);
          resolve(result);
        };

        box.querySelector('#cancelBtn').onclick = () => cleanup(false);
        box.querySelector('#okBtn').onclick = () => cleanup(true);

        // Stäng med Escape
        const onKey = (e) => {
          if (e.key === 'Escape') {
            window.removeEventListener('keydown', onKey);
            cleanup(false);
          }
        };
        window.addEventListener('keydown', onKey);
      });
    }
  </script>
</body>
</html>
