<?php require_once __DIR__ . '/../admin_guard.php'; ?>
<!DOCTYPE html>
<html lang="sv">
<head>
  <meta charset="utf-8" />
  <meta name="viewport" content="width=device-width,initial-scale=1" />
  <title>Hantera User IDs - ByggArkiv Admin</title>
  <link rel="stylesheet" href="../common.css">
  <style>
    /* Sid-specifika stilar för userids.html */
    .container { max-width: 1000px; }
    .userid-list {
      margin: 20px 0;
    }
    
    /* Tre-kolumns layout för userid-content */
    .userid-content {
      display: grid;
      grid-template-columns: 2fr 2fr 1fr;
      gap: 20px;
      align-items: center;
      width: 100%;
    }
    
    .userid-left {
      flex: 1;
    }
    
    .userid-middle {
      /* User ID och Pushover User Key */
      display: flex;
      flex-direction: column;
      gap: 4px;
    }
    
    .userid-id-row,
    .userid-pushover-row {
      font-family: monospace;
      font-size: 13px;
      color: #666;
      word-break: break-all;
    }
    
    .userid-id-row {
      font-weight: 500;
      color: #333;
    }
    
    .userid-item.no-uuid {
      border-left: 4px solid #ff9800;
      opacity: 0.8;
    }
    .userid-text {
      font-family: monospace;
      font-size: 16px;
      font-weight: 500;
      color: #333;
    }
    .userid-actions {
      display: flex;
      gap: 8px;
      align-items: center;
    }
    .add-form {
      display: grid;
      gap: 16px;
      margin-top: 20px;
    }
    .form-row {
      display: grid;
      grid-template-columns: 1fr 1fr;
      gap: 16px;
    }
    .form-row:last-child {
      grid-template-columns: 1fr auto;
    }
    label { 
      display: block; 
      font-weight: 600; 
      margin-bottom: 4px; 
      color: #333;
    }
    input[type="text"] {
      padding: 10px; 
      border: 1px solid #ccc; 
      border-radius: 8px; 
      font-size: 16px; 
      box-sizing: border-box;
      width: 100%;
      font-family: inherit;
    }
    button {
      padding: 12px 20px; 
      border-radius: 8px; 
      border: 0; 
      font-size: 16px;
      cursor: pointer; 
      background: #1a73e8; 
      color: white;
      font-weight: 500;
      white-space: nowrap;
      width: auto;
      display: inline-flex;
      align-items: center;
      justify-content: center;
      transition: all 0.2s;
    }
    button:hover {
      background: #1557b0;
      transform: translateY(-1px);
      box-shadow: 0 4px 12px rgba(26, 115, 232, 0.3);
    }
    .status {
      margin-top: 16px;
      padding: 12px;
      border-radius: 8px;
      font-weight: 500;
    }
    .status.success {
      background: #d4edda;
      color: #155724;
      border: 1px solid #c3e6cb;
    }
    .status.error {
      background: #f8d7da;
      color: #721c24;
      border: 1px solid #f5c6cb;
    }
    
    .status.info {
      background: #d1ecf1;
      color: #0c5460;
      border: 1px solid #bee5eb;
    }
    .form-help {
      font-size: 12px;
      color: #666;
      font-weight: 400;
    }
    
    /* Popup-stilar */
    .confirm-overlay {
      position: fixed;
      top: 0;
      left: 0;
      width: 100%;
      height: 100%;
      background: rgba(0,0,0,0.5);
      display: flex;
      align-items: center;
      justify-content: center;
      z-index: 10000;
      overflow-y: auto;
    }
    
    .confirm-box {
      background: white;
      border-radius: 12px;
      padding: 24px;
      box-shadow: 0 8px 32px rgba(0,0,0,0.3);
      max-width: 500px;
      width: 90%;
      max-height: 90vh;
      overflow-y: auto;
      text-align: left;
    }
    
    .confirm-title {
      margin: 0 0 16px 0;
      font-size: 20px;
      font-weight: 600;
      color: #1a73e8;
    }
    
    
    .checkbox-label {
      display: flex;
      align-items: center;
      gap: 8px;
      font-weight: 500;
    }
    
    .checkbox-label input[type="checkbox"] {
      width: auto;
      margin: 0;
    }
    
    .confirm-actions {
      display: flex;
      gap: 12px;
      justify-content: flex-end;
      margin-top: 20px;
    }
    
    .confirm-btn {
      padding: 10px 14px;
      border-radius: 8px;
      cursor: pointer;
      border: 0;
      font-size: 16px;
      font-weight: 500;
      transition: all 0.2s;
      height: 44px;
      min-height: 44px;
      max-height: 44px;
      box-sizing: border-box;
      display: inline-flex;
      align-items: center;
      justify-content: center;
    }
    
    .confirm-btn-cancel {
      border: 1px solid #6c757d;
      background: #6c757d;
      color: white;
    }
    
    .confirm-btn-cancel:hover {
      background: #5a6268;
    }
    
    .confirm-btn-primary {
      background: #28a745;
      color: white;
    }
    
    .confirm-btn-primary:hover {
      background: #218838;
    }
    
    .confirm-btn-blue {
      background: #1a73e8;
      color: white;
    }
    
    .confirm-btn-blue:hover {
      background: #1557b0;
    }
    
    @keyframes spin {
      0% { transform: rotate(0deg); }
      100% { transform: rotate(360deg); }
    }
    
    .add-user-btn {
      display: block;
      margin: 0;
    }
    
    /* Profilbild-stilar */
    .userid-left-top {
      display: flex;
      align-items: center;
      gap: 12px;
    }
    
    .userid-profile-image {
      width: 48px;
      height: 48px;
      border-radius: 50%;
      object-fit: cover;
      cursor: pointer;
      border: 2px solid #e0e0e0;
      transition: all 0.2s;
    }
    
    .userid-profile-image:hover {
      border-color: #1a73e8;
      transform: scale(1.05);
    }
    
    .userid-profile-placeholder {
      width: 48px;
      height: 48px;
      border-radius: 50%;
      background: #f0f0f0;
      display: flex;
      align-items: center;
      justify-content: center;
      cursor: pointer;
      border: 2px solid #e0e0e0;
      color: #999;
      transition: all 0.2s;
      flex-shrink: 0;
    }
    
    .userid-profile-placeholder:hover {
      background: #e0e0e0;
      border-color: #1a73e8;
      color: #1a73e8;
    }
    
    .userid-name-wrapper {
      flex: 1;
    }
    
    .userid-name {
      cursor: pointer;
      transition: color 0.2s;
    }
    
    .userid-name:hover {
      color: #1a73e8;
    }
    
    /* Popup för bilduppladdning */
    .image-upload-overlay {
      position: fixed;
      top: 0;
      left: 0;
      width: 100%;
      height: 100%;
      background: rgba(0,0,0,0.5);
      display: flex;
      align-items: center;
      justify-content: center;
      z-index: 10001;
    }
    
    .image-upload-box {
      background: white;
      border-radius: 12px;
      padding: 24px;
      box-shadow: 0 8px 32px rgba(0,0,0,0.3);
      max-width: 500px;
      width: 90%;
    }
    
    .image-upload-preview {
      width: 200px;
      height: 200px;
      border-radius: 50%;
      object-fit: cover;
      margin: 0 auto 20px;
      display: block;
      border: 2px solid #e0e0e0;
    }
    
    .image-upload-preview-placeholder {
      width: 200px;
      height: 200px;
      border-radius: 50%;
      background: #f0f0f0;
      margin: 0 auto 20px;
      display: flex;
      align-items: center;
      justify-content: center;
      border: 2px solid #e0e0e0;
      color: #999;
    }
    
    /* Inaktiva användare: ljusröd bakgrund istället för ljusgrön */
    .userid-item.inactive {
      background-color: #f8d7da;   /* ljusröd */
    }
  </style>
</head>
<body>
  <div class="container">
    <header class="header">
      <a href="../../index.html">
        <img src="../../data/logo.png" alt="ByggArkiv Snickare Stefan" class="logo">
      </a>
      <?php
        $script = $_SERVER['SCRIPT_NAME'] ?? '';
        $admPos = strpos($script, '/adm/');
        $adminUrl = ($admPos !== false ? substr($script, 0, $admPos + 4) : '/adm') . '/admin.php';
      ?>
      <div class="back-link">
        <a href="<?= htmlspecialchars($adminUrl) ?>">← Tillbaka till admin</a>
      </div>
    </header>
    
    <div class="card">
      <h1>Hantera användare</h1>
      <p class="subtitle">Lägg till och hantera användare</p>
      <p class="userid-description">
        Klicka på länkikonen (<svg width="16" height="16" viewBox="0 0 24 24" fill="currentColor"><path d="M3.9 12c0-1.71 1.39-3.1 3.1-3.1h4V7H7c-2.76 0-5 2.24-5 5s2.24 5 5 5h4v-1.9H7c-1.71 0-3.1-1.39-3.1-3.1zM8 13h8v-2H8v2zm9-6h-4v1.9h4c1.71 0 3.1 1.39 3.1 3.1s-1.39 3.1-3.1 3.1h-4V17h4c2.76 0 5-2.24 5-5s-2.24-5-5-5z"/></svg>) bredvid varje användare för att se länk till dennes personliga uppladdningsformulär. 
        Denna länk kan delas med den tilltänkte användaren så att de kan ladda upp bilder och dokument direkt till sitt projekt.
      </p>
      
      <div class="add-form">
        <button id="addUserBtn" onclick="showAddUserPopup()" class="add-user-btn">
          Lägg till
        </button>
      </div>
      
      <div class="userid-list" id="useridList">
        <!-- User IDs kommer att laddas här -->
      </div>

      <!-- Popup för att lägga till ny användare -->
      <div id="addUserPopup" class="confirm-overlay" style="display: none;">
        <div class="confirm-box" style="max-width: 500px;">
          <h3 class="confirm-title">Lägg till</h3>
          <p class="form-description">
            UUID och länk-ID genereras automatiskt för nya användare.
          </p>
          <form id="addUserForm">
            <div class="form-group">
              <label for="newFullName">Namn:</label>
              <input type="text" id="newFullName" placeholder="t.ex. Erik Karlsson" maxlength="100" required>
            </div>
            <div class="form-group">
              <label for="newNickname">Nickname (valfritt, unikt):</label>
              <input type="text" id="newNickname" placeholder="Kortnamn som visas i chatten" maxlength="50">
              <div class="form-help" style="font-size: 12px; color: #666; margin-top: 4px;">
                Om satt, visas detta istället för fullständigt namn i chatten. Måste vara unikt.
              </div>
            </div>
            <div class="form-group">
              <label for="newMobile">Mobilnummer:</label>
              <input type="text" id="newMobile" placeholder="t.ex. 070-123 45 67" maxlength="20" required>
            </div>
            <div class="form-group">
              <label for="newPushoverUserKey">Pushover User Key (valfritt):</label>
              <input type="text" id="newPushoverUserKey" placeholder="u1234567890abcdefghijklmnop" maxlength="50" style="font-family: monospace;">
              <div class="form-help" style="font-size: 12px; color: #666; margin-top: 4px;">
                Hämta från Pushover-appen: Inställningar → Your User Key
              </div>
            </div>
            <div class="form-group">
              <label for="newUserType">Användartyp:</label>
              <select id="newUserType" style="padding: 10px; border: 1px solid #ccc; border-radius: 8px; font-size: 16px; width: 100%;">
                <option value="user">Användare</option>
                <option value="admin">Administratör</option>
              </select>
              <div class="form-help">Administratör ger åtkomst till adminsidor</div>
            </div>
            <div class="confirm-actions">
              <button type="submit" class="confirm-btn confirm-btn-primary">Lägg till</button>
              <button type="button" onclick="hideAddUserPopup()" class="confirm-btn confirm-btn-cancel">Avbryt</button>
            </div>
          </form>
        </div>
      </div>
      
      <!-- Popup för att redigera användare -->
      <div id="editUserPopup" class="confirm-overlay" style="display: none;">
        <div class="confirm-box" style="max-width: 500px;">
          <h3 class="confirm-title">Redigera användare</h3>
          <form id="editUserForm">
            <div class="form-group">
              <label for="editUserId">User ID:</label>
              <input type="text" id="editUserId" readonly style="background: #f5f5f5; cursor: not-allowed;">
            </div>
            <div class="form-group">
              <label for="editFullName">Namn:</label>
              <input type="text" id="editFullName" placeholder="t.ex. Erik Karlsson" maxlength="100" required>
            </div>
            <div class="form-group">
              <label for="editNickname">Nickname (valfritt, unikt):</label>
              <input type="text" id="editNickname" placeholder="Kortnamn som visas i chatten" maxlength="50">
              <div class="form-help" style="font-size: 12px; color: #666; margin-top: 4px;">
                Om satt, visas detta istället för fullständigt namn i chatten. Måste vara unikt.
              </div>
            </div>
            <div class="form-group">
              <label for="editMobile">Mobilnummer:</label>
              <input type="text" id="editMobile" placeholder="t.ex. 070-123 45 67" maxlength="20" required>
            </div>
            <div class="form-group">
              <label for="editPushoverUserKey">Pushover User Key (valfritt):</label>
              <input type="text" id="editPushoverUserKey" placeholder="u1234567890abcdefghijklmnop" maxlength="50" style="font-family: monospace;">
              <div class="form-help" style="font-size: 12px; color: #666; margin-top: 4px;">
                Hämta från Pushover-appen: Inställningar → Your User Key
              </div>
            </div>
            <div class="form-group">
              <label for="editUserType">Användartyp:</label>
              <select id="editUserType" style="padding: 10px; border: 1px solid #ccc; border-radius: 8px; font-size: 16px; width: 100%;">
                <option value="user">Användare</option>
                <option value="admin">Administratör</option>
              </select>
              <div class="form-help">Administratör ger åtkomst till adminsidor</div>
            </div>
            <div class="form-group">
              <label for="editNotes">Anteckningar (valfritt):</label>
              <textarea id="editNotes" rows="3" placeholder="Interna anteckningar om denna användare..." style="width: 100%; padding: 10px; border: 1px solid #ccc; border-radius: 8px; font-size: 14px; font-family: inherit; resize: vertical; box-sizing: border-box;"></textarea>
            </div>
            <div class="form-group">
              <label for="editPublicNotes">Publika anteckningar (valfritt):</label>
              <textarea id="editPublicNotes" rows="3" placeholder="Anteckningar som visas i visitkortet..." style="width: 100%; padding: 10px; border: 1px solid #ccc; border-radius: 8px; font-size: 14px; font-family: inherit; resize: vertical; box-sizing: border-box;"></textarea>
              <div class="form-help" style="font-size: 12px; color: #666; margin-top: 4px;">
                Dessa anteckningar visas när visitkortet öppnas, t.ex. i chatten
              </div>
            </div>
            <div class="form-group">
              <div class="checkbox-group">
                <input type="checkbox" id="editInactive">
                <label for="editInactive" class="checkbox-label">Markera som inaktiv (användaren är inte längre aktuell)</label>
              </div>
            </div>
            <div class="confirm-actions">
              <button type="submit" class="confirm-btn confirm-btn-primary">Spara</button>
              <button type="button" onclick="hideEditUserPopup()" class="confirm-btn confirm-btn-cancel">Avbryt</button>
            </div>
          </form>
        </div>
      </div>
      
      <div id="status" class="status"></div>
    </div>
  </div>

  <script>
    let userIds = [];
    
    function loadUserIds() {
      fetch('../api/users.php?action=list')
        .then(response => {
          if (!response.ok) {
            if (response.status === 403) {
              showStatus('Åtkomst nekad. Logga in som admin.', 'error');
            } else {
              showStatus('Kunde inte ladda användare (' + response.status + ')', 'error');
            }
            throw new Error('HTTP ' + response.status);
          }
          return response.text().then(txt => {
            try {
              return JSON.parse(txt);
            } catch (e) {
              throw new Error('Non-JSON response: ' + txt.slice(0, 200));
            }
          });
        })
        .then(data => {
          if (data && data.success) {
            userIds = data.users || [];
            displayUserIds();
          } else {
            showStatus('Kunde inte ladda användare', 'error');
          }
        })
        .catch(error => {
          console.error('Error:', error);
          // status visas redan ovan
        });
    }
    
    function displayUserIds() {
      const listElement = document.getElementById('useridList');
      
      if (!userIds || userIds.length === 0) {
        listElement.innerHTML = '<p class="no-results">Inga användare hittades</p>';
        return;
      }
      
      const validUsers = userIds.filter(user => user && user.user_id);
      const sortedUsers = validUsers.sort((a, b) => {
        // Sortera inaktiva sist
        const aInactive = a.inactive === true || a.inactive === 1 || a.inactive === '1' || a.inactive === 'true' || 
                          (a.active === false || a.active === 0 || a.active === '0' || a.active === 'false');
        const bInactive = b.inactive === true || b.inactive === 1 || b.inactive === '1' || b.inactive === 'true' || 
                          (b.active === false || b.active === 0 || b.active === '0' || b.active === 'false');
        
        if (aInactive !== bInactive) {
          return aInactive ? 1 : -1; // Inaktiva sist
        }
        
        // Sortera efter förnamn (första ordet i full_name)
        const aFirstName = (a.full_name || '').split(' ')[0] || '';
        const bFirstName = (b.full_name || '').split(' ')[0] || '';
        return aFirstName.localeCompare(bFirstName, 'sv');
      });
      
      listElement.innerHTML = sortedUsers.map(user => {
        const isUuid = isUUID(user.user_id);
        const profileImageUrl = user.profile_image ? '../api/users.php?action=get_image&user_id=' + encodeURIComponent(user.user_id) + '&t=' + Date.now() : null;
        
        const userType = user.user_type || 'user';
        const adminClass = userType === 'admin' ? 'admin' : '';
        
        // Skapa profilbild HTML-sträng
        // Använd direkt URL utan HTML-encoding för src-attributet (webbläsaren hanterar det korrekt)
        const safeUserId = user.user_id.replace(/'/g, "\\'");
        const isInactive = user.inactive === true || user.inactive === 1 || user.inactive === '1' || user.inactive === 'true' || 
                           (user.active === false || user.active === 0 || user.active === '0' || user.active === 'false');
        let profileImageHtml = '';
        
        // Om användaren är inaktiv, gör profilbilden icke-klickbar
        if (isInactive) {
          if (profileImageUrl) {
            // För inaktiva användare med profilbild: visa endast bilden, ingen placeholder
            profileImageHtml = '<img src="' + profileImageUrl + '" alt="Profilbild" class="userid-profile-image" style="cursor: not-allowed; opacity: 0.6;" title="Inaktiv användare - profilbild kan inte ändras" onerror="this.onerror=null; this.style.display=\'none\'; const ph=this.nextElementSibling; if(ph) ph.style.display=\'flex\';">' +
              '<div class="userid-profile-placeholder" style="cursor: not-allowed; opacity: 0.6; display:none !important;" title="Inaktiv användare - profilbild kan inte ändras">' +
                '<svg width="24" height="24" viewBox="0 0 24 24" fill="currentColor"><path d="M12 12c2.21 0 4-1.79 4-4s-1.79-4-4-4-4 1.79-4 4 1.79 4 4 4zm0 2c-2.67 0-8 1.34-8 4v2h16v-2c0-2.66-5.33-4-8-4z"/></svg>' +
              '</div>';
          } else {
            profileImageHtml = '<div class="userid-profile-placeholder" style="cursor: not-allowed; opacity: 0.6;" title="Inaktiv användare - profilbild kan inte ändras">' +
              '<svg width="24" height="24" viewBox="0 0 24 24" fill="currentColor"><path d="M12 12c2.21 0 4-1.79 4-4s-1.79-4-4-4-4 1.79-4 4 1.79 4 4 4zm0 2c-2.67 0-8 1.34-8 4v2h16v-2c0-2.66-5.33-4-8-4z"/></svg>' +
            '</div>';
          }
        } else {
          if (profileImageUrl) {
            profileImageHtml = '<img src="' + profileImageUrl + '" alt="Profilbild" class="userid-profile-image" onclick="uploadUserImage(\'' + safeUserId + '\')" title="Klicka för att ändra bild" onerror="this.onerror=null; this.style.display=\'none\'; const ph=this.nextElementSibling; if(ph) ph.style.display=\'flex\';">' +
              '<div class="userid-profile-placeholder" onclick="uploadUserImage(\'' + safeUserId + '\')" title="Klicka för att ladda upp bild" style="display:none;">' +
                '<svg width="24" height="24" viewBox="0 0 24 24" fill="currentColor"><path d="M12 12c2.21 0 4-1.79 4-4s-1.79-4-4-4-4 1.79-4 4 1.79 4 4 4zm0 2c-2.67 0-8 1.34-8 4v2h16v-2c0-2.66-5.33-4-8-4z"/></svg>' +
              '</div>';
          } else {
            profileImageHtml = '<div class="userid-profile-placeholder" onclick="uploadUserImage(\'' + safeUserId + '\')" title="Klicka för att ladda upp bild">' +
              '<svg width="24" height="24" viewBox="0 0 24 24" fill="currentColor"><path d="M12 12c2.21 0 4-1.79 4-4s-1.79-4-4-4-4 1.79-4 4 1.79 4 4 4zm0 2c-2.67 0-8 1.34-8 4v2h16v-2c0-2.66-5.33-4-8-4z"/></svg>' +
            '</div>';
          }
        }
        
        return '<div class="userid-item ' + (user.inactive ? 'inactive' : '') + ' ' + adminClass + ' ' + (isUuid ? 'has-uuid' : 'no-uuid') + '" id="user-' + user.user_id + '" title="UUID: ' + user.user_id + '">' +
          '<div class="userid-content">' +
            '<div class="userid-left">' +
              '<div class="userid-left-top">' +
                profileImageHtml +
                '<div class="userid-name-wrapper">' +
                  '<div class="userid-name" onmouseenter="showUserCard(\'' + safeUserId + '\', event)" onmouseleave="hideUserCard()">' +
                    (user.full_name || 'Namn saknas') +
                  '</div>' +
                  '<div class="userid-details">' +
                    (user.mobile || 'Mobil saknas') +
                  '</div>' +
                '</div>' +
              '</div>' +
            '</div>' +
            '<div class="userid-middle">' +
              '<div class="userid-id-row">u=' + (user.user_id || 'Saknas') + '</div>' +
              '<div class="userid-pushover-row">p=' + (user.pushover_user_key || 'Saknas') + '</div>' +
            '</div>' +
            '<div class="userid-actions">' +
              '<button class="action-btn btn-link" onclick="openUserLink(\'' + user.user_id + '\')" title="Kopiera app-länk">' +
                '<svg width="20" height="20" viewBox="0 0 24 24" fill="currentColor">' +
                  '<path d="M3.9 12c0-1.71 1.39-3.1 3.1-3.1h4V7H7c-2.76 0-5 2.24-5 5s2.24 5 5 5h4v-1.9H7c-1.71 0-3.1-1.39-3.1-3.1zM8 13h8v-2H8v2zm9-6h-4v1.9h4c1.71 0 3.1 1.39 3.1 3.1s-1.39 3.1-3.1 3.1h-4V17h4c2.76 0 5-2.24 5-5s-2.24-5-5-5z"/>' +
                '</svg>' +
              '</button>' +
              '<button class="action-btn btn-edit" onclick="editUser(\'' + user.user_id + '\')" title="Redigera användare">' +
                '<svg width="20" height="20" viewBox="0 0 24 24" fill="currentColor">' +
                  '<path d="M3 17.25V21h3.75L17.81 9.94l-3.75-3.75L3 17.25zM20.71 7.04c.39-.39.39-1.02 0-1.41l-2.34-2.34c-.39-.39-1.02-.39-1.41 0l-1.83 1.83 3.75 3.75 1.83-1.83z"/>' +
                '</svg>' +
              '</button>' +
              (userType === 'admin' ? (
                '<button class="action-btn btn-admin" onclick="openAdminLink(\'' + user.user_id + '\')" title="Öppna admin med denna användare">' +
                  '<svg width="20" height="20" viewBox="0 0 24 24" fill="currentColor">' +
                    '<path d="M19.14,12.94 C19.19,12.64 19.22,12.32 19.22,12 C19.22,11.68 19.19,11.36 19.14,11.06 L21.19,9.47 C21.37,9.33 21.43,9.08 21.34,8.87 L19.34,5.13 C19.24,4.92 19.01,4.84 18.8,4.91 L16.34,5.86 C15.83,5.47 15.27,5.15 14.66,4.92 L14.31,2.3 C14.28,2.08 14.09,1.92 13.87,1.92 L10.13,1.92 C9.91,1.92 9.72,2.08 9.69,2.3 L9.34,4.92 C8.73,5.15 8.17,5.47 7.66,5.86 L5.2,4.91 C4.99,4.84 4.76,4.92 4.66,5.13 L2.66,8.87 C2.57,9.08 2.63,9.33 2.81,9.47 L4.86,11.06 C4.81,11.36 4.78,11.68 4.78,12 C4.78,12.32 4.81,12.64 4.86,12.94 L2.81,14.53 C2.63,14.67 2.57,14.92 2.66,15.13 L4.66,18.87 C4.76,19.08 4.99,19.16 5.2,19.09 L7.66,18.14 C8.17,18.53 8.73,18.85 9.34,19.08 L9.69,21.7 C9.72,21.92 9.91,22.08 10.13,22.08 L13.87,22.08 C14.09,22.08 14.28,21.92 14.31,21.7 L14.66,19.08 C15.27,18.85 15.83,18.53 16.34,18.14 L18.8,19.09 C19.01,19.16 19.24,19.08 19.34,18.87 L21.34,15.13 C21.43,14.92 21.37,14.67 21.19,14.53 L19.14,12.94 Z M12,15.6 C10.01,15.6 8.4,13.99 8.4,12 C8.4,10.01 10.01,8.4 12,8.4 C13.99,8.4 15.6,10.01 15.6,12 C15.6,13.99 13.99,15.6 12,15.6 Z" />' +
                  '</svg>' +
                '</button>'
              ) : '') +
            '</div>' +
          '</div>' +
        '</div>';
      }).join('');
    }
    
    // Visa popup för att lägga till ny användare
    function showAddUserPopup() {
      document.getElementById('addUserPopup').style.display = 'flex';
      // Förhindra scrollning på bakgrunden
      document.body.style.overflow = 'hidden';
      // Rensa formuläret
      document.getElementById('addUserForm').reset();
    }
    
    // Dölj popup för att lägga till ny användare
    function hideAddUserPopup() {
      document.getElementById('addUserPopup').style.display = 'none';
      // Återställ scrollning på bakgrunden
      document.body.style.overflow = '';
    }
    
    function addUserId() {
      const inputFullName = document.getElementById('newFullName');
      const inputMobile = document.getElementById('newMobile');
      const inputPushoverUserKey = document.getElementById('newPushoverUserKey');
      const inputUserType = document.getElementById('newUserType');
      const inputNickname = document.getElementById('newNickname');
      
      const newFullName = inputFullName.value.trim();
      const newMobile = inputMobile.value.trim();
      const newPushoverUserKey = inputPushoverUserKey ? inputPushoverUserKey.value.trim() : '';
      const newUserType = inputUserType ? inputUserType.value : 'user';
      const newNickname = inputNickname ? inputNickname.value.trim() : '';
      
      if (!newFullName) {
        showStatus('Ange fullständigt namn', 'error');
        return;
      }
      
      if (!newMobile) {
        showStatus('Ange mobilnummer', 'error');
        return;
      }
      
      // Generera UUID för ny användare
      const newUserId = generateUUID();
      
      const formData = new FormData();
      formData.append('userId', newUserId);
      formData.append('fullName', newFullName);
      formData.append('mobile', newMobile);
      if (newPushoverUserKey) formData.append('pushoverUserKey', newPushoverUserKey);
      if (newNickname) formData.append('nickname', newNickname);
      formData.append('userType', newUserType);
      // För bakåtkompatibilitet, skicka även admin-flagga om user_type är admin
      if (newUserType === 'admin') {
        formData.append('admin', '1');
      }
      
      fetch('../api/users.php?action=add', {
        method: 'POST',
        body: formData
      })
      .then(response => {
        if (!response.ok) {
          if (response.status === 403) showStatus('Åtkomst nekad. Logga in som admin.', 'error');
          throw new Error('HTTP ' + response.status);
        }
        return response.text().then(txt => {
          try { return JSON.parse(txt); } catch (e) { throw new Error('Non-JSON response: ' + txt.slice(0, 200)); }
        });
      })
      .then(data => {
        if (data.success && data.user) {
          userIds.push(data.user);
          displayUserIds();
          hideAddUserPopup(); // Stäng popup-fönstret
          showStatus('Användare har skapats med UUID', 'success');
        } else {
          showStatus('Kunde inte lägga till användare', 'error');
        }
      })
      .catch(error => {
        console.error('Error:', error);
        if (!/HTTP 403/.test(String(error))) showStatus('Fel vid tillägg', 'error');
      });
    }
    
    function editUser(userId) {
      // Hitta användaren i listan
      const user = userIds.find(u => u.user_id === userId);
      if (!user) {
        showStatus('Användare hittades inte', 'error');
        return;
      }
      
      // Fyll i formuläret med användarens data
      document.getElementById('editUserId').value = user.user_id || '';
      document.getElementById('editFullName').value = user.full_name || '';
      document.getElementById('editNickname').value = user.nickname || '';
      document.getElementById('editMobile').value = user.mobile || '';
      document.getElementById('editPushoverUserKey').value = user.pushover_user_key || '';
      
      // Sätt användartyp
      const userType = user.user_type || 'user';
      document.getElementById('editUserType').value = userType;
      
      // Sätt anteckningar
      document.getElementById('editNotes').value = user.notes || '';
      
      // Sätt publika anteckningar
      document.getElementById('editPublicNotes').value = user.public_notes || '';
      
      // Sätt inaktiv-kryssruta
      document.getElementById('editInactive').checked = user.inactive || false;
      
      // Visa popupen
      document.getElementById('editUserPopup').style.display = 'flex';
      // Förhindra scrollning på bakgrunden
      document.body.style.overflow = 'hidden';
    }
    
    // Dölj popup för att redigera användare
    function hideEditUserPopup() {
      document.getElementById('editUserPopup').style.display = 'none';
      // Återställ scrollning på bakgrunden
      document.body.style.overflow = '';
      // Rensa formuläret
      document.getElementById('editUserForm').reset();
    }
    
    // Spara ändringar för redigerad användare
    function saveEditedUser() {
      const userId = document.getElementById('editUserId').value;
      const fullName = document.getElementById('editFullName').value.trim();
      const mobile = document.getElementById('editMobile').value.trim();
      const pushoverUserKey = document.getElementById('editPushoverUserKey').value.trim();
      const userType = document.getElementById('editUserType').value;
      const notes = document.getElementById('editNotes').value.trim();
      const publicNotes = document.getElementById('editPublicNotes').value.trim();
      const inactive = document.getElementById('editInactive').checked;
      const nickname = document.getElementById('editNickname').value.trim();
      
      if (!fullName) {
        showStatus('Ange fullständigt namn', 'error');
        return;
      }
      
      if (!mobile) {
        showStatus('Ange mobilnummer', 'error');
        return;
      }
      
      const formData = new FormData();
      formData.append('userId', userId);
      formData.append('fullName', fullName);
      formData.append('mobile', mobile);
      formData.append('pushoverUserKey', pushoverUserKey); // Skicka alltid, även om tom, så att API:et kan ta bort den
      formData.append('nickname', nickname); // Skicka alltid, även om tom, så att API:et kan ta bort den
      formData.append('userType', userType);
      formData.append('notes', notes);
      formData.append('publicNotes', publicNotes);
      formData.append('inactive', inactive ? '1' : '0');
      // För bakåtkompatibilitet, skicka även admin-flagga om user_type är admin
      if (userType === 'admin') {
        formData.append('admin', '1');
      }
      
      fetch('../api/users.php?action=update', {
        method: 'POST',
        body: formData
      })
      .then(response => {
        if (!response.ok) {
          if (response.status === 403) showStatus('Åtkomst nekad. Logga in som admin.', 'error');
          throw new Error('HTTP ' + response.status);
        }
        return response.text().then(txt => {
          try { return JSON.parse(txt); } catch (e) { throw new Error('Non-JSON response: ' + txt.slice(0, 200)); }
        });
      })
      .then(data => {
        if (data.success === true || data.success === 'true') {
          // Stäng popupen efter lyckad uppdatering
          hideEditUserPopup();
          // Ladda om listan från servern för att säkerställa att allt är synkroniserat
          setTimeout(() => {
            loadUserIds();
          }, 100);
        } else {
          showStatus('Kunde inte uppdatera användare: ' + (data.message || 'Okänt fel'), 'error');
        }
      })
      .catch(error => {
        console.error('Error:', error);
        if (!/HTTP 403/.test(String(error))) showStatus('Fel vid uppdatering: ' + error.message, 'error');
      });
    }
    
    
    
    function isUUID(str) {
      if (!str || typeof str !== 'string') {
        return false;
      }
      const uuidRegex = /^[0-9a-f]{8}-[0-9a-f]{4}-4[0-9a-f]{3}-[89ab][0-9a-f]{3}-[0-9a-f]{12}$/i;
      return uuidRegex.test(str);
    }
    
    function generateUUID() {
      return 'xxxxxxxx-xxxx-4xxx-yxxx-xxxxxxxxxxxx'.replace(/[xy]/g, function(c) {
        const r = Math.random() * 16 | 0;
        const v = c === 'x' ? r : (r & 0x3 | 0x8);
        return v.toString(16);
      });
    }
    
    // Escape HTML för säkerhet
    function escapeHtml(text) {
      if (!text) return '';
      const div = document.createElement('div');
      div.textContent = text;
      return div.innerHTML;
    }
    
    let userCardTimeout = null;
    let currentUserCard = null;
    
    // Visa visitkort när man hovrar över namnet
    async function showUserCard(userId, event) {
      // Rensa tidigare timeout
      if (userCardTimeout) {
        clearTimeout(userCardTimeout);
        userCardTimeout = null;
      }
      
      // Ta bort tidigare visitkort om det finns
      if (currentUserCard) {
        currentUserCard.remove();
        currentUserCard = null;
      }
      
      // Vänta lite innan visitkortet visas (för att undvika att det blinkar)
      userCardTimeout = setTimeout(async () => {
        try {
          // Hämta användardata
          const response = await fetch(`../../chat/participant_info_api.php?participant_id=${encodeURIComponent(userId)}`);
          const data = await response.json();
          
          if (!data.success || !data.data) {
            return; // Tyst misslyckande
          }
          
          const userData = data.data;
          
          // Hämta position för namnet
          const nameElement = event.target;
          const rect = nameElement.getBoundingClientRect();
          
          // Beräkna position för visitkortet
          let top = rect.bottom + 10;
          let left = rect.left;
          
          // Kontrollera om visitkortet hamnar utanför skärmen
          const cardWidth = 350;
          const cardHeight = 400; // Uppskattad höjd
          
          // Justera horisontellt om det hamnar utanför högerkanten
          if (left + cardWidth > window.innerWidth) {
            left = window.innerWidth - cardWidth - 10;
          }
          
          // Justera vertikalt om det hamnar utanför nedre kanten
          if (top + cardHeight > window.innerHeight) {
            top = rect.top - cardHeight - 10;
          }
          
          // Säkerställ att visitkortet inte hamnar utanför vänsterkanten
          if (left < 10) {
            left = 10;
          }
          
          // Säkerställ att visitkortet inte hamnar utanför övre kanten
          if (top < 10) {
            top = rect.bottom + 10;
          }
          
          // Skapa visitkort
          const card = document.createElement('div');
          card.className = 'user-card-hover';
          card.style.cssText = `
            position: fixed;
            top: ${top}px;
            left: ${left}px;
            background: white;
            border-radius: 12px;
            padding: 20px;
            max-width: 350px;
            width: auto;
            box-shadow: 0 4px 20px rgba(0,0,0,0.3);
            text-align: center;
            z-index: 10000;
            pointer-events: auto;
          `;
          
          // Profilbild URL
          let profileImageUrl = null;
          if (userData.profile_image) {
            if (userData.type === 'user' || userData.type === 'admin') {
              profileImageUrl = `../api/users.php?action=get_image&user_id=${encodeURIComponent(userId)}&t=${Date.now()}`;
            } else if (userData.type === 'staff') {
              profileImageUrl = `../../api/chat/external_staff.php?action=get_image&id=${encodeURIComponent(userId)}&t=${Date.now()}`;
            }
          }
          
          let html = '';
          
          // Profilbild
          html += '<div style="margin-bottom: 16px;">';
          if (profileImageUrl) {
            html += `<img src="${escapeHtml(profileImageUrl)}" alt="Profilbild" style="width: 120px; height: 120px; border-radius: 50%; object-fit: cover; border: 2px solid #e0e0e0; display: block; margin: 0 auto;" onerror="this.style.display='none'; const placeholder = this.nextElementSibling; if(placeholder) placeholder.style.display='flex';">`;
          }
          html += `<div style="width: 120px; height: 120px; border-radius: 50%; background: #f0f0f0; display: ${profileImageUrl ? 'none' : 'flex'}; align-items: center; justify-content: center; margin: 0 auto; border: 2px solid #e0e0e0;">`;
          html += '<svg width="60" height="60" viewBox="0 0 24 24" fill="#999"><path d="M12 12c2.21 0 4-1.79 4-4s-1.79-4-4-4-4 1.79-4 4 1.79 4 4 4zm0 2c-2.67 0-8 1.34-8 4v2h16v-2c0-2.66-5.33-4-8-4z"/></svg>';
          html += '</div>';
          html += '</div>';
          
          // Namn
          html += `<h3 style="margin: 0 0 10px 0; color: #333; font-size: 20px;">${escapeHtml(userData.name)}</h3>`;
          
          // Yrkesroll
          if (userData.role) {
            html += '<div style="margin-bottom: 12px;">';
            html += `<span style="color: #666; font-size: 14px; font-style: italic;">${escapeHtml(userData.role)}</span>`;
            html += '</div>';
          }
          
          // Telefonnummer
          if (userData.mobile) {
            const cleanPhone = userData.mobile.replace(/[\s\-\(\)]/g, '');
            html += '<div style="margin-bottom: 12px; display: flex; align-items: center; justify-content: center; gap: 6px;">';
            html += '<svg width="18" height="18" viewBox="0 0 24 24" fill="#666"><path d="M6.62 10.79c1.44 2.83 3.76 5.14 6.59 6.59l2.2-2.2c.27-.27.67-.36 1.02-.24 1.12.37 2.33.57 3.57.57.55 0 1 .45 1 1V20c0 .55-.45 1-1 1-9.39 0-17-7.61-17-17 0-.55.45-1 1-1h3.5c.55 0 1 .45 1 1 0 1.25.2 2.45.57 3.57.11.35.03.74-.25 1.02l-2.2 2.2z"/></svg>';
            html += `<a href="tel:${escapeHtml(cleanPhone)}" style="color: #1a73e8; font-size: 14px; text-decoration: none; font-weight: 500;">${escapeHtml(userData.mobile)}</a>`;
            html += '</div>';
          }
          
          // Företag (endast för staff)
          if (userData.type === 'extern' && userData.company_name) {
            html += '<div style="margin-bottom: 12px; display: flex; align-items: center; justify-content: center; gap: 6px;">';
            html += '<svg width="18" height="18" viewBox="0 0 24 24" fill="#666"><path d="M12 7V3H2v18h20V7H12zM6 19H4v-2h2v2zm0-4H4v-2h2v2zm0-4H4V9h2v2zm0-4H4V5h2v2zm4 12H8v-2h2v2zm0-4H8v-2h2v2zm0-4H8V9h2v2zm0-4H8V5h2v2zm10 12h-8v-2h2v-2h-2v-2h2v-2h-2V9h8v10zm-2-8h-2v2h2v-2zm0 4h-2v2h2v-2z"/></svg>';
            html += `<span style="color: #666; font-size: 14px;">${escapeHtml(userData.company_name)}</span>`;
            html += '</div>';
          }
          
          // Publika anteckningar
          if (userData.public_notes) {
            html += '<div style="margin-top: 12px; padding: 12px; background: #f5f5f5; border-radius: 8px; text-align: left;">';
            html += `<div style="color: #333; font-size: 13px; line-height: 1.5; white-space: pre-wrap;">${escapeHtml(userData.public_notes)}</div>`;
            html += '</div>';
          }
          
          card.innerHTML = html;
          document.body.appendChild(card);
          currentUserCard = card;
          
          // Håll visitkortet synligt när man hovrar över det
          card.addEventListener('mouseenter', () => {
            if (userCardTimeout) {
              clearTimeout(userCardTimeout);
              userCardTimeout = null;
            }
          });
          
          card.addEventListener('mouseleave', () => {
            hideUserCard();
          });
          
        } catch (error) {
          // Tyst misslyckande
        }
      }, 300); // 300ms delay innan visitkortet visas
    }
    
    // Dölj visitkort
    function hideUserCard() {
      if (userCardTimeout) {
        clearTimeout(userCardTimeout);
        userCardTimeout = null;
      }
      
      if (currentUserCard) {
        currentUserCard.remove();
        currentUserCard = null;
      }
    }
    
    function generateUUIDForUser(userId) {
      const formData = new FormData();
      formData.append('userId', userId);
      
      return fetch('../api/users.php?action=generate_uuid', {
        method: 'POST',
        body: formData
      })
      .then(response => {
        if (!response.ok) {
          if (response.status === 403) showStatus('Åtkomst nekad. Logga in som admin.', 'error');
          throw new Error('HTTP ' + response.status);
        }
        return response.text().then(txt => {
          try { return JSON.parse(txt); } catch (e) { throw new Error('Non-JSON response: ' + txt.slice(0, 200)); }
        });
      })
      .then(result => {
        if (result.success) {
          const userIndex = userIds.findIndex(u => u.user_id === userId);
          if (userIndex !== -1) {
            userIds[userIndex] = result.user;
            displayUserIds();
          }
          return result.user;
        } else {
          showStatus('Fel vid generering av UUID', 'error');
          return null;
        }
      })
      .catch(error => {
        console.error('Error:', error);
        if (!/HTTP 403/.test(String(error))) showStatus('Fel vid generering av UUID', 'error');
        return null;
      });
    }
    
    function openUserLink(userId) {
      if (!isUUID(userId)) {
        showStatus('Genererar UUID för användare...', 'info');
        generateUUIDForUser(userId).then(updatedUser => {
          if (updatedUser && updatedUser.uuid) {
            showUserLinkPopup(updatedUser.uuid);
            showStatus('UUID genererat!', 'success');
          }
        });
      } else {
        showUserLinkPopup(userId);
      }
    }
    
    function showUserLinkPopup(userId) {
      const baseUrl = window.location.origin + window.location.pathname.replace('/adm/users/userids.php', '');
      const userLink = baseUrl + '/external_landing.php?user_id=' + userId;
      
      // Skapa popup
      const popup = document.createElement('div');
      popup.style.cssText = `
        position: fixed;
        top: 0;
        left: 0;
        width: 100%;
        height: 100%;
        background: rgba(0,0,0,0.5);
        display: flex;
        align-items: center;
        justify-content: center;
        z-index: 10000;
      `;
      
      const popupContent = document.createElement('div');
      popupContent.style.cssText = `
        background: white;
        padding: 30px;
        border-radius: 12px;
        box-shadow: 0 8px 32px rgba(0,0,0,0.3);
        max-width: 500px;
        width: 90%;
        text-align: center;
      `;
      
      popupContent.innerHTML = `
        <h3 class="popup-title">Landningslänk för användare</h3>
        <p class="popup-description">Kopiera denna länk och skicka till användaren:</p>
        <div class="popup-code">
          ${userLink}
        </div>
        <div class="popup-buttons">
          <button onclick="copyToClipboard('${userLink}'); this.textContent='Kopierad!'; setTimeout(() => this.textContent='Kopiera länk', 2000)" class="popup-btn popup-btn-primary">
            Kopiera länk
          </button>
          <button onclick="this.closest('.popup-overlay').remove()" class="popup-btn popup-btn-secondary">
            Stäng
          </button>
        </div>
      `;
      
      popup.className = 'popup-overlay';
      popup.appendChild(popupContent);
      document.body.appendChild(popup);
      
      // Stäng popup när man klickar utanför
      popup.addEventListener('click', (e) => {
        if (e.target === popup) {
          popup.remove();
        }
      });
    }
    
    function copyToClipboard(text) {
      navigator.clipboard.writeText(text).then(() => {
        showStatus('Länk kopierad till urklipp!', 'success');
      }).catch(() => {
        // Fallback för äldre webbläsare
        const textArea = document.createElement('textarea');
        textArea.value = text;
        document.body.appendChild(textArea);
        textArea.select();
        document.execCommand('copy');
        document.body.removeChild(textArea);
        showStatus('Länk kopierad till urklipp!', 'success');
      });
    }

    function openAdminLink(userId) {
      const baseUrl = window.location.origin + window.location.pathname.replace('/adm/users/userids.php', '');
      const adminLink = baseUrl + '/adm/check_admin.php?user_id=' + userId;
      
      // Skapa popup
      const popup = document.createElement('div');
      popup.style.cssText = `
        position: fixed;
        top: 0;
        left: 0;
        width: 100%;
        height: 100%;
        background: rgba(0,0,0,0.5);
        display: flex;
        align-items: center;
        justify-content: center;
        z-index: 10000;
      `;
      
      const popupContent = document.createElement('div');
      popupContent.style.cssText = `
        background: white;
        padding: 30px;
        border-radius: 12px;
        box-shadow: 0 8px 32px rgba(0,0,0,0.3);
        max-width: 500px;
        width: 90%;
        text-align: center;
      `;
      
      popupContent.innerHTML = `
        <h3 class="popup-title">Admin-länk för användare</h3>
        <p class="popup-description">Kopiera denna länk för att ge användaren admin-åtkomst:</p>
        <div class="popup-code">
          ${adminLink}
        </div>
        <div class="popup-buttons">
          <button onclick="copyToClipboard('${adminLink}'); this.textContent='Kopierad!'; setTimeout(() => this.textContent='Kopiera länk', 2000)" class="popup-btn popup-btn-primary">
            Kopiera länk
          </button>
          <button onclick="this.closest('.popup-overlay').remove()" class="popup-btn popup-btn-secondary">
            Stäng
          </button>
        </div>
      `;
      
      popup.className = 'popup-overlay';
      popup.appendChild(popupContent);
      document.body.appendChild(popup);
      
      // Stäng popup när man klickar utanför
      popup.addEventListener('click', (e) => {
        if (e.target === popup) {
          popup.remove();
        }
      });
    }
    
    function showStatus(message, type) {
      const statusElement = document.getElementById('status');
      statusElement.textContent = message;
      statusElement.className = 'status ' + type;
      statusElement.style.display = 'block';
      
      setTimeout(function() {
        statusElement.style.display = 'none';
      }, 5000);
    }
    
    // Funktion för att visa bilduppladdningspopup
    function uploadUserImage(userId) {
      // Hitta användaren för att få nuvarande bild
      const user = userIds.find(u => u.user_id === userId);
      
      // Kontrollera om användaren är inaktiv
      if (user) {
        const isInactive = user.inactive === true || user.inactive === 1 || user.inactive === '1' || user.inactive === 'true' || 
                           (user.active === false || user.active === 0 || user.active === '0' || user.active === 'false');
        if (isInactive) {
          alert('Du kan inte ändra profilbild på en inaktiv användare');
          return;
        }
      }
      
      const currentImageUrl = user && user.profile_image ? '../api/users.php?action=get_image&user_id=' + encodeURIComponent(userId) + '&t=' + Date.now() : null;
      
      const popup = document.createElement('div');
      popup.className = 'image-upload-overlay';
      
      // Skapa preview container separat för att undvika HTML-encoding problem
      const previewContainer = document.createElement('div');
      previewContainer.id = 'imagePreviewContainer';
      
      if (currentImageUrl) {
        const img = document.createElement('img');
        img.src = currentImageUrl;
        img.alt = 'Nuvarande bild';
        img.className = 'image-upload-preview';
        img.id = 'imagePreview';
        img.onerror = function() {
          // Om bilden inte kan laddas, visa placeholder
          const placeholder = document.createElement('div');
          placeholder.className = 'image-upload-preview-placeholder';
          placeholder.id = 'imagePreviewPlaceholder';
          placeholder.innerHTML = '<svg width="64" height="64" viewBox="0 0 24 24" fill="currentColor"><path d="M12 12c2.21 0 4-1.79 4-4s-1.79-4-4-4-4 1.79-4 4 1.79 4 4 4zm0 2c-2.67 0-8 1.34-8 4v2h16v-2c0-2.66-5.33-4-8-4z"/></svg>';
          previewContainer.innerHTML = '';
          previewContainer.appendChild(placeholder);
        };
        previewContainer.appendChild(img);
      } else {
        const placeholder = document.createElement('div');
        placeholder.className = 'image-upload-preview-placeholder';
        placeholder.id = 'imagePreviewPlaceholder';
        placeholder.innerHTML = '<svg width="64" height="64" viewBox="0 0 24 24" fill="currentColor"><path d="M12 12c2.21 0 4-1.79 4-4s-1.79-4-4-4-4 1.79-4 4 1.79 4 4 4zm0 2c-2.67 0-8 1.34-8 4v2h16v-2c0-2.66-5.33-4-8-4z"/></svg>';
        previewContainer.appendChild(placeholder);
      }
      
      popup.innerHTML = `
        <div class="image-upload-box">
          <h3 class="confirm-title">Ladda upp profilbild</h3>
        </div>
      `;
      
      const box = popup.querySelector('.image-upload-box');
      box.appendChild(previewContainer);
      
      const fileInput = document.createElement('input');
      fileInput.type = 'file';
      fileInput.id = 'imageFileInput';
      fileInput.accept = 'image/*';
      fileInput.style.display = 'none';
      fileInput.onchange = function(event) {
        handleImageFileSelect(event, userId);
      };
      
      const buttonContainer = document.createElement('div');
      buttonContainer.style.cssText = 'display: flex; gap: 12px; justify-content: flex-start; align-items: center; margin-top: 20px;';
      
      const selectBtn = document.createElement('button');
      selectBtn.type = 'button';
      selectBtn.className = 'confirm-btn confirm-btn-blue';
      selectBtn.textContent = 'Välj bild';
      selectBtn.onclick = function() {
        fileInput.click();
      };
      
      const cancelBtn = document.createElement('button');
      cancelBtn.type = 'button';
      cancelBtn.className = 'confirm-btn confirm-btn-cancel';
      cancelBtn.textContent = 'Avbryt';
      cancelBtn.onclick = function() {
        popup.remove();
      };
      
      // Lägg till "Välj bild" först (längst till vänster)
      buttonContainer.appendChild(selectBtn);
      
      if (currentImageUrl) {
        const deleteBtn = document.createElement('button');
        deleteBtn.type = 'button';
        deleteBtn.className = 'confirm-btn confirm-btn-cancel';
        deleteBtn.textContent = 'Ta bort bild';
        deleteBtn.onclick = function() {
          deleteUserImage(userId);
        };
        buttonContainer.appendChild(deleteBtn);
      }
      
      buttonContainer.appendChild(cancelBtn);
      box.appendChild(fileInput);
      box.appendChild(buttonContainer);
      
      document.body.appendChild(popup);
      
      // Stäng popup när man klickar utanför
      popup.addEventListener('click', (e) => {
        if (e.target === popup) {
          popup.remove();
        }
      });
    }
    
    // Hantera filval
    function handleImageFileSelect(event, userId) {
      const file = event.target.files[0];
      if (!file) return;
      
      // Visa förhandsvisning
      const reader = new FileReader();
      reader.onload = function(e) {
        const previewContainer = document.getElementById('imagePreviewContainer');
        previewContainer.innerHTML = `<img src="${e.target.result}" alt="Förhandsvisning" class="image-upload-preview" id="imagePreview">`;
        
        // Lägg till knapp för att spara
        const uploadBtn = document.createElement('button');
        uploadBtn.type = 'button';
        uploadBtn.className = 'confirm-btn confirm-btn-primary';
        uploadBtn.textContent = 'Spara';
        uploadBtn.onclick = () => uploadImageFile(userId, file);
        
        const buttonContainer = document.querySelector('.image-upload-box > div:last-child');
        if (!buttonContainer.querySelector('.confirm-btn-primary[onclick*="uploadImageFile"]')) {
          // Lägg till "Spara" efter "Välj bild" (som andra knappen)
          const selectBtn = buttonContainer.querySelector('.confirm-btn-blue');
          if (selectBtn && selectBtn.nextSibling) {
            buttonContainer.insertBefore(uploadBtn, selectBtn.nextSibling);
          } else {
            buttonContainer.insertBefore(uploadBtn, buttonContainer.firstChild.nextSibling);
          }
        }
      };
      reader.readAsDataURL(file);
    }
    
    // Ladda upp bildfil
    async function uploadImageFile(userId, file) {
      const formData = new FormData();
      formData.append('user_id', userId);
      formData.append('image', file);
      
      // Hitta popup och knappar
      const popup = document.querySelector('.image-upload-overlay');
      const buttonContainer = popup ? popup.querySelector('.image-upload-box > div:last-child') : null;
      const allButtons = buttonContainer ? buttonContainer.querySelectorAll('button') : [];
      const saveBtn = Array.from(allButtons).find(btn => btn.textContent === 'Spara');
      
      // Disable alla knappar och visa spinner
      allButtons.forEach(btn => {
        btn.disabled = true;
        btn.style.opacity = '0.6';
        btn.style.cursor = 'not-allowed';
      });
      
      // Lägg till spinner om den inte redan finns
      let spinner = popup ? popup.querySelector('.upload-spinner') : null;
      if (!spinner && popup) {
        spinner = document.createElement('div');
        spinner.className = 'upload-spinner';
        spinner.style.cssText = 'text-align: center; margin-top: 15px; color: #1a73e8;';
        spinner.innerHTML = '<div style="display: inline-block; width: 20px; height: 20px; border: 3px solid #f3f3f3; border-top: 3px solid #1a73e8; border-radius: 50%; animation: spin 1s linear infinite;"></div><span style="margin-left: 10px;">Bearbetar bild...</span>';
        const box = popup.querySelector('.image-upload-box');
        if (box) {
          box.appendChild(spinner);
        }
      }
      
      try {
        const response = await fetch('../api/users.php?action=upload_image', {
          method: 'POST',
          body: formData
        });
        
        const responseText = await response.text();
        
        let data;
        try {
          data = JSON.parse(responseText);
        } catch (e) {
          showStatus('Fel vid uppladdning: ' + responseText.slice(0, 200), 'error');
          if (spinner) spinner.remove();
          allButtons.forEach(btn => {
            btn.disabled = false;
            btn.style.opacity = '1';
            btn.style.cursor = 'pointer';
          });
          return;
        }
        
        if (data.success) {
          showStatus('Bild uppladdad', 'success');
          // Uppdatera användarens profilbild i listan
          const user = userIds.find(u => u.user_id === userId);
          if (user) {
            user.profile_image = data.profile_image;
          }
          // Stäng popup
          if (popup && popup.parentNode) {
            popup.remove();
          }
          // Ladda om användarlistan från servern för att säkerställa att allt är synkroniserat
          // Använd en liten delay för att säkerställa att servern har sparat filen
          setTimeout(() => {
            loadUserIds();
          }, 500);
          
          // Ta bort spinner och återställ knappar (om popupen fortfarande finns)
          if (spinner && popup && popup.parentNode) {
            spinner.remove();
          }
          if (popup && popup.parentNode) {
            allButtons.forEach(btn => {
              btn.disabled = false;
              btn.style.opacity = '1';
              btn.style.cursor = 'pointer';
            });
          }
        } else {
          showStatus('Kunde inte ladda upp bild: ' + data.message, 'error');
          // Ta bort spinner och återställ knappar vid fel
          if (spinner) {
            spinner.remove();
          }
          allButtons.forEach(btn => {
            btn.disabled = false;
            btn.style.opacity = '1';
            btn.style.cursor = 'pointer';
          });
        }
      } catch (error) {
        showStatus('Fel vid uppladdning: ' + error.message, 'error');
        // Ta bort spinner och återställ knappar vid fel
        if (spinner) {
          spinner.remove();
        }
        allButtons.forEach(btn => {
          btn.disabled = false;
          btn.style.opacity = '1';
          btn.style.cursor = 'pointer';
        });
      }
    }
    
    // Ta bort bild
    async function deleteUserImage(userId) {
      // Hitta användaren för att kontrollera om de är inaktiva
      const user = userIds.find(u => u.user_id === userId);
      
      // Kontrollera om användaren är inaktiv
      if (user) {
        const isInactive = user.inactive === true || user.inactive === 1 || user.inactive === '1' || user.inactive === 'true' || 
                           (user.active === false || user.active === 0 || user.active === '0' || user.active === 'false');
        if (isInactive) {
          alert('Du kan inte ta bort profilbild på en inaktiv användare');
          return;
        }
      }
      
      if (!confirm('Är du säker på att du vill ta bort profilbilden?')) {
        return;
      }
      
      const formData = new FormData();
      formData.append('user_id', userId);
      
      try {
        const response = await fetch('../api/users.php?action=delete_image', {
          method: 'POST',
          body: formData
        });
        
        const data = await response.json();
        
        if (data.success) {
          showStatus('Bild borttagen', 'success');
          // Uppdatera användarens profilbild i listan
          const user = userIds.find(u => u.user_id === userId);
          if (user) {
            delete user.profile_image;
          }
          // Ladda om användarlistan från servern för att säkerställa att allt är synkroniserat
          loadUserIds();
          // Stäng popup
          document.querySelector('.image-upload-overlay')?.remove();
        } else {
          showStatus('Kunde inte ta bort bild: ' + data.message, 'error');
        }
      } catch (error) {
        showStatus('Fel vid borttagning: ' + error.message, 'error');
      }
    }
    
    document.addEventListener('DOMContentLoaded', function() {
      // Hantera formulärskickning för popup
      const addUserForm = document.getElementById('addUserForm');
      if (addUserForm) {
        addUserForm.addEventListener('submit', function(e) {
          e.preventDefault();
          addUserId();
        });
      }
      
      // Hantera formulärskickning för redigeringspopup
      const editUserForm = document.getElementById('editUserForm');
      if (editUserForm) {
        editUserForm.addEventListener('submit', function(e) {
          e.preventDefault();
          saveEditedUser();
        });
      }
      
      // Stäng popup när man klickar utanför
      const addPopup = document.getElementById('addUserPopup');
      if (addPopup) {
        addPopup.addEventListener('click', function(e) {
          if (e.target === addPopup) {
            hideAddUserPopup();
          }
        });
      }
      
      // Stäng redigeringspopup när man klickar utanför
      const editPopup = document.getElementById('editUserPopup');
      if (editPopup) {
        editPopup.addEventListener('click', function(e) {
          if (e.target === editPopup) {
            hideEditUserPopup();
          }
        });
      }
      
      loadUserIds();
    });
    
  </script>
  
  <style>
    @media (max-width: 768px) {
      .form-row {
        grid-template-columns: 1fr;
      }
      .form-row:last-child {
        grid-template-columns: 1fr;
      }
      .userid-actions {
        flex-direction: column;
        gap: 4px;
      }
      .edit-link, .app-link {
        width: 100%;
        text-align: center;
      }
    }
  </style>
</body>
</html>


