<?php
require_once __DIR__ . '/../config.php';
require_once __DIR__ . '/../security.php';
require_once __DIR__ . '/../lib/participant_helper.php';
require_once __DIR__ . '/../lib/dal/DataStoreFactory.php';
require_once __DIR__ . '/../lib/dal/repositories/ChatMessageRepository.php';
require_once __DIR__ . '/../lib/dal/repositories/ProjectRepository.php';
require_once __DIR__ . '/../lib/dal/repositories/UserRepository.php';
require_once __DIR__ . '/../lib/dal/repositories/SystemMessageRepository.php';
require_once __DIR__ . '/../lib/dal/repositories/MappingRepository.php';
require_once __DIR__ . '/../lib/data_functions.php';

setSecurityHeaders();

// Svara alltid JSON
header('Content-Type: application/json; charset=utf-8');

// Hämta user_id och kontrollera admin-rättigheter
$userId = sanitizeInput($_GET['user_id'] ?? '', 'user_id');
$isAdmin = false;
if (!empty($userId)) {
    $isAdmin = isParticipantAdmin($userId);
}

if (!$isAdmin) {
    http_response_code(403);
    echo json_encode([
        'success' => false,
        'message' => 'Endast admin kan se aktivitetslogg'
    ]);
    exit;
}

$action = $_GET['action'] ?? 'list';

if ($action !== 'list') {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'message' => 'Ogiltig action'
    ]);
    exit;
}

// Hämta datumintervall
$fromDate = $_GET['from'] ?? date('Y-m-d');
$toDate = $_GET['to'] ?? date('Y-m-d');

// Validera datum
if (!preg_match('/^\d{4}-\d{2}-\d{2}$/', $fromDate) || !preg_match('/^\d{4}-\d{2}-\d{2}$/', $toDate)) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'message' => 'Ogiltigt datumformat. Använd YYYY-MM-DD'
    ]);
    exit;
}

$startTimestamp = strtotime($fromDate . ' 00:00:00');
$endTimestamp = strtotime($toDate . ' 23:59:59');

if ($startTimestamp === false || $endTimestamp === false || $startTimestamp > $endTimestamp) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'message' => 'Ogiltigt datumintervall'
    ]);
    exit;
}

// Max antal events för att skydda prestanda
$maxEvents = 1000;

$rootDir = realpath(__DIR__ . '/../') ?: dirname(__DIR__);
$activities = [];

// Initiera repositories
$store = DataStoreFactory::create();
$projectRepo = new ProjectRepository($store);
$userRepo = new UserRepository($store);
$systemMessageRepo = new SystemMessageRepository($store);
$mappingRepo = new MappingRepository($store);

// 1. Hämta chat-meddelanden
$chatsDir = $rootDir . '/chats';
if (is_dir($chatsDir)) {
    $chatRepo = new ChatMessageRepository($store, $chatsDir);
    
    $projectDirs = glob($chatsDir . '/*', GLOB_ONLYDIR);
    foreach ($projectDirs as $projectDir) {
        $projectId = basename($projectDir);
        $messages = $chatRepo->getByProject($projectId);
        
        foreach ($messages as $message) {
            $createdAt = $message['created_at'] ?? '';
            if (empty($createdAt)) {
                continue;
            }
            
            $messageTimestamp = strtotime($createdAt);
            if ($messageTimestamp === false || $messageTimestamp < $startTimestamp || $messageTimestamp > $endTimestamp) {
                continue;
            }
            
            $messageType = $message['type'] ?? 'text';
            $messageText = $message['message'] ?? '';
            $participantId = $message['participant_id'] ?? $message['user_id'] ?? '';
            
            // Hämta användarnamn
            $userName = null;
            if (!empty($participantId)) {
                $userName = getUserFullName($participantId);
            }
            
            // Hämta projektnamn
            $project = $projectRepo->getById($projectId);
            $projectName = $project['name'] ?? $projectId;
            
            $title = '';
            $description = '';
            $imageUrl = null;
            
            if ($messageType === 'image') {
                $title = 'Bild uppladdad i chatt';
                // Hämta image_url om den finns
                $imageUrl = $message['image_url'] ?? $message['media_file'] ?? null;
                if ($imageUrl && !preg_match('/^https?:\/\//', $imageUrl)) {
                    // Relativ URL, gör den absolut
                    $imageUrl = '../chats/' . $projectId . '/' . basename($imageUrl);
                }
                $description = $messageText ?: ($imageUrl ? basename($imageUrl) : '');
            } elseif ($messageType === 'document') {
                $title = 'Dokument uppladdad i chatt';
                $description = $messageText ?: 'Dokument i projekt: ' . $projectName;
            } elseif ($messageType === 'video') {
                $title = 'Video uppladdad i chatt';
                $description = $messageText ?: 'Video i projekt: ' . $projectName;
            } else {
                $title = 'Chattmeddelande';
                $description = mb_substr($messageText, 0, 100);
                if (mb_strlen($messageText) > 100) {
                    $description .= '...';
                }
            }
            
            $activities[] = [
                'timestamp' => $createdAt,
                'type' => 'chat',
                'source' => 'chats',
                'title' => $title,
                'description' => $description,
                'project_id' => $projectId,
                'project_name' => $projectName,
                'user_id' => $participantId,
                'user_name' => $userName,
                'message_type' => $messageType,
                'image_url' => $imageUrl
            ];
            
            if (count($activities) >= $maxEvents) {
                break 2; // Bryt både inner och outer loop
            }
        }
    }
}

// 2. Hämta uppladdade filer (uploads)
$uploadsDir = $rootDir . '/uploads';
if (is_dir($uploadsDir) && count($activities) < $maxEvents) {
    $imageExtensions = ['jpg', 'jpeg', 'png', 'webp', 'heic', 'heif'];
    $documentExtensions = ['pdf', 'txt', 'csv'];
    $videoExtensions = ['mp4', 'avi', 'mov', 'wmv', 'flv', 'webm', 'mkv', 'm4v'];
    
    $iterator = new RecursiveIteratorIterator(
        new RecursiveDirectoryIterator($uploadsDir, RecursiveDirectoryIterator::SKIP_DOTS)
    );
    
    foreach ($iterator as $file) {
        if (count($activities) >= $maxEvents) {
            break;
        }
        
        if (!$file->isFile()) {
            continue;
        }
        
        $extension = strtolower($file->getExtension());
        $filename = $file->getFilename();
        
        // Hoppa över metadata-filer och thumbnails
        if ($extension === 'json' || $extension === 'finfo' || strpos($filename, '-thumb.') !== false) {
            continue;
        }
        
        // Hitta metadata-fil
        $basePath = preg_replace('/\.[^.]+$/', '', $file->getPathname());
        $metaPath = $basePath . '.json';
        if (!file_exists($metaPath)) {
            $metaPath = $basePath . '.finfo';
        }
        
        $storedAt = null;
        $projectId = null;
        $userId = null;
        $originalName = '';
        $fileType = 'file';
        
        if (file_exists($metaPath)) {
            $metadata = json_decode(file_get_contents($metaPath), true);
            if ($metadata && isset($metadata['stored_at'])) {
                $storedAt = $metadata['stored_at'];
                $projectId = $metadata['project_id'] ?? null;
                $userId = $metadata['user_id'] ?? null;
                $originalName = $metadata['original'] ?? '';
                $fileType = $metadata['file_type'] ?? 'file';
            }
        }
        
        // Om ingen metadata, använd filens mtime
        if ($storedAt === null) {
            $fileMtime = $file->getMTime();
            if ($fileMtime < $startTimestamp || $fileMtime > $endTimestamp) {
                continue;
            }
            $storedAt = date('c', $fileMtime);
        } else {
            $storedTimestamp = strtotime($storedAt);
            if ($storedTimestamp === false || $storedTimestamp < $startTimestamp || $storedTimestamp > $endTimestamp) {
                continue;
            }
        }
        
        // Bestäm typ baserat på extension eller metadata
        if (in_array($extension, $imageExtensions) || $fileType === 'image') {
            $title = 'Bild uppladdad';
            $description = $originalName ?: $filename;
        } elseif (in_array($extension, $documentExtensions) || $fileType === 'document') {
            $title = 'Dokument uppladdad';
            $description = $originalName ?: $filename;
        } elseif (in_array($extension, $videoExtensions) || $fileType === 'video') {
            $title = 'Video uppladdad';
            $description = $originalName ?: $filename;
        } else {
            $title = 'Fil uppladdad';
            $description = $originalName ?: $filename;
        }
        
        // Hämta projektnamn om projekt_id finns
        $projectName = null;
        if ($projectId) {
            $project = $projectRepo->getById($projectId);
            $projectName = $project['name'] ?? $projectId;
        }
        
        // Hämta användarnamn
        $userName = null;
        if (!empty($userId)) {
            $userName = getUserFullName($userId);
        }
        
        $activities[] = [
            'timestamp' => $storedAt,
            'type' => 'upload',
            'source' => 'uploads',
            'title' => $title,
            'description' => $description,
            'project_id' => $projectId,
            'project_name' => $projectName,
            'user_id' => $userId,
            'user_name' => $userName,
            'file_type' => $fileType
        ];
    }
}

// 3. Hämta zip-paketeringar (downloads)
$downloadsDir = $rootDir . '/downloads';
if (is_dir($downloadsDir) && count($activities) < $maxEvents) {
    $zipFiles = glob($downloadsDir . '/*.zip');
    foreach ($zipFiles as $zipFile) {
        if (count($activities) >= $maxEvents) {
            break;
        }
        
        if (!is_file($zipFile)) {
            continue;
        }
        
        $fileMtime = filemtime($zipFile);
        if ($fileMtime < $startTimestamp || $fileMtime > $endTimestamp) {
            continue;
        }
        
        $filename = basename($zipFile);
        // Försök extrahera projekt-ID från filnamn (format: projekt_XXXX_YYYY-MM-DD_HH-MM-SS.zip)
        $projectId = null;
        $projectName = null;
        $userId = null;
        $userName = null;
        
        // Försök läsa metadata för zip-filen
        $metadataPath = preg_replace('/\.zip$/', '.json', $zipFile);
        if (file_exists($metadataPath)) {
            $metadata = json_decode(file_get_contents($metadataPath), true);
            if ($metadata) {
                $projectId = $metadata['project_id'] ?? null;
                $userId = $metadata['user_id'] ?? null;
            }
        }
        
        // Om inte i metadata, försök extrahera från filnamn
        if (!$projectId && preg_match('/projekt_(\d+)_/', $filename, $matches)) {
            $projectId = $matches[1];
        }
        
        if ($projectId) {
            $project = $projectRepo->getById($projectId);
            $projectName = $project['name'] ?? $projectId;
        }
        
        // Hämta användarnamn om user_id finns
        if (!empty($userId)) {
            $userName = getUserFullName($userId);
        }
        
        $activities[] = [
            'timestamp' => date('c', $fileMtime),
            'type' => 'package',
            'source' => 'downloads',
            'title' => 'Zip-paket skapat',
            'description' => $filename,
            'project_id' => $projectId,
            'project_name' => $projectName,
            'user_id' => $userId,
            'user_name' => $userName,
            'file_type' => 'zip'
        ];
    }
}

// 4. Hämta systemmeddelanden (om de har created_at)
if (count($activities) < $maxEvents) {
    $messages = $systemMessageRepo->getAll();
    foreach ($messages as $message) {
        if (count($activities) >= $maxEvents) {
            break;
        }
        
        $createdAt = $message['created_at'] ?? '';
        if (empty($createdAt)) {
            continue;
        }
        
        $messageTimestamp = strtotime($createdAt);
        if ($messageTimestamp === false || $messageTimestamp < $startTimestamp || $messageTimestamp > $endTimestamp) {
            continue;
        }
        
        $messageText = $message['message'] ?? '';
        $createdBy = $message['created_by'] ?? null;
        
        // Hämta användarnamn
        $userName = null;
        if (!empty($createdBy)) {
            $userName = getUserFullName($createdBy);
        }
        
        $activities[] = [
            'timestamp' => $createdAt,
            'type' => 'system_message',
            'source' => 'system',
            'title' => 'Systemmeddelande skapat',
            'description' => mb_substr($messageText, 0, 100) . (mb_strlen($messageText) > 100 ? '...' : ''),
            'project_id' => null,
            'project_name' => null,
            'user_id' => $createdBy,
            'user_name' => $userName,
            'file_type' => null
        ];
    }
}

// 5. Hämta projekt-händelser
if (count($activities) < $maxEvents) {
    $projects = $projectRepo->getAll();
    foreach ($projects as $project) {
        if (count($activities) >= $maxEvents) {
            break;
        }
        
        $projectId = $project['id'] ?? '';
        $projectName = $project['name'] ?? $projectId;
        $status = $project['status'] ?? 'active';
        $createdAt = $project['created_at'] ?? '';
        $updatedAt = $project['updated_at'] ?? null;
        $archivedAt = $project['archived_at'] ?? null;
        
        // Normalisera timestamp-format (hantera både 'Y-m-d H:i:s' och ISO 8601)
        $normalizeTimestamp = function($ts) {
            if (empty($ts)) return null;
            $parsed = strtotime($ts);
            return $parsed !== false ? date('c', $parsed) : $ts;
        };
        
        // Projekt skapat
        if (!empty($createdAt)) {
            $createdTimestamp = strtotime($createdAt);
            if ($createdTimestamp !== false && $createdTimestamp >= $startTimestamp && $createdTimestamp <= $endTimestamp) {
                $activities[] = [
                    'timestamp' => $normalizeTimestamp($createdAt),
                    'type' => 'project',
                    'source' => 'projects',
                    'title' => 'Projekt skapat',
                    'description' => $projectName,
                    'project_id' => $projectId,
                    'project_name' => $projectName,
                    'user_id' => null,
                    'file_type' => null
                ];
            }
        }
        
        // Projekt arkiverat (använd archived_at om det finns)
        if (!empty($archivedAt)) {
            $archivedTimestamp = strtotime($archivedAt);
            if ($archivedTimestamp !== false && $archivedTimestamp >= $startTimestamp && $archivedTimestamp <= $endTimestamp) {
                $activities[] = [
                    'timestamp' => $normalizeTimestamp($archivedAt),
                    'type' => 'project',
                    'source' => 'projects',
                    'title' => 'Projekt arkiverat',
                    'description' => $projectName,
                    'project_id' => $projectId,
                    'project_name' => $projectName,
                    'user_id' => null,
                    'file_type' => null
                ];
            }
        }
        
        // Projekt aktiverat (status är 'active' men archived_at finns och updated_at är nyare)
        if ($status === 'active' && !empty($archivedAt) && !empty($updatedAt)) {
            $archivedTimestamp = strtotime($archivedAt);
            $updatedTimestamp = strtotime($updatedAt);
            // Om updated_at är nyare än archived_at, är det troligen en aktivering
            if ($archivedTimestamp !== false && $updatedTimestamp !== false && $updatedTimestamp > $archivedTimestamp) {
                if ($updatedTimestamp >= $startTimestamp && $updatedTimestamp <= $endTimestamp) {
                    $activities[] = [
                        'timestamp' => $normalizeTimestamp($updatedAt),
                        'type' => 'project',
                        'source' => 'projects',
                        'title' => 'Projekt aktiverat',
                        'description' => $projectName,
                        'project_id' => $projectId,
                        'project_name' => $projectName,
                        'user_id' => null,
                        'file_type' => null
                    ];
                }
            }
        }
        
        // Projekt uppdaterat (exkludera om created_at = updated_at eller om det redan är en aktivering/arkivering)
        if (!empty($updatedAt) && $updatedAt !== $createdAt) {
            $updatedTimestamp = strtotime($updatedAt);
            if ($updatedTimestamp !== false && $updatedTimestamp >= $startTimestamp && $updatedTimestamp <= $endTimestamp) {
                // Hoppa över om det redan är en arkivering eller aktivering
                $isArchiving = !empty($archivedAt) && strtotime($archivedAt) === $updatedTimestamp;
                $isActivating = $status === 'active' && !empty($archivedAt) && strtotime($updatedAt) > strtotime($archivedAt);
                
                if (!$isArchiving && !$isActivating) {
                    $activities[] = [
                        'timestamp' => $normalizeTimestamp($updatedAt),
                        'type' => 'project',
                        'source' => 'projects',
                        'title' => 'Projekt uppdaterat',
                        'description' => $projectName,
                        'project_id' => $projectId,
                        'project_name' => $projectName,
                        'user_id' => null,
                        'file_type' => null
                    ];
                }
            }
        }
    }
}

// 6. Hämta användar-händelser
if (count($activities) < $maxEvents) {
    $users = $userRepo->getAll();
    foreach ($users as $user) {
        if (count($activities) >= $maxEvents) {
            break;
        }
        
        $userId = $user['user_id'] ?? '';
        $userName = $user['full_name'] ?? $userId;
        $userType = $user['user_type'] ?? 'user';
        $isActive = ($user['active'] ?? true) !== false;
        $createdAt = $user['created_at'] ?? '';
        $updatedAt = $user['updated_at'] ?? null;
        
        // Hoppa över externa användare här (de hanteras separat)
        if ($userType === 'extern') {
            continue;
        }
        
        // Normalisera timestamp-format
        $normalizeTimestamp = function($ts) {
            if (empty($ts)) return null;
            $parsed = strtotime($ts);
            return $parsed !== false ? date('c', $parsed) : $ts;
        };
        
        // Användare skapad
        if (!empty($createdAt)) {
            $createdTimestamp = strtotime($createdAt);
            if ($createdTimestamp !== false && $createdTimestamp >= $startTimestamp && $createdTimestamp <= $endTimestamp) {
                $activities[] = [
                    'timestamp' => $normalizeTimestamp($createdAt),
                    'type' => 'user',
                    'source' => 'users',
                    'title' => 'Användare skapad',
                    'description' => $userName,
                    'project_id' => null,
                    'project_name' => null,
                    'user_id' => $userId,
                    'user_name' => $userName,
                    'file_type' => null
                ];
            }
        }
        
        // Användare uppdaterad (exkludera om created_at = updated_at)
        if (!empty($updatedAt) && $updatedAt !== $createdAt) {
            $updatedTimestamp = strtotime($updatedAt);
            if ($updatedTimestamp !== false && $updatedTimestamp >= $startTimestamp && $updatedTimestamp <= $endTimestamp) {
                // Kontrollera om status ändrats (aktiv/inaktiv)
                // Om användaren är inaktiv nu, är det en inaktivering
                if (!$isActive) {
                    $activities[] = [
                        'timestamp' => $normalizeTimestamp($updatedAt),
                        'type' => 'user',
                        'source' => 'users',
                        'title' => 'Användare inaktiverad',
                        'description' => $userName,
                        'project_id' => null,
                        'project_name' => null,
                        'user_id' => $userId,
                        'user_name' => $userName,
                        'file_type' => null
                    ];
                } else {
                    // Om användaren är aktiv nu och updated_at är inom intervallet,
                    // kan det vara en aktivering (om användaren tidigare var inaktiv)
                    // eller en vanlig uppdatering. Eftersom vi inte har historik,
                    // antar vi att det kan vara en aktivering om created_at är mycket äldre än updated_at
                    // (dvs användaren skapades länge sedan men uppdaterades nyligen)
                    $createdTimestamp = !empty($createdAt) ? strtotime($createdAt) : false;
                    $isLikelyActivation = false;
                    if ($createdTimestamp !== false && $updatedTimestamp > $createdTimestamp) {
                        $daysBetween = ($updatedTimestamp - $createdTimestamp) / (24 * 3600);
                        // Om det är mer än 7 dagar mellan skapelse och uppdatering, är det troligen en aktivering
                        if ($daysBetween > 7) {
                            $isLikelyActivation = true;
                        }
                    }
                    
                    if ($isLikelyActivation) {
                        $activities[] = [
                            'timestamp' => $normalizeTimestamp($updatedAt),
                            'type' => 'user',
                            'source' => 'users',
                            'title' => 'Användare aktiverad',
                            'description' => $userName,
                            'project_id' => null,
                            'project_name' => null,
                            'user_id' => $userId,
                            'user_name' => $userName,
                            'file_type' => null
                        ];
                    } else {
                        $activities[] = [
                            'timestamp' => $normalizeTimestamp($updatedAt),
                            'type' => 'user',
                            'source' => 'users',
                            'title' => 'Användare uppdaterad',
                            'description' => $userName,
                            'project_id' => null,
                            'project_name' => null,
                            'user_id' => $userId,
                            'user_name' => $userName,
                            'file_type' => null
                        ];
                    }
                }
            }
        }
    }
}

// 7. Hämta deltagar-händelser (tillagda till projekt) via MappingRepository
if (count($activities) < $maxEvents) {
    // Hämta alla projekt-mappings via repository
    $allProjectMappings = $mappingRepo->getProjectMappings();
    
    // Normalisera timestamp-format
    $normalizeTimestamp = function($ts) {
        if (empty($ts)) return null;
        $parsed = strtotime($ts);
        return $parsed !== false ? date('c', $parsed) : $ts;
    };
    
    foreach ($allProjectMappings as $projectId => $participants) {
        if (count($activities) >= $maxEvents) {
            break;
        }
        
        if (!is_array($participants)) {
            continue;
        }
        
        // Hämta projektnamn
        $project = $projectRepo->getById($projectId);
        $projectName = null;
        if ($project && isset($project['name']) && $project['name'] !== '' && $project['name'] !== null) {
            $projectName = $project['name'];
        }
        // Om projektnamn saknas eller är tomt, sätt till projekt-ID så att frontend kan hämta projektnamnet
        if (empty($projectName)) {
            $projectName = $projectId;
        }
        
        foreach ($participants as $participant) {
            if (count($activities) >= $maxEvents) {
                break 2;
            }
            
            $addedAt = $participant['added_at'] ?? '';
            if (empty($addedAt)) {
                continue;
            }
            
            $addedTimestamp = strtotime($addedAt);
            if ($addedTimestamp === false || $addedTimestamp < $startTimestamp || $addedTimestamp > $endTimestamp) {
                continue;
            }
            
            $participantId = $participant['participant_id'] ?? '';
            $participantType = $participant['type'] ?? 'user';
            $addedBy = $participant['added_by'] ?? '';
            
            // Hämta deltagarnamn
            $participantName = null;
            if ($participantType === 'extern') {
                $extUser = $userRepo->getById($participantId);
                $participantName = $extUser['full_name'] ?? $participantId;
            } else {
                $participantName = getUserFullName($participantId) ?: $participantId;
            }
            
            // Beskrivning ska bara visa deltagarnamnet, projektnamnet visas separat i frontend
            $activities[] = [
                'timestamp' => $normalizeTimestamp($addedAt),
                'type' => 'participant',
                'source' => 'projects',
                'title' => 'Deltagare tillagd',
                'description' => $participantName,
                'project_id' => $projectId,
                'project_name' => $projectName,
                'user_id' => $participantId,
                'user_name' => $participantName,
                'file_type' => null
            ];
        }
    }
}

// 8. Hämta fil-händelser (raderade filer i attic, återställda filer)
$uploadsDir = $rootDir . '/uploads';
$atticDir = $uploadsDir . '/attic';
if (is_dir($atticDir) && count($activities) < $maxEvents) {
    $iterator = new RecursiveIteratorIterator(
        new RecursiveDirectoryIterator($atticDir, RecursiveDirectoryIterator::SKIP_DOTS)
    );
    
    foreach ($iterator as $file) {
        if (count($activities) >= $maxEvents) {
            break;
        }
        
        if (!$file->isFile()) {
            continue;
        }
        
        $extension = strtolower($file->getExtension());
        $filename = $file->getFilename();
        
        // Hoppa över metadata-filer och thumbnails
        if ($extension === 'json' || $extension === 'finfo' || strpos($filename, '-thumb.') !== false) {
            continue;
        }
        
        // Använd filens mtime som indikation på när den raderades
        $fileMtime = $file->getMTime();
        if ($fileMtime < $startTimestamp || $fileMtime > $endTimestamp) {
            continue;
        }
        
        // Hitta metadata-fil
        $basePath = preg_replace('/\.[^.]+$/', '', $file->getPathname());
        $metaPath = $basePath . '.json';
        if (!file_exists($metaPath)) {
            $metaPath = $basePath . '.finfo';
        }
        
        $projectId = null;
        $originalName = '';
        $restoredAt = null;
        
        if (file_exists($metaPath)) {
            $metadata = json_decode(file_get_contents($metaPath), true);
            if ($metadata) {
                $projectId = $metadata['project_id'] ?? null;
                $originalName = $metadata['original'] ?? '';
                $restoredAt = $metadata['restored_at'] ?? null;
            }
        }
        
        // Om restored_at finns, är det en återställd fil
        if (!empty($restoredAt)) {
            $restoredTimestamp = strtotime($restoredAt);
            if ($restoredTimestamp !== false && $restoredTimestamp >= $startTimestamp && $restoredTimestamp <= $endTimestamp) {
                $projectName = null;
                if ($projectId) {
                    $project = $projectRepo->getById($projectId);
                    $projectName = $project['name'] ?? $projectId;
                }
                
                $activities[] = [
                    'timestamp' => is_numeric($restoredTimestamp) ? date('c', $restoredTimestamp) : $restoredAt,
                    'type' => 'file',
                    'source' => 'uploads',
                    'title' => 'Fil återställd',
                    'description' => ($originalName ?: $filename) . ($projectName ? ' → ' . $projectName : ''),
                    'project_id' => $projectId,
                    'project_name' => $projectName,
                    'user_id' => null,
                    'file_type' => null
                ];
            }
        } else {
            // Annars är det en raderad fil
            $projectName = null;
            if ($projectId) {
                $project = $projectRepo->getById($projectId);
                $projectName = $project['name'] ?? $projectId;
            }
            
            $activities[] = [
                'timestamp' => date('c', $fileMtime),
                'type' => 'file',
                'source' => 'uploads',
                'title' => 'Fil raderad',
                'description' => ($originalName ?: $filename) . ($projectName ? ' från ' . $projectName : ''),
                'project_id' => $projectId,
                'project_name' => $projectName,
                'user_id' => null,
                'file_type' => null
            ];
        }
    }
}

// 9. Hämta externa användare-händelser
if (count($activities) < $maxEvents) {
    $externalUsers = $userRepo->getByType('extern');
    foreach ($externalUsers as $user) {
        if (count($activities) >= $maxEvents) {
            break;
        }
        
        $userId = $user['user_id'] ?? '';
        $userName = $user['full_name'] ?? $userId;
        $createdAt = $user['created_at'] ?? '';
        $updatedAt = $user['updated_at'] ?? null;
        
        // Normalisera timestamp-format
        $normalizeTimestamp = function($ts) {
            if (empty($ts)) return null;
            $parsed = strtotime($ts);
            return $parsed !== false ? date('c', $parsed) : $ts;
        };
        
        // Externa användare skapad
        if (!empty($createdAt)) {
            $createdTimestamp = strtotime($createdAt);
            if ($createdTimestamp !== false && $createdTimestamp >= $startTimestamp && $createdTimestamp <= $endTimestamp) {
                $activities[] = [
                    'timestamp' => $normalizeTimestamp($createdAt),
                    'type' => 'external_user',
                    'source' => 'users',
                    'title' => 'Extern användare skapad',
                    'description' => $userName,
                    'project_id' => null,
                    'project_name' => null,
                    'user_id' => $userId,
                    'user_name' => $userName,
                    'file_type' => null
                ];
            }
        }
        
        // Externa användare uppdaterad (exkludera om created_at = updated_at)
        if (!empty($updatedAt) && $updatedAt !== $createdAt) {
            $updatedTimestamp = strtotime($updatedAt);
            if ($updatedTimestamp !== false && $updatedTimestamp >= $startTimestamp && $updatedTimestamp <= $endTimestamp) {
                $activities[] = [
                    'timestamp' => $normalizeTimestamp($updatedAt),
                    'type' => 'external_user',
                    'source' => 'users',
                    'title' => 'Extern användare uppdaterad',
                    'description' => $userName,
                    'project_id' => null,
                    'project_name' => null,
                    'user_id' => $userId,
                    'user_name' => $userName,
                    'file_type' => null
                ];
            }
        }
    }
}

// 8.5. Hämta filflytt-händelser
if (count($activities) < $maxEvents) {
    $fileMoveEventsFile = $rootDir . '/data/file_move_events.json';
    if (file_exists($fileMoveEventsFile)) {
        $fileMoveEvents = json_decode(file_get_contents($fileMoveEventsFile), true);
        if (is_array($fileMoveEvents)) {
            foreach ($fileMoveEvents as $event) {
                if (count($activities) >= $maxEvents) {
                    break;
                }
                
                $eventTimestamp = $event['timestamp'] ?? '';
                if (empty($eventTimestamp)) {
                    continue;
                }
                
                $eventTimestampParsed = strtotime($eventTimestamp);
                if ($eventTimestampParsed === false || $eventTimestampParsed < $startTimestamp || $eventTimestampParsed > $endTimestamp) {
                    continue;
                }
                
                $fromProjectId = $event['from_project_id'] ?? '';
                $toProjectId = $event['to_project_id'] ?? '';
                $filename = $event['filename'] ?? '';
                $fromProjectName = $event['from_project_name'] ?? $fromProjectId;
                $toProjectName = $event['to_project_name'] ?? $toProjectId;
                $userId = $event['user_id'] ?? '';
                
                // Hämta användarnamn
                $userName = null;
                if (!empty($userId)) {
                    $userName = getUserFullName($userId) ?: $userId;
                }
                
                // Hämta projektnamn om de saknas
                if ($fromProjectName === $fromProjectId && !empty($fromProjectId)) {
                    $fromProject = $projectRepo->getById($fromProjectId);
                    $fromProjectName = $fromProject['name'] ?? $fromProjectId;
                }
                if ($toProjectName === $toProjectId && !empty($toProjectId)) {
                    $toProject = $projectRepo->getById($toProjectId);
                    $toProjectName = $toProject['name'] ?? $toProjectId;
                }
                
                $activities[] = [
                    'timestamp' => date('c', $eventTimestampParsed),
                    'type' => 'file_move',
                    'source' => 'uploads',
                    'title' => 'Fil flyttad',
                    'description' => $filename . ' från ' . $fromProjectName . ' till ' . $toProjectName,
                    'project_id' => $toProjectId,
                    'project_name' => $toProjectName,
                    'user_id' => $userId,
                    'user_name' => $userName,
                    'file_type' => null
                ];
            }
        }
    }
}

// 9. Hämta chatt rensning/borttag-händelser
if (count($activities) < $maxEvents) {
    $chatEventsFile = $rootDir . '/data/chat_events.json';
    if (file_exists($chatEventsFile)) {
        $chatEvents = json_decode(file_get_contents($chatEventsFile), true);
        if (is_array($chatEvents)) {
            foreach ($chatEvents as $event) {
                if (count($activities) >= $maxEvents) {
                    break;
                }
                
                $eventTimestamp = $event['timestamp'] ?? '';
                if (empty($eventTimestamp)) {
                    continue;
                }
                
                $eventTimestampParsed = strtotime($eventTimestamp);
                if ($eventTimestampParsed === false || $eventTimestampParsed < $startTimestamp || $eventTimestampParsed > $endTimestamp) {
                    continue;
                }
                
                $projectId = $event['project_id'] ?? '';
                $userId = $event['user_id'] ?? '';
                $action = $event['action'] ?? '';
                $messageId = $event['message_id'] ?? null;
                
                // Hämta projektnamn
                $projectName = $projectId;
                if (!empty($projectId)) {
                    $project = $projectRepo->getById($projectId);
                    $projectName = $project['name'] ?? $projectId;
                }
                
                // Hämta användarnamn
                $userName = null;
                if (!empty($userId)) {
                    $userName = getUserFullName($userId) ?: $userId;
                }
                
                // Bestäm titel baserat på action
                $title = '';
                $description = '';
                if ($action === 'clear_messages') {
                    $title = 'Chatt rensad';
                    $description = $projectName;
                } elseif ($action === 'delete_message') {
                    $title = 'Chattmeddelande raderat';
                    $description = $projectName . ($messageId ? ' (meddelande: ' . substr($messageId, 0, 8) . '...)' : '');
                } else {
                    continue; // Hoppa över okända actions
                }
                
                $activities[] = [
                    'timestamp' => date('c', $eventTimestampParsed),
                    'type' => 'chat_cleanup',
                    'source' => 'chat_events',
                    'title' => $title,
                    'description' => $description,
                    'project_id' => $projectId,
                    'project_name' => $projectName,
                    'user_id' => $userId,
                    'user_name' => $userName,
                    'file_type' => null
                ];
            }
        }
    }
}

// 10. Hämta reparation/städning-händelser
if (count($activities) < $maxEvents) {
    $repairLogFile = $rootDir . '/data/repair_log.json';
    if (file_exists($repairLogFile)) {
        $repairEvents = json_decode(file_get_contents($repairLogFile), true);
        if (is_array($repairEvents)) {
            foreach ($repairEvents as $event) {
                if (count($activities) >= $maxEvents) {
                    break;
                }
                
                $eventTimestamp = $event['timestamp'] ?? '';
                if (empty($eventTimestamp)) {
                    continue;
                }
                
                $eventTimestampParsed = strtotime($eventTimestamp);
                if ($eventTimestampParsed === false || $eventTimestampParsed < $startTimestamp || $eventTimestampParsed > $endTimestamp) {
                    continue;
                }
                
                $userId = $event['user_id'] ?? '';
                $repairType = $event['repair_type'] ?? '';
                $changesCount = $event['changes_count'] ?? 0;
                $description = $event['description'] ?? '';
                
                // Hämta användarnamn
                $userName = null;
                if (!empty($userId)) {
                    $userName = getUserFullName($userId) ?: $userId;
                }
                
                // Bestäm titel baserat på repair_type
                $title = 'Reparation utförd';
                $repairTypeNames = [
                    'assign_image_numbers' => 'Bildnummer tilldelade',
                    'cleanup_backup_files' => 'Backup-filer rensade',
                    'cleanup_thumbnails' => 'Thumbnails rensade',
                    'cleanup_project_images' => 'Projektbilder rensade',
                    'cleanup_chat_images' => 'Chattbilder rensade',
                    'cleanup_metadata_missing_image' => 'Metadata utan bild rensade',
                    'cleanup_metadata_missing_thumb' => 'Metadata utan thumbnail rensade',
                    'cleanup_dirs_missing_project' => 'Mappar utan projekt rensade',
                    'cleanup_files_missing_metadata' => 'Filer utan metadata rensade'
                ];
                
                if (isset($repairTypeNames[$repairType])) {
                    $title = $repairTypeNames[$repairType];
                }
                
                $activities[] = [
                    'timestamp' => date('c', $eventTimestampParsed),
                    'type' => 'repair',
                    'source' => 'repair_log',
                    'title' => $title,
                    'description' => $description,
                    'project_id' => null,
                    'project_name' => null,
                    'user_id' => $userId,
                    'user_name' => $userName,
                    'file_type' => null
                ];
            }
        }
    }
}

// 11. Hämta version/deploy-händelser
if (count($activities) < $maxEvents) {
    $deployHistoryFile = $rootDir . '/adm/deploy_history.json';
    if (file_exists($deployHistoryFile)) {
        $deployHistory = json_decode(file_get_contents($deployHistoryFile), true);
        if (is_array($deployHistory)) {
            // Sortera efter deploy_time för att kunna jämföra med tidigare deployer
            usort($deployHistory, function($a, $b) {
                $timeA = strtotime($a['deploy_time'] ?? '');
                $timeB = strtotime($b['deploy_time'] ?? '');
                return $timeA - $timeB; // Äldsta först
            });
            
            foreach ($deployHistory as $index => $deploy) {
                if (count($activities) >= $maxEvents) {
                    break;
                }
                
                $deployTime = $deploy['deploy_time'] ?? '';
                if (empty($deployTime)) {
                    continue;
                }
                
                $deployTimestamp = strtotime($deployTime);
                if ($deployTimestamp === false || $deployTimestamp < $startTimestamp || $deployTimestamp > $endTimestamp) {
                    continue;
                }
                
                // Normalisera timestamp-format
                $normalizeTimestamp = function($ts) {
                    if (empty($ts)) return null;
                    $parsed = strtotime($ts);
                    return $parsed !== false ? date('c', $parsed) : $ts;
                };
                
                $commit = $deploy['commit'] ?? 'unknown';
                $message = $deploy['message'] ?? '';
                $branch = $deploy['branch'] ?? '';
                
                // Kontrollera om detta är en ny version eller omdeploy
                // Om samma commit redan deployades tidigare (i tidigare index), är det en omdeploy
                $isNewVersion = true;
                for ($i = 0; $i < $index; $i++) {
                    if (($deployHistory[$i]['commit'] ?? '') === $commit) {
                        $isNewVersion = false;
                        break;
                    }
                }
                
                // Skapa kort beskrivning från commit-meddelande (första raden)
                $description = $message;
                if (strpos($message, "\n") !== false) {
                    $description = substr($message, 0, strpos($message, "\n"));
                }
                if (mb_strlen($description) > 100) {
                    $description = mb_substr($description, 0, 100) . '...';
                }
                
                $activities[] = [
                    'timestamp' => $normalizeTimestamp($deployTime),
                    'type' => 'version',
                    'source' => 'deploy',
                    'title' => $isNewVersion ? 'Ny version av programvaran' : 'Uppdaterat programvaran',
                    'description' => $description,
                    // Fullständig committext (kan vara flerradig), används t.ex. i aktivitetsloggens popup
                    'full_description' => $message,
                    'project_id' => null,
                    'project_name' => null,
                    'user_id' => null,
                    'file_type' => null,
                    'commit' => $commit,
                    'branch' => $branch,
                    'is_new_version' => $isNewVersion
                ];
            }
        }
    }
}

// Sortera efter timestamp (nyaste först)
usort($activities, function($a, $b) {
    $timeA = strtotime($a['timestamp']);
    $timeB = strtotime($b['timestamp']);
    return $timeB - $timeA; // Nyaste först
});

// Begränsa till maxEvents
$activities = array_slice($activities, 0, $maxEvents);

echo json_encode([
    'success' => true,
    'activities' => $activities,
    'count' => count($activities),
    'from' => $fromDate,
    'to' => $toDate
], JSON_UNESCAPED_UNICODE);
?>

