<?php
require_once __DIR__ . '/../../security.php';
require_once __DIR__ . '/../../config.php';
require_once __DIR__ . '/../../lib/participant_helper.php';
require_once __DIR__ . '/../../lib/mapping_helper.php';

// Global array för att lagra aktiva användare (i minnet)
if (!isset($GLOBALS['activeUsers'])) {
    $GLOBALS['activeUsers'] = [];
}

/**
 * Uppdatera aktivitetsstatus för en användare i ett projekt
 * @param string $projectId Projekt-ID
 * @param string $userId Användar-ID
 */
function updateUserActivity($projectId, $userId) {
    if (empty($projectId) || empty($userId)) {
        return false;
    }
    
    if (!isset($GLOBALS['activeUsers'][$projectId])) {
        $GLOBALS['activeUsers'][$projectId] = [];
    }
    $GLOBALS['activeUsers'][$projectId][$userId] = time();
    return true;
}

/**
 * Hämta lista över aktiva användare i ett projekt
 * @param string $projectId Projekt-ID
 * @return array Array av user_ids som är aktiva
 */
function getActiveUsersInProject($projectId) {
    if (empty($projectId)) {
        return [];
    }
    
    cleanupInactiveUsers($projectId);
    
    if (!isset($GLOBALS['activeUsers'][$projectId])) {
        return [];
    }
    
    return array_keys($GLOBALS['activeUsers'][$projectId]);
}

/**
 * Rensa inaktiva användare från aktivitetsregistret
 * @param string|null $projectId Om angivet, rensa endast för detta projekt, annars alla projekt
 */
function cleanupInactiveUsers($projectId = null) {
    $now = time();
    $timeout = 60; // 60 sekunder
    
    if ($projectId !== null) {
        // Cleanup för specifikt projekt
        if (isset($GLOBALS['activeUsers'][$projectId])) {
            foreach ($GLOBALS['activeUsers'][$projectId] as $userId => $timestamp) {
                if ($now - $timestamp > $timeout) {
                    unset($GLOBALS['activeUsers'][$projectId][$userId]);
                }
            }
            // Ta bort tomma projekt-arrayer
            if (empty($GLOBALS['activeUsers'][$projectId])) {
                unset($GLOBALS['activeUsers'][$projectId]);
            }
        }
    } else {
        // Cleanup för alla projekt
        foreach ($GLOBALS['activeUsers'] as $projId => $users) {
            foreach ($users as $userId => $timestamp) {
                if ($now - $timestamp > $timeout) {
                    unset($GLOBALS['activeUsers'][$projId][$userId]);
                }
            }
            // Ta bort tomma projekt-arrayer
            if (empty($GLOBALS['activeUsers'][$projId])) {
                unset($GLOBALS['activeUsers'][$projId]);
            }
        }
    }
}

// Kör API-endpoint logik endast om filen anropas direkt (inte inkluderad)
// Kontrollera om detta är en direkt request genom att kolla om filen anropas direkt
// Om filen inkluderas, kommer __FILE__ att vara samma som den inkluderande filen
$scriptName = $_SERVER['SCRIPT_NAME'] ?? '';
$isDirectRequest = (basename($scriptName) === 'activity.php' || strpos($scriptName, 'chat/activity.php') !== false);
if ($isDirectRequest && (isset($_GET['action']) || isset($_POST['action']))) {
    // Starta output buffering tidigt för att förhindra att varningar skrivs ut
    ob_start();
    
    // Förhindra att PHP-varningar skrivs ut (de loggas fortfarande)
    error_reporting(E_ALL);
    ini_set('display_errors', 0);
    ini_set('log_errors', 1);
    
    // Säkerhetsheaders tidigt
    setSecurityHeaders();
    
    // Rensa output buffer (tar bort eventuella varningar)
    ob_clean();
    
    // Svara alltid JSON
    header('Content-Type: application/json; charset=utf-8');
    
    // Hämta action och parametrar
    $action = $_GET['action'] ?? $_POST['action'] ?? '';
    $projectId = sanitizeInput($_GET['project_id'] ?? $_POST['project_id'] ?? '', 'project_id');
    $userId = sanitizeInput($_GET['user_id'] ?? $_POST['user_id'] ?? '', 'string');

    // Sökvägar för projekt (för validering)
    $projectsFile = __DIR__ . '/../../data/projects.json';

    try {
        switch ($action) {
            case 'ping':
                if (empty($projectId)) {
                    throw new Exception('Projekt-ID krävs');
                }
                
                if (empty($userId)) {
                    throw new Exception('Användar-ID krävs');
                }
                
                // Kontrollera att användaren har tillgång till projektet
                // (användaren måste vara participant, admin, eller user-typ som kan läggas till automatiskt)
                $hasAccess = false;
                
                // Kontrollera om användaren är admin
                if (isParticipantAdmin($userId)) {
                    $hasAccess = true;
                } else {
                    // Kontrollera användartyp
                    $userType = getParticipantType($userId);
                    
                    // För user-typ, tillåt ping även om de inte är deltagare ännu (de läggs till automatiskt)
                    if ($userType === 'user') {
                        $hasAccess = true;
                    } else {
                        // För externa användare, kontrollera om de är participant eller chatter
                        $participants = readProjectMapping($projectId);
                        foreach ($participants as $participant) {
                            if (($participant['participant_id'] ?? '') === $userId) {
                                $hasAccess = true;
                                break;
                            }
                        }
                        
                        // Kontrollera även om användaren är chatter
                        if (!$hasAccess) {
                            $chatters = readChatMapping($projectId);
                            foreach ($chatters as $chatter) {
                                if (($chatter['participant_id'] ?? '') === $userId) {
                                    $hasAccess = true;
                                    break;
                                }
                            }
                        }
                    }
                }
                
                if (!$hasAccess) {
                    throw new Exception('Du har inte tillgång till detta projekt');
                }
                
                // Kör cleanup först (för att rensa gamla poster)
                cleanupInactiveUsers($projectId);
                
                // Uppdatera aktivitetsstatus
                if (updateUserActivity($projectId, $userId)) {
                    echo json_encode([
                        'success' => true,
                        'message' => 'Aktivitetsstatus uppdaterad'
                    ]);
                } else {
                    throw new Exception('Kunde inte uppdatera aktivitetsstatus');
                }
                break;
                
            default:
                throw new Exception('Ogiltig action');
        }
    } catch (Exception $e) {
        if (ob_get_level() > 0) {
            ob_clean();
        }
        http_response_code(400);
        echo json_encode([
            'success' => false,
            'message' => $e->getMessage()
        ]);
        exit;
    }
}

