<?php
require_once __DIR__ . '/../../security.php';
require_once __DIR__ . '/../../config.php';
require_once __DIR__ . '/../../logger.php';
require_once __DIR__ . '/../../lib/mapping_helper.php';
require_once __DIR__ . '/../../lib/dal/repositories/ConfigRepository.php';
require_once __DIR__ . '/../../lib/dal/repositories/UserRepository.php';
require_once __DIR__ . '/../../lib/dal/repositories/ProjectRepository.php';

// Säkerhetsheaders tidigt
setSecurityHeaders();

// Svara alltid JSON
header('Content-Type: application/json; charset=utf-8');

// Sökvägar till JSON-filer (använd realpath för att få absolut sökväg)
// API:et ligger i api/chat/, så vi går två nivåer upp för att komma till root
$baseDir = realpath(__DIR__ . '/../..');
if ($baseDir === false) {
    // Fallback om realpath misslyckas
    $baseDir = dirname(dirname(__DIR__));
}
$dataDir = $baseDir . '/data';

// Skapa repository-instanser
$configRepo = new ConfigRepository();
$userRepo = new UserRepository();
$projectRepo = new ProjectRepository();

// Funktion för att läsa externa hantverkare (använder repository)
function readExternalStaff($userRepo) {
    return $userRepo->getByType('extern');
}

// Funktion för att spara externa hantverkare (använder repository)
function saveExternalStaff($userRepo, $staff) {
    $allUsers = $userRepo->getAll();
    
    // Ta bort alla externa användare från listan
    $allUsers = array_filter($allUsers, function($user) {
        return ($user['user_type'] ?? '') !== 'extern';
    });
    
    // Uppdatera eller skapa varje extern användare
    foreach ($staff as $member) {
        $member['user_type'] = 'extern'; // Säkerställ att user_type är satt
        $userId = $member['user_id'] ?? '';
        
        if (empty($userId)) {
            continue; // Hoppa över om user_id saknas
        }
        
        // Kontrollera om användaren redan finns
        $existingUser = $userRepo->getById($userId);
        if ($existingUser) {
            // Uppdatera befintlig användare
            $userRepo->update($userId, $member);
        } else {
            // Skapa ny användare
            $userRepo->create($member);
        }
    }
    
    return true;
}

// Funktion för att läsa projekt (använder repository)
function readProjects($projectRepo) {
    return $projectRepo->getAll();
}

// Funktion för att spara projekt (använder repository)
function saveProjects($projectRepo, $projects) {
    foreach ($projects as $project) {
        $projectId = $project['id'] ?? '';
        if (empty($projectId)) {
            continue;
        }
        
        $existingProject = $projectRepo->getById($projectId);
        if ($existingProject) {
            $projectRepo->update($projectId, $project);
        } else {
            $projectRepo->create($project);
        }
    }
    return true;
}

// Funktion för att ta bort en staff från alla projekt (använder repository)
function removeStaffFromAllProjects($staffId, $projectRepo) {
    $projects = $projectRepo->getAll();
    $removedFromProjects = [];
    
    foreach ($projects as $project) {
        $participants = $project['participants'] ?? [];
        $originalCount = count($participants);
        
        // Filtrera bort deltagare med matchande participant_id
        $participants = array_filter($participants, function($p) use ($staffId) {
            $pid = $p['participant_id'] ?? '';
            return $pid !== $staffId;
        });
        
        // Konvertera tillbaka till indexerad array
        $participants = array_values($participants);
        
        // Om någon deltagare togs bort, uppdatera projektet
        if (count($participants) < $originalCount) {
            $project['participants'] = $participants;
            $projectRepo->update($project['id'], $project);
            $removedFromProjects[] = $project['id'] ?? '';
        }
    }
    
    return $removedFromProjects;
}

// Funktioner för att läsa external_types och external_companies (använder repository)
function readExternalTypes($configRepo) {
    return $configRepo->getExternalTypes();
}

function readExternalCompanies($configRepo) {
    return $configRepo->getExternalCompanies();
}

// Hämta user_id för rättighetskontroll (endast admin kan hantera)
$userId = sanitizeInput($_GET['user_id'] ?? $_POST['user_id'] ?? '', 'user_id');

// Kontrollera admin-rättigheter (använder repository)
$isAdmin = false;
if (!empty($userId)) {
    $user = $userRepo->getById($userId);
    if ($user && ($user['user_type'] ?? 'user') === 'admin') {
        $isAdmin = true;
    }
}

// Funktion för att skala ner bild till max 640x640 pixels med bibehållen aspect ratio
function resizeImageToMax($sourcePath, $destPath, $maxWidth = 640, $maxHeight = 640) {
    if (!function_exists('gd_info')) {
        return false;
    }
    
    $info = @getimagesize($sourcePath);
    if ($info === false) {
        return false;
    }
    
    $mime = $info['mime'] ?? '';
    $width = $info[0];
    $height = $info[1];
    
    // Ladda bild baserat på MIME-typ
    switch ($mime) {
        case 'image/jpeg':
        case 'image/jpg':
        case 'image/pjpeg':
            $src = @imagecreatefromjpeg($sourcePath);
            break;
        case 'image/png':
            $src = @imagecreatefrompng($sourcePath);
            break;
        case 'image/webp':
            if (function_exists('imagecreatefromwebp')) {
                $src = @imagecreatefromwebp($sourcePath);
            } else {
                $src = false;
            }
            break;
        default:
            $src = false;
    }
    
    if (!$src) {
        return false;
    }
    
    // Beräkna ny storlek med bibehållen aspect ratio
    if ($width <= $maxWidth && $height <= $maxHeight) {
        // Bilden är redan mindre än max, kopiera direkt
        $targetWidth = $width;
        $targetHeight = $height;
    } else {
        $ratio = min($maxWidth / $width, $maxHeight / $height);
        $targetWidth = (int)round($width * $ratio);
        $targetHeight = (int)round($height * $ratio);
    }
    
    // Skapa ny bild
    $dst = imagecreatetruecolor($targetWidth, $targetHeight);
    
    // För PNG med alfa - fyll vit bakgrund
    if ($mime === 'image/png') {
        $white = imagecolorallocate($dst, 255, 255, 255);
        imagefill($dst, 0, 0, $white);
    }
    
    // Skala om bilden
    imagecopyresampled($dst, $src, 0, 0, 0, 0, $targetWidth, $targetHeight, $width, $height);
    
    // Spara som JPEG med kvalitet 85
    $ok = imagejpeg($dst, $destPath, 85);
    
    imagedestroy($dst);
    imagedestroy($src);
    
    return $ok;
}

$action = $_GET['action'] ?? $_POST['action'] ?? '';

// För get_image action, returnera bild direkt (inte JSON) - måste vara FÖRE user_id-kontroll
if (isset($_GET['action']) && $_GET['action'] === 'get_image') {
    $staffId = sanitizeInput($_GET['id'] ?? '', 'string');
    if (empty($staffId)) {
        http_response_code(404);
        exit;
    }
    
    // Säkerställ att staff_id inte innehåller farliga tecken
    if (preg_match('/[^a-zA-Z0-9\-_]/', $staffId)) {
        http_response_code(400);
        exit;
    }
    
    $staff = readExternalStaff($userRepo);
    $profileImage = null;
    
    foreach ($staff as $member) {
        // Kontrollera både user_id och id (för bakåtkompatibilitet)
        $memberId = $member['user_id'] ?? $member['id'] ?? '';
        if ($memberId === $staffId && isset($member['profile_image'])) {
            $profileImage = $member['profile_image'];
            break;
        }
    }
    
    if (!$profileImage) {
        error_log("get_image: Extern partner $staffId hittades inte eller saknar profile_image");
        http_response_code(404);
        exit;
    }
    
    // Säkerställ att filnamnet är säkert
    $profileImage = basename($profileImage);
    if (!preg_match('/^user_[a-zA-Z0-9\-_]+\.jpg$/', $profileImage)) {
        error_log("get_image: Ogiltigt filnamn: $profileImage");
        http_response_code(400);
        exit;
    }
    
    // Använd samma sökväg som vid uppladdning
    $imagePath = $baseDir . '/data/' . $profileImage;
    
    if (!file_exists($imagePath) || !is_file($imagePath)) {
        error_log("get_image: Bildfilen finns inte: $imagePath");
        http_response_code(404);
        exit;
    }
    
    // Kontrollera att filen faktiskt är läsbar
    if (!is_readable($imagePath)) {
        error_log("get_image: Bildfilen är inte läsbar: $imagePath");
        http_response_code(403);
        exit;
    }
    
    header('Content-Type: image/jpeg');
    header('Cache-Control: public, max-age=3600');
    header('Content-Length: ' . filesize($imagePath));
    readfile($imagePath);
    exit;
}

try {
    switch ($action) {
        case 'list':
            $staff = readExternalStaff($userRepo);
            
            // Filtrera bort inaktiva staff om inte include_inactive är satt
            $includeInactive = isset($_GET['include_inactive']) && $_GET['include_inactive'] === '1';
            if (!$includeInactive) {
                $staff = array_filter($staff, function($member) {
                    $active = $member['active'] ?? true;
                    return ($active === true || $active === 1 || $active === '1' || $active === 'true');
                });
            }
            
            $types = readExternalTypes($configRepo);
            $companies = readExternalCompanies($configRepo);
            
            // Lägg till företagsnamn och yrkeskategorinamn
            // Konvertera till gammalt format för bakåtkompatibilitet (id och name)
            foreach ($staff as &$member) {
                // Lägg till id och name för bakåtkompatibilitet
                if (!isset($member['id'])) {
                    $member['id'] = $member['user_id'] ?? '';
                }
                if (!isset($member['name'])) {
                    $member['name'] = $member['full_name'] ?? '';
                }
                
                foreach ($companies as $company) {
                    if ($company['id'] === ($member['company_id'] ?? '')) {
                        $member['company_name'] = $company['name'];
                        break;
                    }
                }
                foreach ($types as $type) {
                    if ($type['id'] === ($member['type_id'] ?? '')) {
                        $member['type_name'] = $type['name'];
                        break;
                    }
                }
            }
            
            echo json_encode([
                'success' => true,
                'staff' => array_values($staff)
            ]);
            break;
            
        case 'get':
            $id = sanitizeInput($_GET['id'] ?? '', 'string');
            if (empty($id)) {
                throw new Exception('ID krävs');
            }
            
            $staff = readExternalStaff($userRepo);
            $types = readExternalTypes($configRepo);
            $companies = readExternalCompanies($configRepo);
            
            foreach ($staff as $member) {
                // Kontrollera både user_id och id (för bakåtkompatibilitet)
                $memberId = $member['user_id'] ?? $member['id'] ?? '';
                if ($memberId === $id) {
                    // Lägg till id och name för bakåtkompatibilitet
                    if (!isset($member['id'])) {
                        $member['id'] = $member['user_id'] ?? '';
                    }
                    if (!isset($member['name'])) {
                        $member['name'] = $member['full_name'] ?? '';
                    }
                    
                    // Lägg till företagsnamn och yrkeskategorinamn
                    foreach ($companies as $company) {
                        if ($company['id'] === ($member['company_id'] ?? '')) {
                            $member['company_name'] = $company['name'];
                            break;
                        }
                    }
                    foreach ($types as $type) {
                        if ($type['id'] === ($member['type_id'] ?? '')) {
                            $member['type_name'] = $type['name'];
                            break;
                        }
                    }
                    
                    echo json_encode([
                        'success' => true,
                        'staff' => $member
                    ]);
                    exit;
                }
            }
            
            throw new Exception('Hantverkare hittades inte');
            
        case 'list_companies':
            // Ingen admin-kontroll behövs för att läsa företag
            $companies = readExternalCompanies($configRepo);
            echo json_encode([
                'success' => true,
                'companies' => $companies
            ]);
            break;
            
        case 'get_company_users':
            if (!$isAdmin) {
                throw new Exception('Endast admin kan se användare för företag');
            }
            
            $companyId = sanitizeInput($_GET['company_id'] ?? '', 'string');
            if (empty($companyId)) {
                throw new Exception('Företags-ID krävs');
            }
            
            $staff = readExternalStaff($userRepo);
            $types = readExternalTypes($configRepo);
            
            // Filtrera användare som tillhör detta företag
            $companyUsers = [];
            foreach ($staff as $member) {
                if (($member['company_id'] ?? '') === $companyId) {
                    // Lägg till id och name för bakåtkompatibilitet
                    if (!isset($member['id'])) {
                        $member['id'] = $member['user_id'] ?? '';
                    }
                    if (!isset($member['name'])) {
                        $member['name'] = $member['full_name'] ?? '';
                    }
                    
                    // Lägg till yrkeskategorinamn
                    foreach ($types as $type) {
                        if ($type['id'] === ($member['type_id'] ?? '')) {
                            $member['type_name'] = $type['name'];
                            break;
                        }
                    }
                    
                    $companyUsers[] = $member;
                }
            }
            
            echo json_encode([
                'success' => true,
                'users' => array_values($companyUsers)
            ]);
            break;
            
        case 'list_types':
            // Ingen admin-kontroll behövs för att läsa yrkeskategorier
            $types = readExternalTypes($configRepo);
            
            // Debug: Logga till server och inkludera i response för browser console
            $debugInfo = [];
            if (empty($types)) {
                $debugInfo = [
                    'types_count' => 0,
                    'message' => 'No types found via repository'
                ];
                
                logWarning('external_staff', 'list_types returned empty array', [
                    'action' => 'list_types'
                ]);
            }
            
            $response = [
                'success' => true,
                'types' => $types
            ];
            
            // Lägg till debug-info om det finns problem
            if (!empty($debugInfo)) {
                $response['debug'] = $debugInfo;
            }
            
            echo json_encode($response, JSON_UNESCAPED_UNICODE);
            break;
            
        case 'add':
            if (!$isAdmin) {
                throw new Exception('Endast admin kan lägga till hantverkare');
            }
            
            $name = sanitizeInput($_POST['name'] ?? '', 'string');
            $mobile = sanitizeInput($_POST['mobile'] ?? '', 'string');
            $email = sanitizeInput($_POST['email'] ?? '', 'email');
            $companyId = sanitizeInput($_POST['company_id'] ?? '', 'string');
            $typeId = sanitizeInput($_POST['type_id'] ?? '', 'string');
            $pushoverUserKey = sanitizeInput($_POST['pushover_user_key'] ?? '', 'string');
            $notes = sanitizeInput($_POST['notes'] ?? '', 'string');
            $publicNotes = sanitizeInput($_POST['public_notes'] ?? '', 'string');
            $nickname = sanitizeInput($_POST['nickname'] ?? '', 'string');
            
            if (empty($name) || empty($mobile)) {
                throw new Exception('Namn och mobilnummer krävs');
            }
            
            // Validera nickname unikhet om det är satt
            if (!empty($nickname)) {
                require_once __DIR__ . '/../../lib/participant_helper.php';
                if (!isNicknameUnique($nickname)) {
                    throw new Exception('Detta nickname används redan av en annan användare');
                }
            }
            
            $staff = readExternalStaff($userRepo);
            
            // Kontrollera om mobilnummer redan finns
            foreach ($staff as $member) {
                if (($member['mobile'] ?? '') === $mobile) {
                    throw new Exception('En hantverkare med detta mobilnummer finns redan');
                }
            }
            
            // Hantera aktiv-status (default true om inte satt)
            $active = isset($_POST['active']) && $_POST['active'] === '1';
            // Hantera can_chat och can_upload (default true om inte satt)
            $canChat = isset($_POST['can_chat']) && $_POST['can_chat'] === '1';
            $canUpload = isset($_POST['can_upload']) && $_POST['can_upload'] === '1';
            
            // Skapa ny hantverkare med ny struktur
            // upload_projects hanteras nu via participants i projects.json
            $newStaff = [
                'user_id' => 'staff_' . bin2hex(random_bytes(8)),
                'full_name' => $name,
                'mobile' => $mobile,
                'user_type' => 'extern',
                'email' => $email,
                'company_id' => $companyId,
                'type_id' => $typeId,
                'pushover_user_key' => $pushoverUserKey,
                'notes' => $notes,
                'public_notes' => $publicNotes,
                'active' => $active,
                'can_chat' => $canChat,
                'can_upload' => $canUpload,
                'created_at' => date('Y-m-d H:i:s'),
                'created_by' => $userId
            ];
            
            if (!empty($nickname)) {
                $newStaff['nickname'] = $nickname;
            }
            
            $staff[] = $newStaff;
            
            if (saveExternalStaff($userRepo, $staff)) {
                // Lägg till id och name för bakåtkompatibilitet i response
                $responseStaff = $newStaff;
                $responseStaff['id'] = $newStaff['user_id'];
                $responseStaff['name'] = $newStaff['full_name'];
                
                echo json_encode([
                    'success' => true,
                    'message' => 'Hantverkare tillagd',
                    'staff' => $responseStaff
                ]);
            } else {
                throw new Exception('Kunde inte spara hantverkare');
            }
            break;
            
        case 'update':
            if (!$isAdmin) {
                throw new Exception('Endast admin kan uppdatera hantverkare');
            }
            
            $id = sanitizeInput($_POST['id'] ?? '', 'string');
            $name = sanitizeInput($_POST['name'] ?? '', 'string');
            $mobile = sanitizeInput($_POST['mobile'] ?? '', 'string');
            $email = sanitizeInput($_POST['email'] ?? '', 'email');
            $companyId = sanitizeInput($_POST['company_id'] ?? '', 'string');
            $typeId = sanitizeInput($_POST['type_id'] ?? '', 'string');
            $pushoverUserKey = sanitizeInput($_POST['pushover_user_key'] ?? '', 'string');
            $notes = sanitizeInput($_POST['notes'] ?? '', 'string');
            $publicNotes = sanitizeInput($_POST['public_notes'] ?? '', 'string');
            $nickname = sanitizeInput($_POST['nickname'] ?? '', 'string');
            // Hantera aktiv-status (default true om inte satt)
            $active = isset($_POST['active']) && $_POST['active'] === '1';
            // Hantera can_chat och can_upload (default true om inte satt)
            $canChat = isset($_POST['can_chat']) && $_POST['can_chat'] === '1';
            $canUpload = isset($_POST['can_upload']) && $_POST['can_upload'] === '1';
            
            if (empty($id) || empty($name) || empty($mobile)) {
                throw new Exception('ID, namn och mobilnummer krävs');
            }
            
            // Validera nickname unikhet om det är satt
            if (!empty($nickname)) {
                require_once __DIR__ . '/../../lib/participant_helper.php';
                if (!isNicknameUnique($nickname, $id)) {
                    throw new Exception('Detta nickname används redan av en annan användare');
                }
            }
            
            $staff = readExternalStaff($userRepo);
            $found = false;
            $wasActive = true;
            $isNowInactive = false;
            
            foreach ($staff as &$member) {
                // Kontrollera både user_id och id (för bakåtkompatibilitet)
                $memberId = $member['user_id'] ?? $member['id'] ?? '';
                if ($memberId === $id) {
                    // Kontrollera om personen var aktiv innan
                    $wasActive = ($member['active'] ?? true) !== false;
                    $isNowInactive = !$active;
                    
                    // Uppdatera med ny struktur
                    $member['user_id'] = $id; // Behåll samma ID
                    $member['full_name'] = $name;
                    $member['mobile'] = $mobile;
                    $member['email'] = $email;
                    $member['company_id'] = $companyId;
                    $member['type_id'] = $typeId;
                    $member['pushover_user_key'] = $pushoverUserKey;
                    $member['notes'] = $notes;
                    $member['public_notes'] = $publicNotes;
                    $member['active'] = $active;
                    $member['can_chat'] = $canChat;
                    $member['can_upload'] = $canUpload;
                    
                    // Ta bort upload_projects och chat_projects om de finns (hanteras nu via projects.json)
                    unset($member['upload_projects']);
                    unset($member['chat_projects']);
                    $member['user_type'] = 'extern'; // Säkerställ att user_type är satt
                    
                    // Nickname
                    if (!empty($nickname)) {
                        $member['nickname'] = $nickname;
                    } else {
                        unset($member['nickname']);
                    }
                    
                    $found = true;
                    break;
                }
            }
            
            if (!$found) {
                throw new Exception('Hantverkare hittades inte');
            }
            
            if (saveExternalStaff($userRepo, $staff)) {
                // Om personen inaktiverades (var aktiv men är nu inaktiv), ta bort från alla projekt
                if ($wasActive && $isNowInactive) {
                    $removedFromProjects = removeStaffFromAllProjects($id, $projectRepo);
                    $message = 'Hantverkare uppdaterad';
                    if (!empty($removedFromProjects)) {
                        $message .= ' och borttagen från ' . count($removedFromProjects) . ' projekt';
                    }
                } else {
                    $message = 'Hantverkare uppdaterad';
                }
                
                echo json_encode([
                    'success' => true,
                    'message' => $message
                ]);
            } else {
                throw new Exception('Kunde inte spara ändringar');
            }
            break;
            
        case 'delete':
            if (!$isAdmin) {
                throw new Exception('Endast admin kan ta bort hantverkare');
            }
            
            $id = sanitizeInput($_POST['id'] ?? '', 'string');
            if (empty($id)) {
                throw new Exception('ID krävs');
            }
            
            $staff = readExternalStaff($userRepo);
            $initialCount = count($staff);
            
            $staff = array_filter($staff, function($member) use ($id) {
                // Kontrollera både user_id och id (för bakåtkompatibilitet)
                $memberId = $member['user_id'] ?? $member['id'] ?? '';
                return $memberId !== $id;
            });
            
            if (count($staff) === $initialCount) {
                throw new Exception('Hantverkare hittades inte');
            }
            
            if (saveExternalStaff($userRepo, array_values($staff))) {
                // Ta bort från alla projekt när personen raderas
                $removedFromProjects = removeStaffFromAllProjects($id, $projectRepo);
                $message = 'Hantverkare borttagen';
                if (!empty($removedFromProjects)) {
                    $message .= ' och borttagen från ' . count($removedFromProjects) . ' projekt';
                }
                
                echo json_encode([
                    'success' => true,
                    'message' => $message
                ]);
            } else {
                throw new Exception('Kunde inte ta bort hantverkare');
            }
            break;
            
        case 'upload_image':
            if (!$isAdmin) {
                throw new Exception('Endast admin kan ladda upp bilder');
            }
            
            $targetStaffId = sanitizeInput($_POST['id'] ?? '', 'string');
            
            if (empty($targetStaffId)) {
                throw new Exception('Staff ID krävs');
            }
            
            if (!isset($_FILES['image']) || $_FILES['image']['error'] !== UPLOAD_ERR_OK) {
                throw new Exception('Ingen bildfil uppladdad eller uppladdningsfel');
            }
            
            $file = $_FILES['image'];
            
            // Validera MIME-typ
            global $ALLOWED_IMAGE_MIME;
            // config.php är redan inkluderad i början av filen
            
            $finfo = new finfo(FILEINFO_MIME_TYPE);
            $mime = $finfo->file($file['tmp_name']);
            
            if (!isset($ALLOWED_IMAGE_MIME[$mime])) {
                throw new Exception('Otillåten bildtyp: ' . $mime);
            }
            
            // Validera filstorlek (max 25MB)
            if ($file['size'] > MAX_FILE_BYTES) {
                throw new Exception('Bilden är för stor (max 25MB)');
            }
            
            // Kontrollera att extern partner finns
            $staff = readExternalStaff($userRepo);
            $staffFound = false;
            $staffIndex = -1;
            
            foreach ($staff as $index => $member) {
                // Kontrollera både user_id och id (för bakåtkompatibilitet)
                $memberId = $member['user_id'] ?? $member['id'] ?? '';
                if ($memberId === $targetStaffId) {
                    $staffFound = true;
                    $staffIndex = $index;
                    break;
                }
            }
            
            if (!$staffFound) {
                throw new Exception('Extern partner hittades inte');
            }
            
            // Ta bort gammal bild om den finns
            if (isset($staff[$staffIndex]['profile_image'])) {
                $oldImagePath = $baseDir . '/data/' . $staff[$staffIndex]['profile_image'];
                if (file_exists($oldImagePath)) {
                    @unlink($oldImagePath);
                }
            }
            
            // Skapa filnamn (använd user_id om det finns, annars id)
            $userIdForFile = $staff[$staffIndex]['user_id'] ?? $staff[$staffIndex]['id'] ?? $targetStaffId;
            $imageFilename = 'user_' . $userIdForFile . '.jpg';
            $imagePath = $baseDir . '/data/' . $imageFilename;
            
            // Säkerställ att data-mappen finns
            $dataDir = $baseDir . '/data';
            if (!is_dir($dataDir)) {
                @mkdir($dataDir, 0755, true);
            }
            
            // Skala ner och spara bild
            if (!resizeImageToMax($file['tmp_name'], $imagePath, 640, 640)) {
                error_log("upload_image: Kunde inte skala ner bilden för staff_id=$targetStaffId, imagePath=$imagePath");
                throw new Exception('Kunde inte skala ner bilden');
            }
            
            // Verifiera att bilden faktiskt sparades
            if (!file_exists($imagePath) || !is_file($imagePath)) {
                error_log("upload_image: Bildfilen sparades inte korrekt: $imagePath");
                throw new Exception('Bildfilen kunde inte sparas');
            }
            
            // Uppdatera userids.json
            $staff[$staffIndex]['profile_image'] = $imageFilename;
            
            if (!saveExternalStaff($userRepo, $staff)) {
                // Om sparandet misslyckas, ta bort bilden
                @unlink($imagePath);
                throw new Exception('Kunde inte spara data');
            }
            
            error_log("upload_image: Bild uppladdad för staff_id=$targetStaffId, filnamn=$imageFilename, sökväg=$imagePath, filstorlek=" . filesize($imagePath));
            
            echo json_encode([
                'success' => true,
                'message' => 'Bild uppladdad',
                'profile_image' => $imageFilename
            ]);
            break;
            
        case 'delete_image':
            if (!$isAdmin) {
                throw new Exception('Endast admin kan ta bort bilder');
            }
            
            $targetStaffId = sanitizeInput($_POST['id'] ?? '', 'string');
            
            if (empty($targetStaffId)) {
                throw new Exception('Staff ID krävs');
            }
            
            $staff = readExternalStaff($userRepo);
            $staffFound = false;
            
            foreach ($staff as &$member) {
                // Kontrollera både user_id och id (för bakåtkompatibilitet)
                $memberId = $member['user_id'] ?? $member['id'] ?? '';
                if ($memberId === $targetStaffId) {
                    $staffFound = true;
                    
                    // Ta bort bildfil om den finns
                    if (isset($member['profile_image'])) {
                        $imagePath = $baseDir . '/data/' . $member['profile_image'];
                        if (file_exists($imagePath)) {
                            @unlink($imagePath);
                        }
                        unset($member['profile_image']);
                    }
                    break;
                }
            }
            
            if (!$staffFound) {
                throw new Exception('Extern partner hittades inte');
            }
            
            if (!saveExternalStaff($userRepo, $staff)) {
                throw new Exception('Kunde inte spara ändringar');
            }
            
            echo json_encode([
                'success' => true,
                'message' => 'Bild borttagen'
            ]);
            break;
            
        case 'add_company':
            if (!$isAdmin) {
                throw new Exception('Endast admin kan lägga till företag');
            }
            
            $companyName = sanitizeInput($_POST['name'] ?? '', 'string');
            if (empty($companyName)) {
                throw new Exception('Företagsnamn krävs');
            }
            
            $companies = readExternalCompanies($configRepo);
            
            // Kontrollera om företagsnamn redan finns
            foreach ($companies as $company) {
                if (($company['name'] ?? '') === $companyName) {
                    throw new Exception('Ett företag med detta namn finns redan');
                }
            }
            
            // Generera nytt ID
            $maxId = 0;
            foreach ($companies as $company) {
                $id = $company['id'] ?? '';
                if (preg_match('/^comp_(\d+)$/', $id, $matches)) {
                    $num = (int)$matches[1];
                    if ($num > $maxId) {
                        $maxId = $num;
                    }
                }
            }
            $newId = 'comp_' . str_pad((string)($maxId + 1), 3, '0', STR_PAD_LEFT);
            
            $newCompany = [
                'id' => $newId,
                'name' => $companyName,
                'visit_address' => sanitizeInput($_POST['visit_address'] ?? '', 'string'),
                'postal_address' => sanitizeInput($_POST['postal_address'] ?? '', 'string'),
                'phone' => sanitizeInput($_POST['phone'] ?? '', 'string'),
                'contact_person' => sanitizeInput($_POST['contact_person'] ?? '', 'string')
            ];
            
            $companies[] = $newCompany;
            
            // Spara via repository
            if (!$configRepo->setExternalCompanies($companies)) {
                throw new Exception('Kunde inte spara företag');
            }
            
            echo json_encode([
                'success' => true,
                'company' => $newCompany
            ]);
            break;
            
        case 'update_company':
            if (!$isAdmin) {
                throw new Exception('Endast admin kan uppdatera företag');
            }
            
            $companyId = sanitizeInput($_POST['id'] ?? '', 'string');
            $companyName = sanitizeInput($_POST['name'] ?? '', 'string');
            
            if (empty($companyId) || empty($companyName)) {
                throw new Exception('Företags-ID och namn krävs');
            }
            
            $companies = readExternalCompanies($configRepo);
            $companyFound = false;
            
            foreach ($companies as &$company) {
                if (($company['id'] ?? '') === $companyId) {
                    // Kontrollera om namnet redan används av ett annat företag
                    foreach ($companies as $otherCompany) {
                        if (($otherCompany['id'] ?? '') !== $companyId && ($otherCompany['name'] ?? '') === $companyName) {
                            throw new Exception('Ett företag med detta namn finns redan');
                        }
                    }
                    
                    $company['name'] = $companyName;
                    $company['visit_address'] = sanitizeInput($_POST['visit_address'] ?? '', 'string');
                    $company['postal_address'] = sanitizeInput($_POST['postal_address'] ?? '', 'string');
                    $company['phone'] = sanitizeInput($_POST['phone'] ?? '', 'string');
                    $company['contact_person'] = sanitizeInput($_POST['contact_person'] ?? '', 'string');
                    $companyFound = true;
                    break;
                }
            }
            
            if (!$companyFound) {
                throw new Exception('Företag hittades inte');
            }
            
            // Spara via repository
            if (!$configRepo->setExternalCompanies(array_values($companies))) {
                throw new Exception('Kunde inte spara ändringar');
            }
            
            echo json_encode([
                'success' => true,
                'company' => $company
            ]);
            break;
            
        case 'delete_company':
            if (!$isAdmin) {
                throw new Exception('Endast admin kan ta bort företag');
            }
            
            $companyId = sanitizeInput($_POST['id'] ?? '', 'string');
            if (empty($companyId)) {
                throw new Exception('Företags-ID krävs');
            }
            
            // Kontrollera om företaget används av någon extern användare
            $staff = readExternalStaff($userRepo);
            foreach ($staff as $member) {
                if (($member['company_id'] ?? '') === $companyId) {
                    throw new Exception('Företaget kan inte tas bort eftersom det används av externa användare');
                }
            }
            
            $companies = readExternalCompanies($configRepo);
            $companyFound = false;
            
            foreach ($companies as $key => $company) {
                if (($company['id'] ?? '') === $companyId) {
                    unset($companies[$key]);
                    $companyFound = true;
                    break;
                }
            }
            
            if (!$companyFound) {
                throw new Exception('Företag hittades inte');
            }
            
            // Spara via repository
            if (!$configRepo->setExternalCompanies(array_values($companies))) {
                throw new Exception('Kunde inte spara ändringar');
            }
            
            echo json_encode([
                'success' => true,
                'message' => 'Företag borttaget'
            ]);
            break;
            
        case 'get_user_access':
            // Hämta information om vilka projekt och chattar en användare har access till
            $targetUserId = sanitizeInput($_GET['user_id'] ?? '', 'string');
            if (empty($targetUserId)) {
                throw new Exception('user_id krävs');
            }
            
            logInfo('external_staff', 'get_user_access: Anropad', [
                'target_user_id' => $targetUserId
            ]);
            
            try {
                // Hämta användardata för att få can_upload och can_chat
                $staff = readExternalStaff($userRepo);
                $userData = null;
                foreach ($staff as $member) {
                    $memberId = $member['user_id'] ?? $member['id'] ?? '';
                    if ($memberId === $targetUserId) {
                        $userData = $member;
                        break;
                    }
                }
                
                if (!$userData) {
                    logWarning('external_staff', 'get_user_access: Användare hittades inte', [
                        'target_user_id' => $targetUserId
                    ]);
                    throw new Exception('Användare hittades inte');
                }
                
                // Använd samma logik som participant_helper.php: default true om inte satt
                $canUploadRaw = $userData['can_upload'] ?? true;
                $canUpload = ($canUploadRaw === true || $canUploadRaw === 1 || $canUploadRaw === '1' || $canUploadRaw === 'true');
                
                $canChatRaw = $userData['can_chat'] ?? true;
                $canChat = ($canChatRaw === true || $canChatRaw === 1 || $canChatRaw === '1' || $canChatRaw === 'true');
                
                logDebug('external_staff', 'get_user_access: Användardata hämtad', [
                    'target_user_id' => $targetUserId,
                    'can_upload_raw' => $canUploadRaw,
                    'can_upload' => $canUpload,
                    'can_chat_raw' => $canChatRaw,
                    'can_chat' => $canChat
                ]);
                
                // Hämta alla projekt
                $projects = readProjects($projectRepo);
                if (!is_array($projects)) {
                    logWarning('external_staff', 'get_user_access: Kunde inte läsa projekt', [
                        'target_user_id' => $targetUserId
                    ]);
                    $projects = [];
                }
                
                logDebug('external_staff', 'get_user_access: Projekt lästa', [
                    'target_user_id' => $targetUserId,
                    'project_count' => count($projects)
                ]);
                
                // Hitta projekt där användaren är deltagare
                $userProjects = [];
                $userChatProjects = [];
                
                foreach ($projects as $project) {
                    $projectId = $project['id'] ?? '';
                    if (empty($projectId)) {
                        continue;
                    }
                    
                    // Använd readProjectMapping() istället för $project['participants']
                    // Detta säkerställer att projekt-deltagare hittas oavsett om de lagras
                    // i projects.participants eller project_mapping
                    try {
                        $participants = readProjectMapping($projectId);
                        if (!is_array($participants)) {
                            $participants = [];
                        }
                        
                        $isParticipant = false;
                        foreach ($participants as $participant) {
                            if (($participant['participant_id'] ?? '') === $targetUserId) {
                                $isParticipant = true;
                                break;
                            }
                        }
                        
                        if ($isParticipant) {
                            $userProjects[] = [
                                'id' => $projectId,
                                'name' => $project['name'] ?? ''
                            ];
                            
                            logDebug('external_staff', 'get_user_access: Användare är deltagare i projekt', [
                                'target_user_id' => $targetUserId,
                                'project_id' => $projectId,
                                'project_name' => $project['name'] ?? ''
                            ]);
                            
                            // Kolla om användaren också är chatter i projektet
                            try {
                                $chatters = readChatMapping($projectId);
                                if (!is_array($chatters)) {
                                    $chatters = [];
                                }
                                foreach ($chatters as $chatter) {
                                    if (($chatter['participant_id'] ?? '') === $targetUserId) {
                                        $userChatProjects[] = [
                                            'id' => $projectId,
                                            'name' => $project['name'] ?? ''
                                        ];
                                        logDebug('external_staff', 'get_user_access: Användare är chatter i projekt', [
                                            'target_user_id' => $targetUserId,
                                            'project_id' => $projectId,
                                            'project_name' => $project['name'] ?? ''
                                        ]);
                                        break;
                                    }
                                }
                            } catch (Exception $e) {
                                // Logga felet men fortsätt med nästa projekt
                                logWarning('external_staff', 'get_user_access: Fel vid läsning av chat mapping', [
                                    'target_user_id' => $targetUserId,
                                    'project_id' => $projectId,
                                    'error' => $e->getMessage()
                                ]);
                            }
                        }
                    } catch (Exception $e) {
                        logWarning('external_staff', 'get_user_access: Fel vid läsning av project mapping', [
                            'target_user_id' => $targetUserId,
                            'project_id' => $projectId,
                            'error' => $e->getMessage()
                        ]);
                    }
                }
                
                logInfo('external_staff', 'get_user_access: Projekt-sökning klar', [
                    'target_user_id' => $targetUserId,
                    'user_projects_count' => count($userProjects),
                    'user_chat_projects_count' => count($userChatProjects)
                ]);
                
                // Bestäm vilken typ av länk som ska genereras
                $linkType = null;
                $targetProject = null;
                
                // Landningssida kan skapas om användaren är deltagare i minst ett projekt
                // OCH har antingen can_upload eller can_chat
                if (count($userProjects) > 0 && ($canUpload || $canChat)) {
                    // Generera länk till landningssidan
                    $linkType = 'landing';
                    // target_project behålls null eftersom landningssidan visar alla projekt
                    logInfo('external_staff', 'get_user_access: Landing-länk genererad', [
                        'target_user_id' => $targetUserId,
                        'projects_count' => count($userProjects),
                        'can_upload' => $canUpload,
                        'can_chat' => $canChat
                    ]);
                } else {
                    logInfo('external_staff', 'get_user_access: Ingen länk kunde genereras', [
                        'target_user_id' => $targetUserId,
                        'can_upload' => $canUpload,
                        'can_chat' => $canChat,
                        'user_projects_count' => count($userProjects),
                        'user_chat_projects_count' => count($userChatProjects),
                        'reason' => count($userProjects) === 0 ? 'Användaren är inte deltagare i något projekt' : 'Användaren saknar både can_upload och can_chat'
                    ]);
                }
                
                echo json_encode([
                    'success' => true,
                    'can_upload' => $canUpload,
                    'can_chat' => $canChat,
                    'projects' => $userProjects,
                    'chat_projects' => $userChatProjects,
                    'link_type' => $linkType,
                    'target_project' => $targetProject
                ], JSON_UNESCAPED_UNICODE);
            } catch (Exception $e) {
                logWarning('external_staff', 'get_user_access: Oväntat fel', [
                    'target_user_id' => $targetUserId,
                    'error' => $e->getMessage(),
                    'trace' => $e->getTraceAsString()
                ]);
                throw $e;
            }
            break;
            
        default:
            throw new Exception('Ogiltig action');
    }
} catch (Exception $e) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'message' => $e->getMessage()
    ]);
}
?>

