<?php
// Starta output buffering tidigt för att förhindra att varningar skrivs ut
if (ob_get_level() === 0) {
    ob_start();
}

// Förhindra att PHP-varningar skrivs ut (de loggas fortfarande)
error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);

// Funktion för att säkert skicka JSON-svar
function sendJsonResponse($data, $statusCode = 200) {
    // Rensa output buffer om det finns en aktiv
    if (ob_get_level() > 0) {
        ob_clean();
    }
    
    // Sätt headers (endast om de inte redan har skickats)
    if (!headers_sent()) {
        http_response_code($statusCode);
        header('Content-Type: application/json; charset=utf-8');
    }
    
    // Skicka JSON
    $json = json_encode($data, JSON_UNESCAPED_UNICODE);
    if ($json === false) {
        // Om json_encode misslyckas, använd enkel JSON
        $json = json_encode(['success' => false, 'message' => 'Ett fel uppstod vid bearbetning av begäran'], JSON_UNESCAPED_UNICODE);
    }
    
    echo $json;
    exit;
}

// Funktion för att säkert skicka framgångsrika JSON-svar
function sendJsonSuccess($data) {
    // Säkerställ att headers är satta
    if (!headers_sent()) {
        header('Content-Type: application/json; charset=utf-8');
    }
    echo json_encode($data, JSON_UNESCAPED_UNICODE);
    exit;
}

// Global error handler för att fånga alla fel
set_error_handler(function($errno, $errstr, $errfile, $errline) {
    // Ignorera varningar och notices
    if ($errno === E_WARNING || $errno === E_NOTICE || $errno === E_DEPRECATED) {
        return false; // Låt PHP hantera det normalt
    }
    
    // För andra fel, kasta exception
    throw new ErrorException($errstr, 0, $errno, $errfile, $errline);
});

// Global exception handler för att fånga alla exceptions som inte fångas
set_exception_handler(function($e) {
    sendJsonResponse([
        'success' => false,
        'message' => $e->getMessage() ?: 'Ett oväntat fel uppstod',
        'error_type' => get_class($e),
        'file' => basename($e->getFile()),
        'line' => $e->getLine()
    ], 400);
});

require_once __DIR__ . '/../../security.php';
require_once __DIR__ . '/../../config.php';
require_once __DIR__ . '/../../chat/notifications.php';
require_once __DIR__ . '/../../lib/participant_helper.php';
require_once __DIR__ . '/../../lib/mapping_helper.php';

// Säkerhetsheaders tidigt (endast om headers inte redan skickats)
if (!headers_sent()) {
    setSecurityHeaders();
    header('Content-Type: application/json; charset=utf-8');
}

// Rensa output buffer (tar bort eventuella varningar)
// OBS: Rensa endast om det finns output, annars kan det orsaka problem
if (ob_get_level() > 0) {
    $obContents = ob_get_contents();
    if (!empty($obContents)) {
        ob_clean();
    }
}

// Inkludera activity.php säkert (för aktivitetsfiltrering)
// OBS: Inkludera EFTER headers och output buffering är satt
// OBS: activity.php kontrollerar själv om den körs direkt eller inkluderas
if (file_exists(__DIR__ . '/activity.php')) {
    // Spara nuvarande output buffer level
    $obLevelBefore = ob_get_level();
    require_once __DIR__ . '/activity.php';
    // Rensa output buffer igen efter inkludering om det finns ny output
    if (ob_get_level() > $obLevelBefore) {
        // Något startade en ny output buffer, rensa den
        while (ob_get_level() > $obLevelBefore) {
            ob_end_clean();
        }
    } else if (ob_get_level() > 0) {
        ob_clean();
    }
}

// Sökvägar
$chatsDir = __DIR__ . '/../../chats';
$uploadsDir = __DIR__ . '/../../uploads';
$projectsFile = __DIR__ . '/../../data/projects.json';

// Skapa repositories
require_once __DIR__ . '/../../lib/dal/repositories/ProjectRepository.php';
require_once __DIR__ . '/../../lib/dal/repositories/ChatMessageRepository.php';
require_once __DIR__ . '/../../lib/dal/repositories/UserRepository.php';

// Funktion för att logga chatt-händelser (rensning/borttag)
function logChatEvent($projectId, $userId, $action, $messageId = null) {
    $logFile = __DIR__ . '/../../data/chat_events.json';
    $logDir = dirname($logFile);
    
    // Skapa data-mappen om den inte finns
    if (!is_dir($logDir)) {
        @mkdir($logDir, 0755, true);
    }
    
    // Läs befintliga händelser
    $events = [];
    if (file_exists($logFile)) {
        $content = @file_get_contents($logFile);
        if ($content !== false) {
            $decoded = json_decode($content, true);
            if (is_array($decoded)) {
                $events = $decoded;
            }
        }
    }
    
    // Lägg till ny händelse
    $event = [
        'timestamp' => date('c'),
        'project_id' => $projectId,
        'user_id' => $userId,
        'action' => $action
    ];
    
    if ($messageId !== null) {
        $event['message_id'] = $messageId;
    }
    
    $events[] = $event;
    
    // Begränsa till senaste 1000 händelser
    if (count($events) > 1000) {
        $events = array_slice($events, -1000);
    }
    
    // Spara tillbaka
    @file_put_contents($logFile, json_encode($events, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE), LOCK_EX);
}

$projectRepo = new ProjectRepository();
$chatRepo = new ChatMessageRepository(null, $chatsDir);
$userRepo = new UserRepository();

// Funktion för att läsa projekt (bakåtkompatibilitet)
function readProjects($filePath) {
    global $projectRepo;
    return $projectRepo->getAll();
}

// Funktion för att läsa chatmeddelanden (bakåtkompatibilitet)
function readChatMessages($projectId, $chatsDir) {
    global $chatRepo;
    return $chatRepo->getByProject($projectId);
}

// Funktion för att spara chatmeddelanden (bakåtkompatibilitet)
function saveChatMessages($projectId, $messages, $chatsDir) {
    global $chatRepo;
    // För att spara hela arrayen, behöver vi ta bort alla befintliga och lägga till nya
    // Detta är ineffektivt men behåller bakåtkompatibilitet
    // I praktiken bör man använda $chatRepo->create() direkt för nya meddelanden
    $existing = $chatRepo->getByProject($projectId);
    
    // Ta bort alla befintliga meddelanden
    foreach ($existing as $existingMsg) {
        $chatRepo->delete($projectId, $existingMsg['id'] ?? '');
    }
    
    // Lägg till alla nya meddelanden
    foreach ($messages as $message) {
        $chatRepo->create($projectId, $message);
    }
    
    return true;
}

// Funktion för att kontrollera om användare/staff är deltagare
// Admins tillåts alltid automatiskt
// Använder nu isProjectParticipant() från participant_helper.php som läser från mapping-filer
function isParticipant($projectId, $participantId, $projectsFile) {
    return isProjectParticipant($projectId, $participantId, $projectsFile);
}

// Funktion för att konvertera PHP ini-värden (t.ex. "32M") till bytes
function return_bytes($val) {
    $val = trim($val);
    $last = strtolower($val[strlen($val)-1]);
    $val = (int)$val;
    switch($last) {
        case 'g': $val *= 1024;
        case 'm': $val *= 1024;
        case 'k': $val *= 1024;
    }
    return $val;
}

// Funktion för att formatera bytes till läsbar storlek
function formatBytes($bytes, $precision = 2) {
    $units = array('B', 'KB', 'MB', 'GB', 'TB');
    
    for ($i = 0; $bytes > 1024 && $i < count($units) - 1; $i++) {
        $bytes /= 1024;
    }
    
    return round($bytes, $precision) . ' ' . $units[$i];
}

// Kontrollera om POST-requesten är för stor (innan vi försöker läsa $_POST)
$postMaxSize = ini_get('post_max_size');
$postMaxSizeBytes = return_bytes($postMaxSize);
$contentLength = isset($_SERVER['CONTENT_LENGTH']) ? (int)$_SERVER['CONTENT_LENGTH'] : 0;

// Om content-length överskrider post_max_size, så är $_POST tomt
if ($contentLength > 0 && $postMaxSizeBytes > 0 && $contentLength > $postMaxSizeBytes) {
    $fileSizeMB = round($contentLength / (1024 * 1024), 1);
    $maxSizeMB = round($postMaxSizeBytes / (1024 * 1024), 0);
    sendJsonResponse([
        'success' => false,
        'message' => "Filen är för stor: {$fileSizeMB} MB (max {$maxSizeMB} MB). Kontakta administratören för att ladda upp större filer."
    ], 413);
}

// Debug: Logga vad som kommer in
$rawAction = $_GET['action'] ?? $_POST['action'] ?? null;
$action = $rawAction !== null ? trim($rawAction) : '';

// DEBUG: Logga action direkt här för att se vad som händer
error_log("DEBUG messages.php: rawAction=" . var_export($rawAction, true) . ", action=" . var_export($action, true) . ", GET=" . json_encode($_GET));

// Om action är tom, kasta exception med debug-info INNAN try-catch
if (empty($action)) {
    sendJsonResponse([
        'success' => false,
        'message' => 'Action är tom',
        'debug' => [
            'raw_action' => $rawAction,
            'get' => $_GET,
            'post_keys' => array_keys($_POST),
            'request_method' => $_SERVER['REQUEST_METHOD'] ?? 'unknown'
        ]
    ], 400);
}

// Försök sanitize input - om det misslyckas, använd raw värden
try {
    $projectId = sanitizeInput($_GET['project_id'] ?? $_POST['project_id'] ?? '', 'project_id');
} catch (Exception $e) {
    $projectId = $_GET['project_id'] ?? $_POST['project_id'] ?? '';
} catch (Error $e) {
    $projectId = $_GET['project_id'] ?? $_POST['project_id'] ?? '';
}

try {
    $participantId = sanitizeInput($_GET['user_id'] ?? $_POST['user_id'] ?? '', 'string');
} catch (Exception $e) {
    $participantId = $_GET['user_id'] ?? $_POST['user_id'] ?? '';
} catch (Error $e) {
    $participantId = $_GET['user_id'] ?? $_POST['user_id'] ?? '';
}
$participantType = null;
try {
    $participantType = getParticipantType($participantId);
} catch (Exception $e) {
    // Ignorera om getParticipantType misslyckas
} catch (Error $e) {
    // Ignorera om getParticipantType misslyckas
}

// Kontrollera om projektet är arkiverat
$isArchived = false;
if (!empty($projectId)) {
    try {
        $projects = readProjects($projectsFile);
        foreach ($projects as $project) {
            if (($project['id'] ?? '') === $projectId) {
                $isArchived = ($project['status'] ?? 'active') === 'archived';
                break;
            }
        }
    } catch (Exception $e) {
        // Om projekt inte kan läsas, antag att det är aktivt
        $isArchived = false;
    }
}

try {
    // Debug: Logga action-värdet
    if (empty($action)) {
        throw new Exception('Action är tom. GET: ' . json_encode($_GET) . ', POST: ' . json_encode(array_keys($_POST)));
    }
    
    // Normalisera action (ta bort whitespace och konvertera till lowercase för jämförelse)
    $actionNormalized = strtolower(trim($action));
    
    // DEBUG: Alltid skriv ut debug-info i JSON-svaret för att se vad som händer
    $debugInfo = [
        'action_original' => $action,
        'action_normalized' => $actionNormalized,
        'action_length' => strlen($action),
        'get_action' => $_GET['action'] ?? 'NOT_SET',
        'all_get' => $_GET,
        'switch_will_match' => in_array($actionNormalized, ['list', 'send', 'upload_image', 'upload_document', 'update', 'delete', 'clear'])
    ];
    
    switch ($actionNormalized) {
        case 'list':
            if (empty($projectId)) {
                throw new Exception('Projekt-ID krävs');
            }
            
            // Kontrollera om användaren är aktiv
            // OBS: Även admin-användare måste vara aktiva för att kunna chatta
            if (!empty($participantId)) {
                try {
                    $isActive = isParticipantActive($participantId);
                    if (!$isActive) {
                        // Användaren är inaktiv - ta bort dem från alla projekt
                        try {
                            $projects = readProjects($projectsFile);
                            foreach ($projects as $proj) {
                                $projId = $proj['id'] ?? '';
                                if (!empty($projId)) {
                                    // Ta bort från både participants och chatters
                                    try {
                                        removeParticipant($projId, $participantId);
                                    } catch (Exception $e) {
                                        // Ignorera fel vid borttagning
                                    }
                                    try {
                                        removeChatter($projId, $participantId);
                                    } catch (Exception $e) {
                                        // Ignorera fel vid borttagning
                                    }
                                }
                            }
                        } catch (Exception $e) {
                            // Ignorera fel vid läsning av projekt
                        }
                        
                        sendJsonResponse([
                            'success' => false,
                            'message' => 'Din användare är inaktiv och du har loggats ut från chatten'
                        ], 403);
                    }
                } catch (Exception $e) {
                    // Om isParticipantActive kastar exception, antag att användaren är aktiv
                    // (för att undvika att blockera legitima användare)
                } catch (Error $e) {
                    // Om isParticipantActive kastar Error, antag att användaren är aktiv
                }
            }
            
            // För arkiverade projekt, tillåt alla att läsa meddelanden (read-only)
            // För aktiva projekt, kontrollera chattillgång (admins tillåts automatiskt via canUserChat)
            if (!$isArchived) {
                try {
                    $canChat = canUserChat($projectId, $participantId, $projectsFile);
                    if (!$canChat) {
                        require_once __DIR__ . '/../../security.php';
                        logSecurityEvent('unauthorized_chat_access_attempt', [
                            'action' => 'list_messages',
                            'reason' => 'not_participant'
                        ], 'high', $participantId, $projectId, 'list_messages');
                        throw new Exception('Du är inte deltagare i detta projekt');
                    }
                } catch (Exception $e) {
                    // Om canUserChat kastar exception, behandla som otillåten åtkomst
                    require_once __DIR__ . '/../../security.php';
                    logSecurityEvent('unauthorized_chat_access_attempt', [
                        'action' => 'list_messages',
                        'reason' => 'canUserChat_error: ' . $e->getMessage()
                    ], 'high', $participantId, $projectId, 'list_messages');
                    throw new Exception('Du är inte deltagare i detta projekt');
                } catch (Error $e) {
                    // Om canUserChat kastar Error, behandla som otillåten åtkomst
                    require_once __DIR__ . '/../../security.php';
                    logSecurityEvent('unauthorized_chat_access_attempt', [
                        'action' => 'list_messages',
                        'reason' => 'canUserChat_error: ' . $e->getMessage()
                    ], 'high', $participantId, $projectId, 'list_messages');
                    throw new Exception('Du är inte deltagare i detta projekt');
                }
            }
            
            try {
                $messages = readChatMessages($projectId, $chatsDir);
            } catch (Exception $e) {
                throw new Exception('Kunde inte läsa meddelanden: ' . $e->getMessage());
            }
            
            // Ta bort dubbletter av systemmeddelanden (samma meddelande flera gånger)
            $seenSystemMessages = [];
            $deduplicatedMessages = [];
            $hadDuplicates = false;
            foreach ($messages as $message) {
                if (($message['type'] ?? '') === 'system' && !empty($message['system_message'])) {
                    $msgKey = $message['system_message'];
                    // Om vi redan sett detta systemmeddelande, hoppa över dubbletter
                    if (isset($seenSystemMessages[$msgKey])) {
                        $hadDuplicates = true;
                        continue;
                    }
                    $seenSystemMessages[$msgKey] = true;
                }
                $deduplicatedMessages[] = $message;
            }
            
            // Om det fanns dubbletter, spara den rensade listan
            if ($hadDuplicates) {
                saveChatMessages($projectId, $deduplicatedMessages, $chatsDir);
            }
            
            $messages = $deduplicatedMessages;
            
            // Hämta aktuella deltagare för projektet
            try {
                $participants = readProjectMapping($projectId);
                if (!is_array($participants)) {
                    $participants = [];
                }
            } catch (Exception $e) {
                $participants = [];
            } catch (Error $e) {
                $participants = [];
            }
            
            $currentParticipantIds = [];
            foreach ($participants as $p) {
                $pid = $p['participant_id'] ?? '';
                if (!empty($pid)) {
                    $currentParticipantIds[] = $pid;
                }
            }
            
            // Lägg till alla admins till currentParticipantIds (admins är alltid deltagare)
            try {
                $adminIds = getAllAdminUserIds();
                foreach ($adminIds as $adminId) {
                    if (!in_array($adminId, $currentParticipantIds)) {
                        $currentParticipantIds[] = $adminId;
                    }
                }
            } catch (Exception $e) {
                // Ignorera om getAllAdminUserIds misslyckas
            }
            
            // Filtrera bort systemmeddelanden om användare som inte längre är deltagare
            $filteredMessages = [];
            foreach ($messages as $message) {
                // Om det är ett systemmeddelande om att någon lades till, kontrollera om användaren fortfarande är deltagare
                if (($message['type'] ?? '') === 'system' && !empty($message['system_message'])) {
                    $systemMsg = $message['system_message'];
                    // Kontrollera om meddelandet handlar om att någon lades till
                    if (strpos($systemMsg, 'lades till i projektchatten') !== false) {
                        // Extrahera användarnamnet från meddelandet (före " lades till")
                        $namePart = trim(str_replace(' lades till i projektchatten', '', $systemMsg));
                        
                        // Hitta user_id för denna användare
                        $userIdForMessage = null;
                        try {
                            global $userRepo;
                            if (isset($userRepo) && $userRepo !== null) {
                                $allUsers = $userRepo->getAll();
                                foreach ($allUsers as $user) {
                                    $userName = $user['full_name'] ?? '';
                                    if ($userName === $namePart) {
                                        $userIdForMessage = $user['user_id'] ?? '';
                                        break;
                                    }
                                }
                            }
                        } catch (Exception $e) {
                            // Om användardata inte kan läsas, hoppa över användarsökningen
                            // userIdForMessage förblir null
                        } catch (Error $e) {
                            // Om användardata inte kan läsas, hoppa över användarsökningen
                            // userIdForMessage förblir null
                        }
                        
                        // Om användaren inte längre är deltagare, hoppa över meddelandet
                        if (!empty($userIdForMessage) && !in_array($userIdForMessage, $currentParticipantIds)) {
                            continue; // Hoppa över detta meddelande
                        }
                    }
                }
                
                $filteredMessages[] = $message;
            }
            
            // Lägg till avsändarnamn
            foreach ($filteredMessages as &$message) {
                $pid = $message['participant_id'] ?? '';
                if (!empty($pid)) {
                    try {
                        $message['sender_name'] = getParticipantName($pid);
                    } catch (Exception $e) {
                        $message['sender_name'] = $pid; // Fallback till participant_id
                    }
                }
            }
            
            sendJsonSuccess([
                'success' => true,
                'messages' => array_values($filteredMessages)
            ]);
            break;
            
        case 'send':
            if (empty($projectId)) {
                throw new Exception('Projekt-ID krävs');
            }
            
            // Kontrollera om användaren är aktiv
            // OBS: Även admin-användare måste vara aktiva för att kunna skicka meddelanden
            if (!empty($participantId)) {
                if (!isParticipantActive($participantId)) {
                    // Användaren är inaktiv - ta bort dem från alla projekt
                    $projects = readProjects($projectsFile);
                    foreach ($projects as $proj) {
                        $projId = $proj['id'] ?? '';
                        if (!empty($projId)) {
                            // Ta bort från både participants och chatters
                            removeParticipant($projId, $participantId);
                            removeChatter($projId, $participantId);
                        }
                    }
                    
                    sendJsonResponse([
                        'success' => false,
                        'message' => 'Din användare är inaktiv och du har loggats ut från chatten'
                    ], 403);
                }
            }
            
            // Kontrollera chattillgång (admins tillåts automatiskt via canUserChat)
            if (!canUserChat($projectId, $participantId, $projectsFile)) {
                throw new Exception('Du är inte deltagare i detta projekt');
            }
            
            $messageText = sanitizeInput($_POST['message'] ?? '', 'string');
            $notifiedUsers = $_POST['notified_users'] ?? [];
            
            if (!is_array($notifiedUsers)) {
                $notifiedUsers = [];
            }
            
            // Sanitera notified_users array
            $sanitizedNotifiedUsers = [];
            foreach ($notifiedUsers as $nu) {
                $sanitizedNotifiedUsers[] = sanitizeInput($nu, 'string');
            }
            
            // Hämta reply_to om det finns
            $replyTo = sanitizeInput($_POST['reply_to'] ?? '', 'string');
            // Kontrollera att reply_to är ett giltigt meddelande-ID i chatten
            if (!empty($replyTo)) {
                $messages = readChatMessages($projectId, $chatsDir);
                $replyExists = false;
                foreach ($messages as $msg) {
                    if (($msg['id'] ?? '') === $replyTo) {
                        $replyExists = true;
                        break;
                    }
                }
                if (!$replyExists) {
                    $replyTo = ''; // Ogiltigt reply_to, ignorera det
                }
            }
            
            $messages = readChatMessages($projectId, $chatsDir);
            $messageId = time() . '_' . bin2hex(random_bytes(4));
            
            $senderType = $participantType ?? 'user';
            
            // Hämta priority om det finns
            $priority = isset($_POST['priority']) && $_POST['priority'] === '2' ? 2 : null;
            
            $newMessage = [
                'id' => $messageId,
                'sender_type' => $senderType,
                'participant_id' => $participantId,
                'message' => $messageText,
                'type' => 'text',
                'notified_users' => $sanitizedNotifiedUsers,
                'priority' => $priority,
                'created_at' => date('c')
            ];
            
            // Lägg till reply_to om det finns
            if (!empty($replyTo)) {
                $newMessage['reply_to'] = $replyTo;
            }
            
            $messages[] = $newMessage;
            
            if (saveChatMessages($projectId, $messages, $chatsDir)) {
                // Filtrera bort aktiva användare från notifieringar
                try {
                    $activeUsers = getActiveUsersInProject($projectId);
                    $notifiedUsers = array_diff($sanitizedNotifiedUsers, $activeUsers);
                } catch (Exception $e) {
                    // Om aktivitetsfiltrering misslyckas, använd ursprunglig lista
                    error_log("Failed to filter active users: " . $e->getMessage());
                    $notifiedUsers = $sanitizedNotifiedUsers;
                }
                
                // Skicka notifieringar till filtrerad lista
                sendNotifications($projectId, $messageId, $notifiedUsers, 'text', $priority);
                
                sendJsonSuccess([
                    'success' => true,
                    'message' => 'Meddelande skickat',
                    'message_id' => $messageId
                ]);
            } else {
                throw new Exception('Kunde inte spara meddelande');
            }
            break;
            
        case 'upload_image':
            if (empty($projectId)) {
                throw new Exception('Projekt-ID krävs');
            }
            
            // Kontrollera uppladdningstillgång (admins tillåts automatiskt via canUserUpload)
            if (!canUserUpload($projectId, $participantId, $projectsFile)) {
                throw new Exception('Du är inte deltagare i detta projekt');
            }
            
            if (!isset($_FILES['image'])) {
                throw new Exception('Ingen bild uppladdad');
            }
            
            // Kontrollera upload-fel och ge tydliga felmeddelanden
            $uploadError = $_FILES['image']['error'] ?? UPLOAD_ERR_OK;
            if ($uploadError !== UPLOAD_ERR_OK) {
                $errorMessages = [
                    UPLOAD_ERR_INI_SIZE => 'Filen överskrider upload_max_filesize (max 32 MB)',
                    UPLOAD_ERR_FORM_SIZE => 'Filen överskrider MAX_FILE_SIZE som angivits i formuläret',
                    UPLOAD_ERR_PARTIAL => 'Filen laddades bara delvis upp',
                    UPLOAD_ERR_NO_FILE => 'Ingen fil laddades upp',
                    UPLOAD_ERR_NO_TMP_DIR => 'Saknar temporär mapp',
                    UPLOAD_ERR_CANT_WRITE => 'Kunde inte skriva fil till disk',
                    UPLOAD_ERR_EXTENSION => 'En PHP-tillägg stoppade filuppladdningen'
                ];
                $errorMessage = $errorMessages[$uploadError] ?? 'Okänt uppladdningsfel';
                throw new Exception('Fel vid uppladdning: ' . $errorMessage);
            }
            
            $file = $_FILES['image'];
            $errors = validateUploadedImage($file);
            
            if (!empty($errors)) {
                throw new Exception(implode(', ', $errors));
            }
            
            $notifiedUsers = $_POST['notified_users'] ?? [];
            if (!is_array($notifiedUsers)) {
                $notifiedUsers = [];
            }
            
            // Sanitera notified_users array
            $sanitizedNotifiedUsers = [];
            foreach ($notifiedUsers as $nu) {
                $sanitizedNotifiedUsers[] = sanitizeInput($nu, 'string');
            }
            
            // Generera säkert filnamn
            $originalName = $file['name'];
            $secureName = generateSecureFilename($originalName, $projectId);
            
            // VIKTIGT: Chatbilder sparas i chats-mappen, INTE i uploads
            // chats/<projektid>/ ska innehålla både messages.json och bilderna
            $chatDir = $chatsDir . '/' . $projectId;
            if (!is_dir($chatDir)) {
                mkdir($chatDir, 0755, true);
            }
            
            $targetPath = $chatDir . '/' . $secureName;
            
            if (!move_uploaded_file($file['tmp_name'], $targetPath)) {
                throw new Exception('Kunde inte flytta uppladdad fil');
            }
            
            // Skapa meddelande
            $messages = readChatMessages($projectId, $chatsDir);
            $messageId = time() . '_' . bin2hex(random_bytes(4));
            
            $senderType = $participantType ?? 'user';
            
            // Hämta priority om det finns
            $priority = isset($_POST['priority']) && $_POST['priority'] === '2' ? 2 : null;
            
            // Hämta beskrivningstext om den finns
            $messageText = sanitizeInput($_POST['message'] ?? '', 'string');
            
            // Hämta anpassat filnamn om det finns
            $displayName = sanitizeInput($_POST['display_name'] ?? '', 'string');
            
            // Hämta ursprungligt filnamn
            $originalFilename = sanitizeInput($_POST['original_filename'] ?? '', 'string');
            
            $newMessage = [
                'id' => $messageId,
                'sender_type' => $senderType,
                'participant_id' => $participantId,
                'message' => $messageText,
                'type' => 'image',
                'media_file' => $secureName,
                'display_name' => $displayName, // Spara det anpassade filnamnet
                'original_filename' => $originalFilename, // Spara det ursprungliga filnamnet
                'notified_users' => $sanitizedNotifiedUsers,
                'priority' => $priority,
                'created_at' => date('c')
            ];
            
            if ($chatRepo->create($projectId, $newMessage)) {
                // Filtrera bort aktiva användare från notifieringar
                try {
                    $activeUsers = getActiveUsersInProject($projectId);
                    $notifiedUsers = array_diff($sanitizedNotifiedUsers, $activeUsers);
                } catch (Exception $e) {
                    // Om aktivitetsfiltrering misslyckas, använd ursprunglig lista
                    error_log("Failed to filter active users: " . $e->getMessage());
                    $notifiedUsers = $sanitizedNotifiedUsers;
                }
                
                // Skicka notifieringar till filtrerad lista
                sendNotifications($projectId, $messageId, $notifiedUsers, 'image', $priority);
                
                sendJsonSuccess([
                    'success' => true,
                    'message' => 'Bild uppladdad',
                    'message_id' => $messageId,
                    'file' => $secureName
                ]);
            } else {
                throw new Exception('Kunde inte spara meddelande');
            }
            break;
            
        case 'upload_document':
            if (empty($projectId)) {
                throw new Exception('Projekt-ID krävs');
            }
            
            // Kontrollera uppladdningstillgång (admins tillåts automatiskt via canUserUpload)
            if (!canUserUpload($projectId, $participantId, $projectsFile)) {
                throw new Exception('Du är inte deltagare i detta projekt');
            }
            
            if (!isset($_FILES['document'])) {
                throw new Exception('Inget dokument uppladdat');
            }
            
            // Kontrollera upload-fel och ge tydliga felmeddelanden
            $uploadError = $_FILES['document']['error'] ?? UPLOAD_ERR_OK;
            if ($uploadError !== UPLOAD_ERR_OK) {
                $errorMessages = [
                    UPLOAD_ERR_INI_SIZE => 'Filen överskrider upload_max_filesize (max 32 MB)',
                    UPLOAD_ERR_FORM_SIZE => 'Filen överskrider MAX_FILE_SIZE som angivits i formuläret',
                    UPLOAD_ERR_PARTIAL => 'Filen laddades bara delvis upp',
                    UPLOAD_ERR_NO_FILE => 'Ingen fil laddades upp',
                    UPLOAD_ERR_NO_TMP_DIR => 'Saknar temporär mapp',
                    UPLOAD_ERR_CANT_WRITE => 'Kunde inte skriva fil till disk',
                    UPLOAD_ERR_EXTENSION => 'En PHP-tillägg stoppade filuppladdningen'
                ];
                $errorMessage = $errorMessages[$uploadError] ?? 'Okänt uppladdningsfel';
                throw new Exception('Fel vid uppladdning: ' . $errorMessage);
            }
            
            $file = $_FILES['document'];
            $errors = validateUploadedDocument($file);
            
            if (!empty($errors)) {
                throw new Exception(implode(', ', $errors));
            }
            
            $notifiedUsers = $_POST['notified_users'] ?? [];
            if (!is_array($notifiedUsers)) {
                $notifiedUsers = [];
            }
            
            // Sanitera notified_users array
            $sanitizedNotifiedUsers = [];
            foreach ($notifiedUsers as $nu) {
                $sanitizedNotifiedUsers[] = sanitizeInput($nu, 'string');
            }
            
            // Generera säkert filnamn
            $originalName = $file['name'];
            $secureName = generateSecureFilename($originalName, $projectId);
            
            // VIKTIGT: Chatdokument sparas i chats-mappen, INTE i uploads
            // chats/<projektid>/ ska innehålla både messages.json och dokumenten
            $chatDir = $chatsDir . '/' . $projectId;
            if (!is_dir($chatDir)) {
                mkdir($chatDir, 0755, true);
            }
            
            $targetPath = $chatDir . '/' . $secureName;
            
            if (!move_uploaded_file($file['tmp_name'], $targetPath)) {
                throw new Exception('Kunde inte flytta uppladdad fil');
            }
            
            // Skapa meddelande
            $messages = readChatMessages($projectId, $chatsDir);
            $messageId = time() . '_' . bin2hex(random_bytes(4));
            
            $senderType = $participantType ?? 'user';
            
            // Hämta priority om det finns
            $priority = isset($_POST['priority']) && $_POST['priority'] === '2' ? 2 : null;
            
            // Hämta beskrivningstext om den finns
            $messageText = sanitizeInput($_POST['message'] ?? '', 'string');
            
            // Hämta anpassat filnamn om det finns
            $displayName = sanitizeInput($_POST['display_name'] ?? '', 'string');
            
            // Hämta ursprungligt filnamn
            $originalFilename = sanitizeInput($_POST['original_filename'] ?? '', 'string');
            
            $newMessage = [
                'id' => $messageId,
                'sender_type' => $senderType,
                'participant_id' => $participantId,
                'message' => $messageText,
                'type' => 'document',
                'media_file' => $secureName,
                'display_name' => $displayName, // Spara det anpassade filnamnet
                'original_filename' => $originalFilename, // Spara det ursprungliga filnamnet
                'notified_users' => $sanitizedNotifiedUsers,
                'priority' => $priority,
                'created_at' => date('c')
            ];
            
            if ($chatRepo->create($projectId, $newMessage)) {
                // Filtrera bort aktiva användare från notifieringar
                try {
                    $activeUsers = getActiveUsersInProject($projectId);
                    $notifiedUsers = array_diff($sanitizedNotifiedUsers, $activeUsers);
                } catch (Exception $e) {
                    // Om aktivitetsfiltrering misslyckas, använd ursprunglig lista
                    error_log("Failed to filter active users: " . $e->getMessage());
                    $notifiedUsers = $sanitizedNotifiedUsers;
                }
                
                // Skicka notifieringar till filtrerad lista
                sendNotifications($projectId, $messageId, $notifiedUsers, 'document', $priority);
                
                sendJsonSuccess([
                    'success' => true,
                    'message' => 'Dokument uppladdat',
                    'message_id' => $messageId,
                    'file' => $secureName
                ]);
            } else {
                throw new Exception('Kunde inte spara meddelande');
            }
            break;
            
        case 'toggle_like':
            if (empty($projectId)) {
                throw new Exception('Projekt-ID krävs');
            }
            
            $messageId = sanitizeInput($_POST['message_id'] ?? '', 'string');
            if (empty($messageId)) {
                throw new Exception('Meddelande-ID krävs');
            }
            
            // Kontrollera chattillgång (admins tillåts automatiskt via canUserChat)
            if (!canUserChat($projectId, $participantId, $projectsFile)) {
                throw new Exception('Du är inte deltagare i detta projekt');
            }
            
            // Läs meddelanden
            $messages = readChatMessages($projectId, $chatsDir);
            $messageFound = false;
            
            foreach ($messages as &$message) {
                if ($message['id'] === $messageId) {
                    $messageFound = true;
                    
                    // Initiera likes array om den inte finns
                    if (!isset($message['likes']) || !is_array($message['likes'])) {
                        $message['likes'] = [];
                    }
                    
                    // Kontrollera om användaren redan gillat
                    $userIndex = array_search($participantId, $message['likes']);
                    $userLiked = false;
                    
                    if ($userIndex !== false) {
                        // Ta bort användaren från likes
                        array_splice($message['likes'], $userIndex, 1);
                        $userLiked = false;
                    } else {
                        // Lägg till användaren till likes
                        $message['likes'][] = $participantId;
                        $userLiked = true;
                    }
                    
                    break;
                }
            }
            
            if (!$messageFound) {
                throw new Exception('Meddelandet hittades inte');
            }
            
            // Spara tillbaka meddelandena
            if (saveChatMessages($projectId, $messages, $chatsDir)) {
                // Hitta det uppdaterade meddelandet för att returnera korrekt data
                foreach ($messages as $message) {
                    if ($message['id'] === $messageId) {
                        sendJsonSuccess([
                            'success' => true,
                            'user_liked' => in_array($participantId, $message['likes']),
                            'like_count' => count($message['likes']),
                            'likes' => $message['likes']
                        ]);
                        break;
                    }
                }
            } else {
                throw new Exception('Kunde inte spara likes');
            }
            break;

        case 'list_user_projects':
            if (empty($participantId)) {
                throw new Exception('User ID krävs');
            }
            
            // Kontrollera om användaren är admin
            $isAdminUser = isParticipantAdmin($participantId);
            
            // Kontrollera användartyp
            $userType = getParticipantType($participantId) ?? 'user';
            
            $projects = readProjects($projectsFile);
            $userProjects = [];
            
            if ($isAdminUser) {
                // För admin: Visa alla aktiva projekt
                foreach ($projects as $p) {
                    if (($p['status'] ?? 'active') !== 'archived') {
                        $userProjects[] = [
                            'id' => $p['id'] ?? '',
                            'name' => $p['name'] ?? $p['id'] ?? ''
                        ];
                    }
                }
            } else {
                if ($userType === 'extern') {
                    // För externa användare: Visa bara projekt de är participants i (för projektåtkomst)
                    foreach ($projects as $p) {
                        if (($p['status'] ?? 'active') === 'archived') {
                            continue;
                        }
                        $projId = $p['id'] ?? '';
                        if (!empty($projId)) {
                            $projParticipants = readProjectMapping($projId);
                            foreach ($projParticipants as $part) {
                                if (($part['participant_id'] ?? '') === $participantId) {
                                    $userProjects[] = [
                                        'id' => $p['id'] ?? '',
                                        'name' => $p['name'] ?? $p['id'] ?? ''
                                    ];
                                    break;
                                }
                            }
                        }
                    }
                } else {
                    // För interna användare: Visa alla aktiva projekt
                    foreach ($projects as $p) {
                        if (($p['status'] ?? 'active') !== 'archived') {
                            $userProjects[] = [
                                'id' => $p['id'] ?? '',
                                'name' => $p['name'] ?? $p['id'] ?? ''
                            ];
                        }
                    }
                }
            }
            
            // Sortera alfabetiskt efter namn
            usort($userProjects, function($a, $b) {
                return strcasecmp($a['name'], $b['name']);
            });
            
            sendJsonSuccess([
                'success' => true,
                'projects' => array_values($userProjects)
            ]);
            break;
            
        case 'mark_viewed':
            if (empty($projectId)) {
                throw new Exception('Projekt-ID krävs');
            }
            
            if (empty($participantId)) {
                throw new Exception('User ID krävs');
            }
            
            $messageId = sanitizeInput($_POST['message_id'] ?? '', 'string');
            if (empty($messageId)) {
                throw new Exception('Meddelande-ID krävs');
            }
            
            // Kontrollera chattillgång (admins tillåts automatiskt via canUserChat)
            if (!canUserChat($projectId, $participantId, $projectsFile)) {
                throw new Exception('Du är inte deltagare i detta projekt');
            }
            
            // Läs meddelanden
            $messages = readChatMessages($projectId, $chatsDir);
            
            // Hitta meddelandet
            $messageFound = false;
            foreach ($messages as &$message) {
                if (($message['id'] ?? '') === $messageId) {
                    $messageFound = true;
                    
                    // Hämta avsändarens ID
                    $senderId = $message['participant_id'] ?? '';
                    
                    // Initiera viewed_by array om den inte finns
                    if (!isset($message['viewed_by']) || !is_array($message['viewed_by'])) {
                        $message['viewed_by'] = [];
                    }
                    
                    // Lägg till user_id om den inte redan finns OCH användaren inte är avsändaren
                    if (!empty($senderId) && $participantId === $senderId) {
                        // Avsändaren ska inte läggas till i viewed_by
                        // Räkna antal unika användare som har sett meddelandet (exkludera avsändaren)
                        $viewedByWithoutSender = array_filter($message['viewed_by'], function($userId) use ($senderId) {
                            return $userId !== $senderId;
                        });
                        $viewedCount = count($viewedByWithoutSender);
                    } else {
                        // Lägg till user_id om den inte redan finns
                        if (!in_array($participantId, $message['viewed_by'])) {
                            $message['viewed_by'][] = $participantId;
                        }
                        // Räkna antal unika användare som har sett meddelandet (exkludera avsändaren)
                        $viewedByWithoutSender = array_filter($message['viewed_by'], function($userId) use ($senderId) {
                            return $userId !== $senderId;
                        });
                        $viewedCount = count($viewedByWithoutSender);
                    }
                    
                    // Spara uppdaterade meddelanden
                    if (saveChatMessages($projectId, $messages, $chatsDir)) {
                        sendJsonSuccess([
                            'success' => true,
                            'viewed_count' => $viewedCount,
                            'viewed_by' => $message['viewed_by']
                        ]);
                    } else {
                        throw new Exception('Kunde inte spara uppdatering');
                    }
                    break;
                }
            }
            
            if (!$messageFound) {
                throw new Exception('Meddelande hittades inte');
            }
            break;
            
        case 'toggle_pin':
            if (empty($projectId)) {
                throw new Exception('Projekt-ID krävs');
            }
            
            if (empty($participantId)) {
                throw new Exception('User ID krävs');
            }
            
            // Endast admins kan fästa meddelanden
            if (!isParticipantAdmin($participantId)) {
                throw new Exception('Endast admin kan fästa meddelanden');
            }
            
            $messageId = sanitizeInput($_POST['message_id'] ?? '', 'string');
            if (empty($messageId)) {
                throw new Exception('Meddelande-ID krävs');
            }
            
            // Kontrollera chattillgång (admins tillåts automatiskt via canUserChat)
            if (!canUserChat($projectId, $participantId, $projectsFile)) {
                throw new Exception('Du är inte deltagare i detta projekt');
            }
            
            // Läs meddelanden
            $messages = readChatMessages($projectId, $chatsDir);
            
            // Hitta meddelandet
            $messageFound = false;
            foreach ($messages as &$message) {
                if (($message['id'] ?? '') === $messageId) {
                    $messageFound = true;
                    // Växla pinned-status
                    $currentPinned = $message['pinned'] ?? false;
                    $message['pinned'] = !$currentPinned;
                    break;
                }
            }
            
            if (!$messageFound) {
                throw new Exception('Meddelande hittades inte');
            }
            
            // Spara meddelanden
            if (saveChatMessages($projectId, $messages, $chatsDir)) {
                sendJsonSuccess([
                    'success' => true,
                    'message' => $message['pinned'] ? 'Meddelande fäst' : 'Meddelande avfäst',
                    'pinned' => $message['pinned']
                ]);
            } else {
                throw new Exception('Kunde inte spara meddelande');
            }
            break;
            
        case 'clear_messages':
            if (empty($projectId)) {
                throw new Exception('Projekt-ID krävs');
            }
            
            if (empty($participantId)) {
                throw new Exception('User ID krävs');
            }
            
            // Kontrollera att användaren är admin
            if (!isParticipantAdmin($participantId)) {
                throw new Exception('Endast administratörer kan rensa meddelanden');
            }
            
            // Ta bort alla bilder i chats-mappen först
            $chatDir = $chatsDir . '/' . $projectId;
            $imagesDeleted = 0;
            if (is_dir($chatDir)) {
                $files = scandir($chatDir);
                foreach ($files as $file) {
                    if ($file === '.' || $file === '..' || $file === 'messages.json') {
                        continue;
                    }
                    $filePath = $chatDir . '/' . $file;
                    if (is_file($filePath)) {
                        if (unlink($filePath)) {
                            $imagesDeleted++;
                        }
                    }
                }
            }
            
            // Rensa alla meddelanden (ersätt med tom array)
            $emptyMessages = [];
            
            // Lägg till systemmeddelande "Chatten har rensats"
            $systemMessage = [
                'id' => time() . '_' . bin2hex(random_bytes(4)),
                'type' => 'system',
                'system_message' => 'Chatten har rensats',
                'created_at' => date('c')
            ];
            $emptyMessages[] = $systemMessage;
            
            // Spara den tomma listan med systemmeddelandet
            if (saveChatMessages($projectId, $emptyMessages, $chatsDir)) {
                // Logga händelsen
                logChatEvent($projectId, $participantId, 'clear_messages');
                
                sendJsonSuccess([
                    'success' => true,
                    'message' => 'Meddelanden rensade',
                    'images_deleted' => $imagesDeleted
                ]);
            } else {
                throw new Exception('Kunde inte rensa meddelanden');
            }
            break;
            
        default:
            // Debug: Visa exakt vad action är - skriv direkt ut INNAN exception
            $actionDebug = array_merge($debugInfo ?? [], [
                'action_hex' => bin2hex($action),
                'post_action' => $_POST['action'] ?? 'NOT_SET',
                'get_keys' => array_keys($_GET),
                'post_keys' => array_keys($_POST),
                'all_post_keys' => array_keys($_POST)
            ]);
            
            // Skriv ut direkt INNAN exception för att säkerställa att vi ser debug-info
            sendJsonResponse([
                'success' => false,
                'message' => 'Ogiltig ACTION - DEFAULT CASE',
                'debug' => $actionDebug
            ], 400);
    }
    
} catch (Exception $e) {
    $errorData = [
        'success' => false,
        'message' => $e->getMessage() ?: 'Ett oväntat fel uppstod',
        'error_type' => get_class($e),
        'file' => basename($e->getFile()),
        'line' => $e->getLine()
    ];
    
    // Lägg till trace endast om det inte är för långt
    $trace = $e->getTraceAsString();
    if (strlen($trace) < 5000) {
        $errorData['trace'] = $trace;
    }
    
    sendJsonResponse($errorData, 400);
} catch (Error $e) {
    // Fånga även PHP Errors (inte bara Exceptions)
    $errorData = [
        'success' => false,
        'message' => $e->getMessage() ?: 'Ett oväntat fel uppstod',
        'error_type' => get_class($e),
        'file' => basename($e->getFile()),
        'line' => $e->getLine()
    ];
    
    // Lägg till trace endast om det inte är för långt
    $trace = $e->getTraceAsString();
    if (strlen($trace) < 5000) {
        $errorData['trace'] = $trace;
    }
    
    sendJsonResponse($errorData, 400);
}

