<?php
require_once __DIR__ . '/../../security.php';
require_once __DIR__ . '/../../config.php';
require_once __DIR__ . '/../../lib/participant_helper.php';
require_once __DIR__ . '/../../lib/mapping_helper.php';

// Säkerhetsheaders tidigt
setSecurityHeaders();

// Svara alltid JSON
header('Content-Type: application/json; charset=utf-8');

// Sökvägar till JSON-filer
$projectsFile = __DIR__ . '/../../data/projects.json';
$userIdsFile = __DIR__ . '/../../data/userids.json';
$chatsDir = __DIR__ . '/../../chats';

// Funktion för att läsa projekt
function readProjects($filePath) {
    if (!file_exists($filePath)) {
        return [];
    }
    $content = file_get_contents($filePath);
    if ($content === false) {
        return [];
    }
    $projects = json_decode($content, true);
    return is_array($projects) ? $projects : [];
}

// Funktion för att spara projekt
function saveProjects($filePath, $projects) {
    $json = json_encode($projects, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
    return file_put_contents($filePath, $json) !== false;
}

// Funktion för att läsa chatmeddelanden
function readChatMessages($projectId) {
    global $chatsDir;
    $chatFile = $chatsDir . '/' . $projectId . '/messages.json';
    if (!file_exists($chatFile)) {
        return [];
    }
    $content = file_get_contents($chatFile);
    if ($content === false) {
        return [];
    }
    $messages = json_decode($content, true);
    return is_array($messages) ? $messages : [];
}

// Funktion för att spara chatmeddelanden
function saveChatMessages($projectId, $messages) {
    global $chatsDir;
    $chatDir = $chatsDir . '/' . $projectId;
    if (!is_dir($chatDir)) {
        mkdir($chatDir, 0755, true);
    }
    $chatFile = $chatDir . '/messages.json';
    $json = json_encode($messages, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
    return file_put_contents($chatFile, $json) !== false;
}

// Funktion för att skapa systemmeddelande
function createSystemMessage($projectId, $message) {
    $messages = readChatMessages($projectId);
    
    // Ta bort alla befintliga dubbletter av samma systemmeddelande först
    $filteredMessages = [];
    $seenMessages = [];
    foreach ($messages as $existingMessage) {
        if (($existingMessage['type'] ?? '') === 'system' && 
            ($existingMessage['system_message'] ?? '') === $message) {
            // Hoppa över dubbletter av samma systemmeddelande
            continue;
        }
        $filteredMessages[] = $existingMessage;
    }
    
    // Kontrollera om samma systemmeddelande redan finns (förhindra dubbletter)
    // Kontrollera de senaste 30 sekunderna för att undvika duplicering vid snabba anrop
    $currentTime = time();
    $recentThreshold = 30; // sekunder
    
    foreach ($filteredMessages as $existingMessage) {
        if (($existingMessage['type'] ?? '') === 'system' && 
            ($existingMessage['system_message'] ?? '') === $message) {
            // Kontrollera om meddelandet är nyligen skapat
            $createdAt = $existingMessage['created_at'] ?? '';
            if (!empty($createdAt)) {
                try {
                    $createdTimestamp = strtotime($createdAt);
                    if ($createdTimestamp !== false && ($currentTime - $createdTimestamp) < $recentThreshold) {
                        // Samma meddelande skapades nyligen, spara den rensade listan och hoppa över att skapa ett nytt
                        saveChatMessages($projectId, $filteredMessages);
                        return;
                    }
                } catch (Exception $e) {
                    // Om det inte går att parsa datumet, fortsätt och skapa meddelandet
                }
            }
        }
    }
    
    $messageId = time() . '_' . bin2hex(random_bytes(4));
    
    $systemMessage = [
        'id' => $messageId,
        'sender_type' => 'system',
        'type' => 'system',
        'system_message' => $message,
        'created_at' => date('c'),
        'notified_users' => []
    ];
    
    $filteredMessages[] = $systemMessage;
    saveChatMessages($projectId, $filteredMessages);
}

// Funktion för att kontrollera om användare/staff är deltagare
// Admins tillåts alltid automatiskt
function isParticipant($projectId, $participantId, $projectsFile) {
    // Använd den nya helper-funktionen för bakåtkompatibilitet
    return isProjectParticipant($projectId, $participantId, $projectsFile);
}

// Hämta user_id för rättighetskontroll
$userId = sanitizeInput($_GET['user_id'] ?? $_POST['user_id'] ?? '', 'user_id');

// Kontrollera admin-rättigheter
$isAdmin = false;
$userIsAdmin = false; // Initiera tidigt för att undvika undefined variable-warningar

// Om is_admin-flagga skickas med, acceptera den direkt (prioriteras över user_id-kontroll)
// Detta säkerställer att admin-rättigheter fungerar även om user_id inte matchar korrekt
$isAdminFlag = isset($_GET['is_admin']) && ($_GET['is_admin'] === '1' || $_GET['is_admin'] === 1);
if ($isAdminFlag) {
    $isAdmin = true;
    // Logga misstänkt användning av is_admin parameter
    require_once __DIR__ . '/../../security.php';
    logSecurityEvent('is_admin_parameter_used', [
        'user_id' => $userId,
        'is_admin_value' => $_GET['is_admin'] ?? null
    ], 'high', $userId, null, 'admin_check');
} elseif (!empty($userId)) {
    // Om ingen is_admin-flagga finns, kontrollera user_id
    $userIsAdmin = isParticipantAdmin($userId);
    if (file_exists($userIdsFile)) {
        $users = json_decode(file_get_contents($userIdsFile), true) ?: [];
        foreach ($users as $u) {
            if (($u['user_id'] ?? '') === $userId) {
                $userType = $u['user_type'] ?? 'user';
                if ($userType === 'admin') {
                    $isAdmin = true;
                    error_log("DEBUG participants_api.php: isAdmin satt till true för user_id=$userId (user_type=$userType)");
                } else {
                    error_log("DEBUG participants_api.php: user_id=$userId hittades men är inte admin (user_type=$userType)");
                }
                break;
            }
        }
        if (!$isAdmin) {
            error_log("DEBUG participants_api.php: user_id=$userId hittades inte i userids.json eller är inte admin");
        }
    } else {
        error_log("DEBUG participants_api.php: userids.json finns inte");
    }
} else {
    error_log("DEBUG participants_api.php: Ingen user_id angiven");
}

$action = $_GET['action'] ?? $_POST['action'] ?? '';

if (empty($action)) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'message' => 'Action parameter krävs'
    ]);
    exit;
}

try {
    switch ($action) {
        case 'list':
            $projectId = sanitizeInput($_GET['project_id'] ?? '', 'project_id');
            
            if (empty($projectId)) {
                throw new Exception('Projekt-ID krävs');
            }
            
            // Kontrollera att projektet finns
            $projects = readProjects($projectsFile);
            $projectFound = false;
            foreach ($projects as $project) {
                if (($project['id'] ?? '') === $projectId) {
                    $projectFound = true;
                    break;
                }
            }
            
            if (!$projectFound) {
                throw new Exception('Projekt hittades inte');
            }
            
            // Läs participants och chatters från mapping-filer
            $participants = readProjectMapping($projectId);
            $chatters = readChatMapping($projectId);
            
            // Om den aktuella användaren är admin, se till att den finns som deltagare
            // Kontrollera faktiskt i mapping-filen, inte bara om de är admin
            if (!empty($userId) && ($isAdmin || $userIsAdmin)) {
                $isInParticipants = false;
                foreach ($participants as $p) {
                    $pid = $p['participant_id'] ?? '';
                    if ($pid === $userId) {
                        $isInParticipants = true;
                        break;
                    }
                }
                
                $isInChatters = false;
                foreach ($chatters as $c) {
                    $cid = $c['participant_id'] ?? '';
                    if ($cid === $userId) {
                        $isInChatters = true;
                        break;
                    }
                }
                
                if (!$isInParticipants || !$isInChatters) {
                    $detectedType = getParticipantType($userId);
                    // Admin är en roll – använd 'user' som deltagartyp i mappingen
                    $participantType = ($detectedType === 'admin') ? 'user' : $detectedType;
                    
                    // Lägg till i participants om saknas
                    if (!$isInParticipants) {
                        addParticipant($projectId, $userId, $participantType, $userId);
                    }
                    
                    // Lägg till i chatters om saknas
                    if (!$isInChatters) {
                        addChatter($projectId, $userId, $participantType, $userId);
                    }
                    
                    // Läs om mappingen efter tillägg
                    $participants = readProjectMapping($projectId);
                    $chatters = readChatMapping($projectId);
                }
            }
            
            // Kontrollera behörighet: endast admin eller deltagare kan se deltagarlistan
            // Admins kan alltid se deltagarlistan (kontrollera admin-status först)
            $hasPermission = false;
            
            // Först kontrollera om användaren är admin (via $isAdmin eller isParticipantAdmin)
            // Använd isParticipantAdmin() som primär kontroll eftersom den är mer pålitlig
            // $userIsAdmin är redan initierad tidigare i filen, men uppdatera den om den inte är satt
            if (!empty($userId) && !$userIsAdmin) {
                $userIsAdmin = isParticipantAdmin($userId);
            }
            
            // Debug-information
            $debugInfo = [
                'user_id' => $userId,
                'project_id' => $projectId,
                'isAdmin_flag' => $isAdmin,
                'isAdminFlag_param' => $isAdminFlag,
                'userIsAdmin_function' => $userIsAdmin,
                'hasPermission_before_check' => $hasPermission
            ];
            
            if ($isAdmin || $userIsAdmin) {
                // Admin har alltid tillgång
                $hasPermission = true;
                $debugInfo['hasPermission_after_admin_check'] = true;
                $debugInfo['reason'] = 'admin_access';
            } elseif (!empty($userId)) {
                // Om inte admin, kontrollera om användaren är deltagare
                $isParticipantResult = isParticipant($projectId, $userId, $projectsFile);
                
                // Om användaren inte är deltagare, kontrollera om de är user-typ (kan läggas till automatiskt)
                if (!$isParticipantResult) {
                    $userType = getParticipantType($userId);
                    // För user-typ, tillåt åtkomst eftersom de kan läggas till automatiskt
                    // (de kommer att läggas till när de försöker chatta)
                    if ($userType === 'user') {
                        $hasPermission = true;
                        $debugInfo['reason'] = 'user_type_auto_add';
                        $debugInfo['user_type'] = $userType;
                    } else {
                        $hasPermission = false;
                        $debugInfo['reason'] = 'not_participant';
                    }
                } else {
                    $hasPermission = true;
                    $debugInfo['reason'] = 'participant_access';
                }
                
                $debugInfo['isParticipant_result'] = $isParticipantResult;
                $debugInfo['hasPermission_after_participant_check'] = $hasPermission;
            } else {
                $debugInfo['reason'] = 'no_user_id';
            }
            
            if (!$hasPermission) {
                require_once __DIR__ . '/../../security.php';
                logSecurityEvent('unauthorized_data_access_attempt', [
                    'action' => 'list_participants',
                    'reason' => $debugInfo['reason'] ?? 'not_admin_or_participant',
                    'debug' => $debugInfo
                ], 'high', $userId, $projectId, 'list_participants');
                
                // Inkludera debug-info i felmeddelandet
                http_response_code(403);
                echo json_encode([
                    'success' => false,
                    'message' => 'Du saknar behörighet att se deltagare för detta projekt',
                    'action' => $action,
                    'debug' => $debugInfo
                ], JSON_UNESCAPED_UNICODE);
                exit;
            }
            
            // Lägg till användarnamn och staff-information för participants
            $users = [];
            if (file_exists($userIdsFile)) {
                $users = json_decode(file_get_contents($userIdsFile), true) ?: [];
            }
            
            // Externa användare (staff) finns nu i samma fil som interna användare
            // Filtrera ut dem baserat på user_type === 'extern'
            $externalStaff = [];
            foreach ($users as $user) {
                $userType = $user['user_type'] ?? 'user';
                if ($userType === 'extern') {
                    $externalStaff[] = $user;
                }
            }
            
            // Filtrera bort inaktiva participants
            $activeParticipants = [];
            $participantIds = []; // Håll koll på vilka deltagare-ID:n som redan är med
            
            foreach ($participants as $participant) {
                $pid = $participant['participant_id'] ?? '';
                
                // Om participant saknar participant_id, hoppa över
                if (empty($pid)) {
                    error_log("VARNING: Participant saknar participant_id i projekt $projectId: " . json_encode($participant));
                    continue;
                }
                
                // Kontrollera om participant är aktiv
                if (!isParticipantActive($pid)) {
                    continue; // Hoppa över inaktiva participants
                }
                
                // Hämta namn och type - om participant inte hittas i systemet, hoppa över
                $name = getParticipantName($pid);
                $type = getParticipantType($pid);
                
                // Om participant inte hittades i systemet (name är samma som id och type är null), hoppa över
                if ($name === $pid && $type === null) {
                    error_log("VARNING: Participant hittades inte i systemet, hoppar över: $pid");
                    continue;
                }
                
                // Säkerställ att participant-objektet har alla nödvändiga fält
                $participantData = [
                    'participant_id' => $pid,
                    'type' => $type,
                    'name' => $name,
                    'added_at' => $participant['added_at'] ?? null,
                    'added_by' => $participant['added_by'] ?? null
                ];
                
                // Hämta mobilnummer, pushover_user_key och nickname
                if ($type === 'user') {
                    foreach ($users as $user) {
                        if (($user['user_id'] ?? '') === $pid) {
                            $participantData['mobile'] = $user['mobile'] ?? '';
                            $participantData['pushover_user_key'] = $user['pushover_user_key'] ?? '';
                            $participantData['nickname'] = $user['nickname'] ?? '';
                            break;
                        }
                    }
                } elseif ($type === 'extern') {
                    // Externa användare använder nu user_id istället för id
                    foreach ($externalStaff as $staff) {
                        $staffId = $staff['user_id'] ?? $staff['id'] ?? '';
                        if ($staffId === $pid) {
                            $participantData['mobile'] = $staff['mobile'] ?? '';
                            $participantData['pushover_user_key'] = $staff['pushover_user_key'] ?? '';
                            $participantData['nickname'] = $staff['nickname'] ?? '';
                            break;
                        }
                    }
                }
                
                $activeParticipants[] = $participantData;
                $participantIds[] = $pid;
            }
            
            // NOTERA: Admins läggs INTE längre till automatiskt som deltagare.
            // De måste läggas till explicit via "Hantera deltagare"-popupen.
            // Admins kan fortfarande se och hantera chatten via sin admin-behörighet,
            // men visas bara i deltagarlistan om de är explicit tillagda.
            
            // Processera chatters
            $activeChatters = [];
            $chatterIds = [];
            
            foreach ($chatters as $chatter) {
                $cid = $chatter['participant_id'] ?? '';
                
                if (empty($cid)) {
                    continue;
                }
                
                if (!isParticipantActive($cid)) {
                    continue;
                }
                
                $name = getParticipantName($cid);
                $type = getParticipantType($cid);
                
                if ($name === $cid && $type === null) {
                    continue;
                }
                
                $chatterData = [
                    'participant_id' => $cid,
                    'type' => $type,
                    'name' => $name,
                    'added_at' => $chatter['added_at'] ?? null,
                    'added_by' => $chatter['added_by'] ?? null
                ];
                
                if ($type === 'user') {
                    foreach ($users as $user) {
                        if (($user['user_id'] ?? '') === $cid) {
                            $chatterData['mobile'] = $user['mobile'] ?? '';
                            $chatterData['pushover_user_key'] = $user['pushover_user_key'] ?? '';
                            $chatterData['nickname'] = $user['nickname'] ?? '';
                            break;
                        }
                    }
                } elseif ($type === 'extern') {
                    foreach ($externalStaff as $staff) {
                        $staffId = $staff['user_id'] ?? $staff['id'] ?? '';
                        if ($staffId === $cid) {
                            $chatterData['mobile'] = $staff['mobile'] ?? '';
                            $chatterData['pushover_user_key'] = $staff['pushover_user_key'] ?? '';
                            $chatterData['nickname'] = $staff['nickname'] ?? '';
                            break;
                        }
                    }
                }
                
                $activeChatters[] = $chatterData;
                $chatterIds[] = $cid;
            }
            
            echo json_encode([
                'success' => true,
                'participants' => array_values($activeParticipants),
                'chatters' => array_values($activeChatters)
            ]);
            break;
            
        case 'add':
            $projectId = sanitizeInput($_POST['project_id'] ?? '', 'project_id');
            $participantType = sanitizeInput($_POST['type'] ?? '', 'string');
            $participantId = sanitizeInput($_POST['participant_id'] ?? '', 'string');
            $listType = sanitizeInput($_POST['list_type'] ?? 'participants', 'string'); // 'participants' eller 'chatters'
            
            if (empty($projectId) || empty($participantType) || empty($participantId)) {
                throw new Exception('Projekt-ID, typ och deltagar-ID krävs');
            }
            
            // Kontrollera om användaren försöker lägga till sig själv (tillåt för user-typ)
            $isSelfAdd = ($participantId === $userId);
            $userType = getParticipantType($userId);
            
            // Tillåt self-add för user-typ, annars kräv admin
            if (!$isSelfAdd || $userType !== 'user') {
                if (!$isAdmin) {
                    require_once __DIR__ . '/../../security.php';
                    logSecurityEvent('unauthorized_admin_action_attempt', [
                        'action' => 'add_participant',
                        'reason' => 'not_admin'
                    ], 'high', $userId, $projectId, 'add_participant');
                    throw new Exception('Endast admin kan lägga till andra deltagare');
                }
            }
            
            if (!in_array($participantType, ['user', 'extern'])) {
                throw new Exception('Ogiltig deltagartyp');
            }
            
            if (!in_array($listType, ['participants', 'chatters'])) {
                throw new Exception('Ogiltig listtyp');
            }
            
            // Ytterligare kontroll: Verifiera att användaren inte är extern om de försöker läggas till automatiskt
            // Externa användare ska bara kunna läggas till manuellt av admin via kryssrutor i deras profil
            $detectedType = getParticipantType($participantId);
            if ($detectedType === 'extern') {
                // Kontrollera även direkt från userids.json för säkerhet
                $isExternUserDirect = false;
                if (file_exists($userIdsFile)) {
                    $allUsers = json_decode(file_get_contents($userIdsFile), true) ?: [];
                    foreach ($allUsers as $user) {
                        if (($user['user_id'] ?? '') === $participantId) {
                            $userType = $user['user_type'] ?? 'user';
                            if ($userType === 'extern') {
                                $isExternUserDirect = true;
                            }
                            break;
                        }
                    }
                }
                
                // Om användaren är extern, kontrollera att de redan är deltagare i projektet
                // Om de inte är det, tillåt endast om admin lägger till dem manuellt (via 'add' action)
                // Detta förhindrar automatisk tilläggning av externa användare
                if ($isExternUserDirect) {
                    $projects = readProjects($projectsFile);
                    $isAlreadyParticipant = false;
                    foreach ($projects as $project) {
                        if (($project['id'] ?? '') === $projectId) {
                            $participants = $project['participants'] ?? [];
                            foreach ($participants as $p) {
                                $pid = $p['participant_id'] ?? '';
                                if ($pid === $participantId) {
                                    $isAlreadyParticipant = true;
                                    break 2;
                                }
                            }
                        }
                    }
                }
            }
            
            // Kontrollera om deltagaren är aktiv
            if (!isParticipantActive($participantId)) {
                throw new Exception('Inaktiva användare kan inte läggas till i chatten');
            }
            
            // Kontrollera att projektet finns
            $projects = readProjects($projectsFile);
            $projectFound = false;
            foreach ($projects as $project) {
                if (($project['id'] ?? '') === $projectId) {
                    $projectFound = true;
                    break;
                }
            }
            
            if (!$projectFound) {
                throw new Exception('Projekt hittades inte');
            }
            
            // Bestäm typ automatiskt
            if ($detectedType === null) {
                throw new Exception('Deltagaren hittades inte');
            }
            
            // Använd detekterad typ om ingen typ angavs, annars validera
            // 'admin' är en roll, inte en deltagningstyp - acceptera 'user' för admins
            if (empty($participantType)) {
                // Om detekterad typ är 'admin', använd 'user' som deltagningstyp
                $participantType = ($detectedType === 'admin') ? 'user' : $detectedType;
            } elseif ($participantType !== $detectedType && !($participantType === 'user' && $detectedType === 'admin')) {
                throw new Exception('Deltagartyp matchar inte');
            }
            
            // Hämta namn för systemmeddelande
            $participantName = getParticipantName($participantId);
            
            // Använd mapping-funktionerna för att lägga till deltagare
            if ($listType === 'chatters') {
                $result = addChatter($projectId, $participantId, $participantType, $userId);
                if ($result === false) {
                    throw new Exception('Deltagaren finns redan i chatten');
                } elseif ($result !== true) {
                    throw new Exception($result); // Error message
                }
                // Skapa systemmeddelande
                $message = ($participantName ?: $participantId) . ' lades till i projektchatten';
                createSystemMessage($projectId, $message);
            } else {
                $result = addParticipant($projectId, $participantId, $participantType, $userId);
                if ($result === false) {
                    throw new Exception('Deltagaren finns redan i projektet');
                } elseif ($result !== true) {
                    throw new Exception($result); // Error message
                }
            }
            
            echo json_encode([
                'success' => true,
                'message' => 'Deltagare tillagd'
            ]);
            break;
            
        case 'remove':
            if (!$isAdmin) {
                require_once __DIR__ . '/../../security.php';
                logSecurityEvent('unauthorized_admin_action_attempt', [
                    'action' => 'remove_participant',
                    'reason' => 'not_admin'
                ], 'high', $userId, $projectId, 'remove_participant');
                throw new Exception('Endast admin kan ta bort deltagare');
            }
            
            $projectId = sanitizeInput($_POST['project_id'] ?? '', 'project_id');
            $participantType = sanitizeInput($_POST['type'] ?? '', 'string');
            $participantId = sanitizeInput($_POST['participant_id'] ?? '', 'string');
            $listType = sanitizeInput($_POST['list_type'] ?? 'participants', 'string'); // 'participants' eller 'chatters'
            
            if (empty($projectId) || empty($participantType) || empty($participantId)) {
                throw new Exception('Projekt-ID, typ och deltagar-ID krävs');
            }
            
            if (!in_array($listType, ['participants', 'chatters'])) {
                throw new Exception('Ogiltig listtyp');
            }
            
            // Kontrollera om deltagaren är den inloggade användaren (kan inte ta bort sig själv)
            if ($participantId === $userId) {
                throw new Exception('Du kan inte ta bort dig själv från projektet');
            }
            
            // Kontrollera att projektet finns
            $projects = readProjects($projectsFile);
            $projectFound = false;
            foreach ($projects as $project) {
                if (($project['id'] ?? '') === $projectId) {
                    $projectFound = true;
                    break;
                }
            }
            
            if (!$projectFound) {
                throw new Exception('Projekt hittades inte');
            }
            
            // Hämta namn för systemmeddelande innan borttagning
            $participantName = getParticipantName($participantId);
            
            // Ta bort från rätt mapping-fil
            if ($listType === 'chatters') {
                $success = removeChatter($projectId, $participantId);
                if (!$success) {
                    throw new Exception('Deltagaren hittades inte i chatten');
                }
                // Skapa systemmeddelande
                $message = ($participantName ?: $participantId) . ' togs bort från projektchatten';
                createSystemMessage($projectId, $message);
            } else {
                $success = removeParticipant($projectId, $participantId);
                if (!$success) {
                    throw new Exception('Deltagaren hittades inte i projektet');
                }
            }
            
            echo json_encode([
                'success' => true,
                'message' => 'Deltagare borttagen'
            ]);
            break;
            
        case 'clear_chat':
            if (!$isAdmin) {
                throw new Exception('Endast admin kan rensa chattar');
            }
            
            $projectId = sanitizeInput($_POST['project_id'] ?? '', 'project_id');
            
            if (empty($projectId)) {
                throw new Exception('Projekt-ID krävs');
            }
            
            $projects = readProjects($projectsFile);
            $projectFound = false;
            
            foreach ($projects as &$project) {
                if (($project['id'] ?? '') === $projectId) {
                    $projectFound = true;
                    
                    // Ta bort alla deltagare (utom admins)
                    $participants = $project['participants'] ?? [];
                    $adminIds = getAllAdminUserIds();
                    $filteredParticipants = [];
                    
                    foreach ($participants as $p) {
                        $pid = $p['participant_id'] ?? '';
                        // Behåll endast admins
                        if (!empty($pid) && in_array($pid, $adminIds)) {
                            $filteredParticipants[] = $p;
                        }
                    }
                    
                    $project['participants'] = $filteredParticipants;
                    break;
                }
            }
            
            if (!$projectFound) {
                throw new Exception('Projekt hittades inte');
            }
            
            // Ta bort chatten (messages.json och alla bilder)
            $chatDir = $chatsDir . '/' . $projectId;
            $chatFile = $chatDir . '/messages.json';
            $chatDeleted = false;
            $imagesDeleted = 0;
            
            if (is_dir($chatDir)) {
                // Ta bort alla bilder i chats-mappen
                $files = scandir($chatDir);
                foreach ($files as $file) {
                    if ($file === '.' || $file === '..' || $file === 'messages.json') {
                        continue;
                    }
                    $filePath = $chatDir . '/' . $file;
                    if (is_file($filePath)) {
                        if (unlink($filePath)) {
                            $imagesDeleted++;
                        }
                    }
                }
                
                // Ta bort messages.json om den finns
                if (file_exists($chatFile)) {
                    $chatDeleted = unlink($chatFile);
                }
                
                // Ta bort katalogen om den är tom
                if (count(scandir($chatDir)) === 2) { // . och ..
                    rmdir($chatDir);
                }
            }
            
            if (saveProjects($projectsFile, $projects)) {
                echo json_encode([
                    'success' => true,
                    'message' => 'Chatt och deltagare rensade',
                    'chat_deleted' => $chatDeleted,
                    'images_deleted' => $imagesDeleted
                ]);
            } else {
                throw new Exception('Kunde inte spara projekt');
            }
            break;
            
        case 'delete_chat':
            if (!$isAdmin) {
                require_once __DIR__ . '/../../security.php';
                logSecurityEvent('unauthorized_admin_action_attempt', [
                    'action' => 'delete_chat',
                    'reason' => 'not_admin'
                ], 'high', $userId, $projectId, 'delete_chat');
                throw new Exception('Endast admin kan ta bort chattar');
            }
            
            $projectId = sanitizeInput($_POST['project_id'] ?? '', 'project_id');
            
            if (empty($projectId)) {
                throw new Exception('Projekt-ID krävs');
            }
            
            $projects = readProjects($projectsFile);
            $projectFound = false;
            
            foreach ($projects as &$project) {
                if (($project['id'] ?? '') === $projectId) {
                    $projectFound = true;
                    
                    // Ta bort alla bilder i chats-mappen först
                    $chatDir = $chatsDir . '/' . $projectId;
                    $imagesDeleted = 0;
                    if (is_dir($chatDir)) {
                        $files = scandir($chatDir);
                        foreach ($files as $file) {
                            if ($file === '.' || $file === '..' || $file === 'messages.json') {
                                continue;
                            }
                            $filePath = $chatDir . '/' . $file;
                            if (is_file($filePath)) {
                                if (unlink($filePath)) {
                                    $imagesDeleted++;
                                }
                            }
                        }
                    }
                    
                    // Rensa alla meddelanden och lägg till systemmeddelandet
                    // Lägg till systemmeddelande "Chatten har rensats"
                    $systemMessage = [
                        'id' => time() . '_' . bin2hex(random_bytes(4)),
                        'type' => 'system',
                        'system_message' => 'Chatten har rensats',
                        'created_at' => date('c')
                    ];
                    $emptyMessages = [$systemMessage];
                    
                    // Spara den tomma listan med systemmeddelandet
                    saveChatMessages($projectId, $emptyMessages);
                    
                    // Ta bort alla deltagare (utom admins)
                    $participants = $project['participants'] ?? [];
                    $adminIds = getAllAdminUserIds();
                    $filteredParticipants = [];
                    
                    foreach ($participants as $p) {
                        $pid = $p['participant_id'] ?? '';
                        // Behåll endast admins
                        if (!empty($pid) && in_array($pid, $adminIds)) {
                            $filteredParticipants[] = $p;
                        }
                    }
                    
                    $project['participants'] = $filteredParticipants;
                    break;
                }
            }
            
            if (!$projectFound) {
                throw new Exception('Projekt hittades inte');
            }
            
            if (saveProjects($projectsFile, $projects)) {
                echo json_encode([
                    'success' => true,
                    'message' => 'Chatt och deltagare borttagna',
                    'images_deleted' => $imagesDeleted
                ]);
            } else {
                throw new Exception('Kunde inte spara projekt');
            }
            break;
            
        case 'create_chat':
            if (!$isAdmin) {
                require_once __DIR__ . '/../../security.php';
                logSecurityEvent('unauthorized_admin_action_attempt', [
                    'action' => 'create_chat',
                    'reason' => 'not_admin'
                ], 'high', $userId, $projectId, 'create_chat');
                throw new Exception('Endast admin kan skapa chattar');
            }
            
            $projectId = sanitizeInput($_POST['project_id'] ?? '', 'project_id');
            
            if (empty($projectId)) {
                throw new Exception('Projekt-ID krävs');
            }
            
            $projects = readProjects($projectsFile);
            $projectFound = false;
            
            foreach ($projects as &$project) {
                if (($project['id'] ?? '') === $projectId) {
                    $projectFound = true;
                    
                    // Kontrollera om projektet redan har participants
                    $participants = $project['participants'] ?? [];
                    $participantIds = [];
                    foreach ($participants as $p) {
                        $pid = $p['participant_id'] ?? '';
                        if (!empty($pid)) {
                            $participantIds[] = $pid;
                        }
                    }
                    
                    // Lägg till alla admins om de inte redan finns
                    $adminIds = getAllAdminUserIds();
                    $adminsAdded = 0;
                    foreach ($adminIds as $adminId) {
                        if (!in_array($adminId, $participantIds)) {
                            $participants[] = [
                                'type' => 'user',
                                'participant_id' => $adminId,
                                'added_at' => date('c'),
                                'added_by' => 'system'
                            ];
                            $adminsAdded++;
                        }
                    }
                    
                    // Om inga meddelanden finns, skapa ett systemmeddelande för att initiera chatten
                    $chatFile = $chatsDir . '/' . $projectId . '/messages.json';
                    $chatExists = file_exists($chatFile);
                    
                    if (!$chatExists) {
                        // Skapa chat-katalog och ett systemmeddelande för att initiera chatten
                        $chatDir = $chatsDir . '/' . $projectId;
                        if (!is_dir($chatDir)) {
                            mkdir($chatDir, 0755, true);
                        }
                        
                        // Skapa ett systemmeddelande om chatten skapades
                        $systemMessage = [
                            'id' => time() . '_' . bin2hex(random_bytes(4)),
                            'sender_type' => 'system',
                            'type' => 'system',
                            'system_message' => 'Chatten skapades',
                            'created_at' => date('c'),
                            'notified_users' => []
                        ];
                        
                        file_put_contents($chatFile, json_encode([$systemMessage], JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
                    }
                    
                    $project['participants'] = $participants;
                    break;
                }
            }
            
            if (!$projectFound) {
                throw new Exception('Projekt hittades inte');
            }
            
            if (saveProjects($projectsFile, $projects)) {
                echo json_encode([
                    'success' => true,
                    'message' => 'Chatt skapad',
                    'admins_added' => $adminsAdded
                ]);
            } else {
                throw new Exception('Kunde inte spara projekt');
            }
            break;
            
        default:
            throw new Exception('Ogiltig action');
    }
    
} catch (Exception $e) {
    // Logga felet med den befintliga loggfunktionen
    writeLog('participants_errors', $e->getMessage(), [
        'action' => $action ?? 'none',
        'user_id' => $userId ?? 'none',
        'request_method' => $_SERVER['REQUEST_METHOD'] ?? 'unknown',
        'request_uri' => $_SERVER['REQUEST_URI'] ?? 'unknown',
        'get_params' => $_GET,
        'post_params' => array_keys($_POST)
    ]);
    
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'message' => $e->getMessage(),
        'action' => $action ?? 'none',
        'debug' => [
            'user_id' => $userId ?? 'none',
            'request_method' => $_SERVER['REQUEST_METHOD'] ?? 'unknown'
        ]
    ]);
}

