<?php
require_once __DIR__ . '/../../security.php';
require_once __DIR__ . '/../../config.php';
require_once __DIR__ . '/../../lib/participant_helper.php';
require_once __DIR__ . '/../../lib/mapping_helper.php';

// Säkerhetsheaders
setSecurityHeaders();
header('Content-Type: application/json');

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['ok' => false, 'error' => 'Method not allowed']);
    exit;
}

$imagePath = sanitizeInput($_POST['imagePath'] ?? '', 'string');
$jsonPath = sanitizeInput($_POST['jsonPath'] ?? '', 'string');
$description = sanitizeInput($_POST['description'] ?? '', 'string');

if (empty($imagePath) || empty($jsonPath) || empty($description)) {
    echo json_encode(['ok' => false, 'error' => 'Missing required parameters']);
    exit;
}

// Hämta user_id från GET, POST eller session
$userId = '';
if (session_status() === PHP_SESSION_NONE) {
    @session_start();
}
if (isset($_GET['user_id']) && $_GET['user_id'] !== '') {
    $userId = sanitizeInput($_GET['user_id'], 'user_id');
} elseif (isset($_POST['user_id']) && $_POST['user_id'] !== '') {
    $userId = sanitizeInput($_POST['user_id'], 'user_id');
} elseif (isset($_COOKIE['user_id']) && $_COOKIE['user_id'] !== '') {
    $userId = sanitizeInput($_COOKIE['user_id'], 'user_id');
} elseif (isset($_SESSION['user_id']) && $_SESSION['user_id'] !== '') {
    $userId = sanitizeInput($_SESSION['user_id'], 'user_id');
}

// Kontrollera om användaren är admin
$isAdmin = false;
if (!empty($userId)) {
    $isAdmin = isParticipantAdmin($userId);
}

// Säkerhetskontroll: Validera att jsonPath är säker och inom uploads/
$baseDir = realpath(__DIR__ . '/../..') ?: dirname(dirname(__DIR__));
$uploadsDir = $baseDir . '/' . rtrim(UPLOAD_ROOT, '/\\');
$realJsonPath = realpath($jsonPath);

if ($realJsonPath === false || !file_exists($realJsonPath)) {
    http_response_code(404);
    echo json_encode(['ok' => false, 'error' => 'JSON file not found']);
    exit;
}

// Kontrollera att filen är inom uploads-mappen
if (strpos($realJsonPath, $uploadsDir) !== 0) {
    http_response_code(403);
    echo json_encode(['ok' => false, 'error' => 'Access denied']);
    exit;
}

// Om användaren inte är admin, kontrollera att de är deltagare i projektet
if (!$isAdmin && !empty($userId)) {
    // Extrahera projekt-ID från sökvägen (uploads/[project_id]/...)
    $relativePath = str_replace($uploadsDir . '/', '', $realJsonPath);
    $pathParts = explode('/', $relativePath);
    if (count($pathParts) > 0) {
        $projectId = $pathParts[0];
        
        // Kontrollera om användaren är deltagare i projektet
        $participants = readProjectMapping($projectId);
        $isParticipant = false;
        foreach ($participants as $participant) {
            if (($participant['participant_id'] ?? '') === $userId) {
                $isParticipant = true;
                break;
            }
        }
        
        if (!$isParticipant) {
            http_response_code(403);
            echo json_encode(['ok' => false, 'error' => 'Access denied']);
            exit;
        }
    } else {
        http_response_code(403);
        echo json_encode(['ok' => false, 'error' => 'Access denied']);
        exit;
    }
}

try {
    // Read the existing JSON data
    $jsonContent = file_get_contents($realJsonPath);
    if ($jsonContent === false) {
        throw new Exception('Could not read JSON file');
    }
    
    $data = json_decode($jsonContent, true);
    if ($data === null) {
        throw new Exception('Invalid JSON data');
    }
    
    // Update the note field
    $data['note'] = $description;
    
    // Write back to the JSON file
    $result = file_put_contents($realJsonPath, json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
    if ($result === false) {
        throw new Exception('Could not write to JSON file');
    }
    
    echo json_encode(['ok' => true, 'message' => 'Beskrivning tillagd']);
    
} catch (Exception $e) {
    echo json_encode(['ok' => false, 'error' => $e->getMessage()]);
}
?>

