<?php
require_once __DIR__ . '/../config.php';
require_once __DIR__ . '/../security.php';
require_once __DIR__ . '/../lib/participant_helper.php';
require_once __DIR__ . '/../lib/dal/repositories/ProjectRepository.php';

// Säkerhetsheaders tidigt
setSecurityHeaders();

// Svara alltid JSON
header('Content-Type: application/json; charset=utf-8');

// Skapa repository
$projectRepo = new ProjectRepository();

// Hämta action från URL
$action = $_GET['action'] ?? '';

try {
  switch ($action) {
    case 'list':
      $projects = $projectRepo->getAll();
      echo json_encode([
        'success' => true,
        'projects' => $projects
      ]);
      break;
      
    case 'get':
      $projectId = $_GET['project_id'] ?? '';
      if (empty($projectId)) {
        throw new Exception('Projekt-ID krävs');
      }
      
      $project = $projectRepo->getById($projectId);
      if ($project === null) {
        throw new Exception('Projekt hittades inte');
      }
      
      echo json_encode([
        'success' => true,
        'project' => $project
      ]);
      break;
      
    case 'add':
      // Spegla admin-regler: kräver id och name; tillåt bindestreck.
      // Frontend kan skicka 'projectName' och 'projectId'. Stöd även 'name'.
      $id = trim($_POST['projectId'] ?? ($_POST['id'] ?? ''));
      $name = trim($_POST['projectName'] ?? ($_POST['name'] ?? ''));
      $description = trim($_POST['description'] ?? '');
      $startDate = $_POST['startDate'] ?? '';

      if ($id === '') {
        throw new Exception('Projekt-ID krävs');
      }
      if ($name === '') {
        throw new Exception('Projektnamn krävs');
      }

      // Kontrollera om projekt-ID redan finns
      $existingProject = $projectRepo->getById($id);
      if ($existingProject !== null) {
        throw new Exception('Ett projekt med detta ID finns redan');
      }

      // Hämta user_id om det finns (för att lägga till skaparen som admin-deltagare)
      $userId = sanitizeInput($_POST['user_id'] ?? '', 'user_id');
      
      $newProject = [
        'id' => $id,
        'name' => $name,
        'description' => $description,
        'startDate' => $startDate,
        'status' => 'active',
        'created_at' => date('Y-m-d H:i:s'),
        'participants' => []
      ];
      
      // Lägg till skaparen som deltagare om user_id finns
      if (!empty($userId)) {
        $newProject['participants'][] = [
          'type' => 'user',
          'participant_id' => $userId,
          'added_at' => date('c'),
          'added_by' => $userId
        ];
      }

      // Lägg till alla admins som deltagare
      $adminIds = getAllAdminUserIds();
      foreach ($adminIds as $adminId) {
        // Kontrollera att admin inte redan är deltagare (t.ex. om skaparen är admin)
        $alreadyParticipant = false;
        foreach ($newProject['participants'] as $p) {
          $pid = $p['participant_id'] ?? '';
          if ($pid === $adminId) {
            $alreadyParticipant = true;
            break;
          }
        }
        if (!$alreadyParticipant) {
          $newProject['participants'][] = [
            'type' => 'user',
            'participant_id' => $adminId,
            'added_at' => date('c'),
            'added_by' => 'system'
          ];
        }
      }

      if ($projectRepo->create($newProject)) {
        echo json_encode([
          'success' => true,
          'message' => 'Projekt skapat',
          'project' => $newProject
        ]);
      } else {
        throw new Exception('Kunde inte spara projekt');
      }
      break;
      
    case 'update':
      $projectId = $_POST['projectId'] ?? '';
      $name = trim($_POST['name'] ?? '');
      $description = trim($_POST['description'] ?? '');
      $startDate = $_POST['startDate'] ?? '';
      $archived = $_POST['archived'] ?? '0';
      
      // Hantera både 'archived' (från checkbox) och 'status' (från direkt status-ändring)
      if (isset($_POST['archived'])) {
        $status = ($archived === '1' || $archived === 'true') ? 'archived' : 'active';
      } else {
        $status = $_POST['status'] ?? 'active';
      }
      
      if (empty($projectId) || empty($name)) {
        throw new Exception('Projekt-ID och namn krävs');
      }
      
      $project = $projectRepo->getById($projectId);
      if ($project === null) {
        throw new Exception('Projekt hittades inte');
      }
      
      $oldStatus = $project['status'] ?? 'active';
      
      $project['name'] = $name;
      $project['description'] = $description;
      $project['startDate'] = $startDate;
      $project['status'] = $status;
      $project['updated_at'] = date('Y-m-d H:i:s');
      
      // Sätt archived_at när projektet arkiveras
      if ($status === 'archived') {
        // Sätt archived_at om projektet just arkiverades
        if ($oldStatus !== 'archived') {
          $project['archived_at'] = date('Y-m-d H:i:s');
        } else {
          // Projektet var redan arkiverat - sätt archived_at om det saknas eller är ogiltigt
          $archivedAt = $project['archived_at'] ?? null;
          if ($archivedAt === null || $archivedAt === '' || trim((string)$archivedAt) === '') {
            $project['archived_at'] = date('Y-m-d H:i:s');
          }
        }
      } elseif ($status !== 'archived' && $oldStatus === 'archived') {
        // Ta bort archived_at när projektet aktiveras igen
        unset($project['archived_at']);
      }
      
      if ($projectRepo->update($projectId, $project)) {
        echo json_encode([
          'success' => true,
          'message' => 'Projekt uppdaterat'
        ]);
      } else {
        throw new Exception('Kunde inte spara ändringar');
      }
      break;
      
    case 'toggle_status':
      $projectId = $_POST['projectId'] ?? '';
      
      if (empty($projectId)) {
        throw new Exception('Projekt-ID krävs');
      }
      
      $project = $projectRepo->getById($projectId);
      if ($project === null) {
        throw new Exception('Projekt hittades inte');
      }
      
      $project['status'] = $project['status'] === 'active' ? 'archived' : 'active';
      $project['updated_at'] = date('Y-m-d H:i:s');
      
      if ($projectRepo->update($projectId, $project)) {
        $updatedProject = $projectRepo->getById($projectId);
        echo json_encode([
          'success' => true,
          'message' => 'Projektstatus ändrad',
          'project' => $updatedProject
        ]);
      } else {
        throw new Exception('Kunde inte spara ändringar');
      }
      break;
      
    case 'delete':
      $projectId = $_POST['projectId'] ?? '';
      
      if (empty($projectId)) {
        throw new Exception('Projekt-ID krävs');
      }
      
      $project = $projectRepo->getById($projectId);
      if ($project === null) {
        throw new Exception('Projekt hittades inte');
      }
      
      if ($projectRepo->delete($projectId)) {
        echo json_encode([
          'success' => true,
          'message' => 'Projekt borttaget'
        ]);
      } else {
        throw new Exception('Kunde inte spara ändringar');
      }
      break;
      
    default:
      throw new Exception('Ogiltig action');
  }
  
} catch (Exception $e) {
  http_response_code(400);
  echo json_encode([
    'success' => false,
    'message' => $e->getMessage()
  ]);
}
?>
