<?php
require_once __DIR__ . '/../config.php';
require_once __DIR__ . '/../security.php';
require_once __DIR__ . '/../lib/participant_helper.php';

setSecurityHeaders();

// Svara alltid JSON
header('Content-Type: application/json; charset=utf-8');

// Sökvägar till JSON-filer
$messagesFile = __DIR__ . '/../data/system_messages.json';
$readFile = __DIR__ . '/../data/system_message_read.json';
$userIdsFile = __DIR__ . '/../data/userids.json';

// Funktion för att läsa alla systemmeddelanden
function readSystemMessages($filePath) {
    if (!file_exists($filePath)) {
        return [];
    }
    
    $content = file_get_contents($filePath);
    if ($content === false) {
        return [];
    }
    
    $data = json_decode($content, true);
    return is_array($data) ? $data : [];
}

// Funktion för att spara alla systemmeddelanden
function saveSystemMessages($filePath, $messages) {
    $json = json_encode($messages, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
    return file_put_contents($filePath, $json) !== false;
}

// Funktion för att läsa acknowledgments
function readAcknowledgments($filePath) {
    if (!file_exists($filePath)) {
        return [];
    }
    
    $content = file_get_contents($filePath);
    if ($content === false) {
        return [];
    }
    
    $data = json_decode($content, true);
    return is_array($data) ? $data : [];
}

// Funktion för att spara acknowledgments
function saveAcknowledgments($filePath, $acknowledgments) {
    $json = json_encode($acknowledgments, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
    return file_put_contents($filePath, $json) !== false;
}

// Funktion för att kontrollera om användare har läst ett specifikt meddelande
function hasUserReadMessage($acknowledgments, $messageId, $userId) {
    if (empty($userId) || empty($messageId)) {
        return false;
    }
    
    return isset($acknowledgments[$messageId][$userId]);
}

// Funktion för att kontrollera om en användare är extern
function isExternalUser($userId, $userIdsFile) {
    if (empty($userId) || !file_exists($userIdsFile)) {
        return false;
    }
    
    $users = json_decode(file_get_contents($userIdsFile), true) ?: [];
    foreach ($users as $user) {
        if (($user['user_id'] ?? '') === $userId) {
            return ($user['user_type'] ?? 'user') === 'extern';
        }
    }
    
    return false;
}

// Funktion för att hämta olästa meddelanden för en användare
function getUnreadMessages($messages, $acknowledgments, $userId, $userIdsFile) {
    $unread = [];
    $isExternal = isExternalUser($userId, $userIdsFile);
    $now = time();
    
    foreach ($messages as $message) {
        $messageId = $message['id'] ?? '';
        $includeExternal = $message['include_external'] ?? false;
        
        // Kontrollera om meddelandet har gått ut (sista visningsdag)
        $expiresAt = $message['expires_at'] ?? null;
        if ($expiresAt) {
            $expiresTimestamp = strtotime($expiresAt);
            if ($expiresTimestamp !== false && $expiresTimestamp < $now) {
                // Meddelandet har gått ut, hoppa över det
                continue;
            }
        }
        
        // Om användaren är extern och meddelandet inte inkluderar externa, hoppa över
        if ($isExternal && !$includeExternal) {
            continue;
        }
        
        if (!hasUserReadMessage($acknowledgments, $messageId, $userId)) {
            $unread[] = $message;
        }
    }
    
    return $unread;
}

// Hämta user_id och action
$userId = sanitizeInput($_GET['user_id'] ?? $_POST['user_id'] ?? '', 'user_id');
$action = $_GET['action'] ?? $_POST['action'] ?? 'get';

// Kontrollera admin-rättigheter för save och delete
$isAdmin = false;
if (!empty($userId)) {
    $isAdmin = isParticipantAdmin($userId);
    
    // Debug: Logga om admin-kontrollen misslyckas (endast i utvecklingsmiljö)
    if (defined('DEBUG') && DEBUG && !$isAdmin && in_array($action, ['save', 'delete', 'list', 'stats'])) {
        error_log("system_message.php: Admin check failed for user_id: $userId, action: $action");
    }
}

try {
    switch ($action) {
        case 'get':
        case 'get_unread':
            // Hämta alla olästa meddelanden för användaren
            $messages = readSystemMessages($messagesFile);
            $acknowledgments = readAcknowledgments($readFile);
            $unreadMessages = getUnreadMessages($messages, $acknowledgments, $userId, $userIdsFile);
            
            echo json_encode([
                'success' => true,
                'messages' => $unreadMessages
            ]);
            break;
            
        case 'list':
            // Lista alla meddelanden (endast admin)
            if (!$isAdmin) {
                http_response_code(403);
                echo json_encode([
                    'success' => false,
                    'message' => 'Endast admin kan lista meddelanden'
                ]);
                exit;
            }
            
            $messages = readSystemMessages($messagesFile);
            // Sortera efter created_at (nyaste först)
            usort($messages, function($a, $b) {
                $timeA = strtotime($a['created_at'] ?? '0');
                $timeB = strtotime($b['created_at'] ?? '0');
                return $timeB - $timeA;
            });
            
            echo json_encode([
                'success' => true,
                'messages' => $messages
            ]);
            break;
            
        case 'save':
            // Skapa nytt systemmeddelande (endast admin)
            if (!$isAdmin) {
                http_response_code(403);
                echo json_encode([
                    'success' => false,
                    'message' => 'Endast admin kan skapa systemmeddelanden'
                ]);
                exit;
            }
            
            $messageText = sanitizeInput($_POST['message'] ?? '', 'string');
            $includeExternal = isset($_POST['include_external']) && $_POST['include_external'] === '1';
            $expiresAt = !empty($_POST['expires_at']) ? sanitizeInput($_POST['expires_at'], 'string') : null;
            
            if (empty($messageText)) {
                http_response_code(400);
                echo json_encode([
                    'success' => false,
                    'message' => 'Meddelandetext krävs'
                ]);
                exit;
            }
            
            // Validera datumformat om det finns
            if ($expiresAt) {
                $expiresTimestamp = strtotime($expiresAt);
                if ($expiresTimestamp === false) {
                    http_response_code(400);
                    echo json_encode([
                        'success' => false,
                        'message' => 'Ogiltigt datumformat för sista visningsdag'
                    ]);
                    exit;
                }
                // Konvertera till ISO 8601 format
                $expiresAt = date('c', $expiresTimestamp);
            }
            
            $messages = readSystemMessages($messagesFile);
            
            // Skapa nytt meddelande med unikt ID
            $newMessage = [
                'id' => time() . '_' . bin2hex(random_bytes(4)),
                'message' => $messageText,
                'created_at' => date('c'),
                'created_by' => $userId,
                'include_external' => $includeExternal
            ];
            
            // Lägg till expires_at om det finns
            if ($expiresAt) {
                $newMessage['expires_at'] = $expiresAt;
            }
            
            $messages[] = $newMessage;
            
            if (saveSystemMessages($messagesFile, $messages)) {
                echo json_encode([
                    'success' => true,
                    'message' => 'Systemmeddelande skapat',
                    'message_id' => $newMessage['id']
                ]);
            } else {
                http_response_code(500);
                echo json_encode([
                    'success' => false,
                    'message' => 'Kunde inte spara systemmeddelande'
                ]);
            }
            break;
            
        case 'acknowledge':
            // Markera specifikt meddelande som läst
            if (empty($userId)) {
                http_response_code(400);
                echo json_encode([
                    'success' => false,
                    'message' => 'User ID krävs'
                ]);
                exit;
            }
            
            $messageId = sanitizeInput($_POST['message_id'] ?? '', 'string');
            
            if (empty($messageId)) {
                http_response_code(400);
                echo json_encode([
                    'success' => false,
                    'message' => 'Message ID krävs'
                ]);
                exit;
            }
            
            $acknowledgments = readAcknowledgments($readFile);
            
            // Initiera array för meddelandet om den inte finns
            if (!isset($acknowledgments[$messageId])) {
                $acknowledgments[$messageId] = [];
            }
            
            // Markera som läst
            $acknowledgments[$messageId][$userId] = date('c');
            
            if (saveAcknowledgments($readFile, $acknowledgments)) {
                echo json_encode([
                    'success' => true,
                    'message' => 'Meddelande markerat som läst'
                ]);
            } else {
                http_response_code(500);
                echo json_encode([
                    'success' => false,
                    'message' => 'Kunde inte spara acknowledgment'
                ]);
            }
            break;
            
        case 'delete':
            // Ta bort systemmeddelande (endast admin)
            if (!$isAdmin) {
                http_response_code(403);
                echo json_encode([
                    'success' => false,
                    'message' => 'Endast admin kan ta bort systemmeddelanden'
                ]);
                exit;
            }
            
            $messageId = sanitizeInput($_POST['message_id'] ?? '', 'string');
            
            if (empty($messageId)) {
                http_response_code(400);
                echo json_encode([
                    'success' => false,
                    'message' => 'Message ID krävs'
                ]);
                exit;
            }
            
            $messages = readSystemMessages($messagesFile);
            $acknowledgments = readAcknowledgments($readFile);
            
            // Ta bort meddelandet
            $messages = array_filter($messages, function($msg) use ($messageId) {
                return ($msg['id'] ?? '') !== $messageId;
            });
            $messages = array_values($messages);
            
            // Ta bort acknowledgments för detta meddelande
            if (isset($acknowledgments[$messageId])) {
                unset($acknowledgments[$messageId]);
            }
            
            if (saveSystemMessages($messagesFile, $messages) && saveAcknowledgments($readFile, $acknowledgments)) {
                echo json_encode([
                    'success' => true,
                    'message' => 'Systemmeddelande borttaget'
                ]);
            } else {
                http_response_code(500);
                echo json_encode([
                    'success' => false,
                    'message' => 'Kunde inte ta bort systemmeddelande'
                ]);
            }
            break;
            
        case 'stats':
            // Hämta statistik (endast admin)
            if (!$isAdmin) {
                http_response_code(403);
                echo json_encode([
                    'success' => false,
                    'message' => 'Endast admin kan se statistik'
                ]);
                exit;
            }
            
            $messages = readSystemMessages($messagesFile);
            $acknowledgments = readAcknowledgments($readFile);
            
            // Räkna totalt antal användare
            $totalUsers = 0;
            if (file_exists($userIdsFile)) {
                $users = json_decode(file_get_contents($userIdsFile), true) ?: [];
                $totalUsers = count($users);
            }
            
            // Beräkna statistik per meddelande
            $messageStats = [];
            foreach ($messages as $message) {
                $messageId = $message['id'] ?? '';
                $readCount = isset($acknowledgments[$messageId]) ? count($acknowledgments[$messageId]) : 0;
                $messageStats[] = [
                    'id' => $messageId,
                    'read_count' => $readCount,
                    'total_users' => $totalUsers
                ];
            }
            
            echo json_encode([
                'success' => true,
                'total_messages' => count($messages),
                'total_users' => $totalUsers,
                'message_stats' => $messageStats
            ]);
            break;
            
        default:
            http_response_code(400);
            echo json_encode([
                'success' => false,
                'message' => 'Ogiltig action'
            ]);
            break;
    }
    
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => $e->getMessage()
    ]);
}
?>
