<?php
/**
 * Lägg till intern användare som deltagare i projektet om de inte redan är det,
 * och redirecta sedan till chatten
 * Kontrollerar också om användaren har pushover_user_key, annars omdirigeras till instruktionssidan
 */
require_once __DIR__ . '/../config.php';
require_once __DIR__ . '/../lib/participant_helper.php';
require_once __DIR__ . '/../security.php';

setSecurityHeaders();

// Hämta parametrar
$projectId = sanitizeInput($_GET['project_id'] ?? '', 'project_id');
$participantId = sanitizeInput($_GET['user_id'] ?? '', 'string');
$mobile = sanitizeInput($_GET['mobile'] ?? '', 'string');

if (empty($projectId)) {
    die('Projekt-ID krävs');
}

// Om mobilnummer finns, hitta extern partner via mobilnummer
if (!empty($mobile) && empty($participantId)) {
    $userIdsFile = __DIR__ . '/../data/userids.json';
    if (file_exists($userIdsFile)) {
        $allUsers = json_decode(file_get_contents($userIdsFile), true) ?: [];
        foreach ($allUsers as $user) {
            $userType = $user['user_type'] ?? 'user';
            if ($userType === 'extern' && ($user['mobile'] ?? '') === $mobile) {
                $participantId = $user['user_id'] ?? '';
                break;
            }
        }
    }
}

if (empty($participantId) && empty($mobile)) {
    die('Användar-ID eller mobilnummer krävs');
}

// Kontrollera typ av användare
$participantType = null;
$isExternUserDirect = false; // Ytterligare kontroll för externa användare
if (!empty($participantId)) {
    $participantType = getParticipantType($participantId);
    
    // Ytterligare kontroll: Läs direkt från userids.json för att säkerställa att användaren är extern
    $userIdsFile = __DIR__ . '/../data/userids.json';
    if (file_exists($userIdsFile)) {
        $allUsers = json_decode(file_get_contents($userIdsFile), true) ?: [];
        foreach ($allUsers as $user) {
            if (($user['user_id'] ?? '') === $participantId) {
                $userType = $user['user_type'] ?? 'user';
                if ($userType === 'extern') {
                    $isExternUserDirect = true;
                }
                break;
            }
        }
    }
}

// Läs projektdata
$projectsFile = __DIR__ . '/../data/projects.json';
$projects = [];
if (file_exists($projectsFile)) {
    $content = file_get_contents($projectsFile);
    if ($content !== false) {
        $projects = json_decode($content, true) ?: [];
    }
}

// Hitta projektet och kontrollera om användaren redan är deltagare
$projectFound = false;
$isAlreadyParticipant = false;
$participantsChanged = false; // Flagga för att spåra om något ändrats

foreach ($projects as &$project) {
    if (($project['id'] ?? '') === $projectId) {
        $projectFound = true;
        // Säkerställ att participants är en array, inte null
        if (!isset($project['participants']) || !is_array($project['participants'])) {
            $project['participants'] = [];
        }
        $participants = $project['participants'];
        
        // Bestäm participant type för externa partners
        $participantTypeForProject = 'user';
        if (!empty($mobile) || ($participantType === 'extern' || $participantType === null)) {
            // Om mobilnummer finns eller participantType är extern/null, använd extern
            $participantTypeForProject = 'extern';
        }
        
        // Kontrollera om användaren redan är deltagare
        foreach ($participants as $p) {
            $pid = $p['participant_id'] ?? '';
            if ($pid === $participantId) {
                $isAlreadyParticipant = true;
                break;
            }
        }
        
        // Kontrollera om användaren är extern (staff)
        // Använd både getParticipantType() och direkt kontroll från userids.json för säkerhet
        $isExternUser = ($participantType === 'extern') || $isExternUserDirect;
        
        // Om användaren inte redan är deltagare och vi har participantId, lägg till dem
        // MEN: Externa användare (staff) ska INTE läggas till automatiskt om de inte redan är deltagare
        // De ska bara läggas till om en admin har kryssat av projektet i deras profil
        if (!$isAlreadyParticipant && !empty($participantId) && !$isExternUser) {
            $newParticipant = [
                'type' => $participantTypeForProject,
                'participant_id' => $participantId,
                'added_at' => date('c'),
                'added_by' => $participantId // Användaren lägger till sig själv
            ];
            
            $participants[] = $newParticipant;
            $project['participants'] = $participants;
            $participantsChanged = true; // Markera att något ändrats
        }
        break;
    }
}

// Spara endast om något faktiskt ändrats
if ($projectFound && $participantsChanged) {
    $json = json_encode($projects, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
    file_put_contents($projectsFile, $json);
}

// Kontrollera om användaren har pushover_user_key
$userIdsFile = __DIR__ . '/../data/userids.json';
$hasPushoverKey = false;
$redirectUserId = $participantId;
$redirectMobile = $mobile;

if (file_exists($userIdsFile)) {
    $userIds = json_decode(file_get_contents($userIdsFile), true) ?: [];
    foreach ($userIds as $user) {
        // För intern användare: matcha på user_id
        if (!empty($participantId) && ($user['user_id'] ?? '') === $participantId) {
            if (!empty($user['pushover_user_key'] ?? '')) {
                $hasPushoverKey = true;
            }
            break;
        }
        // För extern partner: matcha på mobilnummer
        if (!empty($mobile) && ($user['mobile'] ?? '') === $mobile) {
            $userType = $user['user_type'] ?? 'user';
            if ($userType === 'extern') {
                $redirectUserId = $user['user_id'] ?? '';
                if (!empty($user['pushover_user_key'] ?? '')) {
                    $hasPushoverKey = true;
                }
                break;
            }
        }
    }
}

// Om användaren inte har pushover_user_key, redirecta till instruktionssidan
if (!$hasPushoverKey) {
    $redirectUrl = 'pushover_instr.html?project_id=' . urlencode($projectId);
    if (!empty($redirectUserId)) {
        $redirectUrl .= '&user_id=' . urlencode($redirectUserId);
    }
    if (!empty($redirectMobile)) {
        $redirectUrl .= '&mobile=' . urlencode($redirectMobile);
    }
    header('Location: ' . $redirectUrl);
    exit;
}

// Redirecta till chatten
$chatUrl = 'chat.php?project_id=' . urlencode($projectId);
if (!empty($redirectUserId)) {
    $chatUrl .= '&user_id=' . urlencode($redirectUserId);
}
if (!empty($redirectMobile)) {
    $chatUrl .= '&mobile=' . urlencode($redirectMobile);
}
header('Location: ' . $chatUrl);
exit;

