<?php
require_once __DIR__ . '/../config.php';
require_once __DIR__ . '/../security.php';
require_once __DIR__ . '/../lib/participant_helper.php';
require_once __DIR__ . '/../lib/mapping_helper.php';

setSecurityHeaders();

// Hämta parametrar
$projectId = sanitizeInput($_GET['project_id'] ?? '', 'project_id');
$participantId = sanitizeInput($_GET['user_id'] ?? '', 'string');
$participantType = getParticipantType($participantId);

// Identifiera användartyp från userids.json för att skilja mellan user, admin och extern
$userType = 'user'; // default
if (!empty($participantId)) {
    $userIdsFile = __DIR__ . '/../data/userids.json';
    if (file_exists($userIdsFile)) {
        $allUsers = json_decode(file_get_contents($userIdsFile), true) ?: [];
        foreach ($allUsers as $user) {
            if (($user['user_id'] ?? '') === $participantId) {
                $userType = $user['user_type'] ?? 'user';
                break;
            }
        }
    }
}

if (empty($projectId)) {
    die('Projekt-ID krävs');
}

// Läs projektdata
$projectsFile = __DIR__ . '/../data/projects.json';
$projects = [];
if (file_exists($projectsFile)) {
    $projects = json_decode(file_get_contents($projectsFile), true) ?: [];
}

$project = null;
foreach ($projects as $p) {
    if (($p['id'] ?? '') === $projectId) {
        $project = $p;
        break;
    }
}

if (!$project) {
    die('Projekt hittades inte');
}

// Kontrollera om projektet är arkiverat
$isArchived = ($project['status'] ?? 'active') === 'archived';

// Kontrollera om användaren är admin FÖRE deltagarskap-kontrollen
// Försök först med participantId, sedan med user_id direkt från URL
$isAdmin = false;
if (!empty($participantId)) {
    $isAdmin = isParticipantAdmin($participantId);
} else {
    // Om participantId är tom, kontrollera user_id direkt från URL
    $userIdFromUrl = sanitizeInput($_GET['user_id'] ?? '', 'user_id');
    if (!empty($userIdFromUrl)) {
        $isAdmin = isParticipantAdmin($userIdFromUrl);
        // Om admin hittades via user_id, sätt participantId också
        if ($isAdmin && empty($participantId)) {
            $participantId = $userIdFromUrl;
        }
    }
}

// Kontrollera om användaren är aktiv - om inte, ta bort dem från alla projekt och logga ut
// OBS: Även admin-användare måste vara aktiva för att kunna chatta
if (!empty($participantId)) {
    if (!isParticipantActive($participantId)) {
        // Användaren är inaktiv - ta bort dem från alla projekt
        $projects = [];
        if (file_exists($projectsFile)) {
            $projects = json_decode(file_get_contents($projectsFile), true) ?: [];
        }
        
        foreach ($projects as $proj) {
            $projId = $proj['id'] ?? '';
            if (!empty($projId)) {
                // Ta bort från både participants och chatters
                removeParticipant($projId, $participantId);
                removeChatter($projId, $participantId);
            }
        }
        
        // Redirecta till samma sida som icke-admin användare kommer till när de försöker komma åt adminsidor
        header('Location: ../adm/check_admin.php?user_id=' . urlencode($participantId));
        exit;
    }
}

// Kontrollera chattillgång (hoppa över för arkiverade projekt)
// Admins tillåts alltid automatiskt
$projectsFile = __DIR__ . '/../data/projects.json';

// För arkiverade projekt, tillåt alla att se chatten (read-only)
if ($isArchived) {
    $isParticipant = true;
} else {
    // För aktiva projekt, kontrollera chattillgång (använder canUserChat som kontrollerar can_chat OCH chatters)
    $isParticipant = canUserChat($projectId, $participantId, $projectsFile);
    
    // Om participantId är tom, försök autentisera via mobilnummer
    if (!$isParticipant && empty($participantId)) {
        // Försök autentisera staff via mobilnummer
        $mobile = sanitizeInput($_GET['mobile'] ?? '', 'string');
        if (!empty($mobile)) {
            $userIdsFile = __DIR__ . '/../data/userids.json';
            if (file_exists($userIdsFile)) {
                $allUsers = json_decode(file_get_contents($userIdsFile), true) ?: [];
                foreach ($allUsers as $user) {
                    // Filtrera endast externa användare (user_type === "extern")
                    $userType = $user['user_type'] ?? 'user';
                    if ($userType !== 'extern') {
                        continue;
                    }
                    
                    $staffIdFromFile = $user['user_id'] ?? '';
                    // Kontrollera att staff är aktiv
                    $isActive = $user['active'] ?? true;
                    if ($isActive === false || $isActive === 0 || $isActive === '0' || $isActive === 'false') {
                        continue; // Hoppa över inaktiva staff
                    }
                    
                    // Kontrollera att extern användare kan delta i chatt
                    $canChat = $user['can_chat'] ?? true;
                    if ($canChat === false || $canChat === 0 || $canChat === '0' || $canChat === 'false') {
                        continue; // Hoppa över externa användare som inte kan chatta
                    }
                    
                    if (($user['mobile'] ?? '') === $mobile) {
                        // Kontrollera om staff_id finns i chatters
                        $chatters = $project['chatters'] ?? [];
                        foreach ($chatters as $c) {
                            $cid = $c['participant_id'] ?? '';
                            if ($cid === $staffIdFromFile) {
                                $participantId = $staffIdFromFile;
                                $participantType = 'extern';
                                $isParticipant = true;
                                break 2; // Break både inner och outer loop
                            }
                        }
                    }
                }
            }
        }
    }

    // Kontrollera om extern användare kan delta i chatt
    if ($isParticipant && !empty($participantId) && $userType === 'extern') {
        if (!canExternalUserChat($participantId)) {
            // Redirecta till index.html om extern användare inte kan chatta
            $redirectUrl = '../index.html';
            $redirectUrl .= '?user_id=' . urlencode($participantId);
            header('Location: ' . $redirectUrl);
            exit;
        }
    }
    
    if (!$isParticipant) {
        // Redirecta till index.html om användaren inte är deltagare
        $redirectUrl = '../index.html';
        if (!empty($participantId)) {
            $redirectUrl .= '?user_id=' . urlencode($participantId);
        }
        header('Location: ' . $redirectUrl);
        exit;
    }
}

// Hämta alla projekt användaren är medlem i (eller alla för admin)
$userProjects = [];
if ($isAdmin) {
    // För admin: Visa alla aktiva projekt
    foreach ($projects as $p) {
        if (($p['status'] ?? 'active') !== 'archived') {
            $userProjects[] = [
                'id' => $p['id'] ?? '',
                'name' => $p['name'] ?? $p['id'] ?? ''
            ];
        }
    }
} else {
    if ($userType === 'extern') {
        // För externa partners: Visa bara projekt de är participants i (för projektåtkomst)
        foreach ($projects as $p) {
            if (($p['status'] ?? 'active') === 'archived') {
                continue;
            }
            $projId = $p['id'] ?? '';
            if (!empty($projId)) {
                $projParticipants = readProjectMapping($projId);
                foreach ($projParticipants as $part) {
                    if (($part['participant_id'] ?? '') === $participantId) {
                        $userProjects[] = [
                            'id' => $p['id'] ?? '',
                            'name' => $p['name'] ?? $p['id'] ?? ''
                        ];
                        break;
                    }
                }
            }
        }
    } else {
        // För vanliga users: Visa alla aktiva projekt
        foreach ($projects as $p) {
            if (($p['status'] ?? 'active') !== 'archived') {
                $userProjects[] = [
                    'id' => $p['id'] ?? '',
                    'name' => $p['name'] ?? $p['id'] ?? ''
                ];
            }
        }
    }
}

// Sortera alfabetiskt efter namn
usort($userProjects, function($a, $b) {
    return strcasecmp($a['name'], $b['name']);
});
?>
<!DOCTYPE html>
<html lang="sv">
<head>
    <meta charset="utf-8" />
    <meta name="viewport" content="width=device-width,initial-scale=1" />
    <title><?= htmlspecialchars($project['name'] ?? '') ?> - Chatt - ByggArkiv</title>
    <?php if (!$isArchived): ?>
    <?php endif; ?>
    <style>
        body {
            font-family: system-ui, -apple-system, Segoe UI, Roboto, sans-serif;
            margin: 0;
            padding: 0;
            background-color: #f5f5f5;
            height: 100vh;
            display: flex;
            flex-direction: column;
        }
        .logo-header {
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            z-index: 100;
            background: white;
            border-bottom: 1px solid #e0e0e0;
            box-shadow: 0 2px 4px rgba(0,0,0,0.1);
            margin: 0;
            transition: transform 0.3s ease-out, opacity 0.3s ease-out;
            transform: translateY(0);
            opacity: 1;
        }
        .logo-header.scrolled-up {
            transform: translateY(-100%);
            opacity: 0;
        }
        .logo-section {
            padding: 10px 20px;
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin: 0;
        }
        .logo-header img, .logo-header .logo {
            max-width: 100%;
            height: auto;
            max-height: 80px;
            display: block;
        }
        .header {
            position: fixed;
            top: 0; /* Sätts dynamiskt via JavaScript */
            left: 0;
            right: 0;
            z-index: 99;
            background: white;
            padding: 8px 20px;
            border-top: none;
            border-bottom: 1px solid #e0e0e0;
            box-shadow: 0 2px 4px rgba(0,0,0,0.1);
            margin-top: 0;
            transition: top 0.3s ease-out;
        }
        .header.at-top {
            top: 0 !important;
        }
        .header-content {
            max-width: 1200px;
            margin: 0 auto;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        .header h1 {
            margin: 0;
            font-size: 18px;
            color: #1a73e8;
        }
        #projectSelect {
            font-size: 18px;
            font-weight: 600;
            font-family: system-ui, -apple-system, Segoe UI, Roboto, sans-serif !important;
            border: none;
            background: transparent;
            cursor: pointer;
            color: #333;
            padding: 4px 8px;
            border-radius: 4px;
            transition: background-color 0.2s;
            -webkit-appearance: none;
            -moz-appearance: none;
            appearance: none;
        }
        #projectSelect option {
            font-family: system-ui, -apple-system, Segoe UI, Roboto, sans-serif !important;
            font-size: 18px;
            font-weight: 600;
        }
        #projectSelect:hover {
            background-color: #f0f0f0;
        }
        #projectSelect:focus {
            outline: 2px solid #1a73e8;
            outline-offset: 2px;
        }
        .header .back-link {
            color: #1a73e8;
            text-decoration: none;
        }
        .header .back-link:hover {
            text-decoration: underline;
        }
        .chat-container {
            flex: 1;
            max-width: 1200px;
            margin: 0 auto;
            width: 100%;
            display: flex;
            flex-direction: column;
            overflow: hidden;
            padding-top: 0;
        }
        .participants-section {
            background: white;
            padding: 16px 20px;
            border-top: 1px solid #e0e0e0;
            position: relative;
            z-index: 1002;
        }
        .participants-section h3 {
            margin: 0 0 10px 0;
            font-size: 14px;
            color: #666;
        }
        .participants-list {
            display: flex;
            flex-wrap: wrap;
            gap: 8px;
            max-height: 200px;
            overflow-y: auto;
            overflow-x: hidden;
        }
        .participant-tag {
            background: #e3f2fd;
            color: #1976d2;
            padding: 4px 8px;
            border-radius: 4px;
            font-size: 12px;
            transition: background-color 0.2s;
            height: 24px;
            display: inline-flex;
            align-items: center;
            justify-content: space-between;
            box-sizing: border-box;
            gap: 6px;
        }
        .participant-tag-name {
            flex: 1;
            cursor: pointer;
            min-width: 0;
        }
        .participant-tag-icon {
            cursor: pointer;
            display: flex;
            align-items: center;
            justify-content: center;
            width: 16px;
            height: 16px;
            opacity: 0.7;
            transition: opacity 0.2s;
            flex-shrink: 0;
        }
        .participant-tag-icon:hover {
            opacity: 1;
        }
        .participant-tag-icon svg {
            width: 16px;
            height: 16px;
            fill: currentColor;
        }
        .participant-card-popup {
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(0,0,0,0.5);
            display: flex;
            align-items: center;
            justify-content: center;
            z-index: 10000;
        }
        .participant-card-content {
            background: white;
            border-radius: 12px;
            padding: 30px;
            max-width: 400px;
            width: 90%;
            box-shadow: 0 8px 32px rgba(0,0,0,0.3);
            text-align: center;
        }
        .participant-card-image {
            margin-bottom: 20px;
        }
        .participant-card-placeholder {
            width: 240px;
            height: 240px;
            border-radius: 50%;
            background: #f0f0f0;
            display: flex;
            align-items: center;
            justify-content: center;
            margin: 0 auto;
            border: 3px solid #e0e0e0;
        }
        .participant-card-info {
            margin-bottom: 16px;
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 8px;
        }
        @media (max-width: 768px) {
            .participant-card-content {
                padding: 20px;
                max-width: 90%;
            }
        }
        .search-modal {
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(0,0,0,0.5);
            display: flex;
            align-items: center;
            justify-content: center;
            z-index: 10000;
        }
        .search-modal-content {
            background: white;
            border-radius: 12px;
            padding: 24px;
            max-width: 600px;
            width: 90%;
            max-height: 80vh;
            display: flex;
            flex-direction: column;
            box-shadow: 0 8px 32px rgba(0,0,0,0.3);
        }
        .search-modal-header {
            display: flex;
            align-items: center;
            justify-content: space-between;
            margin-bottom: 16px;
        }
        .search-modal-title {
            font-size: 20px;
            font-weight: 600;
            color: #333;
            margin: 0;
        }
        .search-modal-close {
            background: none;
            border: none;
            font-size: 24px;
            cursor: pointer;
            color: #666;
            padding: 0;
            width: 32px;
            height: 32px;
            display: flex;
            align-items: center;
            justify-content: center;
            border-radius: 4px;
            transition: background-color 0.2s;
        }
        .search-modal-close:hover {
            background-color: rgba(0,0,0,0.05);
        }
        .search-input-container {
            margin-bottom: 16px;
        }
        .search-input {
            width: 100%;
            padding: 12px;
            border: 1px solid #ccc;
            border-radius: 8px;
            font-size: 16px;
            box-sizing: border-box;
        }
        .search-input:focus {
            outline: none;
            border-color: #1a73e8;
        }
        .search-results {
            flex: 1;
            overflow-y: auto;
            max-height: calc(80vh - 200px);
        }
        .search-result-item {
            padding: 12px;
            border-bottom: 1px solid #e0e0e0;
            cursor: pointer;
            transition: background-color 0.2s;
        }
        .search-result-item:hover {
            background-color: #f5f5f5;
        }
        .search-result-item:last-child {
            border-bottom: none;
        }
        .search-result-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 4px;
        }
        .search-result-sender {
            font-weight: 600;
            color: #333;
            font-size: 14px;
        }
        .search-result-date {
            color: #666;
            font-size: 12px;
        }
        .search-result-text {
            color: #555;
            font-size: 14px;
            line-height: 1.4;
        }
        .search-highlight {
            background-color: #fff59d;
            font-weight: 600;
            padding: 2px 0;
        }
        @keyframes highlightPulse {
            0%, 100% {
                transform: scale(1);
                box-shadow: 0 0 0 0 rgba(0, 0, 0, 0);
            }
            10% {
                transform: scale(1.05);
                box-shadow: 0 0 0 8px rgba(0, 0, 0, 0.6);
            }
            20% {
                transform: scale(1);
                box-shadow: 0 0 0 0 rgba(0, 0, 0, 0);
            }
            30% {
                transform: scale(1.05);
                box-shadow: 0 0 0 8px rgba(0, 0, 0, 0.6);
            }
            40% {
                transform: scale(1);
                box-shadow: 0 0 0 0 rgba(0, 0, 0, 0);
            }
            50% {
                transform: scale(1.05);
                box-shadow: 0 0 0 8px rgba(0, 0, 0, 0.6);
            }
            60% {
                transform: scale(1);
                box-shadow: 0 0 0 0 rgba(0, 0, 0, 0);
            }
            70% {
                transform: scale(1.05);
                box-shadow: 0 0 0 8px rgba(0, 0, 0, 0.6);
            }
            80% {
                transform: scale(1);
                box-shadow: 0 0 0 0 rgba(0, 0, 0, 0);
            }
            90% {
                transform: scale(1.05);
                box-shadow: 0 0 0 8px rgba(0, 0, 0, 0.6);
            }
        }
        .message-highlight-pulse .message-bubble {
            animation: highlightPulse 4s ease-in-out;
        }
        .search-no-results {
            text-align: center;
            color: #666;
            padding: 40px 20px;
            font-size: 14px;
        }
        @media (max-width: 768px) {
            .search-modal-content {
                max-width: 100%;
                width: 100%;
                height: 100%;
                max-height: 100vh;
                border-radius: 0;
            }
            .search-results {
                max-height: calc(100vh - 200px);
            }
            .search-input {
                font-size: 16px; /* Förhindra zoom på iOS */
            }
        }
        .participant-tag.notified {
            background: #c8e6c9;
            color: #2e7d32;
        }
        .participant-tag.staff.notified {
            background: #ffcdd2;
            color: #c62828;
        }
        .participant-tag.no-pushover {
            opacity: 0.6;
        }
        .participant-tag.no-pushover .participant-tag-name {
            cursor: not-allowed;
        }
        .messages-container {
            flex: 1;
            overflow-y: auto;
            padding: 20px;
            padding-bottom: 80px; /* Extra padding för scroll-knappen */
            padding-top: 180px; /* Sätts dynamiskt via JavaScript */
            background: #f5f5f5;
            position: relative;
        }
        .scroll-to-bottom-btn {
            position: sticky;
            bottom: 5px;
            display: block;
            width: 50px;
            height: 50px;
            background: rgba(229, 57, 53, 0.4); /* röd knapp med 40% genomskinlighet */
            color: white;
            border: medium;
            border-radius: 50%;
            cursor: pointer;
            font-size: 20px;
            box-shadow: rgba(229, 57, 53, 0.3) 0px 4px 12px;
            transition: all 0.3s;
            margin: 10px auto 0;
            z-index: 100;
        }
        .scroll-to-bottom-btn:hover {
            background: rgba(198, 40, 40, 0.6); /* mörkare röd vid hover med högre opacitet */
            transform: scale(1.1);
        }
        .message {
            margin-bottom: 16px;
            display: flex;
            flex-direction: column;
        }
        .message.system {
            text-align: center;
            color: #666;
            font-style: italic;
            font-size: 14px;
            margin-bottom: 4px;
        }
        .message.system + .message.system {
            margin-top: -8px;
        }
        .message.own {
            align-items: flex-end;
        }
        .message:not(.own):not(.system) {
            align-items: flex-start;
        }
        .message-bubble {
            max-width: 70%;
            min-width: 60px;
            padding: 12px 16px;
            border-radius: 18px;
            word-wrap: break-word;
        }
        .message.user .message-bubble,
        .message.admin .message-bubble {
            background: #1a73e8;
            color: white;
        }
        .message.staff .message-bubble,
        .message.extern .message-bubble {
            background: #ffcdd2;
            color: #333;
        }
        .message.urgent .message-bubble {
            background: #ff0000;
            color: #ffffff;
            border: 4px solid #000000;
        }
        .message.own .message-bubble {
            /* Skarp nedre högra hörnet, övriga rundade */
            border-radius: 18px 18px 0 18px;
        }
        .message:not(.own) .message-bubble {
            /* Skarp nedre vänstra hörnet, övriga rundade */
            border-radius: 18px 18px 18px 0;
        }
        .message.system .message-bubble {
            background: transparent;
            color: #666;
            font-style: italic;
        }
        .message-info {
            font-size: 12px;
            color: #666;
            margin-top: 4px;
            padding: 0 4px;
        }
        .message.own .message-info {
            text-align: right;
        }
        .message:not(.own) .message-info {
            text-align: left;
        }
        .message-image {
            max-width: 100%;
            border-radius: 8px;
            margin-top: 8px;
            cursor: pointer;
            transition: opacity 0.2s;
        }
        .message-image:hover {
            opacity: 0.8;
        }
        .message-document {
            display: flex;
            align-items: center;
            gap: 12px;
            padding: 12px;
            background: #f5f5f5;
            border-radius: 8px;
            margin-top: 8px;
            cursor: pointer;
            transition: background 0.2s;
            max-width: 400px;
        }
        .message-document:hover {
            background: #e8e8e8;
        }
        .message-document-icon {
            width: 48px;
            height: 48px;
            flex-shrink: 0;
        }
        .message-document-info {
            flex: 1;
            min-width: 0;
        }
        .message-document-name {
            font-weight: 500;
            color: #333;
            word-break: break-word;
            margin-bottom: 4px;
        }
        .message-document-size {
            font-size: 12px;
            color: #666;
        }
        .date-separator {
            text-align: center;
            margin: 20px 0;
            position: relative;
        }
        .date-separator::before,
        .date-separator::after {
            content: '';
            position: absolute;
            top: 50%;
            width: 40%;
            height: 1px;
            background: #ddd;
        }
        .date-separator::before {
            left: 0;
        }
        .date-separator::after {
            right: 0;
        }
        .date-separator-text {
            display: inline-block;
            padding: 4px 12px;
            background: #ff9800;
            color: white;
            font-size: 12px;
            font-weight: 500;
            border-radius: 12px;
        }
        .message-reply-btn {
            position: absolute;
            top: 4px;
            left: 50%;
            transform: translateX(-50%);
            background: rgba(255, 255, 255, 0.8);
            border: none;
            border-radius: 4px;
            padding: 6px 8px;
            font-size: 11px;
            cursor: pointer;
            opacity: 0;
            transition: opacity 0.2s;
            z-index: 5;
            display: flex;
            align-items: center;
            justify-content: center;
            height: 24px;
            box-sizing: border-box;
        }
        .message:hover .message-reply-btn {
            opacity: 1;
        }
        .message-reply-btn:hover {
            background: rgba(255, 255, 255, 1);
        }
        .message-pin-btn {
            position: absolute;
            top: 4px;
            left: 50%;
            transform: translateX(calc(-50% - 34px));
            background: rgba(255, 255, 255, 0.8);
            border: none;
            border-radius: 4px;
            padding: 6px 6px;
            cursor: pointer;
            opacity: 0;
            transition: opacity 0.2s;
            z-index: 5;
            display: flex;
            align-items: center;
            justify-content: center;
            height: 24px;
            box-sizing: border-box;
        }
        .message:hover .message-pin-btn {
            opacity: 1;
        }
        .message-pin-btn:hover {
            background: rgba(255, 255, 255, 1);
        }
        .message-pin-btn svg {
            width: 14px;
            height: 14px;
            fill: #666;
        }
        .message-pin-btn.pinned svg {
            fill: #ff9800;
        }
        .message-like-btn {
            position: absolute;
            top: 4px;
            left: 50%;
            transform: translateX(calc(-50% + 34px));
            background: rgba(255, 255, 255, 0.8);
            border: none;
            border-radius: 4px;
            padding: 6px 8px;
            height: 24px;
            box-sizing: border-box;
            font-size: 11px;
            cursor: pointer;
            opacity: 0;
            transition: opacity 0.2s;
            z-index: 5;
            display: flex;
            align-items: center;
            justify-content: center;
        }
        .message:hover .message-like-btn {
            opacity: 1;
        }
        .message-like-btn:hover {
            background: rgba(255, 255, 255, 1);
        }
        .message-like-btn.liked {
            background: rgba(255, 255, 255, 0.8); /* Samma bakgrund som normal */
        }
        .message-like-btn.liked:hover {
            background: rgba(255, 255, 255, 1); /* Samma hover som normal */
        }
        .message-like-btn svg {
            width: 16px;
            height: 16px;
            fill: #666;
        }
        .message-like-btn.liked svg {
            fill: #4caf50; /* Grön tumme istället för vit */
        }
        .message.pinned {
            border: 3px solid #ff9800;
            background: #fff3cd;
            padding: 16px 12px 16px 12px;
            margin: 8px 0 14px 0;
            border-radius: 8px;
        }
        .pinned-indicator {
            display: inline;
            color: #ff9800;
            font-size: 12px;
            margin-left: 8px;
        }
        .pinned-indicator svg {
            width: 12px;
            height: 12px;
            fill: #ff9800;
        }
        .pinned-messages-section {
            margin-bottom: 20px;
            border-bottom: 2px solid #ff9800;
            padding-bottom: 16px;
        }
        .pinned-section-header {
            font-size: 13px;
            font-weight: 600;
            color: #ff9800;
            margin-bottom: 12px;
            padding: 8px 12px;
            background: #fff8e1;
            border-radius: 6px;
            display: inline-block;
        }
        .message-bubble {
            position: relative;
        }
        .reply-popup-overlay {
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(0, 0, 0, 0.6);
            z-index: 10000;
            display: flex;
            align-items: center;
            justify-content: center;
            padding: 20px;
            box-sizing: border-box;
            animation: fadeIn 0.2s ease-out;
        }
        @keyframes fadeIn {
            from { opacity: 0; }
            to { opacity: 1; }
        }
        .reply-popup {
            background: white;
            border-radius: 12px;
            max-width: 500px;
            width: 100%;
            max-height: 85vh;
            display: flex;
            flex-direction: column;
            box-shadow: 0 8px 32px rgba(0, 0, 0, 0.4);
            animation: slideUp 0.3s ease-out;
            overflow: hidden;
        }
        @keyframes slideUp {
            from {
                transform: translateY(20px);
                opacity: 0;
            }
            to {
                transform: translateY(0);
                opacity: 1;
            }
        }
        .reply-popup-header {
            padding: 20px 24px;
            border-bottom: 2px solid #e0e0e0;
            background: #f8f9fa;
        }
        .reply-popup-title {
            font-weight: 600;
            font-size: 18px;
            color: #333;
        }
        .reply-popup-close {
            background: #f0f0f0;
            border: none;
            font-size: 20px;
            color: #666;
            cursor: pointer;
            padding: 0;
            width: 32px;
            height: 32px;
            display: flex;
            align-items: center;
            justify-content: center;
            line-height: 1;
            border-radius: 50%;
            transition: all 0.2s;
        }
        .reply-popup-close:hover {
            background: #e0e0e0;
            color: #333;
        }
        .reply-popup-content {
            padding: 24px;
            overflow-y: auto;
            flex: 1;
            background: white;
        }
        .reply-popup-quoted {
            border-left: 4px solid #007bff;
            padding: 14px 16px;
            margin-bottom: 20px;
            background: #f8f9fa;
            border-radius: 6px;
            box-shadow: 0 2px 4px rgba(0, 0, 0, 0.05);
        }
        .reply-popup-quoted-header {
            font-weight: 600;
            color: #007bff;
            margin-bottom: 8px;
            font-size: 14px;
        }
        .reply-popup-quoted-content {
            color: #555;
            font-size: 15px;
            word-wrap: break-word;
            line-height: 1.5;
        }
        .reply-popup-image-thumbnail {
            display: block;
            max-width: 200px;
            max-height: 150px;
            border-radius: 6px;
            margin-top: 8px;
            object-fit: cover;
            cursor: pointer;
        }
        .reply-popup-image-thumbnail:hover {
            opacity: 0.8;
        }
        .reply-popup-input {
            width: 100%;
            padding: 14px;
            border: 2px solid #ddd;
            border-radius: 6px;
            font-size: 15px;
            font-family: inherit;
            resize: vertical;
            min-height: 100px;
            box-sizing: border-box;
            transition: border-color 0.2s;
        }
        .reply-popup-input:focus {
            outline: none;
            border-color: #007bff;
            box-shadow: 0 0 0 3px rgba(0, 123, 255, 0.1);
        }
        .reply-popup-footer {
            padding: 16px 24px;
            border-top: 2px solid #e0e0e0;
            display: flex;
            justify-content: flex-end;
            gap: 12px;
            background: #f8f9fa;
        }
        .reply-popup-btn {
            padding: 10px 20px;
            border: none;
            border-radius: 6px;
            font-size: 15px;
            cursor: pointer;
            font-weight: 500;
            transition: all 0.2s;
            min-width: 100px;
        }
        .reply-popup-btn-cancel:hover {
            background: #f5f5f5;
            border-color: #bbb;
        }
        .reply-popup-btn-send {
            background: #1a73e8;
            color: white;
            border: none;
            border-radius: 8px;
            width: 40px;
            height: 40px;
            cursor: pointer;
            display: flex;
            align-items: center;
            justify-content: center;
            padding: 0;
            transition: background-color 0.2s;
            min-width: 40px;
        }
        .reply-popup-btn-send:hover {
            background: #1557b0;
        }
        .reply-popup-btn-send:disabled {
            background: #ccc;
            cursor: not-allowed;
        }
        .reply-popup-btn-send svg {
            width: 20px;
            height: 20px;
            fill: white;
        }
        .reply-popup-btn-cancel {
            background: white;
            color: #666;
            border: 2px solid #ddd;
            width: 40px;
            height: 40px;
            padding: 0;
            border-radius: 8px;
            display: flex;
            align-items: center;
            justify-content: center;
            min-width: 40px;
        }
        .reply-popup-btn-cancel svg {
            width: 20px;
            height: 20px;
            fill: #666;
        }
        .upload-popup-input-label {
            font-weight: 600;
            font-size: 14px;
            color: #333;
            margin-bottom: 8px;
            display: block;
        }
        .upload-popup-filename-input {
            width: 100%;
            padding: 12px;
            border: 2px solid #ddd;
            border-radius: 6px;
            font-size: 15px;
            font-family: inherit;
            box-sizing: border-box;
            transition: border-color 0.2s;
            margin-bottom: 20px;
        }
        .upload-popup-filename-input:focus {
            outline: none;
            border-color: #007bff;
            box-shadow: 0 0 0 3px rgba(0, 123, 255, 0.1);
        }
        .quoted-message {
            border-left: 3px solid #007bff;
            padding: 8px 12px;
            margin-bottom: 8px;
            background: #f0f0f0;
            border-radius: 4px;
            font-size: 13px;
        }
        .quoted-message-header {
            font-weight: 600;
            color: #007bff;
            margin-bottom: 4px;
            font-size: 12px;
        }
        .quoted-message-content {
            color: #666;
        }
        .quoted-message-image-thumbnail {
            display: block;
            max-width: 150px;
            max-height: 100px;
            border-radius: 4px;
            margin-top: 6px;
            object-fit: cover;
            cursor: pointer;
        }
        .quoted-message-image-thumbnail:hover {
            opacity: 0.8;
        }
        .message-viewed-badge {
            position: absolute;
            top: -6px;
            right: -6px;
            background: #ff6b6b;
            color: white;
            border-radius: 10px;
            min-width: 18px;
            height: 18px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 11px;
            font-weight: bold;
            padding: 0 5px;
            box-shadow: 0 2px 4px rgba(0,0,0,0.2);
            z-index: 10;
            cursor: pointer;
            transition: background 0.2s;
        }
        .message-viewed-badge:hover {
            background: #ff5252;
        }
        .message.own .message-viewed-badge {
            right: auto;
            left: -6px;
        }
        .message-likes-count {
            color: #4caf50;
            font-size: 12px;
            font-weight: 500;
            margin-left: 8px;
            cursor: pointer;
            transition: color 0.2s;
        }
        .message-likes-count:hover {
            color: #388e3c;
            text-decoration: underline;
        }
        .input-container {
            background: white;
            padding: 2px 20px 8px 20px;
            position: relative;
            z-index: 1002;
        }
        .input-content {
            max-width: 1200px;
            margin: 0 auto;
        }
        .message-input-area {
            display: flex;
            flex-direction: row;
            gap: 8px;
            align-items: flex-start;
        }
        .message-input {
            flex: 1 1 0;
            width: 0;
        }
        .message-buttons {
            display: flex;
            flex-direction: column;
            gap: 8px;
        }
        .icon-button {
            background: #1a73e8;
            color: white;
            border: none;
            border-radius: 8px;
            width: 40px;
            height: 40px;
            cursor: pointer;
            display: flex;
            align-items: center;
            justify-content: center;
            transition: background-color 0.2s;
            padding: 0;
        }
        .icon-button:hover {
            background: #1557b0;
        }
        .icon-button.secondary {
            background: #6c757d;
        }
        .icon-button.secondary:hover {
            background: #5a6268;
        }
        .icon-button svg {
            width: 20px;
            height: 20px;
            fill: currentColor;
        }
        .urgent-icon {
            display: inline-flex;
            align-items: center;
            cursor: pointer;
            padding: 2px;
            border-radius: 4px;
            transition: all 0.2s;
            height: 24px;
            width: 24px;
            box-sizing: border-box;
            margin-left: -4px;
            justify-content: center;
            background: transparent;
        }
        .urgent-icon:hover {
            background-color: transparent;
        }
        .urgent-icon svg {
            width: 16px;
            height: 16px;
            color: #bdbdbd; /* Ljusgrå som standard */
            transition: color 0.2s;
        }
        .urgent-icon.active svg {
            color: #f44336; /* Röd när aktiv */
        }
        .help-icon {
            display: inline-flex;
            align-items: center;
            cursor: pointer;
            padding: 2px;
            border-radius: 4px;
            transition: all 0.2s;
            height: 24px;
            width: 24px;
            box-sizing: border-box;
            margin-left: -4px;
            justify-content: center;
            background: transparent;
        }
        .help-icon:hover {
            background-color: rgba(0, 0, 0, 0.05);
        }
        .help-icon svg {
            width: 16px;
            height: 16px;
        }
        .help-icon svg circle {
            fill: #e3f2fd !important;
        }
        .help-icon svg path {
            fill: #424242 !important;
            stroke: none;
        }
        .pinned-filter-icon {
            display: inline-flex;
            align-items: center;
            cursor: pointer;
            padding: 2px;
            border-radius: 4px;
            transition: all 0.2s;
            height: 24px;
            width: 24px;
            box-sizing: border-box;
            margin-left: -4px;
            justify-content: center;
            background: transparent;
        }
        .pinned-filter-icon:hover {
            background-color: rgba(0, 0, 0, 0.05);
        }
        .pinned-filter-icon svg {
            width: 16px;
            height: 16px;
        }
        .pinned-filter-icon.active svg {
            fill: #ff9800;
        }
        .pinned-filter-icon:not(.active) svg {
            fill: #666;
        }
        .search-icon {
            display: inline-flex;
            align-items: center;
            cursor: pointer;
            padding: 2px;
            border-radius: 4px;
            transition: all 0.2s;
            height: 24px;
            width: 24px;
            box-sizing: border-box;
            margin-left: -4px;
            justify-content: center;
            background: transparent;
        }
        .search-icon:hover {
            background-color: rgba(0, 0, 0, 0.05);
        }
        .search-icon svg {
            width: 16px;
            height: 16px;
            fill: #666;
        }
        .manage-participants-icon {
            display: inline-flex;
            align-items: center;
            cursor: pointer;
            padding: 4px 8px;
            border-radius: 4px;
            transition: all 0.2s;
            height: 24px;
            box-sizing: border-box;
            margin-right: 4px;
            justify-content: center;
            background: #ffebee;
        }
        .manage-participants-icon:hover {
            background-color: #ffcdd2;
        }
        .manage-participants-icon svg {
            height: 16px;
            fill: #666;
        }
        .manage-participants-icon:hover svg {
            fill: #1976d2;
        }
        .notifications-toggle {
            display: inline-flex;
            align-items: center;
            cursor: pointer;
            padding: 2px;
            border-radius: 4px;
            transition: all 0.2s;
            height: 24px;
            width: 24px;
            box-sizing: border-box;
            margin-left: 8px;
            margin-right: -4px;
            justify-content: center;
        }
        .notifications-toggle:hover {
            background-color: transparent;
        }
        .notifications-toggle svg {
            width: 16px;
            height: 16px;
            color: #1976d2;
            transition: color 0.2s;
        }
        .notifications-toggle.active svg {
            color: #2e7d32; /* Grön när alla notifieringar är på */
        }
        .message-input {
            flex: 1;
            padding: 12px;
            border: 1px solid #ccc;
            border-radius: 8px;
            font-size: 16px;
            resize: none;
            min-height: 88px;
            height: 88px;
            max-height: 150px;
            font-family: inherit;
            margin: 0;
            box-sizing: border-box;
            min-width: 0;
        }
        .btn {
            padding: 12px 24px;
            border: none;
            border-radius: 8px;
            cursor: pointer;
            font-size: 16px;
            font-weight: 500;
        }
        .btn-primary {
            background: #1a73e8;
            color: white;
        }
        .btn-primary:hover {
            background: #1557b0;
        }
        .btn-primary:disabled {
            background: #ccc;
            cursor: not-allowed;
        }
        .btn-secondary {
            background: #6c757d;
            color: white;
        }
        .btn-secondary:hover {
            background: #5a6268;
        }
        .file-input-wrapper {
            position: relative;
            display: inline-block;
        }
        .file-input-wrapper input[type="file"] {
            position: absolute;
            opacity: 0;
            width: 0;
            height: 0;
        }
        .file-preview-overlay {
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0; /* Sätts dynamiskt via JavaScript baserat på input-container höjd */
            background: rgba(0, 0, 0, 0.5);
            z-index: 1000;
            display: none;
        }
        .file-preview-container {
            position: fixed;
            bottom: 140px; /* Positionera nära meddelanderutan, sätts dynamiskt via JavaScript */
            left: 50%;
            transform: translateX(-50%);
            z-index: 1003;
            padding: 24px;
            background: white;
            border-radius: 12px;
            border: 1px solid #e0e0e0;
            box-shadow: 0 4px 20px rgba(0, 0, 0, 0.3);
            min-width: 400px;
            max-width: 90vw;
            display: none;
            pointer-events: auto;
        }
        .file-preview-info {
            display: flex;
            align-items: center;
            justify-content: space-between;
            margin-bottom: 8px;
        }
        .file-preview-filename-section {
            flex: 1;
            display: flex;
            flex-direction: column;
            gap: 4px;
        }
        .file-preview-label {
            font-size: 14px;
            color: #333;
            font-weight: 600;
            margin-bottom: 8px;
        }
        .file-preview-filename-input {
            font-size: 16px;
            color: #333;
            font-weight: 500;
            border: 2px solid #ddd;
            border-radius: 6px;
            padding: 10px 12px;
            background: white;
            word-break: break-word;
            width: 100%;
            box-sizing: border-box;
        }
        .file-preview-filename-input:focus {
            outline: none;
            border-color: #1976d2;
            box-shadow: 0 0 0 2px rgba(25, 118, 210, 0.2);
        }
        .file-preview-cancel {
            background: none;
            border: none;
            font-size: 24px;
            color: #666;
            cursor: pointer;
            padding: 0;
            margin-left: 12px;
            line-height: 1;
            width: 24px;
            height: 24px;
            display: flex;
            align-items: center;
            justify-content: center;
            border-radius: 50%;
            transition: background 0.2s;
        }
        .file-preview-cancel:hover {
            background: #e0e0e0;
            color: #333;
        }
        .file-preview-hint {
            font-size: 13px;
            color: #666;
            font-style: italic;
            margin-top: 4px;
        }
        .message-image-filename {
            margin-top: 6px;
            font-size: 13px;
            color: white;
            font-weight: 500;
            font-style: italic;
        }
        .message-image-text,
        .message-document-text {
            margin-top: 8px;
            padding: 8px 0;
            color: white;
            font-size: 14px;
            line-height: 1.5;
            word-wrap: break-word;
        }
        .modal {
            display: none;
            position: fixed;
            z-index: 1003;
            left: 0;
            top: 0;
            width: 100%;
            height: 100%;
            background-color: rgba(0,0,0,0.5);
            align-items: center;
            justify-content: center;
        }
        .modal-content {
            background: white;
            border-radius: 12px;
            padding: 30px;
            padding-bottom: 120px;
            max-width: 600px;
            width: 90%;
            max-height: 90vh;
            overflow-y: auto;
        }
        /* Systemmeddelande modal */
        .system-message-overlay {
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(0, 0, 0, 0.7);
            display: none;
            align-items: center;
            justify-content: center;
            z-index: 10000;
        }
        .system-message-modal {
            background: white;
            border-radius: 12px;
            padding: 32px;
            max-width: 600px;
            width: 90%;
            max-height: 80vh;
            overflow-y: auto;
            box-shadow: 0 8px 32px rgba(0, 0, 0, 0.3);
        }
        .system-message-title {
            font-size: 24px;
            font-weight: 600;
            color: #1a73e8;
            margin: 0 0 20px 0;
        }
        .system-message-content {
            font-size: 16px;
            line-height: 1.6;
            color: #333;
            white-space: pre-wrap;
            word-wrap: break-word;
            margin-bottom: 24px;
            padding: 16px;
            background: #f8f9fa;
            border-radius: 8px;
            border: 1px solid #dee2e6;
        }
        .system-message-actions {
            display: flex;
            justify-content: flex-end;
        }
        .system-message-btn {
            padding: 12px 24px;
            border-radius: 8px;
            border: 0;
            font-size: 16px;
            font-weight: 500;
            cursor: pointer;
            background: #1a73e8;
            color: white;
            transition: all 0.2s;
        }
        .system-message-btn:hover {
            background: #1557b0;
        }
        @media (max-width: 768px) {
            .message-bubble {
                max-width: 85%;
            }
        }
    </style>
</head>
<body>
    <div class="logo-header">
        <div class="logo-section">
            <a href="../index.html<?= !empty($participantId) ? '?user_id=' . htmlspecialchars($participantId) : '' ?>">
                <img src="../data/logo_chat.png" alt="ByggArkiv Snickare Stefan" class="logo">
            </a>
        </div>
    </div>
    <div class="header">
        <div class="header-content">
            <div style="display: flex; flex-direction: column;">
                <div style="font-size: 12px; color: #333; margin-bottom: 2px; font-weight: 500;">Projektchatt:</div>
                <?php if (count($userProjects) > 1): ?>
                    <select id="projectSelect" style="font-size: 18px; font-weight: 600; font-family: system-ui, -apple-system, Segoe UI, Roboto, sans-serif !important; border: none; background: transparent; cursor: pointer; color: #333; padding: 4px 8px; border-radius: 4px; margin: 0; -webkit-appearance: none; -moz-appearance: none; appearance: none;">
                        <?php foreach ($userProjects as $proj): ?>
                            <option value="<?= htmlspecialchars($proj['id']) ?>" <?= ($proj['id'] === $projectId) ? 'selected' : '' ?>>
                                <?= htmlspecialchars($proj['name']) ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                <?php else: ?>
                    <h1 style="margin: 0;"><?= htmlspecialchars($project['name'] ?? '') ?></h1>
                <?php endif; ?>
            </div>
            <?php
            // Bestäm bas-sökväg dynamiskt från nuvarande sökväg
            // Om script ligger i /[mapp]/chat/chat.php, få /[mapp]/
            $scriptPath = $_SERVER['SCRIPT_NAME'] ?? '';
            $basePath = '/';
            if (!empty($scriptPath)) {
                // dirname två gånger för att gå från /[mapp]/chat/chat.php till /[mapp]/
                $baseDir = dirname(dirname($scriptPath));
                if ($baseDir !== '/' && $baseDir !== '\\') {
                    $basePath = rtrim($baseDir, '/') . '/';
                }
            }
            
            // Bestäm länk baserat på användartyp
            if ($isAdmin) {
                // För admin: gå till admin-sidor
                $backUrl = $basePath . 'adm/admin.php' . (!empty($participantId) ? '?user_id=' . htmlspecialchars($participantId) : '');
                $backTitle = 'Tillbaka till admin';
            } else {
                // För extern eller user: gå till bilduppladdning
                $backUrl = $basePath . 'index.html' . (!empty($participantId) ? '?user_id=' . htmlspecialchars($participantId) : '');
                $backTitle = 'Tillbaka till bilduppladdning';
            }
            ?>
            <div style="display: flex; align-items: center; gap: 12px;">
                <a href="<?= $backUrl ?>" class="back-link" title="<?= htmlspecialchars($backTitle) ?>" style="display: inline-flex; align-items: center; justify-content: center; width: 24px; height: 24px; text-decoration: none;">
                    <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                        <path d="M9 21H5a2 2 0 0 1-2-2V5a2 2 0 0 1 2-2h4"></path>
                        <polyline points="16 17 21 12 16 7"></polyline>
                        <line x1="21" y1="12" x2="9" y2="12"></line>
                    </svg>
                </a>
            </div>
        </div>
    </div>
    
    <div class="chat-container">
        <div class="messages-container" id="messagesContainer">
            <p>Laddar meddelanden...</p>
        </div>
        
        <?php if (!$isArchived): ?>
        <div class="participants-section">
            <div class="participants-list" id="participantsList">
                <p>Laddar...</p>
            </div>
        </div>
        
        <div class="input-container">
            <div class="input-content">
                <div class="message-input-area">
                    <textarea 
                        id="messageInput" 
                        class="message-input" 
                        placeholder="Skriv ett meddelande..."
                        rows="1"></textarea>
                    <div class="message-buttons">
                        <div class="file-input-wrapper">
                            <input type="file" id="imageInput" accept="image/*,application/pdf" style="display: none;">
                            <button type="button" class="icon-button secondary" onclick="document.getElementById('imageInput').click()" title="Ladda upp foto eller PDF">
                                <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24">
                                    <path d="M21 19V5c0-1.1-.9-2-2-2H5c-1.1 0-2 .9-2 2v14c0 1.1.9 2 2 2h14c1.1 0 2-.9 2-2zM8.5 13.5l2.5 3.01L14.5 12l4.5 6H5l3.5-4.5z"/>
                                </svg>
                            </button>
                        </div>
                        <button type="button" class="icon-button" id="sendButton" onclick="sendMessage()" title="Skicka meddelande">
                            <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24">
                                <path d="M2.01 21L23 12 2.01 3 2 10l15 2-15 2z"/>
                            </svg>
                        </button>
                    </div>
                </div>
        <!-- Modal overlay för filförhandsvisning -->
        <div id="filePreviewOverlay" class="file-preview-overlay" onclick="cancelFileUpload()"></div>
        
        <!-- Filförhandsvisning (dold som standard) -->
        <div id="filePreviewContainer" class="file-preview-container" style="display: none;">
            <div class="file-preview-info">
                <div class="file-preview-filename-section">
                    <label for="fileDisplayName" class="file-preview-label">Filnamn:</label>
                    <input type="text" id="fileDisplayName" class="file-preview-filename-input" placeholder="Filnamn">
                </div>
                <button type="button" class="file-preview-cancel" onclick="cancelFileUpload()" title="Avbryt">×</button>
            </div>
            <div class="file-preview-hint">Skriv en förklarande text i meddelandefältet ovan (valfritt)</div>
        </div>
            </div>
        </div>
        <?php else: ?>
        <div style="padding: 20px; text-align: center; color: #666; background: #f8f9fa; border-top: 1px solid #e0e0e0;">
            <p style="margin: 0;">Detta projekt är arkiverat. Du kan läsa meddelanden men inte skriva nya.</p>
        </div>
        <?php endif; ?>
    </div>
    
    <script>
        const projectId = '<?= htmlspecialchars($projectId) ?>';
        const isArchived = <?= $isArchived ? 'true' : 'false' ?>;
        const isUserType = <?= ($userType === 'user' && !$isAdmin) ? 'true' : 'false' ?>;
        const isExternType = <?= ($userType === 'extern') ? 'true' : 'false' ?>;
        // Hämta user_id dynamiskt från URL (fungerar för både users och staff)
        function getParticipantIdFromUrl() {
            try {
                const search = window.location.search || '';
                const urlParams = new URLSearchParams(search);
                return urlParams.get('user_id') || '';
            } catch (e) {
                console.error('Fel vid parsing av URL:', e);
                return '';
            }
        }
        let participantId = getParticipantIdFromUrl();
        const isAdmin = <?= $isAdmin ? 'true' : 'false' ?>;
        // Säkerställ att hantera-ikonen bara visas för admin även på klientsidan
        const manageIcon = document.getElementById('manageParticipantsIcon');
        if (manageIcon && !isAdmin) {
            manageIcon.style.display = 'none';
        }
        
        // Spåra vilka meddelanden som redan markerats som "sett" för att förhindra dubbletter
        const viewedMessages = new Set();
        
        // Intersection Observer för att spåra när meddelanden blir synliga
        let messageObserver = null;
        
        // Funktion för att markera ett meddelande som "sett"
        async function markMessageAsViewed(messageId) {
            // Förhindra dubbletter - om meddelandet redan markerats, hoppa över
            if (viewedMessages.has(messageId)) {
                return;
            }
            
            // Kontrollera att vi har nödvändig data
            if (!messageId || !participantId || !projectId) {
                return;
            }
            
            try {
                const formData = new FormData();
                formData.append('action', 'mark_viewed');
                formData.append('project_id', projectId);
                formData.append('user_id', participantId);
                formData.append('message_id', messageId);
                
                const response = await fetch('../api/chat/messages.php', {
                    method: 'POST',
                    body: formData
                });
                
                if (!response.ok) {
                    throw new Error(`HTTP ${response.status}: ${response.statusText}`);
                }
                
                const data = await response.json();
                
                if (data.success) {
                    // Markera meddelandet som "sett" lokalt för att förhindra dubbletter
                    viewedMessages.add(messageId);
                    
                    // Uppdatera meddelandet i messages array
                    const messageIndex = messages.findIndex(msg => msg.id === messageId);
                    if (messageIndex !== -1) {
                        messages[messageIndex].viewed_by = data.viewed_by || [];
                        // Uppdatera visuell indikator (optimistic update)
                        // Räkna exkludera avsändaren
                        const senderId = messages[messageIndex].participant_id || '';
                        const viewedByOthers = (data.viewed_by || []).filter(userId => userId !== senderId);
                        updateViewedBadge(messageId, viewedByOthers.length);
                    }
                }
            } catch (error) {
                console.error('Fel vid markering av meddelande som "sett":', error);
            }
        }
        
        // Funktion för att uppdatera visuell indikator för ett meddelande
        function updateViewedBadge(messageId, viewedCount) {
            // Escape messageId för CSS-selector (fallback om CSS.escape inte finns)
            const escapedMessageId = (typeof CSS !== 'undefined' && CSS.escape) ? CSS.escape(messageId) : messageId.replace(/[!"#$%&'()*+,.\/:;<=>?@[\\\]^`{|}~]/g, '\\$&');
            const messageElement = document.querySelector(`[data-message-id="${escapedMessageId}"]`);
            if (!messageElement) {
                return;
            }
            
            const badge = messageElement.querySelector('.message-viewed-badge');
            if (badge) {
                if (viewedCount > 0) {
                    badge.textContent = viewedCount;
                    badge.style.display = 'flex';
                } else {
                    // Dölj badge om antalet är 0
                    badge.style.display = 'none';
                }
            }
        }
        
        // Initiera Intersection Observer för att spåra när meddelanden blir synliga
        function initMessageViewTracking() {
            // Stäng tidigare observer om den finns
            if (messageObserver) {
                messageObserver.disconnect();
            }
            
            // Kontrollera om Intersection Observer stöds
            if (!('IntersectionObserver' in window)) {
                console.warn('Intersection Observer stöds inte i denna webbläsare');
                return;
            }
            
            const container = document.getElementById('messagesContainer');
            if (!container) {
                return;
            }
            
            // Skapa observer med threshold på 0.5 (50% synligt)
            messageObserver = new IntersectionObserver((entries) => {
                entries.forEach(entry => {
                    if (entry.isIntersecting && entry.intersectionRatio >= 0.5) {
                        const messageElement = entry.target;
                        const messageId = messageElement.getAttribute('data-message-id');
                        
                        if (messageId && !viewedMessages.has(messageId)) {
                            // Vänta 500ms innan vi markerar meddelandet som "sett"
                            // Detta förhindrar att snabba scrollningar räknas
                            const timeoutId = setTimeout(() => {
                                // Kontrollera att elementet fortfarande är synligt
                                // Genom att kontrollera om det fortfarande finns i observern
                                if (messageElement && messageElement.isConnected) {
                                    markMessageAsViewed(messageId);
                                }
                            }, 500);
                            
                            // Spara timeout ID på elementet för att kunna avbryta om användaren scrollar bort
                            messageElement._viewTimeoutId = timeoutId;
                        }
                    } else if (!entry.isIntersecting) {
                        // Om elementet inte längre är synligt, avbryt timeout om den finns
                        const messageElement = entry.target;
                        if (messageElement && messageElement._viewTimeoutId) {
                            clearTimeout(messageElement._viewTimeoutId);
                            delete messageElement._viewTimeoutId;
                        }
                    }
                });
            }, {
                threshold: 0.5, // Minst 50% av meddelandet måste vara synligt
                rootMargin: '0px'
            });
            
            // Observera alla meddelanden
            const messageElements = container.querySelectorAll('[data-message-id]');
            messageElements.forEach(element => {
                // Observera endast icke-systemmeddelanden
                if (!element.classList.contains('system')) {
                    messageObserver.observe(element);
                }
            });
        }
        // Lista över alla admin user_ids
        const adminUserIds = <?= json_encode(getAllAdminUserIds()) ?>;
        
        // Funktion för att kontrollera om en participant är admin
        function isParticipantAdmin(participantId) {
            return adminUserIds.includes(participantId);
        }
        
        let participants = [];
        let messages = [];
        let notifiedParticipants = new Set(); // Håller koll på vilka deltagare som ska notifieras
        
        // State persistence variables
        let chatStateLoaded = false;
        const STATE_EXPIRY_TIME = 60 * 60 * 1000; // 1 hour
        window.urgentNotificationActive = false; // Håller koll på om brådskande notifikation är aktiv
        window.pinnedFilterActive = false; // Håller koll på om filtret för fästa meddelanden är aktivt
        window.notificationsAllEnabled = true; // Håller koll på om notifieringar är på för alla
        
        // ========== AKTIVITETSSPÅRNING FÖR NOTIFIERINGAR ==========
        let lastActivity = Date.now();
        let pingInterval = null;
        const ACTIVITY_TIMEOUT = 30000; // 30 sekunder
        const PING_INTERVAL = 15000; // 15 sekunder
        
        // Spåra aktivitet
        function updateActivity() {
            lastActivity = Date.now();
        }
        
        // Throttle-funktion för att begränsa antal anrop
        function throttle(func, limit) {
            let inThrottle;
            return function() {
                const args = arguments;
                const context = this;
                if (!inThrottle) {
                    func.apply(context, args);
                    inThrottle = true;
                    setTimeout(() => inThrottle = false, limit);
                }
            };
        }
        
        // Event listeners för aktivitet (med throttling för performance)
        const throttledUpdateActivity = throttle(updateActivity, 1000); // Max 1 gång per sekund
        ['keydown', 'mousemove', 'scroll', 'click'].forEach(eventType => {
            document.addEventListener(eventType, throttledUpdateActivity, { passive: true });
        });
        
        // Page Visibility API
        document.addEventListener('visibilitychange', function() {
            if (document.visibilityState === 'visible') {
                startPing();
            } else {
                stopPing();
            }
        });
        
        // Ping-funktioner
        function shouldSendPing() {
            const isVisible = document.visibilityState === 'visible';
            const isActive = (Date.now() - lastActivity) < ACTIVITY_TIMEOUT;
            return isVisible && isActive;
        }
        
        async function sendPing() {
            if (!shouldSendPing()) {
                stopPing();
                return;
            }
            
            if (!projectId || !participantId) {
                return;
            }
            
            try {
                const url = `../api/chat/activity.php?action=ping&project_id=${encodeURIComponent(projectId)}&user_id=${encodeURIComponent(participantId)}`;
                const response = await fetch(url);
                if (!response.ok) {
                    console.error('Ping failed:', response.status);
                }
            } catch (error) {
                console.error('Ping failed:', error);
            }
        }
        
        function startPing() {
            if (pingInterval) return;
            if (!shouldSendPing()) return;
            
            pingInterval = setInterval(sendPing, PING_INTERVAL);
            sendPing(); // Skicka direkt
        }
        
        function stopPing() {
            if (pingInterval) {
                clearInterval(pingInterval);
                pingInterval = null;
            }
        }
        
        // Starta ping när sidan laddas (om fliken är synlig)
        // OBS: Ping startas nu efter att participants har laddats framgångsrikt
        // (se loadParticipants() för när ping faktiskt startas)
        
        // Stoppa ping när sidan stängs
        window.addEventListener('beforeunload', stopPing);
        // ========== SLUT AKTIVITETSSPÅRNING ==========
        
        <?php if (!$isArchived): ?>
        // Auto-resize textarea
        const messageInput = document.getElementById('messageInput');
        messageInput.addEventListener('input', function() {
            this.style.height = 'auto';
            this.style.height = Math.max(88, Math.min(this.scrollHeight, 150)) + 'px';
        });
        
        // Scrolla till botten när meddelandefältet får fokus
        messageInput.addEventListener('focus', function() {
            const container = document.getElementById('messagesContainer');
            if (container) {
                setTimeout(() => {
                    scrollToBottomInstant();
                }, 100);
            }
        });
        
        // Enter för att skicka (Shift+Enter för ny rad)
        messageInput.addEventListener('keydown', function(e) {
            if (e.key === 'Enter' && !e.shiftKey) {
                e.preventDefault();
                sendMessage();
            }
        });
        <?php endif; ?>
        
        async function loadParticipants() {
            try {
                // Uppdatera participantId från URL innan användning
                participantId = getParticipantIdFromUrl();
                if (!projectId) {
                    console.error('projectId är tom');
                    throw new Error('Projekt-ID saknas');
                }
                let url = `../api/chat/participants.php?action=list&project_id=${encodeURIComponent(projectId)}`;
                if (participantId) {
                    url += `&user_id=${encodeURIComponent(participantId)}`;
                }
                // Om användaren är admin enligt PHP-variabeln, skicka alltid med admin-flagga
                // Detta säkerställer att admin-rättigheter fungerar även om user_id inte matchar korrekt
                if (isAdmin) {
                    url += `&is_admin=1`;
                }
                const res = await fetch(url);
                const text = await res.text();
                let data;
                
                // Om 403 och användaren är user-typ (inte admin eller extern), försök lägga till dem automatiskt
                if (!res.ok && res.status === 403 && isUserType && participantId) {
                    try {
                        data = JSON.parse(text);
                    } catch (e) {
                        data = {};
                    }
                    
                    // Kontrollera om det är pga inaktiv användare
                    if (data.message && data.message.includes('inaktiv')) {
                        alert(data.message || 'Din användare är inaktiv och du har loggats ut');
                        window.location.href = '../index.html';
                        return;
                    }
                    
                    // Om det är pga saknad behörighet, försök lägga till användaren automatiskt
                    if (data.message && data.message.includes('behörighet')) {
                        try {
                            const addFormData = new FormData();
                            addFormData.append('action', 'add');
                            addFormData.append('project_id', projectId);
                            addFormData.append('type', 'user');
                            addFormData.append('participant_id', participantId);
                            addFormData.append('user_id', participantId);
                            
                            const addRes = await fetch('../api/chat/participants.php', {
                                method: 'POST',
                                body: addFormData
                            });
                            
                            const addText = await addRes.text();
                            let addData;
                            try {
                                addData = JSON.parse(addText);
                            } catch (parseError) {
                                console.error('Kunde inte parsa JSON-svar vid läggning av deltagare:', addText);
                                addData = { success: false, message: 'Ogiltigt svar från servern' };
                            }
                            
                            if (addData.success) {
                                // Vänta lite för att säkerställa att ändringen är sparad
                                await new Promise(resolve => setTimeout(resolve, 100));
                                // Ladda om deltagare efter att användaren lagts till
                                await loadParticipants();
                                return; // Returnera tidigt eftersom loadParticipants() redan anropas
                            }
                        } catch (addError) {
                            console.error('Kunde inte lägga till användare som deltagare:', addError);
                        }
                    }
                    
                    // Om vi inte kunde lägga till användaren, visa felmeddelande
                    alert(data.message || 'Du saknar behörighet att se deltagare för detta projekt');
                    return;
                }
                
                if (!res.ok) {
                    // Om användaren är inaktiv (403), redirecta
                    if (res.status === 403) {
                        try {
                            data = JSON.parse(text);
                        } catch (e) {
                            data = { message: 'Din användare är inaktiv' };
                        }
                        alert(data.message || 'Din användare är inaktiv och du har loggats ut');
                        window.location.href = '../index.html';
                        return;
                    }
                    throw new Error(`HTTP ${res.status}: ${res.statusText}`);
                }
                
                try {
                    data = JSON.parse(text);
                } catch (parseError) {
                    console.error('Ogiltigt JSON-svar från API:', text);
                    throw new Error('Ogiltigt svar från servern: ' + parseError.message);
                }
                
                // Kontrollera om användaren är inaktiv
                if (!data.success && data.message && data.message.includes('inaktiv')) {
                    alert(data.message);
                    window.location.href = '../index.html';
                    return;
                }
                
                if (data.success) {
                    participants = data.participants || [];
                    const chatters = data.chatters || [];
                    
                    // Om användaren är user (inte admin eller extern) och inte redan är deltagare,
                    // lägg till dem automatiskt som deltagare
                    if (isUserType && participantId) {
                        const isParticipant = participants.some(p => p.participant_id === participantId);
                        if (!isParticipant) {
                            // Lägg till användaren som deltagare automatiskt via direkt API-anrop
                            try {
                                const addFormData = new FormData();
                                addFormData.append('action', 'add');
                                addFormData.append('project_id', projectId);
                                addFormData.append('type', 'user');
                                addFormData.append('participant_id', participantId);
                                addFormData.append('user_id', participantId);
                                
                                const addRes = await fetch('../api/chat/participants.php', {
                                    method: 'POST',
                                    body: addFormData
                                });
                                
                                const addText = await addRes.text();
                                let addData;
                                try {
                                    addData = JSON.parse(addText);
                                } catch (parseError) {
                                    console.error('Kunde inte parsa JSON-svar vid läggning av deltagare:', addText);
                                    addData = { success: false, message: 'Ogiltigt svar från servern' };
                                }
                                
                                if (addData.success) {
                                    // Ladda om deltagare efter att användaren lagts till
                                    await loadParticipants();
                                    // Starta ping efter att användaren lagts till och deltagare har laddats
                                    if (document.visibilityState === 'visible') {
                                        startPing();
                                    }
                                    return; // Returnera tidigt eftersom loadParticipants() redan anropas
                                } else {
                                    console.error('Kunde inte lägga till användare som deltagare:', addData.message);
                                    // Fortsätt ändå med att visa deltagare
                                }
                            } catch (e) {
                                console.error('Kunde inte lägga till användare som deltagare:', e);
                                // Fortsätt ändå med att visa deltagare
                            }
                        }
                    }
                    
                    // Om användaren är admin och inte redan är deltagare, lägg till dem automatiskt
                    if (isAdmin && participantId) {
                        const isParticipant = participants.some(p => p.participant_id === participantId);
                        if (!isParticipant) {
                            // Lägg till adminen som deltagare automatiskt via direkt API-anrop
                            // Lägg till i både participants och chatters
                            try {
                                // Lägg till i participants
                                const addFormData = new FormData();
                                addFormData.append('action', 'add');
                                addFormData.append('project_id', projectId);
                                addFormData.append('type', 'user'); // Admin använder 'user' som deltagartyp
                                addFormData.append('participant_id', participantId);
                                addFormData.append('user_id', participantId);
                                addFormData.append('list_type', 'participants');
                                
                                const addRes = await fetch('../api/chat/participants.php', {
                                    method: 'POST',
                                    body: addFormData
                                });
                                
                                const addText = await addRes.text();
                                let addData;
                                try {
                                    addData = JSON.parse(addText);
                                } catch (parseError) {
                                    console.error('Kunde inte parsa JSON-svar vid läggning av admin:', addText);
                                    addData = { success: false, message: 'Ogiltigt svar från servern' };
                                }
                                
                                // Lägg också till i chatters
                                if (addData.success) {
                                    const addChatterFormData = new FormData();
                                    addChatterFormData.append('action', 'add');
                                    addChatterFormData.append('project_id', projectId);
                                    addChatterFormData.append('type', 'user');
                                    addChatterFormData.append('participant_id', participantId);
                                    addChatterFormData.append('user_id', participantId);
                                    addChatterFormData.append('list_type', 'chatters');
                                    
                                    await fetch('../api/chat/participants.php', {
                                        method: 'POST',
                                        body: addChatterFormData
                                    });
                                }
                                
                                if (addData.success) {
                                    // Ladda om deltagare efter att adminen lagts till
                                    await loadParticipants();
                                    // Starta ping efter att adminen lagts till och deltagare har laddats
                                    if (document.visibilityState === 'visible') {
                                        startPing();
                                    }
                                    return; // Returnera tidigt eftersom loadParticipants() redan anropas
                                } else {
                                    console.error('Kunde inte lägga till admin som deltagare:', addData.message);
                                    // Fortsätt ändå med att visa deltagare
                                }
                            } catch (e) {
                                console.error('Kunde inte lägga till admin som deltagare:', e);
                                // Fortsätt ändå med att visa deltagare
                            }
                        }
                    }
                    
                    // Om användaren är extern och är deltagare i projektet men inte chatter,
                    // lägg till dem automatiskt som chatter (om can_chat är true)
                    if (isExternType && participantId) {
                        const isParticipant = participants.some(p => p.participant_id === participantId);
                        const isChatter = chatters.some(c => c.participant_id === participantId);
                        
                        // Kontrollera om användaren är deltagare men inte chatter
                        if (isParticipant && !isChatter) {
                            // Hämta användardata för att kontrollera can_chat
                            fetch('../api/chat/external_staff.php?action=get&id=' + encodeURIComponent(participantId))
                                .then(res => res.json())
                                .then(userData => {
                                    if (userData.success && userData.staff) {
                                        const canChatRaw = userData.staff.can_chat ?? true;
                                        const canChat = (canChatRaw === true || canChatRaw === 1 || canChatRaw === '1' || canChatRaw === 'true');
                                        
                                        if (canChat) {
                                            // Lägg till extern användare som chatter automatiskt
                                            const addChatterFormData = new FormData();
                                            addChatterFormData.append('action', 'add');
                                            addChatterFormData.append('project_id', projectId);
                                            addChatterFormData.append('type', 'extern');
                                            addChatterFormData.append('participant_id', participantId);
                                            addChatterFormData.append('user_id', participantId);
                                            addChatterFormData.append('list_type', 'chatters');
                                            
                                            fetch('../api/chat/participants.php', {
                                                method: 'POST',
                                                body: addChatterFormData
                                            })
                                            .then(res => res.text())
                                            .then(text => {
                                                try {
                                                    const addData = JSON.parse(text);
                                                    if (addData.success) {
                                                        console.log('[loadParticipants] Extern användare lades till som chatter automatiskt');
                                                        // Ladda om deltagare efter att extern användare lagts till
                                                        loadParticipants();
                                                    } else {
                                                        console.warn('[loadParticipants] Kunde inte lägga till extern användare som chatter:', addData.message);
                                                    }
                                                } catch (e) {
                                                    console.error('[loadParticipants] Kunde inte parsa JSON-svar vid läggning av extern användare som chatter:', e);
                                                }
                                            })
                                            .catch(e => {
                                                console.error('[loadParticipants] Kunde inte lägga till extern användare som chatter:', e);
                                            });
                                        }
                                    }
                                })
                                .catch(e => {
                                    console.error('[loadParticipants] Kunde inte hämta extern användardata:', e);
                                });
                        }
                    }
                    
                    // Sätt alla deltagare som notifierade som standard,
                    // men hoppa över den egna användaren så att man inte notifierar sig själv
                    // och hoppa över de som explicit saknar pushover_user_key
                    // ONLY clear and set defaults if state hasn't been loaded from localStorage
                    if (!chatStateLoaded) {
                        notifiedParticipants.clear();
                    }
                    
                    if (!chatStateLoaded) {
                        participants.forEach(p => {
                        let id = p.participant_id;
                        const pushoverUserKey = p.pushover_user_key;
                        // Bara hoppa över om pushover_user_key är explicit tomt eller null
                        const hasPushover = pushoverUserKey === undefined || (pushoverUserKey && pushoverUserKey.trim() !== '');
                        if (id && (!participantId || id !== participantId) && hasPushover) {
                            notifiedParticipants.add(id);
                        }
                    });
                    }
                    renderParticipants();
                } else {
                    console.error('Fel vid laddning av deltagare:', data.message);
                    // Visa felmeddelande i UI
                    const container = document.getElementById('participantsList');
                    if (container) {
                        container.innerHTML = '<p style="color: red;">Fel vid laddning av deltagare: ' + escapeHtml(data.message) + '</p>';
                    }
                }
            } catch (e) {
                console.error('Fel vid laddning av deltagare:', e);
                // Visa felmeddelande i UI
                const container = document.getElementById('participantsList');
                if (container) {
                        container.innerHTML = '<p style="color: red;">Fel vid laddning av deltagare: ' + escapeHtml(e.message) + '</p>';
                }
            }
        }
        
        function renderParticipants() {
            const container = document.getElementById('participantsList');
            if (participants.length === 0) {
                let html = '';
                // Lägg till "Hantera deltagare"-ikon (tre gubbar) till vänster, endast för admin
                if (isAdmin) {
                    html += '<div id="manageParticipantsIcon" class="manage-participants-icon" onclick="showManageParticipants()" title="Hantera deltagare">';
                    html += '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 -2 34 22">';
                    html += '<path d="M5 7c1.7 0 3-1.3 3-3s-1.3-3-3-3-3 1.3-3 3 1.3 3 3 3zm0 1.5c-2 0-5 1-5 3v2.5h10v-2.5c0-2-3-3-5-3z" fill="currentColor"/>';
                    html += '<path d="M17 7c1.7 0 3-1.3 3-3s-1.3-3-3-3-3 1.3-3 3 1.3 3 3 3zm0 1.5c-2 0-5 1-5 3v2.5h10v-2.5c0-2-3-3-5-3z" fill="currentColor"/>';
                    html += '<path d="M29 7c1.7 0 3-1.3 3-3s-1.3-3-3-3-3 1.3-3 3 1.3 3 3 3zm0 1.5c-2 0-5 1-5 3v2.5h10v-2.5c0-2-3-3-5-3z" fill="currentColor"/>';
                    html += '</svg>';
                    html += '</div>';
                }
                html += '<p>Inga deltagare</p>';
                // Lägg till ikon för att slå av/på notifieringar till alla
                const allEnabled = window.notificationsAllEnabled || false;
                html += '<div id="notificationsToggle" class="notifications-toggle' + (allEnabled ? ' active' : '') + '" onclick="toggleAllNotifications()" title="' + (allEnabled ? 'Stäng av notifieringar till alla' : 'Slå på notifieringar till alla') + '">';
                html += '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24">';
                html += '<path d="M12 22c1.1 0 2-.9 2-2h-4a2 2 0 0 0 2 2zm6-6V11c0-3.07-1.63-5.64-4.5-6.32V4a1.5 1.5 0 0 0-3 0v.68C7.63 5.36 6 7.92 6 11v5l-1.7 1.7a1 1 0 0 0 .7 1.7h14a1 1 0 0 0 .7-1.7L18 16z" fill="currentColor"/>';
                html += '</svg>';
                html += '</div>';
                // Lägg till ikonen för brådskande notifikation
                const urgentActive = window.urgentNotificationActive || false;
                html += '<div id="urgentNotification" class="urgent-icon' + (urgentActive ? ' active' : '') + '" onclick="toggleUrgentNotification()" title="Skicka som brådskande notifikation">';
                html += '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24">';
                html += '<path d="M1 21h22L12 2 1 21zm12-3h-2v-2h2v2zm0-4h-2v-4h2v4z" fill="currentColor"/>';
                html += '</svg>';
                html += '</div>';
                // Lägg till frågeteckensikon för dokumentation
                html += '<div id="helpIcon" class="help-icon" onclick="showChatHelp()" title="Visa hjälp och dokumentation">';
                html += '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24">';
                html += '<circle cx="12" cy="12" r="10" fill="#e3f2fd"/>';
                html += '<path d="M11 18h2v-2h-2v2zm1-16C6.48 2 2 6.48 2 12s4.48 10 10 10 10-4.48 10-10S17.52 2 12 2zm0 18c-4.41 0-8-3.59-8-8s3.59-8 8-8 8 3.59 8 8-3.59 8-8 8zm0-14c-2.21 0-4 1.79-4 4h2c0-1.1.9-2 2-2s2 .9 2 2c0 2-3 1.75-3 5h2c0-2.25 3-2.5 3-5 0-2.21-1.79-4-4-4z" fill="#424242"/>';
                html += '</svg>';
                html += '</div>';
                // Lägg till pin-ikon för att filtrera fästa meddelanden
                const pinnedFilterActive2 = window.pinnedFilterActive || false;
                html += '<div id="pinnedFilterIcon" class="pinned-filter-icon' + (pinnedFilterActive2 ? ' active' : '') + '" onclick="togglePinnedFilter()" title="' + (pinnedFilterActive2 ? 'Visa alla meddelanden' : 'Visa endast fästa meddelanden') + '">';
                html += '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24">';
                html += '<path d="M16,12V4H17V2H7V4H8V12L6,14V16H11.2V22H12.8V16H18V14L16,12Z"/>';
                html += '</svg>';
                html += '</div>';
                
                container.innerHTML = html;
                return;
            }
            
            // Lägg till "Hantera deltagare"-ikon (tre gubbar) till vänster, endast för admin
            let html = '';
            if (isAdmin) {
                html += '<div id="manageParticipantsIcon" class="manage-participants-icon" onclick="showManageParticipants()" title="Hantera deltagare">';
                html += '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 -2 34 22">';
                // Tre gubbar bredvid varann - större storlek
                // Första gubben (vänster) - centrerad på x=5
                html += '<path d="M5 7c1.7 0 3-1.3 3-3s-1.3-3-3-3-3 1.3-3 3 1.3 3 3 3zm0 1.5c-2 0-5 1-5 3v2.5h10v-2.5c0-2-3-3-5-3z" fill="currentColor"/>';
                // Andra gubben (mitten) - centrerad på x=17
                html += '<path d="M17 7c1.7 0 3-1.3 3-3s-1.3-3-3-3-3 1.3-3 3 1.3 3 3 3zm0 1.5c-2 0-5 1-5 3v2.5h10v-2.5c0-2-3-3-5-3z" fill="currentColor"/>';
                // Tredje gubben (höger) - centrerad på x=29
                html += '<path d="M29 7c1.7 0 3-1.3 3-3s-1.3-3-3-3-3 1.3-3 3 1.3 3 3 3zm0 1.5c-2 0-5 1-5 3v2.5h10v-2.5c0-2-3-3-5-3z" fill="currentColor"/>';
                html += '</svg>';
                html += '</div>';
            }
            participants.forEach(p => {
                let id = p.participant_id;
                // Använd nickname om det finns, annars name, annars id
                let name = (p.nickname && p.nickname.trim() !== '') ? p.nickname : (p.name || id);
                let type = p.type || 'user'; // 'user' eller 'extern'
                let pushoverUserKey = p.pushover_user_key; // Ta inte bort undefined med ||
                
                if (!id) {
                    return; // Skip if no ID
                }
                
                // Kontrollera om pushover_user_key finns och inte är tom
                // Om fältet saknas helt (undefined), anta att det kan finnas (för bakåtkompatibilitet)
                // Bara disabla om det är explicit tomt sträng eller null
                const hasPushover = pushoverUserKey === undefined || (pushoverUserKey !== null && pushoverUserKey !== '' && String(pushoverUserKey).trim() !== '');
                const isNotified = notifiedParticipants.has(id);
                const safeId = escapeHtml(id).replace(/'/g, "\\'");
                const isExtern = type === 'extern';
                
                // Om användaren saknar pushover_user_key, visa annan tooltip och gör den icke-klickbar
                let tooltipText;
                if (pushoverUserKey !== undefined && !hasPushover) {
                    tooltipText = 'Saknar Pushover ID - kan inte notifieras';
                } else {
                    tooltipText = isNotified 
                        ? (isExtern ? 'Klicka för att avaktivera notifiering (röd = notifieras)' : 'Klicka för att avaktivera notifiering (grön = notifieras)')
                        : 'Klicka för att aktivera notifiering (ljusblå = notifieras inte)';
                }
                
                // Kontrollera om detta är den inloggade användaren (jämför både med participantId variabel och från URL)
                let currentUserId = participantId;
                if (!currentUserId) {
                    currentUserId = getParticipantIdFromUrl();
                }
                // Normalisera ID:n för jämförelse (ta bort eventuella mellanslag och konvertera till sträng)
                const normalizedId = String(id).trim();
                const normalizedCurrentUserId = currentUserId ? String(currentUserId).trim() : '';
                const isCurrentUser = normalizedCurrentUserId && (normalizedId === normalizedCurrentUserId);
                // Bara lägg till no-pushover klass om pushover_user_key är explicit tomt
                const noPushoverClass = (pushoverUserKey !== undefined && !hasPushover) ? 'no-pushover' : '';
                const classes = `participant-tag ${isExtern ? 'staff' : ''} ${isNotified ? 'notified' : ''} ${noPushoverClass}`;
                const labelSuffix = isCurrentUser ? ' (du)' : '';
                const safeIdForIcon = escapeHtml(id).replace(/"/g, '&quot;');
                html += `<span class="${classes}" data-participant-id="${escapeHtml(id)}">`;
                
                // Om användaren saknar pushover (explicit tomt), gör taggen icke-klickbar
                if (pushoverUserKey !== undefined && !hasPushover) {
                    html += `<span class="participant-tag-name" style="cursor: not-allowed; opacity: 0.6;" title="${tooltipText}${labelSuffix}">${escapeHtml(name)}${labelSuffix}</span>`;
                } else {
                    html += `<span class="participant-tag-name" onclick="toggleNotification('${safeId}')" title="${tooltipText}${labelSuffix}">${escapeHtml(name)}${labelSuffix}</span>`;
                }
                
                html += `<span class="participant-tag-icon" onclick="showParticipantCard('${safeIdForIcon}')" title="Visa kontaktinformation">`;
                html += '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24"><path d="M12 12c2.21 0 4-1.79 4-4s-1.79-4-4-4-4 1.79-4 4 1.79 4 4 4zm0 2c-2.67 0-8 1.34-8 4v2h16v-2c0-2.66-5.33-4-8-4z"/></svg>';
                html += '</span>';
                html += '</span>';
            });
            
            // Lägg till ikon för att slå av/på notifieringar till alla
            const allEnabled = window.notificationsAllEnabled || false;
            html += '<div id="notificationsToggle" class="notifications-toggle' + (allEnabled ? ' active' : '') + '" onclick="toggleAllNotifications()" title="' + (allEnabled ? 'Stäng av notifieringar till alla' : 'Slå på notifieringar till alla') + '">';
            html += '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24">';
            html += '<path d="M12 22c1.1 0 2-.9 2-2h-4a2 2 0 0 0 2 2zm6-6V11c0-3.07-1.63-5.64-4.5-6.32V4a1.5 1.5 0 0 0-3 0v.68C7.63 5.36 6 7.92 6 11v5l-1.7 1.7a1 1 0 0 0 .7 1.7h14a1 1 0 0 0 .7-1.7L18 16z" fill="currentColor"/>';
            html += '</svg>';
            html += '</div>';
            
            // Lägg till ikonen för brådskande notifikation på samma rad
            const urgentActive = window.urgentNotificationActive || false;
            html += '<div id="urgentNotification" class="urgent-icon' + (urgentActive ? ' active' : '') + '" onclick="toggleUrgentNotification()" title="Skicka som brådskande notifikation (kräver bekräftelse)">';
            html += '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24">';
            html += '<path d="M1 21h22L12 2 1 21zm12-3h-2v-2h2v2zm0-4h-2v-4h2v4z" fill="currentColor"/>';
            html += '</svg>';
            html += '</div>';
            // Lägg till frågeteckensikon för dokumentation
            html += '<div id="helpIcon" class="help-icon" onclick="showChatHelp()" title="Visa hjälp och dokumentation">';
            html += '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24">';
            html += '<circle cx="12" cy="12" r="10" fill="#e3f2fd"/>';
            html += '<path d="M11 18h2v-2h-2v2zm1-16C6.48 2 2 6.48 2 12s4.48 10 10 10 10-4.48 10-10S17.52 2 12 2zm0 18c-4.41 0-8-3.59-8-8s3.59-8 8-8 8 3.59 8 8-3.59 8-8 8zm0-14c-2.21 0-4 1.79-4 4h2c0-1.1.9-2 2-2s2 .9 2 2c0 2-3 1.75-3 5h2c0-2.25 3-2.5 3-5 0-2.21-1.79-4-4-4z" fill="#424242"/>';
            html += '</svg>';
            html += '</div>';
            // Lägg till pin-ikon för att filtrera fästa meddelanden
            const pinnedFilterActive = window.pinnedFilterActive || false;
            html += '<div id="pinnedFilterIcon" class="pinned-filter-icon' + (pinnedFilterActive ? ' active' : '') + '" onclick="togglePinnedFilter()" title="' + (pinnedFilterActive ? 'Visa alla meddelanden' : 'Visa endast fästa meddelanden') + '">';
            html += '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24">';
            html += '<path d="M16,12V4H17V2H7V4H8V12L6,14V16H11.2V22H12.8V16H18V14L16,12Z"/>';
            html += '</svg>';
            html += '</div>';
            // Lägg till sökikon
            html += '<div id="searchIcon" class="search-icon" onclick="showSearchModal()" title="Sök i chatten">';
            html += '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24">';
            html += '<path d="M15.5 14h-.79l-.28-.27C15.41 12.59 16 11.11 16 9.5 16 5.91 13.09 3 9.5 3S3 5.91 3 9.5 5.91 16 9.5 16c1.61 0 3.09-.59 4.23-1.57l.27.28v.79l5 4.99L20.49 19l-4.99-5zm-6 0C7.01 14 5 11.99 5 9.5S7.01 5 9.5 5 14 7.01 14 9.5 11.99 14 9.5 14z"/>';
            html += '</svg>';
            html += '</div>';
            
            container.innerHTML = html;
        }
        
        function toggleNotification(participantId) {
            try {
                if (!participantId) {
                    console.warn('toggleNotification: participantId saknas');
                    return;
                }
                
                // Kontrollera om participant har pushover_user_key
                const participant = participants.find(p => p.participant_id === participantId);
                if (!participant) {
                    console.warn('toggleNotification: participant hittades inte');
                    return;
                }
                
                const pushoverUserKey = participant.pushover_user_key;
                // Bara blockera om pushover_user_key är explicit satt till tomt eller null
                if (pushoverUserKey !== undefined && (!pushoverUserKey || pushoverUserKey.trim() === '')) {
                    // Användaren saknar pushover_user_key, tillåt inte att aktivera notifiering
                    alert('Denna användare saknar Pushover ID och kan inte notifieras.');
                    return;
                }
                
                if (notifiedParticipants.has(participantId)) {
                    notifiedParticipants.delete(participantId);
                } else {
                    notifiedParticipants.add(participantId);
                }
                renderParticipants();
                
                // Save state after notification change
                saveParticipantState();
            } catch (e) {
                console.error('Fel i toggleNotification:', e);
                // Fortsätt ändå - detta är inte kritiskt
            }
        }
        
        function toggleAllNotifications() {
            try {
                // Bestäm vad som ska hända baserat på nuvarande tillstånd
                const totalEligible = participants.filter(p => {
                    const pushoverUserKey = p.pushover_user_key;
                    const hasPushover = pushoverUserKey === undefined || (pushoverUserKey && pushoverUserKey.trim() !== '');
                    return hasPushover;
                }).length;
                
                const currentNotified = notifiedParticipants.size;
                const shouldEnableAll = currentNotified < totalEligible; // Om inte alla är notifierade, aktivera alla
                
                notifiedParticipants.clear();
                
                if (shouldEnableAll) {
                    // Aktivera notifieringar för alla deltagare inklusive den egna användaren
                    // Men hoppa över de som explicit saknar pushover_user_key
                    participants.forEach(p => {
                        let id = p.participant_id;
                        const pushoverUserKey = p.pushover_user_key;
                        // Bara hoppa över om pushover_user_key är explicit tomt eller null
                        const hasPushover = pushoverUserKey === undefined || (pushoverUserKey && pushoverUserKey.trim() !== '');
                        if (id && hasPushover) {
                            notifiedParticipants.add(id);
                        }
                    });
                }
                
                // Uppdatera window.notificationsAllEnabled baserat på vad vi faktiskt gjorde
                window.notificationsAllEnabled = shouldEnableAll;
                
                renderParticipants();
                
                // Save state after toggling all notifications
                saveParticipantState();
            } catch (e) {
                console.error('Fel i toggleAllNotifications:', e);
            }
        }
        
        function toggleUrgentNotification() {
            window.urgentNotificationActive = !window.urgentNotificationActive;
            const icon = document.getElementById('urgentNotification');
            if (icon) {
                if (window.urgentNotificationActive) {
                    icon.classList.add('active');
                } else {
                    icon.classList.remove('active');
                }
            }
        }
        
        function togglePinnedFilter() {
            window.pinnedFilterActive = !window.pinnedFilterActive;
            const icon = document.getElementById('pinnedFilterIcon');
            if (icon) {
                if (window.pinnedFilterActive) {
                    icon.classList.add('active');
                    icon.setAttribute('title', 'Visa alla meddelanden');
                } else {
                    icon.classList.remove('active');
                    icon.setAttribute('title', 'Visa endast fästa meddelanden');
                }
            }
            // Rendera om meddelanden med nytt filter
            renderMessages();
        }
        
        async function loadMessages() {
            try {
                // Uppdatera participantId från URL innan användning
                participantId = getParticipantIdFromUrl();
                if (!projectId) {
                    console.error('projectId är tom');
                    throw new Error('Projekt-ID saknas');
                }
                let url = `../api/chat/messages.php?action=list&project_id=${encodeURIComponent(projectId)}`;
                if (participantId) {
                    url += `&user_id=${encodeURIComponent(participantId)}`;
                }
                
                const res = await fetch(url);
                if (!res.ok) {
                    // Om användaren är inaktiv (403), redirecta
                    if (res.status === 403) {
                        const text = await res.text();
                        let data;
                        try {
                            data = JSON.parse(text);
                        } catch (e) {
                            data = { message: 'Din användare är inaktiv' };
                        }
                        alert(data.message || 'Din användare är inaktiv och du har loggats ut');
                        window.location.href = '../index.html';
                        return;
                    }
                    throw new Error(`HTTP ${res.status}: ${res.statusText}`);
                }
                const text = await res.text();
                let data;
                try {
                    data = JSON.parse(text);
                } catch (parseError) {
                    console.error('Ogiltigt JSON-svar från API:', text);
                    throw new Error('Ogiltigt svar från servern: ' + parseError.message);
                }
                
                // Kontrollera om användaren är inaktiv
                if (!data.success && data.message && data.message.includes('inaktiv')) {
                    alert(data.message);
                    window.location.href = '../index.html';
                    return;
                }
                
                if (data.success) {
                    messages = data.messages || [];
                    // Sortera meddelanden kronologiskt (behåller ursprunglig ordning)
                    messages.sort((a, b) => {
                        return new Date(a.created_at) - new Date(b.created_at);
                    });
                    renderMessages();
                } else {
                    document.getElementById('messagesContainer').innerHTML = '<p style="color: red;">' + escapeHtml(data.message) + '</p>';
                }
            } catch (e) {
                console.error('Fel vid laddning av meddelanden:', e);
                document.getElementById('messagesContainer').innerHTML = '<p style="color: red;">Fel vid laddning av meddelanden</p>';
            }
        }
        
        let isFirstLoad = true;
        
        // Hjälpfunktion för att beräkna extra padding för input-container och participants-section
        function getExtraPadding() {
            const inputContainer = document.querySelector('.input-container');
            const participantsSection = document.querySelector('.participants-section');
            return (inputContainer ? inputContainer.offsetHeight : 0) + (participantsSection ? participantsSection.offsetHeight : 0) + 20;
        }
        
        // Scroll to bottom funktionalitet (samma logik som scrollToBottomInstant men med smooth behavior)
        function scrollToBottom() {
            const container = document.getElementById('messagesContainer');
            const inputContainer = document.querySelector('.input-container');
            
            if (container) {
                // Använd requestAnimationFrame för att säkerställa att layouten är klar
                requestAnimationFrame(() => {
                    setTimeout(() => {
                        if (inputContainer) {
                            // Beräkna var input-container börjar relativt till messages-container
                            const containerRect = container.getBoundingClientRect();
                            const inputRect = inputContainer.getBoundingClientRect();
                            
                            // Scrolla så att input-container syns ovanför meddelandena
                            // Beräkna scroll-position: input-container position minus container position minus lite extra utrymme
                            const scrollPosition = inputRect.top - containerRect.top + container.scrollTop - 20;
                            
                            // Se till att vi scrollar minst till maxScroll (hela vägen ner)
                            const maxScroll = container.scrollHeight - container.clientHeight;
                            const finalScrollPosition = Math.max(scrollPosition, maxScroll);
                            
                            container.scrollTo({
                                top: Math.max(0, finalScrollPosition),
                                behavior: 'smooth'
                            });
                        } else {
                            // Fallback: scrolla till botten om input-container inte finns
                            container.scrollTo({
                                top: container.scrollHeight - container.clientHeight,
                                behavior: 'smooth'
                            });
                        }
                    }, 50);
                });
            }
        }
        
        // Hjälpfunktion för att scrolla till botten (instant, utan smooth)
        function scrollToBottomInstant() {
            const container = document.getElementById('messagesContainer');
            const inputContainer = document.querySelector('.input-container');
            
            if (container) {
                // Använd requestAnimationFrame för att säkerställa att layouten är klar
                requestAnimationFrame(() => {
                    if (inputContainer) {
                        // Beräkna var input-container börjar relativt till messages-container
                        const containerRect = container.getBoundingClientRect();
                        const inputRect = inputContainer.getBoundingClientRect();
                        
                        // Scrolla så att input-container syns ovanför meddelandena
                        // Beräkna scroll-position: input-container position minus container position minus lite extra utrymme
                        const scrollPosition = inputRect.top - containerRect.top + container.scrollTop - 20;
                        
                        // Se till att vi scrollar minst till maxScroll (hela vägen ner)
                        const maxScroll = container.scrollHeight - container.clientHeight;
                        const finalScrollPosition = Math.max(scrollPosition, maxScroll);
                        
                        container.scrollTop = Math.max(0, finalScrollPosition);
                    } else {
                        // Fallback: scrolla till botten om input-container inte finns
                        container.scrollTop = container.scrollHeight - container.clientHeight;
                    }
                });
            }
        }
        
        // Visa/dölj scroll-to-bottom knapp baserat på scroll-position
        function updateScrollToBottomButton() {
            const container = document.getElementById('messagesContainer');
            const scrollBtn = document.getElementById('scrollToBottomBtn');
            if (!container || !scrollBtn) return;
            
            // Visa knappen om användaren inte är nära botten (mer än 100px från botten)
            const isNearBottom = container.scrollHeight - container.scrollTop - container.clientHeight < 100;
            if (isNearBottom) {
                scrollBtn.style.display = 'none';
            } else {
                scrollBtn.style.display = 'block';
            }
        }
        
        function renderMessages() {
            // Uppdatera participantId från URL innan rendering
            participantId = getParticipantIdFromUrl();
            
            const container = document.getElementById('messagesContainer');
            
            // Filtrera meddelanden om pin-filter är aktivt
            let messagesToRender = messages;
            if (window.pinnedFilterActive) {
                messagesToRender = messages.filter(msg => msg.pinned === true);
            }
            
            if (messagesToRender.length === 0) {
                if (window.pinnedFilterActive) {
                    container.innerHTML = '<p style="text-align: center; color: #666; margin-top: 40px;">Inga fästa meddelanden.</p><button id="scrollToBottomBtn" class="scroll-to-bottom-btn" onclick="scrollToBottom()" title="Scrolla till senaste meddelandet" style="display: none;">↓</button>';
                } else {
                    container.innerHTML = '<p style="text-align: center; color: #666; margin-top: 40px;">Inga meddelanden än. Var först med att skriva!</p><button id="scrollToBottomBtn" class="scroll-to-bottom-btn" onclick="scrollToBottom()" title="Scrolla till senaste meddelandet" style="display: none;">↓</button>';
                }
                setTimeout(updateScrollToBottomButton, 100);
                return;
            }
            
            // För första laddningen, scrolla till botten för att visa senaste meddelandet
            let shouldScrollToBottom = isFirstLoad;
            if (isFirstLoad) {
                isFirstLoad = false;
            }
            
            // Spara scroll-positionen innan uppdatering (endast om det inte är första laddningen)
            const wasNearBottom = !shouldScrollToBottom && container.scrollHeight - container.scrollTop - container.clientHeight < 100;
            
            let html = '';
            let lastDate = null;
            
            messagesToRender.forEach((msg, index) => {
                const isSystem = msg.type === 'system';
                const msgParticipantId = msg.participant_id;
                const isOwn = msgParticipantId === participantId;
                
                const senderName = msg.sender_name || msgParticipantId || '';
                const msgDate = new Date(msg.created_at);
                const date = msgDate.toLocaleString('sv-SE');
                
                // Kontrollera om detta är en ny dag
                const msgDateOnly = msgDate.toDateString();
                if (index === 0 || (lastDate !== null && lastDate !== msgDateOnly)) {
                    // Lägg till datumseparator när dagen ändras
                    const dateStr = msgDate.toLocaleDateString('sv-SE', { weekday: 'long', year: 'numeric', month: 'long', day: 'numeric' });
                    html += `<div class="date-separator"><span class="date-separator-text">${escapeHtml(dateStr)}</span></div>`;
                }
                lastDate = msgDateOnly;
                
                // Bestäm CSS-klass baserat på sender_type och om det är "own"
                let messageClass = 'system';
                if (!isSystem) {
                    // Hantera olika sender_type värden
                    const senderType = msg.sender_type || 'user';
                    if (senderType === 'admin') {
                        messageClass = 'admin';
                    } else if (senderType === 'extern') {
                        messageClass = 'staff';
                    } else {
                        // user eller okänt -> använd user
                        messageClass = 'user';
                    }
                    // Lägg till "own" klass för positionering
                    if (isOwn) {
                        messageClass += ' own';
                    }
                    // Lägg till "urgent" klass om meddelandet har hög prio
                    const priority = msg.priority || msg.urgent;
                    if (priority === '2' || priority === 2 || priority === 'urgent' || msg.urgent === true) {
                        messageClass += ' urgent';
                    }
                    // Lägg till "pinned" klass om meddelandet är fäst
                    if (msg.pinned === true) {
                        messageClass += ' pinned';
                    }
                }
                
                // Lägg till margin-top på första meddelandet så att det syns under projektchatt-sektionen
                const firstMessageStyle = (index === 0) ? ' style="margin-top: 60px;"' : '';
                html += `<div class="message ${messageClass}" data-message-id="${escapeHtml(msg.id)}"${firstMessageStyle}>`;
                html += `<div class="message-bubble">`;
                
                // Lägg till reply-knapp och pin-knapp (ej för systemmeddelanden)
                if (!isSystem) {
                    // Använd data-attribut för att undvika problem med escaping i onclick
                    const safeMsgId = escapeHtml(msg.id);
                    const safeSenderName = escapeHtml(senderName);
                    const safeMsgText = escapeHtml(msg.message || (msg.type === 'image' ? '[Bild]' : msg.type === 'document' ? '[Dokument]' : ''));
                    // Hämta bild-URL om det är ett bildmeddelande
                    let imageUrl = '';
                    if (msg.type === 'image' && msg.media_file) {
                        // Använd download.php för att servera bilden (chats/-mappen är skyddad)
                        imageUrl = `download.php?project_id=${encodeURIComponent(projectId)}&media_file=${encodeURIComponent(msg.media_file)}`;
                    }
                    const safeImageUrl = escapeHtml(imageUrl);
                    const safeMsgType = escapeHtml(msg.type || '');
                    html += `<button class="message-reply-btn" data-msg-id="${safeMsgId}" data-sender-name="${safeSenderName}" data-msg-text="${safeMsgText}" data-msg-type="${safeMsgType}" data-image-url="${safeImageUrl}" title="Svara på detta meddelande">↩</button>`;
                    
                    // Lägg till pin-knapp för admins
                    if (isAdmin) {
                        const isPinned = msg.pinned === true;
                        const pinClass = isPinned ? 'pinned' : '';
                        html += `<button class="message-pin-btn ${pinClass}" data-msg-id="${safeMsgId}" title="${isPinned ? 'Avfäst meddelande' : 'Fäst meddelande'}">`;
                        html += `<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24">`;
                        html += `<path d="M16,12V4H17V2H7V4H8V12L6,14V16H11.2V22H12.8V16H18V14L16,12Z"/>`;
                        html += `</svg>`;
                        html += `</button>`;
                    }
                    
                    // Lägg till like-knapp (för alla användare)
                    const currentUserId = getParticipantIdFromUrl();
                    const userLikes = msg.likes || [];
                    const hasUserLiked = userLikes.includes(currentUserId);
                    const likeClass = hasUserLiked ? 'liked' : '';
                    html += `<button class="message-like-btn ${likeClass}" data-msg-id="${safeMsgId}" title="${hasUserLiked ? 'Ta bort gillning' : 'Gilla meddelande'}">`;
                    html += `<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" width="16" height="16">`;
                    html += `<path d="M1 21h4V9H1v12zm22-11c0-1.1-.9-2-2-2h-6.31l.95-4.57.03-.32c0-.41-.17-.79-.44-1.06L14.17 1 7.59 7.59C7.22 7.95 7 8.45 7 9v10c0 1.1.9 2 2 2h9c.83 0 1.54-.5 1.84-1.22l3.02-7.05c.09-.23.14-.47.14-.73v-1.91l-.01-.01L23 10z"/>`;
                    html += `</svg>`;
                    html += `</button>`;
                }
                
                // Visa citerat meddelande om detta meddelande är ett svar
                if (!isSystem && msg.reply_to) {
                    // Hitta det citerade meddelandet
                    const quotedMsg = messagesToRender.find(m => m.id === msg.reply_to) || messages.find(m => m.id === msg.reply_to);
                    if (quotedMsg) {
                        const quotedSenderName = quotedMsg.sender_name || quotedMsg.participant_id || 'Okänd';
                        html += `<div class="quoted-message">`;
                        html += `<div class="quoted-message-header">${escapeHtml(quotedSenderName)}</div>`;
                        html += `<div class="quoted-message-content">`;
                        
                        // Om det citerade meddelandet är en bild, visa miniatyr istället för text
                        if (quotedMsg.type === 'image' && quotedMsg.media_file) {
                            // Använd download.php för att servera bilden (chats/-mappen är skyddad)
                            const quotedImageUrl = `download.php?project_id=${encodeURIComponent(projectId)}&media_file=${encodeURIComponent(quotedMsg.media_file)}`;
                            html += `<img src="${quotedImageUrl}" class="quoted-message-image-thumbnail" alt="Bild" title="Bild">`;
                        } else {
                            // För textmeddelanden eller dokument, visa texten
                            const quotedText = quotedMsg.message || (quotedMsg.type === 'document' ? '[Dokument]' : '[Meddelande]');
                            html += escapeHtmlKeepQuotes(quotedText);
                        }
                        
                        html += `</div>`;
                        html += `</div>`;
                    }
                }
                
                if (isSystem) {
                    html += escapeHtmlKeepQuotes(msg.system_message || '');
                } else if (msg.type === 'image') {
                    // Använd download.php för att servera bilden (chats/-mappen är skyddad)
                    const imageUrl = `download.php?project_id=${encodeURIComponent(projectId)}&media_file=${encodeURIComponent(msg.media_file)}`;
                    const safeImageUrl = imageUrl.replace(/'/g, "\\'");
                    
                    // Bestäm nedladdningsfilnamn för download.php
                    let customFilename = '';
                    if (msg.display_name && msg.display_name.trim()) {
                        customFilename = msg.display_name;
                    } else if (msg.original_filename) {
                        customFilename = msg.original_filename.split('.')[0]; // Utan filändelse
                    }
                    
                    const downloadUrl = `download.php?project_id=${encodeURIComponent(projectId)}&media_file=${encodeURIComponent(msg.media_file)}&filename=${encodeURIComponent(customFilename)}`;
                    
                    html += `<img src="${imageUrl}" class="message-image" alt="Bild" onclick="downloadFile('${downloadUrl}')" title="Klicka för att ladda ner bilden">`;
                    // Visa det anpassade filnamnet om det finns
                    if (msg.display_name && msg.display_name.trim()) {
                        html += `<div class="message-image-filename">${escapeHtml(msg.display_name)}</div>`;
                    }
                    // Visa texten under bilden om den finns
                    if (msg.message && msg.message.trim()) {
                        html += `<div class="message-image-text">${escapeHtmlKeepQuotes(msg.message)}</div>`;
                    }
                } else if (msg.type === 'document') {
                    const documentUrl = `../chats/${escapeHtml(projectId)}/${escapeHtml(msg.media_file)}`;
                    const safeDocumentUrl = documentUrl.replace(/'/g, "\\'");
                    
                    // Bestäm nedladdningsfilnamn och visningsnamn för dokument
                    const fileExtension = msg.media_file.split('.').pop() || 'pdf';
                    let customFilename = '';
                    let displayName;
                    
                    if (msg.display_name && msg.display_name.trim()) {
                        // Använd anpassat namn
                        customFilename = msg.display_name;
                        displayName = `${msg.display_name}.${fileExtension}`;
                    } else if (msg.original_filename) {
                        // Använd ursprungligt filnamn
                        customFilename = msg.original_filename.split('.')[0]; // Utan filändelse
                        displayName = msg.original_filename;
                    } else {
                        // Fallback till "Dokument.pdf"
                        customFilename = '';
                        displayName = `Dokument.${fileExtension}`;
                    }
                    displayName = escapeHtml(displayName);
                    
                    const downloadUrl = `download.php?project_id=${encodeURIComponent(projectId)}&media_file=${encodeURIComponent(msg.media_file)}&filename=${encodeURIComponent(customFilename)}`;
                    
                    html += `<div class="message-document" onclick="downloadFile('${downloadUrl}')" title="Klicka för att ladda ner dokumentet">`;
                    html += `<svg class="message-document-icon" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">`;
                    html += `<path d="M14 2H6C5.46957 2 4.96086 2.21071 4.58579 2.58579C4.21071 2.96086 4 3.46957 4 4V20C4 20.5304 4.21071 21.0391 4.58579 21.4142C4.96086 21.7893 5.46957 22 6 22H18C18.5304 22 19.0391 21.7893 19.4142 21.4142C19.7893 21.0391 20 20.5304 20 20V8L14 2Z" stroke="#dc3545" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" fill="white"/>`;
                    html += `<path d="M14 2V8H20" stroke="#dc3545" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/>`;
                    html += `<path d="M16 13H8" stroke="#dc3545" stroke-width="2" stroke-linecap="round"/>`;
                    html += `<path d="M16 17H8" stroke="#dc3545" stroke-width="2" stroke-linecap="round"/>`;
                    html += `<path d="M10 9H8" stroke="#dc3545" stroke-width="2" stroke-linecap="round"/>`;
                    html += `</svg>`;
                    html += `<div class="message-document-info">`;
                    html += `<div class="message-document-name">${displayName}</div>`;
                    html += `<div class="message-document-size">PDF-dokument</div>`;
                    html += `</div>`;
                    html += `</div>`;
                    // Visa texten under dokumentet om den finns
                    if (msg.message && msg.message.trim()) {
                        html += `<div class="message-document-text">${escapeHtmlKeepQuotes(msg.message)}</div>`;
                    }
                } else {
                    html += escapeHtmlKeepQuotes(msg.message || '');
                }
                
                // Visa notisring endast om viewed_by propertyn finns och innehåller användare (exkludera avsändaren)
                if (!isSystem && msg.viewed_by && Array.isArray(msg.viewed_by) && msg.viewed_by.length > 0) {
                    // Filtrera bort avsändaren från räkningen
                    const senderId = msg.participant_id || '';
                    const viewedByOthers = msg.viewed_by.filter(userId => userId !== senderId);
                    if (viewedByOthers.length > 0) {
                        // Spara viewed_by array som data-attribut för att kunna visa namnen vid klick
                        // Använd encodeURIComponent för att säkert sätta JSON i HTML-attribut
                        const viewedByJson = encodeURIComponent(JSON.stringify(viewedByOthers));
                        html += `<span class="message-viewed-badge" onclick="showViewedByNames(event)" data-viewed-by="${viewedByJson}" title="Klicka för att se vem som sett meddelandet">${viewedByOthers.length}</span>`;
                    }
                }
                
                
                html += `</div>`;
                if (!isSystem) {
                    const pinnedIndicator = msg.pinned === true ? '<span class="pinned-indicator">Fäst</span>' : '';
                    
                    // Lägg till like-räknare om det finns likes
                    let likesIndicator = '';
                    if (msg.likes && Array.isArray(msg.likes) && msg.likes.length > 0) {
                        const likesJson = encodeURIComponent(JSON.stringify(msg.likes));
                        const likeText = msg.likes.length === 1 ? '1 gilla' : `${msg.likes.length} gilla`;
                        likesIndicator = `<span class="message-likes-count" onclick="showLikedByNames(event)" data-liked-by="${likesJson}" title="Klicka för att se vem som gillat meddelandet">${likeText}</span>`;
                    }
                    
                    html += `<div class="message-info">${escapeHtml(senderName)} • ${escapeHtml(date)}${pinnedIndicator}${likesIndicator}</div>`;
                }
                html += `</div>`;
            });
            
            // Lägg till scroll-to-bottom knappen i botten av meddelandena
            html += '<button id="scrollToBottomBtn" class="scroll-to-bottom-btn" onclick="scrollToBottom()" title="Scrolla till senaste meddelandet">↓</button>';
            
            container.innerHTML = html;
            
            // Sätt upp event listeners för knappar med data-attribut
            container.querySelectorAll('.message-like-btn').forEach(btn => {
                btn.addEventListener('click', function() {
                    const msgId = this.getAttribute('data-msg-id');
                    if (msgId) {
                        toggleLikeMessage(msgId);
                    }
                });
            });
            
            container.querySelectorAll('.message-pin-btn').forEach(btn => {
                btn.addEventListener('click', function() {
                    const msgId = this.getAttribute('data-msg-id');
                    if (msgId) {
                        togglePinMessage(msgId);
                    }
                });
            });
            
            container.querySelectorAll('.message-reply-btn').forEach(btn => {
                btn.addEventListener('click', function() {
                    const msgId = this.getAttribute('data-msg-id');
                    const senderName = this.getAttribute('data-sender-name');
                    const msgText = this.getAttribute('data-msg-text');
                    const msgType = this.getAttribute('data-msg-type');
                    const imageUrl = this.getAttribute('data-image-url');
                    if (msgId) {
                        replyToMessage(msgId, senderName, msgText, msgType, imageUrl);
                    }
                });
            });
            
            // För första laddningen, scrolla till botten för att visa senaste meddelandet
            if (shouldScrollToBottom) {
                // Hitta det sista meddelandet och scrolla till det (instant för första laddning)
                // Använd requestAnimationFrame för att säkerställa att DOM är helt renderad
                requestAnimationFrame(() => {
                    setTimeout(() => {
                        scrollToBottomInstant();
                        // Ytterligare scrollning efter delay för att hantera bilder som laddas
                        setTimeout(() => {
                            scrollToBottomInstant();
                            // Ytterligare scrollning för att säkerställa att vi är längst ner
                            setTimeout(() => {
                                scrollToBottomInstant();
                            }, 400);
                        }, 600);
                    }, 200);
                });
            } else {
                // För uppdateringar, scrolla bara till botten om användaren redan var nära botten
                // Använd samma logik som scroll-to-bottom knappen (smooth scroll)
                if (wasNearBottom) {
                    requestAnimationFrame(() => {
                        setTimeout(() => {
                            scrollToBottom();
                        }, 100);
                    });
                }
            }
            
            // Uppdatera scroll-to-bottom knappen efter rendering
            setTimeout(updateScrollToBottomButton, 100);
            
            // Initiera Intersection Observer för att spåra när meddelanden blir synliga
            initMessageViewTracking();
        }
        
        async function sendMessage() {
            // Förhindra att skicka meddelanden i arkiverade projekt
            if (isArchived) {
                alert('Detta projekt är arkiverat. Du kan inte skicka nya meddelanden.');
                return;
            }
            
            // Kontrollera om en fil är vald - i så fall ska den skickas via upload-popup
            // (sendFileWithDescription anropas från upload-popup, inte härifrån)
            const fileInput = document.getElementById('imageInput');
            if (fileInput && fileInput.files && fileInput.files.length > 0) {
                // Filen ska skickas via upload-popup, inte härifrån
                return;
            }
            
            // Uppdatera participantId från URL innan skickande
            participantId = getParticipantIdFromUrl();
            
            const messageInputEl = document.getElementById('messageInput');
            if (!messageInputEl) {
                return; // Elementet finns inte (arkiverat projekt)
            }
            
            const messageText = messageInputEl.value.trim();
            if (!messageText) {
                return;
            }
            
            // Hämta valda notifieringar från notifiedParticipants
            const notifiedUsers = Array.from(notifiedParticipants);
            
            // Kontrollera om brådskande notifikation är aktiv
            const urgentNotification = window.urgentNotificationActive || false;
            
            // Hämta reply_to om det finns
            const replyTo = window.currentReplyTo || null;
            
            const formData = new FormData();
            formData.append('action', 'send');
            formData.append('project_id', projectId);
            formData.append('message', messageText);
            if (participantId) {
                formData.append('user_id', participantId);
            }
            if (replyTo) {
                formData.append('reply_to', replyTo);
            }
            if (urgentNotification) {
                formData.append('priority', '2');
            }
            notifiedUsers.forEach(id => {
                formData.append('notified_users[]', id);
            });
            
            try {
                const res = await fetch('../api/chat/messages.php', {
                    method: 'POST',
                    body: formData
                });
                
                // Läs svaret som text först (kan bara läsas en gång)
                const text = await res.text();
                let data;
                
                // Kontrollera om användaren är inaktiv (403)
                if (res.status === 403) {
                    try {
                        data = JSON.parse(text);
                    } catch (e) {
                        data = { message: 'Din användare är inaktiv' };
                    }
                    alert(data.message || 'Din användare är inaktiv och du har loggats ut');
                    window.location.href = '../index.html';
                    return;
                }
                
                // Försök parsa JSON om det finns innehåll
                if (text.trim() === '') {
                    // Tomt svar - behandla som fel om status inte är OK
                    if (!res.ok) {
                        alert('Ett fel uppstod vid skickande av meddelande');
                        return;
                    }
                    // Om status är OK men tomt svar, behandla som framgång
                    data = { success: true };
                } else {
                    try {
                        data = JSON.parse(text);
                    } catch (parseError) {
                        // Om JSON-parsing misslyckas, visa felmeddelandet om det finns
                        if (!res.ok) {
                            alert('Ett fel uppstod vid skickande av meddelande: ' + (text || 'Okänt fel'));
                        } else {
                            alert('Ett fel uppstod vid kommunikation med servern');
                        }
                        return;
                    }
                }
                
                // Kontrollera om användaren är inaktiv
                if (!data.success && data.message && data.message.includes('inaktiv')) {
                    alert(data.message);
                    window.location.href = '../index.html';
                    return;
                }
                
                if (data.success) {
                    if (messageInputEl) {
                        messageInputEl.value = '';
                        // Återställ höjd
                        messageInputEl.style.height = 'auto';
                        messageInputEl.style.height = '88px';
                        // Säkerställ att textarea behåller sin bredd genom att tvinga en reflow
                        const parent = messageInputEl.parentElement;
                        if (parent) {
                            const display = parent.style.display;
                            parent.style.display = 'none';
                            void parent.offsetHeight; // Force reflow
                            parent.style.display = display;
                        }
                    }
                    // Återställ brådskande notifikation efter att meddelandet skickats
                    window.urgentNotificationActive = false;
                    const urgentIcon = document.getElementById('urgentNotification');
                    if (urgentIcon) {
                        urgentIcon.classList.remove('active');
                    }
                    await loadMessages();
                    // Scrolla till botten efter att meddelandet har skickats
                    scrollToBottomInstant();
                    // Uppdatera scroll-to-bottom knappen
                    setTimeout(updateScrollToBottomButton, 100);
                } else {
                    alert('Fel: ' + (data.message || 'Okänt fel'));
                }
            } catch (e) {
                // Logga felet men visa ett användarvänligt meddelande
                console.error('Fel vid skickande:', e);
                console.error('Feltyp:', e.name);
                console.error('Felmeddelande:', e.message);
                if (e.stack) {
                    console.error('Stack trace:', e.stack);
                }
                alert('Ett fel uppstod vid skickande av meddelandet. Kontrollera konsolen för detaljer.');
            }
        }
        
        // Funktion för att avbryta filuppladdning
        function cancelFileUpload() {
            const fileInput = document.getElementById('imageInput');
            const previewContainer = document.getElementById('filePreviewContainer');
            const overlay = document.getElementById('filePreviewOverlay');
            const uploadButton = document.querySelector('.file-input-wrapper');
            
            if (fileInput) {
                fileInput.value = '';
            }
            if (previewContainer) {
                previewContainer.style.display = 'none';
            }
            if (overlay) {
                overlay.style.display = 'none';
            }
            // Visa bilduppladdningsknappen igen
            if (uploadButton) {
                uploadButton.style.display = '';
            }
            // Rensa inte meddelandefältet - användaren kanske vill behålla texten
        }
        
        // Funktion för att hantera bilduppladdning
        async function handleImageUpload(fileInput) {
            // Förhindra att ladda upp filer i arkiverade projekt
            if (isArchived) {
                alert('Detta projekt är arkiverat. Du kan inte ladda upp nya filer.');
                return;
            }
            // Uppdatera participantId från URL innan uppladdning
            participantId = getParticipantIdFromUrl();
            
            const file = fileInput.files[0];
            if (!file) {
                // Om ingen fil valdes, dölj förhandsvisningen
                cancelFileUpload();
                return;
            }
            
            // Kontrollera filstorlek (max 32MB = 33554432 bytes)
            const maxFileSize = 32 * 1024 * 1024; // 32MB
            if (file.size > maxFileSize) {
                const fileSizeMB = (file.size / (1024 * 1024)).toFixed(1);
                const maxSizeMB = (maxFileSize / (1024 * 1024)).toFixed(0);
                alert(`Filen är för stor: ${fileSizeMB} MB (max ${maxSizeMB} MB). Kontakta administratören för att ladda upp större filer.`);
                cancelFileUpload();
                return;
            }
            
            // Visa upload-popup (liknande reply-popup)
            showUploadPopup(file);
        }
        
        // Funktion för att visa upload-popup (liknande reply-popup)
        function showUploadPopup(file) {
            // Skapa popup-overlay
            const overlay = document.createElement('div');
            overlay.className = 'reply-popup-overlay';
            overlay.onclick = function(e) {
                if (e.target === overlay) {
                    closeUploadPopup();
                }
            };
            
            // Skapa popup
            const popup = document.createElement('div');
            popup.className = 'reply-popup';
            
            // Header
            const header = document.createElement('div');
            header.className = 'reply-popup-header';
            const title = document.createElement('div');
            title.className = 'reply-popup-title';
            title.textContent = 'Ladda upp bild/fil';
            header.appendChild(title);
            
            // Content
            const content = document.createElement('div');
            content.className = 'reply-popup-content';
            
            // Filnamn-input
            const filenameLabel = document.createElement('label');
            filenameLabel.className = 'upload-popup-input-label';
            filenameLabel.textContent = 'Filnamn:';
            filenameLabel.setAttribute('for', 'uploadPopupFilename');
            
            const filenameInput = document.createElement('input');
            filenameInput.type = 'text';
            filenameInput.className = 'upload-popup-filename-input';
            filenameInput.id = 'uploadPopupFilename';
            filenameInput.placeholder = 'Filnamn';
            
            // Förifyll med det ursprungliga filnamnet (utan filändelse)
            const originalName = file.name;
            const nameWithoutExtension = originalName.substring(0, originalName.lastIndexOf('.')) || originalName;
            filenameInput.value = nameWithoutExtension;
            
            // Kommentar-input
            const commentLabel = document.createElement('label');
            commentLabel.className = 'upload-popup-input-label';
            commentLabel.textContent = 'Kommentar (valfritt):';
            commentLabel.setAttribute('for', 'uploadPopupComment');
            
            const commentInput = document.createElement('textarea');
            commentInput.className = 'reply-popup-input';
            commentInput.id = 'uploadPopupComment';
            commentInput.placeholder = 'Skriv en förklarande text...';
            
            content.appendChild(filenameLabel);
            content.appendChild(filenameInput);
            content.appendChild(commentLabel);
            content.appendChild(commentInput);
            
            // Footer
            const footer = document.createElement('div');
            footer.className = 'reply-popup-footer';
            const cancelBtn = document.createElement('button');
            cancelBtn.className = 'reply-popup-btn reply-popup-btn-cancel';
            cancelBtn.title = 'Avbryt';
            cancelBtn.onclick = closeUploadPopup;
            // Lägg till X-ikon för cancel-knappen
            const cancelIcon = document.createElementNS('http://www.w3.org/2000/svg', 'svg');
            cancelIcon.setAttribute('xmlns', 'http://www.w3.org/2000/svg');
            cancelIcon.setAttribute('viewBox', '0 0 24 24');
            const cancelPath = document.createElementNS('http://www.w3.org/2000/svg', 'path');
            cancelPath.setAttribute('d', 'M19 6.41L17.59 5 12 10.59 6.41 5 5 6.41 10.59 12 5 17.59 6.41 19 12 13.41 17.59 19 19 17.59 13.41 12z');
            cancelIcon.appendChild(cancelPath);
            cancelBtn.appendChild(cancelIcon);
            
            const sendBtn = document.createElement('button');
            sendBtn.className = 'reply-popup-btn reply-popup-btn-send';
            sendBtn.onclick = function() {
                sendUploadedFile(file, filenameInput.value.trim(), commentInput.value.trim());
            };
            // Lägg till SVG-ikon för send-knappen
            const sendIcon = document.createElementNS('http://www.w3.org/2000/svg', 'svg');
            sendIcon.setAttribute('xmlns', 'http://www.w3.org/2000/svg');
            sendIcon.setAttribute('viewBox', '0 0 24 24');
            const sendPath = document.createElementNS('http://www.w3.org/2000/svg', 'path');
            sendPath.setAttribute('d', 'M2.01 21L23 12 2.01 3 2 10l15 2-15 2z');
            sendIcon.appendChild(sendPath);
            sendBtn.appendChild(sendIcon);
            footer.appendChild(cancelBtn);
            footer.appendChild(sendBtn);
            
            popup.appendChild(header);
            popup.appendChild(content);
            popup.appendChild(footer);
            overlay.appendChild(popup);
            
            document.body.appendChild(overlay);
            
            // Fokusera på filnamnsfältet
            setTimeout(() => {
                filenameInput.focus();
                filenameInput.select();
            }, 100);
            
            // Spara referenser
            window.uploadPopupOverlay = overlay;
            window.uploadPopupFile = file;
            
            // Dölj bilduppladdningsknappen
            const uploadButton = document.querySelector('.file-input-wrapper');
            if (uploadButton) {
                uploadButton.style.display = 'none';
            }
        }
        
        // Funktion för att stänga upload-popup
        function closeUploadPopup() {
            if (window.uploadPopupOverlay) {
                window.uploadPopupOverlay.remove();
                window.uploadPopupOverlay = null;
                window.uploadPopupFile = null;
            }
            
            // Visa bilduppladdningsknappen igen
            const uploadButton = document.querySelector('.file-input-wrapper');
            if (uploadButton) {
                uploadButton.style.display = '';
            }
            
            // Rensa file input
            const fileInput = document.getElementById('imageInput');
            if (fileInput) {
                fileInput.value = '';
            }
        }
        
        // Funktion för att skicka uppladdad fil
        async function sendUploadedFile(file, customFileName, description) {
            if (!file) {
                return;
            }
            
            // Förhindra att ladda upp filer i arkiverade projekt
            if (isArchived) {
                alert('Detta projekt är arkiverat. Du kan inte ladda upp nya filer.');
                closeUploadPopup();
                return;
            }
            
            // Uppdatera participantId från URL innan uppladdning
            participantId = getParticipantIdFromUrl();
            
            // Bestäm filtyp
            const isPDF = file.type === 'application/pdf' || file.name.toLowerCase().endsWith('.pdf');
            const action = isPDF ? 'upload_document' : 'upload_image';
            const fileFieldName = isPDF ? 'document' : 'image';
            
            // Hämta valda notifieringar från notifiedParticipants
            const notifiedUsers = Array.from(notifiedParticipants);
            
            // Kontrollera om brådskande notifikation är aktiv
            const urgentNotification = window.urgentNotificationActive || false;
            
            const formData = new FormData();
            formData.append('action', action);
            formData.append('project_id', projectId);
            formData.append(fileFieldName, file);
            if (description) {
                formData.append('message', description);
            }
            if (customFileName) {
                formData.append('display_name', customFileName);
            }
            // Skicka alltid det ursprungliga filnamnet
            formData.append('original_filename', file.name);
            if (participantId) {
                formData.append('user_id', participantId);
            }
            if (urgentNotification) {
                formData.append('priority', '2');
            }
            notifiedUsers.forEach(id => {
                formData.append('notified_users[]', id);
            });
            
            try {
                const res = await fetch('../api/chat/messages.php', {
                    method: 'POST',
                    body: formData
                });
                
                // Läs svaret som text först (kan bara läsas en gång)
                const text = await res.text();
                let data;
                
                // Kontrollera om användaren är inaktiv (403)
                if (res.status === 403) {
                    try {
                        data = JSON.parse(text);
                    } catch (e) {
                        data = { message: 'Din användare är inaktiv' };
                    }
                    alert(data.message || 'Din användare är inaktiv och du har loggats ut');
                    window.location.href = '../index.html';
                    return;
                }
                
                // Kontrollera om filen är för stor (413)
                if (res.status === 413) {
                    try {
                        data = JSON.parse(text);
                    } catch (e) {
                        data = { message: 'Filen är för stor. Maximal storlek är 32 MB.' };
                    }
                    alert(data.message || 'Filen är för stor. Maximal storlek är 32 MB.');
                    closeUploadPopup();
                    return;
                }
                
                // Försök parsa JSON om det finns innehåll
                if (text.trim() === '') {
                    // Tomt svar - behandla som fel om status inte är OK
                    if (!res.ok) {
                        alert('Ett fel uppstod vid uppladdning');
                        return;
                    }
                    // Om status är OK men tomt svar, behandla som framgång
                    data = { success: true };
                } else {
                    try {
                        data = JSON.parse(text);
                    } catch (parseError) {
                        // Om JSON-parsing misslyckas, visa felmeddelandet om det finns
                        if (!res.ok) {
                            alert('Ett fel uppstod vid uppladdning: ' + (text || 'Okänt fel'));
                        } else {
                            alert('Ett fel uppstod vid kommunikation med servern');
                        }
                        return;
                    }
                }
                
                // Kontrollera om användaren är inaktiv
                if (!data.success && data.message && data.message.includes('inaktiv')) {
                    alert(data.message);
                    window.location.href = '../index.html';
                    return;
                }
                
                if (data.success) {
                    // Stäng popup och rensa
                    closeUploadPopup();
                    
                    // Rensa meddelandefältet
                    const messageInput = document.getElementById('messageInput');
                    if (messageInput) {
                        messageInput.value = '';
                    }
                    
                    // Återställ brådskande notifikation efter att filen skickats
                    window.urgentNotificationActive = false;
                    const urgentIcon = document.getElementById('urgentNotification');
                    if (urgentIcon) {
                        urgentIcon.classList.remove('active');
                    }
                    
                    await loadMessages();
                    // Scrolla till botten efter filuppladdning så att filen syns
                    setTimeout(() => {
                        scrollToBottomInstant();
                    }, 100);
                } else {
                    alert(data.message || 'Ett fel uppstod vid uppladdning');
                }
            } catch (error) {
                console.error('Fel vid uppladdning:', error);
                alert('Ett fel uppstod vid uppladdning. Kontrollera konsolen för detaljer.');
            }
        }
        
        // Funktion för att skicka fil med beskrivning
        async function sendFileWithDescription() {
            const fileInput = document.getElementById('imageInput');
            const messageInput = document.getElementById('messageInput');
            const previewContainer = document.getElementById('filePreviewContainer');
            
            if (!fileInput || !fileInput.files[0]) {
                return; // Ingen fil vald
            }
            
            const file = fileInput.files[0];
            const description = messageInput ? messageInput.value.trim() : '';
            
            // Förhindra att ladda upp filer i arkiverade projekt
            if (isArchived) {
                alert('Detta projekt är arkiverat. Du kan inte ladda upp nya filer.');
                return;
            }
            
            // Uppdatera participantId från URL innan uppladdning
            participantId = getParticipantIdFromUrl();
            
            // Bestäm filtyp
            const isPDF = file.type === 'application/pdf' || file.name.toLowerCase().endsWith('.pdf');
            const action = isPDF ? 'upload_document' : 'upload_image';
            const fileFieldName = isPDF ? 'document' : 'image';
            
            // Hämta valda notifieringar från notifiedParticipants
            const notifiedUsers = Array.from(notifiedParticipants);
            
            // Kontrollera om brådskande notifikation är aktiv
            const urgentNotification = window.urgentNotificationActive || false;
            
            // Hämta det anpassade filnamnet från input-fältet
            const fileDisplayNameInput = document.getElementById('fileDisplayName');
            const customFileName = fileDisplayNameInput ? fileDisplayNameInput.value.trim() : '';
            
            const formData = new FormData();
            formData.append('action', action);
            formData.append('project_id', projectId);
            formData.append(fileFieldName, file);
            if (description) {
                formData.append('message', description);
            }
            if (customFileName) {
                formData.append('display_name', customFileName); // Skicka det valda filnamnet
            }
            // Skicka alltid det ursprungliga filnamnet
            formData.append('original_filename', file.name);
            if (participantId) {
                formData.append('user_id', participantId);
            }
            if (urgentNotification) {
                formData.append('priority', '2');
            }
            notifiedUsers.forEach(id => {
                formData.append('notified_users[]', id);
            });
            
            try {
                const res = await fetch('../api/chat/messages.php', {
                    method: 'POST',
                    body: formData
                });
                
                // Läs svaret som text först (kan bara läsas en gång)
                const text = await res.text();
                let data;
                
                // Kontrollera om användaren är inaktiv (403)
                if (res.status === 403) {
                    try {
                        data = JSON.parse(text);
                    } catch (e) {
                        data = { message: 'Din användare är inaktiv' };
                    }
                    alert(data.message || 'Din användare är inaktiv och du har loggats ut');
                    window.location.href = '../index.html';
                    return;
                }
                
                // Kontrollera om filen är för stor (413)
                if (res.status === 413) {
                    try {
                        data = JSON.parse(text);
                    } catch (e) {
                        data = { message: 'Filen är för stor. Maximal storlek är 32 MB.' };
                    }
                    alert(data.message || 'Filen är för stor. Maximal storlek är 32 MB.');
                    cancelFileUpload();
                    return;
                }
                
                // Försök parsa JSON om det finns innehåll
                if (text.trim() === '') {
                    // Tomt svar - behandla som fel om status inte är OK
                    if (!res.ok) {
                        alert('Ett fel uppstod vid uppladdning');
                        return;
                    }
                    // Om status är OK men tomt svar, behandla som framgång
                    data = { success: true };
                } else {
                    try {
                        data = JSON.parse(text);
                    } catch (parseError) {
                        // Om JSON-parsing misslyckas, visa felmeddelandet om det finns
                        if (!res.ok) {
                            alert('Ett fel uppstod vid uppladdning: ' + (text || 'Okänt fel'));
                        } else {
                            alert('Ett fel uppstod vid kommunikation med servern');
                        }
                        return;
                    }
                }
                
                // Kontrollera om användaren är inaktiv
                if (!data.success && data.message && data.message.includes('inaktiv')) {
                    alert(data.message);
                    window.location.href = '../index.html';
                    return;
                }
                
                if (data.success) {
                    // Rensa filinput och meddelandefält, stäng modal
                    cancelFileUpload();
                    if (messageInput) {
                        messageInput.value = '';
                    }
                    // Återställ brådskande notifikation efter att filen skickats
                    window.urgentNotificationActive = false;
                    const urgentIcon = document.getElementById('urgentNotification');
                    if (urgentIcon) {
                        urgentIcon.classList.remove('active');
                    }
                    await loadMessages();
                    // Scrolla till botten efter filuppladdning så att filen syns
                    const container = document.getElementById('messagesContainer');
                    if (container) {
                        // Använd scrollIntoView för att säkerställa att det senaste meddelandet syns
                        setTimeout(() => {
                            scrollToBottomInstant();
                            // Ytterligare scrollning efter delay för att hantera bilder som laddas
                            setTimeout(() => {
                                scrollToBottomInstant();
                                setTimeout(() => {
                                    scrollToBottomInstant();
                                }, 400);
                            }, 600);
                        }, 200);
                    }
                    // Uppdatera scroll-to-bottom knappen
                    setTimeout(updateScrollToBottomButton, 100);
                } else {
                    alert('Fel: ' + (data.message || 'Okänt fel'));
                }
            } catch (error) {
                console.error('Fel vid uppladdning:', error);
                alert('Ett fel uppstod');
            }
        }
        
        // Bilduppladdning (fotoalbum eller kamera)
        // Lägg till event listener för bilduppladdning endast om elementet finns (ej arkiverade projekt)
        const imageInput = document.getElementById('imageInput');
        if (imageInput) {
            imageInput.addEventListener('change', async function(e) {
                await handleImageUpload(e.target);
            });
        }
        
        function escapeHtml(text) {
            const div = document.createElement('div');
            div.textContent = text;
            return div.innerHTML;
        }
        
        // Funktion för att ladda ner fil via download.php
        function downloadFile(downloadUrl) {
            // Skapa ett temporärt a-element för nedladdning
            const link = document.createElement('a');
            link.href = downloadUrl;
            link.style.display = 'none';
            document.body.appendChild(link);
            link.click();
            document.body.removeChild(link);
        }
        
        // Funktion för att gilla/ogilla meddelande
        async function toggleLikeMessage(messageId) {
            const currentUserId = getParticipantIdFromUrl();
            if (!currentUserId) {
                return;
            }
            
            // Escape messageId för CSS-selector (fallback om CSS.escape inte finns)
            const escapedMessageId = (typeof CSS !== 'undefined' && CSS.escape) ? CSS.escape(messageId) : messageId.replace(/[!"#$%&'()*+,.\/:;<=>?@[\\\]^`{|}~]/g, '\\$&');
            const messageElement = document.querySelector(`[data-message-id="${escapedMessageId}"]`);
            if (!messageElement) {
                return;
            }
            
            const likeBtn = messageElement.querySelector('.message-like-btn');
            const messageInfo = messageElement.querySelector('.message-info');
            
            // Optimistic UI update - uppdatera direkt
            const wasLiked = likeBtn && likeBtn.classList.contains('liked');
            const willBeLiked = !wasLiked;
            
            // Uppdatera like-knappen omedelbart
            if (likeBtn) {
                if (willBeLiked) {
                    likeBtn.classList.add('liked');
                    likeBtn.title = 'Ta bort gillning';
                } else {
                    likeBtn.classList.remove('liked');
                    likeBtn.title = 'Gilla meddelande';
                }
            }
            
            // Uppskatta ny like-count och uppdatera UI
            let currentLikesCount = 0;
            let currentLikes = [];
            const existingLikesCount = messageInfo ? messageInfo.querySelector('.message-likes-count') : null;
            
            if (existingLikesCount) {
                const likesData = existingLikesCount.getAttribute('data-liked-by');
                if (likesData) {
                    try {
                        // Försök först med decodeURIComponent, fallback till direkt parse
                        let decodedData = likesData;
                        try {
                            decodedData = decodeURIComponent(likesData);
                        } catch (e) {
                            // Om decodeURIComponent misslyckas, använd data direkt
                            decodedData = likesData;
                        }
                        currentLikes = JSON.parse(decodedData);
                        if (!Array.isArray(currentLikes)) {
                            currentLikes = [];
                        }
                        currentLikesCount = currentLikes.length;
                    } catch (e) {
                        console.error('Fel vid parsing av likes data:', e);
                        currentLikes = [];
                        currentLikesCount = 0;
                    }
                }
            }
            
            // Beräkna ny like-count optimistically
            let newLikeCount;
            let newLikes;
            if (willBeLiked) {
                newLikeCount = currentLikesCount + 1;
                newLikes = [...currentLikes, currentUserId];
            } else {
                newLikeCount = Math.max(0, currentLikesCount - 1);
                newLikes = currentLikes.filter(id => id !== currentUserId);
            }
            
            // Uppdatera like-räknaren omedelbart
            if (messageInfo) {
                let likesCount = messageInfo.querySelector('.message-likes-count');
                
                if (newLikeCount > 0) {
                    const likeText = newLikeCount === 1 ? '1 gilla' : `${newLikeCount} gilla`;
                    
                    if (likesCount) {
                        // Uppdatera befintlig like-räknare
                        likesCount.textContent = likeText;
                        likesCount.setAttribute('data-liked-by', encodeURIComponent(JSON.stringify(newLikes)));
                    } else {
                        // Skapa ny like-räknare
                        const likesJson = encodeURIComponent(JSON.stringify(newLikes));
                        likesCount = document.createElement('span');
                        likesCount.className = 'message-likes-count';
                        likesCount.onclick = showLikedByNames;
                        likesCount.setAttribute('data-liked-by', likesJson);
                        likesCount.title = 'Klicka för att se vem som gillat meddelandet';
                        likesCount.textContent = likeText;
                        messageInfo.appendChild(likesCount);
                    }
                } else {
                    // Ta bort like-räknare om inga likes finns
                    if (likesCount) {
                        likesCount.remove();
                    }
                }
            }
            
            // Skicka till servern i bakgrunden
            try {
                const formData = new FormData();
                formData.append('action', 'toggle_like');
                formData.append('project_id', projectId);
                formData.append('message_id', messageId);
                formData.append('user_id', currentUserId);
                
                const response = await fetch('../api/chat/messages.php', {
                    method: 'POST',
                    body: formData
                });
                
                // Kontrollera om svaret är giltig JSON
                let data;
                let responseText = '';
                try {
                    // Kontrollera response status först
                    if (!response.ok) {
                        throw new Error(`HTTP ${response.status}: ${response.statusText}`);
                    }
                    
                    responseText = await response.text();
                    if (!responseText || responseText.trim() === '') {
                        // Om svaret är tomt men status är OK, anta att det lyckades
                        // (vissa servrar kan returnera tomt svar vid success)
                        // Funktionen fungerade, inget att göra
                        return;
                    }
                    data = JSON.parse(responseText);
                } catch (parseError) {
                    // Bara logga om det verkligen är ett fel (inte bara tomt svar med OK status)
                    if (response.ok && (!responseText || responseText.trim() === '')) {
                        // Tyst ignorera tomt svar med OK status - funktionen fungerade
                        return;
                    }
                    console.error('Fel vid parsing av svar från servern:', parseError);
                    // Återställ UI:t vid parse-fel
                    if (likeBtn) {
                        if (wasLiked) {
                            likeBtn.classList.add('liked');
                            likeBtn.title = 'Ta bort gillning';
                        } else {
                            likeBtn.classList.remove('liked');
                            likeBtn.title = 'Gilla meddelande';
                        }
                    }
                    return;
                }
                
                if (!data.success) {
                    // Om servern returnerar fel, återställ UI:t
                    console.error('Fel vid gillning:', data.message || 'Okänt fel');
                    
                    // Återställ like-knappen
                    if (likeBtn) {
                        if (wasLiked) {
                            likeBtn.classList.add('liked');
                            likeBtn.title = 'Ta bort gillning';
                        } else {
                            likeBtn.classList.remove('liked');
                            likeBtn.title = 'Gilla meddelande';
                        }
                    }
                    
                    // Återställ like-räknaren
                    if (messageInfo) {
                        let likesCount = messageInfo.querySelector('.message-likes-count');
                        
                        if (currentLikesCount > 0) {
                            const likeText = currentLikesCount === 1 ? '1 gilla' : `${currentLikesCount} gilla`;
                            
                            if (likesCount) {
                                likesCount.textContent = likeText;
                                likesCount.setAttribute('data-liked-by', encodeURIComponent(JSON.stringify(currentLikes)));
                            } else {
                                const likesJson = encodeURIComponent(JSON.stringify(currentLikes));
                                likesCount = document.createElement('span');
                                likesCount.className = 'message-likes-count';
                                likesCount.onclick = showLikedByNames;
                                likesCount.setAttribute('data-liked-by', likesJson);
                                likesCount.title = 'Klicka för att se vem som gillat meddelandet';
                                likesCount.textContent = likeText;
                                messageInfo.appendChild(likesCount);
                            }
                        } else {
                            if (likesCount) {
                                likesCount.remove();
                            }
                        }
                    }
                }
                // Om servern lyckas behöver vi inte göra något - UI:t är redan uppdaterat optimistically
            } catch (error) {
                console.error('Fel vid gillning:', error);
                // Vid nätverksfel, återställ inte UI:t - låt användaren se sin aktion
            }
        }
        
        // State persistence functions
        function saveParticipantState() {
            if (!projectId) {
                return;
            }
            
            try {
                // Get current participants (those who are actually in the chat)
                const currentParticipants = [];
                
                // All participants in the participants array are active participants
                participants.forEach(participant => {
                    const id = participant.participant_id || participant.id;
                    if (id) {
                        currentParticipants.push(id);
                    }
                });
                
                const notifiedArray = Array.from(notifiedParticipants);
                
                const state = {
                    participants: currentParticipants,
                    notifiedParticipants: notifiedArray,
                    timestamp: Date.now(),
                    url: window.location.href,
                    projectId: projectId
                };
                
                const stateKey = `chat_state_${projectId}`;
                const stateJson = JSON.stringify(state);
                
                
                localStorage.setItem(stateKey, stateJson);
                
                // Verify it was saved
                const savedData = localStorage.getItem(stateKey);
                
            } catch (error) {
                console.error('Error saving chat state:', error);
            }
        }
        
        function loadParticipantState() {
            if (!projectId || chatStateLoaded) {
                return false;
            }
            
            try {
                const stateKey = `chat_state_${projectId}`;
                const savedState = localStorage.getItem(stateKey);
                
                if (!savedState) {
                    return false;
                }
                
                const state = JSON.parse(savedState);
                
                // Validate state
                if (state.projectId !== projectId) {
                    return false;
                }
                
                const age = Date.now() - state.timestamp;
                if (age > STATE_EXPIRY_TIME) {
                    clearParticipantState();
                    return false;
                }
                
                // Check if this is a reload of the same page
                const isReload = isPageReload();
                if (!isReload) {
                    clearParticipantState();
                    return false;
                }
                
                // Restore notified participants
                notifiedParticipants = new Set(state.notifiedParticipants || []);
                
                
                chatStateLoaded = true;
                return true;
            } catch (error) {
                console.error('Error loading chat state:', error);
                return false;
            }
        }
        
        function clearParticipantState() {
            if (!projectId) return;
            
            try {
                localStorage.removeItem(`chat_state_${projectId}`);
            } catch (error) {
                console.error('Error clearing chat state:', error);
            }
        }
        
        function isPageReload() {
            try {
                const currentUrl = window.location.href;
                const sessionKey = 'chat_page_token';
                
                // Check if we have a page token from previous load of THIS exact URL
                const storedToken = sessionStorage.getItem(sessionKey);
                const storedUrl = sessionStorage.getItem(sessionKey + '_url');
                
                
                // Set token for this page load (for next time)
                sessionStorage.setItem(sessionKey, Date.now().toString());
                sessionStorage.setItem(sessionKey + '_url', currentUrl);
                
                // If we had a token for the same URL, this is likely a reload/refresh
                if (storedToken && storedUrl === currentUrl) {
                    return true;
                }
                
                // Additional check: Modern Navigation API
                if (performance && performance.getEntriesByType) {
                    const navEntries = performance.getEntriesByType('navigation');
                    if (navEntries.length > 0) {
                        const navType = navEntries[0].type;
                        if (navType === 'reload') {
                            return true;
                        }
                    }
                }
                
                return false;
                
            } catch (error) {
                console.error('Error in isPageReload():', error);
                return false;
            }
        }
        
        function cleanupOldChatStates() {
            try {
                const keysToRemove = [];
                for (let i = 0; i < localStorage.length; i++) {
                    const key = localStorage.key(i);
                    if (key && key.startsWith('chat_state_')) {
                        const data = localStorage.getItem(key);
                        if (data) {
                            const state = JSON.parse(data);
                            if (Date.now() - state.timestamp > STATE_EXPIRY_TIME) {
                                keysToRemove.push(key);
                            }
                        }
                    }
                }
                
                keysToRemove.forEach(key => localStorage.removeItem(key));
                if (keysToRemove.length > 0) {
                }
            } catch (error) {
                console.error('Error cleaning up old chat states:', error);
            }
        }
        
        // Funktion för att visa vem som gillat meddelandet
        function showLikedByNames(event) {
            event.stopPropagation();
            
            const likedByData = event.target.getAttribute('data-liked-by');
            if (!likedByData) return;
            
            try {
                const likedByIds = JSON.parse(decodeURIComponent(likedByData));
                if (!Array.isArray(likedByIds) || likedByIds.length === 0) return;
                
                // Hämta namn för användar-ID:n
                const likedByNames = likedByIds.map(userId => {
                    // Hitta namnet i allParticipants
                    const participant = allParticipants.find(p => p.id === userId);
                    return participant ? participant.name : userId;
                });
                
                const message = likedByNames.length === 1 ? 
                    `Gillad av: ${likedByNames[0]}` : 
                    `Gillad av: ${likedByNames.join(', ')}`;
                
                alert(message);
            } catch (error) {
                console.error('Fel vid visning av likes:', error);
            }
        }
        
        // Funktion för att fästa/avfästa meddelanden
        async function togglePinMessage(messageId) {
            if (!isAdmin) {
                alert('Endast admin kan fästa meddelanden');
                return;
            }
            
            // Escape messageId för CSS-selector (fallback om CSS.escape inte finns)
            const escapedMessageId = (typeof CSS !== 'undefined' && CSS.escape) ? CSS.escape(messageId) : messageId.replace(/[!"#$%&'()*+,.\/:;<=>?@[\\\]^`{|}~]/g, '\\$&');
            
            try {
                participantId = getParticipantIdFromUrl();
                
                const formData = new FormData();
                formData.append('action', 'toggle_pin');
                formData.append('project_id', projectId);
                formData.append('message_id', messageId);
                if (participantId) {
                    formData.append('user_id', participantId);
                }
                
                const res = await fetch('../api/chat/messages.php', {
                    method: 'POST',
                    body: formData
                });
                
                // Läs svaret som text först (kan bara läsas en gång)
                const text = await res.text();
                let data;
                
                // Försök parsa JSON om det finns innehåll
                if (text.trim() === '') {
                    // Tomt svar - behandla som fel om status inte är OK
                    if (!res.ok) {
                        alert('Ett fel uppstod vid fästning av meddelande');
                        return;
                    }
                    // Om status är OK men tomt svar, behandla som framgång
                    data = { success: true };
                } else {
                    try {
                        data = JSON.parse(text);
                    } catch (parseError) {
                        // Om JSON-parsing misslyckas, visa felmeddelandet om det finns
                        if (!res.ok) {
                            alert('Ett fel uppstod vid fästning av meddelande: ' + (text || 'Okänt fel'));
                        } else {
                            alert('Ett fel uppstod vid kommunikation med servern');
                        }
                        return;
                    }
                }
                
                if (data.success) {
                    // Ladda om meddelanden för att visa uppdaterad sortering
                    await loadMessages();
                } else {
                    alert('Fel: ' + (data.message || 'Okänt fel'));
                }
            } catch (e) {
                console.error('Fel vid fästning av meddelande:', e);
                alert('Ett fel uppstod');
            }
        }
        
        // Cache för användarnamn
        let userNameCache = {};
        
        // Funktion för att hämta användarnamn
        async function getUserName(userId) {
            if (userNameCache[userId]) {
                return userNameCache[userId];
            }
            
            try {
                const res = await fetch(`../api/user_info.php?user_id=${encodeURIComponent(userId)}`);
                const data = await res.json();
                if (data.success && data.user) {
                    // Använd nickname om det finns, annars full_name
                    const name = (data.user.nickname && data.user.nickname.trim() !== '') 
                        ? data.user.nickname 
                        : (data.user.full_name || userId);
                    userNameCache[userId] = name;
                    return name;
                }
            } catch (e) {
                console.error('Fel vid hämtning av användarnamn:', e);
            }
            
            // Fallback till user_id om namn inte hittades
            userNameCache[userId] = userId;
            return userId;
        }
        
        // Funktion för att visa namnen på de som sett meddelandet
        async function showViewedByNames(event) {
            event.stopPropagation(); // Förhindra att meddelandet markeras
            
            // Hämta viewed_by array från data-attribut
            const badge = event.target;
            const viewedByJsonEncoded = badge.getAttribute('data-viewed-by');
            if (!viewedByJsonEncoded) {
                return;
            }
            
            let viewedByUserIds;
            try {
                // Dekoda URI-komponenten först, sedan parsa JSON
                const viewedByJson = decodeURIComponent(viewedByJsonEncoded);
                viewedByUserIds = JSON.parse(viewedByJson);
            } catch (e) {
                console.error('Fel vid parsning av viewed_by:', e, 'Encoded:', viewedByJsonEncoded);
                return;
            }
            
            if (!viewedByUserIds || viewedByUserIds.length === 0) {
                return;
            }
            
            // Hämta alla namn
            const names = await Promise.all(viewedByUserIds.map(userId => getUserName(userId)));
            
            // Skapa popup
            const popup = document.createElement('div');
            popup.style.cssText = `
                position: fixed;
                top: 50%;
                left: 50%;
                transform: translate(-50%, -50%);
                background: white;
                border-radius: 8px;
                padding: 12px 16px;
                box-shadow: 0 4px 16px rgba(0,0,0,0.3);
                z-index: 10000;
                max-width: 250px;
                width: auto;
            `;
            
            let html = '<ul style="list-style: none; padding: 0; margin: 0; font-size: 13px;">';
            names.forEach(name => {
                html += `<li style="padding: 4px 0; border-bottom: 1px solid #eee; color: #555;">${escapeHtml(name)}</li>`;
            });
            html += '</ul>';
            html += '<button onclick="this.closest(\'div\').remove()" style="margin-top: 10px; padding: 6px 16px; background: #007bff; color: white; border: none; border-radius: 4px; cursor: pointer; width: 100%; font-size: 12px;">Stäng</button>';
            
            popup.innerHTML = html;
            
            // Lägg till overlay
            const overlay = document.createElement('div');
            overlay.style.cssText = `
                position: fixed;
                top: 0;
                left: 0;
                width: 100%;
                height: 100%;
                background: rgba(0,0,0,0.5);
                z-index: 9999;
            `;
            overlay.onclick = () => {
                popup.remove();
                overlay.remove();
            };
            
            // Ta bort popup och overlay när stäng-knappen klickas
            const closeBtn = popup.querySelector('button');
            if (closeBtn) {
                closeBtn.onclick = () => {
                    popup.remove();
                    overlay.remove();
                };
            }
            
            document.body.appendChild(overlay);
            document.body.appendChild(popup);
        }
        
        // Funktion för att escapea HTML men behålla citationstecken korrekt
        // Variabel för att hålla reda på vilket meddelande man svarar på
        window.currentReplyTo = null;
        
        // Funktion för att svara på ett meddelande
        function replyToMessage(messageId, senderName, messageText, messageType, imageUrl) {
            // Skapa popup-overlay
            const overlay = document.createElement('div');
            overlay.className = 'reply-popup-overlay';
            overlay.onclick = function(e) {
                if (e.target === overlay) {
                    closeReplyPopup();
                }
            };
            
            // Skapa popup
            const popup = document.createElement('div');
            popup.className = 'reply-popup';
            
            // Header
            const header = document.createElement('div');
            header.className = 'reply-popup-header';
            const title = document.createElement('div');
            title.className = 'reply-popup-title';
            title.textContent = 'Svara på meddelande';
            header.appendChild(title);
            
            // Content
            const content = document.createElement('div');
            content.className = 'reply-popup-content';
            
            // Quoted message
            const quoted = document.createElement('div');
            quoted.className = 'reply-popup-quoted';
            const quotedHeader = document.createElement('div');
            quotedHeader.className = 'reply-popup-quoted-header';
            quotedHeader.textContent = senderName;
            const quotedContent = document.createElement('div');
            quotedContent.className = 'reply-popup-quoted-content';
            
            // Om det är ett bildmeddelande, visa miniatyr istället för text
            if (messageType === 'image' && imageUrl) {
                const imageThumbnail = document.createElement('img');
                imageThumbnail.src = imageUrl;
                imageThumbnail.className = 'reply-popup-image-thumbnail';
                imageThumbnail.alt = 'Bild';
                imageThumbnail.onerror = function() {
                    // Om bilden inte kan laddas, visa texten "[Bild]" istället
                    this.style.display = 'none';
                    quotedContent.textContent = messageText;
                };
                quotedContent.appendChild(imageThumbnail);
            } else {
                quotedContent.textContent = messageText;
            }
            
            quoted.appendChild(quotedHeader);
            quoted.appendChild(quotedContent);
            
            // Input
            const input = document.createElement('textarea');
            input.className = 'reply-popup-input';
            input.placeholder = 'Skriv ditt svar...';
            input.id = 'replyPopupInput';
            
            content.appendChild(quoted);
            content.appendChild(input);
            
            // Footer
            const footer = document.createElement('div');
            footer.className = 'reply-popup-footer';
            const cancelBtn = document.createElement('button');
            cancelBtn.className = 'reply-popup-btn reply-popup-btn-cancel';
            cancelBtn.title = 'Avbryt';
            cancelBtn.onclick = closeReplyPopup;
            // Lägg till X-ikon för cancel-knappen
            const cancelIcon = document.createElementNS('http://www.w3.org/2000/svg', 'svg');
            cancelIcon.setAttribute('xmlns', 'http://www.w3.org/2000/svg');
            cancelIcon.setAttribute('viewBox', '0 0 24 24');
            const cancelPath = document.createElementNS('http://www.w3.org/2000/svg', 'path');
            cancelPath.setAttribute('d', 'M19 6.41L17.59 5 12 10.59 6.41 5 5 6.41 10.59 12 5 17.59 6.41 19 12 13.41 17.59 19 19 17.59 13.41 12z');
            cancelIcon.appendChild(cancelPath);
            cancelBtn.appendChild(cancelIcon);
            const sendBtn = document.createElement('button');
            sendBtn.className = 'reply-popup-btn reply-popup-btn-send';
            sendBtn.onclick = function() {
                sendReplyMessage(messageId, input.value.trim());
            };
            // Lägg till SVG-ikon för send-knappen
            const sendIcon = document.createElementNS('http://www.w3.org/2000/svg', 'svg');
            sendIcon.setAttribute('xmlns', 'http://www.w3.org/2000/svg');
            sendIcon.setAttribute('viewBox', '0 0 24 24');
            const sendPath = document.createElementNS('http://www.w3.org/2000/svg', 'path');
            sendPath.setAttribute('d', 'M2.01 21L23 12 2.01 3 2 10l15 2-15 2z');
            sendIcon.appendChild(sendPath);
            sendBtn.appendChild(sendIcon);
            footer.appendChild(cancelBtn);
            footer.appendChild(sendBtn);
            
            popup.appendChild(header);
            popup.appendChild(content);
            popup.appendChild(footer);
            overlay.appendChild(popup);
            
            document.body.appendChild(overlay);
            
            // Fokusera på input
            setTimeout(() => {
                input.focus();
            }, 100);
            
            // Hantera Enter för att skicka (Shift+Enter för ny rad)
            input.addEventListener('keydown', function(e) {
                if (e.key === 'Enter' && !e.shiftKey) {
                    e.preventDefault();
                    sendBtn.click();
                }
            });
            
            // Spara referenser för att kunna stänga
            window.replyPopupOverlay = overlay;
            window.replyPopupMessageId = messageId;
        }
        
        // Funktion för att stänga reply-popup
        function closeReplyPopup() {
            if (window.replyPopupOverlay) {
                window.replyPopupOverlay.remove();
                window.replyPopupOverlay = null;
                window.replyPopupMessageId = null;
            }
        }
        
        // Funktion för att skicka reply-meddelande
        async function sendReplyMessage(replyToMessageId, messageText) {
            if (!messageText.trim()) {
                return;
            }
            
            // Förhindra att skicka meddelanden i arkiverade projekt
            if (isArchived) {
                alert('Detta projekt är arkiverat. Du kan inte skicka nya meddelanden.');
                closeReplyPopup();
                return;
            }
            
            // Uppdatera participantId från URL innan skickande
            participantId = getParticipantIdFromUrl();
            
            // Hämta valda notifieringar från notifiedParticipants
            const notifiedUsers = Array.from(notifiedParticipants);
            
            // Kontrollera om brådskande notifikation är aktiv
            const urgentNotification = window.urgentNotificationActive || false;
            
            const formData = new FormData();
            formData.append('action', 'send');
            formData.append('project_id', projectId);
            formData.append('message', messageText);
            formData.append('reply_to', replyToMessageId);
            if (participantId) {
                formData.append('user_id', participantId);
            }
            if (urgentNotification) {
                formData.append('priority', '2');
            }
            notifiedUsers.forEach(id => {
                formData.append('notified_users[]', id);
            });
            
            // Disable send button
            const sendBtn = document.querySelector('.reply-popup-btn-send');
            if (sendBtn) {
                sendBtn.disabled = true;
                // Behåll ikonen men lägg till text
                const existingText = sendBtn.querySelector('span');
                if (existingText) {
                    existingText.textContent = 'Skickar...';
                } else {
                    const textSpan = document.createElement('span');
                    textSpan.textContent = 'Skickar...';
                    sendBtn.innerHTML = '';
                    sendBtn.appendChild(textSpan);
                }
            }
            
            try {
                const res = await fetch('../api/chat/messages.php', {
                    method: 'POST',
                    body: formData
                });
                
                // Läs svaret som text först (kan bara läsas en gång)
                const text = await res.text();
                let data;
                
                // Kontrollera om användaren är inaktiv (403)
                if (res.status === 403) {
                    try {
                        data = JSON.parse(text);
                    } catch (e) {
                        data = { message: 'Din användare är inaktiv' };
                    }
                    alert(data.message || 'Din användare är inaktiv och du har loggats ut');
                    window.location.href = '../index.html';
                    return;
                }
                
                // Försök parsa JSON om det finns innehåll
                if (text.trim() === '') {
                    // Tomt svar - behandla som fel om status inte är OK
                    if (!res.ok) {
                        alert('Ett fel uppstod vid skickande av svar');
                        closeReplyPopup();
                        return;
                    }
                    // Om status är OK men tomt svar, behandla som framgång
                    data = { success: true };
                } else {
                    try {
                        data = JSON.parse(text);
                    } catch (parseError) {
                        // Om JSON-parsing misslyckas, visa felmeddelandet om det finns
                        if (!res.ok) {
                            alert('Ett fel uppstod vid skickande av svar: ' + (text || 'Okänt fel'));
                        } else {
                            alert('Ett fel uppstod vid kommunikation med servern');
                        }
                        closeReplyPopup();
                        return;
                    }
                }
                
                // Kontrollera om användaren är inaktiv
                if (!data.success && data.message && data.message.includes('inaktiv')) {
                    alert(data.message);
                    window.location.href = '../index.html';
                    return;
                }
                
                if (data.success) {
                    // Stäng popup
                    closeReplyPopup();
                    
                    // Återställ brådskande notifikation efter att meddelandet skickats
                    window.urgentNotificationActive = false;
                    const urgentIcon = document.getElementById('urgentNotification');
                    if (urgentIcon) {
                        urgentIcon.classList.remove('active');
                    }
                    
                    // Ladda om meddelanden
                    await loadMessages();
                    
                    // Scrolla till botten efter att meddelandet har skickats
                    const container = document.getElementById('messagesContainer');
                    if (container) {
                        container.scrollTop = container.scrollHeight;
                    }
                    
                    // Uppdatera scroll-to-bottom knappen
                    setTimeout(updateScrollToBottomButton, 100);
                } else {
                    alert('Fel: ' + (data.message || 'Okänt fel'));
                    if (sendBtn) {
                        sendBtn.disabled = false;
                        // Återställ ikonen
                        const sendIcon = document.createElementNS('http://www.w3.org/2000/svg', 'svg');
                        sendIcon.setAttribute('xmlns', 'http://www.w3.org/2000/svg');
                        sendIcon.setAttribute('viewBox', '0 0 24 24');
                        const sendPath = document.createElementNS('http://www.w3.org/2000/svg', 'path');
                        sendPath.setAttribute('d', 'M2.01 21L23 12 2.01 3 2 10l15 2-15 2z');
                        sendIcon.appendChild(sendPath);
                        sendBtn.innerHTML = '';
                        sendBtn.appendChild(sendIcon);
                    }
                }
            } catch (e) {
                console.error('Fel vid skickande:', e);
                alert('Ett fel uppstod vid skickande av meddelandet.');
                if (sendBtn) {
                    sendBtn.disabled = false;
                    // Återställ ikonen
                    const sendIcon = document.createElementNS('http://www.w3.org/2000/svg', 'svg');
                    sendIcon.setAttribute('xmlns', 'http://www.w3.org/2000/svg');
                    sendIcon.setAttribute('viewBox', '0 0 24 24');
                    const sendPath = document.createElementNS('http://www.w3.org/2000/svg', 'path');
                    sendPath.setAttribute('d', 'M2.01 21L23 12 2.01 3 2 10l15 2-15 2z');
                    sendIcon.appendChild(sendPath);
                    sendBtn.innerHTML = '';
                    sendBtn.appendChild(sendIcon);
                }
            }
        }
        
        function escapeHtmlKeepQuotes(text) {
            if (!text) return '';
            let str = String(text);
            // Först, hantera dubbel-escapeade citationstecken (i omvänd ordning)
            str = str.replace(/&amp;quot;/g, '"').replace(/&amp;#34;/g, '"');
            // Sedan, ersätt redan escapeade citationstecken tillbaka till vanliga
            str = str.replace(/&quot;/g, '"').replace(/&#34;/g, '"');
            // Escapea farliga HTML-tecken men behåll citationstecken
            // Viktigt: escapea & först, men bara om det inte redan är en entity
            return str
                .replace(/&(?!(?:amp|lt|gt|quot|#39|#34);)/g, '&amp;')
                .replace(/</g, '&lt;')
                .replace(/>/g, '&gt;')
                .replace(/'/g, '&#39;');
            // Vi lämnar " som det är eftersom det är säkert i HTML-innehåll
        }
        
        async function showParticipantCard(participantId) {
            if (!participantId) {
                console.warn('showParticipantCard: participantId saknas');
                return;
            }
            
            try {
                // Hämta användardata
                const response = await fetch(`participant_info_api.php?participant_id=${encodeURIComponent(participantId)}`);
                const data = await response.json();
                
                if (!data.success || !data.data) {
                    alert('Kunde inte hämta användarinformation');
                    return;
                }
                
                const userData = data.data;
                
                // Skapa popup-overlay
                const popup = document.createElement('div');
                popup.className = 'participant-card-popup';
                popup.style.cssText = `
                    position: fixed;
                    top: 0;
                    left: 0;
                    width: 100%;
                    height: 100%;
                    background: rgba(0,0,0,0.5);
                    display: flex;
                    align-items: center;
                    justify-content: center;
                    z-index: 10000;
                `;
                
                // Profilbild URL - använd API för både interna användare och externa partners
                let profileImageUrl = null;
                if (userData.profile_image) {
                    if (userData.type === 'user' || userData.type === 'admin') {
                        // Interna användare - använd userids_api.php
                        profileImageUrl = `../adm/api/users.php?action=get_image&user_id=${encodeURIComponent(participantId)}&t=${Date.now()}`;
                    } else if (userData.type === 'extern') {
                        // Externa partners - använd external_staff_api.php
                        profileImageUrl = `../api/chat/external_staff.php?action=get_image&id=${encodeURIComponent(participantId)}&t=${Date.now()}`;
                    }
                }
                
                // Skapa popup-innehåll
                const popupContent = document.createElement('div');
                popupContent.className = 'participant-card-content';
                popupContent.style.cssText = `
                    background: white;
                    border-radius: 12px;
                    padding: 30px;
                    max-width: 400px;
                    width: 90%;
                    box-shadow: 0 8px 32px rgba(0,0,0,0.3);
                    text-align: center;
                `;
                
                let html = '';
                
                // Profilbild
                html += '<div class="participant-card-image" style="margin-bottom: 20px; position: relative;">';
                if (profileImageUrl) {
                    html += `<img id="participantCardImage" src="${escapeHtml(profileImageUrl)}" alt="Profilbild" style="width: 240px; height: 240px; border-radius: 50%; object-fit: cover; border: 3px solid #e0e0e0; display: block; margin: 0 auto;" onerror="console.error('Bildladdningsfel:', this.src); this.style.display='none'; const placeholder = this.nextElementSibling; if(placeholder) placeholder.style.display='flex';">`;
                }
                html += `<div class="participant-card-placeholder" id="participantCardPlaceholder" style="width: 240px; height: 240px; border-radius: 50%; background: #f0f0f0; display: ${profileImageUrl ? 'none' : 'flex'}; align-items: center; justify-content: center; margin: 0 auto; border: 3px solid #e0e0e0;">`;
                html += '<svg width="120" height="120" viewBox="0 0 24 24" fill="#999"><path d="M12 12c2.21 0 4-1.79 4-4s-1.79-4-4-4-4 1.79-4 4 1.79 4 4 4zm0 2c-2.67 0-8 1.34-8 4v2h16v-2c0-2.66-5.33-4-8-4z"/></svg>';
                html += '</div>';
                html += '</div>';
                
                // Namn
                html += `<h3 style="margin: 0 0 12px 0; color: #333; font-size: 24px;">${escapeHtml(userData.name)}</h3>`;
                
                // Yrkesroll
                if (userData.role) {
                    html += '<div class="participant-card-info" style="margin-bottom: 16px;">';
                    html += `<span style="color: #666; font-size: 16px; font-style: italic;">${escapeHtml(userData.role)}</span>`;
                    html += '</div>';
                }
                
                // Telefonnummer (klickbar länk)
                if (userData.mobile) {
                    // Rensa telefonnumret från eventuella blanksteg och specialtecken för tel:-länken
                    const cleanPhone = userData.mobile.replace(/[\s\-\(\)]/g, '');
                    html += '<div class="participant-card-info" style="margin-bottom: 16px; display: flex; align-items: center; justify-content: center; gap: 8px;">';
                    html += '<svg width="20" height="20" viewBox="0 0 24 24" fill="#666"><path d="M6.62 10.79c1.44 2.83 3.76 5.14 6.59 6.59l2.2-2.2c.27-.27.67-.36 1.02-.24 1.12.37 2.33.57 3.57.57.55 0 1 .45 1 1V20c0 .55-.45 1-1 1-9.39 0-17-7.61-17-17 0-.55.45-1 1-1h3.5c.55 0 1 .45 1 1 0 1.25.2 2.45.57 3.57.11.35.03.74-.25 1.02l-2.2 2.2z"/></svg>';
                    html += `<a href="tel:${escapeHtml(cleanPhone)}" style="color: #1a73e8; font-size: 16px; text-decoration: none; font-weight: 500;">${escapeHtml(userData.mobile)}</a>`;
                    html += '</div>';
                }
                
                // Företag (endast för externa användare)
                if (userData.type === 'extern' && userData.company_name) {
                    html += '<div class="participant-card-info" style="margin-bottom: 16px; display: flex; align-items: center; justify-content: center; gap: 8px;">';
                    html += '<svg width="20" height="20" viewBox="0 0 24 24" fill="#666"><path d="M12 7V3H2v18h20V7H12zM6 19H4v-2h2v2zm0-4H4v-2h2v2zm0-4H4V9h2v2zm0-4H4V5h2v2zm4 12H8v-2h2v2zm0-4H8v-2h2v2zm0-4H8V9h2v2zm0-4H8V5h2v2zm10 12h-8v-2h2v-2h-2v-2h2v-2h-2V9h8v10zm-2-8h-2v2h2v-2zm0 4h-2v2h2v-2z"/></svg>';
                    html += `<span style="color: #666; font-size: 16px;">${escapeHtml(userData.company_name)}</span>`;
                    html += '</div>';
                }
                
                // Publika anteckningar
                if (userData.public_notes) {
                    html += '<div class="participant-card-info" style="margin-top: 20px; margin-bottom: 16px; padding: 16px; background: #f5f5f5; border-radius: 8px; text-align: left;">';
                    html += `<div style="color: #333; font-size: 14px; line-height: 1.5; white-space: pre-wrap;">${escapeHtml(userData.public_notes)}</div>`;
                    html += '</div>';
                }
                
                // Stäng-knapp
                html += '<div style="margin-top: 24px;">';
                html += '<button onclick="this.closest(\'.participant-card-popup\').remove()" style="padding: 10px 20px; background: #1a73e8; color: white; border: none; border-radius: 8px; cursor: pointer; font-size: 14px; font-weight: 500;">Stäng</button>';
                html += '</div>';
                
                popupContent.innerHTML = html;
                popup.appendChild(popupContent);
                document.body.appendChild(popup);
                
                // Stäng popup när man klickar utanför
                popup.addEventListener('click', (e) => {
                    if (e.target === popup) {
                        popup.remove();
                    }
                });
                
            } catch (error) {
                console.error('Fel vid hämtning av användarinformation:', error);
                alert('Ett fel uppstod vid hämtning av användarinformation');
            }
        }
        
        async function showManageParticipants() {
            if (!isAdmin) return;
            
            const modal = document.getElementById('manageParticipantsModal');
            const content = document.getElementById('manageParticipantsContent');
            content.innerHTML = '<p>Laddar...</p>';
            modal.style.display = 'flex';
            
            try {
                participantId = getParticipantIdFromUrl();
                
                // Ladda externa partners
                const externalStaffRes = await fetch('../api/chat/external_staff.php?action=list&user_id=' + encodeURIComponent(participantId));
                let externalStaff = [];
                
                if (externalStaffRes.ok) {
                    const externalStaffData = await externalStaffRes.json();
                    if (externalStaffData.success) {
                        externalStaff = externalStaffData.staff || [];
                    }
                }
                
                // Ladda interna användare
                const internalUsersRes = await fetch('../api/user_info.php?action=list&user_id=' + encodeURIComponent(participantId));
                let internalUsers = [];
                
                if (internalUsersRes.ok) {
                    const internalUsersData = await internalUsersRes.json();
                    if (internalUsersData.success) {
                        internalUsers = internalUsersData.users || [];
                    }
                }
                
                // Hämta aktuella deltagare-ID:n
                const currentParticipantIds = participants.map(p => {
                    return p.participant_id || null;
                }).filter(id => id !== null);
                
                // Hämta den inloggade användarens ID
                const currentUserId = participantId || '';
                
                // Filtrera bort inaktiva användare och externa användare (extern ska vara i externalStaff)
                const activeUsers = internalUsers.filter(user => {
                    const active = user.active ?? true;
                    const userType = user.user_type || 'user';
                    // Exkludera externa användare - de ska vara i externalStaff
                    if (userType === 'extern') {
                        return false;
                    }
                    return active === true || active === 1 || active === '1' || active === 'true';
                });
                
                // Filtrera bort inaktiva staff
                const activeStaff = externalStaff.filter(staff => {
                    const active = staff.active ?? true;
                    return active === true || active === 1 || active === '1' || active === 'true';
                });
                
                // Separera användare i grupper: admin, user, extern
                const adminUsers = [];
                const regularUsers = [];
                const externalUsers = [];
                
                // Lägg till interna användare (separera admin och user)
                activeUsers.forEach(user => {
                    const isParticipant = currentParticipantIds.includes(user.user_id);
                    const isAdmin = isParticipantAdmin(user.user_id);
                    const isCurrentUser = user.user_id === currentUserId;
                    // Visa full_name följt av "nickname" om nickname finns
                    const fullName = user.full_name || user.user_id || 'Okänd användare';
                    const nickname = (user.nickname && user.nickname.trim() !== '') ? user.nickname : null;
                    const displayName = nickname ? `${fullName} "${nickname}"` : fullName;
                    
                    const userObj = {
                        id: user.user_id,
                        name: displayName,
                        fullName: fullName,
                        nickname: nickname,
                        type: 'user',
                        isParticipant: isParticipant,
                        isAdmin: isAdmin,
                        isCurrentUser: isCurrentUser,
                        backgroundColor: '#c8e6c9',
                        pushover_user_key: user.pushover_user_key || ''
                    };
                    
                    if (isAdmin) {
                        adminUsers.push(userObj);
                    } else {
                        regularUsers.push(userObj);
                    }
                });
                
                // Lägg till externa partners
                activeStaff.forEach(staff => {
                    const staffId = staff.id || staff.user_id;
                    const isParticipant = currentParticipantIds.includes(staffId);
                    // Visa name/full_name följt av "nickname" om nickname finns
                    const fullName = staff.name || staff.full_name || staffId || 'Okänd partner';
                    const nickname = (staff.nickname && staff.nickname.trim() !== '') ? staff.nickname : null;
                    const displayName = nickname ? `${fullName} "${nickname}"` : fullName;
                    
                    externalUsers.push({
                        id: staffId,
                        name: displayName,
                        fullName: fullName,
                        nickname: nickname,
                        type: 'extern',
                        isParticipant: isParticipant,
                        isAdmin: false,
                        backgroundColor: '#ffcdd2',
                        pushover_user_key: staff.pushover_user_key || ''
                    });
                });
                
                // Sortera varje grupp alfabetiskt på namn
                adminUsers.sort((a, b) => (a.name || '').toLowerCase().localeCompare((b.name || '').toLowerCase()));
                regularUsers.sort((a, b) => (a.name || '').toLowerCase().localeCompare((b.name || '').toLowerCase()));
                externalUsers.sort((a, b) => (a.name || '').toLowerCase().localeCompare((b.name || '').toLowerCase()));
                
                let html = '<h3>Välj deltagare</h3>';
                
                // Funktion för att rendera en grupp
                function renderGroup(title, users, groupType) {
                    if (users.length === 0) return '';
                    
                    let groupHtml = `<h4 style="margin-top: 20px; margin-bottom: 10px; color: #333; font-size: 16px; font-weight: 600;">${title}</h4>`;
                    groupHtml += '<ul style="list-style: none; padding: 0; margin-bottom: 20px;">';
                    
                    users.forEach(user => {
                        const isChecked = user.isParticipant;
                        // Disable endast om det är admin OCH den inloggade användaren
                        const isDisabled = user.isAdmin && user.isCurrentUser;
                        const checkboxId = `participant_${user.type}_${user.id.replace(/[^a-zA-Z0-9]/g, '_')}`;
                        
                        // Kontrollera om notifieringar är aktiverade för denna deltagare
                        const isNotified = notifiedParticipants.has(user.id);
                        
                        // Kontrollera om användaren har pushover_user_key
                        // Kontrollera om pushover_user_key finns och inte är tom
                        const pushoverKey = user.pushover_user_key;
                        const hasPushover = pushoverKey !== undefined && pushoverKey !== null && pushoverKey !== '' && String(pushoverKey).trim() !== '';
                        
                        // Bestäm bakgrundsfärg baserat på om deltagaren är vald och typ
                        // Externa: röd när vald, ljusröd när ej vald
                        // Interna: grön när vald, ljusgrön när ej vald
                        let backgroundColor;
                        if (user.type === 'extern') {
                            // Externa partners: röd/ljusröd
                            backgroundColor = isChecked ? '#ffcdd2' : '#ffebee';
                        } else {
                            // Interna användare: grön/ljusgrön
                            backgroundColor = isChecked ? '#c8e6c9' : '#e8f5e9';
                        }
                        
                        // Alarmklocka-ikon - visa alltid, överkryssad om saknar Pushover eller notifieringar av
                        const canNotify = hasPushover && isNotified;
                        const alarmIconOpacity = canNotify ? '1' : '0.5';
                        const alarmTitle = !hasPushover ? 'Saknar Pushover ID' : (isNotified ? 'Notifieras' : 'Notifieras inte');
                        const alarmIcon = `
                            <span style="display: flex; align-items: center; margin-left: auto; cursor: ${isChecked ? 'pointer' : 'not-allowed'};"
                                  onclick="${isChecked ? `handleAlarmIconClick(event, '${escapeHtml(user.id).replace(/'/g, "\\'")}')` : ''}"
                                  title="${alarmTitle}">
                                <svg width="18" height="18" viewBox="0 0 24 24" fill="currentColor" 
                                     style="color: #666; flex-shrink: 0; position: relative; opacity: ${alarmIconOpacity};">
                                    <path d="M12 22c1.1 0 2-.9 2-2h-4a2 2 0 0 0 2 2zm6-6V11c0-3.07-1.63-5.64-4.5-6.32V4a1.5 1.5 0 0 0-3 0v.68C7.63 5.36 6 7.92 6 11v5l-1.7 1.7a1 1 0 0 0 .7 1.7h14a1 1 0 0 0 .7-1.7L18 16z"/>
                                    ${!canNotify ? '<line x1="4" y1="4" x2="20" y2="20" stroke="#d32f2f" stroke-width="2.5" stroke-linecap="round" opacity="1"/>' : ''}
                                </svg>
                            </span>
                        `;
                        
                        groupHtml += `<li style="padding: 8px 12px; margin-bottom: 6px; border-radius: 6px; background: ${backgroundColor}; display: flex; align-items: center; gap: 12px;">
                            <input type="checkbox" 
                                   id="${checkboxId}"
                                   data-type="${user.type}"
                                   data-id="${escapeHtml(user.id)}"
                                   ${isChecked ? 'checked' : ''}
                                   ${isDisabled ? 'disabled' : ''}
                                   onchange="handleParticipantCheckboxChange(event, '${escapeHtml(user.id)}', '${user.type}', ${isChecked})"
                                   style="width: 20px; height: 20px; cursor: ${isDisabled ? 'not-allowed' : 'pointer'};">
                            <label for="${checkboxId}" 
                                   style="flex: 1; cursor: default; margin: 0;">
                                ${escapeHtml(user.name)}${user.isAdmin ? ' <span style="color: #999; font-size: 12px;">(Admin)</span>' : ''}${isDisabled ? ' <span style="color: #999; font-size: 12px;">(Du)</span>' : ''}
                            </label>
                            ${alarmIcon}
                        </li>`;
                    });
                    
                    groupHtml += '</ul>';
                    return groupHtml;
                }
                
                // Rendera grupper i ordning: Admin, User, Extern
                html += renderGroup('Administratörer', adminUsers, 'admin');
                html += renderGroup('Användare', regularUsers, 'user');
                html += renderGroup('Externa partners', externalUsers, 'extern');
                html += '<p style="color: #666; font-size: 14px; margin-top: 10px;">Gå till <a href="../adm/extusers/external_parties.php?user_id=' + encodeURIComponent(participantId) + '">Extern partner</a> för att skapa extern partner.</p>';
                
                content.innerHTML = html;
            } catch (e) {
                console.error('Fel vid laddning:', e);
                content.innerHTML = '<p style="color: red;">Fel vid laddning. Kontrollera att du har rättigheter.</p>';
            }
        }
        
        async function handleParticipantCheckboxChange(event, id, type, wasChecked) {
            // Förhindra att label-klicket triggas
            event.stopPropagation();
            
            const checkbox = event.target;
            const isChecked = checkbox.checked;
            
            // Om kryssrutan är ikryssad, lägg till deltagaren
            if (isChecked && !wasChecked) {
                try {
                    await addParticipant(type, id, true); // true = skipModalUpdate
                    // Uppdatera bakgrundsfärgen direkt
                    const li = checkbox.closest('li');
                    if (li) {
                        // Vald: mörkare färg (röd för externa, grön för interna)
                        const backgroundColor = type === 'extern' ? '#ffcdd2' : '#c8e6c9';
                        li.style.background = backgroundColor;
                    }
                } catch (e) {
                    // Återställ kryssrutan om det misslyckades
                    checkbox.checked = false;
                    console.error('Fel vid läggning till deltagare:', e);
                    alert('Kunde inte lägga till deltagare: ' + (e.message || 'Okänt fel'));
                }
            } 
            // Om kryssrutan är avkryssad, ta bort deltagaren
            else if (!isChecked && wasChecked) {
                try {
                    await removeParticipant(type, id, true); // true = skipModalUpdate
                    // Uppdatera bakgrundsfärgen direkt
                    const li = checkbox.closest('li');
                    if (li) {
                        // Ej vald: ljusare färg (ljusröd för externa, ljusgrön för interna)
                        const backgroundColor = type === 'extern' ? '#ffebee' : '#e8f5e9';
                        li.style.background = backgroundColor;
                    }
                } catch (e) {
                    // Återställ kryssrutan om det misslyckades
                    checkbox.checked = true;
                    console.error('Fel vid borttagning av deltagare:', e);
                    alert('Kunde inte ta bort deltagare: ' + (e.message || 'Okänt fel'));
                }
            }
            
            // Save state after participant change
            saveParticipantState();
        }
        
        async function handleAlarmIconClick(event, participantId) {
            // Förhindra att andra klickhändelser triggas
            event.preventDefault();
            event.stopPropagation();
            
            // Växla notifieringsstatus direkt
            if (notifiedParticipants.has(participantId)) {
                notifiedParticipants.delete(participantId);
            } else {
                notifiedParticipants.add(participantId);
            }
            
            // Uppdatera huvudvyn (etiketterna)
            renderParticipants();
            
            const isNotified = notifiedParticipants.has(participantId);
            
            // Hitta SVG-elementet (kan vara inuti span som är target)
            let svg = event.target.querySelector('svg');
            if (!svg) {
                svg = event.target.closest('span')?.querySelector('svg');
            }
            if (!svg) {
                svg = event.target.closest('svg');
            }
            
            if (svg) {
                // Ta bort befintlig överkryssningslinje om den finns
                const existingLine = svg.querySelector('line');
                if (existingLine) {
                    existingLine.remove();
                }
                
                // Uppdatera opacity och lägg till/ta bort överkryssningslinje
                if (isNotified) {
                    svg.style.opacity = '1';
                } else {
                    svg.style.opacity = '0.5';
                    // Lägg till överkryssningslinje
                    const line = document.createElementNS('http://www.w3.org/2000/svg', 'line');
                    line.setAttribute('x1', '4');
                    line.setAttribute('y1', '4');
                    line.setAttribute('x2', '20');
                    line.setAttribute('y2', '20');
                    line.setAttribute('stroke', '#d32f2f');
                    line.setAttribute('stroke-width', '2.5');
                    line.setAttribute('stroke-linecap', 'round');
                    line.setAttribute('opacity', '1');
                    svg.appendChild(line);
                }
                
                // Uppdatera title
                svg.setAttribute('title', isNotified ? 'Notifieras - klicka för att stänga av' : 'Notifieras inte - klicka för att slå på');
            }
            
            // Save state after notification change
            saveParticipantState();
        }
        
        async function saveAndCloseManageParticipants() {
            // Eftersom ändringar görs direkt när kryssrutor ändras, behöver vi bara stänga modalen
            const modal = document.getElementById('manageParticipantsModal');
            if (modal) {
                modal.style.display = 'none';
            }
        }
        
        function closeManageParticipants() {
            document.getElementById('manageParticipantsModal').style.display = 'none';
        }
        
        function showChatHelp() {
            const modal = document.getElementById('chatHelpModal');
            modal.style.display = 'flex';
        }
        
        function closeChatHelp() {
            document.getElementById('chatHelpModal').style.display = 'none';
        }
        
        // Sökfunktion - ESC handler för att kunna ta bort den senare
        let searchEscHandler = null;
        
        function showSearchModal() {
            // Skapa modal om den inte finns
            let modal = document.getElementById('searchModal');
            if (!modal) {
                modal = document.createElement('div');
                modal.id = 'searchModal';
                modal.className = 'search-modal';
                modal.innerHTML = `
                    <div class="search-modal-content">
                        <div class="search-modal-header">
                            <h3 class="search-modal-title">Sök i chatten</h3>
                            <button class="search-modal-close" onclick="closeSearchModal()" title="Stäng">×</button>
                        </div>
                        <div class="search-input-container">
                            <input type="text" id="searchInput" class="search-input" placeholder="Sök efter text..." autocomplete="off">
                        </div>
                        <div class="search-results" id="searchResults">
                            <div class="search-no-results">Skriv för att söka...</div>
                        </div>
                    </div>
                `;
                document.body.appendChild(modal);
                
                // Event listeners
                const searchInput = document.getElementById('searchInput');
                searchInput.addEventListener('input', handleSearchInput);
                searchInput.addEventListener('keydown', handleSearchKeydown);
                
                // ESC-tangent för att stänga
                modal.addEventListener('click', (e) => {
                    if (e.target === modal) {
                        closeSearchModal();
                    }
                });
            }
            
            // Lägg till ESC-tangent listener
            if (searchEscHandler) {
                document.removeEventListener('keydown', searchEscHandler);
            }
            searchEscHandler = (e) => {
                if (e.key === 'Escape') {
                    closeSearchModal();
                }
            };
            document.addEventListener('keydown', searchEscHandler);
            
            modal.style.display = 'flex';
            const searchInput = document.getElementById('searchInput');
            if (searchInput) {
                searchInput.focus();
            }
        }
        
        function closeSearchModal() {
            const modal = document.getElementById('searchModal');
            if (modal) {
                modal.style.display = 'none';
                const searchInput = document.getElementById('searchInput');
                if (searchInput) {
                    searchInput.value = '';
                }
                const results = document.getElementById('searchResults');
                if (results) {
                    results.innerHTML = '<div class="search-no-results">Skriv för att söka...</div>';
                }
            }
            // Ta bort ESC-tangent listener
            if (searchEscHandler) {
                document.removeEventListener('keydown', searchEscHandler);
                searchEscHandler = null;
            }
        }
        
        function handleSearchInput(e) {
            const searchTerm = e.target.value.trim();
            if (searchTerm.length === 0) {
                const results = document.getElementById('searchResults');
                if (results) {
                    results.innerHTML = '<div class="search-no-results">Skriv för att söka...</div>';
                }
                return;
            }
            
            const matchingMessages = performSearch(searchTerm);
            displaySearchResults(matchingMessages, searchTerm);
        }
        
        function handleSearchKeydown(e) {
            if (e.key === 'Enter') {
                const results = document.getElementById('searchResults');
                const firstResult = results.querySelector('.search-result-item');
                if (firstResult) {
                    firstResult.click();
                }
            }
        }
        
        function performSearch(searchTerm) {
            if (!messages || messages.length === 0) {
                return [];
            }
            
            const searchLower = searchTerm.toLowerCase();
            const matching = [];
            
            messages.forEach(msg => {
                let matches = false;
                let matchedText = '';
                
                // Sök i meddelandetext
                if (msg.message && typeof msg.message === 'string') {
                    const msgLower = msg.message.toLowerCase();
                    if (msgLower.includes(searchLower)) {
                        matches = true;
                        matchedText = msg.message;
                    }
                }
                
                // Sök i systemmeddelanden
                if (!matches && msg.system_message && typeof msg.system_message === 'string') {
                    const sysLower = msg.system_message.toLowerCase();
                    if (sysLower.includes(searchLower)) {
                        matches = true;
                        matchedText = msg.system_message;
                    }
                }
                
                // Sök i avsändarens namn
                if (!matches && msg.sender_name && typeof msg.sender_name === 'string') {
                    const senderLower = msg.sender_name.toLowerCase();
                    if (senderLower.includes(searchLower)) {
                        matches = true;
                        matchedText = msg.sender_name;
                    }
                }
                
                if (matches) {
                    matching.push({
                        message: msg,
                        matchedText: matchedText
                    });
                }
            });
            
            return matching;
        }
        
        function highlightSearchTerm(text, searchTerm) {
            if (!text || !searchTerm) return escapeHtml(text || '');
            
            const regex = new RegExp(`(${escapeRegex(searchTerm)})`, 'gi');
            const escapedText = escapeHtml(text);
            return escapedText.replace(regex, '<span class="search-highlight">$1</span>');
        }
        
        function escapeRegex(str) {
            return str.replace(/[.*+?^${}()|[\]\\]/g, '\\$&');
        }
        
        function displaySearchResults(matchingMessages, searchTerm) {
            const results = document.getElementById('searchResults');
            if (!results) return;
            
            if (matchingMessages.length === 0) {
                results.innerHTML = '<div class="search-no-results">Inga resultat hittades</div>';
                return;
            }
            
            let html = '';
            matchingMessages.forEach((item, index) => {
                const msg = item.message;
                const senderName = msg.sender_name || msg.participant_id || 'Okänd';
                const msgDate = new Date(msg.created_at);
                const dateStr = msgDate.toLocaleString('sv-SE');
                
                // Hämta text att visa
                let displayText = '';
                if (msg.message) {
                    displayText = msg.message;
                } else if (msg.system_message) {
                    displayText = msg.system_message;
                } else if (msg.type === 'image') {
                    displayText = '[Bild]';
                } else if (msg.type === 'document') {
                    displayText = '[Dokument]';
                } else {
                    displayText = '[Meddelande]';
                }
                
                // Begränsa textlängd för visning
                if (displayText.length > 150) {
                    displayText = displayText.substring(0, 150) + '...';
                }
                
                const highlightedText = highlightSearchTerm(displayText, searchTerm);
                const highlightedSender = highlightSearchTerm(senderName, searchTerm);
                
                html += `<div class="search-result-item" onclick="scrollToMessage('${escapeHtml(msg.id)}')">`;
                html += `<div class="search-result-header">`;
                html += `<span class="search-result-sender">${highlightedSender}</span>`;
                html += `<span class="search-result-date">${escapeHtml(dateStr)}</span>`;
                html += `</div>`;
                html += `<div class="search-result-text">${highlightedText}</div>`;
                html += `</div>`;
            });
            
            results.innerHTML = html;
        }
        
        function scrollToMessage(messageId) {
            closeSearchModal();
            
            // Vänta lite för att modalen ska stängas
            setTimeout(() => {
                const container = document.getElementById('messagesContainer');
                if (!container) return;
                
                const messageElement = container.querySelector(`[data-message-id="${escapeHtml(messageId)}"]`);
                if (!messageElement) {
                    // Om meddelandet inte är synligt (t.ex. pga pin-filter), rendera om utan filter
                    if (window.pinnedFilterActive) {
                        window.pinnedFilterActive = false;
                        const icon = document.getElementById('pinnedFilterIcon');
                        if (icon) {
                            icon.classList.remove('active');
                            icon.setAttribute('title', 'Visa endast fästa meddelanden');
                        }
                        renderMessages();
                        // Försök igen efter rendering
                        setTimeout(() => {
                            const msgEl = container.querySelector(`[data-message-id="${escapeHtml(messageId)}"]`);
                            if (msgEl) {
                                scrollToMessageElement(msgEl);
                            }
                        }, 300);
                    }
                    return;
                }
                
                scrollToMessageElement(messageElement);
            }, 100);
        }
        
        function scrollToMessageElement(messageElement) {
            const container = document.getElementById('messagesContainer');
            if (!container || !messageElement) return;
            
            // Lägg till pulserande animation-klass
            messageElement.classList.add('message-highlight-pulse');
            
            // Scrolla till meddelandet
            messageElement.scrollIntoView({ behavior: 'smooth', block: 'center' });
            
            // Ta bort animation-klassen efter animationen är klar (4 sekunder)
            setTimeout(() => {
                messageElement.classList.remove('message-highlight-pulse');
            }, 4000);
        }
        
        // Flagga för att förhindra flera samtidiga anrop
        let isAddingParticipant = false;
        
        async function addParticipant(type, id, skipModalUpdate = false) {
            // Förhindra flera samtidiga anrop
            if (isAddingParticipant) {
                return;
            }
            
            isAddingParticipant = true;
            
            // Uppdatera participantId från URL innan användning
            participantId = getParticipantIdFromUrl();
            
            try {
                const formData = new FormData();
                formData.append('action', 'add');
                formData.append('project_id', projectId);
                formData.append('type', type);
                formData.append('participant_id', id);
                formData.append('user_id', participantId);
                
                const res = await fetch('../api/chat/participants.php', {
                    method: 'POST',
                    body: formData
                });
                
                const data = await res.json();
                if (data.success) {
                    // Ladda om deltagare och meddelanden för att visa den nya deltagaren
                    await loadParticipants();
                    await loadMessages();
                    // Uppdatera modalen endast om skipModalUpdate är false
                    if (!skipModalUpdate) {
                        showManageParticipants();
                    }
                } else {
                    throw new Error(data.message || 'Okänt fel');
                }
            } catch (e) {
                console.error('Fel:', e);
                throw e; // Kasta vidare så att handleParticipantCheckboxChange kan hantera det
            } finally {
                isAddingParticipant = false;
            }
        }
        
        async function removeParticipant(type, id, skipModalUpdate = false) {
            // Kontrollera om deltagaren är admin och den inloggade användaren
            // Förhindra borttagning endast om det är admin OCH den inloggade användaren
            participantId = getParticipantIdFromUrl();
            if (isParticipantAdmin(id) && id === participantId) {
                throw new Error('Du kan inte ta bort dig själv från projektet');
            }
            
            // Uppdatera participantId från URL innan användning
            participantId = getParticipantIdFromUrl();
            
            try {
                const formData = new FormData();
                formData.append('action', 'remove');
                formData.append('project_id', projectId);
                formData.append('type', type);
                formData.append('participant_id', id);
                formData.append('user_id', participantId);
                
                const res = await fetch('../api/chat/participants.php', {
                    method: 'POST',
                    body: formData
                });
                
                const data = await res.json();
                if (data.success) {
                    await loadParticipants();
                    await loadMessages();
                    // Uppdatera modalen endast om skipModalUpdate är false
                    if (!skipModalUpdate) {
                        showManageParticipants();
                    }
                } else {
                    throw new Error(data.message || 'Okänt fel');
                }
            } catch (e) {
                console.error('Fel:', e);
                throw e; // Kasta vidare så att handleParticipantCheckboxChange kan hantera det
            }
        }
        
        // Event listener för projektdropdown
        document.addEventListener('DOMContentLoaded', function() {
            const projectSelect = document.getElementById('projectSelect');
            if (projectSelect) {
                projectSelect.addEventListener('change', async function() {
                    const newProjectId = this.value;
                    if (newProjectId && newProjectId !== projectId) {
                        // Clear state when changing projects
                        clearParticipantState();
                        
                        try {
                            const currentUserId = getParticipantIdFromUrl();
                            
                            // Om användaren är user (inte admin eller extern) och behöver läggas till som deltagare
                            if (isUserType && currentUserId) {
                                // Kontrollera om användaren redan är deltagare
                                const checkRes = await fetch(`../api/chat/participants.php?action=list&project_id=${encodeURIComponent(newProjectId)}&user_id=${encodeURIComponent(currentUserId)}`);
                                const checkData = await checkRes.json();
                                
                                if (checkData.success) {
                                    const isParticipant = checkData.participants.some(p => p.participant_id === currentUserId);
                                    
                                    if (!isParticipant) {
                                        // Lägg till användaren som deltagare
                                        const addFormData = new FormData();
                                        addFormData.append('action', 'add');
                                        addFormData.append('project_id', newProjectId);
                                        addFormData.append('type', 'user');
                                        addFormData.append('participant_id', currentUserId);
                                        addFormData.append('user_id', currentUserId);
                                        
                                        const addRes = await fetch('../api/chat/participants.php', {
                                            method: 'POST',
                                            body: addFormData
                                        });
                                        
                                        const addData = await addRes.json();
                                        if (!addData.success) {
                                            console.error('Kunde inte lägga till användare i projekt:', addData.message);
                                            alert('Kunde inte lägga till dig i projektet: ' + addData.message);
                                            // Återställ dropdown till nuvarande projekt
                                            this.value = projectId;
                                            return;
                                        }
                                    }
                                }
                            }
                            
                            // Uppdatera URL och redirecta
                            const url = new URL(window.location.href);
                            url.searchParams.set('project_id', newProjectId);
                            if (currentUserId) {
                                url.searchParams.set('user_id', currentUserId);
                            }
                            window.location.href = url.toString();
                        } catch (e) {
                            console.error('Fel vid projektbyte:', e);
                            alert('Ett fel uppstod vid byte av projekt');
                            // Återställ dropdown till nuvarande projekt
                            this.value = projectId;
                        }
                    }
                });
            }
            
            // Sätt header-positionen direkt under logo-header (inga mellanrum)
            let logoHeaderHeight = 0;
            let headerHeight = 0;
            
            function updateHeaderPosition() {
                const logoHeader = document.querySelector('.logo-header');
                const header = document.querySelector('.header');
                if (logoHeader && header) {
                    // Använd getBoundingClientRect för exakt positionering
                    const logoRect = logoHeader.getBoundingClientRect();
                    logoHeaderHeight = logoRect.height;
                    const logoBottom = logoRect.bottom;
                    
                    // Kontrollera om logo är dold
                    const isLogoHidden = logoHeader.classList.contains('scrolled-up');
                    
                    if (isLogoHidden) {
                        // När logo är dold, placera header i toppen
                        header.style.top = '0px';
                        header.classList.add('at-top');
                    } else {
                        // När logo är synlig, placera header under logo
                        header.style.top = logoBottom + 'px';
                        header.classList.remove('at-top');
                    }
                    
                    // Uppdatera padding-top på messages-container
                    const messagesContainer = document.getElementById('messagesContainer');
                    if (messagesContainer) {
                        const headerRect = header.getBoundingClientRect();
                        headerHeight = headerRect.height;
                        const headerBottom = headerRect.bottom;
                        messagesContainer.style.paddingTop = headerBottom + 20 + 'px';
                    }
                }
            }
            
            // Hantera scroll för att dölja/visa logo-header
            function handleScroll() {
                const messagesContainer = document.getElementById('messagesContainer');
                const logoHeader = document.querySelector('.logo-header');
                if (!messagesContainer || !logoHeader) return;
                
                const scrollTop = messagesContainer.scrollTop;
                const scrollThreshold = 50; // Starta döljning efter 50px scroll
                
                if (scrollTop > scrollThreshold) {
                    // Scrolla uppåt - dölj logo
                    logoHeader.classList.add('scrolled-up');
                } else {
                    // Scrolla tillbaka till toppen - visa logo
                    logoHeader.classList.remove('scrolled-up');
                }
                
                // Uppdatera header-positionen
                updateHeaderPosition();
            }
            
            // Kör direkt och efter laddning
            setTimeout(updateHeaderPosition, 0); // Vänta på att DOM är helt renderad
            window.addEventListener('resize', updateHeaderPosition);
            window.addEventListener('load', updateHeaderPosition);
            
            // Lägg till scroll-event listener
            const messagesContainer = document.getElementById('messagesContainer');
            if (messagesContainer) {
                messagesContainer.addEventListener('scroll', handleScroll);
            }
            
            const modal = document.getElementById('manageParticipantsModal');
            if (modal) {
                modal.addEventListener('click', function(e) {
                    if (e.target === modal) {
                        closeManageParticipants();
                    }
                });
            }
            
            const helpModal = document.getElementById('chatHelpModal');
            if (helpModal) {
                helpModal.addEventListener('click', function(e) {
                    if (e.target === helpModal) {
                        closeChatHelp();
                    }
                });
            }
        });
        
        // Funktion för att visa ett systemmeddelande och vänta på kvittering
        async function showSystemMessage(message) {
            return new Promise((resolve) => {
                const overlay = document.getElementById('systemMessageOverlay');
                const content = document.getElementById('systemMessageContent');
                const btn = document.getElementById('systemMessageAcknowledgeBtn');
                
                if (overlay && content) {
                    content.textContent = message.message;
                    overlay.style.display = 'flex';
                    
                    // Blockera scrollning
                    document.body.style.overflow = 'hidden';
                    
                    // När användaren klickar på "Jag förstår"
                    const handleClick = async () => {
                        const currentUserId = getParticipantIdFromUrl();
                        
                        try {
                            const formData = new FormData();
                            formData.append('action', 'acknowledge');
                            formData.append('message_id', message.id);
                            formData.append('user_id', currentUserId || '');
                            
                            const ackResponse = await fetch('../api/system_message.php', {
                                method: 'POST',
                                body: formData
                            });
                            
                            const ackData = await ackResponse.json();
                            
                            if (ackData.success) {
                                // Dölj modal och återställ scrollning
                                overlay.style.display = 'none';
                                document.body.style.overflow = '';
                                btn.removeEventListener('click', handleClick);
                                resolve();
                            } else {
                                console.error('Fel vid kvittering:', ackData.message);
                                // Dölj modal ändå för att inte blockera användaren
                                overlay.style.display = 'none';
                                document.body.style.overflow = '';
                                btn.removeEventListener('click', handleClick);
                                resolve();
                            }
                        } catch (error) {
                            console.error('Fel vid kvittering:', error);
                            // Dölj modal ändå för att inte blockera användaren
                            overlay.style.display = 'none';
                            document.body.style.overflow = '';
                            btn.removeEventListener('click', handleClick);
                            resolve();
                        }
                    };
                    
                    btn.addEventListener('click', handleClick);
                } else {
                    resolve();
                }
            });
        }
        
        // Funktion för att kontrollera och visa alla olästa systemmeddelanden
        async function checkSystemMessage() {
            const currentUserId = getParticipantIdFromUrl();
            
            try {
                const response = await fetch(`../api/system_message.php?action=get&user_id=${encodeURIComponent(currentUserId || '')}`);
                const data = await response.json();
                
                if (data.success && data.messages && data.messages.length > 0) {
                    // Visa varje meddelande i sekvens
                    for (const message of data.messages) {
                        await showSystemMessage(message);
                    }
                }
            } catch (error) {
                console.error('Fel vid kontroll av systemmeddelande:', error);
                // Fortsätt normalt om det uppstår ett fel
            }
        }
        
        // Initialize page
        async function initializePage() {
            // Kontrollera systemmeddelande först (blockerar tills användaren kvitterar)
            await checkSystemMessage();
            
            // Clean up old chat states first
            cleanupOldChatStates();
            
            // Try to load saved state for this chat
            const stateLoaded = loadParticipantState();
            
            // If we didn't load state, or if coming from different page, clear any old state
            if (!stateLoaded) {
                clearParticipantState();
            }
            
            // Load participants and messages
            if (!isArchived) {
                await loadParticipants();
                
                // If state was loaded, make sure UI is updated to reflect restored notifications
                if (stateLoaded) {
                    renderParticipants();
                }
            }
            await loadMessages();
            
        }
        
        // Ladda data när sidan laddas
        initializePage();
        
        // Uppdatera meddelanden var 5:e sekund
        setInterval(loadMessages, 5000);
    </script>
    
    <!-- Modal för att hantera deltagare -->
    <div id="manageParticipantsModal" class="modal" style="display: none;">
        <div class="modal-content">
            <h2>Hantera deltagare</h2>
            <div id="manageParticipantsContent">
                <p>Laddar...</p>
            </div>
            <div style="margin-top: 20px;">
                <button class="btn btn-primary" onclick="saveAndCloseManageParticipants()">Spara</button>
            </div>
        </div>
    </div>
    
    <!-- Systemmeddelande modal -->
    <div id="systemMessageOverlay" class="system-message-overlay">
        <div class="system-message-modal">
            <h2 class="system-message-title">Information</h2>
            <div class="system-message-content" id="systemMessageContent"></div>
            <div class="system-message-actions">
                <button class="system-message-btn" id="systemMessageAcknowledgeBtn">Jag förstår</button>
            </div>
        </div>
    </div>
    
    <!-- Modal för chatdokumentation -->
    <div id="chatHelpModal" class="modal" style="display: none;">
        <div class="modal-content" style="max-width: 800px; max-height: 90vh; overflow-y: auto; padding-bottom: 120px;">
            <h2>ByggChat - Hjälp och dokumentation</h2>
            <div id="chatHelpContent" style="text-align: left; line-height: 1.6;">
                <h3>Översikt</h3>
                <p>ByggChat är ett realtidschatsystem för projektkommunikation. Du kan skicka textmeddelanden, ladda upp bilder och PDF-filer med anpassade namn, gilla meddelanden, svara på meddelanden, fästa viktiga meddelanden och få pushnotifikationer via Pushover. Alla inställningar bevaras automatiskt när du laddar om sidan.</p>
                
                <h3>Användartyper och behörigheter</h3>
                <h4>Administratörer</h4>
                <ul>
                    <li>Ser alla aktiva projekt i dropdown-menyn</li>
                    <li>Kan se och hantera alla projektchatter</li>
                    <li>Kan lägga till och ta bort deltagare via "Hantera"-knappen</li>
                    <li>Måste läggas till som deltagare för att synas i deltagarlistan och få notifieringar</li>
                </ul>
                
                <h4>Vanliga användare</h4>
                <ul>
                    <li>Ser alla aktiva projekt i dropdown-menyn</li>
                    <li>Läggs automatiskt till i projekt när de väljer ett projekt i dropdown-menyn</li>
                    <li>Kan gå med i projektchatter de inte är medlemmar i</li>
                </ul>
                
                <h4>Externa partners</h4>
                <ul>
                    <li>Ser endast projekt de är explicit medlemmar i</li>
                    <li>Läggs INTE till automatiskt i projektchatter</li>
                    <li>Måste läggas till manuellt av administratörer</li>
                    <li>Har globala behörigheter för chatt och uppladdning som styr om de kan delta i chattar och ladda upp filer överhuvudtaget</li>
                    <li>Behörigheter hanteras per projekt - även om de har global behörighet måste de vara medlemmar i specifika projekt för att delta</li>
                </ul>
                
                <h3>Funktioner</h3>
                <h4>Meddelanden</h4>
                <ul>
                    <li><strong>Textmeddelanden</strong>: Skriv meddelanden i textfältet och tryck Enter eller klicka på skicka-knappen</li>
                    <li><strong>Bildmeddelanden</strong>: Klicka på kameraknappen för att ladda upp bilder eller PDF-filer. Du kan ge filer anpassade namn och lägga till beskrivande text</li>
                    <li><strong>Mobilkamera</strong>: På mobila enheter kan du ta bilder direkt från kameran eller välja från galleriet</li>
                    <li><strong>Gilla meddelanden</strong>: Klicka på tumme upp-ikonen som visas när du hovrar över ett meddelande. Antalet gillningar visas under meddelandet</li>
                    <li><strong>Svara på meddelanden</strong>: Klicka på "Svara"-knappen på ett meddelande för att svara direkt på det. Det ursprungliga meddelandet visas i ditt svar</li>
                    <li><strong>Systemmeddelanden</strong>: Automatiska meddelanden när deltagare läggs till eller tas bort</li>
                    <li><strong>Datumseparatorer</strong>: Orange etiketter visar datum när det ändras mellan meddelanden</li>
                </ul>
                
                <h4>Meddelandefärger</h4>
                <ul>
                    <li><strong>Röd bakgrund</strong>: Externa partners</li>
                    <li><strong>Blå bakgrund</strong>: Vanliga användare och administratörer</li>
                    <li><strong>Röd bakgrund med svart kant</strong>: Brådskande meddelanden</li>
                </ul>
                
                <h4>Gilla meddelanden</h4>
                <ul>
                    <li>Alla deltagare kan gilla meddelanden genom att klicka på tumme upp-ikonen</li>
                    <li>Tumme upp-ikonen visas när du hovrar över ett meddelande (till vänster om pin-ikonen, till höger om svara-ikonen)</li>
                    <li>Antalet gillningar visas som "X gilla" under meddelandet bredvid avsändarens namn och datum</li>
                    <li>Klicka på gillningsräknaren för att se vem som gillat meddelandet</li>
                    <li>Din egen gilla-ikon blir grön när du gillat ett meddelande</li>
                    <li>Gillningar uppdateras omedelbart när du klickar (optimistisk uppdatering)</li>
                </ul>
                
                <h4>Fästa meddelanden</h4>
                <ul>
                    <li>Administratörer kan fästa viktiga meddelanden med pin-ikonen</li>
                    <li>Fästa meddelanden visas alltid först i chatten</li>
                    <li>Använd pin-filtret (pin-ikonen i deltagarsektionen) för att visa endast fästa meddelanden</li>
                </ul>
                
                <h4>Brådskande meddelanden</h4>
                <ul>
                    <li>Klicka på varningstriangeln för att markera meddelandet som brådskande</li>
                    <li>Brådskande meddelanden har röd bakgrund, vit text och svart kant</li>
                    <li>Brådskande meddelanden kräver bekräftelse innan de skickas</li>
                </ul>
                
                <h4>Sökfunktion</h4>
                <ul>
                    <li>Klicka på sökikonen (förstoringsglas) i deltagarsektionen för att söka i chatten</li>
                    <li>Sökningen gäller endast den aktuella chatten</li>
                    <li>Matchande meddelanden markeras och du kan klicka på dem för att hoppa till meddelandet</li>
                    <li>Matchande meddelanden blinkar svart när du hoppar till dem</li>
                </ul>
                
                <h4>Notifikationer</h4>
                <ul>
                    <li><strong>Pushover</strong>: Alla som vill ha notifikationer måste ha Pushover installerat och registrerat</li>
                    <li><strong>Per deltagare</strong>: Klicka på alarmklocka-ikonen bredvid deltagarens namn i hantera-popupen för att växla notifikationer</li>
                    <li><strong>Alla deltagare</strong>: Använd klockikon i deltagarsektionen för att växla alla notifikationer samtidigt</li>
                    <li><strong>Alarmklocka-ikoner</strong>: Visas i hantera-popupen för alla som har Pushover ID. Överkryssad och nedtonad när notifieringar är av</li>
                    <li><strong>Standard</strong>: Alla deltagare är notifierade som standard (utom avsändaren)</li>
                    <li><strong>Externa partners</strong>: Måste ha Pushover installerat och registrerat för att kunna ta emot notifikationer</li>
                </ul>
                
                <h4>Projektväljare</h4>
                <ul>
                    <li>Använd dropdown-menyn högst upp för att välja mellan projekt</li>
                    <li>Chatten laddas om automatiskt när projekt ändras</li>
                    <li>Vanliga användare läggs automatiskt till när de väljer ett projekt</li>
                </ul>
                
                <h4>Deltagarskap</h4>
                <ul>
                    <li>Administratörer kan lägga till och ta bort deltagare via "Hantera"-ikonen (tre personer) till vänster om deltagaretiketterna</li>
                    <li>I hantera-popupen kan du kryssa i/ur deltagare för att lägga till eller ta bort dem</li>
                    <li>Deltagare grupperas i "Administratörer", "Användare" och "Externa partners"</li>
                    <li>Externa partners har röd bakgrund när valda, ljusröd när inte valda</li>
                    <li>Interna användare har grön bakgrund när valda, ljusgrön när inte valda</li>
                    <li>Du kan inte ta bort dig själv från chatten</li>
                    <li>Inaktiva användare tas automatiskt bort från alla projektchatter</li>
                </ul>
                
                <h4>Deltagare-etiketter</h4>
                <ul>
                    <li>Etiketterna visar alla som är med i chatten (deltagare i projektet)</li>
                    <li><strong>Grön etikett</strong>: Interna användare</li>
                    <li><strong>Röd etikett</strong>: Externa partners</li>
                    <li><strong>Klicka på namnet</strong>: Växla notifikationer för den personen</li>
                    <li><strong>Klicka på ansiktsikonen</strong>: Visa personens visitkort med kontaktinformation och publika anteckningar</li>
                    <li><strong>Hovra över namnet</strong>: Visa visitkortet automatiskt (för admin-sidor)</li>
                    <li>Etiketten visar "(du)" för din egen etikett</li>
                </ul>
                
                <h4>Visitkort</h4>
                <ul>
                    <li>Klicka på ansiktsikonen på en deltagare-etikett för att visa visitkortet</li>
                    <li>Visitkortet visar profilbild, namn, yrkesroll, telefonnummer, företag (för externa partners) och publika anteckningar</li>
                    <li>Publika anteckningar visas i visitkortet om de finns</li>
                </ul>
                
                <h3>Användning</h3>
                <h4>Skicka meddelande</h4>
                <ol>
                    <li>Skriv meddelandet i textfältet</li>
                    <li>Välj vilka deltagare som ska få notifikationer (klicka på deras etiketter eller alarmklocka-ikoner i hantera-popupen)</li>
                    <li>Markera som brådskande om nödvändigt (varningstriangel)</li>
                    <li>Tryck Enter eller klicka på skicka-knappen</li>
                </ol>
                
                <h4>Svara på meddelande</h4>
                <ol>
                    <li>Klicka på "Svara"-knappen på det meddelande du vill svara på</li>
                    <li>Ett popup-fönster öppnas med det ursprungliga meddelandet</li>
                    <li>Skriv ditt svar i textfältet</li>
                    <li>Om det ursprungliga meddelandet är en bild eller PDF visas en miniatyr i popupen</li>
                    <li>Tryck Enter eller klicka på skicka-knappen</li>
                </ol>
                
                <h4>Ladda upp bild eller PDF</h4>
                <ol>
                    <li>Klicka på kameraknappen</li>
                    <li>Välj bild eller PDF från din enhet, eller ta en bild direkt med kameran (på mobila enheter)</li>
                    <li>Ett förhandsvisningsfönster visas där du kan:
                        <ul>
                            <li>Ge filen ett anpassat namn (ursprungliga filnamnet är förifyllt)</li>
                            <li>Lägga till en beskrivande text i meddelandefältet</li>
                        </ul>
                    </li>
                    <li>Klicka på skicka-knappen för att ladda upp filen</li>
                    <li>När du klickar på en uppladdad fil laddas den ner med det anpassade namnet</li>
                </ol>
                
                <h4>Fästa meddelande (endast administratörer)</h4>
                <ol>
                    <li>Klicka på pin-ikonen på ett meddelande</li>
                    <li>Meddelandet fästs och visas alltid först i chatten</li>
                    <li>Klicka på pin-ikonen igen för att ta bort fästningen</li>
                </ol>
                
                <h4>Söka i chatten</h4>
                <ol>
                    <li>Klicka på sökikonen (förstoringsglas) i deltagarsektionen</li>
                    <li>Skriv in söktermen i sökfältet</li>
                    <li>Matchande meddelanden visas i resultatlistan</li>
                    <li>Klicka på ett resultat för att hoppa till meddelandet</li>
                    <li>Meddelandet blinkar svart när du hoppar till det</li>
                </ol>
                
                <h4>Hantera deltagare (endast administratörer)</h4>
                <ol>
                    <li>Klicka på "Hantera"-ikonen (tre personer) till vänster om deltagaretiketterna</li>
                    <li>En popup öppnas med alla tillgängliga användare grupperade i kategorier</li>
                    <li>Kryssa i/ur deltagare för att lägga till eller ta bort dem från chatten</li>
                    <li>Klicka på alarmklocka-ikonen bredvid ett namn för att växla notifieringar</li>
                    <li>Klicka på "Spara" för att spara ändringarna</li>
                </ol>
                
                <h4>Växla notifikationer</h4>
                <ul>
                    <li>Klicka på deltagarens etikett för att växla notifikationer för den personen</li>
                    <li>Klicka på alarmklocka-ikonen i hantera-popupen för att växla notifieringar</li>
                    <li>Klicka på klockikon i deltagarsektionen för att växla alla notifikationer samtidigt (inklusive din egen)</li>
                    <li>Grön/röd etikett = notifieras, ljusblå = notifieras inte</li>
                    <li>Alarmklocka-ikoner är överkryssade och nedtonade när notifieringar är av</li>
                    <li><strong>Inställningar bevaras</strong>: Deltagaretiketter och notifieringsstatus sparas automatiskt och återställs när du laddar om samma chatt. Om du kommer från en annan sida eller byter till en ny chatt uppdateras inställningarna</li>
                </ul>
                
                <h3>Tips</h3>
                <ul>
                    <li>Meddelanden uppdateras automatiskt var 5:e sekund</li>
                    <li>Chatten scrollar automatiskt till senaste meddelandet när du klickar i textfältet</li>
                    <li>Använd runda knappen längst ner för att scrolla till senaste meddelandet</li>
                    <li>Arkiverade projekt är read-only - du kan läsa men inte skriva</li>
                    <li>När du kommer från en pushover-notifikation scrollar chatten automatiskt till senaste meddelandet</li>
                    <li>Fästa meddelanden visas alltid först, oavsett när de skickades</li>
                    <li>Använd pin-filtret för att snabbt hitta viktiga meddelanden</li>
                    <li>Ge dina uppladdade filer beskrivande namn så är de lättare att hitta senare</li>
                    <li>På mobila enheter kan du ta bilder direkt från kameran - perfekt för att dokumentera på plats</li>
                    <li>Gilla meddelanden för att snabbt visa att du sett eller godkänt något</li>
                    <li>Dina notifieringsinställningar bevaras automatiskt när du laddar om sidan</li>
                </ul>
            </div>
            <div style="margin-top: 20px; text-align: center;">
                <button class="btn btn-primary" onclick="closeChatHelp()">Stäng</button>
            </div>
        </div>
    </div>
    
    <script>
        // Lägg till scroll-event listener på meddelandecontainern när DOM är redo
        document.addEventListener('DOMContentLoaded', function() {
            const container = document.getElementById('messagesContainer');
            if (container) {
                container.addEventListener('scroll', updateScrollToBottomButton);
                // Uppdatera initialt
                updateScrollToBottomButton();
            }
            
            // Set up beforeunload handler to clear state when navigating away
            window.addEventListener('beforeunload', function() {
                // Only clear if we're navigating to a different page (not just reloading)
                if (performance && performance.navigation && performance.navigation.type !== performance.navigation.TYPE_RELOAD) {
                    clearParticipantState();
                }
            });
        });
    </script>
</body>
</html>

