<?php
/**
 * Generell notifieringsfunktion
 * Stödjer olika notifieringsproviders via plugin-system
 * Providers implementeras i separata filer i chat/providers/
 */

require_once __DIR__ . '/../config.php';
require_once __DIR__ . '/../security.php';

/**
 * Skicka notifieringar till användare
 * 
 * @param string $project_id Projekt-ID
 * @param string $message_id Meddelande-ID
 * @param array $notified_users Array av user_id eller external_party_id som ska notifieras
 * @param string $message_type Typ av meddelande ('text', 'image', 'system')
 * @param int|null $priority Pushover priority (0-2, eller null för att använda provider default)
 * @return bool True om notifieringar loggades, false vid fel
 */
function sendNotifications($project_id, $message_id, $notified_users, $message_type = 'text', $priority = null) {
    $logFile = __DIR__ . '/../logs/notifications.log';
    
    // Skapa logs-mappen om den inte finns
    $logDir = dirname($logFile);
    if (!is_dir($logDir)) {
        mkdir($logDir, 0755, true);
    }
    
    $timestamp = date('Y-m-d H:i:s');
    
    $logEntry = [
        'timestamp' => $timestamp,
        'project_id' => $project_id,
        'message_id' => $message_id,
        'notified_users' => $notified_users,
        'message_type' => $message_type,
        'provider' => defined('NOTIFICATION_PROVIDER') ? NOTIFICATION_PROVIDER : 'none'
    ];
    
    // Logga till fil
    $logLine = json_encode($logEntry, JSON_UNESCAPED_UNICODE) . "\n";
    file_put_contents($logFile, $logLine, FILE_APPEND | LOCK_EX);
    
    // Ladda och använd notifieringsprovider
    // För att lägga till en ny provider:
    // 1. Skapa en ny fil i chat/providers/ (t.ex. email_provider.php)
    // 2. Implementera funktionen sendProviderNotifications($project_id, $message_id, $notified_users, $message_type, $config)
    // 3. Lägg till konfiguration i provider-filen (se ntfy_provider.php som exempel)
    // 4. Sätt NOTIFICATION_PROVIDER till provider-namnet i config.php
    $provider = defined('NOTIFICATION_PROVIDER') ? NOTIFICATION_PROVIDER : 'none';
    
    if ($provider === 'none') {
        // Endast loggning, inga faktiska notifieringar
    return true;
    }
    
    // Ladda provider-fil dynamiskt
    $providerFile = __DIR__ . '/providers/' . $provider . '_provider.php';
    if (!file_exists($providerFile)) {
        error_log("Notification provider file not found: $providerFile");
        return false;
    }
    
    try {
        require_once $providerFile;
    } catch (Throwable $e) {
        // Fånga fel vid inladdning av provider-filen
        error_log("Error loading notification provider file $providerFile: " . $e->getMessage());
        return false;
    }
    
    // Anropa standardiserad provider-funktion
    // Provider-filen ansvarar för sin egen konfiguration
    if (function_exists('sendProviderNotifications')) {
        try {
            $providerConfig = [];
            if ($priority !== null) {
                $providerConfig['priority'] = $priority;
            }
            return sendProviderNotifications($project_id, $message_id, $notified_users, $message_type, $providerConfig);
        } catch (Throwable $e) {
            // Fånga alla fel från providern så att de inte kraschar chat_api.php
            error_log("Notification provider error: " . $e->getMessage());
            return false;
        }
    } else {
        error_log("sendProviderNotifications function not found in provider file: $providerFile");
        return false;
    }
}

