<?php
require_once __DIR__ . '/../security.php';
require_once __DIR__ . '/../config.php';
require_once __DIR__ . '/../lib/participant_helper.php';

setSecurityHeaders();

// Svara alltid JSON
header('Content-Type: application/json; charset=utf-8');

// Sökvägar till JSON-filer
$userIdsFile = __DIR__ . '/../data/userids.json';
$externalCompaniesFile = __DIR__ . '/../data/external_companies.json';
$externalTypesFile = __DIR__ . '/../data/external_types.json';

// Funktion för att läsa user IDs från JSON-fil
function readUserIds($filePath) {
    if (!file_exists($filePath)) {
        return [];
    }
    $content = file_get_contents($filePath);
    if ($content === false) {
        return [];
    }
    $userIds = json_decode($content, true);
    return is_array($userIds) ? $userIds : [];
}

// Funktion för att läsa external companies från JSON-fil
function readExternalCompanies($filePath) {
    if (!file_exists($filePath)) {
        return [];
    }
    $content = file_get_contents($filePath);
    if ($content === false) {
        return [];
    }
    $companies = json_decode($content, true);
    return is_array($companies) ? $companies : [];
}

// Funktion för att läsa external types från JSON-fil
function readExternalTypes($filePath) {
    if (!file_exists($filePath)) {
        return [];
    }
    $content = file_get_contents($filePath);
    if ($content === false) {
        return [];
    }
    $types = json_decode($content, true);
    return is_array($types) ? $types : [];
}

try {
    $participantId = sanitizeInput($_GET['participant_id'] ?? '', 'string');
    
    if (empty($participantId)) {
        http_response_code(400);
        echo json_encode([
            'success' => false,
            'message' => 'Participant ID krävs'
        ]);
        exit;
    }
    
    // Bestäm typ av participant
    $type = getParticipantType($participantId);
    
    if ($type === 'user' || $type === 'admin') {
        // Hämta från userids.json (både user och admin är i samma fil)
        $users = readUserIds($userIdsFile);
        foreach ($users as $user) {
            if (($user['user_id'] ?? '') === $participantId) {
                $result = [
                    'success' => true,
                    'data' => [
                        'name' => $user['full_name'] ?? $participantId,
                        'mobile' => $user['mobile'] ?? '',
                        'profile_image' => $user['profile_image'] ?? null,
                        'company_name' => null, // Users har inget företag
                        'role' => null, // Users har ingen yrkesroll ännu
                        'type' => $type, // 'user' eller 'admin'
                        'public_notes' => $user['public_notes'] ?? null
                    ]
                ];
                echo json_encode($result, JSON_UNESCAPED_UNICODE);
                exit;
            }
        }
    } elseif ($type === 'extern') {
        // Hämta från userids.json (filtrerar på user_type === "extern"), external_companies.json och external_types.json
        $allUsers = readUserIds($userIdsFile);
        $companies = readExternalCompanies($externalCompaniesFile);
        $types = readExternalTypes($externalTypesFile);
        
        foreach ($allUsers as $user) {
            // Kontrollera att det är en extern användare och att user_id matchar
            $userType = $user['user_type'] ?? 'user';
            if ($userType === 'extern' && ($user['user_id'] ?? '') === $participantId) {
                // Hitta företagsnamn
                $companyName = null;
                $companyId = $user['company_id'] ?? '';
                if (!empty($companyId)) {
                    foreach ($companies as $company) {
                        if (($company['id'] ?? '') === $companyId) {
                            $companyName = $company['name'] ?? null;
                            break;
                        }
                    }
                }
                
                // Hitta yrkesroll
                $roleName = null;
                $typeId = $user['type_id'] ?? '';
                if (!empty($typeId)) {
                    foreach ($types as $typeItem) {
                        if (($typeItem['id'] ?? '') === $typeId) {
                            $roleName = $typeItem['name'] ?? null;
                            break;
                        }
                    }
                }
                
                $result = [
                    'success' => true,
                    'data' => [
                        'name' => $user['full_name'] ?? $participantId,
                        'mobile' => $user['mobile'] ?? '',
                        'profile_image' => $user['profile_image'] ?? null,
                        'company_name' => $companyName,
                        'role' => $roleName,
                        'type' => 'staff',
                        'public_notes' => $user['public_notes'] ?? null
                    ]
                ];
                echo json_encode($result, JSON_UNESCAPED_UNICODE);
                exit;
            }
        }
    }
    
    // Om participant inte hittades
    http_response_code(404);
    echo json_encode([
        'success' => false,
        'message' => 'Participant hittades inte'
    ]);
    
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => $e->getMessage()
    ]);
}
?>

