<?php
/**
 * Pushover Notification Provider
 * Skickar push-notifikationer via Pushover API.
 *
 * Konfiguration:
 *  - Redigera $PUSHOVER_CONFIG nedan för att anpassa inställningar.
 *  - API-token (application token) hårdkodas här.
 *  - Varje användare får en egen Pushover user key i admin-gränssnittet.
 */

// Hjälpfunktion för Pushover-loggning
function pushover_debug_log($message) {
    $logDir = __DIR__ . '/../../logs';
    if (!is_dir($logDir)) {
        @mkdir($logDir, 0755, true);
    }
    $logFile = $logDir . '/pushover.log';
    $timestamp = date('Y-m-d H:i:s');
    $line = '[' . $timestamp . '] ' . $message . "\n";
    @file_put_contents($logFile, $line, FILE_APPEND | LOCK_EX);
}

// Konfiguration för Pushover-provider
// Redigera dessa värden för att anpassa beteendet
if (!isset($GLOBALS['PUSHOVER_CONFIG'])) {
    $GLOBALS['PUSHOVER_CONFIG'] = [
        'api_token' => 'aacf6h5dgs5852mfsa12411g73k625', // Din Pushover Application API Token (hämta från https://pushover.net/apps)
        // User keys läses från data/userids.json (pushover_user_key fältet per användare)
        // Lägg till Pushover user key i admin-gränssnittet: adm/users/edit_user.php
        'priority' => 0, // -2 (no notification), -1 (quiet), 0 (normal), 1 (high), 2 (emergency)
        'sound' => 'pushover', // Ljud för notifikationer: pushover, incoming, camera, bike, cashregister, cosmic, falling, gamelan, magic, mechanical, pianobar, siren, spacealarm, tugboat, alien, climb, persistent, echo, updown, none
        'title' => 'Ny meddelande i projekt',
        'api_url' => 'https://api.pushover.net/1/messages.json',
        'userids_file' => __DIR__ . '/../../data/userids.json', // Var användardata lagras
    ];
}
$PUSHOVER_CONFIG = &$GLOBALS['PUSHOVER_CONFIG'];

/**
 * Standardiserad provider-funktion för notifikationer
 * Denna funktion måste implementeras av alla providers
 * 
 * @param string $project_id Projekt-ID
 * @param string $message_id Meddelande-ID
 * @param array $notified_users Array av user_id eller external_party_id som ska notifieras
 * @param string $message_type Typ av meddelande ('text', 'image', 'system')
 * @param array $config Konfiguration för provider (används inte här, använder $PUSHOVER_CONFIG istället)
 * @return bool True om notifikationer skickades, false vid fel
 */
function sendProviderNotifications($project_id, $message_id, $notified_users, $message_type = 'text', $config = []) {
    // Inkludera participant_helper för att hämta namn
    require_once __DIR__ . '/../../lib/participant_helper.php';
    
    // Använd konfiguration från denna fil
    global $PUSHOVER_CONFIG;
    
    // Säkerställ att $PUSHOVER_CONFIG är definierad
    // Om den saknas (t.ex. om filen laddats flera gånger), definiera den igen
    if (!isset($PUSHOVER_CONFIG) || !is_array($PUSHOVER_CONFIG)) {
        // Försök hämta från GLOBALS först
        if (isset($GLOBALS['PUSHOVER_CONFIG']) && is_array($GLOBALS['PUSHOVER_CONFIG'])) {
            $PUSHOVER_CONFIG = &$GLOBALS['PUSHOVER_CONFIG'];
        } else {
            // Fallback: skapa ny konfiguration
            $PUSHOVER_CONFIG = [
                'api_token' => 'aacf6h5dgs5852mfsa12411g73k625',
                'priority' => 0,
                'title' => 'Ny meddelande i projekt',
                'api_url' => 'https://api.pushover.net/1/messages.json',
                'userids_file' => __DIR__ . '/../../data/userids.json',
            ];
            $GLOBALS['PUSHOVER_CONFIG'] = &$PUSHOVER_CONFIG;
        }
    }
    
    // Standardkonfiguration som fallback
    $defaultConfig = [
        'api_token' => '',
        'priority' => 0,
        'title' => 'Ny meddelande i projekt',
        'api_url' => 'https://api.pushover.net/1/messages.json',
        'userids_file' => __DIR__ . '/../../data/userids.json',
    ];
    
    // Merga: default -> filens konfiguration -> parametern (om skickad)
    $config = array_merge($defaultConfig, $PUSHOVER_CONFIG, $config);
    
    // Om priority skickas som parameter, använd den (överstyr config)
    if (isset($config['priority']) && $config['priority'] !== null) {
        $config['priority'] = (int)$config['priority'];
    }
    
    // Kontrollera att API-token är konfigurerad
    if (empty($config['api_token'])) {
        pushover_debug_log("Pushover: API token saknas");
        return false;
    }
    
    if (empty($notified_users)) {
        return true; // Inga användare att notifiera
    }
    
    // Ladda user keys från userids.json
    $userKeys = [];
    $useridsFile = $config['userids_file'];
    if (file_exists($useridsFile)) {
        try {
            $useridsJson = file_get_contents($useridsFile);
            $userids = json_decode($useridsJson, true);
            if (is_array($userids)) {
                foreach ($userids as $user) {
                    if (isset($user['user_id']) && !empty($user['pushover_user_key'])) {
                        $userKeys[$user['user_id']] = $user['pushover_user_key'];
                    }
                }
            }
        } catch (Throwable $e) {
            pushover_debug_log("Pushover: fel vid läsning av userids.json: " . $e->getMessage());
        }
    } else {
        pushover_debug_log("Pushover: userids_file hittades inte");
    }
    
    // Förbered titel: "ByggChatt: <projektnamn>" (fallback till projekt-ID)
    $projectName = null;
    try {
        $projectsFile = __DIR__ . '/../../data/projects.json';
        if (file_exists($projectsFile)) {
            $projectsJson = file_get_contents($projectsFile);
            $projectsData = json_decode($projectsJson, true);
            if (is_array($projectsData)) {
                foreach ($projectsData as $project) {
                    if (isset($project['id']) && (string)$project['id'] === (string)$project_id) {
                        if (!empty($project['name'])) {
                            $projectName = $project['name'];
                        }
                        break;
                    }
                }
            }
        }
    } catch (Throwable $e) {
        pushover_debug_log("Pushover: fel vid läsning av projects.json: " . $e->getMessage());
    }
    if ($projectName === null) {
        $projectName = 'Projekt ' . $project_id;
    }
    $title = 'ByggChatt: ' . $projectName;
    $messageText = 'Ny aktivitet i projekt ' . $project_id;

    // Hitta bildfil om message_type är 'image'
    $imageFile = null;
    $imagePath = null;
    
    // Försök läsa chattfilen och hitta meddelandetexten för aktuellt message_id
    $senderName = null;
    try {
        $chatsDir = __DIR__ . '/../../chats';
        $chatFile = $chatsDir . '/' . $project_id . '/messages.json';
        if (file_exists($chatFile)) {
            $chatJson = file_get_contents($chatFile);
            $chatData = json_decode($chatJson, true);
            if (is_array($chatData)) {
                foreach ($chatData as $msg) {
                    if (isset($msg['id']) && $msg['id'] === $message_id) {
                        // Hämta avsändarens namn
                        if (!empty($msg['sender_name'])) {
                            $senderName = $msg['sender_name'];
                        } elseif (!empty($msg['participant_id'])) {
                            // Använd participant_helper för att hämta namn
                            $senderName = getParticipantName($msg['participant_id']);
                            if ($senderName === $msg['participant_id']) {
                                $senderName = null; // Om namnet är samma som ID, använd inte det
                            }
                        }
                        
                        // Om det finns en faktisk meddelandetext, använd den
                        if (!empty($msg['message'])) {
                            $raw = $msg['message'];
                            // Rensa HTML och normalisera whitespace
                            $clean = strip_tags($raw);
                            $clean = preg_replace('/\s+/', ' ', $clean);
                            $clean = trim($clean);
                            if ($clean !== '') {
                                // Lägg till avsändarens namn om det finns
                                if ($senderName) {
                                    $namePart = ' - ' . $senderName;
                                    $maxLen = 512; // Pushover max message length
                                    $maxMsgLen = $maxLen - mb_strlen($namePart, 'UTF-8');
                                    
                                    // Om meddelandet + namn blir för långt, klipp av meddelandet
                                    if (mb_strlen($clean, 'UTF-8') > $maxMsgLen) {
                                        if ($maxMsgLen > 0) {
                                            $clean = mb_substr($clean, 0, $maxMsgLen, 'UTF-8') . '…';
                                        }
                                    }
                                    $clean .= $namePart;
                                } else {
                                    // Om inget namn, kontrollera bara meddelandets längd
                                    $maxLen = 512; // Pushover max message length
                                    if (mb_strlen($clean, 'UTF-8') > $maxLen) {
                                        $clean = mb_substr($clean, 0, $maxLen, 'UTF-8') . '…';
                                    }
                                }
                                $messageText = $clean;
                            }
                        } elseif ($message_type === 'image') {
                            $messageText = 'Ny bild i chatten';
                            if ($senderName) {
                                $messageText .= ' - ' . $senderName;
                            }
                            
                            // Hitta bildfilen (bilder sparas i chats-mappen)
                            if (!empty($msg['media_file'])) {
                                $chatsDir = __DIR__ . '/../../chats';
                                $imagePath = $chatsDir . '/' . $project_id . '/' . $msg['media_file'];
                                if (file_exists($imagePath)) {
                                    $imageFile = $imagePath;
                                }
                            }
                        } elseif ($message_type === 'document') {
                            $messageText = 'Nytt dokument i chatten';
                            if ($senderName) {
                                $messageText .= ' - ' . $senderName;
                            }
                            
                            // Hitta dokumentfilen (dokument sparas i chats-mappen)
                            if (!empty($msg['media_file'])) {
                                $chatsDir = __DIR__ . '/../../chats';
                                $docPath = $chatsDir . '/' . $project_id . '/' . $msg['media_file'];
                                if (file_exists($docPath)) {
                                    $imageFile = $docPath; // Använd samma variabel för att hålla kompatibilitet
                                }
                            }
                        } elseif ($message_type === 'system') {
                            $messageText = 'Systemmeddelande i projekt ' . $project_id;
                        }
                        break;
                    }
                }
            }
        }
    } catch (Throwable $e) {
        pushover_debug_log("Pushover: fel vid läsning av meddelandetext: " . $e->getMessage());
    }
    
    // Skapa miniatyrbild om bilden finns och är för stor (>2.5MB)
    $thumbnailFile = null;
    if ($imageFile && file_exists($imageFile)) {
        $fileSize = filesize($imageFile);
        $maxSize = 2.5 * 1024 * 1024; // 2.5 MB
        
        if ($fileSize > $maxSize) {
            // Skapa miniatyrbild
            try {
                $imageInfo = @getimagesize($imageFile);
                if ($imageInfo !== false) {
                    $mimeType = $imageInfo['mime'];
                    $width = $imageInfo[0];
                    $height = $imageInfo[1];
                    
                    // Skapa miniatyr (max 800px bredd/höjd)
                    $thumbSize = 800;
                    $thumbWidth = $width > $height ? $thumbSize : intval($width * ($thumbSize / $height));
                    $thumbHeight = $height > $width ? $thumbSize : intval($height * ($thumbSize / $width));
                    
                    // Ladda originalbild
                    $source = null;
                    if ($mimeType === 'image/jpeg' || $mimeType === 'image/jpg') {
                        $source = @imagecreatefromjpeg($imageFile);
                    } elseif ($mimeType === 'image/png') {
                        $source = @imagecreatefrompng($imageFile);
                    } elseif ($mimeType === 'image/gif') {
                        $source = @imagecreatefromgif($imageFile);
                    } elseif ($mimeType === 'image/webp') {
                        $source = @imagecreatefromwebp($imageFile);
                    }
                    
                    if ($source) {
                        // Skapa miniatyrbild
                        $thumb = imagecreatetruecolor($thumbWidth, $thumbHeight);
                        
                        // Behåll transparens för PNG/GIF
                        if ($mimeType === 'image/png' || $mimeType === 'image/gif') {
                            imagealphablending($thumb, false);
                            imagesavealpha($thumb, true);
                            $transparent = imagecolorallocatealpha($thumb, 255, 255, 255, 127);
                            imagefilledrectangle($thumb, 0, 0, $thumbWidth, $thumbHeight, $transparent);
                        }
                        
                        imagecopyresampled($thumb, $source, 0, 0, 0, 0, $thumbWidth, $thumbHeight, $width, $height);
                        
                        // Spara miniatyrbild till temporär fil
                        $tempDir = sys_get_temp_dir();
                        $thumbnailFile = $tempDir . '/pushover_thumb_' . basename($imageFile) . '.jpg';
                        
                        if (imagejpeg($thumb, $thumbnailFile, 85)) {
                            imagedestroy($thumb);
                            imagedestroy($source);
                        } else {
                            imagedestroy($thumb);
                            imagedestroy($source);
                            $thumbnailFile = null;
                        }
                    }
                }
            } catch (Throwable $e) {
                pushover_debug_log("Pushover: fel vid skapande av miniatyrbild: " . $e->getMessage());
                $thumbnailFile = null;
            }
        }
    }
    
    $success = true;
    $apiUrl = $config['api_url'];
    
    // Skicka notifikation till varje användare
    foreach ($notified_users as $user_id) {
        // Hitta Pushover user key för denna användare från userids.json
        if (!isset($userKeys[$user_id]) || empty($userKeys[$user_id])) {
            // Ingen user key mappad för denna användare - hoppa över
            continue;
        }
        
        $userKey = $userKeys[$user_id];
        
        // Bygg URL för att öppna chatten
        // Använd scriptPath för att hitta bas-sökvägen (hanterar undermappar korrekt)
        $scriptPath = $_SERVER['SCRIPT_NAME'] ?? '';
        $basePath = '/';
        
        if (!empty($scriptPath)) {
            // Normalisera sökvägen (hantera Windows-backslashes)
            $normalizedPath = str_replace('\\', '/', $scriptPath);
            $normalizedPath = trim($normalizedPath, '/');
            
            if (!empty($normalizedPath)) {
                $pathParts = explode('/', $normalizedPath);
                
                // Om första delen är 'api' eller 'chat', vi är i root
                // Annars är första delen bas-mappen (t.ex. 'byggarkiv_dev' eller 'byggarkiv')
                if (count($pathParts) > 0 && !empty($pathParts[0])) {
                    if ($pathParts[0] === 'api' || $pathParts[0] === 'chat') {
                        $basePath = '/';
                    } else {
                        // Första delen är bas-mappen
                        $basePath = '/' . $pathParts[0] . '/';
                    }
                }
            }
        }
        
        $url = $basePath . 'chat/chat.php?project_id=' . urlencode($project_id) . '&user_id=' . urlencode($user_id);
        
        // Konstruera fullständig URL
        $fullUrl = (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 'https://' : 'http://') 
                   . ($_SERVER['HTTP_HOST'] ?? 'localhost') 
                   . $url;
        
        // Förbered POST-data för Pushover API
        $effectivePriority = isset($config['priority']) && $config['priority'] !== null ? (int)$config['priority'] : 0;
        
        $postData = [
            'token' => $config['api_token'],
            'user' => $userKey,
            'message' => $messageText,
            'title' => $title,
            'priority' => (string)$effectivePriority, // Pushover kräver priority som sträng
            'url' => $fullUrl,
            'url_title' => 'Öppna chatt',
        ];
        
        // Nödläge (priority=2) kräver retry + expire enligt Pushover-specen
        if ($effectivePriority === 2) {
            // Standardvärden: försök var 60:e sekund i upp till 1 timme
            $postData['retry'] = 60;    // sekunder mellan varje försök
            $postData['expire'] = 3600; // totala sekunder innan notisen upphör
            // Använd ett mer påträngande ljud för nödläge
            $postData['sound'] = 'siren';
        } else {
            // Lägg till ljud - alltid skicka sound för att säkerställa att ljudet spelas
            // Använd konfigurerat sound eller default 'pushover'
            $soundToUse = !empty($config['sound']) ? $config['sound'] : 'pushover';
            $postData['sound'] = $soundToUse;
        }
        
        // Lägg till bildbilaga om bild finns
        $useMultipart = false;
        $attachmentFile = null;
        try {
            if ($message_type === 'image' && ($thumbnailFile || $imageFile)) {
                $attachmentFile = $thumbnailFile ?: $imageFile;
                if ($attachmentFile && file_exists($attachmentFile)) {
                    $attachmentSize = filesize($attachmentFile);
                    // Pushover max storlek är 2.5 MB
                    if ($attachmentSize <= 2.5 * 1024 * 1024) {
                        $useMultipart = true;
                    } else {
                        $attachmentFile = null;
                    }
                } else {
                    $attachmentFile = null;
                }
            }
        } catch (Throwable $e) {
            pushover_debug_log("Pushover: fel vid hantering av bildbilaga: " . $e->getMessage());
            // Fortsätt utan bildbilaga - notifikationen ska skickas ändå
            $useMultipart = false;
            $attachmentFile = null;
        }
        
        // Skicka HTTP POST till Pushover API
        $ch = curl_init($apiUrl);
        $curlOptions = [
            CURLOPT_POST => true,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_TIMEOUT => 30 // Öka timeout för bilder
        ];
        
        if ($useMultipart && $attachmentFile) {
            // Försök skicka bilden som Base64-kodad sträng istället för filbilaga
            // Detta undviker problem med multipart/form-data och CURLFile
            try {
                $imageData = @file_get_contents($attachmentFile);
                if ($imageData !== false && strlen($imageData) <= 2.5 * 1024 * 1024) {
                    $base64Image = base64_encode($imageData);
                    $postData['attachment_base64'] = $base64Image;
                    $useMultipart = false; // Använd vanlig POST istället
                } else {
                    throw new Exception("Bildfil för stor eller kunde inte läsas");
                }
            } catch (Throwable $e) {
                pushover_debug_log("Pushover: fel vid Base64-kodning, försöker CURLFile: " . $e->getMessage());
                
                // Fallback till CURLFile-metoden
                // Kopiera alla fält från $postData (inklusive sound, retry, expire etc.)
                $multipartData = [];
                foreach ($postData as $key => $value) {
                    if ($key !== 'attachment_base64') { // Skippa Base64 om det finns
                        $multipartData[$key] = is_string($value) ? $value : (string)$value;
                    }
                }
                
                if (class_exists('CURLFile')) {
                    $mimeType = @mime_content_type($attachmentFile);
                    if (!$mimeType) {
                        $ext = strtolower(pathinfo($attachmentFile, PATHINFO_EXTENSION));
                        $mimeType = 'image/jpeg';
                        if ($ext === 'png') $mimeType = 'image/png';
                        elseif ($ext === 'gif') $mimeType = 'image/gif';
                        elseif ($ext === 'webp') $mimeType = 'image/webp';
                    }
                    $multipartData['attachment'] = new CURLFile($attachmentFile, $mimeType, basename($attachmentFile));
                } else {
                    $multipartData['attachment'] = '@' . $attachmentFile;
                }
                
                curl_setopt($ch, CURLOPT_POSTFIELDS, $multipartData);
            }
        }
        
        if (!$useMultipart) {
            // Använd application/x-www-form-urlencoded för vanliga meddelanden eller Base64-bilder
            curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($postData));
        }
        
        curl_setopt_array($ch, $curlOptions);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error = curl_error($ch);
        curl_close($ch);
        
        // Rensa temporär miniatyrbild om den skapades
        if ($thumbnailFile && file_exists($thumbnailFile)) {
            @unlink($thumbnailFile);
        }
        
        if ($error) {
            pushover_debug_log("Pushover: cURL error for user_id=$user_id – $error");
            $success = false;
        } elseif ($httpCode !== 200) {
            // Pushover returnerar JSON med felmeddelande
            $responseData = json_decode($response, true);
            $errorMsg = $responseData['errors'][0] ?? "HTTP $httpCode";
            pushover_debug_log("Pushover: API error for user_id=$user_id – HTTP $httpCode: $errorMsg");
            $success = false;
        }
        // Vid lyckat svar behåller vi $success = true
    }
    
    return $success;
}

