<?php
require_once __DIR__ . '/config.php';
require_once __DIR__ . '/security.php';
require_once __DIR__ . '/lib/participant_helper.php';
require_once __DIR__ . '/lib/mapping_helper.php';
require_once __DIR__ . '/lib/dal/repositories/UserRepository.php';
require_once __DIR__ . '/lib/dal/repositories/ProjectRepository.php';
require_once __DIR__ . '/logger.php';

setSecurityHeaders();

// Hämta user_id från URL
$userId = sanitizeInput($_GET['user_id'] ?? '', 'string');

if (empty($userId)) {
    die('Användar-ID krävs');
}

// Hämta användardata
$userRepo = new UserRepository();
$userData = $userRepo->getById($userId);

if (!$userData) {
    die('Användare hittades inte');
}

// Hämta användartyp (user, admin eller extern)
$userType = $userData['user_type'] ?? 'user';

// Kontrollera att användaren är aktiv
$active = $userData['active'] ?? true;
if (!($active === true || $active === 1 || $active === '1' || $active === 'true')) {
    die('Användaren är inaktiv');
}

// Hämta rättigheter
// För interna användare (user/admin) är can_upload och can_chat alltid true
// För externa användare läses de från användardata
if ($userType === 'extern') {
    $canUploadRaw = $userData['can_upload'] ?? true;
    $canUpload = ($canUploadRaw === true || $canUploadRaw === 1 || $canUploadRaw === '1' || $canUploadRaw === 'true');
    
    $canChatRaw = $userData['can_chat'] ?? true;
    $canChat = ($canChatRaw === true || $canChatRaw === 1 || $canChatRaw === '1' || $canChatRaw === 'true');
} else {
    // Interna användare (user/admin) har alltid rätt att ladda upp och chatta
    $canUpload = true;
    $canChat = true;
}

// Hämta alla projekt
$projectRepo = new ProjectRepository();
$projects = $projectRepo->getAll();

// Kontrollera om användaren är admin
require_once __DIR__ . '/lib/participant_helper.php';
$isAdmin = isParticipantAdmin($userId);

// Hitta projekt där användaren är deltagare
$userProjects = [];

foreach ($projects as $project) {
    $projectId = $project['id'] ?? '';
    if (empty($projectId)) {
        continue;
    }
    
    // Hoppa över arkiverade projekt
    $status = $project['status'] ?? 'active';
    if ($status === 'archived') {
        continue;
    }
    
    // Admins ser alla aktiva projekt
    if ($isAdmin) {
        $userProjects[] = [
            'id' => $projectId,
            'name' => $project['name'] ?? ''
        ];
        continue;
    }
    
    // För vanliga användare och externa: kontrollera om användaren är deltagare i projektet
    $participants = readProjectMapping($projectId);
    $isParticipant = false;
    
    foreach ($participants as $participant) {
        if (($participant['participant_id'] ?? '') === $userId) {
            $isParticipant = true;
            break;
        }
    }
    
    if ($isParticipant) {
        // Användaren är deltagare i projektet - chatt-länken ska visas om can_chat är true
        // (användaren läggs automatiskt till som chatter när hen öppnar chatten)
        $userProjects[] = [
            'id' => $projectId,
            'name' => $project['name'] ?? ''
        ];
    }
}

// Sortera projekt alfabetiskt
usort($userProjects, function($a, $b) {
    return strcasecmp($a['name'], $b['name']);
});

$userName = $userData['full_name'] ?? $userData['name'] ?? ($isAdmin ? 'Administratör' : 'Användare');
?>
<!DOCTYPE html>
<html lang="sv">
<head>
    <meta charset="utf-8" />
    <meta name="viewport" content="width=device-width,initial-scale=1" />
    <title>Mina projekt - ByggArkiv</title>
    <style>
        * {
            box-sizing: border-box;
        }
        body {
            font-family: system-ui, -apple-system, Segoe UI, Roboto, sans-serif;
            margin: 0;
            padding: 20px;
            background: #f5f5f5;
            color: #333;
        }
        .container {
            max-width: 800px;
            margin: 0 auto;
        }
        .header {
            background: white;
            border-radius: 12px;
            padding: 20px;
            margin-bottom: 20px;
            box-shadow: 0 2px 8px rgba(0,0,0,0.1);
            text-align: center;
        }
        .logo {
            max-width: 100%;
            height: auto;
            max-height: 120px;
            display: block;
            margin: 0 auto 15px;
        }
        h1 {
            color: #1a73e8;
            margin: 0 0 10px 0;
            font-size: 24px;
        }
        .welcome-text {
            color: #666;
            font-size: 14px;
            margin: 0;
        }
        .projects-list {
            display: flex;
            flex-direction: column;
            gap: 16px;
        }
        .project-card {
            background: white;
            border-radius: 12px;
            padding: 16px 20px;
            box-shadow: 0 2px 8px rgba(0,0,0,0.1);
            transition: transform 0.2s, box-shadow 0.2s;
            display: flex;
            align-items: center;
            justify-content: space-between;
            gap: 16px;
        }
        .project-card:hover {
            transform: translateY(-2px);
            box-shadow: 0 4px 12px rgba(0,0,0,0.15);
        }
        .project-name {
            font-size: 18px;
            font-weight: 600;
            color: #333;
            margin: 0;
            flex: 1;
        }
        .project-actions {
            display: flex;
            gap: 8px;
            align-items: center;
        }
        .action-btn {
            width: 40px;
            height: 40px;
            border-radius: 8px;
            border: none;
            cursor: pointer;
            text-decoration: none;
            display: inline-flex;
            align-items: center;
            justify-content: center;
            transition: background-color 0.2s, transform 0.2s;
            padding: 0;
        }
        .action-btn:hover {
            transform: translateY(-1px);
        }
        .action-btn:active {
            transform: translateY(0);
        }
        .btn-chat {
            background: #4caf50;
            color: white;
        }
        .btn-chat:hover {
            background: #45a049;
        }
        .btn-upload {
            background: #2196f3;
            color: white;
        }
        .btn-upload:hover {
            background: #1976d2;
        }
        .action-btn .icon {
            width: 24px;
            height: 24px;
            fill: currentColor;
        }
        .btn-disabled {
            background: #e0e0e0;
            color: #999;
            cursor: not-allowed;
            opacity: 0.6;
        }
        .btn-disabled:hover {
            transform: none;
            background: #e0e0e0;
        }
        .no-projects {
            background: white;
            border-radius: 12px;
            padding: 40px;
            text-align: center;
            box-shadow: 0 2px 8px rgba(0,0,0,0.1);
        }
        .no-projects h2 {
            color: #666;
            margin: 0 0 10px 0;
        }
        .no-projects p {
            color: #999;
            margin: 0;
        }
        @media (max-width: 600px) {
            .project-card {
                flex-direction: column;
                align-items: flex-start;
            }
            .project-actions {
                width: 100%;
                justify-content: flex-start;
            }
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <img src="data/logo.png" alt="ByggArkiv" class="logo" onerror="this.style.display='none'">
            <h1>Mina projekt</h1>
            <p class="welcome-text">Välkommen, <?= htmlspecialchars($userName) ?></p>
        </div>
        
        <?php if (empty($userProjects)): ?>
            <div class="no-projects">
                <h2>Inga projekt</h2>
                <p>Du är inte deltagare i några projekt ännu. Kontakta administratören för att bli tillagd i ett projekt.</p>
            </div>
        <?php else: ?>
            <div class="projects-list">
                <?php foreach ($userProjects as $project): ?>
                    <div class="project-card">
                        <h2 class="project-name"><?= htmlspecialchars($project['name']) ?></h2>
                        <div class="project-actions">
                            <a href="chat/chat.php?project_id=<?= urlencode($project['id']) ?>&user_id=<?= urlencode($userId) ?>" class="action-btn btn-chat" title="Öppna chatt">
                                <svg class="icon" viewBox="0 0 24 24" fill="currentColor">
                                    <path d="M20 2H4c-1.1 0-2 .9-2 2v18l4-4h14c1.1 0 2-.9 2-2V4c0-1.1-.9-2-2-2zm0 14H6l-2 2V4h16v12z"/>
                                </svg>
                            </a>
                            
                            <?php if ($canUpload): ?>
                                <a href="index.html?user_id=<?= urlencode($userId) ?>&project_id=<?= urlencode($project['id']) ?>" class="action-btn btn-upload" title="Ladda upp bilder">
                                    <svg class="icon" viewBox="0 0 24 24" fill="currentColor">
                                        <path d="M9 2L7.17 4H4c-1.1 0-2 .9-2 2v12c0 1.1.9 2 2 2h16c1.1 0 2-.9 2-2V6c0-1.1-.9-2-2-2h-3.17L15 2H9zm3 15c-2.76 0-5-2.24-5-5s2.24-5 5-5 5 2.24 5 5-2.24 5-5 5z"/>
                                    </svg>
                                </a>
                            <?php endif; ?>
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>
        <?php endif; ?>
    </div>
</body>
</html>
