<?php
require_once __DIR__ . '/DataStoreInterface.php';
require_once __DIR__ . '/JsonDataStore.php';
require_once __DIR__ . '/MysqlDataStore.php';

/**
 * Factory för att skapa rätt datastore baserat på konfiguration
 */
class DataStoreFactory {
    private static $instances = [];
    
    /**
     * Skapa eller hämta datastore-instans
     * @param string $type 'json' eller 'mysql'
     * @param array $config Konfiguration för datastore
     * @return DataStoreInterface
     */
    public static function create($type = null, $config = []) {
        // Om typ inte anges, läs från konfiguration
        if ($type === null) {
            $type = defined('DATA_STORE_TYPE') ? DATA_STORE_TYPE : 'json';
        }
        
        // Använd singleton pattern för att undvika flera anslutningar
        $cacheKey = $type . serialize($config);
        
        if (isset(self::$instances[$cacheKey])) {
            return self::$instances[$cacheKey];
        }
        
        switch (strtolower($type)) {
            case 'mysql':
                // Om konfiguration inte anges explicit, läs från DB_CONFIG
                if (empty($config)) {
                    $dbConfig = defined('DB_CONFIG') ? DB_CONFIG : 'primary';
                    
                    if ($dbConfig === 'secondary' && defined('DB_SECONDARY_HOST')) {
                        // Använd sekundär konfiguration
                        $host = DB_SECONDARY_HOST;
                        $dbname = defined('DB_SECONDARY_NAME') ? DB_SECONDARY_NAME : 'byggarkiv_secondary';
                        $username = defined('DB_SECONDARY_USER') ? DB_SECONDARY_USER : 'root';
                        $password = defined('DB_SECONDARY_PASS') ? DB_SECONDARY_PASS : '';
                        $tablePrefix = defined('DB_SECONDARY_TABLE_PREFIX') ? DB_SECONDARY_TABLE_PREFIX : '';
                    } else {
                        // Använd primär konfiguration (standard)
                        $host = defined('DB_PRIMARY_HOST') ? DB_PRIMARY_HOST : (defined('DB_HOST') ? DB_HOST : 'localhost');
                        $dbname = defined('DB_PRIMARY_NAME') ? DB_PRIMARY_NAME : (defined('DB_NAME') ? DB_NAME : 'byggarkiv');
                        $username = defined('DB_PRIMARY_USER') ? DB_PRIMARY_USER : (defined('DB_USER') ? DB_USER : 'root');
                        $password = defined('DB_PRIMARY_PASS') ? DB_PRIMARY_PASS : (defined('DB_PASS') ? DB_PASS : '');
                        $tablePrefix = defined('DB_PRIMARY_TABLE_PREFIX') ? DB_PRIMARY_TABLE_PREFIX : (defined('DB_TABLE_PREFIX') ? DB_TABLE_PREFIX : '');
                    }
                } else {
                    // Explicit konfiguration angiven, använd den
                    $host = $config['host'] ?? (defined('DB_HOST') ? DB_HOST : 'localhost');
                    $dbname = $config['dbname'] ?? (defined('DB_NAME') ? DB_NAME : 'byggarkiv');
                    $username = $config['username'] ?? (defined('DB_USER') ? DB_USER : 'root');
                    $password = $config['password'] ?? (defined('DB_PASS') ? DB_PASS : '');
                    $tablePrefix = $config['table_prefix'] ?? (defined('DB_TABLE_PREFIX') ? DB_TABLE_PREFIX : '');
                }
                
                $charset = $config['charset'] ?? 'utf8mb4';
                
                $store = new MysqlDataStore($host, $dbname, $username, $password, $charset, $tablePrefix);
                break;
                
            case 'json':
            default:
                $basePath = $config['base_path'] ?? __DIR__ . '/../../data';
                $store = new JsonDataStore($basePath);
                break;
        }
        
        self::$instances[$cacheKey] = $store;
        return $store;
    }
    
    /**
     * Skapa datastore med specifik databaskonfiguration
     * @param string $configName 'primary' eller 'secondary'
     * @param string $type 'json' eller 'mysql' (standard: 'mysql')
     * @return DataStoreInterface
     */
    public static function createWithConfig($configName, $type = 'mysql') {
        if ($configName === 'secondary') {
            $config = [
                'host' => defined('DB_SECONDARY_HOST') ? DB_SECONDARY_HOST : 'localhost',
                'dbname' => defined('DB_SECONDARY_NAME') ? DB_SECONDARY_NAME : 'byggarkiv_secondary',
                'username' => defined('DB_SECONDARY_USER') ? DB_SECONDARY_USER : 'root',
                'password' => defined('DB_SECONDARY_PASS') ? DB_SECONDARY_PASS : '',
                'table_prefix' => defined('DB_SECONDARY_TABLE_PREFIX') ? DB_SECONDARY_TABLE_PREFIX : ''
            ];
        } else {
            $config = [
                'host' => defined('DB_PRIMARY_HOST') ? DB_PRIMARY_HOST : (defined('DB_HOST') ? DB_HOST : 'localhost'),
                'dbname' => defined('DB_PRIMARY_NAME') ? DB_PRIMARY_NAME : (defined('DB_NAME') ? DB_NAME : 'byggarkiv'),
                'username' => defined('DB_PRIMARY_USER') ? DB_PRIMARY_USER : (defined('DB_USER') ? DB_USER : 'root'),
                'password' => defined('DB_PRIMARY_PASS') ? DB_PRIMARY_PASS : (defined('DB_PASS') ? DB_PASS : ''),
                'table_prefix' => defined('DB_PRIMARY_TABLE_PREFIX') ? DB_PRIMARY_TABLE_PREFIX : (defined('DB_TABLE_PREFIX') ? DB_TABLE_PREFIX : '')
            ];
        }
        
        return self::create($type, $config);
    }
    
    /**
     * Rensa cacheade instanser (användbart för testing)
     */
    public static function clearCache() {
        self::$instances = [];
    }
}
?>

