<?php
require_once __DIR__ . '/DataStoreInterface.php';
// Logger kan finnas eller inte, använd function_exists() för att kontrollera
if (file_exists(__DIR__ . '/../../logger.php')) {
    require_once __DIR__ . '/../../logger.php';
}

/**
 * JSON-baserad datastore
 * Wrappar befintlig JSON-läsning/skrivning i ett interface
 */
class JsonDataStore implements DataStoreInterface {
    private $basePath;
    
    public function __construct($basePath = null) {
        if ($basePath === null) {
            $basePath = __DIR__ . '/../../data';
        }
        $this->basePath = rtrim($basePath, '/\\');
    }
    
    /**
     * Hämta sökväg till JSON-fil från nyckel
     */
    private function getFilePath($key) {
        // Om nyckeln redan är en sökväg, använd den direkt
        if (strpos($key, '/') !== false || strpos($key, '\\') !== false) {
            // Kontrollera om det är en absolut sökväg eller relativ
            if (file_exists($key)) {
                return $key;
            }
            // Relativ sökväg från basePath
            return $this->basePath . '/' . ltrim($key, '/\\');
        }
        
        // Standard: lägg till .json om det saknas
        $filename = $key;
        if (substr($filename, -5) !== '.json') {
            $filename .= '.json';
        }
        
        return $this->basePath . '/' . $filename;
    }
    
    public function get($key) {
        $filePath = $this->getFilePath($key);
        
        // Debug: Logga om filen inte hittas
        if (!file_exists($filePath)) {
            if (function_exists('logDebug')) {
                logDebug('datastore', 'File not found', [
                    'key' => $key,
                    'file_path' => $filePath,
                    'base_path' => $this->basePath,
                    'file_exists' => false
                ]);
            }
            return null;
        }
        
        // Debug: Logga om filen inte är läsbar
        if (!is_readable($filePath)) {
            if (function_exists('logWarning')) {
                logWarning('datastore', 'File not readable', [
                    'key' => $key,
                    'file_path' => $filePath
                ]);
            }
            return null;
        }
        
        $content = file_get_contents($filePath);
        if ($content === false) {
            if (function_exists('logWarning')) {
                logWarning('datastore', 'Failed to read file', [
                    'key' => $key,
                    'file_path' => $filePath
                ]);
            }
            return null;
        }
        
        $data = json_decode($content, true);
        if (json_last_error() !== JSON_ERROR_NONE) {
            $originalError = json_last_error_msg();
            
            // Försök reparera vanliga JSON-fel (t.ex. trailing comma)
            $repairedContent = $content;
            
            // Ta bort trailing comma före stängande bracket/brace
            $repairedContent = preg_replace('/,\s*([}\]])/', '$1', $repairedContent);
            
            // Försök igen med reparerad content
            $data = json_decode($repairedContent, true);
            
            if (json_last_error() !== JSON_ERROR_NONE) {
                if (function_exists('logError')) {
                    logError('datastore', 'JSON decode error (after repair attempt)', [
                        'key' => $key,
                        'file_path' => $filePath,
                        'json_error' => json_last_error_msg(),
                        'original_error' => $originalError,
                        'content_preview' => substr($content, 0, 200) . '...',
                        'content_length' => strlen($content)
                    ]);
                }
                return null;
            } else {
                // Reparation lyckades - logga varning och spara den reparerade filen
                if (function_exists('logWarning')) {
                    logWarning('datastore', 'JSON file had syntax error but was repaired', [
                        'key' => $key,
                        'file_path' => $filePath,
                        'original_error' => $originalError
                    ]);
                }
                
                // Spara den reparerade filen för framtida användning
                @file_put_contents($filePath, json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
            }
        }
        
        return $data;
    }
    
    public function set($key, $data) {
        $filePath = $this->getFilePath($key);
        
        // Skapa katalog om den inte finns
        $dir = dirname($filePath);
        if (!is_dir($dir)) {
            mkdir($dir, 0755, true);
        }
        
        $json = json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
        return file_put_contents($filePath, $json) !== false;
    }
    
    public function delete($key) {
        $filePath = $this->getFilePath($key);
        
        if (!file_exists($filePath)) {
            return true; // Redan borttagen
        }
        
        return unlink($filePath);
    }
    
    public function exists($key) {
        $filePath = $this->getFilePath($key);
        return file_exists($filePath);
    }
    
    public function getAllKeys($prefix = '') {
        $keys = [];
        
        if (!is_dir($this->basePath)) {
            return $keys;
        }
        
        $files = glob($this->basePath . '/' . $prefix . '*.json');
        foreach ($files as $file) {
            $key = basename($file, '.json');
            if ($prefix !== '') {
                $key = substr($key, strlen($prefix));
            }
            $keys[] = $key;
        }
        
        return $keys;
    }
    
    public function getMultiple($keys) {
        $result = [];
        foreach ($keys as $key) {
            $result[$key] = $this->get($key);
        }
        return $result;
    }
    
    public function setMultiple($data) {
        $success = true;
        foreach ($data as $key => $value) {
            if (!$this->set($key, $value)) {
                $success = false;
            }
        }
        return $success;
    }
}
?>



