<?php
require_once __DIR__ . '/DataStoreInterface.php';

/**
 * MySQL-baserad datastore
 * Använder PDO med prepared statements för säkerhet
 */
class MysqlDataStore implements DataStoreInterface {
    private $pdo;
    private $tablePrefix;
    
    public function __construct($host, $dbname, $username, $password, $charset = 'utf8mb4', $tablePrefix = '') {
        $dsn = "mysql:host={$host};dbname={$dbname};charset={$charset}";
        $options = [
            PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
            PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
            PDO::ATTR_EMULATE_PREPARES => false,
        ];
        
        try {
            $this->pdo = new PDO($dsn, $username, $password, $options);
        } catch (PDOException $e) {
            throw new Exception("Database connection failed: " . $e->getMessage());
        }
        
        $this->tablePrefix = $tablePrefix;
    }
    
    /**
     * Hämta tabellnamn med prefix
     */
    private function getTableName($name) {
        return $this->tablePrefix . $name;
    }
    
    /**
     * Konvertera nyckel till tabellnamn och ID
     * Format: "table:key" eller bara "key" (för generiska tabeller)
     */
    private function parseKey($key) {
        if (strpos($key, ':') !== false) {
            list($table, $id) = explode(':', $key, 2);
            return [$table, $id];
        }
        // Standard: använd 'data' som generisk tabell
        return ['data', $key];
    }
    
    public function get($key) {
        list($table, $id) = $this->parseKey($key);
        $tableName = $this->getTableName($table);
        
        try {
            $stmt = $this->pdo->prepare("SELECT data FROM {$tableName} WHERE `key` = ?");
            $stmt->execute([$id]);
            $row = $stmt->fetch();
            
            if ($row === false) {
                return null;
            }
            
            return json_decode($row['data'], true);
        } catch (PDOException $e) {
            error_log("MysqlDataStore::get error: " . $e->getMessage());
            return null;
        }
    }
    
    public function set($key, $data) {
        list($table, $id) = $this->parseKey($key);
        $tableName = $this->getTableName($table);
        
        try {
            $jsonData = json_encode($data, JSON_UNESCAPED_UNICODE);
            
            // Försök uppdatera först
            $stmt = $this->pdo->prepare("UPDATE {$tableName} SET data = ?, updated_at = NOW() WHERE `key` = ?");
            $stmt->execute([$jsonData, $id]);
            
            // Om ingen rad uppdaterades, skapa ny
            if ($stmt->rowCount() === 0) {
                $stmt = $this->pdo->prepare("INSERT INTO {$tableName} (`key`, data, created_at, updated_at) VALUES (?, ?, NOW(), NOW())");
                $stmt->execute([$id, $jsonData]);
            }
            
            return true;
        } catch (PDOException $e) {
            error_log("MysqlDataStore::set error: " . $e->getMessage());
            return false;
        }
    }
    
    public function delete($key) {
        list($table, $id) = $this->parseKey($key);
        $tableName = $this->getTableName($table);
        
        try {
            $stmt = $this->pdo->prepare("DELETE FROM {$tableName} WHERE `key` = ?");
            $stmt->execute([$id]);
            return true;
        } catch (PDOException $e) {
            error_log("MysqlDataStore::delete error: " . $e->getMessage());
            return false;
        }
    }
    
    public function exists($key) {
        list($table, $id) = $this->parseKey($key);
        $tableName = $this->getTableName($table);
        
        try {
            $stmt = $this->pdo->prepare("SELECT COUNT(*) as count FROM {$tableName} WHERE `key` = ?");
            $stmt->execute([$id]);
            $row = $stmt->fetch();
            return $row['count'] > 0;
        } catch (PDOException $e) {
            error_log("MysqlDataStore::exists error: " . $e->getMessage());
            return false;
        }
    }
    
    public function getAllKeys($prefix = '') {
        // För MySQL behöver vi veta vilken tabell vi ska söka i
        // Standard: sök i 'data' tabellen
        $tableName = $this->getTableName('data');
        
        try {
            if ($prefix !== '') {
                $stmt = $this->pdo->prepare("SELECT `key` FROM {$tableName} WHERE `key` LIKE ?");
                $stmt->execute([$prefix . '%']);
            } else {
                $stmt = $this->pdo->query("SELECT `key` FROM {$tableName}");
            }
            
            $keys = [];
            while ($row = $stmt->fetch()) {
                $keys[] = $row['key'];
            }
            
            return $keys;
        } catch (PDOException $e) {
            error_log("MysqlDataStore::getAllKeys error: " . $e->getMessage());
            return [];
        }
    }
    
    public function getMultiple($keys) {
        $result = [];
        foreach ($keys as $key) {
            $result[$key] = $this->get($key);
        }
        return $result;
    }
    
    public function setMultiple($data) {
        $this->pdo->beginTransaction();
        
        try {
            $success = true;
            foreach ($data as $key => $value) {
                if (!$this->set($key, $value)) {
                    $success = false;
                    break;
                }
            }
            
            if ($success) {
                $this->pdo->commit();
            } else {
                $this->pdo->rollBack();
            }
            
            return $success;
        } catch (Exception $e) {
            $this->pdo->rollBack();
            error_log("MysqlDataStore::setMultiple error: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Hämta PDO-instansen för avancerade queries
     */
    public function getPdo() {
        return $this->pdo;
    }
}
?>




