<?php
require_once __DIR__ . '/../DataStoreFactory.php';

/**
 * Repository för admin-länk-statistik
 * Hanterar klick-statistik för admin-länkar per användare
 */
class AdminLinkStatsRepository {
    private $store;
    private $statsFile;
    
    public function __construct($store = null) {
        if ($store === null) {
            $this->store = DataStoreFactory::create();
        } else {
            $this->store = $store;
        }
        
        // Använd JSON-fil för statistik (enklare än att skapa MySQL-tabell)
        $this->statsFile = __DIR__ . '/../../../data/admin_link_stats.json';
    }
    
    /**
     * Registrera ett klick på en länk
     */
    public function recordClick($userId, $featureId, $url, $title) {
        $stats = $this->loadStats();
        
        if (!isset($stats[$userId])) {
            $stats[$userId] = [];
        }
        
        if (!isset($stats[$userId][$featureId])) {
            $stats[$userId][$featureId] = [
                'count' => 0,
                'last_clicked' => null,
                'url' => $url,
                'title' => $title
            ];
        }
        
        $stats[$userId][$featureId]['count']++;
        $stats[$userId][$featureId]['last_clicked'] = date('c');
        $stats[$userId][$featureId]['url'] = $url;
        $stats[$userId][$featureId]['title'] = $title;
        
        return $this->saveStats($stats);
    }
    
    /**
     * Hämta de populäraste länkarna för en användare
     */
    public function getTopLinks($userId, $limit = 5) {
        $stats = $this->loadStats();
        
        if (!isset($stats[$userId]) || empty($stats[$userId])) {
            return [];
        }
        
        $userStats = $stats[$userId];
        
        // Sortera efter count (högst först), sedan last_clicked (nyast först)
        uasort($userStats, function($a, $b) {
            if ($a['count'] !== $b['count']) {
                return $b['count'] - $a['count']; // Högre count först
            }
            // Om count är samma, sortera efter last_clicked (nyast först)
            $timeA = strtotime($a['last_clicked'] ?? '1970-01-01');
            $timeB = strtotime($b['last_clicked'] ?? '1970-01-01');
            return $timeB - $timeA;
        });
        
        // Ta de första $limit länkarna
        $topLinks = array_slice($userStats, 0, $limit, true);
        
        // Lägg till feature_id i resultatet
        $result = [];
        foreach ($topLinks as $featureId => $data) {
            $result[] = [
                'feature_id' => $featureId,
                'count' => $data['count'],
                'last_clicked' => $data['last_clicked'],
                'url' => $data['url'],
                'title' => $data['title']
            ];
        }
        
        return $result;
    }
    
    /**
     * Nollställ statistiken för en användare
     */
    public function resetStats($userId) {
        $stats = $this->loadStats();
        
        if (isset($stats[$userId])) {
            unset($stats[$userId]);
        }
        
        return $this->saveStats($stats);
    }
    
    /**
     * Ladda statistik från fil
     */
    private function loadStats() {
        if (!file_exists($this->statsFile)) {
            return [];
        }
        
        $content = @file_get_contents($this->statsFile);
        if ($content === false) {
            return [];
        }
        
        $stats = json_decode($content, true);
        return is_array($stats) ? $stats : [];
    }
    
    /**
     * Spara statistik till fil
     */
    private function saveStats($stats) {
        $dir = dirname($this->statsFile);
        if (!is_dir($dir)) {
            @mkdir($dir, 0755, true);
        }
        
        $json = json_encode($stats, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
        return @file_put_contents($this->statsFile, $json, LOCK_EX) !== false;
    }
}
?>

