<?php
require_once __DIR__ . '/../DataStoreFactory.php';

/**
 * Repository för chatmeddelanden
 */
class ChatMessageRepository {
    private $store;
    private $chatsDir;
    
    public function __construct($store = null, $chatsDir = null) {
        if ($store === null) {
            $this->store = DataStoreFactory::create();
        } else {
            $this->store = $store;
        }
        
        if ($chatsDir === null) {
            $this->chatsDir = __DIR__ . '/../../chats';
        } else {
            $this->chatsDir = $chatsDir;
        }
    }
    
    /**
     * Hämta alla meddelanden för ett projekt
     */
    public function getByProject($projectId) {
        if ($this->store instanceof \MysqlDataStore) {
            return $this->getByProjectFromMysql($projectId);
        } else {
            return $this->getByProjectFromJson($projectId);
        }
    }
    
    private function getByProjectFromJson($projectId) {
        $chatFile = $this->chatsDir . '/' . $projectId . '/messages.json';
        if (!file_exists($chatFile)) {
            return [];
        }
        
        $content = file_get_contents($chatFile);
        if ($content === false) {
            return [];
        }
        
        $messages = json_decode($content, true);
        return is_array($messages) ? $messages : [];
    }
    
    private function getByProjectFromMysql($projectId) {
        try {
            $pdo = $this->store->getPdo();
            $stmt = $pdo->prepare("
                SELECT * FROM chat_messages 
                WHERE project_id = ? 
                ORDER BY created_at ASC
            ");
            $stmt->execute([$projectId]);
            
            $messages = [];
            while ($row = $stmt->fetch()) {
                $message = [
                    'id' => $row['id'],
                    'sender_type' => $row['sender_type'],
                    'participant_id' => $row['participant_id'],
                    'message' => $row['message'] ?? '',
                    'type' => $row['type'],
                    'media_file' => $row['media_file'] ?? null,
                    'priority' => $row['priority'] ?? null,
                    'reply_to' => $row['reply_to'] ?? null,
                    'created_at' => $row['created_at'],
                ];
                
                // Parse notified_users JSON
                if (!empty($row['notified_users'])) {
                    $message['notified_users'] = json_decode($row['notified_users'], true);
                    if (!is_array($message['notified_users'])) {
                        $message['notified_users'] = [];
                    }
                } else {
                    $message['notified_users'] = [];
                }
                
                $messages[] = $message;
            }
            
            return $messages;
        } catch (PDOException $e) {
            error_log("ChatMessageRepository::getByProjectFromMysql error: " . $e->getMessage());
            return [];
        } catch (Exception $e) {
            error_log("ChatMessageRepository::getByProjectFromMysql error: " . $e->getMessage());
            return [];
        }
    }
    
    /**
     * Skapa nytt meddelande
     */
    public function create($projectId, $messageData) {
        if ($this->store instanceof \MysqlDataStore) {
            return $this->createInMysql($projectId, $messageData);
        } else {
            return $this->createInJson($projectId, $messageData);
        }
    }
    
    private function createInJson($projectId, $messageData) {
        $messages = $this->getByProject($projectId);
        $messages[] = $messageData;
        return $this->saveMessages($projectId, $messages);
    }
    
    private function createInMysql($projectId, $messageData) {
        $pdo = $this->store->getPdo();
        
        try {
            $stmt = $pdo->prepare("
                INSERT INTO chat_messages 
                (id, project_id, sender_type, participant_id, message, type, media_file, priority, reply_to, notified_users, created_at)
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
            ");
            
            $notifiedUsersJson = !empty($messageData['notified_users']) 
                ? json_encode($messageData['notified_users'], JSON_UNESCAPED_UNICODE) 
                : null;
            
            $stmt->execute([
                $messageData['id'],
                $projectId,
                $messageData['sender_type'] ?? 'user',
                $messageData['participant_id'] ?? '',
                $messageData['message'] ?? '',
                $messageData['type'] ?? 'text',
                $messageData['media_file'] ?? null,
                $messageData['priority'] ?? null,
                $messageData['reply_to'] ?? null,
                $notifiedUsersJson,
                $messageData['created_at'] ?? date('c')
            ]);
            
            return true;
        } catch (PDOException $e) {
            error_log("ChatMessageRepository::createInMysql error: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Uppdatera meddelande
     */
    public function update($projectId, $messageId, $messageData) {
        if ($this->store instanceof \MysqlDataStore) {
            return $this->updateInMysql($projectId, $messageId, $messageData);
        } else {
            return $this->updateInJson($projectId, $messageId, $messageData);
        }
    }
    
    private function updateInJson($projectId, $messageId, $messageData) {
        $messages = $this->getByProject($projectId);
        $found = false;
        
        foreach ($messages as &$message) {
            if (($message['id'] ?? '') === $messageId) {
                $message = array_merge($message, $messageData);
                $message['id'] = $messageId; // Säkerställ att ID inte ändras
                $found = true;
                break;
            }
        }
        
        if (!$found) {
            return false;
        }
        
        return $this->saveMessages($projectId, $messages);
    }
    
    private function updateInMysql($projectId, $messageId, $messageData) {
        $pdo = $this->store->getPdo();
        
        try {
            $fields = [];
            $values = [];
            
            if (isset($messageData['message'])) {
                $fields[] = 'message = ?';
                $values[] = $messageData['message'];
            }
            if (isset($messageData['notified_users'])) {
                $fields[] = 'notified_users = ?';
                $values[] = json_encode($messageData['notified_users'], JSON_UNESCAPED_UNICODE);
            }
            
            if (empty($fields)) {
                return true; // Inga ändringar
            }
            
            $values[] = $projectId;
            $values[] = $messageId;
            
            $sql = "UPDATE chat_messages SET " . implode(', ', $fields) . " WHERE project_id = ? AND id = ?";
            $stmt = $pdo->prepare($sql);
            $stmt->execute($values);
            
            return true;
        } catch (PDOException $e) {
            error_log("ChatMessageRepository::updateInMysql error: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Ta bort meddelande
     */
    public function delete($projectId, $messageId, $userId = null) {
        $result = false;
        if ($this->store instanceof \MysqlDataStore) {
            $result = $this->deleteFromMysql($projectId, $messageId);
        } else {
            $result = $this->deleteFromJson($projectId, $messageId);
        }
        
        // Logga händelsen om delete lyckades
        if ($result && $userId !== null) {
            $this->logChatEvent($projectId, $userId, 'delete_message', $messageId);
        }
        
        return $result;
    }
    
    /**
     * Logga chatt-händelse
     */
    private function logChatEvent($projectId, $userId, $action, $messageId = null) {
        $logFile = __DIR__ . '/../../../data/chat_events.json';
        $logDir = dirname($logFile);
        
        // Skapa data-mappen om den inte finns
        if (!is_dir($logDir)) {
            @mkdir($logDir, 0755, true);
        }
        
        // Läs befintliga händelser
        $events = [];
        if (file_exists($logFile)) {
            $content = @file_get_contents($logFile);
            if ($content !== false) {
                $decoded = json_decode($content, true);
                if (is_array($decoded)) {
                    $events = $decoded;
                }
            }
        }
        
        // Lägg till ny händelse
        $event = [
            'timestamp' => date('c'),
            'project_id' => $projectId,
            'user_id' => $userId,
            'action' => $action
        ];
        
        if ($messageId !== null) {
            $event['message_id'] = $messageId;
        }
        
        $events[] = $event;
        
        // Begränsa till senaste 1000 händelser
        if (count($events) > 1000) {
            $events = array_slice($events, -1000);
        }
        
        // Spara tillbaka
        @file_put_contents($logFile, json_encode($events, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE), LOCK_EX);
    }
    
    private function deleteFromJson($projectId, $messageId) {
        $messages = $this->getByProject($projectId);
        $found = false;
        
        foreach ($messages as $index => $message) {
            if (($message['id'] ?? '') === $messageId) {
                unset($messages[$index]);
                $found = true;
                break;
            }
        }
        
        if (!$found) {
            return false;
        }
        
        $messages = array_values($messages); // Re-index
        return $this->saveMessages($projectId, $messages);
    }
    
    private function deleteFromMysql($projectId, $messageId) {
        $pdo = $this->store->getPdo();
        
        try {
            $stmt = $pdo->prepare("DELETE FROM chat_messages WHERE project_id = ? AND id = ?");
            $stmt->execute([$projectId, $messageId]);
            return true;
        } catch (PDOException $e) {
            error_log("ChatMessageRepository::deleteFromMysql error: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Hjälpmetod för att spara meddelanden (JSON)
     */
    private function saveMessages($projectId, $messages) {
        $chatDir = $this->chatsDir . '/' . $projectId;
        if (!is_dir($chatDir)) {
            mkdir($chatDir, 0755, true);
        }
        $chatFile = $chatDir . '/messages.json';
        $json = json_encode($messages, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
        return file_put_contents($chatFile, $json) !== false;
    }
}
?>

