<?php
require_once __DIR__ . '/../DataStoreFactory.php';

/**
 * Repository för konfigurationsfiler
 * Hanterar external_companies, external_staff, billing_config, etc.
 */
class ConfigRepository {
    private $store;
    
    public function __construct($store = null) {
        if ($store === null) {
            $this->store = DataStoreFactory::create();
        } else {
            $this->store = $store;
        }
    }
    
    /**
     * Hämta konfiguration
     */
    public function get($configKey) {
        return $this->store->get($configKey);
    }
    
    /**
     * Spara konfiguration
     */
    public function set($configKey, $data) {
        return $this->store->set($configKey, $data);
    }
    
    /**
     * Hämta external_companies
     */
    public function getExternalCompanies() {
        $data = $this->get('external_companies');
        return is_array($data) ? $data : [];
    }
    
    /**
     * Spara external_companies
     */
    public function setExternalCompanies($data) {
        return $this->set('external_companies', $data);
    }
    
    /**
     * Hämta external_types
     */
    public function getExternalTypes() {
        $data = $this->get('external_types');
        
        // Debug: Logga om data är null eller inte en array
        if ($data === null) {
            if (function_exists('logWarning')) {
                logWarning('config', 'DataStore returned null for external_types', [
                    'store_type' => get_class($this->store),
                    'key' => 'external_types'
                ]);
            }
        } elseif (!is_array($data)) {
            if (function_exists('logWarning')) {
                logWarning('config', 'DataStore returned non-array for external_types', [
                    'store_type' => get_class($this->store),
                    'key' => 'external_types',
                    'data_type' => gettype($data)
                ]);
            }
        }
        
        return is_array($data) ? $data : [];
    }
    
    /**
     * Spara external_types
     */
    public function setExternalTypes($data) {
        return $this->set('external_types', $data);
    }
    
    /**
     * Hämta billing_config
     */
    public function getBillingConfig() {
        $data = $this->get('billing_config');
        return is_array($data) ? $data : [];
    }
    
    /**
     * Spara billing_config
     */
    public function setBillingConfig($data) {
        return $this->set('billing_config', $data);
    }
    
    /**
     * Hämta upload_limits
     */
    public function getUploadLimits() {
        $data = $this->get('upload_limits');
        return is_array($data) ? $data : [];
    }
    
    /**
     * Spara upload_limits
     */
    public function setUploadLimits($data) {
        return $this->set('upload_limits', $data);
    }
    
    /**
     * Hämta image_numbers (singleton)
     */
    public function getImageNumbers() {
        if ($this->store instanceof \MysqlDataStore) {
            return $this->getImageNumbersFromMysql();
        } else {
            return $this->getImageNumbersFromJson();
        }
    }
    
    private function getImageNumbersFromJson() {
        $data = $this->get('image_numbers');
        if (!is_array($data)) {
            return ['next_number' => 1];
        }
        return $data;
    }
    
    private function getImageNumbersFromMysql() {
        $pdo = $this->store->getPdo();
        $stmt = $pdo->query("SELECT next_number FROM image_numbers LIMIT 1");
        $row = $stmt->fetch();
        
        if (!$row) {
            // Skapa initialt värde
            $stmt = $pdo->prepare("INSERT INTO image_numbers (next_number) VALUES (1)");
            $stmt->execute();
            return ['next_number' => 1];
        }
        
        return ['next_number' => (int)$row['next_number']];
    }
    
    /**
     * Öka image number atomiskt
     */
    public function incrementImageNumber() {
        if ($this->store instanceof \MysqlDataStore) {
            return $this->incrementImageNumberInMysql();
        } else {
            return $this->incrementImageNumberInJson();
        }
    }
    
    private function incrementImageNumberInJson() {
        $numbersFile = __DIR__ . '/../../data/image_numbers.json';
        $lockFile = $numbersFile . '.lock';
        
        // Skapa data-mappen om den inte finns
        $dataDir = dirname($numbersFile);
        if (!is_dir($dataDir)) {
            if (!@mkdir($dataDir, 0755, true)) {
                // Om mappen inte kan skapas, logga och returnera ett standardvärde
                error_log("Kunde inte skapa data-mapp: $dataDir");
                return 1;
            }
        }
        
        // Skapa filen om den inte finns
        if (!file_exists($numbersFile)) {
            $initialData = ['next_number' => 1];
            if (@file_put_contents($numbersFile, json_encode($initialData, JSON_PRETTY_PRINT)) === false) {
                error_log("Kunde inte skapa image_numbers.json: $numbersFile");
                return 1;
            }
        }
        
        // Enkel filbaserad låsning
        $maxWait = 5;
        $waited = 0;
        while (file_exists($lockFile) && $waited < $maxWait) {
            usleep(100000);
            $waited += 0.1;
        }
        
        // Försök skapa lock-fil, men fortsätt även om det misslyckas
        @file_put_contents($lockFile, getmypid());
        
        try {
            $content = @file_get_contents($numbersFile);
            if ($content === false) {
                // Om filen inte kan läsas, returnera ett standardvärde
                error_log("Kunde inte läsa image_numbers.json: $numbersFile");
                return 1;
            }
            
            $data = json_decode($content, true);
            if (!is_array($data) || !isset($data['next_number'])) {
                $data = ['next_number' => 1];
            }
            
            $nextNumber = (int)$data['next_number'];
            $data['next_number'] = $nextNumber + 1;
            
            // Försök spara, men fortsätt även om det misslyckas
            if (@file_put_contents($numbersFile, json_encode($data, JSON_PRETTY_PRINT)) === false) {
                error_log("Kunde inte spara image_numbers.json: $numbersFile");
                // Returnera numret ändå eftersom vi redan har räknat upp det
            }
            
            return $nextNumber;
        } finally {
            if (file_exists($lockFile)) {
                @unlink($lockFile);
            }
        }
    }
    
    private function incrementImageNumberInMysql() {
        $pdo = $this->store->getPdo();
        
        try {
            $pdo->beginTransaction();
            
            // Lås raden för update
            $stmt = $pdo->query("SELECT next_number FROM image_numbers FOR UPDATE");
            $row = $stmt->fetch();
            
            if (!$row) {
                // Skapa initialt värde
                $stmt = $pdo->prepare("INSERT INTO image_numbers (next_number) VALUES (1)");
                $stmt->execute();
                $nextNumber = 1;
            } else {
                $nextNumber = (int)$row['next_number'];
            }
            
            // Öka numret
            $stmt = $pdo->prepare("UPDATE image_numbers SET next_number = next_number + 1");
            $stmt->execute();
            
            $pdo->commit();
            
            return $nextNumber;
        } catch (PDOException $e) {
            $pdo->rollBack();
            error_log("ConfigRepository::incrementImageNumberInMysql error: " . $e->getMessage());
            return false;
        }
    }
}
?>

