<?php
require_once __DIR__ . '/../DataStoreFactory.php';

/**
 * Repository för chat och project mappings
 */
class MappingRepository {
    private $store;
    
    public function __construct($store = null) {
        if ($store === null) {
            $this->store = DataStoreFactory::create();
        } else {
            $this->store = $store;
        }
    }
    
    /**
     * Hämta chat mappings för ett projekt
     */
    public function getChatMappings($projectId = null) {
        if ($this->store instanceof \MysqlDataStore) {
            return $this->getChatMappingsFromMysql($projectId);
        } else {
            return $this->getChatMappingsFromJson($projectId);
        }
    }
    
    private function getChatMappingsFromJson($projectId = null) {
        $mapping = $this->store->get('chat_mapping');
        if (!is_array($mapping)) {
            return $projectId ? [] : [];
        }
        
        if ($projectId !== null) {
            return $mapping[$projectId] ?? [];
        }
        
        return $mapping;
    }
    
    private function getChatMappingsFromMysql($projectId = null) {
        $pdo = $this->store->getPdo();
        
        if ($projectId !== null) {
            $stmt = $pdo->prepare("
                SELECT * FROM chat_mappings 
                WHERE project_id = ?
                ORDER BY added_at ASC
            ");
            $stmt->execute([$projectId]);
        } else {
            $stmt = $pdo->query("
                SELECT * FROM chat_mappings 
                ORDER BY project_id, added_at ASC
            ");
        }
        
        $mappings = [];
        while ($row = $stmt->fetch()) {
            $mapping = [
                'type' => $row['type'],
                'participant_id' => $row['participant_id'],
                'added_at' => $row['added_at'],
                'added_by' => $row['added_by'] ?? '',
            ];
            
            if ($projectId === null) {
                if (!isset($mappings[$row['project_id']])) {
                    $mappings[$row['project_id']] = [];
                }
                $mappings[$row['project_id']][] = $mapping;
            } else {
                $mappings[] = $mapping;
            }
        }
        
        return $mappings;
    }
    
    /**
     * Hämta project mappings för ett projekt
     */
    public function getProjectMappings($projectId = null) {
        if ($this->store instanceof \MysqlDataStore) {
            return $this->getProjectMappingsFromMysql($projectId);
        } else {
            return $this->getProjectMappingsFromJson($projectId);
        }
    }
    
    private function getProjectMappingsFromJson($projectId = null) {
        $mapping = $this->store->get('project_mapping');
        if (!is_array($mapping)) {
            return $projectId ? [] : [];
        }
        
        if ($projectId !== null) {
            return $mapping[$projectId] ?? [];
        }
        
        return $mapping;
    }
    
    private function getProjectMappingsFromMysql($projectId = null) {
        $pdo = $this->store->getPdo();
        
        if ($projectId !== null) {
            $stmt = $pdo->prepare("
                SELECT * FROM project_mappings 
                WHERE project_id = ?
                ORDER BY added_at ASC
            ");
            $stmt->execute([$projectId]);
        } else {
            $stmt = $pdo->query("
                SELECT * FROM project_mappings 
                ORDER BY project_id, added_at ASC
            ");
        }
        
        $mappings = [];
        while ($row = $stmt->fetch()) {
            $mapping = [
                'type' => $row['type'],
                'participant_id' => $row['participant_id'],
                'added_at' => $row['added_at'],
                'added_by' => $row['added_by'] ?? '',
            ];
            
            if ($projectId === null) {
                if (!isset($mappings[$row['project_id']])) {
                    $mappings[$row['project_id']] = [];
                }
                $mappings[$row['project_id']][] = $mapping;
            } else {
                $mappings[] = $mapping;
            }
        }
        
        return $mappings;
    }
    
    /**
     * Lägg till chatter i ett projekt
     */
    public function addChatter($projectId, $participantId, $type, $addedBy) {
        if ($this->store instanceof \MysqlDataStore) {
            return $this->addChatterInMysql($projectId, $participantId, $type, $addedBy);
        } else {
            return $this->addChatterInJson($projectId, $participantId, $type, $addedBy);
        }
    }
    
    private function addChatterInJson($projectId, $participantId, $type, $addedBy) {
        $mapping = $this->getChatMappings();
        if (!isset($mapping[$projectId])) {
            $mapping[$projectId] = [];
        }
        
        // Kontrollera om redan finns
        foreach ($mapping[$projectId] as $chatter) {
            if (($chatter['participant_id'] ?? '') === $participantId) {
                return false; // Redan finns
            }
        }
        
        $mapping[$projectId][] = [
            'type' => $type,
            'participant_id' => $participantId,
            'added_at' => date('c'),
            'added_by' => $addedBy
        ];
        
        return $this->store->set('chat_mapping', $mapping);
    }
    
    private function addChatterInMysql($projectId, $participantId, $type, $addedBy) {
        $pdo = $this->store->getPdo();
        
        try {
            // Kontrollera om redan finns
            $stmt = $pdo->prepare("
                SELECT COUNT(*) as count FROM chat_mappings 
                WHERE project_id = ? AND participant_id = ?
            ");
            $stmt->execute([$projectId, $participantId]);
            $row = $stmt->fetch();
            
            if ($row['count'] > 0) {
                return false; // Redan finns
            }
            
            $stmt = $pdo->prepare("
                INSERT INTO chat_mappings (project_id, participant_id, type, added_at, added_by)
                VALUES (?, ?, ?, NOW(), ?)
            ");
            $stmt->execute([$projectId, $participantId, $type, $addedBy]);
            
            return true;
        } catch (PDOException $e) {
            error_log("MappingRepository::addChatterInMysql error: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Ta bort chatter från ett projekt
     */
    public function removeChatter($projectId, $participantId) {
        if ($this->store instanceof \MysqlDataStore) {
            return $this->removeChatterFromMysql($projectId, $participantId);
        } else {
            return $this->removeChatterFromJson($projectId, $participantId);
        }
    }
    
    private function removeChatterFromJson($projectId, $participantId) {
        $mapping = $this->getChatMappings();
        if (!isset($mapping[$projectId])) {
            return true; // Redan borttagen
        }
        
        $mapping[$projectId] = array_filter($mapping[$projectId], function($chatter) use ($participantId) {
            return ($chatter['participant_id'] ?? '') !== $participantId;
        });
        
        $mapping[$projectId] = array_values($mapping[$projectId]); // Re-index
        
        return $this->store->set('chat_mapping', $mapping);
    }
    
    private function removeChatterFromMysql($projectId, $participantId) {
        $pdo = $this->store->getPdo();
        
        try {
            $stmt = $pdo->prepare("
                DELETE FROM chat_mappings 
                WHERE project_id = ? AND participant_id = ?
            ");
            $stmt->execute([$projectId, $participantId]);
            return true;
        } catch (PDOException $e) {
            error_log("MappingRepository::removeChatterFromMysql error: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Lägg till participant i ett projekt
     */
    public function addParticipant($projectId, $participantId, $type, $addedBy) {
        if ($this->store instanceof \MysqlDataStore) {
            return $this->addParticipantInMysql($projectId, $participantId, $type, $addedBy);
        } else {
            return $this->addParticipantInJson($projectId, $participantId, $type, $addedBy);
        }
    }
    
    private function addParticipantInJson($projectId, $participantId, $type, $addedBy) {
        $mapping = $this->getProjectMappings();
        if (!isset($mapping[$projectId])) {
            $mapping[$projectId] = [];
        }
        
        // Kontrollera om redan finns
        foreach ($mapping[$projectId] as $participant) {
            if (($participant['participant_id'] ?? '') === $participantId) {
                return false; // Redan finns
            }
        }
        
        $mapping[$projectId][] = [
            'type' => $type,
            'participant_id' => $participantId,
            'added_at' => date('c'),
            'added_by' => $addedBy
        ];
        
        return $this->store->set('project_mapping', $mapping);
    }
    
    private function addParticipantInMysql($projectId, $participantId, $type, $addedBy) {
        $pdo = $this->store->getPdo();
        
        try {
            // Kontrollera om redan finns
            $stmt = $pdo->prepare("
                SELECT COUNT(*) as count FROM project_mappings 
                WHERE project_id = ? AND participant_id = ?
            ");
            $stmt->execute([$projectId, $participantId]);
            $row = $stmt->fetch();
            
            if ($row['count'] > 0) {
                return false; // Redan finns
            }
            
            $stmt = $pdo->prepare("
                INSERT INTO project_mappings (project_id, participant_id, type, added_at, added_by)
                VALUES (?, ?, ?, NOW(), ?)
            ");
            $stmt->execute([$projectId, $participantId, $type, $addedBy]);
            
            return true;
        } catch (PDOException $e) {
            error_log("MappingRepository::addParticipantInMysql error: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Ta bort participant från ett projekt
     */
    public function removeParticipant($projectId, $participantId) {
        if ($this->store instanceof \MysqlDataStore) {
            return $this->removeParticipantFromMysql($projectId, $participantId);
        } else {
            return $this->removeParticipantFromJson($projectId, $participantId);
        }
    }
    
    private function removeParticipantFromJson($projectId, $participantId) {
        $mapping = $this->getProjectMappings();
        if (!isset($mapping[$projectId])) {
            return true; // Redan borttagen
        }
        
        $mapping[$projectId] = array_filter($mapping[$projectId], function($participant) use ($participantId) {
            return ($participant['participant_id'] ?? '') !== $participantId;
        });
        
        $mapping[$projectId] = array_values($mapping[$projectId]); // Re-index
        
        return $this->store->set('project_mapping', $mapping);
    }
    
    private function removeParticipantFromMysql($projectId, $participantId) {
        $pdo = $this->store->getPdo();
        
        try {
            $stmt = $pdo->prepare("
                DELETE FROM project_mappings 
                WHERE project_id = ? AND participant_id = ?
            ");
            $stmt->execute([$projectId, $participantId]);
            return true;
        } catch (PDOException $e) {
            error_log("MappingRepository::removeParticipantFromMysql error: " . $e->getMessage());
            return false;
        }
    }
}
?>




