<?php
require_once __DIR__ . '/../DataStoreFactory.php';

/**
 * Repository för projekt-operationer
 */
class ProjectRepository {
    private $store;
    
    public function __construct($store = null) {
        if ($store === null) {
            $this->store = DataStoreFactory::create();
        } else {
            $this->store = $store;
        }
    }
    
    /**
     * Hämta alla projekt
     */
    public function getAll() {
        if ($this->store instanceof \MysqlDataStore) {
            return $this->getAllFromMysql();
        } else {
            return $this->getAllFromJson();
        }
    }
    
    private function getAllFromJson() {
        $projects = $this->store->get('projects');
        return is_array($projects) ? $projects : [];
    }
    
    private function getAllFromMysql() {
        $pdo = $this->store->getPdo();
        $stmt = $pdo->query("
            SELECT p.*, 
                   JSON_ARRAYAGG(
                       JSON_OBJECT(
                           'type', pp.type,
                           'participant_id', pp.participant_id,
                           'added_at', pp.added_at,
                           'added_by', pp.added_by
                       )
                   ) as participants
            FROM projects p
            LEFT JOIN project_participants pp ON p.id = pp.project_id
            GROUP BY p.id
            ORDER BY p.created_at DESC
        ");
        
        $projects = [];
        while ($row = $stmt->fetch()) {
            $project = [
                'id' => $row['id'],
                'name' => $row['name'],
                'description' => $row['description'] ?? '',
                'status' => $row['status'],
                'created_at' => $row['created_at'],
                'updated_at' => $row['updated_at'] ?? null,
                'archived_at' => $row['archived_at'] ?? null,
            ];
            
            if (!empty($row['participants'])) {
                $project['participants'] = json_decode($row['participants'], true);
            } else {
                $project['participants'] = [];
            }
            
            $projects[] = $project;
        }
        
        return $projects;
    }
    
    /**
     * Hämta projekt efter ID
     */
    public function getById($projectId) {
        $projects = $this->getAll();
        foreach ($projects as $project) {
            if (($project['id'] ?? '') === $projectId) {
                return $project;
            }
        }
        return null;
    }
    
    /**
     * Hämta endast aktiva projekt
     */
    public function getActive() {
        $projects = $this->getAll();
        return array_filter($projects, function($project) {
            return ($project['status'] ?? 'active') === 'active';
        });
    }
    
    /**
     * Skapa nytt projekt
     */
    public function create($projectData) {
        if ($this->store instanceof \MysqlDataStore) {
            return $this->createInMysql($projectData);
        } else {
            return $this->createInJson($projectData);
        }
    }
    
    private function createInJson($projectData) {
        $projects = $this->getAll();
        
        // Kontrollera om ID redan finns
        foreach ($projects as $project) {
            if (($project['id'] ?? '') === ($projectData['id'] ?? '')) {
                return false; // ID finns redan
            }
        }
        
        $projects[] = $projectData;
        return $this->store->set('projects', $projects);
    }
    
    private function createInMysql($projectData) {
        $pdo = $this->store->getPdo();
        
        try {
            $pdo->beginTransaction();
            
            // Skapa projekt
            $stmt = $pdo->prepare("
                INSERT INTO projects (id, name, description, status, created_at, updated_at)
                VALUES (?, ?, ?, ?, NOW(), NOW())
            ");
            $stmt->execute([
                $projectData['id'],
                $projectData['name'],
                $projectData['description'] ?? '',
                $projectData['status'] ?? 'active'
            ]);
            
            // Lägg till participants om de finns
            if (!empty($projectData['participants']) && is_array($projectData['participants'])) {
                $stmt = $pdo->prepare("
                    INSERT INTO project_participants (project_id, participant_id, type, added_at, added_by)
                    VALUES (?, ?, ?, ?, ?)
                ");
                
                foreach ($projectData['participants'] as $participant) {
                    $stmt->execute([
                        $projectData['id'],
                        $participant['participant_id'] ?? $participant['user_id'] ?? '',
                        $participant['type'] ?? 'user',
                        $participant['added_at'] ?? date('c'),
                        $participant['added_by'] ?? ''
                    ]);
                }
            }
            
            $pdo->commit();
            return true;
        } catch (PDOException $e) {
            $pdo->rollBack();
            error_log("ProjectRepository::createInMysql error: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Uppdatera projekt
     */
    public function update($projectId, $projectData) {
        if ($this->store instanceof \MysqlDataStore) {
            return $this->updateInMysql($projectId, $projectData);
        } else {
            return $this->updateInJson($projectId, $projectData);
        }
    }
    
    private function updateInJson($projectId, $projectData) {
        $projects = $this->getAll();
        $found = false;
        
        foreach ($projects as &$project) {
            if (($project['id'] ?? '') === $projectId) {
                $oldStatus = $project['status'] ?? 'active';
                $newStatus = $projectData['status'] ?? $oldStatus;
                
                // Merge update data into existing project
                $project = array_merge($project, $projectData);
                $project['id'] = $projectId; // Säkerställ att ID inte ändras
                
                // Om projektet avarkiveras (status ändras från 'archived' till 'active'),
                // ta bort archived_at explicit eftersom array_merge behåller fält som inte finns i update data
                if ($oldStatus === 'archived' && $newStatus !== 'archived') {
                    unset($project['archived_at']);
                }
                
                $found = true;
                break;
            }
        }
        
        if (!$found) {
            return false;
        }
        
        return $this->store->set('projects', $projects);
    }
    
    private function updateInMysql($projectId, $projectData) {
        $pdo = $this->store->getPdo();
        
        try {
            $pdo->beginTransaction();
            
            // Uppdatera projekt
            $stmt = $pdo->prepare("
                UPDATE projects 
                SET name = ?, description = ?, status = ?, updated_at = NOW(), archived_at = ?
                WHERE id = ?
            ");
            $stmt->execute([
                $projectData['name'] ?? '',
                $projectData['description'] ?? '',
                $projectData['status'] ?? 'active',
                $projectData['archived_at'] ?? null,
                $projectId
            ]);
            
            // Uppdatera participants om de finns
            if (isset($projectData['participants'])) {
                // Ta bort gamla
                $stmt = $pdo->prepare("DELETE FROM project_participants WHERE project_id = ?");
                $stmt->execute([$projectId]);
                
                // Lägg till nya
                if (!empty($projectData['participants']) && is_array($projectData['participants'])) {
                    $stmt = $pdo->prepare("
                        INSERT INTO project_participants (project_id, participant_id, type, added_at, added_by)
                        VALUES (?, ?, ?, ?, ?)
                    ");
                    
                    foreach ($projectData['participants'] as $participant) {
                        $stmt->execute([
                            $projectId,
                            $participant['participant_id'] ?? $participant['user_id'] ?? '',
                            $participant['type'] ?? 'user',
                            $participant['added_at'] ?? date('c'),
                            $participant['added_by'] ?? ''
                        ]);
                    }
                }
            }
            
            $pdo->commit();
            return true;
        } catch (PDOException $e) {
            $pdo->rollBack();
            error_log("ProjectRepository::updateInMysql error: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Ta bort projekt
     */
    public function delete($projectId) {
        if ($this->store instanceof \MysqlDataStore) {
            return $this->deleteFromMysql($projectId);
        } else {
            return $this->deleteFromJson($projectId);
        }
    }
    
    private function deleteFromJson($projectId) {
        $projects = $this->getAll();
        $found = false;
        
        foreach ($projects as $index => $project) {
            if (($project['id'] ?? '') === $projectId) {
                unset($projects[$index]);
                $found = true;
                break;
            }
        }
        
        if (!$found) {
            return false;
        }
        
        $projects = array_values($projects); // Re-index
        return $this->store->set('projects', $projects);
    }
    
    private function deleteFromMysql($projectId) {
        $pdo = $this->store->getPdo();
        
        try {
            $pdo->beginTransaction();
            
            // Ta bort participants först
            $stmt = $pdo->prepare("DELETE FROM project_participants WHERE project_id = ?");
            $stmt->execute([$projectId]);
            
            // Ta bort projekt
            $stmt = $pdo->prepare("DELETE FROM projects WHERE id = ?");
            $stmt->execute([$projectId]);
            
            $pdo->commit();
            return true;
        } catch (PDOException $e) {
            $pdo->rollBack();
            error_log("ProjectRepository::deleteFromMysql error: " . $e->getMessage());
            return false;
        }
    }
}
?>




