<?php
require_once __DIR__ . '/../DataStoreFactory.php';

/**
 * Repository för systemmeddelanden
 */
class SystemMessageRepository {
    private $store;
    
    public function __construct($store = null) {
        if ($store === null) {
            $this->store = DataStoreFactory::create();
        } else {
            $this->store = $store;
        }
    }
    
    /**
     * Hämta alla systemmeddelanden
     */
    public function getAll() {
        if ($this->store instanceof \MysqlDataStore) {
            return $this->getAllFromMysql();
        } else {
            return $this->getAllFromJson();
        }
    }
    
    private function getAllFromJson() {
        $messages = $this->store->get('system_messages');
        return is_array($messages) ? $messages : [];
    }
    
    private function getAllFromMysql() {
        $pdo = $this->store->getPdo();
        $stmt = $pdo->query("
            SELECT * FROM system_messages 
            ORDER BY created_at DESC
        ");
        
        $messages = [];
        while ($row = $stmt->fetch()) {
            $messages[] = [
                'id' => $row['id'],
                'message' => $row['message'],
                'priority' => $row['priority'] ?? null,
                'created_at' => $row['created_at'],
                'created_by' => $row['created_by'] ?? null,
                'expires_at' => $row['expires_at'] ?? null,
            ];
        }
        
        return $messages;
    }
    
    /**
     * Skapa nytt systemmeddelande
     */
    public function create($messageData) {
        if ($this->store instanceof \MysqlDataStore) {
            return $this->createInMysql($messageData);
        } else {
            return $this->createInJson($messageData);
        }
    }
    
    private function createInJson($messageData) {
        $messages = $this->getAll();
        $messages[] = $messageData;
        return $this->store->set('system_messages', $messages);
    }
    
    private function createInMysql($messageData) {
        $pdo = $this->store->getPdo();
        
        try {
            $stmt = $pdo->prepare("
                INSERT INTO system_messages (id, message, priority, created_at, expires_at)
                VALUES (?, ?, ?, ?, ?)
            ");
            $stmt->execute([
                $messageData['id'],
                $messageData['message'],
                $messageData['priority'] ?? null,
                $messageData['created_at'] ?? date('c'),
                $messageData['expires_at'] ?? null
            ]);
            
            return true;
        } catch (PDOException $e) {
            error_log("SystemMessageRepository::createInMysql error: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Markera meddelande som läst för en användare
     */
    public function markAsRead($messageId, $userId) {
        if ($this->store instanceof \MysqlDataStore) {
            return $this->markAsReadInMysql($messageId, $userId);
        } else {
            return $this->markAsReadInJson($messageId, $userId);
        }
    }
    
    private function markAsReadInJson($messageId, $userId) {
        $read = $this->store->get('system_message_read');
        if (!is_array($read)) {
            $read = [];
        }
        
        if (!isset($read[$messageId])) {
            $read[$messageId] = [];
        }
        
        $read[$messageId][$userId] = date('c');
        return $this->store->set('system_message_read', $read);
    }
    
    private function markAsReadInMysql($messageId, $userId) {
        $pdo = $this->store->getPdo();
        
        try {
            // Kontrollera om redan markerad som läst
            $stmt = $pdo->prepare("
                SELECT COUNT(*) as count FROM system_message_read 
                WHERE message_id = ? AND user_id = ?
            ");
            $stmt->execute([$messageId, $userId]);
            $row = $stmt->fetch();
            
            if ($row['count'] > 0) {
                return true; // Redan markerad
            }
            
            $stmt = $pdo->prepare("
                INSERT INTO system_message_read (message_id, user_id, read_at)
                VALUES (?, ?, NOW())
            ");
            $stmt->execute([$messageId, $userId]);
            
            return true;
        } catch (PDOException $e) {
            error_log("SystemMessageRepository::markAsReadInMysql error: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Hämta olästa meddelanden för en användare
     */
    public function getUnreadForUser($userId) {
        if ($this->store instanceof \MysqlDataStore) {
            return $this->getUnreadForUserFromMysql($userId);
        } else {
            return $this->getUnreadForUserFromJson($userId);
        }
    }
    
    private function getUnreadForUserFromJson($userId) {
        $messages = $this->getAll();
        $read = $this->store->get('system_message_read');
        if (!is_array($read)) {
            $read = [];
        }
        
        $unread = [];
        foreach ($messages as $message) {
            $messageId = $message['id'] ?? '';
            if (empty($messageId)) {
                continue;
            }
            
            // Kontrollera om meddelandet är utgånget
            if (!empty($message['expires_at'])) {
                $expiresAt = strtotime($message['expires_at']);
                if ($expiresAt < time()) {
                    continue; // Hoppa över utgångna meddelanden
                }
            }
            
            // Kontrollera om användaren har läst meddelandet
            if (!isset($read[$messageId][$userId])) {
                $unread[] = $message;
            }
        }
        
        return $unread;
    }
    
    private function getUnreadForUserFromMysql($userId) {
        $pdo = $this->store->getPdo();
        $stmt = $pdo->prepare("
            SELECT sm.* 
            FROM system_messages sm
            LEFT JOIN system_message_read smr ON sm.id = smr.message_id AND smr.user_id = ?
            WHERE smr.message_id IS NULL
            AND (sm.expires_at IS NULL OR sm.expires_at > NOW())
            ORDER BY sm.created_at DESC
        ");
        $stmt->execute([$userId]);
        
        $messages = [];
        while ($row = $stmt->fetch()) {
            $messages[] = [
                'id' => $row['id'],
                'message' => $row['message'],
                'priority' => $row['priority'] ?? null,
                'created_at' => $row['created_at'],
                'created_by' => $row['created_by'] ?? null,
                'expires_at' => $row['expires_at'] ?? null,
            ];
        }
        
        return $messages;
    }
    
    /**
     * Kontrollera om användare har läst ett meddelande
     */
    public function hasUserRead($messageId, $userId) {
        if ($this->store instanceof \MysqlDataStore) {
            return $this->hasUserReadFromMysql($messageId, $userId);
        } else {
            return $this->hasUserReadFromJson($messageId, $userId);
        }
    }
    
    private function hasUserReadFromJson($messageId, $userId) {
        $read = $this->store->get('system_message_read');
        if (!is_array($read)) {
            return false;
        }
        
        return isset($read[$messageId][$userId]);
    }
    
    private function hasUserReadFromMysql($messageId, $userId) {
        $pdo = $this->store->getPdo();
        $stmt = $pdo->prepare("
            SELECT COUNT(*) as count FROM system_message_read 
            WHERE message_id = ? AND user_id = ?
        ");
        $stmt->execute([$messageId, $userId]);
        $row = $stmt->fetch();
        
        return $row['count'] > 0;
    }
}
?>




