<?php
require_once __DIR__ . '/../DataStoreFactory.php';

/**
 * Repository för användar-operationer
 */
class UserRepository {
    private $store;
    
    public function __construct($store = null) {
        if ($store === null) {
            $this->store = DataStoreFactory::create();
        } else {
            $this->store = $store;
        }
    }
    
    /**
     * Hämta alla användare
     */
    public function getAll() {
        if ($this->store instanceof \MysqlDataStore) {
            return $this->getAllFromMysql();
        } else {
            return $this->getAllFromJson();
        }
    }
    
    private function getAllFromJson() {
        $users = $this->store->get('userids');
        return is_array($users) ? $users : [];
    }
    
    private function getAllFromMysql() {
        $pdo = $this->store->getPdo();
        $stmt = $pdo->query("SELECT * FROM users ORDER BY created_at DESC");
        
        $users = [];
        while ($row = $stmt->fetch()) {
            $users[] = [
                'user_id' => $row['user_id'],
                'full_name' => $row['full_name'],
                'mobile' => $row['mobile'] ?? '',
                'user_type' => $row['user_type'],
                'active' => (bool)$row['active'],
                'can_chat' => isset($row['can_chat']) ? (bool)$row['can_chat'] : true,
                'can_upload' => isset($row['can_upload']) ? (bool)$row['can_upload'] : true,
                'pushover_user_key' => $row['pushover_user_key'] ?? '',
                'nickname' => $row['nickname'] ?? '',
                'created_at' => $row['created_at'],
                'updated_at' => $row['updated_at'] ?? null,
            ];
        }
        
        return $users;
    }
    
    /**
     * Hämta användare efter ID
     */
    public function getById($userId) {
        if ($this->store instanceof \MysqlDataStore) {
            return $this->getByIdFromMysql($userId);
        } else {
            return $this->getByIdFromJson($userId);
        }
    }
    
    private function getByIdFromJson($userId) {
        $users = $this->getAll();
        foreach ($users as $user) {
            if (($user['user_id'] ?? '') === $userId) {
                return $user;
            }
        }
        return null;
    }
    
    private function getByIdFromMysql($userId) {
        $pdo = $this->store->getPdo();
        $stmt = $pdo->prepare("SELECT * FROM users WHERE user_id = ?");
        $stmt->execute([$userId]);
        $row = $stmt->fetch();
        
        if (!$row) {
            return null;
        }
        
        return [
            'user_id' => $row['user_id'],
            'full_name' => $row['full_name'],
            'mobile' => $row['mobile'] ?? '',
            'user_type' => $row['user_type'],
            'active' => (bool)$row['active'],
            'can_chat' => isset($row['can_chat']) ? (bool)$row['can_chat'] : true,
            'can_upload' => isset($row['can_upload']) ? (bool)$row['can_upload'] : true,
            'pushover_user_key' => $row['pushover_user_key'] ?? '',
            'nickname' => $row['nickname'] ?? '',
            'created_at' => $row['created_at'],
            'updated_at' => $row['updated_at'] ?? null,
        ];
    }
    
    /**
     * Hämta användare efter typ
     */
    public function getByType($userType) {
        $users = $this->getAll();
        return array_filter($users, function($user) use ($userType) {
            return ($user['user_type'] ?? 'user') === $userType;
        });
    }
    
    /**
     * Skapa ny användare
     */
    public function create($userData) {
        if ($this->store instanceof \MysqlDataStore) {
            return $this->createInMysql($userData);
        } else {
            return $this->createInJson($userData);
        }
    }
    
    private function createInJson($userData) {
        $users = $this->getAll();
        
        // Kontrollera om user_id redan finns
        foreach ($users as $user) {
            if (($user['user_id'] ?? '') === ($userData['user_id'] ?? '')) {
                return false; // User ID finns redan
            }
        }
        
        $users[] = $userData;
        return $this->store->set('userids', $users);
    }
    
    private function createInMysql($userData) {
        $pdo = $this->store->getPdo();
        
        try {
            $stmt = $pdo->prepare("
                INSERT INTO users (user_id, full_name, mobile, user_type, active, can_chat, can_upload, pushover_user_key, nickname, created_at, updated_at)
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, NOW(), NOW())
            ");
            $stmt->execute([
                $userData['user_id'],
                $userData['full_name'] ?? '',
                $userData['mobile'] ?? '',
                $userData['user_type'] ?? 'user',
                isset($userData['active']) ? ($userData['active'] ? 1 : 0) : 1,
                isset($userData['can_chat']) ? ($userData['can_chat'] ? 1 : 0) : 1,
                isset($userData['can_upload']) ? ($userData['can_upload'] ? 1 : 0) : 1,
                $userData['pushover_user_key'] ?? '',
                $userData['nickname'] ?? ''
            ]);
            
            return true;
        } catch (PDOException $e) {
            error_log("UserRepository::createInMysql error: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Uppdatera användare
     */
    public function update($userId, $userData) {
        if ($this->store instanceof \MysqlDataStore) {
            return $this->updateInMysql($userId, $userData);
        } else {
            return $this->updateInJson($userId, $userData);
        }
    }
    
    private function updateInJson($userId, $userData) {
        $users = $this->getAll();
        $found = false;
        
        foreach ($users as &$user) {
            if (($user['user_id'] ?? '') === $userId) {
                $user = array_merge($user, $userData);
                $user['user_id'] = $userId; // Säkerställ att ID inte ändras
                $found = true;
                break;
            }
        }
        
        if (!$found) {
            return false;
        }
        
        return $this->store->set('userids', $users);
    }
    
    private function updateInMysql($userId, $userData) {
        $pdo = $this->store->getPdo();
        
        try {
            $fields = [];
            $values = [];
            
            if (isset($userData['full_name'])) {
                $fields[] = 'full_name = ?';
                $values[] = $userData['full_name'];
            }
            if (isset($userData['mobile'])) {
                $fields[] = 'mobile = ?';
                $values[] = $userData['mobile'];
            }
            if (isset($userData['user_type'])) {
                $fields[] = 'user_type = ?';
                $values[] = $userData['user_type'];
            }
            if (isset($userData['active'])) {
                $fields[] = 'active = ?';
                $values[] = $userData['active'] ? 1 : 0;
            }
            if (isset($userData['can_chat'])) {
                $fields[] = 'can_chat = ?';
                $values[] = $userData['can_chat'] ? 1 : 0;
            }
            if (isset($userData['can_upload'])) {
                $fields[] = 'can_upload = ?';
                $values[] = $userData['can_upload'] ? 1 : 0;
            }
            if (isset($userData['pushover_user_key'])) {
                $fields[] = 'pushover_user_key = ?';
                $values[] = $userData['pushover_user_key'];
            }
            if (isset($userData['nickname'])) {
                $fields[] = 'nickname = ?';
                $values[] = $userData['nickname'];
            }
            
            if (empty($fields)) {
                return true; // Inga ändringar
            }
            
            $fields[] = 'updated_at = NOW()';
            $values[] = $userId;
            
            $sql = "UPDATE users SET " . implode(', ', $fields) . " WHERE user_id = ?";
            $stmt = $pdo->prepare($sql);
            $stmt->execute($values);
            
            return true;
        } catch (PDOException $e) {
            error_log("UserRepository::updateInMysql error: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Ta bort användare
     */
    public function delete($userId) {
        if ($this->store instanceof \MysqlDataStore) {
            return $this->deleteFromMysql($userId);
        } else {
            return $this->deleteFromJson($userId);
        }
    }
    
    private function deleteFromJson($userId) {
        $users = $this->getAll();
        $found = false;
        
        foreach ($users as $index => $user) {
            if (($user['user_id'] ?? '') === $userId) {
                unset($users[$index]);
                $found = true;
                break;
            }
        }
        
        if (!$found) {
            return false;
        }
        
        $users = array_values($users); // Re-index
        return $this->store->set('userids', $users);
    }
    
    private function deleteFromMysql($userId) {
        $pdo = $this->store->getPdo();
        
        try {
            $stmt = $pdo->prepare("DELETE FROM users WHERE user_id = ?");
            $stmt->execute([$userId]);
            return true;
        } catch (PDOException $e) {
            error_log("UserRepository::deleteFromMysql error: " . $e->getMessage());
            return false;
        }
    }
}
?>




