<?php
require_once __DIR__ . '/../config.php';
require_once __DIR__ . '/dal/repositories/ProjectRepository.php';
require_once __DIR__ . '/dal/repositories/UserRepository.php';
require_once __DIR__ . '/dal/repositories/ConfigRepository.php';

/**
 * Hämtar alla uppladdningsdata och returnerar strukturerad data
 */
function getAllUploadData() {
    // Hämta projektdata
    $projectsData = [];
    $projectNames = []; // Lookup array for project names
    
    $projectRepo = new ProjectRepository();
    $allProjects = $projectRepo->getActive();
    
    foreach ($allProjects as $project) {
        $projectsData[] = $project;
        $projectNames[$project['id']] = $project['name'];
    }

    // Hämta alla uppladdningar och gruppera efter projekt och upload_id
    $uploadRoot = __DIR__ . '/../' . rtrim(UPLOAD_ROOT, '/\\');
    $allUploads = [];
    $uploadIds = [];
    $userIds = [];
    $projectUploads = [];

    if (is_dir($uploadRoot)) {
        $projectDirs = glob($uploadRoot . '/*', GLOB_ONLYDIR);
        
        foreach ($projectDirs as $projectDir) {
            $project_id = basename($projectDir);
            $projectFiles = [];
            
            // Hämta alla JSON-filer (metadata)
            $jsonFiles = glob($projectDir . '/*.json');
            
            foreach ($jsonFiles as $jsonFile) {
                $metadata = json_decode(file_get_contents($jsonFile), true);
                if ($metadata) {
                    $actualImageFile = null;
                    $imageExt = null;
                    
                    // Först: Försök använda stored_file från JSON-metadata om det finns
                    if (!empty($metadata['stored_file'])) {
                        $storedFile = $projectDir . '/' . basename($metadata['stored_file']);
                        if (file_exists($storedFile) && is_file($storedFile)) {
                            $actualImageFile = $storedFile;
                            $imageExt = strtolower(pathinfo($metadata['stored_file'], PATHINFO_EXTENSION));
                        }
                    }
                    
                    // Fallback: Om stored_file inte finns eller inte matchar, försök hitta baserat på JSON-filnamn
                    if (!$actualImageFile) {
                        // Kontrollera alla stödda bildformat och videoformat
                        $supportedImageFormats = ['jpg', 'jpeg', 'png', 'webp', 'heic', 'heif'];
                        $supportedVideoFormats = ['mp4', 'webm', 'mov', 'avi'];
                        $allSupportedFormats = array_merge($supportedImageFormats, $supportedVideoFormats);
                        
                        foreach ($allSupportedFormats as $format) {
                            $testFile = str_replace('.json', '.' . $format, $jsonFile);
                            if (file_exists($testFile)) {
                                $actualImageFile = $testFile;
                                $imageExt = $format;
                                break;
                            }
                        }
                    }
                    
                    if ($actualImageFile) {
                        // Kontrollera att projektet är aktivt innan bilden läggs till
                        // Om projektet inte finns i projects.json, använd projekt-ID som namn
                        $projectName = $projectNames[$project_id] ?? $project_id;
                        $projectExists = isset($projectNames[$project_id]);
                        
                        // Inkludera bilden alltid (projektet är redan filtrerat till aktiva i $projectNames,
                        // men vi inkluderar också projekt som saknas i projects.json)
                        // Detta gör att bilder i projekt som saknas i projects.json fortfarande visas
                        $metadata['image_path'] = $actualImageFile;
                        $metadata['image_url'] = '../../view_file.php?project_id=' . urlencode($project_id) . '&file=' . urlencode(basename($actualImageFile));
                        
                        // Bestäm filtyp (använd från JSON om det finns, annars gissa från filändelse)
                        if (!isset($metadata['file_type'])) {
                            $supportedVideoFormats = ['mp4', 'webm', 'mov', 'avi'];
                            $isVideo = in_array($imageExt, $supportedVideoFormats);
                            $metadata['file_type'] = $isVideo ? 'video' : 'image';
                        }
                        
                        // Thumb detection (använd thumb_file från JSON om det finns, annars gissa)
                        if (($metadata['file_type'] ?? 'image') !== 'video') {
                            if (!empty($metadata['thumb_file'])) {
                                $thumbPath = $projectDir . '/' . basename($metadata['thumb_file']);
                                if (file_exists($thumbPath)) {
                                    $metadata['thumb_path'] = $thumbPath;
                                    $metadata['thumb_url'] = '../../view_file.php?project_id=' . urlencode($project_id) . '&file=' . urlencode(basename($thumbPath));
                                }
                            } else {
                                // Fallback: Gissa thumbnail-namn
                                $baseNoExt = preg_replace('/\.(jpg|jpeg|png|webp|heic|heif)$/i', '', $actualImageFile);
                                $thumbPath = $baseNoExt . '-thumb.jpg';
                                if (file_exists($thumbPath)) {
                                    $metadata['thumb_path'] = $thumbPath;
                                    $metadata['thumb_url'] = '../../view_file.php?project_id=' . urlencode($project_id) . '&file=' . urlencode(basename($thumbPath));
                                }
                            }
                        }
                        $metadata['project_id'] = $project_id;
                        $metadata['project_name'] = $projectName;
                        
                        $projectFiles[] = $metadata;
                        $allUploads[] = $metadata;
                        
                        // Samla unika upload_ids
                        if (isset($metadata['upload_id'])) {
                            $uploadIds[$metadata['upload_id']] = $metadata['upload_id'];
                        }
                        
                        // Samla unika user_ids
                        if (isset($metadata['user_id']) && $metadata['user_id'] !== '') {
                            $userIds[$metadata['user_id']] = $metadata['user_id'];
                        } else {
                            // Lägg till tom sträng för uppladdningar utan user_id
                            $userIds[''] = '';
                        }
                    }
                }
            }
            
            // Sortera efter datum (nyaste först)
            usort($projectFiles, function($a, $b) {
                return strtotime($b['stored_at']) - strtotime($a['stored_at']);
            });
            
            if (!empty($projectFiles)) {
                $projectUploads[$project_id] = $projectFiles;
            }
        }
    }

    // Sortera upload_ids (nyaste först)
    arsort($uploadIds);
    
    // Sortera user_ids
    sort($userIds);

    // Sortera projekt efter ID
    ksort($projectUploads);

    return [
        'projectsData' => $projectsData,
        'projectNames' => $projectNames,
        'allUploads' => $allUploads,
        'uploadIds' => $uploadIds,
        'userIds' => $userIds,
        'projectUploads' => $projectUploads
    ];
}

/**
 * Läs in en karta user_id => full_name från userids
 */
function getUserIdToNameMap() {
    static $cachedMap = null;
    if ($cachedMap !== null) {
        return $cachedMap;
    }

    $cachedMap = [];
    $userRepo = new UserRepository();
    $users = $userRepo->getAll();

    foreach ($users as $entry) {
        if (isset($entry['user_id']) && isset($entry['full_name']) && $entry['user_id'] !== '') {
            $cachedMap[$entry['user_id']] = $entry['full_name'];
        }
    }

    return $cachedMap;
}

/**
 * Hämta fullständigt namn för ett user_id (eller tom sträng om okänt)
 */
function getUserFullName($userId) {
    if ($userId === null || $userId === '') {
        return '';
    }
    $map = getUserIdToNameMap();
    return $map[$userId] ?? '';
}

/**
 * Resolve short user ID to full UUID (for backward compatibility)
 */
function resolveLinkToUserId($userId) {
    if (empty($userId)) {
        return null;
    }
    
    $userRepo = new UserRepository();
    $user = $userRepo->getById($userId);
    
    if ($user && isset($user['user_id'])) {
        return $user['user_id'];
    }
    
    return null;
}

/**
 * Hämtar nästa tillgängliga bildnummer och ökar det atomiskt
 * @return int Nästa bildnummer
 */
function getNextImageNumber() {
    $configRepo = new ConfigRepository();
    return $configRepo->incrementImageNumber();
}

/**
 * Tilldelar ett bildnummer till en bild och sparar det i metadata
 * @param string $imagePath Sökväg till bildfilen
 * @param string $jsonPath Sökväg till JSON-metadatafilen
 * @return int|false Bildnumret om lyckat, annars false
 */
function assignImageNumber($imagePath, $jsonPath) {
    // Kontrollera att JSON-filen finns
    if (!file_exists($jsonPath)) {
        return false;
    }
    
    // Läs befintlig metadata
    $content = file_get_contents($jsonPath);
    if ($content === false) {
        return false;
    }
    
    $metadata = json_decode($content, true);
    if (!is_array($metadata)) {
        return false;
    }
    
    // Om bilden redan har ett nummer, returnera det
    if (isset($metadata['image_number']) && is_numeric($metadata['image_number'])) {
        return (int)$metadata['image_number'];
    }
    
    // Hämta nästa nummer
    $imageNumber = getNextImageNumber();
    
    // Lägg till numret i metadata
    $metadata['image_number'] = $imageNumber;
    
    // Spara tillbaka
    $result = file_put_contents($jsonPath, json_encode($metadata, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
    
    if ($result === false) {
        return false;
    }
    
    return $imageNumber;
}
?>
