<?php
/**
 * Helper-funktioner för att hantera chat_mapping.json och project_mapping.json
 */
require_once __DIR__ . '/dal/repositories/MappingRepository.php';

/**
 * Läsa chat_mapping.json
 * @param string|null $projectId Om angivet, returnera bara chatters för detta projekt
 * @param string|null $mappingFile Sökväg till chat_mapping.json (optional, ignoreras nu)
 * @return array Array med chatters (eller array med projekt som nyckel om $projectId är null)
 */
function readChatMapping($projectId = null, $mappingFile = null) {
    $mappingRepo = new MappingRepository();
    return $mappingRepo->getChatMappings($projectId);
}

/**
 * Läsa project_mapping.json
 * @param string|null $projectId Om angivet, returnera bara participants för detta projekt
 * @param string|null $mappingFile Sökväg till project_mapping.json (optional, ignoreras nu)
 * @return array Array med participants (eller array med projekt som nyckel om $projectId är null)
 */
function readProjectMapping($projectId = null, $mappingFile = null) {
    $mappingRepo = new MappingRepository();
    return $mappingRepo->getProjectMappings($projectId);
}

/**
 * Spara chat_mapping.json
 * @param string $projectId Projekt-ID
 * @param array $chatters Array av chatters
 * @param string|null $mappingFile Sökväg till chat_mapping.json (optional, ignoreras nu)
 * @return bool True om lyckat, false annars
 */
function saveChatMapping($projectId, $chatters, $mappingFile = null) {
    // För MySQL behöver vi hantera detta annorlunda
    // För nu, ta bort alla och lägg till igen
    $mappingRepo = new MappingRepository();
    
    // Hämta befintliga för att se vilka som ska tas bort
    $existing = $mappingRepo->getChatMappings($projectId);
    $existingIds = array_map(function($c) { return $c['participant_id'] ?? ''; }, $existing);
    $newIds = array_map(function($c) { return $c['participant_id'] ?? ''; }, $chatters);
    
    // Ta bort de som inte finns i nya listan
    foreach ($existingIds as $id) {
        if (!in_array($id, $newIds)) {
            $mappingRepo->removeChatter($projectId, $id);
        }
    }
    
    // Lägg till nya
    foreach ($chatters as $chatter) {
        $id = $chatter['participant_id'] ?? '';
        if (!in_array($id, $existingIds)) {
            $mappingRepo->addChatter(
                $projectId,
                $id,
                $chatter['type'] ?? 'user',
                $chatter['added_by'] ?? ''
            );
        }
    }
    
    return true;
}

/**
 * Spara project_mapping.json
 * @param string $projectId Projekt-ID
 * @param array $participants Array av participants
 * @param string|null $mappingFile Sökväg till project_mapping.json (optional, ignoreras nu)
 * @return bool True om lyckat, false annars
 */
function saveProjectMapping($projectId, $participants, $mappingFile = null) {
    // För MySQL behöver vi hantera detta annorlunda
    // För nu, ta bort alla och lägg till igen
    $mappingRepo = new MappingRepository();
    
    // Hämta befintliga för att se vilka som ska tas bort
    $existing = $mappingRepo->getProjectMappings($projectId);
    $existingIds = array_map(function($p) { return $p['participant_id'] ?? ''; }, $existing);
    $newIds = array_map(function($p) { return $p['participant_id'] ?? ''; }, $participants);
    
    // Ta bort de som inte finns i nya listan
    foreach ($existingIds as $id) {
        if (!in_array($id, $newIds)) {
            $mappingRepo->removeParticipant($projectId, $id);
        }
    }
    
    // Lägg till nya
    foreach ($participants as $participant) {
        $id = $participant['participant_id'] ?? '';
        if (!in_array($id, $existingIds)) {
            $mappingRepo->addParticipant(
                $projectId,
                $id,
                $participant['type'] ?? 'user',
                $participant['added_by'] ?? ''
            );
        }
    }
    
    return true;
}

/**
 * Lägg till chatter i ett projekt
 * @param string $projectId Projekt-ID
 * @param string $participantId Deltagar-ID
 * @param string $type Typ ('user' eller 'extern')
 * @param string $addedBy Användare som lade till
 * @param string|null $mappingFile Sökväg till chat_mapping.json (optional, ignoreras nu)
 * @return bool|string True om lyckat, false om redan finns, error message som string om fel
 */
function addChatter($projectId, $participantId, $type, $addedBy, $mappingFile = null) {
    $mappingRepo = new MappingRepository();
    $result = $mappingRepo->addChatter($projectId, $participantId, $type, $addedBy);
    
    if ($result === false) {
        return false; // Redan finns
    } elseif ($result === true) {
        return true;
    } else {
        return 'Kunde inte spara chat_mapping';
    }
}

/**
 * Ta bort chatter från ett projekt
 * @param string $projectId Projekt-ID
 * @param string $participantId Deltagar-ID
 * @param string|null $mappingFile Sökväg till chat_mapping.json (optional, ignoreras nu)
 * @return bool True om lyckat, false annars
 */
function removeChatter($projectId, $participantId, $mappingFile = null) {
    $mappingRepo = new MappingRepository();
    return $mappingRepo->removeChatter($projectId, $participantId);
}

/**
 * Lägg till participant i ett projekt
 * @param string $projectId Projekt-ID
 * @param string $participantId Deltagar-ID
 * @param string $type Typ ('user' eller 'extern')
 * @param string $addedBy Användare som lade till
 * @param string|null $mappingFile Sökväg till project_mapping.json (optional, ignoreras nu)
 * @return bool|string True om lyckat, false om redan finns, error message som string om fel
 */
function addParticipant($projectId, $participantId, $type, $addedBy, $mappingFile = null) {
    $mappingRepo = new MappingRepository();
    $result = $mappingRepo->addParticipant($projectId, $participantId, $type, $addedBy);
    
    if ($result === false) {
        return false; // Redan finns
    } elseif ($result === true) {
        return true;
    } else {
        return 'Kunde inte spara project_mapping';
    }
}

/**
 * Ta bort participant från ett projekt
 * @param string $projectId Projekt-ID
 * @param string $participantId Deltagar-ID
 * @param string|null $mappingFile Sökväg till project_mapping.json (optional, ignoreras nu)
 * @return bool True om lyckat, false annars
 */
function removeParticipant($projectId, $participantId, $mappingFile = null) {
    $mappingRepo = new MappingRepository();
    return $mappingRepo->removeParticipant($projectId, $participantId);
}

